import { consume } from '@lit/context';
import { LitElement, css, html } from 'lit';
import { customElement, property, state } from 'lit/decorators.js';
import { classMap } from 'lit/directives/class-map.js';
import { ifDefined } from 'lit/directives/if-defined.js';
import { createRef, ref, type Ref } from 'lit/directives/ref.js';
import { avatars, profiles } from '../../config.js';
import type { LightTheme, Profile } from '../../types.js';
import { DiscordInputText } from '../discord-input-text/DiscordInputText.js';
import { messagesLightTheme } from '../discord-messages/DiscordMessages.js';
import ModalClose from '../svgs/ModalClose.js';
import ModalWarning from '../svgs/ModalWarning.js';

@customElement('discord-modal')
export class DiscordModal extends LitElement implements LightTheme {
	public static override readonly styles = css`
		@keyframes modal-pop {
			0% {
				opacity: 0;
			}
		}

		*,
		:before,
		:after {
			border-width: 0 !important;
		}

		/* This does positioning, sizing, and transition */
		.discord-modal {
			pointer-events: none;
			position: fixed;
			top: 0;
			right: 0;
			bottom: 0;
			left: 0;
			margin: 0;
			display: grid;
			height: 100%;
			width: 100%;
			max-height: none;
			max-width: none;
			justify-items: center;
			padding: 0;
			opacity: 0;
			overscroll-behavior: contain;
			z-index: 1002;
			background-color: transparent;
			color: inherit;
			transition-duration: 0.2s;
			transition-timing-function: ease;
			transition-property: all;
			overflow: hidden;
		}

		.discord-modal-open,
		.discord-modal:target,
		.discord-modal-toggle:checked + .discord-modal,
		.discord-modal[open] {
			pointer-events: auto;
			visibility: visible;
			opacity: 1;
		}

		:where(.discord-modal) {
			align-items: center;
		}

		.discord-modal-box {
			display: none;
			flex-direction: column;
			min-height: 0;
			max-width: 100%;
			background-color: oklab(0.321088 -0.000220731 -0.00934622);
			transform: translate(0, 0) rotate(0) skew(0) skewY(0) scaleX(0.9) scaleY(0.9);
			transition-property: all;
			transition-timing-function: ease;
			transition-duration: 0.2s;
			box-shadow: none;
			overflow-y: visible;
			border-radius: 4px;
		}

		.discord-modal-box:focus {
			outline: none;
			border: none;
		}

		.discord-modal-open .discord-modal-box,
		.discord-modal-toggle:checked + .discord-modal .discord-modal-box,
		.discord-modal:target .discord-modal-box,
		.discord-modal[open] .discord-modal-box {
			transform: translate(0, 0px) rotate(0) skew(0) skewY(0) scaleX(1) scaleY(1);
		}

		.discord-modal:not(dialog:not(.discord-modal-open)),
		.discord-modal::backdrop {
			background-color: rgba(0, 0, 0, 0.7);
			animation: modal-pop 0.2s ease-out;
		}

		.discord-modal-light-theme:not(dialog:not(.discord-modal-open)),
		.discord-modal-light-theme::backdrop {
			background-color: rgba(0, 0, 0, 0.54);
			backdrop-filter: blur(0px);
		}

		.discord-modal-root {
			background-color: oklab(0.321088 -0.000220731 -0.00934622);
			border-radius: 4px;
			display: flex;
			flex-direction: column;
			margin: 0 auto;
			pointer-events: all;
			position: relative;
			box-shadow:
				rgba(30, 31, 34, 0.6) 0px 0px 0px 1px,
				rgba(0, 0, 0, 0.2) 0px 2px 10px 0px;
			opacity: 1;
			transform: scale(1);
			width: 440px;
			max-height: 720px;
			min-height: 200px;
		}

		.discord-modal-content-container::-webkit-scrollbar {
			width: 5px;
			background-color: transparent;
		}

		.discord-modal-content-container::-webkit-scrollbar-track {
			background-color: transparent;
		}

		.discord-modal-content-container::-webkit-scrollbar-thumb {
			border-radius: 10px;
			background-color: rgba(0, 0, 0, 0.5);
		}

		.discord-modal-root-light-theme {
			box-shadow:
				0 0 0 1px hsl(210 calc(1 * 9.3%) 78.8% / 0.3),
				0 2px 10px 0 hsl(0 calc(1 * 0%) 0% / 0.1);
			background-color: color-mix(in oklab, hsl(0 calc(1 * 0%) 100% / 1) 100%, black 0%);
		}

		.discord-modal-header-container {
			flex: 0 0 auto;
			border-radius: 4px 4px 0 0;
			transition: box-shadow 0.1s ease-out;
			word-wrap: break-word;
			position: relative;
			padding: 16px;
			z-index: 1;
			overflow: hidden;
			flex-wrap: nowrap;
			justify-content: flex-start;
			align-items: center;
			flex-direction: row;
			display: flex;
		}

		.discord-modal-content-container {
			overflow: hidden auto;
			padding-right: 8px;
			position: relative;
			z-index: 0;
			border-radius: 5px 5px 0 0;
			padding-left: 16px;
			position: relative;
			z-index: 0;
			border-radius: 5px 5px 0 0;
			padding-left: 16px;
		}

		.discord-modal-actions-container {
			flex: 0 0 auto;
			box-shadow: inset 0 1px 0 hsl(220 calc(1 * 6.5%) 18% / 0.6);
			border-radius: 0 0 5px 5px;
			background-color: color-mix(in oklab, hsl(220 calc(1 * 6.5%) 18% / 1) 100%, black 0%);
			position: relative;
			padding: 16px;
			z-index: 1;
			overflow-x: hidden;
			flex-wrap: nowrap;
			justify-content: flex-start;
			align-items: center;
			flex-direction: row-reverse;
			display: flex;
		}

		.discord-modal-actions-container-light-theme {
			box-shadow: inset 0 1px 0 hsl(0 calc(1 * 0%) 97.6% / 0.6);
			background-color: color-mix(in oklab, hsl(220 calc(1 * 13%) 95.5% / 1) 100%, black 0%);
		}

		.discord-modal-avatar {
			margin-right: 0.5em;
			position: relative;
			min-width: 24px;
			width: 24px;
			height: 24px;
			border-radius: 50%;
		}

		.discord-modal-avatar img {
			width: 24px;
			height: 24px;
			border-radius: 50%;
		}

		.discord-modal-title {
			font-size: 24px;
			line-height: 30px;
			color: color-mix(in oklab, hsl(220 calc(1 * 13%) 95.5% / 1) 100%, black 0%);
		}

		.discord-modal-title-light-theme {
			color: color-mix(in oklab, hsl(240 calc(1 * 7.7%) 2.5% / 1) 100%, black 0%);
		}

		.discord-modal-close-button {
			position: absolute;
			top: 16px;
			right: 16px;
			height: 26px;
			padding: 4px;
			transition: opacity.2s ease-in-out;
			opacity: 0.5;
			cursor: pointer;
			border-radius: 3px;
			color: color-mix(in oklab, hsl(215 calc(1 * 8.8%) 73.3% / 1) 100%, black 0%);
			box-sizing: content-box;
			width: auto;
			background: transparent;
			border: 0;
			margin: 0;
			display: flex;
			justify-content: center;
			align-items: center;
			font-size: 14px;
			font-weight: 500;
			line-height: 16px;
			-moz-user-select: none;
			user-select: none;
			outline: 0;
			font-family: 'gg sans', 'Noto Sans', Whitney, 'Helvetica Neue', Helvetica, Roboto, Arial, sans-serif;
			text-rendering: optimizeLegibility;
		}

		.discord-modal-close-button-light-theme {
			color: color-mix(in oklab, hsl(228 calc(1 * 6%) 32.5% / 1) 100%, black 0%);
		}

		.discord-modal-close-button:hover {
			opacity: 1;
			color: oklab(0.899401 -0.00192499 -0.00481987);
		}

		.discord-modal-close-button-light-theme:hover {
			color: color-mix(in oklab, hsl(223 calc(1 * 6.7%) 20.6% / 1) 100%, black 0%);
		}

		.discord-modal-close-button-content {
			background-image: linear-gradient(
				to top,
				transparent,
				transparent 1px,
				transparent 1px,
				transparent calc(1px + 1px),
				transparent calc(1px + 1px)
			);
		}

		.discord-modal-warning-container {
			margin-bottom: 8px;
			background: color-mix(in oklab, hsl(40 calc(1 * 86.4%) 56.9% / 0.1) 100%, hsl(0 0% 0% / 0.1) 0%);
			border: 1px solid color-mix(in oklab, hsl(40 calc(1 * 86.4%) 56.9% / 1) 100%, black 0%);
			color: color-mix(in oklab, hsl(0 calc(1 * 0%) 100% / 1) 100%, black 0%);
			display: flex;
			border-radius: 4px;
			font-weight: 500;
			padding: 8px;
			width: 100%;
			box-sizing: border-box;
		}

		.discord-modal-warning-container-light-theme {
			background: color-mix(in oklab, hsl(38 calc(1 * 78.6%) 38.4% / 0.1) 100%, hsl(0 0% 0% / 0.1) 0%);
			border: color-mix(in oklab, hsl(38 calc(1 * 78.6%) 38.4% / 1) 100%, black 0%);
			color: color-mix(in oklab, hsl(0 calc(1 * 0%) 0% / 1) 100%, black 0%);
		}

		.discord-modal-warning-icon {
			display: flex;
		}

		.discord-modal-warning {
			color: color-mix(in oklab, hsl(210 calc(1 * 9.1%) 87.1% / 1) 100%, black 0%);
			margin-left: 10px;
			flex: 1;
			align-self: center;
			font-family: 'gg sans', 'Noto Sans', Whitney, 'Helvetica Neue', Helvetica, Roboto, Arial, sans-serif;
			font-size: 14px;
			line-height: 1.2857142857142858;
			font-weight: 500;
		}

		.discord-modal-warning-light-theme {
			color: color-mix(in oklab, hsl(223 calc(1 * 6.7%) 20.6% / 1) 100%, black 0%);
		}

		.discord-modal-warning-icon-svg {
			fill: color-mix(in oklab, hsl(215 calc(1 * 8.8%) 73.3% / 1) 100%, black 0%);
			color: color-mix(in oklab, hsl(40 calc(1 * 86.4%) 56.9% / 1) 100%, black 0%);
			width: 24px;
			height: 24px;
			flex-shrink: 0;
		}

		.discord-modal-warning-icon-svg-light-theme {
			color: color-mix(in oklab, hsl(38 calc(1 * 78.6%) 38.4% / 1) 100%, black 0%);
			fill: none;
		}

		.discord-modal-warning-icon-svg-light-theme > path {
			fill: color-mix(in oklab, hsl(228 calc(1 * 6%) 32.5% / 1) 100%, black 0%);
		}

		.discord-modal-warning-author {
			font-weight: 600;
		}

		.discord-modal-button {
			align-items: center;
			background: none;
			border-radius: 3px;
			border: none;
			box-sizing: border-box;
			color: hsl(0 calc(1 * 0%) 100% / 1);
			display: flex;
			font-size: 14px;
			font-weight: 500;
			height: 38px;
			justify-content: center;
			line-height: 16px;
			min-height: 38px;
			min-width: 96px;
			padding: 2px 16px;
			position: relative;
			-moz-user-select: none;
			user-select: none;
			width: auto;
			cursor: pointer;
		}

		.discord-modal-button-submit {
			background-color: hsl(235 calc(1 * 85.6%) 64.7% / 1);
			transition:
				background-color 170ms ease,
				color 170ms ease;
		}

		.discord-modal-button-submit:hover {
			background-color: hsl(235 calc(1 * 51.4%) 52.4% / 1);
		}

		.discord-modal-button-submit:active {
			background-color: hsl(235 calc(1 * 46.7%) 44.1% / 1);
		}

		.discord-modal-button-cancel-light-theme {
			color: hsl(223 calc(1 * 5.8%) 52.9% / 1);
		}

		.discord-modal-button-cancel:hover > .discord-modal-button-content {
			--button--underline-color: hsl(0 calc(1 * 0%) 100% / 1);
		}

		.discord-modal-button-cancel-light-theme:hover > .discord-modal-button-content-light-theme {
			--button--underline-color: hsl(223 calc(1 * 5.8%) 52.9% / 1);
		}

		.discord-modal-button-content {
			--button--underline-color: transparent;
			margin: 0 auto;
			white-space: nowrap;
			text-overflow: ellipsis;
			overflow: hidden;
			background-image: linear-gradient(
				to top,
				transparent,
				transparent 1px,
				var(--button--underline-color) 1px,
				var(--button--underline-color) calc(1px + 1px),
				transparent calc(1px + 1px)
			);
		}

		.discord-modal-slot-input-text {
			padding-right: 8px;
			padding-left: 20px;
			position: sticky;
			z-index: 3;
		}
	`;

	/**
	 * The id of the profile data to use.
	 */
	@property()
	public accessor profile: string | undefined = undefined;

	/**
	 * The message author's username.
	 *
	 * @defaultValue 'User'
	 */
	@property()
	public accessor author: string | undefined = 'User';

	/**
	 * The modal's author's avatar. Can be an avatar shortcut, relative path, or external link.
	 */
	@property()
	public accessor avatar: string | undefined = undefined;

	/**
	 * The `id` property of the `dialog` component. This is used to open the modal from the button and should match with `modal-id` on the button.
	 */
	@property({ reflect: true, attribute: 'modal-id' })
	public accessor modalId: string;

	/**
	 * The title of the modal, displayed at the top
	 */
	@property({ reflect: true, attribute: 'modal-title' })
	public accessor modalTitle: string;

	@property({ reflect: false, noAccessor: true, attribute: false })
	public accessor submitForm: (...args: unknown[]) => void;

	@property({ reflect: false, noAccessor: true, attribute: false })
	public accessor closeForm: (...args: unknown[]) => void;

	@consume({ context: messagesLightTheme, subscribe: true })
	@property({ type: Boolean, reflect: true, attribute: 'light-theme' })
	public accessor lightTheme = false;

	@state()
	public accessor originalBodyOverflow: string;

	protected dialogRef: Ref<HTMLDialogElement> = createRef();

	protected handleClickCloseIcon() {
		if (this.dialogRef.value) {
			this.dialogRef.value.close();
		}
	}

	protected handleFormSubmit(event: SubmitEvent) {
		event.preventDefault();
		event.stopPropagation();

		const slottedItems = this.shadowRoot?.querySelector('slot')?.assignedElements() ?? [];

		for (const slottedItem of slottedItems) {
			if (slottedItem instanceof DiscordInputText) {
				const shadowRootSlot = slottedItem.shadowRoot;

				const input = shadowRootSlot?.querySelector('input') ?? shadowRootSlot?.querySelector('textarea');

				if (input?.attributes.getNamedItem('required') && !input.value) {
					const messageNeeded = shadowRootSlot?.querySelector('div.discord-text-input-message-needed-input');

					if (messageNeeded instanceof HTMLDivElement && !messageNeeded.style.display) {
						messageNeeded.style.display = 'flex';

						globalThis.setTimeout(() => {
							messageNeeded.style.opacity = '1';
						}, 1);
					}

					return;
				}

				if (
					(input instanceof HTMLTextAreaElement &&
						input.value.length < Number(input.attributes.getNamedItem('minlength')?.value) &&
						input.value.length > 0) ||
					(input instanceof HTMLTextAreaElement &&
						input.value.length < Number(input.attributes.getNamedItem('minlength')?.value) &&
						input?.attributes.getNamedItem('required'))
				) {
					const messageNeededMinLength = shadowRootSlot?.querySelector('div.discord-text-input-message-needed-min-length');

					if (messageNeededMinLength instanceof HTMLDivElement && !messageNeededMinLength.style.display) {
						messageNeededMinLength.style.display = 'flex';
						slottedItem.hasWarning = true;

						globalThis.setTimeout(() => {
							messageNeededMinLength.style.opacity = '1';
						}, 1);
					}

					return;
				}
			}
		}

		this.submitForm?.();
		this.handleClickCloseIcon();
	}

	protected onCloseDialog() {
		globalThis.document.body.style.overflow = this.originalBodyOverflow ?? 'scroll';

		const divRootModal = this.shadowRoot?.querySelector('div.discord-modal-box');
		if (divRootModal instanceof HTMLDivElement) {
			divRootModal.style.display = 'none';
		}

		const slottedItems = this.shadowRoot?.querySelector('slot')?.assignedElements() ?? [];

		for (const slottedItem of slottedItems) {
			if (slottedItem instanceof DiscordInputText) {
				slottedItem.resetState();

				const shadowRootSlot = slottedItem.shadowRoot;

				const messageNeeded = shadowRootSlot?.querySelector('div.discord-text-input-message-needed-input');
				const messageNeededMinLength = shadowRootSlot?.querySelector('div.discord-text-input-message-needed-min-length');

				if (messageNeeded instanceof HTMLDivElement && messageNeeded.style.display) {
					messageNeeded.style.display = '';
					messageNeeded.style.opacity = '0';
				}

				if (messageNeededMinLength instanceof HTMLDivElement && messageNeededMinLength.style.display) {
					messageNeededMinLength.style.display = '';
					messageNeededMinLength.style.opacity = '0';
				}
			}
		}

		this.closeForm?.();
	}

	protected override render() {
		const defaultData: Pick<Profile, 'author'> = {
			author: this.author
		};

		const profileData: Profile = ((this.profile !== undefined && Reflect.get(profiles, this.profile)) as Profile) || {};
		const profile: Profile = { ...defaultData, ...profileData, avatar: this.resolveAvatar(profileData.avatar ?? this.avatar) };

		return html`
			<dialog
				${ref(this.dialogRef)}
				id="${ifDefined(this.modalId)}"
				class=${classMap({ 'discord-modal': true, 'discord-modal-light-theme': this.lightTheme })}
				@close=${this.onCloseDialog}
			>
				<div class="discord-modal-box">
					<form @submit=${this.handleFormSubmit}>
						<div aria-labelledby="discord-modal-title" role="dialog" tabindex="-1" aria-modal="true">
							<div class=${classMap({ 'discord-modal-root': true, 'discord-modal-root-light-theme': this.lightTheme })}>
								<div class="discord-modal-header-container">
									<div class="discord-modal-avatar">
										<img src="${ifDefined(profile.avatar)}" alt="${ifDefined(profile.author)}" />
									</div>
									<div
										id="discord-modal-title"
										class=${classMap({ 'discord-modal-title': true, 'discord-modal-title-light-theme': this.lightTheme })}
									>
										${this.modalTitle}
									</div>
									<button
										aria-label="Close"
										@click=${this.handleClickCloseIcon}
										type="button"
										class=${classMap({
											'discord-modal-close-button': true,
											'discord-modal-close-button-light-theme': this.lightTheme
										})}
									>
										<div class="discord-modal-close-button-content">${ModalClose()}</div>
									</button>
								</div>
								<div
									class=${classMap({
										'discord-modal-content-container': true,
										'discord-modal-content-container-light-theme': this.lightTheme
									})}
								>
									<div class="discord-modal-warning-container">
										<div class="discord-modal-warning-icon">
											${ModalWarning({
												class:
													'discord-modal-warning-icon-svg' +
													(this.lightTheme ? ' discord-modal-warning-icon-svg-light-theme' : '')
											})}
										</div>
										<div
											class=${classMap({
												'discord-modal-warning': true,
												'discord-modal-warning-light-theme': this.lightTheme
											})}
										>
											This form will be submitted to <strong class="discord-modal-warning-author">${profile.author}</strong>. Do
											not share passwords or other sensitive information.
										</div>
									</div>
									<div
										aria-hidden="true"
										style="position: absolute; pointer-events: none; min-height: 0px; min-width: 1px; flex: 0 0 auto; height: 0px;"
									></div>
								</div>
								<div class="discord-modal-slot-input-text">
									<slot></slot>
								</div>
								<div
									class=${classMap({
										'discord-modal-actions-container': true,
										'discord-modal-actions-container-light-theme': this.lightTheme
									})}
								>
									<button type="submit" class="discord-modal-button discord-modal-button-submit">
										<div class="discord-modal-button-content">Submit</div>
									</button>
									<button
										type="button"
										class=${classMap({
											'discord-modal-button': true,
											'discord-modal-button-cancel': true,
											'discord-modal-button-cancel-light-theme': this.lightTheme
										})}
										@click=${this.handleClickCloseIcon}
									>
										<div
											class=${classMap({
												'discord-modal-button-content': true,
												'discord-modal-button-content-light-theme': this.lightTheme
											})}
										>
											Cancel
										</div>
									</button>
								</div>
							</div>
						</div>
					</form>
				</div>
			</dialog>
		`;
	}

	private resolveAvatar(avatar: string | undefined): string {
		return avatar === undefined ? avatars.default : (avatars[avatar] ?? avatar ?? avatars.default);
	}
}

declare global {
	interface HTMLElementTagNameMap {
		'discord-modal': DiscordModal;
	}
}
