﻿/* ===============================================================================================================================================
   createThreadedText

   Description
   This script creates a threaded text. It can also be made from a mixture of point types and path objects.
   Vertical text is also supported.

   Usage
   1. Select two or more text and/or path objects, run this script from File > Scripts > Other Script...
      There is no need to convert to the area types in advance.
   2. Select an order of concatenation.
   3. Enter a value of the alignment position tolerance. (0 or higher number)
   4. To convert the paths to the texts, check the Convert Path to Type checkbox and select either the Area Type or Type on a Path.

   Notes
   The stacking order of objects and layers has no bearing on the order of concatenation.
   When converting the path to the area type, ignore anchor points with fewer than two anchor points.
   The units of the alignment position tolerance depend on the ruler units.
   In rare cases, the script may not work if you continue to use it.
   In this case, restart Illustrator and try again.

   Requirements
   Illustrator CC or higher

   Version
   1.1.0

   Homepage
   github.com/sky-chaser-high/adobe-illustrator-scripts

   License
   Released under the MIT license.
   https://opensource.org/licenses/mit-license.php
   =============================================================================================================================================== */

(function() {
    if (app.documents.length && isValidVersion()) main();
})();


function main() {
    var items = app.activeDocument.selection;
    var texts = getTextFrames(items);
    var shapes = getPathItems(items);
    if (texts.length + shapes.length < 2) return;

    var dialog = showDialog(texts, shapes);

    dialog.ok.onClick = function() {
        if (!dialog.shape.value && texts.length < 2) return dialog.close();
        var config = getConfiguration(dialog);
        app.executeMenuCommand('deselectall');
        convertPointTypeToAreaType(texts);
        if (dialog.shape.value) {
            if (dialog.area.value) convertPathToAreaType(shapes);
            if (dialog.path.value) convertPathToTypeOnAPath(shapes);
        }
        createThreadedText(config);
        dialog.close();
    }

    dialog.show();
}


function getConfiguration(dialog) {
    var value = Number(dialog.tolerance.text);
    if (isNaN(value)) value = 0;
    var tolerance = convertUnits(Math.abs(value) + dialog.units, 'pt');
    return {
        tolerance: round(tolerance),
        order: {
            row: dialog.state.rowLR,
            column: dialog.state.columnLR,
            left: dialog.state.LtoR,
            top: dialog.state.TtoB,
            rowRtoL: dialog.state.rowRL,
            columnRtoL: dialog.state.columnRL,
            right: dialog.state.RtoL,
            bottom: dialog.state.BtoT
        }
    };
}


function convertPointTypeToAreaType(texts) {
    for (var i = 0; i < texts.length; i++) {
        var text = texts[i];
        text.convertPointObjectToAreaObject();
        text.selected = true;
    }
}


function convertPathToAreaType(items) {
    for (var i = items.length - 1; i >= 0; i--) {
        var item = items[i];
        var points = item.pathPoints;
        if (points.length <= 2) continue;
        var layer = app.activeDocument.activeLayer;
        var text = layer.textFrames.areaText(item);
        // work around a bug
        text.selected = false;
        text.selected = true;
    }
}


function convertPathToTypeOnAPath(items) {
    for (var i = items.length - 1; i >= 0; i--) {
        var item = items[i];
        var layer = app.activeDocument.activeLayer;
        var text = layer.textFrames.pathText(item);
        // work around a bug
        text.selected = false;
        text.selected = true;
    }
}


function createThreadedText(config) {
    var texts = sortTexts(config);

    for (var i = 0; i < texts.length - 1; i++) {
        var text = texts[i];
        var next = texts[i + 1];

        if (!hasLinefeed(text.contents)) {
            insertLinefeed(text);
        }

        text.nextFrame = next;
    }
}


function insertLinefeed(text) {
    var ranges = text.textRanges;
    var last = text.insertionPoints[ranges.length];
    last.characters.add('\n');
}


function hasLinefeed(text) {
    var linefeed = /\n$|\r$|\r\n$|\u0003$/;
    return linefeed.test(text);
}


function sortTexts(config) {
    var items = app.activeDocument.selection;
    var texts = getTextFrames(items);
    if (texts.length < 2) return [];

    var order = config.order;
    var tolerance = config.tolerance;

    if (order.row) return sortRow(texts, tolerance);
    if (order.column) return sortColumn(texts, tolerance);
    if (order.rowRtoL) return sortRowRtoL(texts, tolerance);
    if (order.columnRtoL) return sortColumnRtoL(texts, tolerance);
    if (order.left) return texts.sort(orderLeftToRight);
    if (order.right) return texts.sort(orderRightToLeft);
    if (order.top) return texts.sort(orderTopToBottom);
    if (order.bottom) return texts.sort(orderBottomToTop);
}


function sortRow(texts, tolerance) {
    return texts.sort(function(a, b) {
        var distance = Math.abs(b.top - a.top);
        if (distance <= tolerance) {
            return a.left - b.left;
        }
        return b.top - a.top;
    });
}


function sortRowRtoL(texts, tolerance) {
    return texts.sort(function(a, b) {
        var distance = Math.abs(b.top - a.top);
        if (distance <= tolerance) {
            return b.left - a.left;
        }
        return b.top - a.top;
    });
}


function sortColumn(texts, tolerance) {
    return texts.sort(function(a, b) {
        var distance = Math.abs(a.left - b.left);
        if (distance <= tolerance) {
            return b.top - a.top;
        }
        return a.left - b.left;
    });
}


function sortColumnRtoL(texts, tolerance) {
    return texts.sort(function(a, b) {
        var distance = Math.abs(a.left - b.left);
        if (distance <= tolerance) {
            return b.top - a.top;
        }
        return b.left - a.left;
    });
}


function orderRow(a, b) {
    if (b.top - a.top == 0) {
        return a.left - b.left;
    }
    return b.top - a.top;
}


function orderRowRtoL(a, b) {
    if (b.top - a.top == 0) {
        return b.left - a.left;
    }
    return b.top - a.top;
}


function orderColumn(a, b) {
    if (a.left - b.left == 0) {
        return b.top - a.top;
    }
    return a.left - b.left;
}


function orderColumnRtoL(a, b) {
    if (a.left - b.left == 0) {
        return b.top - a.top;
    }
    return b.left - a.left;
}


function orderLeftToRight(a, b) {
    return a.left - b.left;
}


function orderRightToLeft(a, b) {
    return b.left - a.left;
}


function orderTopToBottom(a, b) {
    return b.top - a.top;
}


function orderBottomToTop(a, b) {
    return a.top - b.top;
}


function getTextFrames(items) {
    var texts = [];
    for (var i = 0; i < items.length; i++) {
        var item = items[i];
        if (item.typename == 'TextFrame') {
            texts.push(item);
        }
        if (item.typename == 'GroupItem') {
            texts = texts.concat(getTextFrames(item.pageItems));
        }
    }
    return texts;
}


function getPathItems(items) {
    var shapes = [];
    for (var i = 0; i < items.length; i++) {
        var item = items[i];
        if (item.typename == 'PathItem') {
            shapes.push(item);
        }
        if (item.typename == 'GroupItem') {
            shapes = shapes.concat(getPathItems(item.pageItems));
        }
    }
    return shapes;
}


function round(value) {
    var digits = 1000;
    return Math.round(value * digits) / digits;
}


function convertUnits(value, unit) {
    try {
        return Number(UnitValue(value).as(unit));
    }
    catch (err) {
        return Number(UnitValue('1pt').as('pt'));
    }
}


function getRulerUnits() {
    var unit = getUnitSymbol();
    if (!app.documents.length) return unit.pt;

    var document = app.activeDocument;
    var src = document.fullName;
    var ruler = document.rulerUnits;
    try {
        switch (ruler) {
            case RulerUnits.Pixels: return unit.px;
            case RulerUnits.Points: return unit.pt;
            case RulerUnits.Picas: return unit.pc;
            case RulerUnits.Inches: return unit.inch;
            case RulerUnits.Millimeters: return unit.mm;
            case RulerUnits.Centimeters: return unit.cm;

            case RulerUnits.Feet: return unit.ft;
            case RulerUnits.Yards: return unit.yd;
            case RulerUnits.Meters: return unit.meter;
        }
    }
    catch (err) {
        switch (xmpRulerUnits(src)) {
            case 'Feet': return unit.ft;
            case 'Yards': return unit.yd;
            case 'Meters': return unit.meter;
        }
    }
    return unit.pt;
}


function xmpRulerUnits(src) {
    if (!ExternalObject.AdobeXMPScript) {
        ExternalObject.AdobeXMPScript = new ExternalObject('lib:AdobeXMPScript');
    }
    var xmpFile = new XMPFile(src.fsName, XMPConst.FILE_UNKNOWN, XMPConst.OPEN_FOR_READ);
    var xmpPackets = xmpFile.getXMP();
    var xmp = new XMPMeta(xmpPackets.serialize());

    var namespace = 'http://ns.adobe.com/xap/1.0/t/pg/';
    var prop = 'xmpTPg:MaxPageSize';
    var unit = prop + '/stDim:unit';

    var ruler = xmp.getProperty(namespace, unit).value;
    return ruler;
}


function getUnitSymbol() {
    return {
        px: 'px',
        pt: 'pt',
        pc: 'pc',
        inch: 'in',
        ft: 'ft',
        yd: 'yd',
        mm: 'mm',
        cm: 'cm',
        meter: 'm'
    };
}


function isValidVersion() {
    var cc = 17;
    var aiVersion = parseInt(app.version);
    if (aiVersion < cc) return false;
    return true;
}


function showDialog(texts, shapes) {
    $.localize = true;
    var ui = localizeUI();
    var icon = getUIIcon();
    var ruler = getRulerUnits();

    var state = {
        rowLR: true,
        columnLR: false,
        LtoR: false,
        TtoB: false,
        rowRL: false,
        columnRL: false,
        RtoL: false,
        BtoT: false
    };

    var dialog = new Window('dialog');
    dialog.text = ui.title;
    dialog.orientation = 'column';
    dialog.alignChildren = ['fill', 'top'];
    dialog.spacing = 10;
    dialog.margins = 16;

    var panel1 = dialog.add('panel', undefined, undefined, { name: 'panel1' });
    panel1.text = ui.order;
    panel1.orientation = 'column';
    panel1.alignChildren = ['fill', 'top'];
    panel1.spacing = 10;
    panel1.margins = 10;

    var group1 = panel1.add('group', undefined, { name: 'group1' });
    group1.orientation = 'column';
    group1.alignChildren = ['fill', 'center'];
    group1.spacing = 10;
    group1.margins = [0, 6, 0, 0];

    var group2 = group1.add('group', undefined, { name: 'group2' });
    group2.orientation = 'row';
    group2.alignChildren = ['fill', 'center'];
    group2.spacing = 10;
    group2.margins = 0;

    var image1 = group2.add('image', undefined, File.decode(icon.checked.rowLR), { name: 'image1' });
    var image2 = group2.add('image', undefined, File.decode(icon.normal.columnLR), { name: 'image2' });
    var image3 = group2.add('image', undefined, File.decode(icon.normal.LtoR), { name: 'image3' });
    var image4 = group2.add('image', undefined, File.decode(icon.normal.TtoB), { name: 'image4' });

    var group3 = group1.add('group', undefined, { name: 'group3' });
    group3.orientation = 'row';
    group3.alignChildren = ['fill', 'center'];
    group3.spacing = 10;
    group3.margins = 0;

    var image5 = group3.add('image', undefined, File.decode(icon.normal.rowRL), { name: 'image5' });
    var image6 = group3.add('image', undefined, File.decode(icon.normal.columnRL), { name: 'image6' });
    var image7 = group3.add('image', undefined, File.decode(icon.normal.RtoL), { name: 'image7' });
    var image8 = group3.add('image', undefined, File.decode(icon.normal.BtoT), { name: 'image8' });

    var panel2 = dialog.add('panel', undefined, undefined, { name: 'panel2' });
    panel2.text = ui.option;
    panel2.orientation = 'column';
    panel2.alignChildren = ['left', 'top'];
    panel2.spacing = 10;
    panel2.margins = 10;

    var group4 = panel2.add('group', undefined, { name: 'group4' });
    group4.orientation = 'row';
    group4.alignChildren = ['left', 'center'];
    group4.spacing = 10;
    group4.margins = [0, 8, 0, 0];

    var statictext1 = group4.add('statictext', undefined, undefined, { name: 'statictext1' });
    statictext1.text = ui.tolerance;
    statictext1.preferredSize.height = 20;

    var edittext1 = group4.add('edittext', undefined, undefined, { name: 'edittext1' });
    edittext1.text = '1';
    edittext1.preferredSize.width = 60;
    edittext1.active = true;

    var statictext2 = group4.add('statictext', undefined, undefined, { name: 'statictext2' });
    statictext2.text = ruler;

    var divider1 = panel2.add('panel', undefined, undefined, { name: 'divider1' });
    divider1.alignment = 'fill';

    var group5 = panel2.add('group', undefined, { name: 'group5' });
    group5.orientation = 'column';
    group5.alignChildren = ['left', 'center'];
    group5.spacing = 10;
    group5.margins = [0, 4, 0, 0];
    if (!shapes.length) group5.enabled = false;

    var checkbox1 = group5.add('checkbox', undefined, undefined, { name: 'checkbox1' });
    checkbox1.text = ui.shape;
    if (!texts.length) checkbox1.value = true;

    var group6 = group5.add('group', undefined, { name: 'group6' });
    group6.orientation = 'row';
    group6.alignChildren = ['left', 'center'];
    group6.spacing = 10;
    group6.margins = [18, 0, 0, 0];
    group6.enabled = false;
    if (!texts.length) group6.enabled = true;

    var radiobutton1 = group6.add('radiobutton', undefined, undefined, { name: 'radiobutton1' });
    radiobutton1.text = ui.area;
    radiobutton1.value = true;

    var radiobutton2 = group6.add('radiobutton', undefined, undefined, { name: 'radiobutton2' });
    radiobutton2.text = ui.path;

    var group7 = dialog.add('group', undefined, { name: 'group7' });
    group7.orientation = 'row';
    group7.alignChildren = ['right', 'center'];
    group7.spacing = 10;
    group7.margins = 0;

    var button1 = group7.add('button', undefined, undefined, { name: 'button1' });
    button1.text = ui.cancel;
    button1.preferredSize.width = 90;

    var button2 = group7.add('button', undefined, undefined, { name: 'button2' });
    button2.text = ui.ok;
    button2.preferredSize.width = 90;

    image1.addEventListener('mouseover', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.rollover.rowLR);
    });
    image1.addEventListener('mousedown', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.pressed.rowLR);
    });
    image1.addEventListener('mouseup', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.rowLR) {
            this.image = ScriptUI.newImage(icon.checked.rowLR);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.rowLR);
        }
    });
    image1.addEventListener('mouseout', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.rowLR) {
            this.image = ScriptUI.newImage(icon.checked.rowLR);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.rowLR);
        }
    });
    image1.addEventListener('click', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.checked.rowLR);
        image2.image = ScriptUI.newImage(image2.image);
        image2.image = ScriptUI.newImage(icon.normal.columnLR);
        image3.image = ScriptUI.newImage(image3.image);
        image3.image = ScriptUI.newImage(icon.normal.LtoR);
        image4.image = ScriptUI.newImage(image4.image);
        image4.image = ScriptUI.newImage(icon.normal.TtoB);
        image5.image = ScriptUI.newImage(image5.image);
        image5.image = ScriptUI.newImage(icon.normal.rowRL);
        image6.image = ScriptUI.newImage(image6.image);
        image6.image = ScriptUI.newImage(icon.normal.columnRL);
        image7.image = ScriptUI.newImage(image7.image);
        image7.image = ScriptUI.newImage(icon.normal.RtoL);
        image8.image = ScriptUI.newImage(image8.image);
        image8.image = ScriptUI.newImage(icon.normal.BtoT);
        group4.enabled = true;
        edittext1.active = false;
        edittext1.active = true;
        state.rowLR = true;
        state.columnLR = false;
        state.LtoR = false;
        state.TtoB = false;
        state.rowRL = false;
        state.columnRL = false;
        state.RtoL = false;
        state.BtoT = false;
    });

    image2.addEventListener('mouseover', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.rollover.columnLR);
    });
    image2.addEventListener('mousedown', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.pressed.columnLR);
    });
    image2.addEventListener('mouseup', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.columnLR) {
            this.image = ScriptUI.newImage(icon.checked.columnLR);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.columnLR);
        }
    });
    image2.addEventListener('mouseout', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.columnLR) {
            this.image = ScriptUI.newImage(icon.checked.columnLR);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.columnLR);
        }
    });
    image2.addEventListener('click', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.checked.columnLR);
        image1.image = ScriptUI.newImage(image1.image);
        image1.image = ScriptUI.newImage(icon.normal.rowLR);
        image3.image = ScriptUI.newImage(image3.image);
        image3.image = ScriptUI.newImage(icon.normal.LtoR);
        image4.image = ScriptUI.newImage(image4.image);
        image4.image = ScriptUI.newImage(icon.normal.TtoB);
        image5.image = ScriptUI.newImage(image5.image);
        image5.image = ScriptUI.newImage(icon.normal.rowRL);
        image6.image = ScriptUI.newImage(image6.image);
        image6.image = ScriptUI.newImage(icon.normal.columnRL);
        image7.image = ScriptUI.newImage(image7.image);
        image7.image = ScriptUI.newImage(icon.normal.RtoL);
        image8.image = ScriptUI.newImage(image8.image);
        image8.image = ScriptUI.newImage(icon.normal.BtoT);
        group4.enabled = true;
        edittext1.active = false;
        edittext1.active = true;
        state.rowLR = false;
        state.columnLR = true;
        state.LtoR = false;
        state.TtoB = false;
        state.rowRL = false;
        state.columnRL = false;
        state.RtoL = false;
        state.BtoT = false;
    });

    image3.addEventListener('mouseover', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.rollover.LtoR);
    });
    image3.addEventListener('mousedown', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.pressed.LtoR);
    });
    image3.addEventListener('mouseup', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.LtoR) {
            this.image = ScriptUI.newImage(icon.checked.LtoR);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.LtoR);
        }
    });
    image3.addEventListener('mouseout', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.LtoR) {
            this.image = ScriptUI.newImage(icon.checked.LtoR);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.LtoR);
        }
    });
    image3.addEventListener('click', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.checked.LtoR);
        image1.image = ScriptUI.newImage(image1.image);
        image1.image = ScriptUI.newImage(icon.normal.rowLR);
        image2.image = ScriptUI.newImage(image2.image);
        image2.image = ScriptUI.newImage(icon.normal.columnLR);
        image4.image = ScriptUI.newImage(image4.image);
        image4.image = ScriptUI.newImage(icon.normal.TtoB);
        image5.image = ScriptUI.newImage(image5.image);
        image5.image = ScriptUI.newImage(icon.normal.rowRL);
        image6.image = ScriptUI.newImage(image6.image);
        image6.image = ScriptUI.newImage(icon.normal.columnRL);
        image7.image = ScriptUI.newImage(image7.image);
        image7.image = ScriptUI.newImage(icon.normal.RtoL);
        image8.image = ScriptUI.newImage(image8.image);
        image8.image = ScriptUI.newImage(icon.normal.BtoT);
        group4.enabled = false;
        state.rowLR = false;
        state.columnLR = false;
        state.LtoR = true;
        state.TtoB = false;
        state.rowRL = false;
        state.columnRL = false;
        state.RtoL = false;
        state.BtoT = false;
    });

    image4.addEventListener('mouseover', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.rollover.TtoB);
    });
    image4.addEventListener('mousedown', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.pressed.TtoB);
    });
    image4.addEventListener('mouseup', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.TtoB) {
            this.image = ScriptUI.newImage(icon.checked.TtoB);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.TtoB);
        }
    });
    image4.addEventListener('mouseout', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.TtoB) {
            this.image = ScriptUI.newImage(icon.checked.TtoB);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.TtoB);
        }
    });
    image4.addEventListener('click', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.checked.TtoB);
        image1.image = ScriptUI.newImage(image1.image);
        image1.image = ScriptUI.newImage(icon.normal.rowLR);
        image2.image = ScriptUI.newImage(image2.image);
        image2.image = ScriptUI.newImage(icon.normal.columnLR);
        image3.image = ScriptUI.newImage(image3.image);
        image3.image = ScriptUI.newImage(icon.normal.LtoR);
        image5.image = ScriptUI.newImage(image5.image);
        image5.image = ScriptUI.newImage(icon.normal.rowRL);
        image6.image = ScriptUI.newImage(image6.image);
        image6.image = ScriptUI.newImage(icon.normal.columnRL);
        image7.image = ScriptUI.newImage(image7.image);
        image7.image = ScriptUI.newImage(icon.normal.RtoL);
        image8.image = ScriptUI.newImage(image8.image);
        image8.image = ScriptUI.newImage(icon.normal.BtoT);
        group4.enabled = false;
        state.rowLR = false;
        state.columnLR = false;
        state.LtoR = false;
        state.TtoB = true;
        state.rowRL = false;
        state.columnRL = false;
        state.RtoL = false;
        state.BtoT = false;
    });

    image5.addEventListener('mouseover', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.rollover.rowRL);
    });
    image5.addEventListener('mousedown', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.pressed.rowRL);
    });
    image5.addEventListener('mouseup', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.rowRL) {
            this.image = ScriptUI.newImage(icon.checked.rowRL);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.rowRL);
        }
    });
    image5.addEventListener('mouseout', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.rowRL) {
            this.image = ScriptUI.newImage(icon.checked.rowRL);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.rowRL);
        }
    });
    image5.addEventListener('click', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.checked.rowRL);
        image1.image = ScriptUI.newImage(image1.image);
        image1.image = ScriptUI.newImage(icon.normal.rowLR);
        image2.image = ScriptUI.newImage(image2.image);
        image2.image = ScriptUI.newImage(icon.normal.columnLR);
        image3.image = ScriptUI.newImage(image3.image);
        image3.image = ScriptUI.newImage(icon.normal.LtoR);
        image4.image = ScriptUI.newImage(image4.image);
        image4.image = ScriptUI.newImage(icon.normal.TtoB);
        image6.image = ScriptUI.newImage(image6.image);
        image6.image = ScriptUI.newImage(icon.normal.columnRL);
        image7.image = ScriptUI.newImage(image7.image);
        image7.image = ScriptUI.newImage(icon.normal.RtoL);
        image8.image = ScriptUI.newImage(image8.image);
        image8.image = ScriptUI.newImage(icon.normal.BtoT);
        group4.enabled = true;
        edittext1.active = false;
        edittext1.active = true;
        state.rowLR = false;
        state.columnLR = false;
        state.LtoR = false;
        state.TtoB = false;
        state.rowRL = true;
        state.columnRL = false;
        state.RtoL = false;
        state.BtoT = false;
    });

    image6.addEventListener('mouseover', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.rollover.columnRL);
    });
    image6.addEventListener('mousedown', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.pressed.columnRL);
    });
    image6.addEventListener('mouseup', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.columnRL) {
            this.image = ScriptUI.newImage(icon.checked.columnRL);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.columnRL);
        }
    });
    image6.addEventListener('mouseout', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.columnRL) {
            this.image = ScriptUI.newImage(icon.checked.columnRL);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.columnRL);
        }
    });
    image6.addEventListener('click', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.checked.columnRL);
        image1.image = ScriptUI.newImage(image1.image);
        image1.image = ScriptUI.newImage(icon.normal.rowLR);
        image2.image = ScriptUI.newImage(image2.image);
        image2.image = ScriptUI.newImage(icon.normal.columnLR);
        image3.image = ScriptUI.newImage(image3.image);
        image3.image = ScriptUI.newImage(icon.normal.LtoR);
        image4.image = ScriptUI.newImage(image4.image);
        image4.image = ScriptUI.newImage(icon.normal.TtoB);
        image5.image = ScriptUI.newImage(image5.image);
        image5.image = ScriptUI.newImage(icon.normal.rowRL);
        image7.image = ScriptUI.newImage(image7.image);
        image7.image = ScriptUI.newImage(icon.normal.RtoL);
        image8.image = ScriptUI.newImage(image8.image);
        image8.image = ScriptUI.newImage(icon.normal.BtoT);
        group4.enabled = true;
        edittext1.active = false;
        edittext1.active = true;
        state.rowLR = false;
        state.columnLR = false;
        state.LtoR = false;
        state.TtoB = false;
        state.rowRL = false;
        state.columnRL = true;
        state.RtoL = false;
        state.BtoT = false;
    });

    image7.addEventListener('mouseover', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.rollover.RtoL);
    });
    image7.addEventListener('mousedown', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.pressed.RtoL);
    });
    image7.addEventListener('mouseup', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.RtoL) {
            this.image = ScriptUI.newImage(icon.checked.RtoL);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.RtoL);
        }
    });
    image7.addEventListener('mouseout', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.RtoL) {
            this.image = ScriptUI.newImage(icon.checked.RtoL);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.RtoL);
        }
    });
    image7.addEventListener('click', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.checked.RtoL);
        image1.image = ScriptUI.newImage(image1.image);
        image1.image = ScriptUI.newImage(icon.normal.rowLR);
        image2.image = ScriptUI.newImage(image2.image);
        image2.image = ScriptUI.newImage(icon.normal.columnLR);
        image3.image = ScriptUI.newImage(image3.image);
        image3.image = ScriptUI.newImage(icon.normal.LtoR);
        image4.image = ScriptUI.newImage(image4.image);
        image4.image = ScriptUI.newImage(icon.normal.TtoB);
        image5.image = ScriptUI.newImage(image5.image);
        image5.image = ScriptUI.newImage(icon.normal.rowRL);
        image6.image = ScriptUI.newImage(image6.image);
        image6.image = ScriptUI.newImage(icon.normal.columnRL);
        image8.image = ScriptUI.newImage(image8.image);
        image8.image = ScriptUI.newImage(icon.normal.BtoT);
        group4.enabled = false;
        state.rowLR = false;
        state.columnLR = false;
        state.LtoR = false;
        state.TtoB = false;
        state.rowRL = false;
        state.columnRL = false;
        state.RtoL = true;
        state.BtoT = false;
    });

    image8.addEventListener('mouseover', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.rollover.BtoT);
    });
    image8.addEventListener('mousedown', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.pressed.BtoT);
    });
    image8.addEventListener('mouseup', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.BtoT) {
            this.image = ScriptUI.newImage(icon.checked.BtoT);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.BtoT);
        }
    });
    image8.addEventListener('mouseout', function() {
        this.image = ScriptUI.newImage(this.image);
        if (state.BtoT) {
            this.image = ScriptUI.newImage(icon.checked.BtoT);
        }
        else {
            this.image = ScriptUI.newImage(icon.normal.BtoT);
        }
    });
    image8.addEventListener('click', function() {
        this.image = ScriptUI.newImage(this.image);
        this.image = ScriptUI.newImage(icon.checked.BtoT);
        image1.image = ScriptUI.newImage(image1.image);
        image1.image = ScriptUI.newImage(icon.normal.rowLR);
        image2.image = ScriptUI.newImage(image2.image);
        image2.image = ScriptUI.newImage(icon.normal.columnLR);
        image3.image = ScriptUI.newImage(image3.image);
        image3.image = ScriptUI.newImage(icon.normal.LtoR);
        image4.image = ScriptUI.newImage(image4.image);
        image4.image = ScriptUI.newImage(icon.normal.TtoB);
        image5.image = ScriptUI.newImage(image5.image);
        image5.image = ScriptUI.newImage(icon.normal.rowRL);
        image6.image = ScriptUI.newImage(image6.image);
        image6.image = ScriptUI.newImage(icon.normal.columnRL);
        image7.image = ScriptUI.newImage(image7.image);
        image7.image = ScriptUI.newImage(icon.normal.RtoL);
        group4.enabled = false;
        state.rowLR = false;
        state.columnLR = false;
        state.LtoR = false;
        state.TtoB = false;
        state.rowRL = false;
        state.columnRL = false;
        state.RtoL = false;
        state.BtoT = true;
    });

    statictext1.addEventListener('click', function() {
        edittext1.active = false;
        edittext1.active = true;
    });

    edittext1.addEventListener('keydown', function(event) {
        var value = Number(this.text);
        if (isNaN(value)) value = 0;
        var keyboard = ScriptUI.environment.keyboardState;
        var step = keyboard.shiftKey ? 5 : 1;
        var tolerance;
        if (event.keyName == 'Up') {
            tolerance = value + step;
            this.text = tolerance;
            event.preventDefault();
        }
        if (event.keyName == 'Down') {
            tolerance = value - step;
            if (tolerance < 0) tolerance = 0;
            this.text = tolerance;
            event.preventDefault();
        }
    });

    checkbox1.onClick = function() {
        group6.enabled = (checkbox1.value) ? true : false;
    }

    button1.onClick = function() {
        dialog.close();
    }

    dialog.state = state;
    dialog.tolerance = edittext1;
    dialog.shape = checkbox1;
    dialog.area = radiobutton1;
    dialog.path = radiobutton2;
    dialog.units = ruler;
    dialog.ok = button2;
    return dialog;
}


function getUIIcon() {
    var lightest = {
        normal: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01aIDATH%C2%89%C3%AD%C2%96%C2%AD%C2%8E%C3%82%40%14%C2%85%C2%BF%C2%925%C3%A36%60%C2%AB%C2%B1%C2%B5%C3%A36%60k%10%C2%8C%C2%9D%07%C2%A8%C3%A3%1DH-%C2%8E%04%5B%C2%8DX3%16%C2%B6%C2%AE%C3%8FP%C2%8Dm2%C2%AErW-aC%7F%C2%A6%C3%BC%C2%B4%2B8%C3%B2%C3%A6%C3%9E9%C3%B7%C2%9C%C2%99%C2%9C%C2%8Cg%C2%AD%7D%07%3E%C2%81%0F%C3%BAC%0A%2C%3Ck%C3%ADW%C3%8F%C3%84%C3%A7%05%3Ck%C3%AD%C3%B7%00%C3%84%00%C2%8C%C2%86%22~%C2%91%0F%C2%86%C2%B7%C2%AAb%14EW%C2%B5%C3%ADv%C3%AB%7C%C2%A8%C3%AB%C3%BC%C3%BFS%5E%C2%85*5%C2%BD%C2%91%C3%9Fk%7B%15%1En%7B%C2%96e%C3%8E%C2%BD%0FK%C2%B8%C2%A2(H%C2%92%C2%84%C2%B2%2CY.%C2%97L%C2%A7%C3%93%C3%96%19g%C3%9B%C2%9Bp8%1C0%C3%860%C2%9F%C3%8F%09%C3%83%C3%90y%C3%AE.%C3%A5%C2%97j%C2%B5%C3%96%C3%B8%C2%BE%C3%9Fi%C3%BEf%C3%A5%C3%86%18%C2%8C1%C2%84a%C3%98I%C3%AD%25%3A%2B%3F%C2%9DN%24I%C2%82%10%02%C2%AD5%C2%93%C3%89%C3%A4%26b%C3%A8%C3%B8%C3%9A%C2%8D1l6%1B%C2%A4%C2%94%C2%ACV%C2%ABZ%C3%A28%C2%8E)%C2%8A%C2%A2%C3%B5%3C\'%C3%A5y%C2%9E%C2%B3%C3%9F%C3%AF%C2%9D%C3%95FQt%C3%AE%0D%C2%82%C2%A0%C2%B6%C3%8F9%C3%9B%C2%B5%C3%96H)%C3%9B%C3%B6%3C%C2%A3%2CKv%C2%BB%1D%C2%B3%C3%99%0C%C2%A5Te%C2%8F%C2%B3%C3%AD%5D%C2%88%2Fq%3C%1Ek%C2%AF%C2%A1%C3%92%C3%B6g%C3%A4%C3%B8x%3Cf%C2%BD%5E%C3%BF%C2%A9%C3%B5%C2%92%C3%AD%C2%BE%C3%AF%C2%A3%C2%B5%C2%BE%C3%AA%7BH%C3%825AJ%C2%89R%0A!%C2%84%1By%17%C2%95u%10B%C2%A0%C2%94j%7C%2BOQ%C3%BEks%5B%C3%9C%C2%BE%C3%BE%C3%AD%C2%83%C2%91%C2%A7%03q%C2%A7%23%601%C3%80%02)%C2%B0%C3%B8%011%26%C2%8B%1C%C3%B2G%C2%9D%C2%AC%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01XIDATH%C2%89%C3%A5%C2%97!%C2%8F%C2%830%18%C2%86%1F%C2%963u%C2%97%60%C2%AB%C2%B1X%C3%9C%C2%85Z%0C%06%C2%8B%25A%C2%81%C3%9D%0F%C3%80%C3%8E%C3%AD%07%C3%B0%0B%C3%8E%60%C3%89%C3%A1%C3%B8%0D%C3%93X%C2%92%C2%BA%C3%89%3B%C2%B5%25%C2%BB%0Dn%40%03%C3%89%C2%AD%C3%B2%C3%AB%C3%B7%C3%A6i%C3%9F%C3%B4%C3%BB%C3%9AZZ%C3%ABw%C3%A0%13%C3%B8%60%C2%BD%C3%91%00%C2%A1%C2%A5%C2%B5%C3%BEZ%19%7C%5D%C2%80%C2%A5%C2%B5%C3%BE%C3%9E%00%0C%C3%80n%2B%C3%B0%C3%A6%C3%B0%C2%B7%C2%B1%C3%894M%C3%AFb%C3%87%C3%A3%C3%91X%C3%BE%C3%AB%C3%9A%C3%BE%3F%C3%A0%C2%A7%C3%93i%C2%B2%C3%A6%C2%A9%3AO%C3%93t%C3%B4%C3%A0%00%14E%C2%81R%0A%C3%8F%C3%B3%C2%9E%C3%8A%07C%3B%C2%AF%C2%AA%0A!%04%C2%9E%C3%A7M%C3%92-%C2%86w%5DG%5D%C3%97%C3%84q%3CY%C2%BB%18%5E%C2%96%25A%10%60%C3%9B%C3%B6%C2%BA%C3%B0%C2%8B%C3%9DJ%C2%A9Y%C3%BA%C3%99%C3%B0%C2%BE%C3%AF%C2%A9%C2%AAj%C2%96%C3%9D%C2%8B%C3%A1K%C3%AC%5E%04%C2%AF%C3%AB%C2%9A%C3%B3%C3%B9L%10%04%C2%B3%C3%81%C2%B3%C3%A0%26%C3%AC%C2%9E%0D%2F%C3%8B%12%C2%A5%14R%C3%8Au%C3%A1m%C3%9B%1A%C2%B1%C3%BB2F%C3%9B%C3%AB%C2%A3%C3%BB9%C3%8B2%1C%C3%87%C2%B9%C2%89%15EA%C3%97u%C2%83%10)%25%C3%BB%C3%BD%C3%BE.%3E%C3%99%C3%B6%C3%9F%60%C2%80%24I%10B%3C%C3%8C%17B%C2%90%24%C3%89%C3%839%23%C2%BD%C3%9D%C2%B6%C3%AD%C3%81%03%18%C3%87%C3%B1%609%1A%C2%BBR%5D%C3%97%C3%85%C3%B7%C3%BD%C2%9B%C2%98%C3%AF%C3%BB%C2%B8%C2%AE%3B%C2%A81%C3%BA%C2%98%C2%88%C2%A2%C3%A8Z%05RJ%C2%A2(%1A%C3%8D7%C3%BEn%C3%AF%C3%BB%C2%9E%C3%83%C3%A1%40%C2%9E%C3%A7%7Fv%C2%BF%C3%97%C3%BE44%1B%C2%B1%C2%9B%1D%10n%C2%B0%C2%80%06%08%7F%00%C2%BB%1Ex%C3%A5%C2%A5O%C2%82%C2%86%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%A0IDATH%C2%89%C3%AD%C2%96%C2%BD%C2%89%C3%83%40%10%C2%85%3F%19\'%C2%9B%1D%C2%8A%C2%B7%C2%90%C3%8D%0Eu%C2%A0dR%15%C2%A0%3A%C3%84%C2%96%20P%17%C3%AE%40V%09%C3%AAA%C2%A9%602%C2%85w%C3%91%1D%C2%9C%C2%BD%16%C2%92%0Fy1%C3%AC%0B%C2%87%07%C3%9F%C2%9B%1F%C3%98%C3%8DT%C3%B5%03%C2%B8%00%C2%9F%C2%BCN%03Pf%C2%AAz%7D1%C3%B87%40%C2%A6%C2%AA_%11%C3%80%00%C2%9Cb%C2%81%13%3C%C3%81%13%C3%BC%C2%BD%C3%A1%C3%9E%7B%C3%A6y%C2%8E%03%C2%9F%C2%A6%09%C3%AF%3D%C3%A38%C2%AE%C3%BA%C3%8E%C2%A1b%5D%C3%97w%C2%B5%C2%B6mw%05X%C2%96%C2%85%C2%AE%C3%AB(%C2%8A%02%11%09z%0E%C3%9Fy%C3%9F%C3%B7%0F%C3%97%10%7CXB%C2%9D%C3%BFWy%C2%9E%C3%934%C3%8D%C2%9FZp%C3%AC!%C3%AD%19%C3%BBmxk-UU%C3%9D%C3%B96%C3%83%C2%9F%C2%95s%0E%11%C3%81%18%C2%B3%0D%C2%BE%C3%B7%C2%B8B2%C3%86%20%228%C3%A7%1Ez%0E%C3%A9%C3%BCg%C3%8C%C3%96%C3%9AU_%C3%BA%C3%89%24x%C2%82\'%C3%B8%C3%A1%C3%B0!%12%7B8%01e%C2%84%00%03P~%03%C2%A5%C2%97BM%C2%B2%C2%8D%C3%A6%15%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%A0IDATH%C2%89%C3%AD%C2%97%C2%BDm%C3%830%10F%C2%9F%0C7%C3%AC%02%C2%B1%C3%A5%0E%1C!%C3%A0%06j%C3%98%C2%B2%3D%C2%80%23h%00%C2%8D%C2%A092B%C2%B4%C2%8B%06%C2%B8%0D%C2%92%22H%10%C3%832%60%C3%92%C2%88%09D%C2%BA%C3%B2%0E%C3%BC%C3%9E%C3%BD%C2%91%00%3BU%7D%01%C3%9E%C2%80W%C2%9Eg%0B0t%C2%AA%C3%BA%C3%BEd%C3%B0O%02%C2%9D%C2%AA~4%00%03pj%05n%0E%3F%C3%97%1C%C3%8A9_%C3%B9%C3%A6y.%C3%96%C3%99o%C3%9B%C3%B7%0B%7F%C3%A8%C2%9E%C3%A7%C2%9C%C2%AB%16%C3%AD%C3%9B%C3%B6%C3%9B%C3%B6%03~%C3%80%0F%C3%B8%C3%BF%C2%87%17%3D%C2%AF%C3%934%C2%B1%C2%AE%C3%AB%C3%8D%C2%B8s%C2%8Eq%1C%C3%AF%C2%86%17U.%22%18c6c%C3%86%18D%C2%A4D%C2%AE%0Cn%C2%AD%25%C2%A5%C2%B4%19K)a%C2%AD%C3%BD%3B8%C2%80%C3%B7%C2%9E%10%C3%82%C2%85%2F%C2%84%C2%80%C3%B7%C2%BET%C2%AAn%C3%A1b%C2%8C8%C3%A7%C2%80%C2%AF9%C3%87%18kd%C3%AA%C2%B7%5DD%C3%A8%C3%BB%C2%BEx%C3%8E%C2%BFm%C3%9F%C2%9F%C2%86%C2%A5%11%7B9%01C%C2%83%04%16%60%C3%B8%04%1Fa5%C2%B7%C3%91%C3%AE%C2%85%0F%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01XIDATH%C2%89%C3%AD%C2%96%C2%A1r%021%10%C2%86%3F%C2%98%1A%5Cgb%C2%A3y%C2%86%C2%B8%C3%8E%60cx%C2%82h%06%C3%87K%60%C3%8F%C3%B1%00k%C3%8F%C3%94%C3%84B%C3%8F%C3%85b%C3%91g%C3%83%C3%84!%5B%C3%85M%3Bw%03%01%C3%9A%C2%BB%0A~w%7BI%C2%BE%C3%AC%C3%8E%C3%A6OF)%C2%A5W%C3%A0%1Dx%C2%A3%3FU%C3%80%7C%C2%94R%C3%BA%C3%A8%19%C3%9Cl%60%C2%94R%C3%BA%1C%00%0C%C3%80x(%C3%B0%13%3E%C2%98%5E%C2%BA%C2%82%C3%8B%C3%A5%C2%B2%15%C3%9Bl6%C3%99%C2%8B%C3%A6%C3%8E%C3%BF%7F%C2%99w%C2%A9%2B%C2%9B%C3%9E%C3%A0%C2%8F%C2%96%C2%BDK%C3%99e%0F!d%C3%83s%C2%95%C3%A5p%C2%87%C3%83%01%11A)%C2%85s%0E%C2%A5T%7F%C3%B0%C2%B3%C2%BC%C3%B7l%C2%B7%5B%C2%AC%C2%B5%C3%8Cf%C2%B3~%C3%A1%00u%5D%23%22L%26%C2%93%C2%87%C2%ABp%C3%B7%C3%85%C3%A2%C2%BD%C3%87%7B%C2%8F%C2%B5%16km%C2%BFp%C2%80%18%23%22%C3%82%C3%A9t%C3%829%C2%87%C3%96%C3%BA%C2%A6%C3%B9W%C2%BB%3D%C3%86%C3%88z%C2%BD%C3%AE%C3%BC%C2%A7%C2%94b%C2%B5Za%C2%8C%C2%A1(%0A%C2%BC%C3%B77%C3%81%2Ff%C2%BE%C3%9F%C3%AF%C2%9B%C3%8C%C2%AE%C2%9D%C3%B3s%15b%C2%8C8%C3%A7%C2%98N%C2%A7%C3%B7%C3%83%C3%8B%C2%B2d%C2%B7%C3%9B5%C3%9F%C2%B9%26%13B%40DZ%C3%B1%2Co%3F%C2%97%C3%B9%3B%C3%B8%16%19c%C2%B2%C3%87%C2%B6%C3%AC%C2%B5(%0A%C2%8E%C3%87ck%60%2F%C3%9E%C2%BEX%2C%10%11%C3%AA%C2%BA%C3%BE%11%C3%AF%C3%85%C3%9B%C2%B5%C3%96M%07%C3%BF%C2%B5.v%7B%08%C2%81%C2%B2%2C%C2%B3%C2%BA%C3%BD%1E%5D%C2%BCR%C2%8D1h%C2%AD%3B%C2%BB%C3%B77%C3%B4%7C%C2%B7%0F%06%C2%AF%06bWc%60%3E%C3%80%06*%60%C3%BE%05U%C2%BF%C2%ADF%C2%B9%C3%A8%C3%BC%C3%B0%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01aIDATH%C2%89%C3%A5%C3%96!o%C2%83%40%14%C3%80%C3%B1%7F%C2%9B%C2%99sK%C3%8E%C2%9E%C3%86b%C3%AB%16%C2%B0g0%C3%98%C2%B3%24%C2%B8%C3%9A%C3%AA%C2%A6%C2%B6U%7C%C2%80~%C2%82%C2%89b%C3%89p%7C%064%C3%B6%C2%92s%C2%95%C2%9BX%C2%B2%C2%A4i%C3%97A%C2%B9A%C2%96%C2%9D%7C%C2%BC%C2%97%C3%9F%C3%B1%C3%88%C2%BDc%C3%A1%C2%9C%7B%06%5E%C2%81%17%C2%A6%5B5%C2%90%2C%C2%9Cso%13%C3%83_%1BX8%C3%A7%C3%9Eg%C2%80%01X%C3%8E%05%C3%8F%C2%8E%3F%C3%9D%7B%C2%98%C3%A7%C3%B9U%C2%AC(%0Ao%C3%B9%C3%BF%C2%B7%C3%AD%7F%0Bo%C3%9B%C3%96%1B%C3%9E%C3%AB%C2%9C%C3%A7yNQ%144M%C3%83%C3%A9tb%C2%BB%C3%9D%C3%B6%C3%8A%C3%BFi%0Dz%C3%B3%C3%95j%C2%85%C2%94%C2%92%C2%B2%2C%C2%87%C2%94%C3%B9%C3%81%01%C2%8C1TUE%C3%97u%C3%93%C3%A3RJ%C2%B4%C3%96%1C%C2%8F%C3%87%C3%A9q%C2%808%C2%8E%11B%C2%8Cn%C3%BF%C3%83G%C3%8D%18CY%C2%96Xk%C2%A7%C3%87%7D%C2%B4%7F%C3%94%C2%90%C3%91Zs%3E%C2%9F%C2%A9%C2%AAjz%1C%C3%86%C2%B5%7F4%C2%AE%C2%94%22%C2%8E%C3%A3%C2%87%C3%9A%C3%AFe%C2%B6k%C2%AD%C2%B1%C3%96%C3%924%C3%8D%C2%A0%C2%BA%C2%9B%C3%A3u%C2%B7%C3%9B%C3%9D%1D%22J)6%C2%9B%C3%8DE%C2%ACm%5B%0E%C2%87%C3%83U%C3%AE%C3%A0%C3%BB%3C%C3%8B2%C2%84%107%0B%C2%84%10dYv%15%0F%C2%82%C3%A0%5Bd%10.%C2%A5%C3%84%18s%C2%B3%C3%80%18%C2%83%C2%94r0%C3%94%1B%07%08%C3%83%C2%90(%C2%8A.bQ%14%11%C2%86%C2%A1%17%C3%B8.%0E%C2%90%C2%A6)J)%C3%A0%C3%B3%3B%C2%A7i%C3%AA%0D%C2%86%1E%C3%B7%C2%B9%C2%B5%C2%96%C3%BD~%C3%8Fz%C2%BD%C3%B6%C3%96%C3%AE%C3%9E%C3%B8o%C2%AE%C3%99%C3%BF%C3%A1%C3%AA%C2%99%C3%ACz%09%243l%C2%A0%06%C2%92%0F%C2%88%C2%9C~%C3%A5%C2%A2%C2%9Ds8%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%A2IDATH%C2%89%C3%AD%C3%95%C2%B1%C2%AD%C2%830%14%C2%85%C3%A1%C2%9F(%03d%00%0F%C3%A2.%2B0%06%C2%A2c%09Z%3A%24%C3%9A%C3%9B%C2%BA%C3%89%06%C2%84%C2%8E%C2%96%1Dh%C2%8D%C3%9C%C2%B9%7C%C2%AFK%C3%A3%C2%84%10D%20%C2%85Oi%2C%7F%C3%B7Z%C2%B6I%C2%9Cs%17%C3%A0%06%5C%C3%99%2F%1D%C2%90%26%C3%8E%C2%B9%C3%BB%C3%8E%C3%B0%C2%A3%C2%80%C3%849%C3%B7w%00%0C%C3%80%C3%A9(8%C3%A2%11%C2%8F%C3%B8%C3%AF%C3%A0%C3%96Z%C3%8A%C2%B2%C3%BC%0A~%C2%9E%C3%BB8%0C%03%22%C2%82%C3%B7~_%C3%9C%18C%C3%9B%C2%B6%C2%AB%16%C3%8D%C3%B3%3C%18%C2%AB%C3%AB%C3%BA%3Dn%C2%AD%C2%A5i%1A%C3%86q%5C%05%7F%C2%92%00%C2%AF%C2%AA%C2%8Ai%C2%9A%C2%82%C2%89%C3%8F%C2%BA%C3%99%1C%C3%8F%C2%B2%0C%11%09%3A%7F%C2%B6m%C2%AF%C2%B2%C2%B4%C3%90%C3%A0%C2%B4%2B%C2%A5(%C2%8A%02%C2%AD%C3%B5blmf%C3%BF%C3%A7%7D%C3%9Fc%C2%8C%C3%81%7B%C3%BFQ%C3%A7K3%7B%C3%95%C2%B4%C3%96(%C2%A5%10%C2%91%C3%8Dax%C3%93%C3%B9%C2%B7%C3%B3%C3%9B%C3%8Fk%C3%84%23%1E%C3%B1%C2%AD%C3%B0%C3%AE%20%C2%BB%3B%01%C3%A9%01%05t%40%C3%BA%0F%C2%B0%0CO%C2%8A%C3%87%C3%93%C3%B0%C2%9E%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%83IDATH%C2%89%C3%AD%C3%96%C2%B1%0D%021%0C%40%C3%91%7FL%C3%80F%C2%ACpsd%C2%86%C2%9B%C3%A3%24%C2%B7i%C3%93%C2%B0A%C2%B8%11%C2%B2CZ%C2%9F%C3%9C%C2%A5%C2%84%0E%09A%C3%81%1D%22%11%22)mE%C3%8F%C2%8Ae%2B%C2%83%C2%99%1D%C2%813p%C2%A2%C3%9EY%C2%80q0%C2%B3Ke%C3%B8%5E%C3%80%60f%C3%97%060%00%C2%87V%C3%B0G%C2%B8%C2%AA2M%13%C2%AAZ%1F%17%11%C3%96uED%C3%AA%C3%A2!%04r%C3%8E%00%C3%A4%C2%9C%09!%C3%94%C3%81SJ%C3%84%18%1Fb1FRJ%C3%9F%C3%85U%15%C3%AF%C3%BD%C3%8B%C2%9C%C3%B7~s%C3%BF7%C3%A1%22B)%C3%A5e%C2%AE%C2%94%C2%B2%C2%B9%C3%BF%1F%C3%8D%C2%B9s%C2%8Ey%C2%9E%C3%B7%5E%C3%BF%C3%919%C3%AFx%C3%87%3B%C3%9E%C3%B1w%C3%8E%C2%AE%C3%B5%C3%AA%C2%9C%7B%C2%8A%C3%ADY%C2%B3%C3%BF%C3%BB%C3%AC%C3%BF%C2%8B%C3%B7%7F%7B3%7Cid%2F%07%60lP%C3%80%02%C2%8C7%C2%8C%C2%A6U_G%C3%B6%C2%A2%C2%8E%00%00%00%00IEND%C2%AEB%60%C2%82')
        },
        rollover: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C2%BEIDATH%C2%89%C3%8D%C2%97%C2%A1%C2%8F%C3%820%18%C3%85%1F%C3%8B%C2%A5a5CL%0F%C2%8B%C2%BC%04%3B%2CX%C3%90%15%18%C3%B6G%40%C3%80%11%C3%9C%0Cv%0A%C2%81.%16%2C%C3%BF%C3%80I%04f%C2%B5%2C9f%C2%BA%C2%A4%C2%82%3Bs%10%C2%92%2B%C2%B01%18%3C%C2%B7%C3%A6k%7F%7D%C3%9F%C2%9A%C2%B7%C2%B5%C2%B4%C3%9B%C3%AD%C2%AAI%C2%92%7C)%C2%A5*(H%C2%84%C2%90%C2%BDi%C2%9A%C2%9F%25!%C3%847%C2%80%C2%8AeY%C2%A0%C2%94%3E%1D%2C%C2%A5D%1C%C3%87%00%C2%B07%C2%94R%C2%85%C2%81%01%C2%80R%0A%C3%8B%C2%B2%C2%A0%C2%94%C2%AA%18%C3%87%C2%81%22U.%C2%97%01%00F%C2%A1%C3%94%3F%19%C2%86%C3%B1%3A%C3%B8i%13%C2%AF%C2%84%7F%C3%A8%06%C2%BB%C3%9D%C3%AE%C2%BF%C2%B1%C3%99l%C2%96z%C3%91%C2%B4%C3%B3%C3%9F%C3%8F%C2%B9N%3A7%C2%85%C3%81%C3%B3%C2%B6%5D%C2%A7%C2%87%C2%B7%7D%C2%BD%5E%C2%A7%C2%AE-m%C2%B7%C3%9B%C2%9Fj%C2%B5%C2%9A%1B%1AE%11%C2%82%20%C2%80%C2%94%12%C2%8C1%C3%94j%C2%B5%C2%AB%C3%B5a%18%C2%A6o%C3%BB5-%C2%97Kp%C3%8E%C3%91j%C2%B5%C3%90%C3%A9tR%C3%8F%C3%8B%05%3Fw%3B%18%0C%C3%A08N%C2%A6%C3%B9w%C3%839%C3%A7X%2C%16h%C2%B7%C3%9B%C2%99%C3%9C%C3%A6%C2%82%0B!%10%04%01(%C2%A5%C3%B0%7D%1F%C2%B6m%C3%9F%05%062%C2%9Ev%C3%8E9%26%C2%93%09%5C%C3%97E%C2%BF%C3%9F%C2%BF%08%1E%C2%8DF%C2%88%C2%A2%C3%A8%C3%A6z%C2%A9%C2%9Co6%1B%C3%8C%C3%A7sPJ1%1E%C2%8Fo%C2%BA%15B%608%1C%C3%82%C3%B3%3C%C3%94%C3%AB%C3%B5lp%5DH%C3%B4z%3D4%1A%C2%8D4%7B%05%00%24I%C2%82%C3%A9t%C2%8Af%C2%B3%09%C3%86%C2%98%C2%B6%26u%C3%9B%C2%B3%C2%80%C3%8F%C2%B5Z%C2%AD.%C2%BE%06m%C3%88%3C%23%C3%87m%C3%9B%C2%86%C3%AF%C3%BB%C2%A7%C3%A7L!%C2%93\'%C3%9B%1D%C3%87%C2%81%C3%A7y%C3%BF%C3%AA%1E%C2%92p%C3%97%C3%A4%C2%BA.%18c%C3%9A%C3%BFD-%3C%C2%8B%C3%8BK2M%13%C2%8C%C2%B1%C2%ABg%C3%A5)%C3%8E%C2%8Fm%C2%BE%15%C2%B7%0F%C3%BB%C2%AAeU%18%C2%86o%C3%B0%C3%B7*%C2%A5%2C%14z%C3%A4%19%C2%84%C2%90%7D%1C%C3%87%C2%85l%C3%A0p8%C2%9C%C3%AEj%C2%84%C2%90%7D%C3%A9%C2%95%C2%B7%C3%94_%C3%9A%C2%A2%C2%B8%C2%9F7%C3%9E%15!%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C2%BDIDATH%C2%89%C3%AD%C2%97%C2%AF%C2%AE%C3%A2%40%14%C3%86%3F%C2%9AM%C3%83%C2%8C)%C2%A2z%C2%B0%C3%88M%C2%AA%C2%B1%C2%AD%C2%A5%C2%96%0AT-%C2%B5%20P%3C%C3%86(%C2%9E%C2%A0%C3%98%1A%0C%2Fp%25b%0Cc*h%C2%B2%C3%94%0C%C3%89%C2%90%C3%90%15%C2%BB%C2%90%C2%B0%0B%C3%9C%C3%9B%3F%C2%B7%C3%97%C3%9C%C2%93%C2%8C9s%C2%BE%C3%BCf%C2%BE%C3%8C%C3%B4t%3A%C2%87%C3%83%C2%A1%7F%3A%C2%9D%C3%9E%C2%B4%C3%96%3D%C2%B4%14%C2%A6i%1E%09!%3F%3BR%C3%8A_%00z%C2%96e%C2%81R%C3%BA%C3%A9%60%C2%A5%14%C3%B2%3C%07%C2%80%C2%A3%C2%A1%C2%B5n%0D%0C%00%C2%94RX%C2%96%05%C2%ADu%C3%8F%C2%B8%26%C3%9A%C2%8Cn%C2%B7%0B%000Z%C2%A5%C3%BE%0D%C3%83%C3%B8%C2%83%C3%BD%C3%B1%C2%AAh2%C2%99%C3%BC%C2%97%5B%C2%ADV%C2%8D%C3%95%7F%C3%89%C3%8E%C2%BF%C3%A1%C2%8D%C3%81w%C2%BB%5Dy%C2%91%10%C2%A28%C2%9F%C3%8F%2F%C3%87x%3C~%C2%B7f6%C2%9B%15%C2%9B%C3%8D%C3%A6%C3%83%C3%B5B%C2%88%C2%A2%C2%91%C2%9D%C3%87q%0CJ)%C2%86%C3%83a)%5Dm%C2%B8%C2%94%12I%C2%92%20%0C%C3%83%C3%92%C3%9A%C3%9Ap%C3%8E9%7C%C3%9F%C2%87m%C3%9B%C3%AD%C3%82%C2%AFv%7B%C2%9EWI_%19%C2%9Ee%19%C3%96%C3%ABu%25%C2%BBk%C3%839%C3%A7%18%C2%8DF%C2%95%C3%AC%C2%AE%05O%C2%92%04J)%C3%B8%C2%BE_%19%5C%09%C2%9Ee%19%C3%A28%C2%AEewe8%C3%A7%1C%C2%9E%C3%A7%C2%811%C3%96.%7C%C2%BB%C3%9D6b%C3%B75%3AB%C2%88%C2%A2%C3%9F%C3%AF%3F%C2%9C%7C%C3%94%C2%9F%C3%A7%C3%B39%06%C2%83%C3%81%5Dn%C2%B1X%40J%C3%B9%14%C3%82%18%C3%83r%C2%B9%C2%BC%C3%8B%C3%AD%C3%B7%C3%BB%C3%B2%C2%B6%C3%BF%0B%06%C2%80(%C2%8A%40%08yXO%08A%14E%0F%C3%A7%1A%C3%B9%C2%B6%C3%9B%C2%B6%C3%BD%C3%B4%00%C2%86a%C3%B8%C3%B4%3A6%C3%96R%1D%C3%87%C2%81%C3%AB%C2%BAw9%C3%97u%C3%A18%C3%8ESM%C2%A3%3F%13A%10%C3%9Cn%01c%0CA%10%C2%BC%16%7C%C2%A4%C2%9F%C2%97%19i%C2%9A%16%C3%93%C3%A9%C2%B4H%C3%93%C3%B4%C3%9D~%C3%BE%C3%B2%C2%B4%7Ff%C3%9CN%C2%BBR%C2%AAU%C3%B0%C2%95g%C2%98%C2%A6y%C3%8C%C3%B3%C2%BC%C2%95%05%5C.%C2%97%C3%9B%5B%C3%8D4%C3%8Dc%C3%A7%2B_%C2%A9%C2%BF%01%C2%9C%04%7F%C2%95%C2%8E%C3%9A%05%C3%BB%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%2FIDATH%C2%89%C3%AD%C2%971n%C2%840%10E%3FVd%01%C2%8D)%C2%A8%C3%A1%0A9%C3%80%C3%92r%09%17T%C3%9C%02%C2%8E%40%C3%83%0D%C2%B8%06w%40%C3%8A%01h%C2%A0%05i%C2%A11%C2%92%0B%C2%92%26%20%25%C3%AB%5D%C2%85D%C2%98%24%C3%A2w%1E%7D%C3%A9%C2%8DgF%C2%B6%C3%86%C3%A8%C2%BA%C3%8E%C2%9F%C2%A6%C3%A9EJ%C3%A9%40%C2%93(%C2%A5%C2%83eY%C3%8FF%C3%9B%C2%B6W%00%0Ec%0C%C2%B6m%C3%AF%0E%16B%60%1CG%00%18%C2%88%C2%94R%1B%18%00l%C3%9B%06c%0CRJ%C2%87%2C%01%C2%9D2M%13%00%40%C2%B4R%C3%9FE%089%0E%C2%BE%26q%C3%82O%C3%B8%C2%9F%C2%87%C2%A7i%C2%8A%C2%BE%C3%AF%C2%8F%C2%81%C2%B7m%C2%8B%24IPU%C3%95C%C3%9F%C2%93*%18E%C3%91M%C2%AC(%C2%8AM%09L%C3%93%C2%84%3C%C3%8F%11%C2%86!8%C3%A7J%C3%8F%C3%AE%3D%2F%C3%8B%C3%B2n%1B%C2%8C%C2%BA%C2%AE_%7D%C3%9F%C3%BF%10T%C3%9D%C3%BC%C2%A7r%5D%17Y%C2%96%C2%AD%C3%A7%C2%A6i%C3%94eWiK%C3%99%3F\'%C3%AFy%1E%C3%A28%C2%BE%C3%B1%7D%19%C3%BE%5D%5D.%17p%C3%8E%C2%95%3F%C2%A7%12%C2%BEu%C2%B8T%C2%B2%2C%0B%C2%9Cs%04Ap%C3%97%C2%B3%C3%8B%C3%8D%C2%972%7B%C2%9E%C3%B7%C3%90%C2%A7%1C8%1Dj%C2%9A%C3%A6%1F%3E%C2%AF\'%C3%BC%C2%84%C3%BF%5E%C2%B8%10B%2Bt%C3%A1%11J%C3%A90%C2%8E%C2%A3%C2%96%04%C3%A6y%5Ew5J%C3%A9%60%1C%C2%B9%C2%A5%C2%BE%01%C3%95%C2%BCtq%C3%95%0A%C3%9Bn%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01*IDATH%C2%89%C3%AD%C2%971%C2%8E%C2%84%20%18%C2%85%C2%9FdC%C2%84%06%0Bj%C2%BD%C3%82%26%C3%9C%C2%81KPXYs%02%2BO%C3%A2%C2%8D%C3%B6%006%C3%92Z%C2%AC6%C2%9AP8%5B%C2%B8nf%C2%B2S%2Cd%C3%87Iv%7D%C3%A5Ox%1F%3F%C2%BC%40H%C2%86a(%C2%96ey%C3%B3%C3%9Eg8H%C2%94%C3%92%C2%911%C3%B6%C2%9A8%C3%A7%C3%9E%01dB%08p%C3%8E%1F%0E%C2%9E%C3%A7%19%C3%934%01%C3%80H%C2%BC%C3%B7%C2%87%C2%81%01%C2%80s%0E!%04%C2%BC%C3%B7%19%C3%99%0BG*MS%00%009%C2%94%C3%BA)B6%C3%ACK%C3%8C%C3%A4%C2%B2%2C%C2%BF%C3%95%C3%9A%C2%B6%0D_D%0C%C3%BC%C2%B7t%C3%82%C2%9F%C2%A2%C2%A8%C3%80%C3%AD%C3%A1*%C3%8B2*h%C2%BB%C3%BE%C3%AF%C2%B6%C2%9F%C3%B0%13~%C3%82%C3%BF%3E%3C%C3%A8z%C2%AD%C3%AB%1A%C3%8E%C2%B9%C2%9B%C3%9A%C3%B5%C3%9B%C2%9E%C3%A79%C2%9A%C2%A6%C3%B9%C2%B1_P%C3%A7%C3%96Z0%C3%86%C3%AE%C2%8E1%C3%86%60%C2%AD%0D%C2%B1%0B%C2%83K)QU%C3%95%C3%9D%C2%B1%C2%AA%C2%AA%20%C2%A5%7C%1C%1C%00%C2%94R%C3%90Z%C3%9F%C3%94%C2%B4%C3%96PJ%C2%85Z%C3%85%05%C3%8E%18%C2%83%3C%C3%8F%01l%C3%A7l%C2%8C%C2%89%C2%B1%C2%89O%C2%BB%C2%B5%16R%C3%8A%C3%A0s%C2%BEV%C3%92u%C3%9D%C2%A5(%C2%8Ah%C2%83X%C3%B5%7D%C2%BFu%3E%C3%8F%C3%B3%C2%A1%C3%A0%C2%9DG(%C2%A5%C3%A34M%C2%87%2C%60%5D%C3%97%C2%AF%C2%BF%1A%C2%A5tL%C2%9E%C3%B9K%C3%BD%00%3CoaK6_%1F%C2%A4%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%89IDATH%C2%89%C3%8D%C2%97%C2%AF%C2%8E%C3%AA%40%18%C3%85%0F%C3%8Df%02c%C2%8A%C2%A8%C2%9E%C3%9A%C2%91%C3%BB%00T%C3%B3%18%C2%A8jD%15IQ%10%C2%AAx%C2%82Q%C2%88j4%C3%98j%C2%92%C2%95%230%C2%8Ce%C2%92KM%C2%9BTp%C2%AF%C2%B8%0BK%C3%92.%19%C3%A8%C3%9D%C3%A1%C3%BE%C3%A4%C3%BC%C3%B9%C3%8E%C2%9C%C2%AF%C2%93%C3%93L%C3%A7x%3C%C3%BAeY~TU%C3%95%C2%87%25%08!%C2%A7%5E%C2%AF%C3%B7%C3%9EQJ%C3%BD%02%C3%90w%5D%17%C2%94%C3%92%1F%17.%C2%8A%02y%C2%9E%03%C3%80%C3%89%C2%A9%C2%AA%C3%8A%C2%9A0%00PJ%C3%A1%C2%BA.%C2%AA%C2%AA%C3%AA%3B%C2%97%01%C2%9Bt%C2%BB%5D%00%C2%80cU%C3%B5%13%C3%87q%5E\'~%3D%C3%84%2B%C3%85%C3%9F%C2%9A%06G%C2%A3Qml%C2%B5Z%19%175%C3%9D%C3%BF%C3%BF9o%C2%A2%C3%89%C2%8D5%C3%B1%C2%B6mo%C3%82%C2%B8%C3%ADY%C2%96%19%C2%8B%C2%9B%C3%92%C3%99%C3%AF%C3%B7%C2%BF%7D%C3%9F%C2%BF%C2%BBHJ%09!%04%3C%C3%8FC%18%C2%86%C3%B0%3C%C2%AF%C2%B5%C3%B0%C3%A1p0s%C3%8E9%C3%87r%C2%B9%04%C3%A7%1Cq%1Cc%C2%B3%C3%99%C2%B4%16%07%0C%C2%9D%C3%9F%C2%A2%C2%94%C2%82%10%02%C2%94%C3%92V%5D0v~%0Bc%0C%C2%B3%C3%99%0C%C2%9CsDQ%C2%84%C3%B5z%C3%BD%C2%948%C3%B0%C2%84%C3%B3%5B%C2%B4%C3%96%10B%C2%A0(%0A%C2%84a%08%C3%86%C2%98%C3%B1%5E%23%C3%A7ZkL%C2%A7%C3%93%C3%869%C3%8F%C3%B30%C2%99L0%18%0C%C2%B0X%2C%1E%C3%AE%C3%82%5D%C3%B1%C3%9Dn%C2%878%C2%8E%C2%A1%C2%94%C2%BA%5Bd8%1Cb%3E%C2%9FCJ%C2%89(%C2%8A%20%C2%A54%12%C3%BF6d%C3%924%C3%85v%C2%BB5*%02%7Cu!%C3%8B2%24IR%C2%9B7%C3%8A%C3%B6K%C2%9B%1F%11%C2%BE%25%08%02%C3%A3%C2%B55%C3%A7I%C2%92%40k%5D%5Bh%25%C3%9B%C3%87%C3%A31%C2%84%10%C2%B5%C3%AFl%25%C3%9B%19c%C3%97%1B%C3%BC%C3%934%5E%C2%B8Kzq%C3%8E%C2%91%C2%A6)%C3%8A%C2%B2%7C%C2%A8%C2%A8i%C2%97%C3%AE%C3%BER%C2%83%20%C2%80%C3%AF%C3%BB%10B%3C%24nJ%C2%AB%C2%84k%C3%83S%C3%99%C3%BE%2Fq%C2%80%C2%BFO%18%C2%9B%5C%C3%B4%1CB%C3%88)%C3%8Fs%2B%078%C2%9F%C3%8F%C3%97%C2%B7%1A!%C3%A4%C3%94y%C3%A5%2B%C3%B5%0F%07%C3%84%C3%8Dy%C3%87B%C3%AC%C2%86%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%8CIDATH%C2%89%C3%AD%C2%97%C2%B1%C2%8E%C2%B2%40%10%C3%87%C3%BF%C2%92%C3%8BF%C2%B6%C3%81b%C3%AB%C2%B5%C2%B5%C2%BC%C2%84%C3%9A%16Zi%C2%A5%C2%B0%C3%9A%16%C2%9E%C3%80%C3%8A%C3%82%C2%87%C2%B0%C3%B2%09%C2%B0%C2%A5%C2%B1%C3%B1%05%C2%AE%C2%A4%C3%98F%1A%0AIN%1AL%20%C2%91%2B%C2%BC%C2%BB%C3%84%C2%88%C3%9C%C2%A1%7C%7B%C3%8D7%C3%894%C2%93%C2%99%C3%BCv%C3%BE%2C3%C3%99%C3%9E%C3%A1p%18%C2%9EN%C2%A7%C2%B7%C2%A2(%06Pd%C2%84%C2%90%C2%A3%C2%AE%C3%AB%C2%AF%C2%BD8%C2%8E%C3%9F%01%0C%0C%C3%83%00%C2%A5%C3%B4%C2%9F%C2%83%C3%B3%3CG%C2%96e%00p%C3%94%C2%8A%C2%A2P%06%06%00J)%0C%C3%83%40Q%14%03%C3%AD%2B%C2%A0%C3%92%C3%BA%C3%BD%3E%00%40SJ%C3%BD4M%C2%BB%60_%C2%9A%C2%92f%C2%B3%C3%99Ml%C2%BD%5Ew%C2%96%C3%BF\'%C2%9D%C3%BF%C2%87%C2%B7%C2%86GQ%C3%94%1D%5DJY%C2%95e%C3%99%C3%A8%C3%93%C3%A9%C2%B4*%C3%8B%C2%B2%C3%9An%C2%B7%C2%95%C3%A7y%C2%BF%C3%8Eor)e%C3%95%C2%AA%C3%B3%C3%B1x%0C%C3%86%18%C2%82%20%C3%A8%C2%A4%C3%B1%C3%96%C2%B2%0B!%10%C2%86!%C3%A28V%0Fg%C2%8C%C3%81q%1C%C2%ACV%2B%C3%B5p%00%C2%B0m%1B%C2%94%C3%92%C2%A7%C3%A5%7F%C3%B8W%13B%60%C2%B3%C3%99%20MS%C3%B5p%C3%86%18%26%C2%93%C3%89S%C3%B2%3F5d%1C%C3%87A%C2%9E%C3%A7%08%C3%83P%3D%1C%C2%B8%C3%88%1F%04%C3%81C%C3%B2%3F%0D%C3%A7%C2%9C%C3%83%C2%B6%C3%AD%C2%87%C3%A4%C3%AFd%C2%B6%3B%C2%8E%C2%834M%C2%B1%C3%9B%C3%ADZ%C3%95%C3%B5%C2%A4%C2%94%C3%95p8%C2%BC%0A%C3%8E%C3%A7%C3%B3%C3%86!%C3%829%C3%87b%C2%B1%C2%B8%C2%8AEQ%C2%84%C3%A5ry%C2%93%7Bo%C2%9F%C3%AF%C3%B7%C3%BB%C3%BA%C3%8E%7D%C3%9F%C2%87%C2%AE%C3%AB%C2%B5E%C2%BA%C2%AE%C3%83%C3%B7%C3%BD%C2%9B%C3%B8h4%C2%BA%7B%C3%98%7BV%0Bg%C2%8CA%08Q%5B%20%C2%84%00c%C2%AC5%C3%A8%C3%97p%000M%13%C2%96e%5D%C3%85%2C%C3%8B%C2%82i%C2%9A%C2%9D%C2%80%1B%C3%A1%00%C3%A0%C2%BA.8%C3%A7%00.%C3%9F%C3%99u%C3%9D%C3%8E%C3%80%00~%C3%9E%C3%A7I%C2%92T%C2%9E%C3%A7UI%C2%92%C3%BC%C2%B8%C2%A3%C3%9B%C2%B8%C2%94%C2%B2%C2%AA%C2%BD%C3%AD*%C3%AC%C3%BB%C2%B6%C3%A7y%C2%AE%14%C3%BC%C3%85%C3%93%08!%C3%87%2C%C3%8B%C2%94%1C%C3%A0%7C%3E%7F%C2%BF%C3%95%08!%C3%87%C3%9E_%C2%BER%3F%00%24%C2%91%C2%82w%C3%8F%C2%BA%C3%9E5%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%011IDATH%C2%89%C3%AD%C3%971n%C2%830%14%06%C3%A0%1F%C2%AB%C2%B2%C3%80%0B%C2%9C%00%C2%AF%C2%8C%3D%009%0B%1333%0333%C2%B3%C2%A7%0C%5C%C2%82%3B%20%C3%B5%00%2C%C2%B0%C3%86R%60%01%C3%89%03%C3%AD%C3%92d%C2%81%C2%A6%C2%80%C2%8A%13U%C3%BC%C2%A3%C2%8D%C3%BC%3D%5B%0FK6.%C2%97%0B%1F%C2%86%C3%A1C)%C3%A5%40S(%C2%A5%C2%ADeY%C3%AFF%C3%934W%00%C2%8Em%C3%9B%60%C2%8C%C3%AD%0E%C3%B7%7D%C2%8F%C2%AE%C3%AB%00%C2%A0%25J)m0%000%C3%86%60%C3%9B6%C2%94R%0E%C2%B9%0D%C3%A8%C2%8Ci%C2%9A%00%00%C2%A2U%C3%BD%0E!%C3%A4y%C3%B8%C2%BD%C2%88%03%3F%C3%B0%C2%97%C3%82%C2%A5%C2%94H%C2%92d%17%C3%BC%C3%AD%C3%91dY%C2%96%10B%60%18%06%C2%BDx%C2%9E%C3%A7(%C2%8Ab%C3%93%C2%A2A%10L%C3%86%C3%8E%C3%A7%C3%B3%C3%AF%C2%B8%C2%94%12Y%C2%96%C2%A1i%C2%9AM%C3%B0%C2%9AL%C3%B04M!%C2%A5%C2%9C%7C8%C2%B7%C2%9B%3F%C3%87%C2%A3(%C2%82%10b%C2%B2%C3%B3%C2%B9c%C3%BB)K%0B%C2%9Dt%C2%BB%C3%AB%C2%BA%C2%88%C3%A3%18%C2%BE%C3%AF%2F%C3%86%C2%B6f%C2%B6%C3%A1%18c%08%C3%83%10%C2%9E%C3%A7!%C3%8F%C3%B3%C3%95%C3%9D%C2%BE%C3%B4%C2%94%1E%C3%BEj%C2%A7%C3%93%09%C2%9Cs%08!V%C3%A1KcTU%C3%B5%C3%899%C3%9Fe%C3%B1G%C2%A9%C3%AB%C3%BA%C3%85%C2%AF%C3%97%03%3F%C3%B0%C3%BF%C2%85%C3%B7%7D%C2%AF%15%C2%BDy%C2%84R%C3%9Av%5D%C2%A7%C2%A5%C2%80q%1C%C3%AFo5Jik%3C%C3%B3%C2%95%C3%BA%05%C2%8A%C2%A4y%C3%91%C2%95%19%1C%23%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%13IDATH%C2%89%C3%AD%C2%971n%C3%85%20%0C%C2%86%C3%BF%C2%A0%0A%C3%A5%C2%B1%C2%90%1B%C3%A4%0A%C2%BD%11%C2%87%C3%88I%C2%98%C2%992p%09%C3%AE%C2%80%C3%94%03d%09k%C2%90%1A%16%C2%90%18%C3%92N%C2%A9%C3%94%C3%B6-%C2%A1%0Dox%C3%B9\'d%C3%8B%C3%BE%C2%90m%C2%81%C3%9C%2C%C3%8B%C3%92%C2%A7%C2%94%C3%9Er%C3%8E%1D*%C2%89R%C2%BA%C3%9En%C2%B7%C3%97%C3%869%C3%B7%0E%C2%A0%C3%A3%C2%9C%C2%831v%3A8%C3%86%C2%88%10%02%00%C2%AC%24%C3%A7%5C%0D%0C%00%C2%8C1p%C3%8E%C2%91s%C3%AE%C3%88n%C2%A8%C2%A9%C2%B6m%01%00%C2%A44%C2%81%C3%B7%1E%C3%830%C3%80%7B%7F8%C2%96%10%C3%B27%C2%B8%C2%94%12%C3%9E%7BH)KS%C2%94%C3%81%C2%B5%C3%96p%C3%8E%01%00%C2%9Cs%C3%90Z%C3%97%C2%81%5Bka%C2%8C%C3%B9f3%C3%86%C3%80Z%7B.%C3%9C%7B%0F%C2%A5%C3%94%5D%C2%9FR%C3%AAp%C3%BF%0F%C3%81%C2%A5%C2%94H)%C3%9D%C3%B5%C2%A5%C2%94%0E%C3%B7%C2%BF%C2%99%C2%A6%C3%A9%C2%A3%C3%AF%C3%BBCA%C2%BB%C2%84%10%18%C3%87%C2%B1(v%C2%9E%C3%A7%C3%B2i%C3%BF%0F%5D%C3%B0%0B~%C3%81%2F%C3%B8%C2%A9z)%09%12B%C3%BC%3A%C2%97%3C%C2%B3%C3%8F%5B%C3%B6%C3%A7%C2%85%17%0D%5C%C3%A9%1F%C3%BES%C2%8F%2F%7B%C2%8C%C2%B1*t%C3%A7%11J%C3%A9%1AB%C2%A8r%C2%81m%C3%9B%C2%BEv5J%C3%A9%C3%9A%3CrK%C3%BD%04yA~%C3%9AS%08%C3%83Q%00%00%00%00IEND%C2%AEB%60%C2%82')
        },
        pressed: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%88IDATH%C2%89%C3%8D%C2%97!%C2%8F%C3%820%1C%C3%85%C3%9F%C3%9A%5BBQ%C2%80%C2%9C9%C2%81E%C2%9C%60%12%C3%A4l%25%C3%BB%0A%20%09%0A%C2%BE%02%C2%99D%23A%22%C2%A9%C2%9D%C3%84%C3%9C%07%00%12%C3%8C%C3%A4%0E5%20K%C3%8B%C2%89%3B.%C2%97%C3%9C%C2%80%C2%8E%C3%81%C2%B8g%C2%965%C3%BF%C3%AE%C3%97%C3%B7_%C3%B3%C2%B6%1A%C2%9B%C3%8D%C3%A6UJ%C3%B9%C2%AE%C2%94*!\'%11B%C2%B6%C2%94%C3%927c%C2%BD%5E%7F%00(%C2%99%C2%A6%09J%C3%A9%C3%83%C3%81RJ%C3%84q%0C%00%5B%C2%A2%C2%94%C3%8A%0D%0C%00%C2%94R%C2%98%C2%A6%09%C2%A5T%C2%89%C2%9C%06%C3%B2%14!%C3%A4%C3%AB%C2%9A%2B%C3%B5%5B%C2%86a%3C%0F~%C3%92S%C3%A1%2FI%C2%83%C2%BD%5E%C3%AF%C3%8F%C3%98p8%C3%94~%C2%A8%C3%AE%C3%BC%C3%BF%C3%A7%3CIInr%C2%83gm%7B%C2%92%C3%AE%C3%9E%C3%B6%C3%85b%C2%A1%5Dk%2C%C2%97%C3%8Bc%C2%B1X%C3%8C%0C%0D%C3%83%10%C3%93%C3%A9%14%C2%BB%C3%9D%0E%C2%9CsT%C2%AB%C3%95%C2%8B%C3%B5Q%14%C3%A9%C2%B7%C3%BD%C2%92%7C%C3%9F%C3%87%7C%3EG%C2%B3%C3%99%C2%84%C3%A38%C3%9A%C3%B32%C3%81%7F%C2%BB%C3%ADt%3A%C2%B0%2C%2B%C3%95%C3%BC%C2%9B%C3%A1B%08%08!%C3%A08N*%C2%B7%C2%99%C3%A0A%10%602%C2%99%C2%801%C2%86~%C2%BF%C2%8FJ%C2%A5r%13%18H%C2%B9%C3%9B%C2%85%10%18%C2%8DF%C2%B0m%1B%C3%ADv%C3%BB%2C%C3%98%C3%B3%3C%C2%84ax%C3%B5yZ%C3%8EW%C2%AB%15f%C2%B3%19%18c%C3%A8v%C2%BBW%C3%9D%06A%00%C3%8F%C3%B3%C3%A0%C2%BA.j%C2%B5Z%3AxRH%C2%B4Z-%C3%98%C2%B6%C2%AD%C2%B3V%00%C3%80~%C2%BF%C3%87x%3CF%C2%A3%C3%91%00%C3%A7%3C%C2%B1F%C2%BB%C3%ADi%C3%80%C2%BF%C3%A5%C3%BB%C3%BE%C3%99%C3%97%C2%90%182%C2%8F%C3%88%C3%B1r%C2%B9%C2%8C%C3%81%60%C3%B0s%C2%9F*d%C2%B2d%C2%BBeYp%5D%C3%B7O%C3%9D%5D%12%C3%AE%C2%92%C3%AA%C3%B5%3A8%C3%A7%60%C2%8C%C3%A9%C3%81%C3%93%C2%B8%3C%C2%A7B%C2%A1%00%C3%8E%C3%B9%C3%85%C2%BD%C3%B2%10%C3%A7%C2%A76_%C2%8B%C3%9B%C2%BB%7D%C3%95%C3%92*%C2%8A%C2%A2%7F%C3%B0%C3%B7*%C2%A5%C3%8C%15z%C3%A2%11B%C3%886%C2%8E%C3%A3%5C%16p%3C%1E%7F%C3%8Ej%C2%84%C2%90%C3%AD%0B%C2%A5%C3%B4MJ%C3%B9~8%1Cr%3F%C2%A5~%02%0B%C2%A9%C2%B58Q%16%C3%94a%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C2%BCIDATH%C2%89%C3%AD%C2%971o%C2%82%40%14%C3%87%C3%BF%C3%9C%C2%95%04%C2%9D%0C%C3%A3-%1D%5C%19%C2%BA0%C2%9E%23%C2%8E%C2%8C2%C2%B2%C2%B2%1A%3E%C2%86a%C3%B5%2B%C3%88G%C2%90QG%C2%97~%005qa%C2%BC%3Aa%0C%C2%B9%C2%B3C%C2%ABI%5B%25%0A%C3%B4%C2%BA%C3%B4%C2%BF%C2%90%3C%C3%BE%2F%3F%C3%AE%C3%A5%C3%9D%3B%C3%8E%C3%98%C3%ADv%C3%8FR%C3%8AW%C2%A5T%0F%C2%9AD%08%C3%99SJ_%C2%8C%C3%ADv%C3%BB%06%C2%A0g%C2%9A%26(%C2%A5%C2%BF%0E%C2%96R%C2%A2%2CK%00%C3%98%13%C2%A5%C2%9460%00PJa%C2%9A%26%C2%94R%3Dr%0E%C3%A8%14!%C3%A4%C3%A3%C2%A9%C2%95%C3%BA)%C3%830%00%00OU%C2%A68%C2%8E%7F%C3%84%26%C2%93Ik%C3%BE%3FY%C3%B9%3F%C2%BC5%C3%B8f%C2%B3y8%C3%87X%C2%AF%C3%97%C2%A7n%C2%B7%5Bi%C2%8A%C3%A3%C2%B8%C2%B2q%00%20I%12p%C3%8E%C3%A1%C2%BA%C3%AE%5D%C3%BE%C2%A2(%C3%9AYy%C2%96e%C3%A8t%3Ap%5D%C3%B7%C2%A1%C2%BC%C3%86%C3%B0%3C%C3%8F%C2%B1X%2C0%1A%C2%8D%1E%C3%8Em%0C%C2%9F%C3%8Df%18%0E%C2%87%C2%B0m%5B%2F%C3%BC%5Cn%C3%8Ey%C2%AD%C3%BC%C3%9Ap!%04%C2%B2%2C%C2%ABU%C3%AE%C3%86%C3%B04M%C3%A1y%5E%C2%ADr7%C2%82%2F%C2%97K%1C%0E%07x%C2%9EW%1B%5C%0B.%C2%84%C3%80%7C%3EG%10%04%C2%8D%C3%80%C2%B5%C3%A0i%C2%9Ab0%18%C2%801%C2%A6%17%C2%BEZ%C2%ADZ)%C3%B7Y%C2%95%C3%A3%C3%B5%C3%9A%C3%B9%1CE%11%C3%BA%C3%BD%C3%BE%C2%97X%C2%92%24%C3%88%C3%B3%C3%BC%26%C2%841%C2%86%C3%B1x%C3%BC%25Vk%C2%BC~%07%03%40%18%C2%86%C2%B0%2C%C3%AB%C2%AA%C3%9F%C2%B2%2C%C2%84ax%C3%B5%5D%2B%C2%B3%C3%9D%C2%B6%C3%AD%C2%9B%0D%18%04%C3%81%C3%8D%C3%AD%C3%98%C3%9A%C2%91%C3%AA8%C3%8E%C2%8FI%C3%879%C2%87%C3%A387sZ%C3%BD%C2%99%C3%B0%7D%C3%BF%C2%B2%0B%18c%C3%B0%7D%C2%BF%C3%92%7F%C3%97y%C3%BE%C2%88%C2%84%10%C2%98N%C2%A7%C2%88%C2%A2%C2%A8r%C3%BA%15E%C3%91%3E%C3%BC%5E%5D%C2%BA%5DJ%C2%A9%15%7C%C3%A6%11B%C3%88%C2%BE%2CK-%1Fp%3A%C2%9D.w5B%C3%88%C3%BE%C2%89R%C3%BA%22%C2%A5%7C%3D%1E%C2%8F%C3%9Ao%C2%A9%C3%AF%C3%81H%C2%AC%C3%93%3C)%C3%AF%0D%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%014IDATH%C2%89%C3%AD%C2%971n%C3%820%14%C2%86%7F%3Fc%C2%890%C2%855K%C2%AF%C3%80%40F%C3%8E%C3%A0%C2%95%C2%83%C2%B0%C3%A5%0A%C3%88g%C3%A0.%5C%C2%80!%23B%C3%8A%C2%921%C3%8D%14%C2%81%C2%AC%C3%A7th%C2%83%C3%94b%10i%C2%85i%C2%AB%7C%C2%8B%C2%95%C2%A7_%C3%BA%C2%9E%C2%9F%C2%ADH%16EQ%C2%BC0%C3%B3%C3%8E9%17%23%10DTK)g%C3%A2p8%C2%BC%02%C2%88%C2%95R%C2%90R%3E%5C%C3%8C%C3%8C%C2%B0%C3%96%02%40M%C3%8E%C2%B9%60b%00%C2%90RB)%05%C3%A7%5CL%5D!%24D%C3%B4%C2%BE%06%C2%B5~%20%C2%84x%C2%9E%C2%BCc%C2%90%0F%C3%B2%C2%BF%2F7%C3%86%C2%A0%C2%AA%C2%AA%C3%A7%C3%88%C3%8B%C2%B2%C2%841%06y%C2%9E%C3%9F%C3%8C%C2%8D%7C%C3%85%C3%95juQ%5B%C2%AF%C3%97%C2%BD%1A8%1E%C2%8F%C3%98l6X%2C%16%C3%90Z%7B3%0F%3F%C3%B3%C3%ADv%7B%C3%B5%18%C3%84~%C2%BFo\'%C2%93%C3%89%C2%A7%C2%A2o%C3%A7%3Fe%3A%C2%9D%22%C3%8B%C2%B2%C3%B3w%C3%934%C3%BE%C2%B1%C3%BB%C3%A83%C3%B6%C2%AF%C3%8D\'I%C2%82%C3%A5ry%C2%91%C2%BB%5B%C3%BE%5D%C3%A6%C3%B39%C2%B4%C3%96%C2%88%C2%A2%C3%A8%3Ey%C3%9F%C3%8B%C3%A5c%3C%1ECk%C2%8D4M%C2%AFf%1E%C2%B2%C3%B3n%C3%8CI%C2%92%C3%9C%C3%8Cy%2F%5C%08%C2%9A%C2%A6%C3%B9%C2%87%C2%BF%C3%97A%3E%C3%88%7F%C2%AF%C2%9C%C2%99%C2%83J%3B%1F%11Qm%C2%AD%0D%C3%92%40%C3%9B%C2%B6%C3%A7%C2%B7%1A%11%C3%95%23)%C3%A5%C2%8C%C2%99w%C2%A7%C3%93)%C3%B8%2B%C3%B5%0D%C2%BD%C2%A4q%3A%267%C3%84%5B%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%2FIDATH%C2%89%C3%AD%C2%97%3F%C2%8A%C2%840%14%C3%86%3F_F%10%2B%C2%B5%C2%B4%C3%99%0BX%C3%8C%05%3CC%C3%9A%C2%949%C2%83%C3%A4%18%C3%A2%C3%99%C3%A6%00%22%C3%98%C3%98%C2%AE%C2%95%20!q%C2%8A%C3%99%19v%C3%99Y%C3%98d%C2%98%0C%C3%AC%C3%BA5%C2%81%C3%B7x%C3%9F%2F%7F%1E!%C2%89%C3%86q%7C3%C3%86%C2%9C%C2%AC%C2%B5%19%02%C2%89%C2%88f%C3%86%C3%981%1A%C2%86%C3%A1%1D%40%16%C3%871%18cO%07%1Bc%C2%A0%C2%B5%06%C2%80%C2%99%C2%AC%C2%B5%C3%81%C3%80%00%C3%80%18C%1C%C3%87%C2%B0%C3%96ft%0D%C2%84%14%11%5D%C3%86%C2%A0%C3%94%0FEQ%04%008%C3%B8%14%2B%C2%A5%C2%BE%C3%85%C3%9A%C2%B6u%C3%B6y%C3%89%C3%8Aw%C3%B8K%C3%A1Q%C3%9F%C3%B7%5B%C2%9A%C2%A6%5E%C3%85J)%C2%AFF%03%C2%80eY%C3%BE%C3%B1%C2%B6%C3%AF%C3%B0%1D%C2%BE%C3%83%C3%BF%3E%C3%9C%C3%A9z%C3%AD%C2%BA%0E%C3%934%C3%BD%C2%98%2F%C3%8B%12M%C3%93%C3%BC%C3%8A%C3%8B%C3%B9z%C2%95R%22I%C2%92%C2%BB%C2%B9%24I%20%C2%A5t%C2%B1s%C2%83%17E%01!%C3%84%C3%9D%C2%9C%10%02EQ%3C%0F%0E%00UU%C2%A1%C2%AE%C3%AB%2F%C2%B1%C2%BA%C2%AEQU%C2%95%C2%AB%C2%95_%C3%83q%C3%8EQ%C2%96%25%C2%80%C3%8B9s%C3%8E%7Dl%C3%BC%C2%BB%5DJ%C2%89%3C%C3%8F%C2%9D%C3%8F%C3%B9%C2%B3%1EzL%3C%C2%A2%5B%C2%B7%1Bc%C2%82%C2%82%C2%AF%3C%22%C2%A2Yk%1Dd%02%C3%9B%C2%B6%C3%9D%C3%BEjD4%1F%18cGc%C3%8Ci%5D%C3%97%C3%A0%C2%BF%C3%943%C3%B8%C3%81f%C2%B4%C2%A4%C3%B1%0B%C2%B0%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%8FIDATH%C2%89%C3%8D%C2%97%C2%A1%C2%92%C3%AA0%14%C2%86%C3%BF%26%5B%01%C2%8A%C2%A9%2Cb_a%05%C2%91%C2%B5%C3%A5%11%C3%80%22k%19%5E%01%C3%87T%C3%92W%08%C2%B26%12d%C3%8C%0A%24a%06SYP%05%C2%A6%C3%93%C2%B2bo%C3%AF%C2%BD3%C3%AD%C2%B2a%7B7%7B%3F%C3%93%C2%99%24%C3%8D%C2%97s%C3%9A%C2%9CL%C2%AC%C3%83%C3%A1%C3%B0%5C%14%C3%85kY%C2%96%3D%18%C2%82%10r%C2%A2%C2%94%C2%BEX%C3%BB%C3%BD%C3%BE%08%C2%A0g%C3%9B6(%C2%A5%C3%9F..%C2%8A%02y%C2%9E%03%C3%80%C2%89%C2%94eiL%0C%00%C2%94R%C3%98%C2%B6%C2%8D%C2%B2%2C%7B%C2%A4j0%09!%C3%A4%C3%BDi%C3%94%C3%BA%0B%C3%8B%C2%B2~N%5E%C3%B1%C2%A3%C3%B2%C2%A7%C2%A6%C3%86%C3%99lVk%5B%2C%16%C3%9A%C2%93%C3%AA%C2%BE%C3%BF%C3%BFE%C3%9EDS4%C3%86%C3%A4m%C3%93%C3%9E%C2%84v%C3%9A%C2%A5%C2%94%C3%9Ar%5D%C2%AC%C3%9Dnw%C3%ABv%C2%BBw%07)%C2%A5%C3%809%C2%87%C3%A38%18%C2%8DFp%1C%C2%A7%C2%B58%C3%8B2%3Dy%C2%85%10%02%C3%AB%C3%B5%1A%C3%83%C3%A1%10%C2%9E%C3%A7%C2%99%C2%95%03%40%C2%92%24%C3%A0%C2%9C%C2%A3%C3%93%C3%A9%C2%B4%C3%8A%C3%82%C2%97%C3%A4%15B%08%08!%C3%A0%C3%BB%3E%7C%C3%9F7%2B%07%C2%804M%C2%B1Z%C2%ADp%3E%C2%9F1%1E%C2%8F%C3%A1%C2%BA%C3%AEC%C3%B2O%C3%BF%C3%B64M%11%C2%86ac%C2%9F%C3%A38%08%C2%82%00%C2%8C1%2C%C2%97K%08!%C3%B4W%C2%8EO%C2%B6%C3%9Av%C2%BBE%18%C2%86H%C2%92%C3%A4%C3%AE%24%C2%9E%C3%A7a%3A%C2%9DB)%C2%85%C3%B9%7C%0E%C2%A5%C2%94%C2%96%C3%BC%C3%83%22%13%C3%8716%C2%9B%C2%8D%C3%96%24%C3%80%C2%9F%2CH)%11EQ%C2%AD_%C2%AB%C2%B6Wi~D%C3%BC7%C2%8C1%C3%AD%C2%B1%C2%B5%C3%88%C2%A3(%C3%82%C3%B1x%C2%AC%0D4R%C3%9B\'%C2%93%098%C3%A7%C2%B5%C3%AFl%C2%A4%C2%B6%C2%BB%C2%AE%C2%8B%20%080%18%0C%C2%B4e_%C3%A5%C3%AE%3E%C2%97R%22%C2%8Ec%5C.%C2%97%C2%87%22%C3%97!%C3%8B%C2%B2%C3%BBG*c%0C%C3%BD~%1F%C2%9C%C3%B3%7F*%C2%AEhU%C3%A1%C3%9A%C2%A0U%C3%A1%C2%BE%13%02%C2%BC_aLR%C3%B9%08!%C3%A4%C2%94%C3%A7%C2%B9%C2%91%05%C3%9Cn%C2%B7%C3%9Fw5B%C3%88%C3%A9%C2%89R%C3%BAR%14%C3%85%C3%AB%C3%B5z5~K%7D%033%C2%95%C3%A0Ku%C3%96%C2%88%C3%B9%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%8EIDATH%C2%89%C3%AD%C2%97%C2%B1n%C3%AA0%14%C2%86%C3%BF%C3%98%C2%8D%14%C2%98%C2%90%C3%87%2C%1DX3ta4%C2%A3%193%26cVV%C2%94%C2%8DW%C2%88X%C3%B3%0A%C3%B0%08x%C2%841K%1F%C2%80%20%C2%B10%C2%BAL%20%149%C3%9C%C2%81%C3%9BJ%15%24%224%C3%B5%5D%C3%AE%C2%BFX%3A%3AG%C2%9F%7Dl%C3%BF%C2%89%C2%AD%C3%9Dn%C3%B7%C2%AA%C2%B5~%2F%C3%8B%C2%B2%07C%22%C2%84%1C(%C2%A5o%C3%96v%C2%BB%C3%BD%00%C3%90%C2%B3m%1B%C2%94%C3%92_%07k%C2%ADQ%14%05%00%1CHY%C2%96%C3%86%C3%80%00%40)%C2%85m%C3%9B(%C3%8B%C2%B2G%3E%03%26E%08%C2%B9%C2%8EF%C2%A9%7FeY%16%00%C3%A0%C2%A5.)%C2%8E%C3%A3%C2%9BX%C2%92%24%C2%AD%C3%A5%C3%BF%C2%93%C2%95%C3%BF%C2%877%C2%86%C3%A7y%C3%9E%1A%C3%9C%C3%9Al6%C2%97n%C2%B7%5B%C2%9B%14%C3%871%C2%92%24A%C2%96e%C2%90Rb%3A%C2%9D%3E%C2%94_%C2%A7%C3%A3%C3%B1%C3%98l%C3%A5%C2%83%C3%81%00%C2%8C1H)%C2%9B%C2%94U%C2%AAq%C3%9B%C2%83%20%C3%80j%C2%B5%C3%82~%C2%BF7%0Fg%C2%8Ca4%1Aa%3E%C2%9F%C2%9B%C2%87%03%00%C3%A7%1C%C2%9DN%C3%A7%C3%87%C3%AD%7F%C3%BA%C2%AA%05A%00)%25%C2%94R%C3%A6%C3%A1%C2%8C1%08!%C2%B0X%2C%C3%8C%C3%83%01%40%08%C2%81%C3%93%C3%A9%C2%84%C3%B5zm%1E%0E%00a%18b%C2%B9%5C%3E%C3%95%C3%BE%1F%C3%83%5D%C3%97%C3%85p8%7C%C2%AA%C3%BD%C2%ADx%C2%BB%10%02J)dY%C3%96%C2%A8%C3%AE%C2%AE%C2%BD%C3%8Ef%C2%B3Z%13q%5D%17%C2%93%C3%89%C3%A4%5B%2C%C3%8Fs%C2%A4iz%C2%93%5Be%C2%B3%C2%95%C3%B6%1AE%11%1C%C3%87%C2%B9%5B%C3%A48%0E%C2%A2(%C2%BA%C2%89%C3%B7%C3%BB%C3%BD%C3%8A%C3%89V%C3%A9.%C2%9C1%C2%860%0C%C3%AF%16%C2%84a%08%C3%86Xc%C3%90%C3%83p%00%C3%B0%3C%0F%C2%9C%C3%B3o1%C3%8E9%3C%C3%8Fk%05%5C%0B%07%00%C3%9F%C3%B7%C3%A1%C2%BA.%C2%80%C3%AB%3E%C3%BB%C2%BE%C3%9F%1A%18x%C3%A0%7B%C2%AE%C2%94B%C2%9A%C2%A6%18%C2%8F%C3%87%C2%AD%C2%B5%1B%C2%B8%1E%C2%B8%C2%87~%26~C_%C2%A7%5Dkm%14%C3%BC%C3%89%23%C2%84%C2%90CQ%14F%26p%C2%B9%5C%C2%BE%C3%9Ej%C2%84%C2%90%C3%83%0B%C2%A5%C3%B4Mk%C3%BD~%3E%C2%9F%C2%8D%C2%BFR%C3%BF%00Z%C2%8B%C2%B2%C3%93%20%C2%A5%C3%A4.%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%013IDATH%C2%89%C3%AD%C3%97%C2%A1%C2%8E%C2%830%18%07%C3%B0%7F%C2%BF%0E%C2%83B%C2%83%C2%B8W%C2%98%C2%80%07%C2%9A%C3%84%C3%B3%18%04%C2%8B%C2%9D%C2%AC%C3%AD3L%C3%8EL%20%09%C3%89%0C%12P%04BZ%26%C3%AE8%03%C2%B7%C3%9B%C2%96%C2%AD%C2%B7%5C%C3%B8%1B%C2%92%C2%96%C3%B4%C3%B7%7D%0D4)%3B%C2%9F%C3%8F%1FJ%C2%A9%C2%93%C3%96%C3%9A%C2%81%C2%A1%10Q%C3%839%C3%9F%C2%B2%C2%A2(j%00%C2%8EeY%C3%A0%C2%9C%C2%BF%1CVJa%18%06%00hHkm%0C%06%00%C3%8E9%2C%C3%8B%C2%82%C3%96%C3%9A%C2%A1i%C3%80d%C2%88%C3%A8%C3%B3iT%C3%BD%0Ac%C3%AC%C3%AF%C3%B0)%2B%C2%BE%C3%A2%C3%AF%C2%85WU%C2%85%24I%5E%C2%82o%C2%AEMfY%06!%04%C2%BA%C2%AE3%C2%8BK)q8%1C%1EZ4%C2%8A%C2%A2%C3%99X%1C%C3%87%C2%BF%C3%A3UUa%C2%BF%C3%9F%C2%A3%2C%C3%8B%C2%87%C3%A0%7B2%C3%83%C3%934E%5D%C3%97%C2%B3%17%C2%97%C2%BAy%3A%C2%BE%C3%9B%C3%AD%20%C2%84%C2%98u%C2%BE%C2%B4m%3F%C3%A5%C3%96Bg_%C2%BB%C3%AB%C2%BA%08%C3%83%10%C2%BE%C3%AF%C3%9F%C2%8C%3D%1A%C2%96%C3%A7%C3%B9h%C3%9B%C3%B6%C3%A2%C3%A4%C3%B1x%C2%84%C2%94%12%5D%C3%97%C3%9D%C3%95%C3%B9-i%C3%9B%C3%B6%C3%BA%C2%AF%16%04%01%3C%C3%8F%C2%83%10%C3%A2%C2%A9%C3%B0%C2%94%C2%AB%C2%9D%C2%BF2m%C3%9B%C2%BE%C3%B9%C3%B1%C2%BA%C3%A2%2B%C3%BE%C2%BFp%C2%A5%C2%94Qt%C3%B2%C2%88%C2%88%C2%9Aa%18%C2%8C%140%C2%8E%C3%A3%C3%B7%5D%C2%8D%C2%88%C2%9A%0D%C3%A7%7C%C2%AB%C2%94%3A%C3%B5%7Do%C3%BC%C2%96z%01%C2%B3%C2%81%7F1%C3%AB%03%C2%B5%C2%AB%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%16IDATH%C2%89%C3%AD%C2%971j%C3%850%0C%C2%86%C3%BF%C3%88%C3%B5%C2%92)7%C3%A8%15%C3%9E%C3%A1r%C2%85l9%C2%84G%C2%AF%3E%C2%83%2F%C3%B0%06%C2%8F!%C2%90%C3%85%C2%A3%C2%93)%24%18%3B%1DJ%0A%C2%A5o%C2%89%C3%93%C2%98B%C3%B3%2F%06%09%C3%A9%13H%C3%88%C2%A8%18%C2%86%C3%A1%3D%C2%84%C3%B0%C2%8C1V%C3%88%24%22%C2%9A%18c%C2%8F%C2%A2%C3%AF%C3%BB%11%40%C3%859%07c%C3%ACrp%08%01%C3%9E%7B%00%C2%98(%C3%86%C2%98%0D%0C%00%C2%8C1p%C3%8E%11c%C2%ACh7%C3%A4%14%11%7D%C2%BE%C2%A9%09%C2%9Csh%C2%9A%06%C3%8E%C2%B9%C3%83%C2%B1EQ%C2%9C%C2%83%0B!0%C2%8E%23%C2%84%10%C2%A9)%C3%92%C3%A0J)Xk%01%00%C3%96Z(%C2%A5%C3%B2%C3%80%C2%8D1%C3%90Z%7F%C2%B3i%C2%ADa%C2%8C%C2%B9%16%C3%AE%C2%9C%C2%83%C2%94%C3%B2%C2%A5OJy%C2%B8%C3%BF%C2%87%C3%A0B%08%2C%C3%8B%C3%B2%C3%92%C2%B7%2C%C3%8B%C3%A1%C3%BE%17%5D%C3%97meY%1E%0A%C3%9AU%C3%975%C3%9A%C2%B6M%C2%8A%C2%9D%C3%A79%7D%C3%9A%7FC7%C3%BC%C2%86%C3%9F%C3%B0%1B~%C2%A9%C2%92%C3%96k%5D%C3%97%3FlG%C3%97%C3%AC%C2%BD%5E%C3%BF\'%C3%BC%C3%94%7F~F%7Fc%C3%A0B%08Y%C2%A1%3B%C2%8F%C2%88h%C3%B2%C3%9Eg)%60%C3%9B%C2%B6%C2%AF%5B%C2%8D%C2%88%C2%A67%C3%86%C3%98%23%C2%84%C3%B0%5C%C3%975%C3%BB%C2%95%C3%BA%01H%C2%89%C2%88%3F!%C2%B6eo%00%00%00%00IEND%C2%AEB%60%C2%82')
        },
        checked: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C2%BAIDATH%C2%89%C3%8D%C2%97%C2%B1n%C3%820%10%C2%86%7F%C2%90e%0FA%22%C2%82%C2%89%C2%89)%2CY%0A%C2%93\'%3F%00L%C2%99%C3%B2%00%3C%17%C3%8F%10%09%09%26%16%C2%98%C2%98%0CK%26%1E%C2%80%C2%89%08%C2%A4d%C2%B8(%12%5D%C2%9A%C2%AAj%C3%93%C2%92%040%C3%BD%C3%87%C3%93%C2%9D%3F%C3%BFg%C3%ABd7%C3%A6%C3%B3y%C2%9F%C2%88%C3%B6Y%C2%96%C3%990%24%C3%86%C3%98Y%08%C3%B1%C3%86%C2%88h%0F%C3%80n%C2%B7%C3%9B%10B%3C%1DLDH%C2%92%C3%84%26%C2%A2%7D3%C3%8B2%C3%9B%C2%B2%2C%23%60%00%10B%C3%80%C2%B2%2CdYf7%C3%B3%C2%80Iq%C3%8E%01%00M%C2%A3%C3%94%0F5%1A%C2%8D%C3%97%C3%81s%C2%BD%14%C3%8E%C2%8A%C2%82%C2%B3%C3%99%C3%ACGl%3A%C2%9D%C2%96%5E%C2%B4l%C3%BD%C3%BFs%5E%C2%A4%227%C3%86%C3%A0%C3%B7%C2%B6%C2%BDH%0Fo%C3%BB%C3%A1p(%C2%9D%5B%C3%A8%C2%BC%C2%8A%C3%8B%5Cq%1Cc%C2%BD%5E%23MS%C2%8C%C3%87c%C3%B4z%C2%BDz%C3%B0%C2%AA%0A%C3%83%10Zk%C2%B8%C2%AE%C2%8B%C3%91hT%C2%BA%C3%AE.%C3%B8w%C2%B7%C3%9Dn%C2%B7R%7Dm%C2%B8%C3%96%1A%C2%BB%C3%9D%0E%C3%83%C3%A1%C2%B0%C2%92%C3%9B%C2%BB%C3%A0%C2%A7%C3%93%09%C2%9B%C3%8D%06%C2%9Cs%C3%B8%C2%BE%C2%8FV%C2%ABU%0B%0CT%C2%BC%C3%ADZk%2C%C2%97K%0C%06%03L%26%C2%93_%C3%81A%10%20%C2%8E%C3%A3%C2%9B%C3%AB%C2%95r~%3C%1E%C2%B1%C3%9Dn%C3%819%C2%87%C3%A7y7%C3%9DFQ%C2%84%20%08%C2%A0%C2%94B%C2%BF%C3%9F%C2%AF%06%2F%1A%12J)8%C2%8ESf%C2%AF%00%C2%804M%C2%B1Z%C2%AD%C3%A0%C2%BA.%C2%A4%C2%94%C2%859%C2%A5%C3%9B%5E%05%C3%BCUa%18%C3%BEz%0CFf%7B%14EX%2C%16%C3%B0%7D%C2%BF%1E%C3%BC%C2%9E%C3%99%C3%9E%C3%A9t%C2%A0%C2%94%C3%BA%C2%91%C3%B7%C2%90%09%C3%B7%C2%97%1C%C3%87%C2%81%C2%94%C3%B2%C3%B3%C3%9Dv%13%5Eg%C2%B6%7F%17%C3%A7%1CR%C3%8A%3F%C3%AF%C3%8AS%C2%9C%C3%A7m%C2%BE5n%C2%9F%02%C3%B7%3C%C2%AFT%C3%9E%C3%AB_%C2%AFDd%14%C2%9A%C3%B3%C2%9A%C2%8C%C2%B1s%C2%92%24F6p%C2%BD%5E%C3%B3%C2%BF%1A%18cg%26%C2%84x%23%C2%A2%C3%BD%C3%A5r1%C3%BEK%7D%07%C3%81%C3%BF%C2%AA%0B%C3%B0f%04%C2%99%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%8EIDATH%C2%89%C3%AD%C2%97%C2%B1%C2%8A%C3%82%40%10%C2%86%C3%BF%C2%84e%C2%B7H%C3%80%10A%C2%B0J%C2%95*%C3%8D%C2%99*U%C2%AC%C3%92he%C3%A5%03%C3%B8%5C%C2%A9-%05%C3%81%07%C2%88%C2%95U%C2%B4I%C3%A5%03%08%01EaS%C2%AC%2Cx%C3%85%5D%0E%C3%AEN%C3%83%C2%99%C3%A4%C3%B6%C2%9A%C3%BB%C3%8B%C3%99%19%C2%BE%C3%AC%C3%ACdvG%5B.%C2%97%C2%8E%10b\'%C2%A5%C2%B4%C2%A0H%C2%84%C2%903c%C3%AC%C2%85%08!v%00%C2%ACN%C2%A7%03%C3%86%C3%98%C2%AF%C2%83%C2%85%10(%C2%8A%C3%82%12B%C3%ACt)%C2%A5e%18%C2%86%120%000%C3%86%60%18%06%C2%A4%C2%94%C2%96%5E%1AT%C2%8AR%0A%00%C3%90%C2%95R%C3%9F%C2%A5i%1A%00%C2%80T9%C3%85q%C3%BC%C3%8D6%C2%9B%C3%8DZ%C3%B3%C3%BF%C2%93%C2%9D%C3%BF%C3%83%5B%C2%83%1F%0E%C2%87%C2%A7c*%0B%C2%AE%2C%C2%968%C2%8E%2B%0B%07%006%C2%9B%0D%C3%820%C2%84%C3%AB%C2%BA%3F%C3%B2%07Z%C3%9Ay%C2%9A%C2%A6%C2%A0%C2%94%C3%82u%C3%9D%C2%A7%C3%A2%1A%C3%83%C2%8F%C3%87%23%C2%B2%2C%C3%83p8%7C%3A%C2%B61%7C%C2%BD%5E%C3%83%C3%B7%7D%C2%98%C2%A6%C2%A9%16%5E%C2%A6%C3%9B%C3%B3%C2%BCZ%C3%B1%C2%B5%C3%A1%C2%9Csl%C2%B7%C3%9BZ%C3%A9n%0CO%C2%92%04%C2%83%C3%81%C2%A0V%C2%BA%1B%C3%81%C2%B3%2C%C3%83%C3%B5z%C2%85%C3%AF%C3%BB%C2%B5%C3%81%C2%B5%C3%A0%C2%9Cs%C2%A4i%C2%8A0%0C%1B%C2%81k%C3%81%C2%93%24%C2%81%C3%A7y%C3%A8v%C2%BBj%C3%A1%C3%BB%C3%BD%C2%BE%C2%95t%C2%97%C3%92%C3%A6%C3%B3%C3%B9%C2%AD%C3%97%C3%AB%C3%9D%5D%C2%BCw%3F%C2%8FF%23%C3%B4%C3%BB%C3%BDO%C2%B6%C3%85b%C2%81%C3%93%C3%A9%C3%B4%10b%C3%9B6%26%C2%93%C3%89\'%5B%C2%9E%C3%A7%C3%8F%C2%A7%C3%BD%2B%18%00%C2%A2(%C3%BAx%1A%7D%15%C2%A5%14Q%14%C3%9D%5Dk%C2%A5%C2%B7%C2%9B%C2%A6%C3%B9%C2%B0%00%C3%830%7C%C3%B8%3B%C2%B6v%C2%A5%3A%C2%8E%C3%B3%C2%AD%C3%93y%C2%9E%07%C3%87q%1E%C3%86%C2%B4%C3%BA%C2%98%08%C2%82%00%C2%B6m%03x%3B%C3%A7%20%08*%C3%BD%2B%0B%C2%AE%C2%8E8%C3%A7X%C2%ADV%18%C2%8F%C3%87%C2%95%C3%9D%2F%C3%8F%C3%B3%C3%AA%C3%87D%1D%C2%99%C2%A6%C2%89%C3%A9t%C3%BA%23_%1Dx%1BaT%C2%AA%C3%A4%C3%A9%C2%84%C2%90sQ%14J%3E%C3%A0v%C2%BB%C2%95%C2%B3%1A%08!g%C3%82%18%7B%11B%C3%AC.%C2%97%C2%8B%C3%B2)%C3%B5%15%C2%B3%C2%A0%C2%A0%C3%AF_%C3%85%C2%A3%C2%97%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%2CIDATH%C2%89%C3%AD%C2%971%C2%8A%C2%840%14%C2%86%7F%25%24%C2%85%C2%85%C2%A2%07%C3%98%C3%8A%C3%8Af%C3%8AT%C3%9E%C3%81%C3%8A%03x.%C3%8F%20%08v%1E%C3%82%C3%86%03x%C2%81%C2%A0%C2%A0%C3%85%C2%93%C2%80%C3%93%C2%AC%5B%C3%AC8%C2%B3%C3%A3%C2%80%C2%99%C3%9D%C3%85%C2%AF%7C%C3%BC%C3%B0%C2%BD%C2%BC%C2%84%C2%84XeY~%10Q%C2%A3%C2%B5%C3%B6%60%08%C3%86X%2F%C2%84%C2%B80%22j%00x%C2%AE%C3%ABB%08q%C2%B8%C2%98%C2%880M%C2%93GD%C2%8D%C2%AD%C2%B5%C3%B6%1C%C3%871%22%06%00!%04%1C%C3%87%C2%81%C3%96%C3%9A%C2%B3%C3%97%C2%82I8%C3%A7%00%00%C3%9B%C2%A8%C3%B5%13%C3%8B%C2%B2%C3%9E\'_9%C3%A5%C2%A7%C3%BC%C3%AF%C3%8B%C2%8B%C2%A2%C3%808%C2%8E%C3%AF%C2%91%2B%C2%A5P%14%05%C2%BA%C2%AE%7B%C2%98c%5B%C3%85%3C%C3%8FojY%C2%96%C3%ADj%60%C2%9Eg%C3%94u%C2%8D(%C2%8A%20%C2%A5%C3%9C%C3%8C%1C%C2%BE%C3%A7m%C3%9B%C3%9E%C3%9D%C2%86%C3%8D%C2%95o%C2%B15%C2%8DgQJ%C2%A1%C2%AA*%C2%A4i%C3%BA%C2%9A%7C%C3%8F%C3%98%C2%BF7%C3%AA%C3%BB%3E%C3%A28%C2%BE%C3%89%3D-%7F%C2%950%0C!%C2%A5%C3%BCz%C3%89~%C2%94%C3%AF%3D%5C%5Bp%C3%8E!%C2%A5D%18%C2%86w3%C2%87%C2%AC%7C%1Ds%10%04%0Fs%C2%87%C3%88%C2%93%24y*%C3%B7%C3%BF%C2%AE%C3%97S~%C3%8A%7F%C2%AF%C2%9C%C2%88%C2%8CJW%C2%9F%C3%8D%18%C3%AB%C2%A7i2%C3%92%C3%80%C2%B2%2C%C3%AB_%0D%C2%8C%C2%B1%C2%9E%09!.D%C3%94%0C%C3%83%60%C3%BC%C2%97z%05%5D%C2%B1k%C2%AA*%C3%B3F%60%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%011IDATH%C2%89%C3%AD%C2%97%3Dn%C2%840%10%C2%85%1F%C3%88%C3%82%C2%85-%C2%81%C2%B8%C2%80%0F%C3%A0fKW%C3%AE%7C%00*%0E%C3%80%C2%B98%03%12\'%C3%A1%24%08%0A%17%23Yb%C2%8B%5D%C2%A2%C3%9D%24E%C3%AChY)%C3%A1%C2%95%C2%83x%1F%C3%83%3C%C3%B9\'%1B%C3%87Q%11%C3%91%14B%C2%A8p%C2%90%18c%0B%C3%A7%C3%BC%C3%82%C2%88h%02P%C2%95e%09%C3%8E%C3%B9%C3%8B%C3%81D%04%C3%AF%7DEDS%1EB%C2%A8%C2%84%10%C2%87%C2%80%01%C2%80s%0E!%04B%08U%C2%BE%17%C2%8ETQ%14%00%C2%80%C3%BCP%C3%AA%5DY%C2%96%01%00X%C3%8A%C3%8B%7D%C3%9F%7F%C2%A9u%5D%17%C3%AD%C3%B3%C2%96%C3%8EO%C3%B8%5B%C3%A1I%C2%81%C3%9B%C3%83%C3%95%C3%B7%7DR%C3%90v%C3%BD%C3%9F%C3%9F~%C3%82O%C3%B8%09%C3%BF%C3%BB%C3%B0%C2%A8%C3%A5u%18%06%C3%8C%C3%B3%C3%BCT%7B%C3%9C%C3%9B%C3%AB%C2%BAF%C3%934%3F%C3%B6%C2%8B%C3%AA%C3%9C9%C3%B7q%04%C3%BA%C2%AC%C2%A2(%C3%A0%C2%9C%C2%8B%C2%B1%C2%8B%C2%83K)a%C2%AD%C3%BD%C3%B6%C2%99%C2%B5%16R%C3%8A%C3%97%C3%81%01%40)%05%C2%AD%C3%B5SMk%0D%C2%A5T%C2%ACUZ%C3%A0%C2%8C1%C2%A8%C3%AB%1A%C3%80m%C3%8E%C3%86%C2%98%14%C2%9B%C3%B4%C2%B4%3B%C3%A7%20%C2%A5%C2%8C%C2%9E%C3%B3%C2%A3%C2%92%0E%13%C3%80m%C3%BEm%C3%9B%26%C2%83%C2%81%7B%C3%A7D%C3%B4%2B%C2%93X%C3%AD%C2%BC%C2%9C1%C2%B6x%C3%AF%0F%C3%B9%C2%80m%C3%9B%C3%B6%C2%BB%1A%18c%0B%C3%A3%C2%9C_%C2%88hZ%C3%97%C3%B5%C3%B0%5B%C3%AA%15eaV%23H%03%C2%A65%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%92IDATH%C2%89%C3%8D%C2%97%C2%B1%C2%AE%C2%A2%40%14%C2%86%7F%C3%88%C2%84)%20%C2%91Hb2%C3%95V%C3%98%C3%98%C3%9C%C3%B0%00%C3%9AXIE%C3%85%03%C3%B8%5C%C3%96%C2%86j%12%13y%00**cc%23%C2%95%C3%95%24%26%1AM%C2%A0%18C%C3%A2%16%C2%BBTr%C2%BD%C3%83u%C3%AF%C2%B8%7Fyf%C2%98%C2%8F%C3%B3s%C3%A6%C2%90c%C2%ACV%C2%AB_R%C3%8Am%5D%C3%97.4%C2%89%10r%C2%A1%C2%94~%10)%C3%A5%16%C2%80%C3%9B%C3%AB%C3%B5%40)%C3%BDq%C2%B0%C2%94%12UU%C2%B9R%C3%8A%C2%ADY%C3%97%C2%B5k%C3%9B%C2%B6%160%00PJa%C3%9B6%C3%AA%C2%BAv%C3%8D%26%C2%A0S%C2%96e%01%00L%C2%AD%C3%94%C2%BF2%0C%C3%A3%7D%C3%B0Fo%C2%85%C2%93%C2%B6%C3%A0b%C2%B1x%C2%88%C3%8D%C3%A7s%C3%A5CU%C2%9F%C3%BF%C3%BF2oS%5B6%C3%9A%C3%A0%C2%AF%C3%9A%C3%9E%26e%C3%9B%C2%8B%C2%A2P%C2%86%C2%AB%C3%8AX.%C2%97%C3%B7%C3%81%60%C3%B0t%C2%93%10%02Y%C2%96%C3%81q%1CL%26%138%C2%8E%C3%B32%C3%B8x%3C%C2%AAe%C3%8E%18C%1C%C3%87%60%C2%8C%C2%81s%C2%8E%C3%9Dn%C3%B72%1C%C3%A8X%C3%ADA%10%606%C2%9Ba%C2%BF%C3%9Fc%C2%BD%5E%C2%A3%2CK%7Dp%00%C3%B0%3C%0FQ%14%C2%811%C2%86%24I%C2%B0%C3%99l%C3%B4%C3%81%1B%05A%C2%808%C2%8E!%C2%84%00%C3%A7%1C%C2%A7%C3%93%C3%A9%C3%9F%C3%83%C3%8B%C2%B2%04%C3%A7%C2%BCu%C3%8Dq%1C%C2%84a%C2%88%C3%A1p%C2%884M%3B%C2%BB%C3%B0%14~8%1C%C3%809%C3%87%C3%B9%7C~z%C3%88h4B%14E%10B%20I%12%08!%C2%94%C3%A0%C2%9F6%C2%99%3C%C3%8F%3BUu%C3%A3BQ%14H%C3%93%C3%B4a%5D%C2%A9%C2%B776%7F%C3%B7%3A%C3%B9%C2%BE%C2%AF%C2%BC%C3%B7!%C3%B3%C3%8F%C2%AE%C2%90%C2%96%C3%9E%3E%C2%9DN%C2%91e%C3%99%C3%83w%C3%96%C3%92%C3%9B%3D%C3%8FC%18%C2%86%C2%9D%C3%AC%C3%BB%C2%AEZ%0B%C3%8E%C2%B2%2C%C2%8C%C3%87c0%C3%86%C2%90%C3%A79n%C2%B7%5B%C2%A7CU%5Dz%C3%BAK%C3%B5%7D%1F%C3%BD~%1FY%C2%96u%C2%82%C2%AB%C3%AA%C3%8B%26%C3%93%C2%B4%C3%93%C2%B7%C3%80%7FR%26%C3%B0g%C2%84%C3%91%C2%A9%C2%86g%12B.UUiy%C2%81%C3%BB%C3%BD%C3%9E%C3%8Cj%20%C2%84%5C%08%C2%A5%C3%B4CJ%C2%B9%C2%BD%5E%C2%AF%C3%9A%C2%A7%C3%94%C3%9F%C3%B0I%C3%80k%C3%9F%C2%946%C2%B2%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%97IDATH%C2%89%C3%AD%C2%97%C2%B1%C2%8A%C3%B2%40%10%C3%87%C3%BF%09K%C2%B6%C3%88%C2%82!%C2%82%60%C2%95%C3%8Aj%C2%9B3U%C2%AA%C2%A4Jee%C3%A5%03%C3%A4%C2%B9R%5B%0A%C2%82%C3%B6%C3%BA%00%C3%91%26U%1E%20%20(%0AI%C2%B1%C2%B2%C3%A0%15%C2%9E%07w%C2%9A%7C%C3%86%C3%8B%C2%B7%C3%97%C3%9C%C2%BF%1Cf%C3%B8%C3%AD%C3%8E%C3%AE%C3%8E%C3%8Eh%C3%B3%C3%B9%C3%9C%11Bl%C2%A5%C2%94%16%14%C2%89%10r%C2%A4%C2%94%C2%BE%11!%C3%84%16%C2%80%C3%95%C3%A9t%40)%C3%BD%C3%AF%60!%04%C3%8A%C2%B2%C2%B4%C2%84%10%5B%5DJi%C2%99%C2%A6%C2%A9%04%0C%00%C2%94R%C2%98%C2%A6%09)%C2%A5%C2%A5%C3%9F%0C*e%18%06%00%40WJ%C3%BD%C2%90%C2%A6i%00%00R%C3%A7%14%C3%87%C3%B1%C2%9D-%C2%8A%C2%A2%C3%96%C3%BC%7Fe%C3%A7%7F%C3%B0%C3%86%C3%B0%3C%C3%8F%5B%C2%83k%C3%93%C3%A9%C3%B4%C3%92%C3%AB%C3%B5j%C2%9D%C3%A28F%14E%C3%88%C2%B2%0CI%C2%92%602%C2%99%3C%C3%A5_%C2%A7%C3%9Dn%C3%97l%C3%A7%C2%83%C3%81%00%C2%8C1%24I%C3%92%24%C2%ACR%C2%8D%C3%93%1E%04%01%C3%924%C3%85~%C2%BFW%0Fg%C2%8C%C3%81u%5D%C2%AC%C3%97k%C3%B5p%00%C3%A0%C2%9C%C3%830%C2%8C%1F%C2%A7%C3%BF%C3%A5%C2%A7%16%04%016%C2%9B%0D%C2%8A%C2%A2P%0Fg%C2%8Ca8%1Cb%C2%B5Z%C2%A9%C2%87%03%C2%80%C3%AB%C2%BA8%C2%9F%C3%8FH%C3%93T%3D%1C%00%7C%C3%9FG%C2%92%24%2F%C2%A5%C3%BF%C3%87%C3%B0n%C2%B7%0B%C3%8E%C3%B9K%C3%A9o%C2%A5%C2%B6%C2%BB%C2%AE%C2%8B%C2%A2(%C2%90eY%C2%A3%C2%B8%C2%87%C3%A5u6%C2%9B%C3%A1p8T%06%C3%99%C2%B6%C2%8D%C3%B1x%C3%BC%C3%85%C2%96%C3%A79%C2%96%C3%8B%C3%A5%C2%9DoU%C2%99%C2%AD%2C%C2%AFa%18~%C2%B6%3A%C3%9Fe%18%06%C3%820%C2%BC%C2%B3%C3%B7%C3%BB%C3%BD%C3%8A%C3%85V%C3%A9!%C2%9C1%06%C3%9F%C3%B7%1F%06%C3%B8%C2%BE%0F%C3%86Xc%C3%90%C3%93p%00p%1C%07%C2%9C%C3%B3%2F6%C3%8E9%1C%C3%87i%05%5C%0B%07%00%C3%8F%C3%B3%60%C3%9B6%C2%80%C3%AB9%7B%C2%9E%C3%97%1A%18x%C3%A2%3F%2F%C2%8A%02%C2%8B%C3%85%02%C2%A3%C3%91%C2%A8%C2%B5t%03%C3%97%0BW%C3%9B%C2%BD%02%C3%97%C3%B3%C3%BFW%C3%B3%C3%B0%C2%AAt%C3%A0%3A%C3%82%C2%A8%C3%94%C2%8D%C2%A7%13B%C2%8EeY*Y%C3%80%C3%A5r%C2%B9%C3%8Dj%20%C2%84%1C%09%C2%A5%C3%B4M%08%C2%B1%3D%C2%9DN%C3%8A%C2%A7%C3%94w%C2%88q%C2%AB6%C3%B8%C2%A8F%C2%B0%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%015IDATH%C2%89%C3%AD%C3%97%C2%B1%C2%8E%C3%82%20%18%07%C3%B0%7F%1B%02%03%C2%83M%C2%99%C2%9Cnb%C3%B7%09%3Au%C3%AB%C3%94g%C3%B0%C2%B9%7C%C2%86%26%26NnL%C2%8C%5D%7C%00_%C3%80h%02%C3%83gH%C3%AAr%C2%9D%C3%9A%C3%93jN%C2%BC%C3%A4%C3%BA%1F%C2%81%C3%B0%03%C3%B2AB%C2%B2%C3%9Dn%C2%BF%C2%88%C2%A8%0D!d%C2%88%14%C3%86%C3%98Y%08%C2%B1bD%C3%94%02%C3%88%16%C2%8B%05%C2%84%10o%C2%87%C2%89%08%C3%9E%C3%BB%C2%8C%C2%88%C3%9A4%C2%84%C2%90I)%C2%A3%C3%80%00%20%C2%84%C2%80%C2%94%12!%C2%84%2C%C3%AD%1Bb%C2%86s%0E%00H%C2%A3%C2%AA%C3%9FI%C2%92%C3%A4sx%C2%9F%19%C2%9F%C3%B1%C2%BF%C2%85%3B%C3%A7%C3%904%C3%8D%5Bpv%C2%AF%C3%B3x%3C%C3%82%18%C2%83%C3%AB%C3%B5%1A%17%C2%B7%C3%96%C3%A2p8%C2%BC4%C3%A9f%C2%B3%19%C2%B4%C2%AD%C3%97%C3%AB%C3%87%C2%B8s%0E%C3%BB%C3%BD%1E%C2%A7%C3%93%C3%A9%25%C3%B8%C2%99%0C%C3%B0%C3%9Dn%07%C3%A7%C3%9C%60%C3%A0%C3%98n~%1D%2F%C3%8B%12%C3%86%C2%98%C3%81%C3%8E%C3%87%C2%8E%C3%AD%C2%A7L%5D%C3%A8%C2%A0%C3%9A%C2%95R%C2%A8%C2%AA%0AZ%C3%AB%C3%89%C3%98%C2%AB%19-8%C3%8E9%C2%8A%C2%A2%C3%80r%C2%B9%C2%84%C2%B5%C3%B6%C3%A9j%C2%9FzJw%C2%AF%C2%9A%C3%96%1Ay%C2%9E%C3%83%18%C3%B3%14%3E5%0F%1F%19%C2%A5%14%C3%AA%C2%BA%C3%BE%0C%C3%BE%C3%8E%C3%8C%C3%B8%C2%8C%C3%BF3%C2%9C%C2%88%C2%A2%C2%A2%C2%BD%C2%972%C3%86%C3%8E%C3%9E%C3%BB(%0B%C3%A8%C2%BA%C2%AE%C3%BF%C2%AB%C2%811vfB%C2%88%15%11%C2%B5%C2%97%C3%8B%25%C3%BA%2F%C3%B5%06%C3%9B\'v%C2%8A!%C2%9E%C2%88%7C%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%1AIDATH%C2%89%C3%AD%C2%971%C2%AE%C2%84%20%10%C2%86%7F%0D%C2%91%C2%82Bo%C3%B0%C2%AE%C2%B0W%C3%B1%0C%C2%9E%C2%8B%03lkKGei%C3%83%01%C2%BC%C2%80%C3%91d(%C2%86%C2%90%C2%B8%C2%95%C3%89n%C3%9E6%C2%B8O%5E%C2%B1%C3%BE%15%C3%8C%C3%80%7Cdf%02%C2%A1%C3%A8%C3%BB%C3%BE%C2%87%C2%99%C3%87%18c%C2%83L%12B%2CR%C3%8A%C2%9B%60%C3%A6%11%40S%C3%975%C2%A4%C2%94%C2%A7%C2%83%C2%99%19%C3%9E%C3%BB%C2%86%C2%99%C3%872%C3%86%C3%98(%C2%A5%C2%B2%C2%80%01%40J%09%C2%A5%14b%C2%8CM%C2%B9%1Br%C2%AA%C2%AA*%00%40y4%00%11%C3%A1~%C2%BF%C2%83%C2%88%C2%92%C3%B7%16E%C3%B1%19%C3%9C%18%03%22%C2%821%C3%A6h%C2%88c%C3%B0a%180%C3%8F3%00%60%C2%9Eg%0C%C3%83%C2%90%07%3EM%13%C2%9Cs%2F6%C3%A7%1C%C2%A6i%3A%17ND%C2%B0%C3%96%C2%BE%C3%B5Yk%C2%93%C3%AB%C2%9F%047%C3%86%20%C2%84%C3%B0%C3%96%17BH%C2%AE%C2%BFHY%C3%9C%C2%B6%C3%AD%C3%8B%5Ck%C2%8D%C2%AE%C3%AB%C2%92%C2%80%C3%8F%3A%C3%9C%C3%AD%7F%C2%A1%0B~%C3%81%2F%C3%B8%05%3FUI%C3%97%C3%AB.%C2%AD%C3%B5%C2%AF%C3%B1%C2%91k%C3%B6%7B%C3%93%C3%BE%C2%BD%C3%B0C%0D%C3%B7%C3%89%1B%C3%BE%C2%AC%C3%BFO%3B3g%C2%85%C3%AE%C2%BCR%08%C2%B1x%C3%AF%C2%B3%1C%60%C3%9B%C2%B6%C3%BD%C2%AF%06!%C3%84%22%C2%A4%C2%947f%1E%C3%97u%C3%8D%C3%BEK%7D%00%C2%9C%C3%AF%7F.%C2%9DIb%60%00%00%00%00IEND%C2%AEB%60%C2%82')
        }
    };

    var light = {
        normal: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01wIDATH%C2%89%C3%AD%C2%97%C2%AF%C2%8E%C3%82%40%10%C2%87%3F%C2%9AKX%C3%97S%C3%A0%C2%908%C3%AA%C3%AA%20g%16E%05%C2%BE%C2%B2%C2%AEo%C3%90\'%C3%A0)*%C2%91%24%C2%88%C2%9A%06E%16W%C3%87%23%C3%A0%C3%80%C3%95%C2%81%C2%BA%13%C3%B7\'%5CX%C3%A8%16%0A%C2%BD%C2%84%C3%BB%C3%89%C3%89%C3%8C~%C3%B3%C2%9Bn%C2%A6m%23M%C3%93W%60%0E%C2%BC%C3%B18-%C2%81%C2%B1U%03%C2%98%2F%C3%9E%C3%9C%C2%AA%01%C3%BC%C3%93%C2%80U%13%18%C2%80%7Fx-z%C3%91%05\'%C2%93%C3%89I%2C%C2%8A%22%C3%A3CM%C3%AB%C3%BF%C2%9Es%C2%9Dtn%1E%06%C2%BFu%C3%AC%3AU%3E%C3%B6%C3%B5zm%C2%9C%C3%9BH%C3%93%C3%B4%C2%BD%0Ah%C2%9E%C3%A7%24I%C3%82~%C2%BFg8%1C%C3%92%C3%A9t%0Ak%C2%8C%C3%87~IY%C2%96%C2%A1%C2%94%C3%82u%5D%06%C2%83%C2%81q%C3%9DM%C3%B0c%C2%B7%C2%BE%C3%AF%C3%93n%C2%B7K%C3%95_%0DWJ%C2%B1Z%C2%AD%C3%A8%C3%B7%C3%BB%C2%A5%C3%9C%C3%9E%04%C3%9Fn%C2%B7%24I%C2%82%10%C2%820%0C%C2%B1m%C3%BB*0%C2%94%C2%BC%C3%ADJ)%C2%A6%C3%93)%C2%8E%C3%A3%C3%A0%C3%BB%C3%BEYp%1C%C3%87%C3%A4y%5Ex%C2%9E%C2%91%C3%B3%C3%8Df%C3%83b%C2%B1%40%08A%10%04%C2%85nw%C2%BB%1Dq%1C%C3%A3y%1E%C3%9Dn%C2%B7%1C%5C%C2%B7%24F%C2%A3%11%C2%8E%C3%A3%C2%98%C3%B4%0A%C3%80%C3%A1p%606%C2%9B%C3%A1%C2%BA.RJm%C2%8E%C3%B1%C3%98%C3%8B%C2%80%C2%8F%C2%95e%C3%99%C3%99%C3%87%C2%A0%5D2%C3%B7%C3%98%C3%A3%C2%B6m%13%C2%86%C3%A1%C2%AF%C3%98Cv%7B%C2%AB%C3%95%C3%82%C3%B3%C2%BC%C2%93%C2%BCJ6%C3%9C%25%C3%B5z%3D%C2%A4%C2%94%08!%C3%8C%C3%A0e%5C%C2%9ES%C2%B3%C3%99DJy%C3%B1%C2%AE%C3%9C%C3%85%C3%B9%C3%B7%C2%98%C2%8B%C3%96meo%C2%B5k%C3%B4%C2%BC_%C2%AF%16%C2%9F%3Fmuhi%01%C3%A3%1A%1AX%02%C3%A3%0F%C3%89%C2%A6z%3E%1Es%C2%97%C3%B6%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01hIDATH%C2%89%C3%A5%C2%96%C2%A1r%C2%83%40%10%C2%86%C2%BF0%C2%9D!%C2%8E*%C3%A2%22%C2%91%C2%B8saj%C2%90%C2%88%08%1C2O%C2%91\'%C3%A0)%C2%90%C3%B1%11g%C2%90%0Cq%C2%B8%3CB%5Cp%C3%A7p%C2%ADh%C3%93%C2%996%C2%81%C2%84%C3%A3%267%C3%93%C2%AE%C3%9C%C3%BBw%C2%BE%C2%BB%C2%9Fc%C3%B7feY%C2%BE%02%7B%C3%A0%C2%8D%C3%A7E%05%C2%AC%1D%0B%60%C2%BEx%7B%C3%87%02%C3%B8%7B%03%C2%8E%250%00V%C3%A1%2FC%C2%8By%C2%9E_%C3%A5%C2%B6%C3%9B%C2%AD1%C3%BD%C3%BF%C2%B5%C3%BDo%C3%80O%C2%A7%C3%93%C3%A8%C2%9AYY%C2%96%C3%AF%C3%B7Dy%C2%9E%0F%5E%1C%C2%80%C2%A2(%10B%10%C2%86%C3%A1Cz0t%C3%B2%C2%BA%C2%AE%C2%99%C3%8F%C3%A7%C2%84a8%C2%AAn2%C3%BC%7C%3E%C3%934%0DI%C2%92%C2%8C%C2%AE%C2%9D%0C%C2%97R%12E%11%C2%9E%C3%A7%3D%17~%C2%B1%5B%08%C2%A1U%C2%AF%0DWJq8%1C%C2%B4%C3%AC%C2%9E%0C%C2%97R%C2%B2Z%C2%AD%C2%B4%C3%AC%C2%9E%04o%C2%9A%C2%86%C2%AE%C3%AB%C2%88%C2%A2H%1B%C2%AC%05WJQ%C3%97%C3%B5%24%C2%BB%C2%B5%C3%A1RJ%C2%84%10%2C%16%C2%8B%C3%A7%C3%82%C2%8F%C3%87%C2%A3%11%C2%BB%2F1%C3%98%5Eo%C3%8D%C3%A7%2C%C3%8BX.%C2%97%3FrEQ%C3%90%C2%B6m%2F%C3%84%C3%B7%7D6%C2%9B%C3%8DU~%C2%B4%C3%AD%C2%BF%C3%81%00i%C2%9A%C3%A2%C2%BA%C3%AEM%C2%BD%C3%AB%C2%BA%C2%A4izs%C3%8DHo%C3%B7%3C%C2%AF%C3%B7%02%26I%C3%92%C3%BB%3B%1A%1B%C2%A9A%10%5Cu%3A!%04A%10%C3%B4%C3%96%18%7DL%C3%84q%C2%8C%C3%AF%C3%BB%C3%80%C3%A7w%C2%8E%C3%A3xP%C3%BF%C3%90%3C%1F%13J)v%C2%BB%1DY%C2%96%C3%9D%C3%AD~%C3%86%C3%A1c%C3%82%C3%BA%1B%C2%AE%C2%B2%C3%84%C2%AE%1C%60ma%03%15%C2%B0%C3%BE%00%17um%C2%A1%C3%9E4%0B%C2%A5%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%A4IDATH%C2%89%C3%AD%C2%941%0E%C2%82%40%10E%C2%9F%C3%84%04%3A%C2%AC%C3%A0%14%C3%90Q%C3%9BLI%C3%819%C2%B8%01\'%C3%A0%14%7B%04%12%C3%9B-%C2%A9%C2%B9%01G%C2%A0%C2%A3%C2%A4%C3%92FM%C3%94%C3%85%C2%88%067%C2%9A%C3%BD%C3%A5%C3%8FO%C3%9E%C3%9F%C2%99%C3%89n%C2%B4%C3%96%3B%C3%A0%00%C3%AC%C3%B9%C2%9EZ%C2%A0%C3%B0%2C%C2%809%C3%B3%0E%C2%9E%05%C3%B0%C2%B5%C2%80g%09%0C%C2%80%C2%83%3B%C2%B8%C2%83%C3%BF.%5C)%C3%858%C2%8Ev%C3%A0%C3%830%C2%A0%C2%94%C2%A2%C3%AF%C3%BB%C2%A7%C2%B9%C2%AD%C3%89%C2%AC%C3%AB%C3%BA%C3%81%C2%AB%C2%AAjQ%C2%81i%C2%9Ah%C2%9A%C2%86%2C%C3%8B%10%11cf%C3%B5%C2%9Dw%5D7%C2%BB%C2%86%C2%8D%C3%96%C3%BAxo%C2%9A%5E%C3%BE%C2%A9%C3%820%C2%A4%2C%C3%8B%1B%C3%8F8v%C2%93%C2%96%C2%8C%C3%BD%C2%BE%7C%14E%C3%A4y%C3%BE%C2%90%7B%19%C3%BE%C2%AE%C2%92%24AD%08%C2%82%C3%A05%C3%B8%C3%92%C3%A32%C3%89%C3%B7%7DD%C2%844Mg3%C2%AB%C2%BC%C3%BC2%C3%A68%C2%8E%C2%9F%C3%A6%C2%8C%07%C3%B7-%C3%BD%C3%9F%C3%B7%C3%AA%C3%A0%0E%C3%AE%C3%A0%26xk%C2%89%C3%9Dz%40a%C2%A1%40%0B%14\'%C2%B7A9%C3%9C%C2%BF%C3%9F%C2%BA0%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%A1IDATH%C2%89%C3%AD%C2%96%C2%B1%C2%8D%C3%830%0CE_%C2%8C%00*%C2%99J%23h%03%C3%95%C3%97%C2%A8T%C3%A1%C2%82%1Bx%0AO%C2%A0)%C2%BCCZ%C2%95%5E%C3%87K%5C%C2%8A%C3%9C%05%09%C3%A2%03N%0Ar%02%C3%8EfI%C2%82%C3%BFQ%24A%C3%A8%C2%90s%3E%01g%C3%A0%C2%83%C2%BF%C2%B3%19%C3%A8%C2%BB%06%60%C2%BEx%C3%A7%C2%AE%01%C3%B8V%40%C3%97%08%0C%40S%C3%B8%C2%B1%26)%C2%A5%C3%B4%C3%A4%1B%C3%87%C2%B1Xg%C2%BBm%C3%9F.%C3%BC%C2%90s%C3%BE%C2%ACMN)U-%C3%9A%C2%B7m%C2%B7%C3%AD%3B%7C%C2%87%C3%AF%C3%B0%C3%BF%0F%2F%3A%C2%AF%C3%934%C2%B1%2C%C3%8B%C2%8Fqk-%C3%830%C3%BC%1A%5E%C3%B4rU%C3%85%18%C2%B3%1A3%C3%86%C2%A0%C2%AA%25rep%11!%C3%86%C2%B8%1A%C2%8B1%22%22%C3%AF%C2%83%038%C3%A7%C3%B0%C3%9E%3F%C3%B8%C2%BC%C3%B78%C3%A7J%C2%A5%C3%AA%16.%C2%84%C2%80%C2%B5%16%C2%B8%C3%8E9%C2%84P%23S%C2%BF%C3%AD%C2%AA%C2%8A%C2%88%14%C3%8F%C3%B9%C3%9E%5E%C3%BAL%C2%BCj%C3%8D%C2%8F%C3%8C%C3%9C%C2%88%3Dw%40%C3%9F%C2%A0%C2%80%19%C3%A8%2F%C3%B1%C3%A8*%C2%9D%C3%BE%C3%96%C3%A8W%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01xIDATH%C2%89%C3%AD%C2%97%C2%ADn%C3%830%14%C2%85%C2%BFE%03e%1E3%C3%AB%1B%C3%94%2C%C2%AC%C3%91%5E%20%C2%A0Ra%60%C2%9F%C2%A2O%C3%907(%0B4%1F0%C2%89%14R%C2%B9%2C%2C%C2%B0%C3%90%C2%ACef-%C3%9A%C3%80%C3%96mR%C2%B2%C3%96Y%C2%A7d%C3%92v%C2%A0%7F%C3%AE%C3%A7%7Bb%C2%9D%24wEQ%3C%00O%C3%80%23%C3%BDi%03%C3%8C%C2%A2%01%C3%80%C2%BC%C3%B1%C2%9E%C2%A2%01%C3%80%C3%AF%07%C2%88%06%02%03%C3%B0%0F%1FD%C3%B7m%C2%83%C2%AB%C3%95%C2%AA1%C2%B6%5C.%C2%83%C2%8B%C2%86%C3%AE%C3%BF%7D%C2%9D%C2%B7%C2%A9%C2%AD%C2%9B%C3%9E%C3%A0%C2%B7%C3%9A%C3%9E%C2%A6%60%C3%9B%C3%AB%C2%BA%0E%C2%86%C2%87%C3%AA%C2%AE(%C2%8A%C3%A7k%C2%8B%C2%9Cs%18c%10B%C2%90%C2%A6)B%C2%88%C3%BE%C3%A0gYk%C2%A9%C2%AA%C2%8A%24I%C2%88%C3%A3%C2%B8_8%C3%80~%C2%BF%C3%87%18%C3%83h4%C2%BA%C3%99%C2%85%C3%8E%C3%B0%C2%B3%C2%AC%C2%B5l%C2%B7%5B%C2%A6%C3%93)I%C2%92%C3%B4%0B%07%C3%B0%C3%9Ec%C2%8C%C3%A1x%3C%C2%92%C2%A6)R%C3%8AN%C3%BB%C2%AF%C3%9Ev%C3%AF%3Dy%C2%9E%C2%B7%C3%8E%09!%C3%88%C2%B2%0C%C2%A5%14Zk%C2%AC%C2%B5%3F%07%C3%9F%C3%ADv%C3%A4y%C3%8E%C3%A1p%C2%B8X%24%C2%8Ec%16%C2%8B%05%C3%8E9%C3%96%C3%AB5%C3%8E%C2%B9%20%C3%B8%C2%97!S%C2%96%25UU%05%15%C2%81%0F%17%C3%AA%C2%BAFk%C3%9D%C2%98%0F%C3%8A%C3%B6%C2%B3%C3%8D%5D%C3%80%C2%9F%C2%A5%C2%94%0A%5E%C3%9B%C3%A8%5Ck%C2%8D%C3%B7%C2%BE%C2%B1%C2%B0%C2%97l%C2%9F%C3%8F%C3%A7%18c%1A%C3%8F%C2%B9%C2%97l%C2%97R%C2%92e%19%C2%93%C3%89%24%18%C3%B6%5D%C2%B5%5E%C2%B8sz%C2%8D%C3%87c%C3%8A%C2%B2%C3%A4t%3Au*%1A%C3%AA%C3%92%C3%85W%C2%AAR%0A)%25%C3%86%C2%98N%C3%B0P%C3%9D%C2%94p%C2%B7%C3%AA%C3%AF~%C2%BDF%C2%BC%C3%BE%C2%B4%0D%C2%A1M%04%C3%8C%068%C3%80%06%C2%98%C2%BD%00%12W%C2%957%5E\'%C3%8D%5D%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01nIDATH%C2%89%C3%A5%C3%96%C2%AD%C2%8E%C2%83%40%14%C2%86%C3%A1%C2%B7%C3%8D%26%C3%94%C2%9DU%C3%94U%22%C3%AB%C3%86u%C2%B2%069%C2%A2%02WY%C3%97%3B%C3%A0%0A%C2%B8%C2%8AJ%7CE%0D%C2%92%C2%80%C3%83q%09%C2%B8%C3%96%C2%8D%C2%AB%C3%9B%15%C3%9Dl%C3%92%C3%B4%0F%0A%0B%C3%99%C3%AC%C3%88%C3%839y%C3%A0%23%C3%8C0J%C2%92%C3%A4%1D%C3%98%01%1F%C3%B4%C2%B7R%609%1E%00%C3%A6%C3%9B%C3%9B%C2%8D%07%C2%80%7Fn%60%3C%10%0C%C3%80%C2%A0%C3%B8%C3%9B%C2%A3%C2%8BQ%14%5D%C3%95%C3%820%C3%AC%C2%AC%C3%BF%C3%BF%C3%86%C3%BE%C2%B7%C3%B0%C2%AA%C2%AA%3A%C3%83GI%C2%92%7C%3Ek%C2%8A%C2%A2%C2%880%0C)%C3%8B%C2%92%3C%C3%8F%C3%99l6%C2%B5%C3%BA%C2%9F%C2%ADFO%3E%C2%9F%C3%8F%11%11%C2%B2%2Ck2%C3%96%0D%0E%60%C2%8C%C2%A1(%0A%0E%C2%87C%C3%BF%C2%B8%C2%88%C2%A0%C2%B5f%C2%BF%C3%9F%C3%B7%C2%8F%03(%C2%A5%C2%98L%26%C2%AD%C3%A3%7F%C3%B9S3%C3%86%C2%90%C3%A79%C3%96%C3%9A%C3%BEq%11a%C2%B1X%C2%B4%C2%8A%C2%BF%C3%95%26%C2%A3%C2%B5%C3%A6t%3AQ%14E%C3%BF8%C2%9C%C3%A3%C3%8F%C2%B2%C3%AC%C2%A5%C3%B8%5B%C3%A3%C3%93%C3%A9%14%C2%A5%C3%94K%C3%B1w%C2%B2%C2%B7k%C2%AD%C2%B1%C3%96R%C2%96e%C2%A3%C2%B9%C2%9B%C3%9B%C3%ABv%C2%BB%C3%A5x%3C%C3%9E%1Dr%5D%C2%97%C3%B5z%7DQ%C2%AB%C2%AA%C2%8A8%C2%8E%C2%AFz%1B%C2%9F%C3%A7A%10%C3%A08%C3%8E%C3%8D%01%C3%87q%08%C2%82%C3%A0%C2%AA%3E%C2%9B%C3%8D%C3%AE%22%C2%8Dp%11%C3%81%18ss%C3%80%18%C2%83%C2%884%C2%86j%C3%A3%00%C2%9E%C3%A7%C2%A1%C2%94%C2%BA%C2%A8)%C2%A5%C3%B0%3C%C2%AF%13%C3%B8!%0E%C3%A0%C3%BB%3E%C2%AE%C3%AB%02%C3%A7%C3%B7%C3%AC%C3%BB~g0%C3%948%C3%8F%C2%AD%C2%B5%C3%84q%C3%8Cj%C2%B5%C3%AA%2C%C3%AE%C3%9A%C3%B8o%C2%AE%C3%81%C3%BF%C3%A1%C3%92%C2%81%C3%ACt%0C%2C%07%C2%B8%C2%81%14X~%01J%C3%93s%C2%A1A%C3%B0my%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%ACIDATH%C2%89%C3%AD%C3%94!%C2%8E%C2%840%14%C3%86%C3%B1%C3%BF%C2%90%15%C3%88uu%C3%9C%00%C2%AE%C2%B0%17%40%C2%90%209%09\'%C3%A0%14H%1CbDM%C2%93%1A47%C3%A0%08%C2%B8\'Q%3Bf%C2%B3%C2%A6%C3%ACl!3%C3%93%11%C3%BDd%C3%9B%C3%B4%C3%B7%C3%9A%C2%BC%C3%B6b%C2%8C%C3%B9%04%C2%AE%C3%80%17%C2%AF%C3%8B%04TI%00%C2%98%1F%C3%AF%C2%9A%04%C2%80%7F%0BH%02%C3%81%00D%3C%C3%A2%11%7F%0F%5CD%C3%A8%C3%BB%C3%BE)%C3%B8%C3%87%C2%BD%C3%89eY%C3%90Z%C2%B3m%C3%9Bkqk-%C3%B3%3C%C2%9F%C3%9A%C2%B4%C3%AB%3Ag%C2%ACm%C3%9B%C3%BFq%11a%1CG%C3%96u%3D%05%1F%C2%89%C2%83%0F%C3%83%C2%80%C2%888%0B%C3%B7N%C3%B3p%C2%BC%C2%AEk%C2%B4%C3%96%C3%8E%C3%89%C3%B7%C2%AE%C3%AD%C2%AF%C3%B8%16%C3%AAt%C2%BBR%C2%8A%C2%A6i%C3%88%C3%B3%C3%9C%1B%3B%C2%9B%C3%9D%C2%86K%C3%93%C2%94%C2%B2%2C%C3%89%C2%B2%0Ck%C3%AD%C3%A1n%C3%B7%C2%BD%C2%A5%C2%BBO%C2%AD(%0A%C2%94Rh%C2%AD%0F%C3%A1%C2%BE%C2%B9%18c%C2%BE%C2%9F%C2%B2%C2%B3G%C3%9E%C3%BB%7B%C2%8Dx%C3%84%23%C3%BE(%7C%0AdO%09P%05(%60%02%C2%AA%1B%C3%9C%02D%C3%ABE%C2%BE%15H%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%8CIDATH%C2%89%C3%AD%C3%97%C2%B1%0D%C3%83%20%10%C2%85%C3%A1%C3%9FL%C2%90%C2%8D%C2%B2%C2%82%C3%97%C3%B1%04l%C3%A0)%C2%AEpA%C2%83D%C3%A3%C2%9A%0D%18%C2%81%C3%AEJ%C2%AA%C2%A4H%14%25J%1A%C3%A3%C3%98%14%C3%B6%2BA%C3%A8%3B%01%3AD%C3%A7%C2%BD%C2%BF%00%13pe%C2%BF%C3%8C%40o%1A%C3%80%3C%C2%BD%C3%894%C2%80_%05%C2%98F0%00%C3%95%C2%B8%C2%AA2%C2%8E%23%C2%AA%C2%BA%3F.%22%C2%A8*%22%C2%B2%2F%1EB%20%C3%A7%0C%40%C3%8E%C2%99%10%C3%82%3ExJ%C2%89%18%C3%A3%C3%87X%C2%8C%C2%91%C2%94%C3%92%C2%B6%C2%B8%C2%AA%C3%A2%C2%9C%C3%BB9%C3%A7%C2%9C%5B%7C%C3%BE%C2%8Bp%11%C2%A1%C2%94%C3%B2s%C2%AE%C2%94%C2%B2%C3%B8%C3%BC%3B%C3%AF%C3%BDm%C3%91%C2%8A%C2%B7Xk%19%C2%86%C2%A1vy%C3%BDm%C3%BFGN%C3%BC%C3%84O%C3%BC%C3%847MU%7B%C2%B5%C3%96~%C2%8D%C3%95%C2%B4%C3%99%C3%A3n%C3%BBq%C3%B1U%C3%AF%C3%B9%C3%9A%1Cw%C3%9B%0D%C2%8FO%5B%C2%8B%C3%8C%06%C3%A8%1B%140%03%C3%BD%1D%18%3EO%C2%B5%C3%86%C2%99T%09%00%00%00%00IEND%C2%AEB%60%C2%82')
        },
        rollover: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%8FIDATH%C2%89%C3%8D%C2%97%C2%B1%C2%AA%C3%AA%40%10%C2%86%C3%BF%13%0Cf%C2%9B%10%C2%B1Q%10.%C2%96v%C3%87tZ%C2%A64%C2%9Dy%C2%84%C2%94%C3%9Aii%C3%A9%23h%C2%99R%C2%B0%C3%91N%C3%8B-%C2%B5%13%C3%9F%C3%80F0%C2%8DD%C3%92%C2%8C%C2%B0%C3%A0%C2%BD%C3%8D%C3%B5r%C3%A0D%C3%9D%C2%98%C3%A3z%C3%BFr%C2%98%C3%99o%C3%BE%C3%892d%3F%C2%96%C3%8B%C3%A5%2F%22%C3%9A%0A!%2C(%C2%92%C2%AE%C3%AB\'%C3%86%C3%98g%C2%8E%C2%88%C2%B6%00%C2%ACb%C2%B1%08%C3%86%C3%98%C3%8B%C3%81D%C2%848%C2%8E-%22%C3%9AjB%08%C3%8B4M%25%60%00%60%C2%8C%C3%814M%08!%2C%C3%AD%1AP%C2%A9%7C%3E%0F%00%C3%90%C2%94R%C3%BFJ%C3%93%C2%B4%C3%B7%C3%81%C3%BF5%C3%B1Nx.)8%18%0C%C2%BE%C3%85%C2%86%C3%83%C2%A1%C3%B4%C2%A1%C2%B2%C3%B5%C3%BF%C2%9F%C3%B3%24%25%C2%B9Q%06%C3%8F%3A%C3%B6%24%C3%BD%C3%B8%C3%987%C2%9B%C2%8Dt%C3%AE%C3%87t%3A%C3%BD%5D%C2%A9T2C%C2%A3(%C3%82%7C%3E%07%11%C3%81u%5DT%C2%AB%C3%95%C2%BB%C3%B9%C3%BB%C3%BD%5E~%C3%AC%C3%B7%C2%B4Z%C2%AD%C3%809G%C2%B3%C3%99%C2%84%C3%A38%C3%92u%C2%99%C3%A0_%C3%9D%C3%BA%C2%BE%C2%8Fr%C2%B9%C2%9C%C2%AA%C3%BEi8%C3%A7%1C%C2%9Cs8%C2%8E%C2%93%C3%8Am%26%C3%B8%C3%A1p%C3%80l6%03c%0C%C2%BD%5E%0F%C2%85B%C3%A1)0%C2%90%C3%B2%C2%B6s%C3%8E%11%04%01l%C3%9B%C2%86%C3%AF%C3%BB7%C3%81%C2%A3%C3%91%08Q%14%3D%3CO%C3%8A%C3%B9n%C2%B7%C3%83b%C2%B1%00c%0C%C2%9DN%C3%A7%C2%A1%C3%9B0%0C1%1E%C2%8F%C3%A1y%1Ej%C2%B5Z%3Ax%C3%92%C2%92h%C2%B7%C3%9B%C2%B0m%5B%C2%A6W%00%C3%80%C3%B9%7C%C3%86d2A%C2%A3%C3%91%C2%80%C3%AB%C2%BA%C2%899%C3%92cO%03%C3%BE%C2%AA%C3%B5z%7D%C3%B33(%C3%99%C3%ADa%18%22%08%02%C3%B4%C3%BB%C3%BD%C3%A7%C3%A0Yv%7B%C2%A9T%C2%82%C3%A7y%C3%9F%C3%B2~d%C3%83%C3%9DS%C2%BD%5E%C2%87%C3%AB%C2%BA0%0CC%0E%C2%9E%C3%86%C3%A5-%19%C2%86%C2%81V%C2%ABu%C3%B7%C2%AE%C2%BC%C3%84%C3%B9u%C3%8C%C2%8F%C3%96%C3%ADK%C3%A0%C3%9DnW*%C3%AF%C3%BD%7F%C2%AFD%C2%A4%14z%C3%A5i%C2%BA%C2%AE%C2%9F%C3%A28V%C3%92%C3%80%C3%A5r%C2%B9%C2%BE%C3%95%C2%A0%C3%AB%C3%BA)%C3%87%18%C3%BB%24%C2%A2%C3%AD%C3%B1xT%C3%BEJ%C3%BD%03%24%1D%C2%AFi%C3%A2%C3%A5l%2F%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%8AIDATH%C2%89%C3%AD%C2%97%2F%C2%AF%C2%A3%40%14%C3%85%C3%8F%23%C2%9D%C2%941%C2%84I%0DM%C2%9A%C2%BCT%C3%A2%5Ep%20%C2%B18%C2%90Hle%25%C2%B2%1F%C2%A1%C2%B6%C2%BE%C2%A6%C2%B8V%22%C2%91M%C2%BFA%0DIk%1A%1A%C3%8C%C2%88I%C2%BA%2B%C2%BA%C2%88%C3%AE%C2%B6%C3%A4u%60g%C3%8D%1Ey%C3%A7%C3%9E%C3%BC%C2%983%C2%97%C3%B9%C3%B3%C2%B1%C3%9B%C3%AD%3E9%C3%A7%07!%C2%84%09E%22%C2%84%5C)%C2%A5_%03%C3%8E%C3%B9%01%C2%809%1A%C2%8D%40)%C3%BD%C3%AB%60%C3%8E9%C3%AA%C2%BA69%C3%A7%07M%08a%1A%C2%86%C2%A1%04%0C%00%C2%94R%18%C2%86%01!%C2%84%C2%A95%01%C2%95%1A%0E%C2%87%00%00M)%C3%B5%C2%974%C3%AD%C2%8E%1D%C2%B4%25%C2%A5i%C3%BAGl%C2%B1X%C3%B4%C2%96%C3%BFOf%C3%BE%1F%C3%9E%1B%C3%BCx%3C%C2%BE%5D%C3%93%C3%9ApM%C2%B3%C2%A4i%C3%9A%C3%9A8%00%C2%B0%C3%9Dn%11%C2%86!%1C%C3%87%C3%B9V%3E%C3%90%C3%93%C3%8C%C3%B3%3C%07%C2%A5%14%C2%8E%C3%A3%C2%BCU%C3%97%19~%3A%C2%9DP%14%05%C3%820%7C%C2%BB%C2%B63%7C%C2%B3%C3%99%C3%80%C3%B7%7D0%C3%86%C3%94%C3%82%1B%C2%BB%3D%C3%8F%C2%93%C2%AA%C2%97%C2%86WU%C2%85%3C%C3%8F%C2%A5%C3%AC%C3%AE%0C%C3%8F%C2%B2L%C3%9A%C3%AEN%C3%B0%C2%A2(%C3%809%C2%87%C3%AF%C3%BB%C3%92%60)xcw%14E%C2%9D%C3%80R%C3%B0%2C%C3%8B%C3%A0y%1E%C3%86%C3%A3%C2%B1Z%C3%B8~%C2%BF%C3%AF%C3%85%C3%AEF%1F%C3%AB%C3%B5%C3%BA%C3%87d2y%3A%C3%B8%C3%AC%7CN%C2%92%04%C3%93%C3%A9%C3%B4!%C2%B6%5C.q%3E%C2%9F_B%2C%C3%8B%C3%82l6%7B%C2%88%C2%95e%C3%B9%C2%BE%C3%AD%C2%BF%C2%83%01%20%C2%8Ec%C3%A8%C2%BA%C3%BE4_%C3%97u%C3%84q%C3%BCt%C2%AC%C2%97%C2%BD%C2%9D1%C3%B6%C2%B2%01%C2%A3(z%C3%B9%3B%C3%B6v%C2%A4%C3%9A%C2%B6%0D%C3%97u%1Fb%C2%AE%C3%AB%C3%82%C2%B6%C3%AD%C2%975%C2%BD%5E%26%C2%82%20%C2%80eY%00%C3%AE%C3%AB%1C%04Ak~k%C3%83%C3%89%C2%A8%C2%AA*%C2%ACV%2B%24I%C3%92%C2%BA%C3%BB%C2%95e%C3%99~%C2%99%C2%90%11c%0C%C3%B3%C3%B9%C3%BC%5B%C2%B9%1Ap%7F%C3%82%C2%A8T%C3%83%C3%93%08!%C3%97%C2%BA%C2%AE%C2%95%7C%C3%80%C3%ADvk%C3%9Ej%20%C2%84%5C%07%C2%94%C3%92%2F%C3%8E%C3%B9%C3%A1r%C2%B9(%7F%C2%A5%C3%BE%04%C3%A2%C3%A0%C2%A47b%C3%82%C3%B5%C3%83%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%015IDATH%C2%89%C3%AD%C2%971%C2%8A%C3%830%10E%C3%BF%0A%C2%8BX%C2%8D0%C2%B8I%C2%AA%C2%BDB%C3%AC%C3%8EW%C2%B0J_%C3%83e%C3%8A%C2%949BR%C2%A6%C3%8C%15r%C2%85%C2%B4%C3%AE%7C%055%C3%86F%C3%8D%18%04%26%C3%8Dz%C2%8B%C2%8D%13%C2%9C%C2%80%C2%95%C3%9D%C3%85%C2%BF%1C%3E%C2%BC%3F3B0%1F%C3%A7%C3%B3%C3%B9%C2%93%C2%88%0Akm%00G%C3%A2%C2%9C7B%C2%88%C2%B5GD%05%C2%80%20%0CC%08!%26%07%13%11%C2%8C1%01%11%15%C3%8CZ%1BH)%C2%9D%C2%80%01%40%08%01)%25%C2%AC%C2%B5%01%C3%AB%0B.%C2%B5X%2C%00%00%C3%8C)%C3%B5K%C2%8C%C2%B1%C3%B7%C3%81%C2%BFC%C3%8C%C3%B0%19%C3%BE%C3%A7%C3%A1%C3%BB%C3%BD%1Eu%5D%C2%BF%07%C2%AE%C2%B5%C3%86%C3%A1p%40Y%C2%96%0F%7D%C3%9EPq%C2%BB%C3%9D%C3%9E%C3%94v%C2%BB%C3%9DS%01%C3%9A%C2%B6%C3%85%C3%A9tB%C2%92%24PJ%0Dz%26%C3%9F%C3%B9%C3%A5r%C2%B9%C2%BB%C2%86%C3%81%C3%8E%C2%8744%C2%8D%C2%B1%C3%92Z%C3%A3x%3Cb%C2%B3%C3%99%C2%BC%06%7Ff%C3%AC%3F%C2%83.%C2%97KdYv%C3%A3%1B%0D%7FUQ%14A)%05%C3%9F%C3%B7%C3%87%C3%81%C2%9F%7D%5CC%C3%B2%7D%1Fi%C2%9A%22%C2%8E%C3%A3%C2%BB%C2%9EI%3A%C3%AF%C3%87%C2%BCZ%C2%AD%1E%C3%BA%26%C2%81%C3%A7y%3E%C3%8A%C3%B7%C3%BF%C2%BE%C3%97%19%3E%C3%83%7F%2F%C2%9C%C2%88%C2%9CB%7B%1E%C3%A3%C2%9C7%C3%86%18\'%01%C2%BA%C2%AE%C3%ABo5p%C3%8E%1BO%08%C2%B1%26%C2%A2%C2%A2%C2%AA*%C3%A7W%C3%AA%15%10%C3%96l%1F%C2%9B%C3%82%C3%B9j%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%015IDATH%C2%89%C3%AD%C2%971%C2%8E%C2%84%20%18%C2%85%C3%9F%1A%C2%89%C3%92%18%C2%88%0D%C3%9D%C3%96t%C3%93r%06KK%C2%AE%60ii%C3%A9%11%3C%C2%8B%C3%A7%C3%B0%186%26%13%C2%9A%C2%BF%201%5B8%263%C3%99)%16v%C3%87Mv%7D%C3%A5O%7C%1F%C3%B0%C2%BF%08%C2%BC%C2%8D%C3%A3%C3%B8ND%C2%93%C3%B7%5E%C3%A0%201%C3%86%C2%AE%C2%9C%C3%B3KJD%13%00Q%C2%96%258%C3%A7%2F%07%13%11%C2%9Cs%C2%82%C2%88%C2%A6%C3%84%7B%2F%C2%8A%C2%A28%04%0C%00%C2%9Cs%14E%01%C3%AF%C2%BDH%C3%B6%C3%82%C2%91%C3%8A%C2%B2%0C%00%C2%90%1CJ%C2%BD)I6l%1A%C3%B3q%C3%97u%C2%9Fj%7D%C3%9F%C2%87O%22%06%C3%BES%3A%C3%A1%C2%BF%C2%A2%C2%A8%C3%80%C3%AD%C3%A1%C3%AA%C2%BA.*h%C2%BB%C3%BE%C3%AF%C2%B6%C2%9F%C3%B0%13~%C3%82%C3%BF%3E%3C%C3%A8%C3%B7%3A%0C%03%C3%A6y~%C2%A8%C3%9D%C2%9F%C3%ADJ)4M%C3%B3e%C2%BF%C2%A0%C2%95%5Bk%C2%91%C3%A7%C3%B9%C3%93%C2%B1%3C%C3%8Fa%C2%AD%0D%C2%B1%0B%C2%83K)Q%C3%97%C3%B5%C3%93%C2%B1%C2%BA%C2%AE!%C2%A5%7C%1D%1C%00%C2%B4%C3%960%C3%86%3C%C3%94%C2%8C1%C3%90Z%C2%87Z%C3%85%05%C2%AE%C2%AA*(%C2%A5%00l%7D%C2%AE%C2%AA*%C3%86%26%3E%C3%AD%C3%96Z%08!%C2%82%C3%BB%7C%C2%AF%C2%A8%C3%8B%04%C2%B0%C3%B5%C2%BFm%C3%9Bh0p%5B9%11%7D%C3%8B%24T%3B%2Fa%C2%8C%5D%C2%9Ds%C2%87L%60%5D%C3%97%C3%BD%C2%AD%06%C3%86%C3%985%C3%A5%C2%9C_%C2%88hZ%C2%96%C3%A5%C3%B0W%C3%AA%07%3E%C2%A4Yh%C3%AEH%C3%9A%C2%B4%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%A4IDATH%C2%89%C3%8D%C2%97!%C2%AF%C3%A2%40%14%C2%85%C3%8Fkh%60L%C3%93%06%C3%95%C2%84d3%12%C3%B7%C2%A8D%C3%B6\'%C2%80D%C2%82%03%07%12%C2%89%04%C3%97%C3%8Az%C3%8CH*%C2%91%C3%A0%1A%5C%C3%A5%18%120%C2%84%C2%A6fH%C2%9A%C2%B0%2B%C3%B6uw%C2%93v%C3%99%C3%A9%C2%B2o%C3%98%23%C3%AF%C3%9C%C3%9Eo%C3%AE%C2%99%C3%A9M%C3%A6-%0C%C3%83%2FB%C2%88C%C2%96e%26%14I%C3%97%C3%B5%C2%84%10%C3%B2%5E%13B%1C%00%C2%98%C3%8Df%13%C2%84%C2%90O%07%0B!%C2%90%C2%A6%C2%A9)%C2%848hY%C2%96%C2%99%C2%86a(%01%03%00!%04%C2%86a%20%C3%8B2S%C3%8B%03*U%C2%AF%C3%97%01%00%C2%9AR%C3%AA%C2%874M%7B%1D%C3%BC%C3%87%26%5E%09%C2%AF%C2%95%05%C3%A7%C3%B3y!%C2%B6X%2C%C2%A4%C2%8B%C3%8A~%C3%BF%C3%BFu%5E%C2%A6%C2%B2n%C2%94%C3%81%C2%9F%C2%B5%C2%BDL%C3%92%C2%B6GQ%24%0D%C2%97%C3%95%C3%9Bz%C2%BD%C3%BE%C3%9Aj%C2%B5%1E%26q%C3%8E%C3%81%18%C2%83eY%C3%A8%C3%B5z%C2%B0%2C%C3%ABi%C3%B0%C3%B1x%C2%94%C3%AB%C2%9CR%C2%8A%C3%99l%06J)%7C%C3%9F%C3%87n%C2%B7%7B%1A%0ET8s%00p%5D%17%C3%ADv%1B%C2%8C1%C3%84q%C3%BC%C2%B4%0B%C2%95%7F5%C3%9B%C2%B61%C2%99L%40)%C3%85j%C2%B5%C3%82v%C2%BBU%07%C3%8F%C3%A5%C2%BA.%C2%A6%C3%93)8%C3%A7%C3%B0%3C%0F%C2%A7%C3%93%C3%A9%C3%9F%C3%83%C2%AF%C3%97%2B%3C%C3%8F%2B%5D%C2%B3%2C%0B%C3%83%C3%A1%10%C2%8E%C3%A3%20%08%C2%82%C3%8A.%3C%C2%84%C3%87q%0C%C3%9F%C3%B7q%3E%C2%9F%1F%16%C3%A9v%C2%BB%18%C2%8F%C3%87%C3%A0%C2%9Cc%C2%B9%5C%C2%82s.%05%C3%BF%C3%AD%C2%85%C3%9Bl6%C3%98%C3%AF%C3%B7RE%C2%80%C2%9F.DQ%C2%84%20%08%0A%C3%ABR%C2%B3%3D%C2%B7%C2%B9%0A%C3%B8W9%C2%8E%23%C2%9D%5B%C3%A8%3C%08%02%24IRHT2%C3%9B%07%C2%83%01%18c%C2%85sV2%C3%9Bm%C3%9B%C3%86h4B%C2%A7%C3%93%C2%91%C2%86%C3%BD%C2%ADJ%2F%5C%C2%A3%C3%91%40%C2%BF%C3%9F%07%C2%A5%14a%18%C3%A2v%C2%BBU**%C3%AB%C3%92%C3%83%C3%B1%C3%AA8%0El%C3%9B%06c%C2%AC%12%5CV%7F%1C2%C3%B98%7D%09%C3%BC3%C2%A5%01%C3%9F%C2%9F0*%C2%95%C3%B34%5D%C3%97%C2%934M%C2%95l%C3%A0~%C2%BF%C3%A7o5%C3%A8%C2%BA%C2%9E%C3%94%08!%C3%AFB%C2%88%C3%83%C3%A5rQ%C3%BEJ%C3%BD%06%C3%B15%C2%BD%22%C2%BF%C3%A3%C2%A3J%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%92IDATH%C2%89%C3%AD%C2%97!%C2%8F%C2%AB%40%14%C2%85%C3%8F%C2%92N%C3%8A%18%C3%82%C2%A4%C2%86%26M6%C2%95%C2%B8M%1D%C3%88%C2%B18F%22k%2B%2B%C2%91%C3%BC%C2%84Z%7CM%2B%2B%C2%91H%C3%92%7F%C2%80i%C3%92%C2%9A%C2%86%063%C2%82%C2%A4o%05%5B%C2%B1iaK%C2%977%C3%8F%C2%BC%23o%C3%AE%C3%8D7sf%C2%B8%C3%8C%7D%C3%9B%C3%ADv%C3%AFR%C3%8A%7DUU%26%14%C2%89%10r%C2%A1%C2%94~%0C%C2%A4%C2%94%7B%00%C3%A6h4%02%C2%A5%C3%B4%C2%AF%C2%83%C2%A5%C2%94(%C3%8B%C3%92%C2%94R%C3%AE%C2%B5%C2%AA%C2%AAL%C3%830%C2%94%C2%80%01%C2%80R%0A%C3%830PU%C2%95%C2%A9%C3%9D%02*5%1C%0E%01%00%C2%9AR%C3%AA%C2%974%C2%AD%C3%86%0E%C3%9A%C2%92%C3%820%C2%BC%C2%8BEQ%C3%94%5B%C3%BE%3F%C3%99%C3%B9%7Fxgx%C2%9E%C3%A7%C2%BD%C3%81%C3%9F%C3%96%C3%AB%C3%B5%C2%9F%C3%89d%C3%92%C2%9A%14%C2%86!%C2%A2(B%C2%96eH%C2%92%04%C3%8B%C3%A5%C3%B2%C2%A9%C3%BC6%1D%0E%C2%87n%3B%C2%9F%C3%8Df%60%C2%8C!I%C2%92.e%C2%8D%C3%AAl%C2%BB%C3%AF%C3%BBH%C3%93%14%C3%87%C3%A3Q%3D%C2%9C1%06%C3%8E96%C2%9B%C2%8Dz8%00%C2%B8%C2%AE%0BJ%C3%A9%C2%AF%C3%AD%7F%C3%B9S%C3%B3%7D%1FI%C2%92%C2%A0(%0A%C3%B5%C3%B0%C2%9B%C3%BD%C3%9B%C3%ADV%3D%1C%008%C3%A7%C2%90R%22MS%C3%B5p%00%10B%C2%BCl%C3%BF%C2%AF%C3%A1%C3%A3%C3%B1%18%C2%AE%C3%AB%C2%BEd%7F%2F%C2%BD%C2%9Ds%C2%8E%C2%A2(%C2%90eY%C2%A7%C2%BA%C2%87%C3%ADu%C2%B5Z%C3%A1t%3A5%16Y%C2%96%C2%85%C3%85b%C3%B1-%C2%96%C3%A79%C3%A28%C2%BE%C3%8Bmj%C2%B3%C2%8D%C3%AD5%08%02%C3%A8%C2%BA%C3%BE%C2%B0H%C3%97u%04Ap%17%C2%9FN%C2%A7%C2%8D%C2%8Bm%C3%92C8c%0CB%C2%88%C2%87%05B%080%C3%86%3A%C2%83%C2%9E%C2%86%03%C2%80m%C3%9Bp%1C%C3%A7%5B%C3%8Cq%1C%C3%98%C2%B6%C3%9D%0B%C2%B8%15%0E%00%C2%9E%C3%A7%C3%81%C2%B2%2C%00%C3%B59%7B%C2%9E%C3%97%1B%18x%C3%A2%7F%5E%14%05%C3%A28%C3%86%7C%3E%C3%AF%C3%8Dn%C2%A0%C2%BEp%C2%AD%C2%AFW%C2%A0%3E%C3%BF%C2%9F%1E%0F%C2%AFJ%03%C3%AA%11F%C2%A5n%3C%C2%8D%10r)%C3%8BR%C3%89%02%C2%AE%C3%97%C3%ABmV%03!%C3%A42%C2%A0%C2%94~H)%C3%B7%C3%A7%C3%B3Y%C3%B9%C2%94%C3%BA%09%02Y%C2%AD%C3%B4%C3%91%175%1D%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%3AIDATH%C2%89%C3%AD%C3%971n%C3%820%14%06%C3%A0%C2%BF%16%16x%C2%B1%22e%C3%B2Tq%03%C3%88U2%C2%B22r%C2%84%C2%8C%1C!kFF%C2%8F%C3%A4%0A%C2%AC%C3%99r%05%C2%96%C3%88%C2%96%C2%97%17%C3%89%12bi%C2%BA%24%C2%85%C2%80%C2%8A%C2%A9%C3%94%C3%BC%C2%A3c%C3%B9%7B%C2%B6%C2%9E%23%C3%B9%C2%A3%2C%C3%8BO%22%C2%AA%C2%BC%C3%B7%11%02%C2%85sn%C2%85%10%C2%AB%19%11U%00%C2%A28%C2%8E!%C2%84x9LDp%C3%8EEDT1%C3%AF%7D%24%C2%A5%0C%02%03%C2%80%10%02RJx%C3%AF%23%C3%96%0D%C2%84%C3%8C%7C%3E%07%00%C2%B0%C2%A0%C3%AAW%18c%C3%AF%C3%83%C2%BF%C2%8B%C2%98%C3%B0%09%C3%BFS%C2%B81%06y%C2%9E%C2%BF%04%C2%9F%C3%9D%C3%BAX%C3%975%C2%B4%C3%96h%C3%9B6%2C~%3C%1Eq%3A%C2%9D%C2%9EZ4%C3%8B%C2%B2%C3%9E%C3%98~%C2%BF%C2%BF%C2%8F%1Bcp8%1Cp%3E%C2%9F%C2%9F%C2%82%1FI%0F%2F%C2%8A%02%C3%96%C3%9A%C3%9E%C3%84%C2%A1%C3%9D%C3%BC%3A%C2%BE%C3%99l%C2%A0%C2%B5%C3%AE%C3%AD%7C%C3%A8%C3%98~%C3%8A%C3%98B%7B%C3%9D%C2%AE%C2%94%C3%82v%C2%BB%C3%85z%C2%BD%1E%C2%8D%3D%C2%9B%C3%81%C2%86%5B%2C%16H%C3%93%14%C3%8B%C3%A5%12eY%3E%C3%9C%C3%ADcO%C3%A9%C3%A6UK%C2%92%04J)h%C2%AD%1F%C3%82%C3%87%C3%A6%C3%AEOF)%C2%85%C3%9Dn%C3%B7%1E%C3%BC%C2%95%C2%99%C3%B0%09%C3%BFg8%11%05E%3B%C2%8Fq%C3%8E%C2%ADs.H%01%C2%97%C3%8B%C2%A5%7B%C2%AB%C2%81sngB%C2%88%15%11UM%C3%93%04%7F%C2%A5%5E%01%C3%9D%C2%BBu%C2%A0%15%C2%87h%C2%94%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%1BIDATH%C2%89%C3%AD%C2%971%C2%AA%C3%84%20%10%C2%86%C3%BF\'%C2%91%60%23%C2%81%1C%C3%A0%5Da%0F%C2%96%23Xn%C2%99kXZ%26WH%C2%9B%C3%8E%2B%C2%A4%09%C2%8A%C3%8D%08BxU%60%C2%97%C2%B7%C2%8D%C3%99%17_%C2%B1%C3%B9%2B%C2%9D%C3%91%C3%B9dfP%C3%BC%1A%C3%87%C3%B1%C2%9B%C2%88%C3%A6%C2%94R%C2%83B%C3%A2%C2%9C%7B!%C3%84%C2%AD%22%C2%A2%19%40%C3%93%C2%B6-%C2%84%10%C2%A7%C2%83%C2%89%08!%C2%84%C2%86%C2%88f%C2%96Rj%C2%A4%C2%94E%C3%80%00%20%C2%84%C2%80%C2%94%12)%C2%A5%C2%86%C3%AD%C2%86%C2%92%C2%AA%C3%AB%1A%00%C3%80%C2%8E%06p%C3%8E%C2%A1%C3%AF%7B8%C3%A7%C2%B2%C3%B72%C3%86%C3%9E%C2%83k%C2%AD%C3%A1%C2%BD%C2%87%C3%96%C3%BAh%C2%88c%C3%B0a%18%C2%B0%2C%0B%00%60Y%16%0C%C3%83P%06n%C2%AD%C3%854MO%C2%B6i%C2%9A%60%C2%AD%3D%17%C3%AE%C2%9C%C2%831%C3%A6%C2%A5%C3%8F%18%C2%93%5D%C3%BF%2C%C2%B8%C3%96%1A1%C3%86%C2%97%C2%BE%18cv%C3%BD%C2%AB%C2%9C%C3%85%5D%C3%97%3D%C3%8D%C2%95R%C2%B8%C3%9F%C3%AFY%C3%80G%1D%C3%AE%C3%B6%C2%BF%C3%90%05%C2%BF%C3%A0%17%C3%BC%C2%82%C2%9F%C2%AA%C2%AC%C3%ABu%C2%97R%C3%AA%C3%97%C3%B8%C3%885%C3%BB%C2%B9i%C3%BF%5C%C3%B8%C2%A1%C2%86%7B%C3%A7%0D%7F%C3%94%C3%BF%C2%A7%C2%9D%C2%88%C2%8ABw%1E%C3%A3%C2%9C%C3%BB%10B%C2%91%03l%C3%9B%C2%B6%C3%BF%C3%95%C3%809%C3%B7%C2%95%10%C3%A2FD%C3%B3%C2%BA%C2%AE%C3%85%7F%C2%A9%3F%C3%94%C3%AF%7C%25%C2%95%11%C3%9D%C3%9E%00%00%00%00IEND%C2%AEB%60%C2%82')
        },
        pressed: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%83IDATH%C2%89%C3%8D%C2%97%C2%A1%C2%AE%C2%AB%40%10%C2%86%7F%06%C2%9A%C2%ADj0%C2%B8%26W%C3%94%C2%A1%0E%0E%C2%B7%C2%8E\'%C3%80%C3%A1x%C2%82%3EJ_%C2%82%C2%A4%02W%C2%87%C2%AB%5BGU%C2%9F%C2%A0%0E%C3%93%C2%A0J%C2%80%C3%A1%C2%8As%C3%88%3D%C3%89%C2%A1%C3%AD%C3%92%C2%9E%C3%92%C3%BB%1B%C2%92%C3%8D%C3%8C~%C3%B3%C3%8Fn%26%C2%AC%C2%91%C2%A6%C3%A9%1Ff%3E0%C2%B3%C2%8D%C2%89DDg%22%C3%BA%C2%B0%C2%98%C3%B9%C3%90u%C2%9D%3D%C2%9B%C3%8D%60%C2%9A%C3%A6%C3%8B%C3%81m%C3%9B%C2%A2i%1A%C2%9B%C2%99%0F%C3%84%C3%8C%C2%B6eY%C2%93%C2%80%01%C3%804MX%C2%96%05f%C2%B6%C2%A9_%C2%98RD%C3%B4%C3%B9%C2%9D%C2%94%C3%BA%25%C3%830%C3%9E%07%C3%AF%C3%B5V%C2%B85%C2%B4%C2%B8%C3%99l~%C2%AC%C2%AD%C3%97k%C3%ADMu%C3%B3%C3%BF%3F%C3%A7C%1Ar3%19%C3%BC%C3%99%C2%B6%0F%C3%A9%C3%97%C3%9B~%3C%1E%C2%B5c%C2%8D%C3%ADv%C3%9B%C3%8D%C3%A7%C3%B3%C2%A7%C2%A1eY%22%C3%8B2TU%05)%25%C2%96%C3%8B%C3%A5%C3%8D%C3%B8%C3%8B%C3%A5%C2%A2%C3%9F%C3%B6%5B%C3%8A%C3%B3%1CJ)x%C2%9E%07%C3%9F%C3%B7%C2%B5%C3%B3%C2%9E%C2%82%7Fw%1B%C2%86!%1C%C3%87%19%C2%95%C3%BF0%5C)%05%C2%A5%14%7C%C3%9F%1F%C3%A5%C3%B6)xQ%14%C3%88%C2%B2%0CB%08%C3%84q%C2%8C%C3%85b%C3%B1%10%18%18y%C3%9B%C2%95RH%C3%93%14%C2%AE%C3%AB%22%0C%C3%83%C2%AB%C3%A0%24IP%C2%96%C3%A5%C3%9D%C3%BD%C2%B4%C2%9C%C2%9FN\'%C3%AC%C3%B7%7B%08!%10E%C3%91%5D%C2%B7EQ%20I%12%04A%C2%80%C3%95j5%0E%3E4%24%C2%82%20%C2%80%C3%AB%C2%BA%3A%C2%B5%02%00%C2%AA%C2%AA%C3%82n%C2%B7%C2%83%C3%A7y%C2%90R%0E%C3%86h%C2%B7%7D%0C%C3%B8%C2%BB%C3%B2%3C%C2%BFz%0C%C2%93%C3%8C%C3%B6%C2%A2(%C2%90%C2%A6)%C3%A28~%0C%C3%BE%C3%8Clw%1C%07A%10%C3%BC%C2%88%C3%BB%C2%95%09wK%C2%AE%C3%ABBJ%09!%C2%84%1E%7C%C2%8C%C3%8Bk%12B%40Jy%C3%B3%C2%AE%C2%BC%C3%84y%C3%9F%C3%A6%7B%C3%A3%C3%B6%25%C3%B0(%C2%8A%C2%B4%C3%A2%C3%9E%C3%BF%C3%B7%C3%9A%C2%B6%C3%AD%C2%A4%C3%90%C2%9EGDtn%C2%9Af%C2%92%02%C2%BA%C2%AE%C3%AB%C3%9Fj%20%C2%A2%C2%B3ED%1F%C3%8C%7C%C2%A8%C3%AB%C3%9A%C2%AE%C3%AB%C3%BA%C3%A5%05%00%C3%BF%5E%C2%A9%7F%01%0E%C2%82%C2%B4%C3%A8%C2%B2\'l%C2%91%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C2%B1IDATH%C2%89%C3%AD%C2%971%C2%8E%C2%A3%40%10E%3F%05V%3BB%24%C2%9DY%C3%9A%C2%80%C2%8Ch%3A%23%23%23%23%23%C3%B3%25%7C%14.%C2%81%C3%A4%C2%80%13p%022%1C%C3%B9%04%C3%8EH%2C%22%2C%C2%A0%C2%98%60%C2%96Y%C3%8D%C2%8E%C2%8Dl%C3%A8a%C2%93%C3%BD%09Ru%C2%95%5Ew%C2%A9%C2%BA%C2%8A6%C2%B2%2C%C3%BB%C3%85%C3%8C\'fv%C2%B0%C2%92%C2%88%C3%A8JDo%163%C2%9F%C2%86ap6%C2%9B%0DL%C3%93%C3%BCqp%C3%9F%C3%B7%C3%A8%C2%BA%C3%8Ea%C3%A6%131%C2%B3cY%C3%96*%60%000M%13%C2%96e%C2%81%C2%99%1D%1A%0Dk%C2%8A%C2%88%3E%C2%BE%C2%ABR%7F%C3%8B0%0C%00%C2%805%C3%A5%C2%94%24%C3%897%C3%9B%C3%A1p%C3%90%C3%A6%C3%BFON%C3%BE%1F%C2%AE%0D~%C2%B9%5C%5E%C2%8E1%C2%8E%C3%87%C3%A3%C2%B0%C3%9Dn\'%C2%9D%C2%92%24%C2%99%2C%1C%00H%C3%93%14J)x%C2%9E%C3%B7%C2%94%7F%C3%934zN%5E%14%05%C2%84%10%C3%B0%3C%C3%AF%C2%A5%C2%B8%C3%85%C3%B0%C2%AA%C2%AAP%C2%96%25%C3%820%7C9v1%3C%C3%8Fs%C3%B8%C2%BE%0F%C3%9B%C2%B6%C3%97%C2%85%C2%8F%C3%A9VJ%C3%8D%C2%8A%C2%9F%0D%C2%AF%C3%AB%1AEQ%C3%8CJ%C3%B7b%C3%B8%C2%92t%2F%C2%82%C2%97e%C2%89%C3%9B%C3%AD%06%C3%9F%C3%B7g%C2%83g%C3%81u%C2%A4%7B6%3C%C3%8Fs(%C2%A5%20%C2%A5%5C%17~%3E%C2%9F%C2%B5%C2%A4%7B%C3%94d%7B%C2%BD7%C2%9F%C3%A38%C3%86n%C2%B7%C3%BBbK%C3%93%14UU%3D%C2%84H)%C2%B1%C3%9F%C3%AF%C2%BF%C3%98f%C2%B5%C3%97%C2%BF%C3%81%00%10E%11%C2%84%10w%C3%BD%C2%85%10%C2%88%C2%A2%C3%A8%C3%AE%C2%9A%C2%96%C3%9En%C3%9B%C3%B6%C3%83%02%0C%C3%83%C3%B0%C3%A1u%C3%946R%5D%C3%97%C3%BD%C3%96%C3%A9%C2%94Rp%5D%C3%B7a%C2%8C%C3%96%C2%9F%C2%89%20%08%3Eo%C2%81%C2%94%12A%10L%C3%BA%3F5%C3%8F_Q%5D%C3%97%C3%88%C2%B2%0Cq%1COv%C2%BF%C2%A6i%C3%B4%C3%83%C2%9F%C3%95g%C2%B5%C3%B7%7D%C2%BF*x%C3%A4%11%11%5D%C2%BB%C2%AE%5Be%03%C3%830%C2%8Co5%10%C3%91%C3%95%22%C2%A27f%3E%C2%B5m%C3%AB%C2%B4m%C3%BB%C3%A3%1B%00%C3%BE%C2%BCR%C3%9F%01%C2%B8%3F%C2%B0%C3%AE%C2%A2%C2%85%C2%BA%C3%B2%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%2CIDATH%C2%89%C3%AD%C2%971n%C2%840%10E%3Fc%23Z7%C3%94%C3%A9%C2%A9%C2%96%13%C2%B8%C3%9BK%C3%90q%02%C2%8E%C3%82%25%C3%9C%C3%91q%10%C2%B8%00\'%C2%A0At%08%18o%C2%91%C2%A0HY%C3%AF%0AV%C2%82M%22~ci%C3%B4%C2%A5%C3%B7%3DcY%1A%C2%AF(%C2%8A%0Ff%C2%AE%C2%99Y%C3%A1%20%11QGD%17%C3%89%C3%8C%C2%B5%C2%B5V%C3%B9%C2%BE%0F!%C3%84%C3%AE%C3%A0y%C2%9E1M%C2%93b%C3%A6%C2%9A%C2%98YI)%0F%01%03%C2%80%10%02RJ0%C2%B3%C2%A2%C2%A5p%C2%A4%C2%88%C3%A8%C3%B3%3C%C2%94%C3%BA%25%C3%8F%C3%B3%C3%9E%07_t%C3%82O%C3%B8%C3%9F%C2%87%1Bc%C3%90%C3%B7%C3%BD%7B%C3%A0m%C3%9B%C3%82%18%C2%83%C2%A6i%C2%9E%C3%BA%C2%A4%C2%AB%C2%98%C3%A7%C3%B9%5D-%C3%8B%C2%B2M%01%C2%86a%40Y%C2%96%C2%88%C3%A3%18Zk%C2%A7g%C3%B7%C2%99WU%C3%B5p%0C%C3%8E%C2%9B%C2%BB%C3%A4%C3%AA%C3%86Z%C2%B5m%C2%8B%C2%A2(%C2%90%C2%A6%C3%A9k%C3%B0-m%C3%BF%194%0CC%5C%C2%AF%C3%97%3B%C3%9Fj%C3%B8%C2%AB%C2%8A%C2%A2%08Zk%04A%C2%B0%0E%C2%BE%C3%B5q%C2%B9%14%04%01%C2%B4%C3%96%C2%88%C2%A2%C3%A8%C2%A1g%C2%97%C2%9B%2Fm%0E%C3%83%C3%B0%C2%A9o%17x%C2%92%24%C2%AB%7C%C3%BF%C3%AF%7B%3D%C3%A1\'%C3%BC%C3%B7%C3%82%C3%A7y%3E%14%C2%BA%C3%B0%C2%88%C2%88%C2%BAi%C2%9A%0E%09%60%C2%AD%5Dv5%10Q\'%C2%89%C3%A8%C3%82%C3%8C%C3%B58%C2%8Ej%1C%C3%87%C3%9D%03%00%C3%9F%5B%C3%AA%0D%05%5Cq%C3%91nU%C2%93n%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%014IDATH%C2%89%C3%AD%C2%97%3Dn%C2%840%10%C2%85%1F%C2%83%C2%91%C2%A9%2C7%C2%AESp%C2%81%C3%A5%04tt%3E%01%C2%97%C3%A0(%C2%BE%04%12%05%C2%87%C3%A1%1E4%C2%94%2B%01%C3%A3%14%1B%C3%B2%C2%A3l%C2%A4%C2%98%15%C2%AC%C2%94%C3%B0%1AK3%C3%B2%C3%BB%C3%86%C3%B6%C3%88%C2%B2%C2%A3%C2%AE%C3%AB%5E%C2%98%C2%B9gf%C2%8D%C2%83DD%23%11%5D%043%C3%B7%C3%9E%7B%C2%9D%24%09%C3%A28%C3%9E%1D%C2%BC%2C%0B%C3%A6y%C3%96%C3%8C%C3%9C%133k!%C3%84!%60%00%C2%88%C3%A3%18B%080%C2%B3%C2%A65p%C2%A4%C2%88%C3%A86%1EJ%7DS%14E%00%00%C2%B1e%C2%B2s%C3%AE%5B%C2%AC%C2%AE%C3%AB%60%C2%9F%C2%A7%C2%AC%C3%BC%C2%84%3F%15%1E%C2%B5m%C3%AB%C3%934%C3%9D4%C3%999%C2%B7%C2%A9%C3%91%00%C3%A0z%C2%BD%C3%BE%C3%A3m%3F%C3%A1\'%C3%BC%C2%84%C3%BF%7Dx%C3%90%C3%B5%C3%9A4%0D%C2%86a%C3%B81o%C2%8CAUU%C2%BF%C3%B2%0A%C2%BE%5E%C2%AD%C2%B5%C2%90R%C3%9E%C3%8DI)a%C2%AD%0D%C2%B1%0B%C2%83%2B%C2%A5P%C2%96%C3%A5%C3%9D%5CY%C2%96PJ%C3%AD%07%07%C2%80%2C%C3%8B%C2%90%C3%A7%C3%B9%C2%97X%C2%9E%C3%A7%C3%88%C2%B2%2C%C3%94j%5B%C3%83%15E%01c%0C%C2%80%C3%9B9%17E%C2%B1%C3%85f%7B%C2%B7%5Bk%C2%A1%C2%94%0A%3E%C3%A7%C3%8Fz%C3%A81%C3%B1%C2%88%C3%9E%C2%BB%7DY%C2%96C%C3%81%2B%C2%8F%C2%88h%C2%9C%C3%A7%C3%B9%C2%90%02%C2%BC%C3%B7%C3%AB_%0DD4%0A%22%C2%BA0s%3FM%C2%93%C2%9E%C2%A6i%C3%B7%02%C2%80%C2%8F_%C3%AA%2B~%C3%AAj%C3%B0%C3%83%C2%BF%C3%AF%C2%B9%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%8BIDATH%C2%89%C3%8D%C2%97%C2%BF%C2%8E%C2%A30%10%C3%86%3F%06%C2%A2%C2%A4B4%C3%AE%2C%5DO%C2%B5%C3%A9%C3%B2%04t%C3%A9%C3%A8%C3%A8%C3%B2%04y%C2%94%3C%01%1DR%0A%C2%9E%C3%80%1D%5D%C2%BA%C2%A4%C2%A2K%C3%A7.MD%15%04%19%C2%AE%C3%98%C3%A3N%C2%BA%C2%90%C3%88%2C%C2%8Aw%C2%BF%06%C3%89%C3%98%C3%BEy%C3%BEx%C2%ACq%C3%B2%3C%C3%BF%C3%85%C3%8C\'f%0E%60IDt%25%C2%A2%0F%C2%8F%C2%99O%5D%C3%97%05%C2%B3%C3%99%0C%C2%AE%C3%AB%C2%BE%1D%7C%C2%BF%C3%9F%C3%91%C2%B6m%C3%80%C3%8C\'b%C3%A6%C3%80%C3%B3%3C%2B%60%00p%5D%17%C2%9E%C3%A7%C2%81%C2%99%03%C3%AA%07l%C2%8A%C2%88%3E%C2%BFV%C2%A9%7F%C3%A48%C3%8E%C3%B7%C3%81%7B%7D%2B%C3%9C%1B%1A%C3%9C%C3%ADv%0Fc%C3%9B%C3%AD%C3%96xS%C3%93%C3%B5%3F%C3%8F%C3%B2!%0DYc%0D%3E%C3%95%C3%ADC2v%7BY%C2%96%C3%86pS9%C3%BB%C3%BD%C2%BE%5B%2C%16%2F\'i%C2%AD%C2%A1%C2%94%C2%82%C3%AF%C3%BB%C2%88%C2%A2%08%C2%BE%C3%AFO%06%C3%9Fn73%C3%8B%C2%A5%C2%94%C3%98l6%C2%90R%22%C3%8B2%1C%C2%8F%C3%87%C3%89p%60d%C2%B6%C2%AFV%2B%C3%84q%C2%8C%C2%B2%2C%C2%91%C3%A79%C2%AA%C2%AA%C2%B2%07%07%00!%04%C2%92%24%C2%81%C2%94%12i%C2%9A%C3%A2p8%7C%19n%14%C3%B3g%C2%AA%C2%AA%0AJ)%C3%94u%C2%8D(%C2%8A%20%C2%840%5Ek%14%C3%B3%C2%AA%C2%AA%C2%90e%C3%99%C3%A0%3F%C3%9F%C3%B7%11%C3%871%C3%820D%C2%9E%C3%A7%C2%A3%C2%BD%C3%B0%12~%3E%C2%9F%C2%91e%19.%C2%97%C3%8B%C3%8BM%C2%96%C3%8B%25%C2%92%24%C2%81%C3%96%1Ai%C2%9ABkm%04%7FZd%C2%8A%C2%A2%18%C2%95%C3%95%C2%BD%17%C3%BAd%C3%BC_F%C2%B5%C2%BDw%C3%B3W%C2%AFS%18%C2%86%C3%86s%1F%2C%7Fv%C2%85%C2%AC%C3%94%C3%B6%C3%B5z%0D%C2%A5%C3%94C%C2%9C%C2%AD%C3%94v!%C3%84%C3%9F%0C~%C2%B7%06%13n%3E%C2%9F%23%C2%8A%22H)Q%14%05%C3%AA%C2%BA%1E%C2%B5%C2%A9%C2%A9%C2%97%5E%3E%C2%A9a%18B%08%01%C2%A5%C3%94(%C2%B8%C2%A9%26U%C2%B8)2~%C3%95%C3%9E%25%02%3E%5B%18%C2%9B%C3%AAyDD%C3%97%C2%B6m%C2%AD%1C%C2%A0%C3%AB%C2%BA%C2%BEW%03%11%5D%3D%22%C3%BA%60%C3%A6S%C3%934A%C3%934o%3F%00%C3%B0%C2%AFK%C3%BD%0DJ%C3%95%C3%90S%0DM%C3%8F%08%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%80IDATH%C2%89%C3%AD%C2%97%C2%B1%C2%AE%C2%AA%40%10%C2%86%7F%06%C3%8CZmh%C3%A8LnAgu%C2%B6%C2%A3%C2%A3%C3%9B%C2%8E%C3%8E%C3%8E%C3%8E\'%C3%A0Qx%09%13%0B%C2%9E%C3%80\'%C2%A0%C3%83%C2%8A\'%C2%A0%C2%B31T%1A%60%C2%B8%C2%85%C3%87ss%C2%A2%109%C3%A2%C2%9E%C3%A6%C3%BE%0D%C3%89d%26%C3%9F%C3%AC%C2%BF%C3%8B%C3%80Zi%C2%9A%C3%BEa%C3%A6%033%C2%BB0%24%22%3A%11%C3%91%C2%87%C3%83%C3%8C%C2%87%C2%AE%C3%AB%C3%9C%C3%99l%06%C3%9B%C2%B6%C3%9F%0En%C3%9B%16M%C3%93%C2%B8%C3%8C%7C%20fv%1D%C3%871%02%06%00%C3%9B%C2%B6%C3%A18%0E%C2%98%C3%99%C2%A5%5B%C3%80%C2%A4%C2%88%C3%A8%C3%BA4J%C3%BD%C2%94eY%00%00g()I%C2%92%C2%BBX%1C%C3%87%C2%93%C3%A5%C3%BF%C3%8A%C3%8A%C3%BF%C3%83G%C3%83%C3%8B%C2%B2%C2%9C%0Cn%C3%ADv%C2%BBn%3E%C2%9F%0F%26%25I%C2%828%C2%8EQ%14%05%C2%B2%2C%C3%83f%C2%B3y*%7FH%C3%A7%C3%B3y%C3%9C%C3%8A%C2%97%C3%8B%25%C2%A4%C2%94%C3%88%C2%B2lLY%C2%AFF%C3%9B%C2%AE%C2%B5F%C2%9E%C3%A78%1E%C2%8F%C3%A6%C3%A1RJ%04A%C2%80%C3%BD~o%1E%0E%00J)%08!%5E%C2%B6%C3%BF%C3%87%C2%AF%C2%9A%C3%96%1AY%C2%96%C2%A1%C2%AA*%C3%B3%C3%B0)%C3%AC%7Fi%C3%88%04A%C2%80%C3%8B%C3%A5%C2%82%3C%C3%8F%C3%8D%C3%83%C2%81%C3%97%C3%AC%7F%19%C3%AEy%1E%C2%94R%3F%C2%B2%7F%C2%92%C3%99%1E%04%01%C2%AA%C2%AABQ%14%C2%A3%C3%AA%1E%C2%8E%C3%97%C3%ADv%3B8D%3C%C3%8F%C3%83z%C2%BD%C3%BE%16%2B%C3%8B%12i%C2%9A%C3%9E%C3%A5%C3%B6%C2%8D%C3%99%C3%9E%C3%B1%1AE%11%C2%84%10%0F%C2%8B%C2%84%10%C2%88%C2%A2%C3%A8.%C2%BEX%2Cz%C2%9B%C3%AD%C3%93C%C2%B8%C2%94%12Z%C3%AB%C2%87%05ZkH)G%C2%83%C2%9E%C2%86%03%C2%80%C3%AF%C3%BBPJ%7D%C2%8B)%C2%A5%C3%A0%C3%BB%C3%BE%24%C3%A0A8%00%C2%84a%08%C3%8F%C3%B3%00%5C%C3%B79%0C%C3%83%C3%89%C3%80%C3%80%13%C3%9F%C3%B3%C2%AA%C2%AA%C2%90%C2%A6)V%C2%AB%C3%95dv%03%C3%97%03%C3%B7%C3%94%C3%8F%C3%84%3B%C3%B4u%C3%9A%C3%9B%C2%B65%0A%C2%BE%C3%B1%C2%88%C2%88NM%C3%93%18i%C2%A0%C3%AB%C2%BA%C3%9B%5D%0DDtr%C2%88%C3%A8%C2%83%C2%99%0Fu%5D%C2%BBu%5D%C2%BF%C2%BD%01%C3%A0%C3%9F-%C3%B5%2F%C3%86%C3%9D%C2%B6%C3%AE%7D%7C%C3%9D%C2%99%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%3CIDATH%C2%89%C3%AD%C3%971n%C2%830%14%06%C3%A0%C2%9F%C2%87%23%18Y%C3%98%22u%C3%B7%C2%94Sd%C3%8B%C3%86A8%0A%C2%97%40%C3%8A%C3%80%C3%86%C3%A6%C2%8D%13%C3%80%05r%02%16%C3%A4%C2%89%08x%C3%AE%C3%90%C2%A2%0E%C2%A4%C2%94%C2%A0%C3%A2F%15%C3%BF%C2%82d%23%7F%C3%8F%C3%96%03%C3%89N%C2%96eo%C3%8C%5C1s%00K!%C2%A2%C2%86%C2%88N%C2%82%C2%99%2BcLp8%1C%C3%A0%C2%BA%C3%AE%C3%A6%C3%B00%0C%C3%A8%C3%BB%3E%60%C3%A6%C2%8A%C2%989%10BX%C2%81%01%C3%80u%5D%08!%C3%80%C3%8C%01%C2%8D%036CD%1FO%C2%AB%C3%AAg%1C%C3%87%C3%B9%3B%7C%C3%8C%C2%8E%C3%AF%C3%B8k%C3%A1Zk%C2%A4i%C2%BA%09.%C3%A6%26o%C2%B7%1B%C2%94R%C2%B8%C3%9F%C3%AFv%C3%B1%C2%A2(P%C2%96%C3%A5%C2%AAE%C2%93%24%C2%99%C2%8C%C3%85q%C3%BC3%C2%AE%C2%B5F%C2%9E%C3%A7%C2%A8%C3%ABz%15%C3%BCL%26x%C2%96e%C3%90ZO%5E%7C%C2%B4%C2%9B_%C3%87%2F%C2%97%0B%C2%94R%C2%93%C2%9D%3F%3A%C2%B6%C3%AF%C2%B2%C2%B4%C3%90I%C2%B7%C2%87a%C2%88(%C2%8A%20%C2%A5%5C%C2%8C%C2%AD%C3%8D%C3%83%C2%86%C3%B3%3C%0F%C3%A7%C3%B3%19%C3%87%C3%A3%11EQ%3C%C3%9D%C3%ADKOi%C3%B6S%C2%93R%22%0CC(%C2%A5%C2%9E%C3%82%C2%97%C3%86%C2%B9%5E%C2%AF%C3%86%C3%B7%C3%BDM%16%C2%9FK%C3%9B%C2%B6%2F%C3%BE%7B%C3%9D%C3%B1%1D%C3%BF_%C3%B80%0CV%C3%91%C3%91%23%22j%C3%BA%C2%BE%C2%B7R%C2%801f%C2%BC%C2%AB%C2%81%C2%88%1AAD\'f%C2%AE%C2%BA%C2%AE%0B%C2%BA%C2%AE%C3%9B%C2%BC%00%C3%A0%C3%AB%C2%96%C3%BA%0Eb%C3%91%C2%80%C3%8A*%20%C2%A7%05%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%24IDATH%C2%89%C3%AD%C2%971%C2%8E%C2%840%0CE%3F%26HSr%C2%83%C2%BD%C3%82%C3%9C%C2%87Kp%14N%40MA%C2%97%C2%8E.\'%C2%80%0B%C3%A4%04iP*%C2%A2%04g%C2%8B%15%C3%9A%1D%C3%AD4a%07%C2%B4%C3%92%C3%B0%C2%9B(v%C3%ACg%C3%85V%C2%A4d%7D%C3%9F%7F0%C3%B3%C3%84%C3%8C%25N%12%11%C3%8DDt%17%C3%8C%3C%C3%85%18%C3%8B%C2%A2(%C2%90%C3%A7%C3%B9%C3%A1%C3%A0u%5D%11B(%C2%99y%22f.%C2%85%10%C2%A7%C2%80%01%20%C3%8Fs%08!%C3%80%C3%8C%25m%C2%863ED_%C3%AB%C3%9E%04%C3%96Z%C2%B4m%0Bkmrl%C2%96e%7F%C2%83K)a%C2%AD%C2%85%C2%94ro%C2%8A%7Dp%C2%A5%14%C2%8C1%00%00c%0C%C2%94R%C3%A7%C3%80%C2%B5%C3%96%18%C3%87%C3%B1%C3%816%C2%8E%23%C2%B4%C3%96%C3%87%C3%82%C2%AD%C2%B5%18%C2%86%C3%A1%C2%A9o%18%C2%86%C3%A4%C3%BE\'%C3%81%C2%A5%C2%94p%C3%8E%3D%C3%B59%C3%A7%C2%92%C3%BB%2FR%0EWU%C3%B5%C2%B0o%C2%9A%06u%5D\'%01%7Fj%C3%B7%C2%B4%C2%BFB%17%C3%BC%C2%82_%C3%B0%0B~%C2%A8%C2%B2%C2%AE%C3%AB%C3%A2%C3%ADvK%0Aj%C2%9A%C3%A6%C2%97-%C3%B5%C2%99%5D%C2%96%C3%A5%C2%8D%C2%AF%C3%BD%7D%C3%A1%C2%BB%06%C3%AE%15%C3%BA%1F%03%C2%B7%C2%AE%C3%AB%C2%A9%C3%90%C2%8DGD4%C2%87%10N)%20%C3%86%C2%B8%C3%BD%C3%95%40D%C2%B3%20%C2%A2%3B3O%C3%9E%C3%BB%C3%92%7B%7Fx%01%C3%80%C3%B7%2F%C3%B5%13y%3A%C2%8C%C3%A1UY6%C2%98%00%00%00%00IEND%C2%AEB%60%C2%82')
        },
        checked: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C2%B3IDATH%C2%89%C3%8D%C2%97!%C2%93%C3%AA0%14%C2%85%0FL%12Qf%C2%98(%0C%C3%A2%C3%85%C2%80%C2%A9%C3%99%C2%9Ahjj%40%C3%B3%C2%AB%C3%B8%25%C3%BC%C2%84%C2%9A*LUM%0C%184*%C3%83L%2Bn3%C3%833%C3%9B%C2%9D%C2%9D%C2%B7%C3%9D%C2%A5-KxGfn%C3%B2%C3%A5%C3%9CfN%C2%93%C3%91n%C2%B7%C3%BBCD%C2%85sN%C3%82%C2%93%18cV%08%C3%B1%C3%86%C2%88%C2%A8%00%20%C2%A7%C3%93)%C2%84%10O%07%13%11%C2%AA%C2%AA%C2%92DT%C2%8C%C2%9Ds2%08%02%2F%60%00%10B%20%08%028%C3%A7%C3%A4%C2%B8%19%C3%B0)%C3%8E9%00%60%C3%AC%C2%95%C3%BA%C2%AE%C3%91h%C3%B4%3Ax%C2%A3%C2%97%C3%82Y%C3%9B%C3%A0~%C2%BF%C3%BF2%C2%B6%C3%9Dn%3B%2F%C3%9Au%C3%BE%C3%BF%C3%A7%C2%BCMmn%C2%BC%C3%81%1Fm%7B%C2%9B~%C2%BD%C3%AD%C3%A7%C3%B3%C2%B9sm%C2%AB%C3%B3%3E.%1B%C2%95e%C2%89%3C%C3%8FQ%C3%975V%C2%AB%15f%C2%B3%C3%990x_%1D%C2%8FG%18c%C2%B0%5C.%11%C2%86a%C3%A7y%0F%C3%81%3F%C2%BB%C2%8D%C3%A3%18R%C3%B6%C3%BB1%0E%C2%86%1Bc%60%C2%8CA%18%C2%86%C2%BD%C3%9C%3E%04%C2%B7%C3%96%22%C3%8Fsp%C3%8E%C2%B1%5E%C2%AF1%C2%99L%06%C2%81%C2%81%C2%9E%C2%A7%C3%9D%18%C2%83%2C%C3%8B%C2%A0%C2%94B%1C%C3%87%C3%9F%C2%82%C3%934EY%C2%96w%C3%97%C3%AB%C3%A4%C3%BCr%C2%B9%C2%A0(%0Ap%C3%8E%C2%91%24%C3%89%5D%C2%B7%C3%96Z%C2%A4i%0A%C2%AD5%C3%A6%C3%B3y%3Fx%5BHh%C2%AD%C2%A1%C2%94%C3%AA%C2%B2W%00%40%5D%C3%978%1C%0EX%2C%16%C2%88%C2%A2%C2%A8%C2%B5%C2%A6s%C3%9B%C3%BB%C2%80%3F%C3%ABt%3A%7D%C3%BB%19%C2%BCd%C2%BB%C2%B5%16Y%C2%96a%C2%B3%C3%99%0C%C2%83%3F%C2%92%C3%ADRJh%C2%AD%C2%BF%C3%94%C3%BDJ%C3%82%C3%BD%24%C2%A5%14%C2%A2(%C3%BA%C2%B8%C2%B7%C3%9D%C2%85%0F%C3%89%C3%B6%7F%C3%859G%14E%3F%C2%9E%C2%95%C2%A78o%C3%9A%7C%2Fn%C2%9F%02O%C2%92%C2%A4S%C3%9D%C3%ABo%C2%AFD%C3%A4%15%C3%9A%C3%B0%C3%86%C2%8C1%5BU%C2%95%C2%97%0D%C3%9Cn%C2%B7%C3%A6%C2%AD%06%C3%86%C2%98eB%C2%887%22*%C2%AE%C3%97%C2%AB%C3%B7W%C3%AA_%24%20%C2%A9\'%C2%8A%C2%98%C3%A8P%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C2%A3IDATH%C2%89%C3%AD%C2%97%2F%C2%8F%C2%830%18%C3%86%1FH%5BA%C2%93%C2%A5j%06q%180%C2%98ajj%C2%A6%C2%A6%C2%A6%C3%B7%C2%A9%C3%B6I%C3%B8%083%C3%B3(%0C%065%3D%C3%95%2C%19%C2%A2k%C2%B2%13w%2C%C2%B9%C3%A3%20%C3%BC%3B%C3%8E%C3%9C%C2%A3%C3%88%C3%8B%C3%BB%C3%A4%C3%97%C2%BE%C2%BC%C2%A5%C2%ADs%3C%1E%C3%9F%C2%8C1%C2%B9%C2%B5V%60!%11B4clC%C2%8C19%00%C2%B1Z%C2%AD%C3%80%18%C3%BBu%C2%B01%06UU%09cL%C3%AEZk%C2%85%C3%A7y%C2%8B%C2%80%01%C2%801%06%C3%8F%C3%B3%60%C2%AD%15n%1DXR%C2%94R%00%C2%80%C2%BB(%C3%B5S%C2%8E%C3%A3%00%00HWR%C2%9A%C2%A6%C2%8D%C3%98%C3%A1p%C2%98-%C3%BFOf%C3%BE%0F%C2%9F%0D~%C2%BD%5E%07%7B%3A%1B%C2%AEn%C2%964M%3B%1B%07%00%C3%B2%3C%C2%87%C2%94%12A%10%C3%B4%C3%8A%07f%C2%9AyQ%14%C2%A0%C2%94%22%08%C2%82A%C2%BE%C3%89p%C2%AD5%C3%8A%C2%B2%C2%84%C2%94r%C2%B0w2%3C%C3%8B2%C3%84q%0C%C3%8E%C3%B9%C2%B2%C3%B0%C2%BA%C3%9CQ%14%C2%8D%C3%B2%C2%8F%C2%86%C3%9F%C3%AFw%14E1%C2%AA%C3%9C%C2%93%C3%A1S%C3%8A%3D%09%5E%C2%96%25%1E%C2%8F%07%C3%A28%1E%0D%1E%05%C2%9F%C2%A3%C3%9C%C2%A3%C3%A1Y%C2%96!%C2%8A%22%081%C3%BD%C3%945%08~%C2%B9%5Cf)w%C2%AD%C3%9E%C3%BBy%C3%BD%C2%BC%C3%9Dn%1By%C2%A7%C3%93%09Z%C3%ABV%C2%AF%10%02%C2%BB%C3%9D%C2%AE%C3%A1%1B%5C%C3%B6%C3%B5z%C3%9D%C2%88)%C2%A5%5EG%C2%A3%C3%AF%C2%A2%C2%94B)%C3%B5%C3%A3%C2%BBY%C3%BE%C3%AD%C2%9C%C3%B3%C3%96%06%C2%94R%C2%B6.%C3%87%C3%99%C2%B6T%C3%9F%C3%B7%11%C2%86%C3%A1%C2%97X%18%C2%86%C3%B0%7D%C2%BF%C3%953%C3%ABa%22I%C2%92%C3%97*%10B%20I%C2%92%C3%8E%C3%BC%5E%C3%BB%C3%B9%10)%C2%A5p%3E%C2%9F%5B%C2%BFso%C3%B8%18q%C3%8E%C2%B1%C3%9F%C3%AF%7B%C3%A5%C2%BA%C3%80%C3%87%15fI%C3%95%3C%C2%97%10%C2%A2%C2%AB%C2%AAZd%00%C3%8F%C3%A7%C2%B3%C2%BE%C2%AB%C2%81%10%C2%A2%09clc%C2%8C%C3%89o%C2%B7%C3%9B%C3%A2%C2%B7%C3%94w%C3%BD%05%C2%95%C2%B8N%C2%83\'%C3%BF%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%25IDATH%C2%89%C3%AD%C2%97%3Dn%C2%840%10%C2%85%1F%C3%88vA%C2%81%7C%C2%83%C3%90%C3%90%C2%AF%C2%9B%3D%02%155%C2%A7%C3%A2%24%1C%C3%81%0D%15%3D%3D5%C2%BD%C2%85%04%C3%85%60%C2%89mB%C2%8A%C2%ACw%03%2B%C3%A1M%22%5E9z%C3%927%3F%C2%96%C3%AD%09%C3%8A%C2%B2%C3%BC%20%C2%A2%C3%96Z%2B%C3%A1I%C2%8C1%23%C2%84%C2%B80%22j%01%C3%888%C2%8E!%C2%848%1CLD%C2%98%C2%A6I%12Q%1BZke%14E%5E%C3%80%00%20%C2%84%40%14E%C2%B0%C3%96%C3%8Ap%0D%C3%B8%14%C3%A7%1C%00%10z%C2%A5~*%08%C2%82%C3%B7%C3%81W%C2%9D%C3%B0%13%C3%BE%C3%B7%C3%A1Zk%C2%8C%C3%A3%C3%B8%1E%C2%B81%06Zk%C3%B4%7D%C3%BF%C3%94%C3%87%5C%C3%81%C2%AA%C2%AA%C3%AEbEQ%C3%ACJ%60%C2%9Eg4M%C2%834M%C2%A1%C2%94rz%0E%C2%9Fy%C3%97u%0F%C3%87%C3%A0%C2%AC%C3%9C%25W7%C2%B6%C3%8A%18%C2%83%C2%BA%C2%AE%C2%91%C3%A7%C3%B9k%C3%B0%3Dm%C3%BF%C2%9E%C2%A8%C2%94%12%C3%97%C3%AB%C3%B5%C3%8E%C2%B7%19%C3%BE%C2%AA%C2%92%24%C2%81R%C3%AA%C3%AB%25%C3%BB%11%C2%BE%C3%B7p%C2%B9%C3%849%C2%87R%0AI%C2%92%3C%C3%B4%1CR%C3%B9%C3%9Af)%C2%9F%C3%BF%C3%8C%0E%C2%81gY%C2%B6%C3%89%C3%B7%C3%BF%C2%AE%C3%97%13~%C3%82%7F%2F%C2%9C%C2%88%C2%BCBW%5E%C3%88%183%C3%934yI%60Y%C2%96uW%03c%C3%8C0!%C3%84%C2%85%C2%88%C3%9Aa%18%C2%BCo%C2%A97%C2%A3%12k%11%5D%C3%BC%C3%95%C2%97%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01-IDATH%C2%89%C3%AD%C2%971n%C2%840%10E%3F%C3%88va%244%3DE*%C3%B7KC%C3%A3%C2%9E%C2%8A%C2%9ASq%12%1F%C2%81Sp%14%C2%8B%C3%82%C3%85%C3%88%12)v%C2%89v%C2%93%14%C2%B1%C2%A3e%C2%A5%C2%84_%0E%C3%A2%3F%06%7F%C3%99%C2%9Eb%C2%9A%C2%A67f%5Eb%C2%8C%C2%84%C2%83%24%C2%84%C3%B0J%C2%A9%C2%8B%60%C3%A6%05%00%C3%95u%0D%C2%A5%C3%94%C3%93%C3%81%C3%8C%C2%8C%10%021%C3%B3R%C3%86%18Ik%7D%08%18%00%C2%94R%C3%90Z%23%C3%86H%C3%A5%5E8RRJ%00%40y(%C3%B5%C2%A6%C2%A2(%00%00%22%C3%A7e%C3%A7%C3%9C%C2%97%C3%9A8%C2%8E%C3%89%3E%2F%C3%A9%C3%BC%C2%84%C2%BF%14%C2%9E%15%C2%B8%3D%5C%C3%8E%C2%B9%C2%AC%C2%A0%C3%AD%C3%BA%C2%BF%C2%BF%C3%BD%C2%84%C2%9F%C3%B0%13%C3%BE%C3%B7%C3%A1I%C3%9B%C3%AB%3C%C3%8F%C3%B0%C3%9E%3F%C3%94%C3%AE%C3%8Fv%22B%C3%9F%C3%B7%3F%C3%B6K%C3%AA%C3%9CZ%C3%BBq%05%C3%BA%2C)%25%C2%AC%C2%B5)vi%C3%B0%C2%AA%C2%AA%C3%90u%C3%9D%C2%B7%C3%8F%C2%BA%C2%AECUU%C3%8F%C2%83%03%40%C3%9340%C3%86%3C%C3%94%C2%8C1h%C2%9A%26%C3%95*%2Fpm%C3%9B%C2%82%C3%A8%3Ac%10%11%C3%9A%C2%B6%C3%8D%C2%B1%C3%89O%C2%BB%C2%B5%16Z%C3%AB%C3%A4u%C2%BEW%C3%96e%02%C2%B8%C2%AE%C3%BF0%0C%C3%99%60%C3%A0%C3%9693%C3%BF%C3%8A%24U%3B%C2%AF%14B%C3%B8%10%C3%82!%1F%C2%B0m%C3%9B%3E%C2%ABA%08%C3%A1%C2%85R%C3%AA%C3%82%C3%8C%C3%8B%C2%BA%C2%AE%C2%87O%C2%A9%C3%AFVpTg%C2%8E%24%C3%9D%C2%80%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%8CIDATH%C2%89%C3%8D%C2%97!%C2%AF%C3%AA0%14%C3%87%C3%BF%5B%C2%BA%C2%8A5!%15%C2%A4%06%C3%B1%C2%A603WMaf0%C2%A0%C3%B9%40h%3E%09%1E3St%C3%8D0%18%10%C3%8C%60%16%C3%84B%C2%B2%C2%89%C2%B2%C2%84%2B%C3%AE%C3%9B%C3%8D%C3%8B%C3%9B%02%C3%A5%C3%B2(%C3%AF\'%C3%9B%C2%B3%C3%BEvNNORg%C2%B1X%C3%BC%C3%92Zo%C3%AA%C2%BA%C3%A6%C2%B0%04!%C2%A4%C2%A0%C2%94~%10%C2%AD%C3%B5%06%00%C3%AF%C3%B5z%C2%A0%C2%94%C2%BE%5C%C2%AC%C2%B5FUU%5Ck%C2%BDq%C3%AB%C2%BA%C3%A6%C2%BE%C3%AF%5B%11%03%00%C2%A5%14%C2%BE%C3%AF%C2%A3%C2%AEk%C3%AE6%0B6%C3%B1%3C%0F%00%C3%A0Z%C2%B5%C3%BE%C3%86q%C2%9C%C3%B7%C3%89%1B%C3%9E*\'%5D%C2%8B%C3%8B%C3%A5%C2%B2%C2%B56%C2%9B%C3%8D%C2%8C%0F5%C3%BD%C3%BE%C3%BF%C3%8B%C2%BC%C2%8B%C2%AEl%C2%AC%C3%89%C2%9F-%7B%17%C3%86e%C3%8F%C2%B2%C3%8CXn%C2%8A3%C2%9F%C3%8F%C2%AF%C3%BD~%C3%BFfP%C2%9E%C3%A7PJ%C2%811%C2%86(%C2%8A%C3%80%18%7BZ%7C%3A%C2%9D%C3%8C2%17B%60%3A%C2%9DB%08%C2%81%24I%C2%B0%C3%9B%C3%AD%C2%9E%C2%96%03%0Fv%7B%18%C2%86%C2%88%C3%A3%18Y%C2%96AJ%C2%89%C2%B2%2C%C3%AD%C3%89%01%C2%80s%C2%8E%C3%B1x%0C!%04V%C2%AB%15%C2%B6%C3%9B%C2%AD%3DyC%18%C2%86%C2%98L%26%C3%88%C3%B3%1CI%C2%92%C2%A0(%C2%8A%7F%2F%2F%C3%8B%12I%C2%92t%C3%AE1%C3%86%10%C3%871%C2%82%20%C2%80%C2%94%C3%B2%C3%A1*%C3%9C%C2%BC%C3%A7%C3%87%C3%A3%11J)%5C.%C2%97%C2%9B%C2%87%0C%C2%87C%0C%06%03(%C2%A5p8%1C%10E%11%C2%84%10%3F%C2%97%C2%A7i%C2%8A%C3%BD~%7F%C3%B7%C2%80%C2%86%C2%A6%0AY%C2%96a%C2%BD%5E%C2%B7%C3%B6%C2%8Df%7BS%C3%A6G%C3%84%7F%12%04%C2%81ql%2Bs)%25%C2%AA%C2%AAj%05Z%C2%99%C3%AD%C2%A3%C3%91%08J%C2%A9V%C3%B7Z%C2%99%C3%AD%C2%9C%C3%B3%C3%AF%0E~5%C2%9D%0D%C3%A7y%C3%9Ew%C3%87%C2%A6iz%C2%B7%C3%9B%C3%BF%C3%86%C2%B4J7%C2%AFZ%10%04%C3%A0%C2%9CC)%C3%B5%C2%90%C3%9C%C2%94%C2%BBC%C2%A6%19%C2%A7o%C2%91%C2%BF%12%17%C3%B8z%C3%82%C3%98%C2%A4%C3%B1%C2%B9%C2%84%C2%90%C2%A2%C2%AA*%2B%3Fp%C2%BD%5E%C2%9B%C2%B7%1A%08!%05%C2%A1%C2%94~h%C2%AD7%C3%A7%C3%B3%C3%99%C3%BA%2B%C3%B5%13R!%C2%BE*%C3%89%C2%B0%C2%9F3%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C2%BEIDATH%C2%89%C3%AD%C2%97%C2%A1%C2%8E%C3%AA%40%14%C2%86%C3%BF63%23%3A%09%19A0%15%C2%8Bi%C3%8D%18j%C3%86%C3%94%C2%A3%C3%90%7D%204O%C3%92G%C3%80%C2%80F!%C2%A8%C2%A9%C2%AAn%10%0DI%2B%C2%86I%C2%B8%C2%82%C3%8B%26%C2%BB%C3%90%C2%B2%C2%85%C3%AE%5Cs%7FyrN%C2%BE%C2%993%C3%A9%C3%9Fs%C2%9C%C3%95j%C3%B5%C2%A1%C2%B5%C3%9E%1Bc%04%2C%C2%89%10R1%C3%86fDk%C2%BD%07%20F%C2%A3%11%18c%C2%BF%0E%C3%96Z%C2%A3i%1A%C2%A1%C2%B5%C3%9E%C2%BB%C3%86%18%C3%A1y%C2%9E%150%000%C3%86%C3%A0y%1E%C2%8C1%C3%82%C2%BD%05l%C2%8AR%0A%00p%C2%ADR%C3%BF%C3%8Aq%1C%00%00%C3%A9JJ%C3%93%C3%B4.%C2%96%24%C3%89%60%C3%B9%C3%BF%C3%A4%C3%A6%C3%BF%C3%A1%C2%BD%C3%A1eY%0E%06w%C2%96%C3%8B%C3%A5e%3C%1Ew%26%C2%A5i%C2%8A%24IP%14%05%0E%C2%87%03%16%C2%8B%C3%85%C2%8F%C3%B2%C2%BBt%3C%1E%C3%BB%C3%9D%7C%3A%C2%9D%C2%82s%C2%8E%2C%C3%8B%C3%BA%C2%94%C2%B5%C2%AAw%C3%9B%C2%95R%C3%88%C3%B3%1CUU%C3%99%C2%87s%C3%8E!%C2%A5%C3%84n%C2%B7%C2%B3%0F%07%C2%800%0CA)%7D%C2%BB%C3%BD%2F%7FjJ)dY%C2%86%C2%BA%C2%AE%C3%AD%C3%83%C2%87h%C3%BF%5B%26%23%C2%A5%C3%84%C3%B9%7CF%C2%9E%C3%A7%C3%B6%C3%A1%C3%80%7B%C3%AD%7F%1B.%C2%84%40%18%C2%86%2F%C2%B5%7F%10o%C2%97R%C2%A2%C2%AEk%14E%C3%91%C2%AB%C3%AE%C2%A1%C2%BD%C2%AE%C3%97%C3%ABN%13%11B%60%3E%C2%9F%7F%C2%89%C2%95e%C2%89%C3%ADv%7B%C2%97%C3%9Bf%C2%B3%C2%AD%C3%B6%1A%C3%87%C3%B1%C3%A7%C2%A8%C3%B3%5D%C2%94R%C3%84q%7C%17%C2%9FL%26%C2%AD%C2%87m%C3%93C8%C3%A7%1CJ%C2%A9%C2%87%05J)p%C3%8E%7B%C2%83~%0C%07%00%C3%9F%C3%B7%11%04%C3%81%C2%97X%10%04%C3%B0%7D%7F%10p\'%1C%00%C2%A2(%C2%82%10%C3%97%5DB%08%C2%81(%C2%8A%06%03%03O%06H%C3%A0%C3%BA%C3%BE%C2%9B%C3%8D%C3%A6%C3%A1%3B%7F%C3%97%C2%B3%7Fxo8%C3%A7%C3%BC%C3%A9%C3%B0%C3%B0%C2%AA%5C%C3%A0%C2%BA%C3%82%C3%98%C3%94%C2%8D%C3%A7%12B%C2%AA%C2%A6i%C2%AC%1C%C3%A0r%C2%B9%C3%9Cv5%10B*%C3%82%18%C2%9Bi%C2%AD%C3%B7%C2%A7%C3%93%C3%89%C3%BA%C2%96%C3%BA%07%C3%B3%22%C2%A1%C2%8E%60%2FOY%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01.IDATH%C2%89%C3%AD%C3%97%C2%A1%C2%AE%C2%830%14%06%C3%A0%1F%C3%92V%20%C3%88Q%C2%B3%C2%B7%C2%8F0%C2%85%C3%82%60%C3%B6%08%3C%15O%C3%82%23%C3%8CLaj%C3%80O%C3%A3%09%09%15gM%C2%98%C2%B9(%C2%B8%1B%23w%C3%9DM.%C2%BFl%C2%9B~ms%C3%9A%C2%A4AQ%14_%C3%8C%C3%9C8%C3%A7%08%C2%9E%22%C2%84%C3%A8%C2%94RG%C3%81%C3%8C%0D%00%C2%8A%C3%A3%18J%C2%A9%C2%B7%C3%83%C3%8C%0Ck-1s%13%3A%C3%A7(%C2%8A%22%2F0%00(%C2%A5%10E%11%C2%9Cs%14N%0D%3E%23%C2%A5%04%00%C2%84%5E%C3%95%C3%AF%04A%C3%B09%7C%C3%8A%C2%8E%C3%AF%C3%B8%C3%9F%C3%82%C2%87a%C3%80%C3%B9%7C~%0B.%1Eu%C2%B6m%0Bc%0Cn%C2%B7%C2%9B_%C2%BC%C2%AEk%5C%C2%AF%C3%97M%C2%93%C2%96e9k%C3%8B%C3%B3%C3%BC9%3E%0C%03%C2%AA%C2%AAB%C3%97u%C2%9B%C3%A0W2%C3%83%2F%C2%97%0B%C2%AC%C2%B5%C2%B3%C2%81K%C2%BB%C3%B9u%3CMS%18cf%3B_%3A%C2%B6%C2%9F%C2%B2v%C2%A1%C2%B3j\'%22dY%06%C2%AD%C3%B5jlk%16%0BNJ%C2%89%24Ip8%1CP%C3%97%C3%B5%C3%8B%C3%95%C2%BE%C3%B6%C2%94%1E%5E5%C2%AD5%C2%88%08%C3%86%C2%98%C2%97%C3%B0%C2%B5y%C3%BA%C3%88%10%11N%C2%A7%C3%93g%C3%B0wf%C3%87w%C3%BC%C2%9F%C3%A1%C3%8C%C3%AC%15%C2%9D%C2%BCP%08%C3%91Yk%C2%BD%2C%60%1C%C3%87%C3%A9%C2%AF%06!D\'%C2%94RGfn%C3%BA%C2%BE%C3%B7%C3%BEK%C2%BD%03%C3%9A%25vxps~%2F%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%16IDATH%C2%89%C3%AD%C2%971%C2%8E%C3%83%20%10E%C2%BF-LAa%C3%B9%06%7B%C2%85%C2%9C%C3%82G%C3%B0%C2%A9%7C%12j_%C2%81%C2%86%C3%86%C3%AE%C2%A9%C3%A9%C2%91%25(%06%C2%A4le)Q%C3%92%40%C3%96l%11%C3%BF%0Af%60%1E%C2%9A%19%C2%81h%C3%A6y%C3%BE!%C2%A2-%C2%A54%C2%A0%C2%92%18c%C2%8Es~cD%C2%B4%01%18%C3%BA%C2%BE%07%C3%A7%C3%BCt0%11!%C2%840%10%C3%91%C3%96%C2%A6%C2%94%06!D%150%00p%C3%8E!%C2%84%40Jih%0FCMu%5D%07%00hK%03x%C3%AF%C2%B1%2C%0B%C2%BC%C3%B7%C3%99%7B%C2%9B%C2%A6%C3%B9%0C%C2%AE%C2%94B%08%01J%C2%A9%C3%92%10e%C3%B0u%5D%C3%A1%C2%9C%03%008%C3%A7%C2%B0%C2%AEk%1D%C2%B8%C2%B5%16%C3%86%C2%98\'%C2%9B1%06%C3%96%C3%9As%C3%A1%C3%9E%7Bh%C2%AD%C3%9F%C3%BA%C2%B4%C3%96%C3%99%C3%B5%C3%8F%C2%82%2B%C2%A5%10c%7C%C3%AB%C2%8B1f%C3%97%C2%9F%C3%A5%2C%1E%C3%87%C3%B1i.%C2%A5%C3%844MY%C3%80G%15w%C3%BB_%C3%A8%C2%82_%C3%B0%0B~%C3%81OU%C3%96%C3%B5zHJ%C3%B92.%C2%B9f%C2%BF7%C3%AD%C3%9F%0B%2Fj%C2%B8O%C3%9E%C3%B0G%C3%BD%7F%C3%9A%C2%89%C2%A8*%C3%B4%C3%A0%C2%B5%C2%8C1%17B%C2%A8r%C2%80%C3%BB%C3%BD~%C3%BC%C3%95%C3%80%18s%C2%8Cs~%23%C2%A2m%C3%9F%C3%B7%C3%AA%C2%BF%C3%94_%C3%90%1B%7F%C3%B4O%C2%AC%1C%01%00%00%00%00IEND%C2%AEB%60%C2%82')
        }
    };

    var dark = {
        normal: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01vIDATH%C2%89%C3%AD%C2%97%C2%AF%C2%8E%C3%82%40%10%C2%87%3F%C2%9AK%C3%AA%C2%9A%13%C3%A0x%02TQ%C2%AD%C2%AB%C2%ABB%C2%90%C2%AC%C2%A3%0F%C3%81%C2%B3%C3%B0%02%C3%85%C2%A1%20AS%C2%B5(pUE%C3%A1p%C2%A0%C2%A8Zu\'%C3%AEO%C2%B8%C2%B0%C3%90-%14z%C3%89%C3%9DONf%C3%B6%C2%9B%C3%9Ft3m%1BB%C2%88W%60%0E%04%3CO%12%C3%A8%5B5%C2%80%C3%B9%C3%A4%C3%8D%C2%AD%1A%C3%80%C3%9F%0DX5%C2%81%01%C3%B8%C2%87%C3%97%C2%A2%17%5Dp8%1C%C2%9E%C3%85F%C2%A3%C2%91%C3%B1%C2%A1%C2%A6%C3%B5%C2%BF%C3%8F%C2%B9N%3A7O%C2%83%C3%9F%3Bv%C2%9D*%1F%7B%C2%A7%C3%931%C3%8Em%08!%C3%9E%C2%AA%C2%80%3A%C2%8EC%18%C2%86%C3%98%C2%B6%C2%8D%C2%94%C2%92%C3%9DnWXc%3C%C3%B6k%C3%AAv%C2%BBx%C2%9EG%C2%9A%C2%A6%C2%ACV%2B%C3%A3%C2%BA%C2%BB%C3%A0%C2%A7ng%C2%B3%19%C3%BB%C3%BD%C2%BET%C3%BD%C3%8Dp%C3%9F%C3%B7%C3%B1%3C%C2%8F%C3%B5z%5D%C3%8A%C3%AD%5D%C3%B0V%C2%ABE%18%C2%86(%C2%A5%18%C2%8F%C3%87%1C%C2%8F%C3%87%C2%9B%C3%80P%C3%B2%C2%B6%C3%BB%C2%BE%C2%8F%10%C2%82%2C%C3%8B%C2%98N%C2%A7%17%C3%81Q%14%C3%A18N%C3%A1yF%C3%8E%C3%9B%C3%AD6A%10%C2%A0%C2%94b2%C2%99%14%C2%BAm6%C2%9B%0C%06%03%C2%92%24a%C2%BB%C3%9D%C2%96%C2%83%C3%AB%C2%96D%C2%92%24dYf%C3%92%2B%00%C2%B6m%C3%93%C3%AB%C3%B5H%C3%93%14)%C2%A56%C3%87x%C3%ACe%C3%80%C2%A7r%5D%C3%B7%C3%A2c%C3%90.%C2%99G%C3%AC%C3%B1%3C%C3%8F%C2%89%C3%A3%C3%B8G%C3%AC)%C2%BB%C3%BDp8%C2%B0X%2C%C3%8E%C3%B2*%C3%99p%C3%97%C2%B4%C3%99l%C2%90R%C2%A2%C2%942%C2%83%C2%97qyIJ)%C2%96%C3%8B%C3%A5%C3%95%C2%BB%C3%B2%10%C3%A7_c.Z%C2%B7%C2%95%C2%BD%C3%95n%C3%91%C3%9F%C3%BDz%C2%B5%C3%B8%C3%B8i%C2%ABC%C3%92%02%C3%BA54%20%C2%81%C3%BE%3B%C3%B3%19%C2%8A%5E%2F%C3%A0%C2%8B%12%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01fIDATH%C2%89%C3%A5%C2%96%C2%AD%C2%8E%C2%83%40%14FO%C3%89%268%C2%B2%C2%A2u%7D%C2%82%C2%AA%C2%A2Z%C2%B7%0E%C3%9D%C2%84%3A%1E%C2%82g%C3%A1!%C3%AA%20i%C2%82%2Bj%5CQ%C2%8C%C3%82%C3%A1%C3%AA%C2%8A%C3%82%C2%8D%C3%9A%15%C2%BB%C3%9D%C2%A4%5B%C2%A0%05fK%C2%B2%7B%C3%A5%C3%BD%C3%89%C2%99%C3%B9f%C3%A6%C3%9E%C2%99%C2%B8%C2%AE%C3%BB%0A%C3%AC%C2%817%C2%9Eg%02%C3%98%18%23%C2%80%C3%B9%C3%A2%C3%AD%C2%8D%11%C3%80%C3%9F%0B0F%02%030*%C3%BC%C2%A5-%C3%A8%C3%BB%C3%BE%C2%8D%2F%08%02m%C3%B9%C3%BFW%C3%B6%C2%BF%01%C2%9F%C3%8F%C3%A7%C2%9Dk%26%C2%AE%C3%AB%C2%BE%C3%9FK%C3%B2%7D%C2%BF%C3%B5%C3%A2%00x%C2%9EG%C2%96e%C3%A4y%C3%BEP%3Eh%C3%9A%C3%B9z%C2%BDF)E%C2%9E%C3%A7%C2%9D%C3%AA%06%C3%83g%C2%B3%19%C3%8B%C3%A5%C2%92%C3%83%C3%A1%C3%90%C2%B9v0%C3%9Cq%1C%C3%924%C2%A5%C2%AA%C2%AA%C3%A7%C3%82%2FrgY%C3%96%C2%AB%C2%BE7%C3%9C%C2%B2%2CV%C2%ABU%2F%C2%B9%07%C3%83%C2%87%C3%88%3D%08n%C3%9B6%C2%A6ir%3C%1E%7B%C2%83%7B%C3%81u%C3%88%C3%9D%1B%C3%AE8%0ERJ%C3%8E%C3%A7%C3%B3s%C3%A1%C2%8B%C3%85B%C2%8B%C3%9C%17km%C2%AFu%C3%B39%C2%8A%22N%C2%A7%C3%93%C2%95%C3%8F%C3%B3%3C%C2%A6%C3%93i%23%C2%A4%2CKv%C2%BB%C3%9D%C2%8D%C2%BF%C2%B3%C3%AC%3F%C3%81%00q%1C%C2%A3%C2%94%C2%AA%C3%8DWJ%11%C3%87qmLKo%C2%AF%C2%AA%C2%8A%24IjcI%C2%924%3EGm%23%C2%B5(%0A%C2%A4%C2%94W%3E)%25EQ4%C3%96h%C3%BDL%08!(%C3%8B%12%C3%B8%3Cg!Dk%C3%BEC%C3%B3%C2%BC%C2%8BY%C2%96%C3%85v%C2%BB%25%0C%C3%83%C2%BB%C3%9DO%3B%C2%BC%C2%8B%C2%8D%C3%BE%C2%87k%3F%C2%98%C3%9F3a%00%C2%9B%11%16%20%C2%80%C3%8D%07%C2%B7t%C2%8DW%13a%C2%A0%C3%B1%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%A9IDATH%C2%89%C3%AD%C2%95%C2%BB%C2%8D%C2%83%40%14E%C2%8F%C2%91%252%C2%B4%01%1D%C3%90%00%1D%C2%B8%01%02%C2%A4%C3%89%C2%A0%08j%C2%A1%01jpL%C3%B6%C2%A8%C2%80%08%C2%8A%C2%80%C2%8Ch%C2%A2u%C2%B2k%C3%89%C3%B6%C2%B3el%C3%A1%C3%91%C2%AE%C3%A6%C2%86WW%3A%C3%AF%C2%A7%C2%99%C2%9D1%C3%A6%0B8%02%07%3E\'%01%C3%B2%C3%80%01%C2%98%1F%C3%9E1p%00%3E%17%108%02%03%C3%A0%C3%A1%1E%C3%AE%C3%A1%7F%17%5E%C2%96%25Q%14%C2%B9%C2%81%C3%87qLQ%14%24I%C3%B20%C2%B7%C3%97%C3%8C%C2%AA%C2%AAn%C2%BC%C2%BA%C2%AEW%15%10%C2%86!Y%C2%96%C3%91%C3%B7%3D%22%C2%A2f6%C3%9Fy%C2%9A%C2%A6w%C3%97%C2%B03%C3%86%7C_%C2%9BZ%C3%A7%C3%AFjY%16%C2%9A%C2%A6%C2%B9%C3%B0%C3%94%C2%B1kZ3%C3%B6%C3%AB%C3%A2%C3%A7y%C2%A6m%C3%9B%C2%9B%C3%9C%C3%93%C3%B0W5%C2%8E%23%22%C2%82%C2%B5%C3%B69%C3%B8%C3%9A%C3%A3%C3%92d%C2%AD%C2%A5%C3%AB%3A%C2%86a%C2%B8%C2%9B%C3%99%C2%A4%C3%B3%C3%9F1O%C3%93%C3%B40%C2%A7%1E%C3%9C%C2%A7%C3%B4%C3%BF%C2%9EW%0F%C3%B7p%0F%C3%97%C3%A0%C3%BAg%C2%BB%C2%BD%24%00r%07%05%08%C2%90%C2%9F%00%5B%06Ab%C2%8D%1F%C2%BB1%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%A3IDATH%C2%89%C3%AD%C2%96%C2%B1%C2%8D%C3%830%0CE_%C2%8C%0Cp%C2%85%C3%B7%C3%89%0A%06Tz%08%C3%8D%C3%A2!%C3%9C%07%60%C2%A9%C2%8A%1Bh%01m%C2%A0J%C2%AET%5D%C2%8A%C3%9C%1D.H%0E8%C3%89H%04%C3%84%C3%BE%25%09%C3%B2Q%24A%C3%A8%60%C2%8C%C3%B9%00%C3%8E%C3%80%C2%89%C3%97I%C2%81%C2%A1k%00%C3%A6%C2%8Bw%C3%AE%1A%C2%80%7F%0A%C3%A8%1A%C2%81%01h%0A%3F%C3%96%04Yk%C3%AFl%C3%934%15%C3%A7%C3%99n%C3%9B%C2%B7%0B%3F%18c%3Ek%C2%83%C2%AD%C2%B5U%C2%8B%C3%B6%C2%AD%C3%AD%C2%B6%7D%C2%87%C3%AF%C3%B0%1D%C3%BE%C3%BE%C3%B0%C2%A2%C3%B3%3A%C2%8E%23%7D%C3%9F%C3%BF%C3%A9%C2%8F12%C3%8F%C3%B3%C2%BF%C3%A1E%2F%17%11r%C3%8E%0F%7D9gD%C2%A4%24%5D%19%3C%C2%A5%C2%84s%C3%AE%C2%A1%C3%8F9GJ%C3%A9yp%C2%80%10%02%C3%9E%C3%BB%1B%C2%9B%C3%B7%C2%9E%10Bi%C2%AA%C2%BA%C2%85SUb%C2%8C%C3%80u%C3%8E%C2%AAZ%C2%93%C2%A6~%C3%9BE%C2%84eY%C2%8A%C3%A7%C3%BC%5B%C2%AB%3E%13k%C3%95%C3%BC%C3%88%C3%94%0Dl%C2%BD%C2%B4%03%C2%86%06%05(0%5C%00-%C2%83Jk%C2%94%C2%90%C2%92%C3%A8%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C2%A2IDATH%C2%89%C3%AD%C2%97%C2%B1%C2%AE%C3%9A0%14%C2%86%C2%BF%C2%A4%C2%912au%C2%80%C3%8D%1B%5B%26%C2%9E%00%C2%A6xb%40r%26%C3%98x%01%5E%C2%A1%C2%AF%C3%80K0e%C2%B8%12%5B6%C2%9E%20C%24%C3%B3%02%19%C2%90%C2%B2y%40%C3%8A%C2%80%C3%9A%C3%A1%C2%B6%15RRpn*r%C2%A5%C3%B6%1F%C3%AD%C3%A3%C3%B3%C3%B9%3F%C2%89O%1COk%C3%BD%15x%03%16%C2%BCN\'%60%C3%A5%0F%00%C3%A6\'%C3%AF%C3%8D%1F%00%C3%BC%7B%03%C3%BE%40%60%00%C3%BE%C3%83%07Q%C3%906%C2%B8%C3%9B%C3%AD%1Ac%C3%BB%C3%BD%C3%9E9%C2%A9%C3%AB%C3%BA%C3%8F%C3%A7%C2%BCMmn%5E%06%C3%AF%5B%C3%B669%C2%97%3D%C2%8A%22g%C2%B8%C2%AB%3C%C2%AD%C3%B5%C3%B7gARJ%C2%94RXk%C3%89%C2%B2%0Ck%C3%AD_%C2%81%7F%C2%89%C2%A2%C3%A8%C3%9B%C2%B3%20k-y%C2%9E%23%C2%84%20%C2%8Ecn%C2%B7%1B%C2%97%C3%8B%C2%A57%C3%9C%C3%89%C3%B9%C2%BD%26%C2%93%09J)%C3%AA%C2%BA%C3%AE%5D%05\'%C3%A7%C3%B7%C2%BA%5E%C2%AF%14E%C2%81%10%C2%82%C3%A5r%C2%89%C3%A7y%C2%94e%C3%B9!xg%C3%A7%C3%B7%12B%C2%A0%C2%94%22%0CC%C2%B2%2C%C2%A3%C2%AA%C2%AAN%C3%AB%C2%9F%3A%17B%C2%90%24%09EQ4%C3%A6%C3%AA%C2%BA%C3%86%18C%10%04%C3%84qL%10%04%C2%9D%C2%AA%C3%B0%C3%B0%C2%A8M%C2%A7S%C3%96%C3%AB5%C3%A3%C3%B1%C3%B8a%C2%92%3C%C3%8F9%1C%0EH)%C3%99n%C2%B7H)%C2%9D%C3%A0%7Fl2%C2%8B%C3%85%C2%82%C3%99l%C3%A6%C2%94%04%C3%9EOD%C2%9A%C2%A6DQ%C2%84%C3%96%C2%BA1%C3%AF%C3%94%C3%9B%C2%85%10l6%C2%9BN%C3%A0%7B%19c%C2%9Cc%1B%C3%8E%C2%93%24a4%1A5%02_%C3%92%C3%9B%C2%8F%C3%87%23J%C2%A9%C3%86s~Io%C2%AF%C2%AA%C2%8A4M9%C2%9F%C3%8F%C3%8E%C2%B0%C2%8F%C2%AA%C3%B5%C2%85%C3%BB%C3%95%C2%BD%C3%8A%C2%B2d%3E%C2%9F%13%C2%86a%C2%A7%C2%A4%C2%AEUz%C3%B8I5%C3%86PU%15J%C2%A9NpW%C3%B5%C3%AAp%7D%C3%B5%C3%AF%C3%9E%5E%7D%C3%9E%7F%C3%9A%C2%86%C3%90%C3%89%07V%03l%C3%A0%04%C2%AC~%00%C3%B0(%C2%8EX%7BL%11%7C%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01hIDATH%C2%89%C3%A5%C3%96%C2%AD%C2%8E%C3%82%40%10%C3%80%C3%B1%3F%C3%8D%25%C2%B8%C3%8D%09p%7D%C2%82*P%C3%94%C3%A1%C3%90%24%C2%8B%C3%A3!%C3%BA%2C%7D%01%1C%0EAR%C3%97%C2%AAu%C2%ADb%15%C2%AE%0EG%C3%95%C2%BAUw%C3%A2.%C2%97%10%C2%BEZ%C2%BA%C3%97%C3%A6r%23%C2%A73%C3%B9mg%C3%93%C3%9D%0E%C2%A4%C2%94%C3%AF%C3%80%1E%C2%98%C3%93%5D(%60%C3%A9%C3%B5%00%C3%B3%C3%AD%C3%AD%C2%BD%1E%C3%A0%C2%9F%05x%3D%C3%81%00%C3%B4%C2%8A%C2%BF%3Dz%18E%C3%91U.%C2%8Ecg%C3%B5%C3%BFw%C3%AC%7F%0B%C3%B7%7D%C3%9F%19%3E%C2%90R~%3C%2B%C2%8A%C2%A2%C2%888%C2%8E%09%C2%82%C2%800%0C%C3%99l6%C2%B5%C3%AA%C2%9FE%C2%A37%3F%1E%C2%8F%18c%08%C3%83%C2%B0I%C2%9B%1B%1C%20MS%26%C2%93%09%C3%A3%C3%B1%C2%B8%7B%C3%9C%18CQ%14%2C%16%C2%8B%C3%AEq%C2%80%C3%83%C3%A1%C2%80%C2%B5%C2%B6%C3%B5%C3%B8_%C3%BE%C3%94%C3%924e6%C2%9B!%C2%84%C3%A8%1Ew1%C3%BEV%C2%87L%C2%9E%C3%A7%0C%C2%87C%C2%A6%C3%93i%C3%B78%C2%B4%1B%7Fk%C3%BC%7C%3E%C2%A3%C2%B5~i%C3%BCN%C3%8E%C3%B6%3C%C3%8F%11B%10%04A%C2%A3%C2%BE%C2%9B%C3%87%C3%ABz%C2%BDf4%1A%C3%9Dm%C2%AA%C2%AA%C2%8A%C3%ADv%7B%C2%91%C3%B3%7D%1F)%C3%A5Um%C3%A3%C3%BB%3CI%12%C2%AC%C2%B57%1B%C2%AC%C2%B5%24Ir%C2%95%3F%C2%9DNw%C2%91F%C2%B81%C2%86%2C%C3%8Bn6dY%C2%861%C2%A61T%1B%07(%C3%8B%12%C2%AD%C3%B5ENkMY%C2%96N%C3%A0%C2%878%C2%80R%C2%8A%C2%AA%C2%AA%C2%80%C2%AF%7DVJ9%C2%83%C2%A1%C3%86%7D.%C2%84%60%C2%B5Z%C2%B1%C3%9B%C3%AD%C2%9C%C2%8D%C2%BB6%C3%BE%C2%9B%C3%91%C3%BB%3F%C2%9C%C3%9B%C2%8D%C2%AC%1F%C3%8A%03%C2%96%3D%2C%40%01%C3%8BO%C3%8C%C3%B9%C2%87W%1E%C2%93r%C3%85%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%ADIDATH%C2%89%C3%AD%C3%94%C2%B1%C2%8D%C2%830%18%C2%86%C3%A17%C3%A8%24*%C2%AC%2B%C3%98%C2%80%05%C2%98%00*%C3%9C!!%C2%99%C2%8E!%C2%98%C2%85%3D(%22%C3%91%C3%91%C2%B1%04%2C%40%C3%A7%09%C2%A8%C3%AE%C2%9AS%1A%C2%93%C2%9CAI%C2%9C%C3%82_i%5B~~%5B%C2%BF%7DQJ%7D%03W%20%C3%A7%7D%C2%99%C2%80*p%00%C3%B3%C3%A7%5D%03%07%C3%B0%C2%AD%C2%80%C3%80%11%0C%C2%80%C3%87%3D%C3%AE%C3%B1%C3%8F%C3%80%C2%85%104M%C3%B3%12%C3%BC%C3%AB%C3%91d%C2%92%24%14EA%18%C2%86%C3%AF%C3%85%C3%B3%3C\'M%C3%93S%C2%9B%C2%B6mk%C2%8Cu%5D%C3%B7%3F.%C2%84%C2%A0%2CK%C3%A28%3E%05%1F%C2%89%C2%81%C3%97uM%14E%C3%86%C3%82%C2%BD%C3%93%3C%1D%1F%C2%86%01)%C2%A5q%C3%B2%C2%BDk%C2%BB%17%C3%9BB%C2%8Dn%C3%97Z%C3%93%C3%B7%3D%C3%8B%C2%B2Xcg%C2%B3%C3%9Bp%C3%9B%C2%B61%C2%8E%23%C3%AB%C2%BA%C2%92e%C3%99%C3%A1n%C2%B7%C2%BD%C2%A5%C2%87Om%C2%9Eg%C2%B4%C3%96H)%0F%C3%A1%C2%B6%C2%B9(%C2%A5~%5E%C2%B2%C2%B3E%3E%C3%BB%7B%C3%B5%C2%B8%C3%87%3D%C3%BE%2C%7CrdO%01P9(%60%02%C2%AA_G%C3%BD3%C2%AB%C3%B9%C2%8E%083%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%8DIDATH%C2%89%C3%AD%C3%97%C2%B1%0D%C3%83%20%10%C2%85%C3%A1%3F4T%3E%C2%A5%60%03%16%60%03*%C3%8A%C2%93%2C%C2%B1%C2%81%17%60%16o%C2%92%15%3CYR%24%C2%8A%12%25%0D81%C2%85%C3%BDJ%C3%90%C3%A9C%C2%80%0Eq%C3%8A9%C2%9F%C2%81%0B%10%C3%99.%0B0%C2%9A%0E0%0F%C3%AFb%3A%C3%80%C3%8F%05%C2%98N0%00%C3%8D%C2%B8%C2%880M%13%22%C2%B2%3D%C2%AE%C2%AA%0C%C3%83%C2%80%C2%AAn%C2%8B%C3%87%18q%C3%8E%01%C3%A0%C2%9C%23%C3%86%C2%B6kS%C2%8D%7B%C3%AF%09!%C2%BC%C2%8D%C2%85%10%C3%B0%C3%9E%C3%BF%17%17%11RJ_%C3%A7RJ%C3%95%C3%A7_%C2%85%C2%AB*%C3%96%C3%9A%C2%AFs%C3%96%C3%9A%C3%AA%C3%B3%3F%C3%A5%C2%9C%C2%AFU%15%2F)%C2%A50%C3%8Fsky%C3%BBm%C3%BFE%0E%C3%BC%C3%80%0F%C3%BC%C3%80%C3%BF%C2%9A%C2%A6%C3%B6ZJ%C3%B9%18ki%C2%B3%C3%BB%C3%9D%C3%B6%C3%BD%C3%A2%C2%AB%C3%9E%C3%B3%C2%B5%C3%99%C3%AF%C2%B6%1B%C3%AE%C2%9F%C2%B6%1EY%0C0vX%C3%80%02%C2%8C7-%C2%9D%20i%1E%C3%81%5Cj%00%00%00%00IEND%C2%AEB%60%C2%82')
        },
        rollover: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C2%BBIDATH%C2%89%C3%8D%C2%97%C2%B1n%C3%820%10%C2%86%7F%C2%90%C3%A5HQ%C2%84%C2%A201ub%C3%8AR%3C1%C3%81%C3%82%04S%C2%9E%C2%81%C2%8Dw%C3%A2%012e%C2%82%07%C3%B0%C3%A4%09X%C2%B2%C2%90%C2%89%C2%89%09%C2%84%10%C2%8A%C2%94%C2%93%25%C2%BA4U%C3%95%C2%A6%25%09%C3%85%C3%B4%1FOw%C3%BE%C3%BC%C2%9F%C2%AD%C2%93%C3%9D%C2%98%C3%8Df%2F%C2%9C%C3%B3%0Dc%C3%8C%C2%85!i%C2%ADOD%C3%B4%C3%8A8%C3%A7%1B%00%C3%AE%C3%B9%7CF%C2%96e%0F%07%5B%C2%96%05%C3%9B%C2%B6%5D%C3%8E%C3%B9%C2%A6%C3%89%18s%C3%9345%02%06%C2%80%2C%C3%8B%C2%90%C2%A6)%18cn3%0F%C2%98%14%11%01%00%C2%9AF%C2%A9%C3%AF%C2%BA%5E%C2%AF%C3%8F%C2%83%C3%A7z*%C2%9C%15%05%C2%A7%C3%93%C3%A9%C2%B7%C3%98%7C%3E%2F%C2%BDh%C3%99%C3%BA%C3%BF%C3%A7%C2%BCHEn%C2%8C%C3%81%C3%AFm%7B%C2%91%C3%BE%C2%BC%C3%AD%C3%9Dn%C2%B7tn%C2%A1%C3%B3*.s9%C2%8E%C2%83%C3%A1p%08%C3%8E9%C2%96%C3%8B%25%C3%B6%C3%BB%7D%3DxU%C3%B9%C2%BE%0F!%04%C3%A28%C3%86j%C2%B5*%5Dw%17%C3%BC%C2%AB%C3%9B%C3%83%C3%A1P%C2%A9%C2%BE6%5C%08%C2%81%5E%C2%AF%C2%87%C3%B5z%5D%C3%89%C3%AD%5D%C3%B0v%C2%BB%C2%8D%C3%81%60%00%22B%18%C2%86%C2%B8%5C.%C2%B5%C3%80%40%C3%85%C3%9B.%C2%84%C3%80x%3C%C3%86v%C2%BB%C3%85b%C2%B1%C3%B8%11%1C%04%01%1C%C3%87%C2%B9%C2%B9%5E)%C3%A7%C2%9DN%07%C3%BD~%1FD%C2%84(%C2%8An%C2%BA%C3%B5%3C%0FA%10%40J%C2%89%C3%9DnW%0D%5E4%24%C2%A4%C2%94H%C2%92%C2%A4%C3%8C%5E%01%00%C2%9Cs%C2%8CF%23%C3%84q%0C%C2%A5TaN%C3%A9%C2%B6W%01%7F%C2%96%C3%AF%C3%BB%3F%1E%C2%83%C2%91%C3%99%C3%AEy%1E%26%C2%93%09%C3%820%C2%AC%07%C2%BFg%C2%B6%1F%C2%8FGH)%C2%BF%C3%A5%C3%BD%C3%89%C2%84%C3%BBMI%C2%92%40)%C3%B5%C3%B1n%C2%BB%09%C2%AF3%C3%9B%C2%BF%C2%8A%C2%88%C2%A0%C2%94%C3%BA%C3%B5%C2%AE%3C%C3%84y%C3%9E%C3%A6%5B%C3%A3%C3%B6!%C3%B0(%C2%8AJ%C3%A5%3D%C3%BF%C3%B5jY%C2%96Qh%C3%8Ekj%C2%ADO%C2%B6m%1B%C3%99%40%C2%A3%C3%91%C3%88%C3%BFj%C3%90Z%C2%9F%18%11%C2%BDr%C3%8E7%C2%ADV%C3%8B%C3%B8%2F%C3%B5%0Di%C3%B6%C2%B7%25%02%C3%A9%23%C3%B8%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%89IDATH%C2%89%C3%AD%C2%97%C2%BFn%C3%820%10%C3%86%C2%BFD%C2%96%C2%91%C2%A2%08E%20%06%C2%A6NLYJ%26%26%C3%B2%00L%3CC6f%5E%C2%87%07%60bc%C3%89%C2%84%C2%A7L%C2%81%25S%C2%A6L%0C%C2%88%08!%2BR%2CKthS%C2%A9-A%C3%A4O%C3%9D%C2%A5%C3%9Fx%C2%BE%C3%93%2F%3E_%7C%3Em%C2%B1X%C2%BCPJ%0F%C2%84%10%0B%C2%8A%24%C2%A5%C2%BC%08!%5E%09%C2%A5%C3%B4%00%C3%80%C2%BA%5E%C2%AF%C3%88%C3%B3%C3%BC%C3%97%C3%81%C2%9DN%07%C2%86aX%C2%94%C3%92%C2%83N%08%C2%B1%C2%B2%2CS%02%06%C2%80%3C%C3%8F%C2%91e%19%08!%C2%96%5E%18TJ%08%01%00%C3%90%C2%95R%3Ft%C2%BB%C3%9D%00%00%C3%A4%C2%91%C2%93%C3%A7y%3Fl%C2%AB%C3%95%C2%AA5%C3%BF%3F%C3%99%C3%B9%3F%C2%BC5%C3%B8p8%C2%AC%1C%C3%B3%C2%B0%C3%A0%C2%8Ab%C3%B1%3C%C3%AFa%C3%A1%00%C3%80d2%01c%0Cq%1C%3F%C3%A5%0F%C2%B4%C2%B4s%C3%87q%20%C2%84%40%1C%C3%87%C2%95%C3%A2%1A%C3%83%C3%BB%C3%BD%3El%C3%9B%C3%86n%C2%B7%C2%AB%1C%C3%9B%18%3E%C2%9DN%11%C2%86!8%C3%A7j%C3%A1E%C2%BA%C2%A3(%C2%AA%15_%1Bn%C2%9A%26%C3%86%C3%A3q%C2%ADt7%C2%86%C2%BB%C2%AE%C2%8B%C3%BD~_%2B%C3%9D%C2%8D%C3%A0%C2%B6m%C2%83R%C2%8A0%0Ck%C2%83k%C3%81M%C3%93%C2%84%C3%A38%60%C2%8C5%02%C3%97%C2%82%C2%BB%C2%AE%C2%8B(%C2%8Ap%3E%C2%9F%C3%95%C3%82G%C2%A3Q%2B%C3%A9.%C2%A4-%C2%97%C3%8B%C3%9B%C3%A9t%C2%BA%C2%BBx%C2%AF%3Fo%C2%B7%5B%1C%C2%8F%C3%87%2F%C2%B6%C3%B9%7C%C2%8E%5E%C2%AFW%0AI%C3%93%14%C2%9B%C3%8D%C3%A6%C2%8Bm0%18TO%C3%BBw0%00%C3%B8%C2%BE%C3%BF%C3%B94%C3%BA.!%04%7C%C3%9F%C2%BF%C2%BB%C3%96%C3%8A%C3%9D%C3%8E9%2F-%40%C3%86X%C3%A9%C3%AF%C3%98ZKM%C2%92%C3%A4%C3%87M%17E%11%C2%92%24)%C2%8Di%C3%B51%11%04%01%C3%924%05%C3%B0~%C3%8EA%10%3C%C3%B4%7F%C2%AA%C2%9FW%C2%91%C3%AF%C3%BB%C2%98%C3%8Df%C2%A5%C3%A7%C3%BC4%C2%BC%C2%8E8%C3%A7X%C2%AF%C3%97O%C3%B9%C3%AA%C3%80%C3%BB%08%C2%A3R%05O%C2%97R%5E%0C%C3%83P%C3%B2%01%C2%9A%C2%A6%15%C2%B3%1A%C2%A4%C2%94%17%22%C2%84x%C2%A5%C2%94%1E%C2%BA%C3%9D%C2%AE%C3%B2)%C3%B5%0D%C2%A9%C3%83%C3%83i%C3%AE%C3%A6%C2%9DP%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%2BIDATH%C2%89%C3%AD%C2%971%C2%8A%C2%840%14%C2%86%7F%25%3CAD%C3%84%0Bx%00%C2%9B9%C2%84%C2%9D%C2%95g%C2%B0%C3%B3N%1E%C3%80%C3%8A%3BX%C2%A5%C2%B4%C2%B1%C3%8A%19%14%C2%B1%08%18%02n%C2%B3n%C2%B1%C3%A3%C3%8C%C3%AA%C2%80%C2%99%C3%9D%C3%85%C2%AF%7C%C3%BC%C3%B0%C2%BD%C2%BC%C2%84%C2%84XEQDD%C3%942%C3%86%02%18Bk%3D*%C2%A5n%C2%8C%C2%88Z%00%C3%814M%C2%98%C3%A7%C3%B9t%C2%B1%C3%A38p%5D7%20%C2%A2%C3%96f%C2%8C%05RJ%23b%00%C2%98%C3%A7%19RJ0%C3%86%02%7B-%C2%98D)%05%00%C2%B0%C2%8DZ%3FY%C2%96%C3%A5%7D%C3%B2%C2%95K~%C3%89%C3%BF%C2%BE%3C%C3%8B2x%C2%9E%C3%B7%1Ey%18%C2%86%C3%88%C2%B2%0CQ%14%3D%C3%8D%C2%B1%C2%ADb%C2%9E%C3%A7w%C2%B5%C2%B2%2C%0F5%40DH%C2%92%04%5D%C3%97%C2%81s%C2%BE%C2%999%7D%C3%8F%C3%A38~%C2%B8%0D%C2%9B%2B%C3%9Fbk%1A%7B%09%C3%83%10i%C2%9A%C2%A2%C2%AA%C2%AA%C3%97%C3%A4G%C3%86%C3%BE%C2%BD%C3%91a%18%C3%904%C3%8D%5Dn%C2%B7%C3%BCU%C2%84%10%C3%A0%C2%9C%7F%C2%BDd%3F%C3%8A%C2%8F%1E%C2%AE-%C2%94R%C3%A0%C2%9CC%08%C3%B10s%C3%8A%C3%8A%C3%971%C3%B7%7D%C3%BF4w%C2%8A%C2%BC%C2%AE%C3%AB%5D%C2%B9%C3%BFw%C2%BD%5E%C3%B2K%C3%BE%7B%C3%A5%C2%8E%C3%A3%18%C2%95%C2%AE%3E%5Bk%3D%C2%BA%C2%AEk%C2%A4%01%C3%8B%C2%B2%C3%96%C2%BF%1A%C2%B4%C3%96%23SJ%C3%9D%C2%88%C2%A8%C3%B5%7D%C3%9F%C3%B8%2F%C3%B5%03%18%C2%A0nb%C2%92%C3%BF%3E%05%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%016IDATH%C2%89%C3%AD%C2%97Oj%C2%840%18%C3%85%C2%9F%12%12%C3%88B%C3%84%0B%C3%B4%0As%15%C3%8F%C3%A0%C3%8E%3By%00W%1E%20%C2%BB%C2%AC%C2%B2t%C3%A3%01r%C2%81%20%12%C2%84%C2%84%C2%80%5D%C3%8CXf%C3%9A.%C2%9A%C2%94q%C2%A0%C3%B5-%3F%C3%B1%C3%BD%C3%BC%C3%BC%1E%C3%B9%C2%93%C2%B5m%C3%BBF)%1D%09!%25%0ER%08a%C3%B6%C3%9E_%08%C2%A5t%04P.%C3%8B%02%C3%A7%C3%9C%C3%93%C3%81%C2%8C1p%C3%8EKJ%C3%A9%C2%98%13B%C3%8Au%5D%0F%01%03%C2%80s%0E%C3%AB%C2%BA%C2%82%10R%C3%A6%7B%C3%A1Hy%C3%AF%01%00%C3%B9%C2%A1%C3%94%C2%9B%C2%B6m%03%00%C2%90%C2%94%C2%97%C2%9B%C2%A6%C3%B9R%C3%AB%C2%BA.%C3%9A%C3%A7%25%C2%9D%C2%9F%C3%B0%C2%97%C3%82%C2%93%02%C2%B7%C2%87%C2%ABi%C2%9A%C2%A4%C2%A0%C3%AD%C3%BA%C2%BF%C2%BF%C3%BD%C2%84%C2%9F%C3%B0%13%C3%BE%C3%B7%C3%A1Q%C3%8Bk%5D%C3%97%C2%A8%C2%AA%C3%AA%C2%A1v%C2%BF%C2%B7%1Bc0%0C%C3%83%C2%8F%C3%BD%C2%A2%3A%17B%7C%1C%C2%81%3E%C3%8B%7B%0F!D%C2%8C%5D%1C%C3%9CZ%0B)%C3%A5%C2%B7%C3%8F%C2%A4%C2%94%C2%B0%C3%96%3E%0F%0E%00ZkL%C3%93%C3%B4P%C2%9B%C2%A6%09Z%C3%ABX%C2%AB%C2%B4%C3%80)%C2%A5%60%C2%8C%01p%C2%9D%C2%B3R*%C3%85%26%3D%C3%ADB%08Xk%C2%A3%C3%A7%7C%C2%AF%C2%A4%C3%83%04p%C2%9D%7F%C3%9F%C3%B7%C3%89%60%C3%A0%C3%969c%C3%ACW%26%C2%B1%C3%9Ayy%08a%C3%A6%C2%9C%1F%C3%B2%01Y%C2%96%C3%ADw5%C2%84%10f%C3%A2%C2%BD%C2%BFPJ%C3%87%C2%A2(%0E%C2%BF%C2%A5%C2%BE%03%C2%9E%06%7D%C3%9D%19%C2%AB%C3%8F!%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%93IDATH%C2%89%C3%8D%C2%97%C2%B1%C2%8E%C2%AA%40%14%C2%86%7F%C3%88dL%081Dc1%C2%95%1566%1B%2B*%1E%00%C2%AB%C2%A9x%00%3Bk_%C3%87%07%C2%A0%C2%A2%C3%82%C3%8AF%1F%C2%80%C3%98%C3%98%C3%88%03La0%C3%86%10%12%26%24n%C2%B1%C2%97J%C3%96%3B%C2%ACw%C3%87%C3%BB%C2%97g%C2%86%C3%B98%3Fg%0E9%C3%86r%C2%B9%1CSJ%0F%C2%84%10%07%C2%9AT%C3%97%C3%B5UJ%C3%B9A(%C2%A5%07%00%C3%8E%C3%ADvCUU%C2%BF%0E%C3%AE%C3%B5z%C2%B0%2C%C3%8B%C2%A1%C2%94%1ELB%C2%88S%C2%96%C2%A5%160%00TU%C2%85%C2%B2%2CA%08q%C3%8C%26%C2%A0SRJ%00%C2%80%C2%A9%C2%95%C3%BAG%C3%B7%C3%BB%C3%BD%7D%C3%B0Fo%C2%85%C2%93%C2%B6%C3%A0b%C2%B1x%C2%88%C2%AD%C3%97k%C3%A5CU%C2%9F%C3%BF%C3%BF2oS%5B6%C3%9A%C3%A0%C2%AF%C3%9A%C3%9E%26e%C3%9B%5D%C3%97U%C2%86%C2%AB%C3%8AX%C2%ADV%C3%B7%C3%B3%C3%B9%C3%BCt%13c%0C%C2%BE%C3%AF%C2%A3(%0A%C3%ACv%3B%14E%C3%B12x4%1A%C2%A9e.%C2%84%40%14E%10B%C2%80s%C2%8E%C3%A9t%C3%BA2%1C%C3%A8X%C3%ADi%C2%9Ab%C2%B3%C3%99%602%C2%99%60%3E%C2%9F%C3%83%C2%B6m%7Dp%00%C3%88%C3%B3%1Cq%1CC%08%C2%810%0C1%C2%9B%C3%8D%C3%B4%C3%81%1B%C2%A5i%C2%8A(%C2%8A%C3%80%18%03%C3%A7%1C%C3%83%C3%A1%C3%B0%C3%9F%C3%83m%C3%9B%06%C3%A7%C2%BCu%C2%AD(%0A%24I%C2%82%C3%93%C3%A9%C2%84%20%08%3A%C2%BB%C3%B0%14%3E%1E%C2%8F%C3%819%C3%87%600xz%C3%88%C3%B1xD%1C%C3%87%60%C2%8C!%0CC0%C3%86%C2%94%C3%A0%C3%9F6%19%C3%8F%C3%B3%3AUu%C3%A3%C2%82%C3%AB%C2%BA%08%C2%82%C3%A0a%5D%C2%A9%C2%B776%C3%BF%C3%B4%3AeY%C2%A6%C2%BC%C3%B7!%C3%B3%C3%AF%C2%AE%C2%90%C2%96%C3%9E%C2%BE%C3%9Dn%C3%A1%C3%BB%C3%BE%C3%83w%C3%96%C3%92%C3%9B%C3%B3%3CG%C2%92%24%C2%9D%C3%AC%C3%BB%C2%A9Z%0BNJ%C2%89%C3%BD~%0F!%04%3C%C3%8F%03%C2%A5%C2%B4%C3%93%C2%A1%C2%AA.%3D%C3%BD%C2%A5fY%C2%86%C3%8B%C3%A5%02%C3%9F%C3%B7%3B%C3%81U%C3%B5%C3%97%26%C3%93%C2%B4%C3%93%C2%B7%C3%80%7FS%26%C3%B05%C3%82%C3%A8T%C3%833%C3%AB%C2%BA%C2%BEZ%C2%96%C2%A5%C3%A5%05%0C%C3%83hf5%C3%94u%7D%25R%C3%8A%0FJ%C3%A9%C2%A1%C3%9F%C3%AFk%C2%9FR%3F%01s%C3%87%C2%B4%C3%B2%5E%C2%8D%7C%15%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%95IDATH%C2%89%C3%AD%C2%971%C2%8E%C3%AA0%10%C2%86%C3%BFD%C2%96%C2%91%C2%A2%08E%20%C2%8AT%C2%AFJ%C2%95f%C3%B1%01%C2%A0%C2%87%C2%8A3%C2%A4%C2%A3%C3%A6%3A9%00%15%07H%15W%C2%A9%02%C2%8D%2B%C2%AAT%14%C2%88%08!%2BR%C2%ACH%C2%BC%C2%82%C3%87%C2%93v!%C2%B0a%C2%B3%C3%9Ef%C3%BFr4%C2%A3%C3%8F%C2%9E%C2%91%C3%873%C3%86%7C%3E%C3%BFC)%C3%9D%10B%1ChRUUG%C2%A5%C3%94%1B%C2%A1%C2%94n%008%C2%A7%C3%93%09eY~%3B%C2%B8%C3%93%C3%A9%C3%80%C2%B2%2C%C2%87R%C2%BA1%09!NQ%14Z%C3%80%00P%C2%96%25%C2%8A%C2%A2%00!%C3%841%C2%AF%06%C2%9DRJ%01%00L%C2%AD%C3%94%7F%3A%C2%9F%C3%8F%00%00%C3%B2%C3%88)%08%C2%82%1B%5B%18%C2%86%C2%AD%C3%B9%C3%BF%C3%88%C3%8D%7F%C3%A1%C2%8D%C3%A1%C2%AE%C3%AB%C2%B6%067%16%C2%8B%C3%85y%C2%BF%C3%9F%3Ft%0A%C2%82%00a%18%C3%82%C3%B3%3C0%C3%86%C2%B0%5C.%3F%C3%A5%C3%BFH%C2%83%C3%81%C2%A0%C3%99%C3%8D%C2%B7%C3%9B-%C2%A4%C2%94%60%C2%8C5%09%C2%ABU%C3%A3%C2%B4%C3%87q%0C%C3%9F%C3%B7%C3%91%C3%AF%C3%B7%C3%B5%C3%83%C2%A5%C2%94H%C3%93%14%C2%A3%C3%91H%3F%1C%00%C2%84%10PJ%7D9%C3%BD%2F%3F%C2%B58%C2%8E1%1C%0Ea%C3%9B%C2%B6~%C2%B8%C2%94%12%C3%AB%C3%B5%1A%C3%A3%C3%B1X%3F%1C%00%C3%924%05%C2%A5%14%C2%BE%C3%AF%C3%AB%C2%87%03%00%C3%A7%1C%C2%8C%C2%B1%C2%97%C3%92%C3%BFe%C3%B8%C3%A1p%C2%80%10%C3%A2%C2%A5%C3%B4%C2%B7%C3%92%C3%9B%C3%934%C2%85m%C3%9B%C3%B0%3C%C2%AFQ%C3%9C%C3%9D%C3%B6%3A%C2%9B%C3%8D%C3%90%C3%AB%C3%B5j%C2%83%C3%B2%3C%C3%87j%C2%B5zgs%5D%17%C2%93%C3%89%C3%A4%C3%86%C2%B7%C2%AE%C3%8D%C3%96%C2%B6%C3%97(%C2%8A%C3%BE%C2%8F%3A%1F%C2%A5%C2%94B%14E7%C3%B6%C3%9DnW%7B%C3%98%3A%C3%9D%C2%85K)%C3%819%C2%BF%1B%C3%809%C2%87%C2%94%C2%B21%C3%A8%C3%93p%00%C3%88%C2%B2%0CB%C2%88w6!%04%C2%B2%2Ck%05%C3%BC%10%0E%00I%C2%92%20%C3%8Fs%00%C2%97%3A\'I%C3%92%1A%18x2%40%02%C2%97%C3%BAO%C2%A7%C3%93%C2%BBu%C3%BE%C2%A8g%7Fxc%C2%B8%C2%94%C3%B2%C3%A9%C3%B0%C3%B0%C2%AAL%C3%A0%C2%B2%C3%82%C3%A8%C3%94%C2%95gVUu%C2%B4%2CK%C3%8B%01%0C%C3%83%C2%B8%C3%AEj%C2%A8%C2%AA%C3%AAH%C2%94Ro%C2%94%C3%92M%C2%B7%C3%9B%C3%95%C2%BE%C2%A5%C3%BE%05%60%3B%C3%81k%19Fh%C2%8D%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%015IDATH%C2%89%C3%AD%C3%971%C2%AA%C3%83%20%18%07%C3%B0%7F%C2%82%7C%C2%82H%09%C2%B9%40%0E%C3%90%C2%A5%C2%A3%C2%93%5B%C2%A7N%3DC%C2%B7%C3%9E%C2%A9g%C3%88%C3%94%C2%ADSN%C2%90%C3%9D%03d%C2%B2%C2%84%0E%01E%C3%A8%5B%5E%C2%A6%C3%B4%C2%B5ix%C2%B1%0F%5E%C3%BE%C2%A3%C2%8A%3F%C2%95O%C3%81%C3%A4x%3C%16DT3%C3%862DJ%08%C2%A1%C3%B5%C3%9Eo%18%11%C3%95%00%C2%B2%C3%9B%C3%AD%06%C3%A7%C3%9C%C3%AC0%C3%A7%1CB%C2%88%C2%8C%C2%88%C3%AA%C2%941%C2%96u%5D%17%05%06%00%C3%A7%1C%C2%BA%C2%AE%03c%2CK%C3%BB%C2%86%C2%98%C3%B1%C3%9E%03%00%C3%92%C2%A8%C3%AAw%C3%AE%C3%B7%C3%BB%C3%A7%C3%B0%3E%0B%C2%BE%C3%A0%7F%0B%C2%97Rb%C2%BF%C3%9F%C3%8F%C2%82%C2%B3g%C2%9DEQ%40k%0D%22%C2%8A%C2%8B%2B%C2%A5%C2%B0%5E%C2%AF\'Mz8%1C%06m%C2%A7%C3%93%C3%A95.%C2%A5%C3%84v%C2%BBE%C2%9E%C3%A7%C2%93%C3%A0w2%C3%80w%C2%BB%1D%C2%A4%C2%94%C2%83%C2%81%C2%8Fv%C3%B3%C3%AB%C3%B8%C3%A5r%C2%81%C3%96z%C2%B0%C3%B3G%C3%87%C3%B6S%C3%86.tP%C3%AD%C3%96Z%C2%9C%C3%8Fg%18cFcS%C3%B3%C2%B0%C3%A0%C2%BC%C3%B7%C2%A8%C2%AA%0AM%C3%93%40)%C3%B5v%C2%B5%C2%8F%3D%C2%A5%C2%A7W%C3%8D%18%C2%83%C3%AB%C3%B5%0A%C2%AD%C3%B5%5B%C3%B8%C3%98%C2%BC%7Cd%C2%AC%C2%B5(%C3%8B%C3%B23%C3%B8%C2%9CY%C3%B0%05%C3%BFg8%C3%A7%3C*%C3%9A%7Bi%08%C2%A1%15BDY%40%C2%92%24%C3%BD_%0D!%C2%84%C2%96y%C3%AF7DT%C2%AFV%C2%AB%C3%A8%C2%BF%C3%94%2F%C2%B3%C2%B4f%C2%82X%5Cn%C3%99%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%18IDATH%C2%89%C3%AD%C2%971%C2%AA%C3%84%20%10%C2%86%C3%BF%04%19A%24%04%2F%C3%A0%01l%C2%B6Le%C3%A7%19r%00%C2%BB%C3%9C)g%C3%88Ur%C2%94%C2%90%22%10%11%C3%B2%C2%AA%C3%80.o%1B%C3%9D%17_%C2%B1%C3%B9%2B%C2%9D%C3%91%C3%B9dfP%C2%AC%C2%86a%C3%90D43%C3%86Z%14R%C2%8Cq%09!%3C%18%11%C3%8D%00%C3%9Au%5D%C2%B1%C3%AF%C3%BB%C3%A5%60%C3%8E9%C2%84%10-%11%C3%8D5c%C2%AC%C3%9D%C2%B6%C2%AD%08%18%00%C3%B6%7D%C3%87%C2%B6m%60%C2%8C%C2%B5%C3%B5i(%C2%A9%10%02%00%C2%A0%C3%8E%0D%20%C2%A5D%C3%9F%C3%B7%C2%90R%26%C3%AF%3D%C2%8E%C3%A33%C2%B8s%0ERJ8%C3%A7rC%C3%A4%C3%81%C2%BB%C2%AE%C2%83R%0A%00%C2%A0%C2%94B%C3%97ue%C3%A0Zk%18c%5El%C3%86%18h%C2%AD%C2%AF%C2%85K)a%C2%AD%7D%C3%AB%C2%B3%C3%96%26%C3%97%3F%09%C3%AE%C2%9C%03%11%C2%BD%C3%B5%11Qr%C3%BDY%C3%8A%C3%A2i%C2%9A%5E%C3%A6%C3%9E%7B%C2%8C%C3%A3%C2%98%04%7CVv%C2%B7%C3%BF%C2%85n%C3%B8%0D%C2%BF%C3%A17%C3%BCR%25%5D%C2%AF%C2%A7%C2%BC%C3%B7%C2%BF%C3%869%C3%97%C3%AC%C3%B7%C2%A6%C3%BD%7B%C3%A1Y%0D%C3%B7%C3%89%1B%C3%BE%C2%AC%C3%BFO%3B%C3%A7%C2%BC(%C3%B4%C3%A4%C3%951%C3%86E%08Q%C3%A4%00UU%C2%9D%7F5%C3%84%18%17%16Bx%10%C3%91%C3%9C4M%C3%B1_%C3%AA%0F%14%C2%A6N%C3%8E%C2%94e%1A%14%00%00%00%00IEND%C2%AEB%60%C2%82')
        },
        pressed: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%80IDATH%C2%89%C3%8D%C2%97%C2%B1%C2%8A%C2%AB%40%14%C2%86%C3%BF%2B%C2%81%C2%81%11D%C2%84)R%C2%84%C3%BB%00i%16%C2%BB%C3%A9RZ%C2%A5%C2%B4%C2%B3%C3%8A%1B%C3%A4A%C3%B2%04vv%C3%A6%05b%15%3B%3BIc%C2%AAt)R%08%12%02%1AN%13%C2%B6Ya%C2%B9%C3%ABnF%C2%B3%C2%99%C3%9C%C2%BF%1C%C3%8E%C2%99o%C3%BE%C3%A3%C3%B0%C3%A3%C3%BCY%2C%16%7F9%C3%A7%3B%C3%86%C2%98%0DM%22%C2%A2s%C3%934o%23%C3%8E%C3%B9%0E%C2%80%7D%3A%C2%9DP%C3%97%C3%B5%C3%93%C3%81%C2%A6i%C3%82q%1C%C2%9Bs%C2%BE3%18cvUUZ%C3%80%00P%C3%975%C2%AA%C2%AA%02c%C3%8C6%C3%9A%05%C2%9D%C2%BA%5E%C2%AF%00%00C%2B%C3%B5C%C2%B7%C3%9B%C3%ADu%C3%B0V%2F%C2%85%C2%8F%C2%BA%16%C2%97%C3%8B%C3%A5%C2%97%C2%B5%C3%95j%C2%A5%C2%BC%C2%A9j%C3%BF%C3%BF%C3%A7%C2%BCK%5Dn%C2%B4%C3%81%1F%1D%7B%C2%97~%7D%C3%AC%C3%93%C3%A9T%C2%B9%C2%B6%C3%93y%1F%C2%97%C2%AD%2C%C3%8B%C2%82%C3%A7y%60%C2%8Ca%C2%BD%5E%C3%A3x%3C%0E%C2%83%C3%B7%C2%95%C3%AB%C2%BA%C2%90R%22%C3%8FsdY%C2%A6%C3%9C%C3%B7%10%C3%BC%C2%B3%C3%9B8%C2%8EQ%C2%96e%C2%AF%C3%BE%C3%81p)%25%C2%A4%C2%94%C3%88%C2%B2%C2%AC%C2%97%C3%9B%C2%87%C3%A0B%08x%C2%9E%07%22B%18%C2%86%C2%B8%5C.%C2%83%C3%80%40%C3%8F%C3%9B.%C2%A5%C2%84%C3%AF%C3%BB(%C2%8A%02q%1C%7F%0B%0E%C2%82%00%C2%96e%C3%9D%C3%9DO%C3%89%C3%B9d2%C3%81l6%03%11!%C2%8A%C2%A2%C2%BBn%C2%85%10%08%C2%82%00I%C2%92%C3%A0p8%C3%B4%C2%83w%C2%85D%C2%92%24(%C2%8AB%C3%A5%C2%AC%00%00%C3%86%18%C3%A6%C3%B39%C3%B2%3CG%C2%9A%C2%A6%C2%9D5%C3%8Ac%C3%AF%03%C3%BE%2C%C3%97u%C2%BF%C3%BD%0CZ%C2%B2%5D%08%01%C3%9F%C3%B7%11%C2%86%C3%A10%C3%B8%23%C3%99%5E%C2%96%256%C2%9B%C3%8D%C2%97%C2%BA_I%C2%B8%C2%9F%C2%B4%C3%9F%C3%AF%C2%B1%C3%9DnADj%C3%B0!%C3%99%C3%BE%C2%AF%C2%88%08i%C2%9A%C3%BExW%C2%9E%C3%A2%C2%BC%1D%C3%B3%C2%BD%C2%B8%7D%0A%3C%C2%8A%22%C2%A5%C2%BA%C3%97%C3%BF%C2%BD%C2%9A%C2%A6%C2%A9%15%C3%9A%C3%B2%0C%22%3A%3B%C2%8E%C2%A3%C3%A5%00%C2%86a%C2%B4o5%10%C3%91y%C3%944%C3%8D%1B%C3%A7%7C7%1E%C2%8F%C2%B5%C2%BFR%C3%9F%01%C2%9E%C3%A1%C2%B6%C2%9E%C2%95%60%C3%93%02%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C2%BDIDATH%C2%89%C3%AD%C2%97%C2%BF%C2%8A%C3%AA%40%14%C3%86%C2%BF%0D%C3%82%C3%80%08%12%02)%2Cd%1B%3B%C2%9B%25%5D%3A%C3%8B%3C%C2%81%C2%9D%C2%95o%10%7C%0E%C2%9F%C3%80%C3%8E%C3%92j%3A%C2%A7K%C2%97n%C2%B0I%C2%97%C3%8E%C3%82b%20H%20%C2%91%C3%93%C2%B8%C2%B7%C2%B8%C2%9B%C2%85%C3%9D%C2%8Db%C3%BE%C3%9C%C2%B9%C3%8D~%C3%A5%C2%99s%C3%B8%C3%8D%C2%9C93%C2%87%C3%B3%C2%B2Z%C2%AD%5E9%C3%A7G%C3%86%C2%98%0DC%22%C2%A2KY%C2%96o%03%C3%8E%C3%B9%11%C2%80%7D%3E%C2%9FQ%14%C3%85%3F%07%0F%C2%87C8%C2%8Ecs%C3%8E%C2%8F%16c%C3%8C%C3%8E%C2%B2%C3%8C%08%18%00%C2%8A%C2%A2%40%C2%96e%60%C2%8C%C3%99Ve0%C2%A9%C3%AB%C3%B5%0A%00%C2%B0%C2%8CR%3Ft%C2%BB%C3%9D%00%00%C2%83GNa%18%C3%BE%C2%B0m6%C2%9B%C3%9E%C3%BC%C3%BF%C3%8B%C3%89%7F%C3%A1%C2%BD%C3%81\'%C2%93I%C3%A3%C2%98%C2%87%05W%15K%18%C2%86%0F%0B%07%00%C3%A6%C3%B39%C2%A4%C2%94H%C2%92%C3%A4)%7F%C2%A0%C2%A7%C2%93%C3%BB%C2%BE%0F%22B%C2%92%24%C2%8D%C3%A2%3A%C3%83%5D%C3%97%C2%85%C3%A7y8%1C%0E%C2%8Dc%3B%C3%83%C2%83%20%40%1C%C3%87%C3%88%C3%B3%C3%9C%2C%C2%BCJ%C2%B7R%C2%AAU%7Ck%C3%B8h4%C2%82%C3%AF%C3%BB%C2%AD%C3%92%C3%9D%19%C3%9E%25%C3%9D%C2%9D%C3%A0%C2%9E%C3%A7%C2%811%C2%868%C2%8E%5B%C2%83%5B%C3%81%C3%BBHwkx%10%04PJAkm%16%3E%C2%9B%C3%8DzIw%C2%A5%C2%97%C3%B5z%C3%BD%C2%9E%C2%A6i%C3%ADb%5D%7F%C3%9E%C3%AF%C3%B78%C2%9DN_l%C3%8B%C3%A5%12%C2%AE%C3%AB%C3%9E%C2%85h%C2%AD%C2%B1%C3%9B%C3%AD%C2%BE%C3%98%C2%A6%C3%93i%C3%B3%C2%B4%7F%07%03%C2%80%10%02DT%C3%ABOD%10B%C3%94%C2%AE%C3%B5%C3%B2%C2%B7%C3%A7y%0E)e%C3%AD%C2%9A%C2%94%C3%B2%C3%AEs%C3%AC%C2%AD%C2%A5%C2%A6i%C3%BA%C3%A3%C2%A7SJ%C3%A1%C3%9E%C2%95%C3%B6%0A%07%C2%80(%C2%8A%3E_%C2%81%C3%96%1AQ%14%3D%C3%B4%7F%C2%AA%C2%9F7%C2%91%10%02%C2%8B%C3%85%C3%A2%C3%AE%3D%3F%0Do%C2%A3%3C%C3%8F%C2%B1%C3%9Dn%C2%9F%C3%B2%C2%B5%C2%80%C2%BF%23%C2%8CIU%3C%C2%8B%C2%88.%C2%8E%C3%A3%18%C3%99%C2%80eY%C3%95%C2%AC%06%22%C2%BA%0C%C3%8A%C2%B2%7C%C3%A3%C2%9C%1F%C3%87%C3%A3%C2%B1%C3%B1)%C3%B5%0F%C2%B0%19%C3%81%C2%AD%C2%91%019%C3%9F%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%010IDATH%C2%89%C3%AD%C2%971%C2%8A%C2%84%40%10E%C3%BF6BC%0B%22%C2%82%C2%81%C3%91%26%1E%60%C3%B0%20%C2%86fF%C3%9E%C3%80%C2%83x%023%2Fa%C2%A6\'%10%13%C2%BD%C2%82A%C2%83%C2%98%C2%A8T%22%C2%9B%C2%AC%C3%89%C2%8E3%C3%A8%C2%80%3D%C2%BB%C2%8B%2F%2C%3E%C2%BC%C3%AA%C2%AA%C2%A6%C2%A1%3F%C2%A2(%C3%BA%14B%C3%94%C2%9Cs%13%C2%8A%20%C2%A2a%C2%9A%C2%A6%C2%9B%26%C2%84%C2%A8%01%C2%98%5D%C3%97a%1C%C3%87%C3%93%C3%85%C2%BA%C2%AE%C3%83%C2%B2%2CS%08Q3%C3%8E%C2%B9%C3%99%C3%B7%C2%BD%121%00%C2%8C%C3%A3%C2%88%C2%BE%C3%AF%C3%8197%C3%99ZP%C3%89%3C%C3%8F%00%00%C2%A6%C3%94%C3%BA%C3%8D%C2%B2%2C%C3%AF%C2%93%C2%AF%5C%C3%B2K%C3%BE%C3%B7%C3%A5a%18%C3%820%C2%8C%C3%B7%C3%88m%C3%9BF%18%C2%86p%5D%C3%B7iN%C3%9B*%C3%86q%7CWK%C2%92%C3%A4P%03%C2%9Cs%C3%B8%C2%BE%C2%8F%C2%AA%C2%AAP%C2%96%C3%A5f%C3%A6%C3%B4%C2%9D%7B%C2%9E%C3%B7p%0D%C2%9B\'%C3%9Fbk%1A%7B%C2%B1m%1BA%10%20M%C3%93%C3%97%C3%A4G%C3%86%C3%BE%C2%B3Q)%25%C3%B2%3C%C2%BF%C3%8B%C3%AD%C2%96%C2%BFJ%C3%9B%C2%B6(%C2%8A%02D%C2%B4O~%C3%B4rmAD(%C3%8B%12M%C3%93%3C%C3%8C%C2%9Cr%C3%B2u%C3%8CR%C3%8A%C2%A7%C2%B9S%C3%A4Y%C2%96%C3%AD%C3%8A%C3%BD%C2%BF%C3%A7%C3%B5%C2%92_%C3%B2%C3%9F%2B%C3%97u%5D%C2%A9t%C3%B51%22%1A%2C%C3%8BR%C3%92%00cl%C3%BD%C2%AB%C2%81%C2%88%06m%C2%9A%C2%A6%C2%9B%10%C2%A2v%1CG%C3%B9%2F%C3%B5%0Bh%C3%B6n5%C2%9E%20%3A%C2%8D%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%018IDATH%C2%89%C3%AD%C2%971%C2%8A%C2%840%14%C2%86%C3%BF%0DB%20iD%C2%B0%C2%B0%C3%9A%2B%C3%8C%C2%85%C2%AC%C2%BC%C2%81%07%C3%B1%04v%C2%9E%20e%3AO%20%1E%20%C2%BD%C2%85%20%12%C2%88%C3%B2%1A%C3%99bF%C2%98a%C2%A7%C3%98d%19%07v%C3%BD%C3%8B%17%C3%BC%3F%C2%9F%C3%AF\'1%1FEQ%7C%0A!z%C3%8Ey%C2%8C%C2%83DD%C3%B3%C2%B2%2C%C2%97H%08%C3%91%03%C2%88%C2%87a%C2%80s%C3%AE%C3%A5%60)%25%C2%92%24%C2%89%C2%85%10%3D%C3%A3%C2%9C%C3%87%C3%934%1D%02%06%00%C3%A7%1C%C2%A6i%02%C3%A7%3Cf%7B%C3%A1H%C2%AD%C3%AB%0A%00%60%C2%87Ro%C3%9A%C2%B6%0D%00%10%C2%85%3C%5C%C2%96%C3%A5%C2%B7ZUU%C3%9E%3Eo%C3%A9%C3%BC%C2%84%C2%BF%15%1E%14%C2%B8%3D%5CeY%06%05m%C3%97%C3%BF%C3%BD%C3%AC\'%C3%BC%C2%84%C2%9F%C3%B0%C2%BF%0F%C3%B7%C3%9A%5E%C3%B3%3CG%C2%9A%C2%A6%0F%C2%B5%C3%BB%C2%B3%7D%1CG4M%C3%B3c%3F%C2%AF%C3%8E%C2%95R%20%C2%A2%C2%A7kD%04%C2%A5%C2%94%C2%8F%C2%9D%1F%C3%9CZ%0B%C2%AD%C3%B5%C3%935%C2%AD5%C2%AC%C2%B5%C2%AF%C2%83%03%C2%801%06%5D%C3%97%3D%C3%94%C2%BA%C2%AE%C2%831%C3%86%C3%97*%2Cpm%C3%9Bb%1CG%00%C3%979%C2%B7m%1Bb%13%C2%9Ev%C2%A5%14%C2%AC%C2%B5%C3%9Es%C2%BEW%C3%90%C3%8F%04p%C2%9D%7F%5D%C3%97%C3%81%60%C3%A0%C3%96%C2%B9%C2%94%C3%B2W%26%C2%BE%C3%9Ay%C2%8C%C2%88%C3%A6%24I%0Ey%01%C3%86%C3%98~W%03%11%C3%8D%C3%91%C2%B2%2C%17!D%C2%9Fe%C3%99%C3%A1%C2%B7%C3%94%2F%C3%B33%7Ci%C3%A2%C3%96%C3%AD%C2%A5%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%97IDATH%C2%89%C3%8D%C2%97%C2%B1%C2%8A%C3%AA%40%14%C2%86%C3%BF%0D%C3%82%C3%80%08!%04%2C%2C%C3%82m%C3%92M%C2%B3%C3%98%C3%99%C2%A5K%5E%C3%80t%C2%A9%C3%B4%09%C3%84%C3%A7%C3%B0%09%C3%92%C3%99%C2%A5%C2%B3%C2%8BO%C2%90Nl%C3%86*%C3%9D%14%C2%82%C2%81%20%C2%81D%C2%A6qoqo%C2%9Ak%C2%AE%3B%C3%AA%C3%AE%C2%B8_93%C2%99%2F%C3%A7p%C3%8E%C2%81y%C2%9BN%C2%A7%C2%BF(%C2%A5%3BB%C2%88%05MH)OM%C3%93%C2%BC%C3%B7(%C2%A5%3B%00%C3%96%C3%A1p%40%5D%C3%97%C3%9F.%C3%AE%C3%B7%C3%BB%C2%B0m%C3%9B%C2%A2%C2%94%C3%AE%0CB%C2%88U%C2%96%C2%A5%161%00%C3%94u%C2%8D%C2%B2%2CA%08%C2%B1%C2%8CvA\'%C3%A7%C3%B3%19%00%60h%C2%B5%C3%BE%C3%A5r%C2%B9%C2%BCN%C3%9E%C3%B2Ry%C2%AFkq%3E%C2%9F_%C2%AD-%C2%97K%C3%A5KU%C2%BF%C3%BFy%C2%91w%C3%91%15%C2%8D6%C3%B9%C2%B3i%C3%AFB9%C3%AD%C2%8C1e%C2%B9*o%C2%8B%C3%85%C3%A2%23%C3%8F%C3%B3%C2%9B%C2%87%1C%C3%87%C2%81%C3%AF%C3%BB%C2%A8%C2%AA%0Ai%C2%9A%C2%A2%C2%AA%C2%AA%C2%A7%C3%85%C2%AE%C3%AB%C2%AAE.%C2%84%40%1C%C3%87%10B%20%C2%8A%22%C2%8CF%C2%A3%C2%A7%C3%A5%C3%80%C2%9D%C3%95%C2%9Ee%19%C2%92%24%01c%0Ca%18%C3%824M%7Dr%00(%C2%8A%02%C2%AB%C3%95%0AB%08%C3%8Cf3%C2%8C%C3%87c%7D%C3%B2%C2%96%2C%C3%8B%10%C3%871%1C%C3%87A%14E%18%0C%06_%2F7M%13Q%14u%C3%AEUU%C2%85%24I%C3%809G%18%C2%86wg%C3%A1f%C2%9F%C2%BB%C2%AE%0B%C3%9F%C3%B7A%08%C2%B9y%C3%89v%C2%BBE%C2%9E%C3%A7%08%C2%82%00%C2%8C1l6%1B%08!%1E%C2%97%7B%C2%9EwWU%C2%B7Y%60%C2%8Ca2%C2%99%5C%C3%AD%2B%C3%8D%C3%B66%C3%8D%C2%8F%C2%B6%13%C3%A7%5C%C3%B9%C3%ACU%C3%A4%C3%BFk!-%C2%B3%7D%C2%BD%5E%23%08%C2%82%C2%AB%C3%AA%C3%952%C3%9B%C2%8B%C2%A2%40%C2%92%24%C3%98%C3%AF%C3%B7%C3%8A%C2%B2G%C3%A9%2C8)%25%C3%924%C2%85%10%02%C2%9E%C3%A7%7DZ%C3%AD%C3%BF%C2%A2%C2%9A%C2%A5%C2%9B%C2%AD%C3%869%C3%87%C3%B1xD%10%04w%C3%89U%C3%B9t%C3%88%C2%B4%C3%A3%C3%B4%25%C3%B2%C3%AF%C3%84%00%C3%BE%3Cat%C3%92%C3%BA%0C)%C3%A5%C3%89%C2%B6m-%3F%60%18F%C3%BBV%C2%83%C2%94%C3%B2%C3%94k%C2%9A%C3%A6%C2%9DR%C2%BA%1B%0E%C2%87%C3%9A_%C2%A9%C2%BF%01n%C2%83%C2%BA(%06%C3%8ET%C2%88%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%86IDATH%C2%89%C3%AD%C2%971%C2%8E%C3%AA0%10%C2%86%C3%BF%C2%8D%C2%90%2C%19)%C2%8A%22QPD%C3%9B%C3%90%C2%A5Y%C2%A5%C3%B3%0D%C2%B8%00tT%C3%9C%20%C3%A2%1C9%01%1D\'H%C2%97t%C3%A9%C3%9CY4t%C3%A9((%22E%C3%88R%C2%82%C2%A6a_%C3%81%C2%A3%C3%98%25%C2%81%0D%C3%A4%C3%B95%C3%BB%C2%97%C2%A3%19%7D%C3%B6%C2%8C%C3%BC%C3%9B~%5B.%C2%97%C3%AF%C2%9C%C3%B3-c%C3%8C%C2%81!%11%C3%91%C2%B1%C2%AE%C3%AB%C2%8F%01%C3%A7%7C%0B%C3%809%1C%0E%C2%A8%C2%AA%C3%AA%C2%9F%C2%83%C2%87%C3%83!%5C%C3%97u8%C3%A7%5B%C2%8B1%C3%A6%C2%94ei%04%0C%00UU%C2%A1%2CK0%C3%86%1C%C3%AB%1A0%C2%A9%C3%93%C3%A9%04%00%C2%B0%C2%8CR%C3%BF%C3%AA%7C%3E%03%00%06%C3%B7%C2%92%C3%820%C2%BC%C2%89EQ%C3%94%5B%C3%BE%7F%C3%99%C3%B9%2F%C2%BC3%C3%9C%C3%B3%C2%BC%C3%9E%C3%A0o%C2%AB%C3%95%C3%AA3%C3%8F%C3%B3%C2%BBIa%18%22%C2%8A%22%C3%B8%C2%BE%0F!%04%C3%96%C3%AB%C3%B5%C2%8F%C3%B2%C3%AFi2%C2%99t%C3%9B%C3%B9n%C2%B7%C2%83%C3%96%1AB%C2%88.e%C2%AD%C3%AA%C3%9C%C3%B6%24I%10%04%01F%C2%A3%C2%91y%C2%B8%C3%96%1ARJL%C2%A7S%C3%B3p%00PJ%C2%81%C2%88%5En%C3%BF%C3%93G-I%12%08!%60%C3%9B%C2%B6yx%1F%C3%AD%7F%C3%89d%C2%A4%C2%94%60%C2%8C!%08%02%C3%B3p%C3%A0%C2%B5%C3%B6%C2%BF%0C%2F%C2%8A%02J%C2%A9%C2%A7%C3%9A%C3%9F%C2%8B%C2%B7K)a%C3%9B6%7C%C3%9F%C3%AFT%C3%97h%C2%AF%C2%8B%C3%85%C3%A2%C2%AE%C2%89%14E%C2%81%C3%8Df%C3%B3%25%C3%A6y%1Ef%C2%B3%C3%99Mn%C2%9B%C3%8D%C2%B6%C3%9Ak%1C%C3%87%20%C2%A2%C3%86%22%22B%1C%C3%877%C3%B1%C3%BD~%C3%9F%C2%BA%C3%9865%C3%82%C2%B5%C3%96H%C3%93%C2%B4%C2%B1%20MSh%C2%AD%3B%C2%83~%0C%07%C2%80%3C%C3%8F%C2%A1%C2%94%C3%BA%12SJ%C3%A1%C3%91%0D%C3%98%0B%1C%00%C2%B2%2CCQ%14%00.s%C3%8E%C2%B2%C2%AC70%C3%B0%C3%A0%01%09%5C%C3%A6%3F%C2%9F%C3%8F%1B%C3%A7%C3%BC%5D%C2%8F%C3%AE%C3%B0%C3%8Ep%C2%AD%C3%B5%C3%83%C3%87%C3%83%C2%B3%C2%B2%C2%80%C3%8B%17%C3%86%C2%A4%C2%AE%3C%C2%8B%C2%88%C2%8E%C2%AE%C3%AB%1AY%C2%80eY%C3%97%C2%BF%1A%C2%88%C3%A88%C2%A8%C3%AB%C3%BA%C2%83s%C2%BE%1D%C2%8F%C3%87%C3%86%7F%C2%A9%7F%00%C2%9Dv%C2%BF%C3%B1%C2%A6%5B%C3%A52%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%3CIDATH%C2%89%C3%AD%C3%971%C2%8A%C2%840%18%05%C3%A0%C2%B7A%08D%08%22XXm3%07%18%C3%A6%04v%C3%9A%C3%99%C3%99Y%C3%8D%0D%3C%C2%88%C2%97%C2%B0%C2%B3%C2%B3%C3%B3%186z%01%C2%8B%01%05%C2%B1QI%23%C3%9B%C2%AC%C2%95%C3%AE%C2%8C%C3%8A%C2%9AYX_%C2%99%C2%84%7CI%C3%B8%13%C3%88%C3%87%C3%BD~%C3%BFd%C2%8Ce%C2%94R%0D%C2%92%22%C2%84h%C3%BB%C2%BE%C2%BF*%C2%8C%C2%B1%0C%C2%80%C3%B6x%3C%C3%90u%C3%9D%C3%A1%C2%B0%C2%AA%C2%AA%C3%90u%5Dc%C2%8Ce%C2%84R%C2%AA5M%23%05%06%C2%80%C2%AE%C3%AB%C3%904%0D(%C2%A5%1A%C2%99%1Adf%18%06%00%00%C2%91%C2%AA~g%1C%C3%87%C3%B7%C3%A1SN%C3%BC%C3%84%C3%BF%16%C3%8E9%C2%87%C3%AF%C3%BB%C2%87%C3%A0%C3%8A%C2%B3%C3%8E%C3%8B%C3%A5%02%C3%9B%C2%B6A)%C2%95%C2%8B%5B%C2%96%C2%85%C3%9B%C3%AD%C2%B6k%C3%92%20%08fma%18%C2%BE%C3%869%C3%A7p%5D%17%C2%86a%C3%AC%C2%82%C2%B7d%C2%86%7B%C2%9E%07%C3%8E%C3%B9l%C3%A0%C3%92n~%1DO%C2%92%04%C2%8E%C3%A3%C3%8Cv%C2%BEtl%3Fe%C3%ADBg%C3%95%5E%C3%975%C3%A28FQ%14%C2%AB%C2%B1%C2%BDY%2C8!%04%C3%924EY%C2%96%C2%B0%2Cks%C2%B5%C2%AF%3D%C2%A5%C2%A7W-%C3%8FsTU%05%C3%87q6%C3%A1k%C3%B3%C3%B2%C2%91%C2%A9%C3%AB%1AQ%14%C2%BD%07%3F2\'~%C3%A2%C3%BF%0CWUU*%3AyD%08%C3%91%C3%AA%C2%BA.e%01%C2%84%C2%90%C3%A9%C2%AF%06!D%C2%AB%C3%B4%7D%7Fe%C2%8Ce%C2%A6iJ%C3%BF%C2%A5~%01wzf%C3%B79M%7B%C2%AE%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%1FIDATH%C2%89%C3%AD%C2%971%C2%AA%C3%84%20%10%C2%86%C3%BF\'%C2%81%01%03%12%02%16%C2%A9%5E%C3%A3%01%16O%C2%90.%C2%A5%C2%87HN%C3%A0Ar%C2%89%5C(%C2%B7H%17%C3%92%C3%A8b%13%5E%15%C3%98%C3%A5m%C2%A3%C3%BB%C3%A2%2B6%7F%C2%A53%3A%C2%9F%C3%8C%0C%C2%8A_%7D%C3%9F%7Fs%C3%8Eg%22%C2%AA%C2%90I!%C2%84%C3%8D%7B%7F%2B8%C3%A73%C2%80jY%168%C3%A7N%07%C2%97e%C2%89%C2%BA%C2%AE%2B%C3%8E%C3%B9%C3%8C%C2%88%C2%A8Z%C3%975%0B%18%00%C2%9CsX%C3%97%15DT%C2%B1%C3%83%C2%90S%C3%B7%C3%BB%1D%00%C3%80R%03%08!0%0C%03%C2%84%10%C3%91%7B%C3%B7%7D%7F%0Fn%C2%8C%C2%81%10%02%C3%86%C2%98%C3%94%10i%C3%B0%C2%B6m!%C2%A5%04%00H)%C3%91%C2%B6m%1E%C2%B8R%0AZ%C3%AB\'%C2%9B%C3%96%1AJ%C2%A9s%C3%A1B%08t%5D%C3%B7%C3%92%C3%97u%5Dt%C3%BD%C2%A3%C3%A0%C3%86%18%10%C3%91K%1F%11E%C3%97%C2%BF%C2%88Y%3CM%C3%93%C3%93%C3%9CZ%C2%8Bq%1C%C2%A3%C2%80%C2%8FJ%C3%AE%C3%B6%C2%BF%C3%90%05%C2%BF%C3%A0%17%C3%BC%C2%82%C2%9F%C2%AA%C2%A8%C3%AB%C3%B5%C2%90%C2%B5%C3%B6%C3%978%C3%A5%C2%9A%C3%BD%C3%9C%C2%B4%7F.%3C%C2%A9%C3%A1%C3%9Ey%C3%83%1F%C3%B5%C3%BFi%2F%C3%8B2%2B%C3%B4%C3%A0%C2%B1%10%C3%82V%C3%97u%C2%96%030%C3%86%C2%8E%C2%BF%1AB%08%5B%C3%A1%C2%BD%C2%BFq%C3%8E%C3%A7%C2%A6i%C2%B2%C3%BFR%7F%00v%3EP6%C2%85%06%C3%B1%7C%00%00%00%00IEND%C2%AEB%60%C2%82')
        },
        checked: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C2%BCIDATH%C2%89%C3%8D%C2%97!%C2%8F%C2%AB%40%14%C2%85%C3%8F%23M%26d%10%04%5CG%3CU%C2%BF%C2%AE%16Q3%C2%B6%1AY%C2%8D%C3%85b%C2%B1%C3%93%C2%BFPd%C2%B1X%2CUMm5u4%15%C3%90%C3%89%C2%98f%C3%8D%C2%92l%C3%9E%C2%B2%5B%C2%A0%C3%9B%C3%A9%3Brr%C3%AF%7Cs.%C2%93%13%C3%A6%C3%8Fb%C2%B1%C3%B8K)%C3%9D%13Blh%C2%92R%C3%AA%C3%924%C3%8D%C3%9B%C2%84R%C2%BA%07%60%C2%9FN\'%C3%94u%C3%BDt%C2%B0eYp%5D%C3%97%C2%A6%C2%94%C3%AE%0DB%C2%88%5DU%C2%95%160%00%C3%94u%C2%8D%C2%AA%C2%AA%40%08%C2%B1%C2%8DvA%C2%A7%C2%AE%C3%97%2B%00%C3%80%C3%90J%C3%BD%C3%90%C3%ADv%7B%1D%C2%BC%C3%95K%C3%A1%C2%93%C2%AEE!%C3%84%C2%97%C2%B5%20%08zo%C3%9A%C2%B7%C3%BF%C3%BFs%C3%9E%C2%A5.7%C3%9A%C3%A0%C2%8F%C2%8E%C2%BDK%C2%BF%3E%C3%B6%C3%B9%7C%C3%9E%C2%BB%C2%B6%C3%93%C3%B9%10%C2%97%C2%AD%1C%C3%87%C2%81%C3%AF%C3%BB0M%13%C3%AB%C3%B5%1A%C3%87%C3%A3q%1C%7C%C2%A8%3C%C3%8F%03%C3%A7%1Cy%C2%9E%23%C3%8B%C2%B2%C3%9E%7D%0F%C3%81%3F%C2%BB%15B%C2%A0%2C%C3%8BA%C3%BD%C2%A3%C3%A1%C2%9Csp%C3%8E%C2%91e%C3%99%20%C2%B7%0F%C3%81%19c%C3%B0%7D%1FRJDQ%C2%84%C3%B3%C3%B9%3C%0A%0C%0C%C2%BC%C3%AD%C2%9Cs%04A%C2%80%C2%A2(%20%C2%84%C3%B8%16%1C%C2%86!%1C%C3%87%C2%B9%C2%BB_%2F%C3%A7%C2%B3%C3%99%0C%C3%8B%C3%A5%12RJ%C3%84q%7C%C3%97-c%0Ca%18%22I%12%1C%0E%C2%87a%C3%B0%C2%AE%C2%90H%C2%92%04EQ%C3%B49%2B%00%C3%804M%C2%ACV%2B%C3%A4y%C2%8E4M%3Bkz%C2%8F%7D%08%C3%B8%C2%B3%3C%C3%8F%C3%BB%C3%B63h%C3%89v%C3%86%18%C2%82%20%40%14E%C3%A3%C3%A0%C2%8Fd%7BY%C2%96%C3%98l6_%C3%AA~%25%C3%A1~%C3%92n%C2%B7%C3%83v%C2%BB%C2%85%C2%94%C2%B2%1F%7CL%C2%B6%C3%BF%2B)%25%C3%924%C3%BD%C3%B1%C2%AE%3C%C3%85y%3B%C3%A6%7Bq%C3%BB%14x%1C%C3%87%C2%BD%C3%AA%5E%C3%BF%C3%B7jY%C2%96Vh%C3%8B3%C2%94R%17%C3%97u%C2%B5%1C%C3%800%C2%8C%C3%B6%C2%AD%06%C2%A5%C3%94e%C3%924%C3%8D%1B%C2%A5t%3F%C2%9DN%C2%B5%C2%BFR%C3%9F%01u%C3%9B%C2%B5%15y%22%C2%9F%25%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%82IDATH%C2%89%C3%AD%C2%97-%C2%B3%C2%A30%14%C2%86%C3%9F%C3%8Bt%26%C3%93I%05%03%C2%AE%C2%88UE_W%C2%8B%C2%A8%C3%81VT%C3%B1%1B%C2%B0%C2%B5%C2%B5%C2%B5%C2%A9%C3%846%16%C2%8B%C2%8D%C3%85u%C2%90%C2%ADN%1DL%05%C2%94%C2%89%C3%A9%C2%AE%C3%98eg%C3%AE%5E%C3%8A%C3%9C%02%C2%9B5%C3%BB%C3%8A%C3%B31Or%C3%A6%C3%A4%24y%5B%C2%ADV%C3%9F(%C2%A5\'B%C2%88%09MRJ%C3%9D%C2%AA%C2%AAz%C2%9FPJO%00%C3%8C%C3%AB%C3%B5%C2%8A%C2%B2%2C%C3%BF%3Ax6%C2%9B%C3%81%C2%B6m%C2%93Rz2%08!f%C2%9E%C3%A7Z%C3%80%00P%C2%96%25%C3%B2%3C%07!%C3%844%1A%C2%83N%C3%9D%C3%AFw%00%C2%80%C2%A1%C2%95%C3%BAK%C2%8F%C3%87%03%000%C3%A9%0Ab%C2%8C%7D%C2%B2%C2%85a8Z%C3%BC%3F%C3%99%C3%B9%7F%C3%B8h%C3%B0%C3%85b%C3%B1rNg%C3%835%C3%8D%C3%82%18%C3%ABl%1C%00X%C2%AF%C3%97%C3%A0%C2%9C%23M%C3%93%2F%C3%85%03%23%C3%AD%C3%9C%C3%B7%7D%C3%94u%C2%8D4M_%C3%8A%1B%0Cw%1C%07%C2%9E%C3%A7%C3%A1x%3C%C2%BE%C2%9C%3B%18%1E%04%01%C2%92%24AQ%14z%C3%A1M%C2%B9%C2%85%10%C2%BD%C3%B2%7B%C3%83-%C3%8B%C2%82%C3%AF%C3%BB%C2%BD%C3%8A%3D%18%3E%C2%A4%C3%9C%C2%83%C3%A0%C2%9E%C3%A7a%3A%C2%9D%22I%C2%92%C3%9E%C3%A0%5E%C3%B01%C3%8A%C3%9D%1B%1E%04%01%C2%84%10%C2%90R%C3%AA%C2%85%2F%C2%97%C3%8BQ%C3%8A%C3%9D%C3%A8m%C2%B3%C3%99%7C%3F%C2%9F%C3%8F%C2%AD%C3%8E%C2%B6%C3%BB%C3%B9p8%C3%A0r%C2%B9%7C%C2%B0m%C2%B7%5B8%C2%8E%C3%B3%14%22%C2%A5%C3%84~%C2%BF%C3%BF%60s%5D%C3%B7%C3%B5%C2%B2%C3%BF%09%06%C2%80(%C2%8AP%C3%97uk%7C%5D%C3%97%C2%88%C2%A2%C2%A8%C3%957%C3%8Al%2F%C2%8A%02%C2%9C%C3%B3V%1F%C3%A7%C3%BC%C3%A9q%1C%C3%ADJ%C3%8D%C2%B2%C3%AC%C3%93%C2%A4%13B%20%C3%8B%C2%B2%C2%A79%C2%A3%3E%26%C3%A28%C3%BE%7D%0A%C2%A4%C2%94%C2%88%C3%A3%C2%B83%C2%BE%C2%B3%C3%A1%C3%BA%C3%88%C2%B2%2C%C2%84a%08%C3%86X%C3%A7%C3%B4s%5D%C2%B7%C3%BB1%C3%91GEQ%60%C2%B7%C3%9B%7D)%C3%96%00~~at%C2%AA%C3%A1%19J%C2%A9%C2%9Bm%C3%9BZ%16%60%18F%C3%B3W%C2%83R%C3%AA6%C2%A9%C2%AA%C3%AA%C2%9DRz%C2%9A%C3%8F%C3%A7%C3%9A%7F%C2%A9%3F%00%C3%B5-%C3%81%C3%BE%7Fo%C3%8E%C2%B9%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%010IDATH%C2%89%C3%AD%C2%971%C2%8A%C2%840%14%C2%86%C3%BF%0DB%C2%90X%C2%88vZ%C3%AC%0Dd%C2%8E%60%C3%A1Q%C2%AC%C3%93%C3%9A%C3%9A%C3%9Az%06%5B%3D%C2%83%C2%ADV%5EB%3B%C3%81%22*id%C2%9B%C2%B5%C3%99q%06%1D0%C2%B3%C2%BB%C3%B8%C2%95%C2%8F%1F%C2%BE%C2%97%C3%B7B%20%1FA%10%7C2%C3%86%1AJ%C2%A9%09EH)%C2%87q%1Co%1Ac%C2%AC%01%60v%5D%07!%C3%84%C3%A9b%C3%830%60%C3%9B%C2%B6%C3%89%18k%08%C2%A5%C3%94%C3%AC%C3%BB%5E%C2%89%18%00%C2%84%10%C3%A8%C3%BB%1E%C2%94R%C2%93%C2%AC%05%C2%95L%C3%93%04%00%20J%C2%AD%C3%9F%2C%C3%8B%C3%B2%3E%C3%B9%C3%8A%25%C2%BF%C3%A4%7F_%1EE%11%2C%C3%8Bz%C2%8F%C3%9Cu%5DDQ%04%C3%8F%C3%B3%C2%9E%C3%A6%C2%B4%C2%ADb%C2%9A%C2%A6w5%C3%8E%C3%B9%C2%A1%06t%5DG%18%C2%86(%C3%8B%12EQlfN%C3%9F%C2%B9%C3%AF%C3%BB%0F%C3%97%C2%B0y%C3%B2-%C2%B6%C2%A6%C2%B1%17%C3%97u%C3%819G%1C%C3%87%C2%AF%C3%89%C2%8F%C2%8C%C3%BDg%C2%A3m%C3%9B%22%C3%8B%C2%B2%C2%BB%C3%9Cn%C3%B9%C2%AB%C3%94u%C2%8D%3C%C3%8F1%C3%8F%C3%B3%3E%C3%B9%C3%91%C3%8B%C2%B5%C3%85%3C%C3%8F(%C2%8A%02UU%3D%C3%8C%C2%9Cr%C3%B2u%C3%8Cm%C3%9B%3E%C3%8D%C2%9D%22O%C2%92dW%C3%AE%C3%BF%3D%C2%AF%C2%97%C3%BC%C2%92%C3%BF%5E%C2%B9a%18J%C2%A5%C2%AB%C2%8FH)%07%C3%9B%C2%B6%C2%954%40%08Y%C3%BFj%C2%90R%0E%C3%9A8%C2%8E7%C3%86X%C3%A38%C2%8E%C3%B2_%C3%AA%17%C2%A6%C3%AAmli%2B0%C2%85%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%018IDATH%C2%89%C3%AD%C2%97!%C2%8E%C2%840%18%C2%85%C3%9F6%24%15E%C2%90%C3%A0%40%C3%AC%15%C3%A6%10%C2%9C%07%C2%8B%C3%85b%C2%B1%C3%95%C2%B5%3DC%C2%8F%40%C2%B8Cq4%08%0A%C2%A9!%2BfHf%C2%B2%23%C2%B6%C3%9D%0C%C2%93%C3%AC%C3%B2%C3%A4%C3%9F%C3%B0%3E~%C3%BE%C2%97%C2%96~%14E%C3%B1%C3%89%18%C3%AB(%C2%A5%09%0E%C2%92sn%C2%B2%C3%96%5E%22%C3%86X%07%20%19%C2%86%01%C3%B3%3C%C2%BF%1C%1C%C3%871%C3%924M%18c%1D%C2%A1%C2%94%26%C3%A38%1E%02%06%C2%80y%C2%9E1%C2%8E%23(%C2%A5%09%C3%99%0BGjY%16%00%009%C2%94z%C3%93%C2%B6m%00%C2%80(%C3%A4%C3%A1%C2%B6m%C2%BF%C3%95%C3%8A%C2%B2%C3%B4%C3%B6yK%C3%A7\'%C3%BC%C2%AD%C3%B0%C2%A0%C3%80%C3%AD%C3%A1j%C3%9B6(h%C2%BB%C3%BE%C3%AFg%3F%C3%A1\'%C3%BC%C2%84%C3%BF%7D%C2%B8%C3%97%C3%B6ZU%15%C3%B2%3C%7F%C2%A8%C3%9D%C2%9F%C3%ADZk4M%C3%B3c%3F%C2%AF%C3%8E9%C3%A7X%C3%97%C3%B5%C3%A9%C3%9A%C2%BA%C2%AE%C3%A0%C2%9C%C3%BB%C3%98%C3%B9%C3%81%C2%8D1%10B%3C%5D%13B%C3%80%18%C3%B3%3A8%00%C3%B4%7D%0F%C2%A5%C3%94CM)%C2%85%C2%BE%C3%AF%7D%C2%AD%C3%82%02\'%C2%A5%C2%84%C3%96%1A%C3%80u%C3%8ER%C3%8A%10%C2%9B%C3%B0%C2%B4s%C3%8Ea%C2%8C%C3%B1%C2%9E%C3%B3%C2%BD%C2%82~%26%C2%80%C3%AB%C3%BC%C3%AB%C2%BA%0E%06%03%C2%B7%C3%8E%C3%A38%C3%BE%C2%95%C2%89%C2%AFv%1Eq%C3%8EMi%C2%9A%1E%C3%B2%02%C2%84%C2%90%C3%BD%C2%AE%06%C3%A7%C3%9C%14Yk%2F%C2%8C%C2%B1.%C3%8B%C2%B2%C3%83o%C2%A9_%5D5x%C3%A2%C3%9E%3D%C3%80%3C%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%BAIDATH%C2%89%C3%8D%C2%97%C2%A1%C2%8F%C3%A2%40%14%C2%87%7F%C3%9B%23i6%C2%83%20S%07%C3%A2T1%24d%C3%9DX%04%06%5B%C2%81%C3%A9X%24%C2%B5%C2%B5X%C3%ACXl%C3%B9%1Bjk%10E%11%24E%20(%C2%AE%0D%09-%C2%A4%09aO%C3%9C5!G%0F%C3%9A%25%5B%C3%AE%C2%93%C3%93%C3%A9%C3%BB%C3%A6%C2%BD%C2%BCy%C3%89%C2%BCu%C2%BB%C3%9D%C2%9F%C2%84%C2%90%C2%85%2C%C3%8B5%C2%94D%C2%92%24%C3%BB8%C2%8E%3F*%C2%84%C2%90%05%C2%80%C3%9An%C2%B7C%14E%C3%9F.%C2%AEV%C2%ABP%14%C2%A5F%08YH%C2%B2%2C%C3%97%C2%82%20(E%0C%00Q%14!%08%02%C3%88%C2%B2%5C%C2%93%C3%92%C2%8529%1E%C2%8F%00%00%C2%A9T%C3%AB%1F.%C2%97%C3%8B%C3%AB%C3%A4)%2F%C2%95W%C2%B2%16%C2%85%107k%C2%86a%C3%A4%0E%C2%9A%C3%B7%C3%BF%C3%BF%2F%C3%B3%2C%C2%B2%C2%B2)M%C3%BEl%C3%99%C2%B3%C3%88%5Dv%C3%86Xny%5E%C3%9E%C3%BA%C3%BD%C3%BE%C3%A7j%C2%B5%C2%BA%C2%BBIUU%C3%A8%C2%BA%C2%8E0%0CaY%16%C3%820%7CZ%C3%9Cl6%C3%B1%C2%A3%C3%95j%C2%8D%C2%82%20%C2%B8%C2%BB1%0CC8%C2%8E%03J)8%C3%A78%C2%9F%C3%8F%C3%98l6O%C3%89%15E)%C3%96%C3%AD%C2%B6mC%08%01%C3%86%18%0C%C3%83%00%C2%A5%C3%B4%C2%A9%03%C3%A4%C3%8A%C3%BC%C2%9A%C3%83%C3%A1%C2%80%C3%99l%06J)%06%C2%83%01%00%60%C2%BD%5E%17%16%17%C3%8E%C3%BC%1A%C3%9B%C2%B61%1A%C2%8D%C2%A0%C2%AA*L%C3%93D%C2%A3%C3%91(%1C%C3%A3%C2%A1%C2%9CR%0A%C3%9343%C2%BF%C2%85a%08!%04%5C%C3%97%C2%85a%18%C3%A8%C3%B5z%C2%85%C3%A4w%C3%AFy%C2%BB%C3%9D%C2%86%C2%AE%C3%ABx%7F%7F%C2%BF%1B%C3%84q%1C%2C%C2%97Kp%C3%8E%C3%81%18%C3%83t%3A%C2%85%C3%A7y_%C2%97k%C2%9A%C2%86N%C2%A7%C3%B30%40JZ%05%C3%86%18%C2%86%C3%83%C3%A1%C3%8D%C3%B7%5C%C2%B3%3D-s%11%C3%B15%C2%AE%C3%AB%C3%A6%C3%9E%7B%C2%93%C3%B9%C2%BF%C2%AEP)%C2%B3%7D2%C2%99%C2%80s~%C3%93%C2%BD%C2%A5%C3%8Cv%C3%9F%C3%B7!%C2%84%C3%80%7C%3E%C3%8F-%C3%BB*%C2%99%0Dw%3A%C2%9D%60Y%16%3C%C3%8F%C2%83%C2%A6i%0F%C2%BB%C3%BDo%C3%B2V%C3%A9%C3%AEUs%5D%17%C3%9B%C3%AD%16%C2%9C%C3%B3B%C3%B2%C2%BC%3C%1C2%C2%BE%C3%AFc%3C%1E%C2%BFF%C3%BE%C2%9DH%C3%80%C3%AF\'L%C2%99%C2%A4%3E)I%C2%92%C2%BD%C2%A2(%C2%A5%1C%40%C2%92%C2%A4%C3%B4%C2%AD%C2%86%24I%C3%B6%C2%958%C2%8E%3F%08!%C2%8Bz%C2%BD%5E%C3%BA%2B%C3%B5%17a%1D%C3%85_%C2%87R%C3%93%C3%8A%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%87IDATH%C2%89%C3%AD%C2%97!%C2%B3%C3%A20%14%C2%85%C3%8F%C3%AB0%C2%93a%C2%82%C3%A8%C2%B4%0E%C3%84%C2%AA%C2%A2%C2%9F%C2%AB%C2%AD%C3%80%C3%B4%0F%C2%A0%C3%B8%0B%C3%84b%C2%B1%C3%98%C3%98%C3%9A%C3%96%C3%96%C3%86%C3%86%C3%A2%C2%98J%C3%90%C3%855%C2%83h%C3%A9%C3%84%C3%B0V%C2%B0%C2%887%C3%90.%C2%85n%C3%96%C2%BC%23%C3%AF%C3%9C%3B_r%C2%92%C3%9E%C3%A6~%C3%8Cf%C2%B3_%C2%94%C3%92%1D!%C3%84%C2%86!i%C2%ADOUU%7D%0E(%C2%A5%3B%00%C3%B6%C3%B1xDY%C2%96%C3%BF%1C%3C%1A%C2%8D%C3%A0%C2%BA%C2%AEM)%C3%9DY%C2%84%10%C2%BB(%0A%23%60%00(%C3%8B%12EQ%C2%80%10b%5B%C2%B7%C2%80I%C2%9D%C3%8Fg%00%C2%80e%C2%94%C3%BAG%C2%97%C3%8B%05%000hK%C3%A2%C2%9C%C3%9F%C3%85%18c%C2%BD%C3%A5%C3%BF%C2%97%C2%9D%C3%BF%C3%80%3B%C3%83%3D%C3%8F%C3%AB%0D%C3%BE1%C2%9F%C3%8F%C2%BF%C3%B6%C3%BB%7Dk%12%C3%A7%1C%C2%8C1%C3%B8%C2%BE%C2%8F0%0C%C2%B1%5E%C2%AF%C2%9F%C3%8Ao%C3%93t%3A%C3%AD%C2%B6%C3%B3%C3%ADv%0B%C2%A5%14%C3%820%C3%ACR%C3%96%C2%A8%C3%8E%C2%B6%C3%87q%C2%8C%20%080%C2%99L%C3%8C%C3%83%C2%95R%10B%60%C2%B1X%C2%98%C2%87%03%C2%80%C2%94%12u%5D%C2%BFm%C3%BF%C3%8B%C2%9FZ%1C%C3%87%08%C3%83%10%C2%8E%C3%A3%C2%98%C2%87%C3%B7a%C3%BF%5BMF%08%C2%81%C3%A1p%C2%88%20%08%C3%8C%C3%83%C2%81%C3%B7%C3%AC%7F%1B%C2%9E%C3%A79%C2%A4%C2%94%2F%C3%99%C3%9FKo%17B%C3%80q%1C%C3%B8%C2%BE%C3%9F%C2%A9%C3%AEa%7B%5D%C2%ADV%C2%ADM%24%C3%8Fsl6%C2%9Bo1%C3%8F%C3%B3%C2%B0%5C.%C3%AFr%C2%9B%C3%9Alc%7B%C2%8D%C2%A2%08u%5D%3F%2C%C2%AA%C3%AB%1AQ%14%C3%9D%C3%85%0F%C2%87C%C3%A3b%C2%9B%C3%B4%10%C2%AE%C2%94B%C2%92%24%0F%0B%C2%92%24%C2%81R%C2%AA3%C3%A8i8%00dY%06)%C3%A5%C2%B7%C2%98%C2%94%12Y%C2%96%C3%B5%02n%C2%85%03%40%C2%9A%C2%A6%C3%88%C3%B3%1C%C3%80%C3%B5%C2%9C%C3%934%C3%AD%0D%0C%3C%C3%B1%3Fw%1C%07%C2%8C1p%C3%8E%7B%C2%B3%1B%C2%B8%5E%C2%B8%C3%96%C3%97%2Bp%3D%C3%BF%C2%BF%3D%1E%5E%C2%95%05%5CG%18%C2%93%C2%BA%C3%B1%2C%C2%AD%C3%B5%C3%89u%5D%23%0B%C2%B0%2C%C3%AB6%C2%ABAk%7D%1ATU%C3%B5I)%C3%9D%C2%8D%C3%87c%C3%A3S%C3%AAo%C2%81%C2%92%C3%80r%C2%BB%14%C2%94%C2%99%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%3BIDATH%C2%89%C3%AD%C3%97%C2%B1%C2%8E%C3%82%20%1C%06%C3%B0%C3%AF%C2%88%09i%C3%A8%C3%90%C3%80%C2%A6%C3%83%C2%BD%C2%81%C3%B1%11%1C%C3%BA%04%C2%AC%C2%B2ve%C3%AD3%C2%B8v%C3%AD%C3%AA%C3%8B%C3%A8%C3%94%C3%9D%C3%95%C2%BA%C2%918%C2%94%1A%C2%96%C3%A6%C2%96%C3%AB%C3%94%C2%9E%C3%96%C3%A6%C2%8A%C2%97%5C%C2%BF%11%08%3F%20%7FH%C3%B8%C2%88%C3%A3%C3%B8%C2%931VPJ%23x%C2%8As%C3%AEf%C2%AD%C3%9D%2C%18c%05%C2%80%C3%A8z%C2%BD%C2%A2%C2%AA%C2%AA%C3%89%C3%A10%0C!%C2%84%C2%88%18c%05%C2%A1%C2%94F%C3%86%18%2F0%00TU%05c%0C(%C2%A5%11i%1B%7C%C2%A6%C2%AEk%00%00%C3%B1%C2%AA~%C2%A7i%C2%9A%C3%B7%C3%A1mf%7C%C3%86%C3%BF%16%C3%8E9G%C2%9A%C2%A6%C2%93%C3%A0%C2%8BG%C2%9D%C3%AB%C3%B5%1A%C2%BB%C3%9D%0EA%10%C3%B8%C3%85%C2%A5%C2%94%C3%98n%C2%B7%C2%A3%26%C3%8D%C2%B2%C2%AC%C3%93%C2%A6%C2%B5~%C2%8Es%C3%8E%C2%91%24%09V%C2%AB%C3%95(%C3%B8%C2%95tp%C2%AD58%C3%A7%C2%9D%C2%81%7D%C2%BB%C3%B9u%3C%C3%8Fs(%C2%A5%3A%3B%C3%AF%3B%C2%B6%C2%9F2t%C2%A1%C2%9Dj%2F%C3%8B%12Y%C2%96%C3%A1t%3A%0D%C3%86%C3%86%C2%A6%C2%B7%C3%A0%C3%AE%C3%B7%3B%0E%C2%87%03%C3%8E%C3%A73%C2%A4%C2%94%2FW%C3%BB%C3%90Szx%C3%95%C2%8E%C3%87%23.%C2%97%0B%C2%94R%2F%C3%A1C%C3%B3%C3%B4%C2%91)%C3%8B%12%C3%BB%C3%BD%C3%BE%3D%C3%B8%C2%94%C2%99%C3%B1%19%C3%BFgx%18%C2%86%5E%C3%91%C3%96%23%C3%8E%C2%B9%C2%9B%10%C3%82%C3%8B%02%08!%C3%AD_%0D%C3%8E%C2%B9%C3%9B%C3%82Z%C2%BBa%C2%8C%15%C3%8B%C3%A5%C3%92%C3%BB%2F%C3%B5%0B%1D%C2%90g%C2%84%1E%C2%AD%C2%8B%C2%A1%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%20IDATH%C2%89%C3%AD%C2%97%C2%B1%C2%8E%C2%84%20%10%C2%86%C3%BF%23%26%C3%84%C2%8C%C2%85%19%3B)%C3%AE%0D%C3%8C%3E%C2%82%C2%85O%40kM%C3%AB3%C3%98%C3%9A%C3%B2d%3E%C2%85%1D%C3%89%16jBc%C2%AE2%C3%99%C3%8Dm%C2%83%7Br%C3%85%C3%BAW0%03%C3%B3%C2%91%C2%99%09%C2%84%C2%AF%C2%A6i%C2%BE%C2%89h%C2%94R%C3%A6%C2%88%24%C3%AF%C3%BD%7DY%C2%96%5BBD%23%C2%80%7C%C2%9A%26%C3%8C%C3%B3%7C%3A8%C3%8B2%14E%C2%91%13%C3%91(%C2%A4%C2%94%C2%B9s.%0A%18%00%C3%A6y%C2%86s%0ER%C3%8A%5C%C3%AC%C2%86%C2%98Z%C3%97%15%00%20%C2%8E%06%60f%C3%B4%7D%0Ff%0E%C3%9E%C2%BBm%C3%9B%7Bpc%0C%C2%98%19%C3%86%C2%98%C2%A3!%C2%8E%C3%81%C2%B5%C3%96PJ%01%00%C2%94R%C3%90Z%C3%87%C2%81WU%C2%85%C2%BA%C2%AE%C2%9Flu%5D%C2%A3%C2%AA%C2%AAs%C3%A1%C3%8C%C2%8C%C2%B6m_%C3%BA%C3%9A%C2%B6%0D%C2%AE%7F%10%C3%9C%18%C2%834M_%C3%BA%C3%924%0D%C2%AE%7F%12%C2%B2x%18%C2%86%C2%A7%C2%B9%C2%B5%16%5D%C3%97%05%01%1Fu%C2%B8%C3%9B%C3%BFB%17%C3%BC%C2%82_%C3%B0%0B~%C2%AA%C2%82%C2%AE%C3%97%5D%C3%96%C3%9A_%C3%A3%23%C3%97%C3%AC%C3%A7%C2%A6%C3%BDs%C3%A1%C2%87%1A%C3%AE%C2%9D7%C3%BCQ%C3%BF%C2%9F%C3%B6%2C%C3%8B%C2%A2Bw%C2%9E%C3%B0%C3%9E%C3%9F%C2%8B%C2%A2%C2%88r%00!%C3%84%C3%BEW%C2%83%C3%B7%C3%BE%C2%9E%2C%C3%8Br%23%C2%A2%C2%B1%2C%C3%8B%C3%A8%C2%BF%C3%94%1F%02%13R%C3%8D%C3%80%C3%A4%C2%9A%C2%96%00%00%00%00IEND%C2%AEB%60%C2%82')
        }
    };

    var darkest = {
        normal: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01xIDATH%C2%89%C3%AD%C2%97-%C2%8F%C3%82%40%10%C2%86%1F%C2%9A%13%C3%A0%C2%AE%0E%24%7Fa%1D%C2%98K%C3%96%C3%96%10%C2%90H%24%C3%BC%04%24%12%24%C2%B2%3F%C2%A1%0Df%C2%91M0%C3%A0%C2%8AD%22%C3%81%C2%81k%C3%9D%C2%9D%C2%B8%C2%8Fpa%C2%A1%C3%9BR(%C3%89%C3%9D%2B\'3%C3%BB%C3%8C%3B%C3%9DL%C3%9B%C2%92%10%C3%A2%15%C2%98%01o%3CN%0B%C2%A0e%15%00%C3%A6%C2%8B7%C2%B3%0A%00%C3%BF4%60%15%04%06%C3%A0%1F%5E%C2%88%5Et%C3%81%C3%91ht%16%1B%0E%C2%87%C3%86%C2%87%C2%9A%C3%96%3F%C2%9Fs%C2%9Dtn%1E%06%C2%BFu%C3%AC%3A%C3%A5%3Ev!%C2%84qnI%08%C3%B1%C2%9E%07%C3%94%C2%B6m%C3%9A%C3%AD6%C2%95J%05%C2%A5%14%C3%9B%C3%AD6%C2%B1%C3%86x%C3%AC%C3%97%C3%94l6%C2%91R%C2%B2%5C.%09%C2%82%C3%80%C2%B8%C3%AE%26%C3%B8%C2%A9%5B%C3%97u%C3%99%C3%ADv%C2%A9%C3%AA3%C3%83%C2%A5%C2%94H)%09%C2%82%20%C2%95%C3%9B%C2%9B%C3%A0%C2%B5Z%C2%8DN%C2%A7C%14EL%26%13%0E%C2%87C%260%C2%A4%C2%BC%C3%ADRJz%C2%BD%1Ea%18%C3%A2%C2%BA%C3%AEE%C3%B0%600%C3%80%C2%B6%C3%AD%C3%84%C3%B3%C2%8C%C2%9C%C3%97%C3%ABu%1C%C3%87!%C2%8A%22%C2%A6%C3%93i%C2%A2%C3%9Bj%C2%B5J%C2%BF%C3%9F%C3%87%C3%B3%3C6%C2%9BM%3A%C2%B8nI%C3%B8%C2%BEO%18%C2%86%26%C2%BD%02P.%C2%97%C3%A9v%C2%BB%C2%ACV%2B%C2%94R%C3%9A%1C%C3%A3%C2%B1%C2%A7%01%C2%9F%C2%AA%C3%91h%5C%7C%0C%C3%9A%25s%C2%8F%3D~%3C%1E%19%C2%8F%C3%87%C2%BFb%0F%C3%99%C3%AD%C3%BB%C3%BD%1E%C3%8F%C3%B3%C3%8E%C3%B2r%C3%99p%C3%97%C2%B4%5E%C2%AFQJ%11%C3%87%C2%B1%19%3C%C2%8D%C3%8BK%C2%8A%C3%A3%C2%98%C3%B9%7C~%C3%B5%C2%AE%C3%9C%C3%85%C3%B9%C3%B7%C2%98%C2%93%C3%96mno%C2%B5%2C%C3%BA%C2%BB_%C2%AF%16%C2%9F%3FmEha%01%C2%AD%02%1AX%00%C2%AD%0F%C3%854%C2%88%3F0%C2%88%C3%8E%C3%A5%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01gIDATH%C2%89%C3%A5%C2%96%C2%A1%C2%8E%C2%83%40%10%C2%86%C2%BF%C2%92%C2%9A%C2%BA%C3%83Q%C3%99WX%C2%89%C2%B9%C2%84\'h%C3%A8%23%C3%94%C3%B2%08%C2%95%3CB-%12IS%C2%83%25%C3%81%2C%0E%C2%8B%C2%AC%04%C2%87%2C%C3%AEN%C3%9C%C3%B5%C2%92%5E%0B-%C3%8B%C2%A6%C2%9B%C3%9C%C2%8D%C2%9C%C3%BD\'%C3%9F%C3%AE%C3%8F2%C2%B33!%C3%84%1Bp%04%C3%9Ey%5D%C3%A4%C3%80%C3%9A2%00%C3%A6%C2%9Bw%C2%B4%0C%C2%80%7F6%60%19%02%03%60%14%3E%1FZ%0C%C3%83%C3%B0%26%C2%B7%C3%9B%C3%AD%C2%B4%C3%A9%C3%BF%C2%AF%C3%AD%7F%03%C2%BEZ%C2%ADF%C3%97%C3%8C%C2%84%10%1F%C2%8FDa%18%0E%5E%1C%C2%80%20%08%C2%90RR%C2%96%C3%A5Sz%C3%90tr%C3%8F%C3%B38%C2%9F%C3%8F%C2%94e9%C2%AAn2%7C%C2%B9%5C%C3%A2%C2%BA.%C2%87%C3%83at%C3%ADd%C3%B8f%C2%B3!%C3%8B2%C3%9A%C2%B6%7D-%C3%BCb%C2%B7%C2%94R%C2%A9%5E%19n%C3%9B6%C2%9E%C3%A7)%C3%99%3D%19%C3%AE%C3%BB%C2%BE%C2%B2%C3%9D%C2%93%C3%A0%C2%AE%C3%AB%C2%B2X%2C%C3%88%C2%B2L%19%C2%AC%04%C2%BF%C3%98%C2%9D%24%C3%89%24%C2%B0%12%C3%9C%C3%B7%7D%C2%A4%C2%94%C3%94u%C3%BDZ%C2%B8%10B%C2%8B%C3%9D%C2%97%18l%C2%AF%C3%B7%C3%A6s%14E%C2%9CN%C2%A7%C2%AB%5C%10%048%C2%8E%C3%93%0Bi%C2%9A%C2%86%C3%BD~%7F%C2%93%1Fm%C3%BBo0%40%1C%C3%87t%5DwW%C3%9Fu%1Dq%1C%C3%9F%5D%C3%93%C3%92%C3%9B%C3%9B%C2%B6%C3%AD%C2%BD%C2%80I%C2%92%C3%B4%C3%BE%C2%8E%C3%9AFjUU%14Eq%C2%95%2B%C2%8A%C2%82%C2%AA%C2%AAzk%C2%B4%3E%26%C3%924%C2%A5i%1A%C3%A0%C3%AB%3B%C2%A7i%3A%C2%A8%7Fj%C2%9E%C2%8F%09%C3%9B%C2%B6%C3%99n%C2%B7DQ%C3%B4%C2%B0%C3%BBi%C2%87%C2%8F%09%C3%A3o%C2%B8%C3%9C%10%3B%C2%B7%C2%80%C2%B5%C2%81%0D%C3%A4%C3%80%C3%BA%13%C3%B7%5E%C2%89%C3%AE%C3%9F%C3%BEf%C2%94%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%A4IDATH%C2%89%C3%AD%C2%941%0E%C2%82%40%14D%1F%C3%84%C2%82R%3A%C2%AE%40%C2%BB%1D%C2%95W%C2%A0%C3%A7%08p%04JJ%C2%AEBb%C2%B3%C3%B4Ttp%0C()%C2%B7%C3%93FM%C2%84%C3%95%C2%88%067%C2%9A%C2%9Dr2%C3%89%C2%9B%C3%BD%C3%BFg%1D!%C3%84%1E8%02%07%C2%BE%C2%A7%06%C2%88%5D%03%60.%C2%BC%C2%A3k%00%7C%2B%C3%A0%1A%02%03%60%C3%A1%16n%C3%A1%C2%BF%0B%C3%8F%C2%B2%0C%C3%9F%C3%B7%C3%8D%C3%80%C2%83%20%20MS%C3%820%7C%C2%9A%C3%9B%C3%A9%C3%8C%C2%A2(%16%5E%C2%9E%C3%A7%C2%AB%0Ax%C2%9EG%C2%92%24%C2%B4m%C2%8B%C2%94R%C2%9B%C3%99%7C%C3%A7Q%14%3D%5C%C2%83%23%C2%848%C3%8DM%C3%9D%C3%8B%3F%C3%954M%C2%94ey%C3%A7i%C3%87%C2%AE%C3%93%C2%9A%C2%B1%C3%8F%C3%8B%C2%8F%C3%A3HUU%C2%8B%C3%9C%C3%8B%C3%B0w%C3%95%C3%B7%3DRJ%C2%94R%C2%AF%C3%81%C3%97%1E%C2%97NJ)%C3%AA%C2%BA%C2%A6%C3%AB%C2%BA%C2%87%C2%99M%5E~%1D%C3%B30%0COs%C3%9A%C2%83%C3%BB%C2%96%C3%BE%C3%AF%7B%C2%B5p%0B%C2%B7p%1D%C2%BC1%C3%84n%5C%206P%C2%A0%01%C3%A23_%C3%84%3F%C3%9F%C2%AFH%02%C3%B6%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%A5IDATH%C2%89%C3%AD%C2%96%C2%B1%C2%8D%C3%830%0CE_%C2%8C%0C%C2%90%C3%92sh%C2%81%5B%C3%81%C2%83x%04%C2%97%1A%C3%81%2B%C2%B0%14%C3%A0F%03%C2%B8%C3%92%08ZA%C2%9D%5C%C2%AAK%C2%8A%C3%9C%1D.%C2%88%03%C2%9Cd%C3%A4%04%C2%9C%C3%8D%C2%92%04%C3%BF%C2%A3H%C2%82%C3%90I)u%01%26%C3%A0%C2%83%C2%BF%C2%B3%19%C3%A8%C2%9A%0A%60%3EySS%01%C3%BC%5D%40S%09%0C%40U%C3%B8%C2%B9%24Ik%C3%BD%C3%A4%1B%C2%86!%5Bg%C2%BFm%C3%9F%2F%C3%BC%C2%A4%C2%94%C2%BA%C2%96%26k%C2%AD%C2%8B%16%C3%AD%C3%8B%C3%B6%C3%9B%C3%B6%03~%C3%80%0F%C3%B8%C3%BF%C2%87g%C2%9D%C3%97%C2%BE%C3%AFi%C3%9B%C3%B6e%3C%C2%84%C3%808%C2%8E%C2%BF%C2%86g%C2%BD%5CDH)%C2%AD%C3%86RJ%C2%88H%C2%8E%5C%1E%3C%C3%86%C2%881f5f%C2%8C!%C3%86%C3%B8%3E8%C2%80%C3%B7%1E%C3%A7%C3%9C%C2%83%C3%8F9%C2%87%C3%B7%3EW%C2%AAl%C3%A1%C2%AC%C2%B5%C2%84%10%C2%80%C3%BB%C2%9C%C2%AD%C2%B5%252%C3%A5%C3%9B.%22%2C%C3%8B%C2%92%3D%C3%A7%C2%9F%C2%B6%C3%A93%C2%B1%C3%95%C2%AA%1F%C2%99%C2%B9%12%7Bn%C2%80%C2%AEB%013%C3%90%C3%9D%00%C3%AF%C2%A9G%082%C2%82%05%15%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C2%9CIDATH%C2%89%C3%AD%C2%97!%C2%8F%C3%A2%40%14%C3%87%7F%C3%9B%C2%AB%00%C3%878*Gb%C3%87%C2%A1%C3%8E%C2%83!%3C%C2%89%2C%0E%24%12%C2%B1%C2%82%C2%AF%C2%80%03%C2%89%C2%84%C3%ACW%C3%80TT%C2%B4%1F%01%C2%81%18Y%04%09%08%C2%92%3Bq%C2%97%0BIY%C2%98n7%C3%B4%C2%92%C3%9D%C2%BF%C2%9Cy%C3%B3~%C3%B3%7F%C3%8D%C2%BC%C2%99%C2%BE%18c%1A%C3%80%1B%C3%B0%C2%93%C3%A7i%0B%C3%B4%C2%BC%0A%C3%80%C3%BC%C3%A5%C2%BDy%15%C2%80%C3%BFm%C3%80%C2%AB%08%0C%C3%807%C2%BC%12%C3%B9%C2%B7%06g%C2%B3Ynl%3A%C2%9D%3A\'u%5D%C3%BF%C3%BF9%C2%BF%C2%A5%5Bn%C2%9E%06%2F%5B%C3%B6%5Br.%C2%BB1%C3%86%19%C3%AE%C2%AA%17c%C3%8C%C2%AFGAZkD%C2%84%2C%C3%8B%C3%98l6dY%C3%B6)%C3%B0%1FA%10%C2%BC%3E%0A%3A%1C%0EDQ%C2%84R%0A%11%C3%A1r%C2%B9%C2%B0%C3%9F%C3%AFK%C3%83%C2%9D%C2%9C_%2B%08%02D%C2%84%C3%93%C3%A9T%C2%BA%0AN%C3%8E%C2%AFu%3C%1E%C2%89%C3%A3%18%C2%A5%14%C2%83%C3%81%00%C2%80%C3%9Dn%C3%B7!xa%C3%A7%C3%97RJ%C3%91%C3%AF%C3%B7%C2%A9%C3%97%C3%AB%C2%AC%C3%97k%C2%AC%C2%B5%C2%85%C3%96%3Ft%C2%AE%C2%94b8%1C%12%C3%87qn%C3%AE%7C%3E%C2%93%C2%A6)%C2%BE%C3%AF%23%22%C3%B8%C2%BE_%C2%A8%0Aw%C2%8FZ%C2%AB%C3%95b4%1A%C3%91l6%C3%AF%26%C2%89%C2%A2%C2%88%C3%B9%7C%C2%8E%C3%96%C2%9A%C3%89d%C2%82%C3%96%C3%9A%09%C3%BEn%C2%93%C3%A9v%C2%BB%C2%B4%C3%9Bm%C2%A7%24%00Y%C2%96%C2%B1%5C.1%C3%86%10%C2%86an%C3%9E%C2%A9%C2%B7%2B%C2%A5%18%C2%8F%C3%87%C2%85%C3%80%C3%97J%C2%92%C3%8496%C3%A7%3C%0CC%1A%C2%8DF.%C3%B0)%C2%BD%7D%C2%B5Z!%22%C2%B9%C3%AF%C3%BC%C2%94%C3%9En%C2%ADe%C2%B1X%C2%90%C2%A6%C2%A93%C3%AC%C2%A3%C2%BA%7B%C3%8E%C2%8D1t%3A%1Dj%C2%B5Z!%C3%A7%C2%AE%C2%BA%7B%C2%A5%26I%C2%82%C2%B5%16%11%C3%B9t0%C2%94%C3%ACpe%C3%B5u_%C2%AF%1E%7F~%C3%9A%C2%AA%C3%90%C3%96%03z%15l%60%0B%C3%B4~%03%C3%81%C3%B9%C2%87k%C3%B3%17%C3%ADU%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01jIDATH%C2%89%C3%A5%C3%96%C2%AD%C2%8E%C2%83%40%10%C3%80%C3%B1%7F%C3%89%C2%99%C2%BA%5B%C3%97J%5Ea%25%C3%A6%C2%92%7D%C2%82%C2%A6%7D%04l%1F%01%C2%89%C2%AC%C2%AC%C2%ADD.%C2%A9%C3%81n%C2%82%01W%C2%8B%C2%AC%2Cneqw%C3%A2%3E%C2%92%C2%A6_%C3%90rl.7r%C2%98%C3%89ow%08%C2%BB%C2%8C%C2%A4%C2%94%C2%AF%C3%80%16xc%C2%B8%C3%88%C2%81%C2%99%C3%A7%00%C3%A6%C3%8B%C3%9Bz%0E%C3%A0%C2%9F%05x%C2%8E%60%00%C2%9C%C3%A2%2F%C2%B7%1E%C3%86q%7C%C2%96%C2%8B%C2%A2%C2%A8%C2%B7%C3%BA%C3%BF%3B%C3%B6%C2%BF%C2%85%C3%BB%C2%BE%C3%9F%1B%3E%C2%92R%C2%BE%C3%9F%2B%C2%8A%C3%A3%C2%98(%C2%8A%C2%90R%C2%A2%C2%94b%C2%B5Z%C2%B5%C2%AA%C2%BF%17%C2%9Dv%C2%BE%C3%9B%C3%AD%C2%B0%C3%96%C2%A2%C2%94%C3%AA%C3%92%C3%96%0F%0E%C2%90%C2%A6)A%100%C2%9DN%C2%87%C3%87%C2%AD%C2%B5%18cX%2C%16%C3%83%C3%A3%00EQp%3C%1E%C2%9F%1E%C3%BF%C3%83%C2%9FZ%C2%9A%C2%A6(%C2%A5%10B%0C%C2%8F%7F%C2%8F%7F%3E%C2%9F%0F%C2%8F%03%18c%18%C2%8F%C3%87%04A0%3C%0E%C2%A0%C2%B5~x%C3%BCO%C3%A3%C2%87%C3%83%C2%81%C2%A2(%1E%1A%7F%2Fg%C2%BB1%06!%04R%C3%8AN%7D%17%C2%8F%C3%97%C3%A5r%C3%89d2%C2%B9%C3%9AT%C3%975%C3%AB%C3%B5%C3%BA%24%C3%A7%C3%BB%3Ea%18%C2%9E%C3%95v%C2%BE%C3%8F%C2%93%24%C2%A1i%C2%9A%C2%8B%0DM%C3%93%C2%90%24%C3%89Y~%C2%BF%C3%9F_E%3A%C3%A1%C3%96Z%C2%B4%C3%96%17%1B%C2%B4%C3%96Xk%3BC%C2%ADq%C2%80%C2%AA%C2%AA(%C3%8B%C3%B2%24W%C2%96%25UU%C3%B5%02%C3%9F%C3%84%01%C2%B2%2C%C2%A3%C2%AEk%C3%A0%C3%B3%3DgY%C3%96%1B%0C-%C3%AEs!%04a%18%C2%B2%C3%99lz%1Bwk%C3%BC7%C3%83%C3%B9%3F%5C%C3%AE%C3%88%C3%8E%3D%60%C3%A6%60%0190%C3%BB%00%C2%AB%C3%98%C2%83%C3%AE%C3%9D%C3%912%C3%83%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%A2IDATH%C2%89%C3%AD%C3%94-%C2%B2%C2%830%14%C2%86%C3%A1%C2%B7%C3%8C%15%C2%91%C3%84%C2%B1%05%C3%ACq%C2%A8z0%C3%8C%1C%C2%89%C3%84%C2%B2%04%24%C3%8B%C3%80%C3%A2%C2%BB%056%00%C3%8B%C3%882z%C3%8D%C2%9Dk%C3%92%1F%C3%8A%C3%90%C2%A6%22%C2%9FL2y%C3%8E%C3%89%249%C2%89H%0A%5C%C2%803%C2%9F%C3%8B%0C%C3%94I%00%C2%98%3F%C3%AF%C2%92%04%C2%80%C3%BF%0BH%02%C3%81%00D%3C%C3%A2%11%C3%BF%0E%C3%9CZK%C3%97uo%C3%81%7F%1EM%C3%A6y%C2%8E%C2%AAb%C2%8C%C3%B9%2C%5EU%15EQ%C3%AC%C3%9At%18%06o%C2%AC%C3%AF%C3%BB%C3%A7%C2%B8%C2%B5%C2%96%C2%A6i%C3%88%C2%B2l%17%C3%BCJ%3C%C2%BCm%5B%C3%924%C3%B5%16%C3%9E%C3%AA%C3%A6p%7C%C2%9A%26T%C3%95%C3%AB%C3%BC%C3%96%C2%B1%C3%9D%C3%8B%C3%96B%C2%BD%C3%9B%C3%AE%C2%9Cc%1CG%C3%96u%C3%9D%C2%8C%C3%AD%C3%8DID%C2%AE%C3%B7%26E%C2%84%C2%B2%2C1%C3%86%C2%BC%C3%94%C3%B9%C3%96%3C%7Cj%C3%8B%C2%B2%C3%A0%C2%9CCU%0F%C2%87%C3%A1I%C3%A7%C3%AF%C3%8Ew%7F%C2%AF%11%C2%8Fx%C3%84%C2%8F%C3%82%C3%A7%40%C3%B6%C2%9C%00u%C2%80%02f%C2%A0%C3%BE%05c%06.VE%20%C2%8D%1B%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%00%C3%8BIDATH%C2%89%C3%AD%C3%97!%0E%C3%83%20%14%C3%86%C3%B1%7FQ%C3%88%3E%C3%87%15%C2%B08%C3%94%0E%C2%80i%C2%82%7C%C3%97%C2%A8%C2%AC%C3%A4%40%C2%BBBO%C2%B6%C2%89-%C3%8B%C2%96%C3%95%C2%94nE%C2%B4%C2%9F%C2%84%C2%90%C3%9F%0B%C2%90G%C3%A8B%08%3Dp%05.%C3%AC%C2%97%19%18L%03%C2%98%C2%A7w5%0D%C3%A0W%01%C2%A6%11%0C%405.%22%C2%8C%C3%A3%C2%88%C2%88%C3%AC%C2%8F%C2%AB*%7D%C3%9F%C2%A3%C2%AA%C3%BB%C3%A2)%25%C2%9Cs%008%C3%A7H)%C3%AD%C2%83%7B%C3%AF%C2%891~%C2%8C%C3%85%18%C3%B1%C3%9E%C3%BF%17%17%11r%C3%8E%C2%8Bs9%C3%A7%C3%95%C3%A7%C2%BF%0AWU%C2%AC%C2%B5%C2%8Bs%C3%96%C3%9A%C3%95%C3%A7%C3%9F%C2%85%10n%C2%ABV%C2%BC%C2%A5%C2%94%C3%824M%C2%B5%C3%8B%C3%ABo%C3%BB%2Fr%C3%A2\'~%C3%A2\'%C3%BE%C3%97T%C2%B5%C3%97R%C3%8A%C3%97XM%C2%9B%3D%C3%AE%C2%B6%1F%17%C3%9F%C3%B4%C2%9Eo%C3%8Dq%C2%B7%C3%9D%C3%B0%C3%B8%C2%B4%C2%B5%C3%88l%C2%80%C2%A1A%0130%C3%9C%01Q%C3%8A%20%C3%BC%C3%B5g%C2%92%11%00%00%00%00IEND%C2%AEB%60%C2%82')
        },
        rollover: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%89IDATH%C2%89%C3%8D%C2%97%C2%A1%C3%8E%C3%B20%14%C2%86%C3%9F%C2%91%C2%A6%C2%90%096A%C2%86%C3%A3%C2%BF%02%C2%92O%C2%80%C2%ABD%C2%90%C3%8CMN%C3%AE*%C2%B8%04%C3%AE%60nrwP7%C3%9C%1C%08%C2%929%24%0E%C2%82%00%C3%842J%13~%C3%B3%2F%C3%B9%C2%92o%40%C3%87%3E%C3%8A%C3%BF%C3%8A%C2%93s%C3%BA%C3%B4%3DmNZc%3A%C2%9D%C3%BE%C2%A1%C2%94%C2%AE%09!64IJy%14B%7C%11J%C3%A9%1A%C2%80%7D%3E%C2%9F!%C2%84x%3B%C2%98R%0A%C3%934mJ%C3%A9%C2%BAE%08%C2%B1%C3%B3%3C%C3%97%02%06%00!%04%C3%B2%3C%07!%C3%84n%C2%95%01%C2%9D%C2%BA%5E%C2%AF%00%C2%80%C2%96V%C3%AA%3F%C3%9Dn%C2%B7%C3%8F%C3%81K%7D%14N%C2%AA%C2%82%C2%B3%C3%99%C3%ACGl%3E%C2%9F%2B%2F%C2%AAZ%C3%BF%C3%BF9%C2%AFR%C2%95%1Bm%C3%B0%C2%A6m%C2%AF%C3%92%C2%AF%C2%B7%7D8%1C*%C3%A7%1A%C2%9E%C3%A7%C3%9D%0E%C2%87Cc%C2%A8eYp%5D%17%C2%9DN%07I%C2%92%60%C2%BB%C3%9D%3E%C3%8C%C3%AF%C3%B5z%C3%AAm%7F%C2%A4%C3%B1x%0C%C3%86%18%C2%96%C3%8B%25%C3%924U%C2%AEk%04%C3%BF%C3%AE6%C2%8Ec%C3%ACv%C2%BBZ%C3%B5%2F%C3%83%19c%60%C2%8C!M%C3%93Zn%1B%C3%81%C3%BB%C3%BD%3E%5C%C3%97EQ%14%08%C3%83%10%C2%A7%C3%93%C3%A9%250P%C3%B3%C2%B63%C3%86%C3%A0%C3%BB%3E%C2%B2%2CC%1C%C3%87w%C3%81A%10%C3%80%C2%B2%C2%AC%C2%A7%C3%AB)9%1F%0C%06%C2%98L%26(%C2%8A%02Q%14%3Du%C3%AB8%0E%C2%82%20%00%C3%A7%1C%C2%9B%C3%8D%C2%A6%1E%C2%BCjHp%C3%8E%C2%91e%C2%99%C3%8A%5E%01%00%C3%ADv%1B%C2%9E%C3%A7a%C2%B5Z!I%C2%92%C3%8A%1C%C3%A5%C2%B6%C3%97%01%7F%C3%97h4%C2%BA%7B%0CZf%C2%BB%C3%A38%C3%B0%7D%1Fa%18%C2%BE%06o2%C3%9B%C3%B7%C3%BB%3D8%C3%A7%3F%C3%B2~e%C3%82%3DR%C2%96eX%2C%16(%C2%8AB%0D%5E%C3%87%C3%A5%3D%5D.%17%24I%C3%B2%C3%B0%C2%AE%C2%BC%C3%85y%C3%99%C3%A6g%C3%A3%C3%B6-%C3%B0(%C2%8A%C2%94%C3%B2%3E%C3%BFz%C2%A5%C2%94j%C2%85%C2%96%C2%BC%C2%96%C2%94%C3%B2h%C2%9A%C2%A6%C2%96%0D%18%C2%86Q%C3%BE%C3%95%20%C2%A5%3C%12!%C3%84%17%C2%A5t%C3%9D%C3%ADv%C2%B5%C3%BFR%C3%BF%02u%C2%BA%C2%B9%C2%8Ct%C2%AC%C2%A1%05%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%84IDATH%C2%89%C3%AD%C2%97%C2%AF%C2%8E%C2%B3%40%14%C3%85%0Fd2i%10%05AZ%C3%97%7D%C2%82%26kp%C3%88%C2%8A%26%C2%B8%C3%8A%C3%8A%3E%05%C2%8F%C3%907%C2%A8%C2%AB%C3%84!%C3%87!q%20%C2%9A%C3%A0%C2%90u%25%15%C2%9D%0AR%C2%A6%C2%93t%C3%85.%C2%A2%C2%BB-%C3%99%02%C3%9F%7Cf%C2%8F%C2%BCso~%C3%8C%C2%99%C3%8B%C3%BC%C3%91%C3%A6%C3%B3%C3%B9%1B%C2%A5tG%08%C2%B1%C2%A0HR%C3%8A%C2%93%10%C3%A2%C2%9DPJw%00%C2%AC%C3%B3%C3%B9%0C!%C3%84%3F%07SJa%18%C2%86E)%C3%9D%C3%A9%C2%84%10%C2%AB%2CK%25%60%00%10B%C2%A0%2CK%10B%2C%C2%BD%0E%C2%A8%C3%94%C3%B5z%05%00%C3%A8J%C2%A9_%C2%BA%C3%9Dn%00%00%C3%92%C2%94%C3%A4%C3%BB%C3%BE%C2%8F%C3%98z%C2%BD%C3%AE-%C3%BF%C2%BF%C3%8C%C3%BC%0F%C3%9E%1B%7C2%C2%99%C2%BC%5C%C3%93%C3%98pu%C2%B3%C3%B8%C2%BE%C3%9F%C3%988%000%C2%9B%C3%8D%C3%80%18C%C2%96e%C2%BF%C3%8A%07z%C2%9A%C2%B9%C3%AB%C2%BA%C2%B8%5C.%C3%88%C2%B2%C3%AC%C2%A5%C2%BA%C3%8E%C3%B0%C3%B1x%0C%C3%87q%C3%80%18%7B%C2%B9%C2%B63%C3%9C%C3%B3%3C%C3%84q%0C%C3%8E%C2%B9Zxmw%C2%92%24%C2%AD%C3%AA%5B%C3%83M%C3%93%C2%84%C3%AB%C2%BA%C2%AD%C3%AC%C3%AE%0C%C3%AFbw\'%C2%B8%C3%A38%18%0C%06%C2%88%C3%A3%C2%B85%C2%B8%15%C2%BC%0F%C2%BB%5B%C3%83%3D%C3%8FC%C2%92%248%1C%0Ej%C3%A1%C3%93%C3%A9%C2%B4%17%C2%BBki%C2%8B%C3%85%C3%A2v%3C%1E%1F%0E%3E%3A%C2%9F%C2%83%20%C3%80~%C2%BF%C2%BF%C2%8B%C2%ADV%2B%C2%8CF%C2%A3%C2%A7%C2%90%C2%A2(%C2%B0%C3%9Dn%C3%AFb%C2%B6m%C2%BFn%C3%BBw0%00%C2%84a%C2%88%C2%AA%C2%AA%1E%C3%A6WU%C2%850%0C%1F%C2%8E%C3%B5%C2%B2%C2%B7s%C3%8E%C2%9F6%20c%C3%AC%C3%A9%C3%AF%C3%98%C3%9B%C2%91%C2%9A%C3%A79%C3%924%C2%BD%C2%8B%C2%A5i%C2%8A%3C%C3%8F%C2%9F%C3%96%C3%B4z%C2%99%C2%88%C2%A2%08EQ%00%C3%B8%5C%C3%A7(%C2%8A%1A%C3%B3%1B%1B%C2%AE%C2%8DL%C3%93%C3%84r%C2%B9D%10%04%C2%8D%C2%BB%C2%9Fm%C3%9B%C3%8D%C2%97%C2%896%C3%A2%C2%9Cc%C2%B3%C3%99%C3%BC*W%07%3E%C2%9F0*U%C3%B3t)%C3%A5%C3%890%0C%25%1F%C2%A0iZ%C3%BDV%C2%83%C2%94%C3%B2D%C2%84%10%C3%AF%C2%94%C3%92%C3%9Dp8T%C3%BEJ%C3%BD%00%0D%C2%A2%C3%84b%C3%B1M%07N%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01-IDATH%C2%89%C3%AD%C2%971%C2%8A%C2%84%40%10E%C2%BF%C3%92%C2%94%60%C2%A0F%C2%A6%C2%9B%19O%C3%A0!%C2%8C%0D%0D%3D%C2%85G%C3%B0%06f%C3%9E%C3%82Lo%60%20x%0A%C2%8D%C3%84%40%C2%B4mp%C2%925%C3%99q%06%1D%C2%B0gw%C3%B1%C2%85%C3%85%C2%87W%5D%C3%954%C2%B4%C3%A2y%C3%9E%17%11U%C2%8C1%0B%C2%92%10Bt%C2%9C%C3%B3%1B%23%C2%A2%0A%C2%80%C3%95%C3%B7%3D8%C3%A7%C2%A7%C2%8B%C2%89%08%C2%BA%C2%AE%5BDT%C2%A9%C2%8C1k%18%06)b%00%C3%A0%C2%9Cc%18%060%C3%86%2Cu-%C3%88d%C2%9Eg%00%C2%80*%C3%95%C3%BA%C3%8D%C2%B2%2C%C2%9F%C2%93%C2%AF%5C%C3%B2K%C3%BE%C3%B7%C3%A5a%18%C3%824%C3%8D%C3%8F%C3%88m%C3%9BF%18%C2%86p%1C%C3%A7e%C2%8Em%15%C2%A3(z%C2%A8%C3%85q%7C%C2%A8%01M%C3%93%C3%A0%C3%BB%3E%C3%8A%C2%B2D%C2%9E%C3%A7%C2%9B%C2%99%C3%93w%C3%AE%C2%BA%C3%AE%C3%935l%C2%9E%7C%C2%8B%C2%ADi%C3%AC%C3%85%C2%B6m%04A%C2%80%24I%C3%9E%C2%93%1F%19%C3%BB%C3%8FF%C3%9B%C2%B6E%C2%96e%0F%C2%B9%C3%9D%C3%B2w%C2%A9%C3%AB%1AEQ%60%1C%C3%87%7D%C3%B2%C2%A3%C2%97k%C2%8Bi%C2%9A%C2%90%C3%A79%C3%AA%C2%BA~%C2%9A9%C3%A5%C3%A4%C3%AB%C2%98%C2%9B%C2%A6y%C2%99%3BE%C2%9E%C2%A6%C3%A9%C2%AE%C3%9C%C3%BF%7B%5E%2F%C3%B9%25%C3%BF%C2%BDr%22%C2%92*%5D%7D%C2%AA%10%C2%A2%C3%93u%5DJ%03%C2%8A%C2%A2%C2%AC%7F5%08!%3A%C3%869%C2%BF%11Qe%18%C2%86%C3%B4_%C3%AA%1D%C2%85%C3%82m%C2%B1%C3%8C%3F%C3%BBB%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%014IDATH%C2%89%C3%AD%C2%97%C2%B1%C2%8D%C2%840%10E%3F%C3%88%C2%9A%C3%80%01%22%22%C2%BD%16%C2%B6%0C%C3%AA%C2%A0%0A%C2%97%40%07dt%C3%A0%C3%90%19%25%10%10S%02%11%26%C2%B0dc%C2%89%0Dv%C2%91vu%1B%C2%9C%7DZV%C2%BA%C3%A3%C2%87c%C3%B1%1F%C3%83%7C%C3%998)%C3%8B%C3%B2%C2%8B%C2%88%06%C3%86X%C2%8E%C2%83%C3%A4%C2%BD%C2%9F%C2%9Ds%17FD%03%C2%80%7CY%168%C3%A7%C3%9E%0E%26%22p%C3%8Es%22%1AR%C3%86Xn%C2%8C9%04%0C%00%C3%8E9%18c%C3%80%18%C3%8B%C3%93%C2%BDp%C2%A4%C3%96u%05%00%C2%A4%C2%87R%C3%AF%C3%9A%C2%B6%0D%00%C3%80b%1E%16B%7C%C2%AB%C3%95u%1D%C3%AC%C3%B3%C2%91%C3%8EO%C3%B8G%C3%A1Q%C2%81%C3%9B%C3%83%25%C2%84%C2%88%0A%C3%9A%C2%AE%C3%BF%C3%BB%C3%99O%C3%B8%09%3F%C3%A1%7F%1F%1E%C2%B4%C2%BDVU%C2%85%C2%A2(%C2%9Ej%C2%8Fg%C3%BB4Mh%C3%9B%C3%B6%C3%87~A%C2%9DK)a%C2%AD%7D%C2%B9f%C2%AD%C2%85%C2%942%C3%84.%0C%C2%AE%C2%B5%C2%86R%C3%AA%C3%A5%C2%9AR%0AZ%C3%AB%C3%B7%C3%81%01%60%1CG%C3%B4%7D%C3%BFT%C3%AB%C3%BB%1E%C3%A38%C2%86Z%C3%85%05%C2%AE%C3%AB%3AL%C3%93%04%C3%A06%C3%A7%C2%AE%C3%ABbl%C3%A2%C3%93.%C2%A5%C2%84%C3%96%3Ax%C3%8E%C2%8F%C2%8A%C3%BA%C2%99%00n%C3%B3o%C2%9A%26%1A%0C%C3%9C%3B\'%C2%A2_%C2%99%C2%84j%C3%A7%C2%A5%C3%9E%C3%BB%C2%99s~%C3%88%0B%24I%C2%B2%C3%9F%C3%95%C3%A0%C2%BD%C2%9F%C2%99s%C3%AEBDC%C2%96e%C2%87%C3%9FR%C2%AF%C2%A7%C3%A5%7Ba%C2%81K%7F%23%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%B9IDATH%C2%89%C3%8D%C2%97%C2%A1%C2%AE%C3%A2%40%14%C2%86%7F%C3%98%C3%89%C3%9Cd%04%C2%83%20%C3%A0f%15%24%C2%88%26W%20%C2%91%15%24%C2%83%02WY%C3%87%1B%20W%C3%B2%06%C2%B8J%24%C2%AEoP%07%09%24%C2%95%C2%B8%0A%C2%92%12%04CBC%C2%86%12V%C3%AC%C2%B2%7B%13X%C3%AE%C2%B0%2Ce%3F9%3D%3D_%C3%8F%C2%99%C3%A9I%26%C3%97j%C2%B5%C2%BERJ%C3%A7%C2%84%C2%90%222%22M%C3%93%C2%8D%C3%96%C3%BA%C2%9DPJ%C3%A7%00%C2%8A%C3%9B%C3%AD%16Z%C3%AB%C2%A7%C2%8B)%C2%A5%60%C2%8C%15)%C2%A5%C3%B3%3C!%C2%A4%C2%98%24I%26b%00%C3%90Z%23I%12%10B%C2%8A%C3%B9%C3%B3B%C2%96%1C%0E%07%00%40%3ES%C3%ABON%C2%A7%C3%93%C3%AB%C3%A4g%5E*\'%C3%97%16%C3%BB%C3%BD%C3%BE%C3%85%C3%9A%6000Nj%C3%BA%C3%BE%C3%BFW%C3%B95%C2%AEU%C2%93%C2%99%C3%BC%C3%91%C2%B6_%C3%83%C2%B8%C3%AD%C2%96e%19%C3%8BM%C3%89u%3A%C2%9D%C3%93z%C2%BD%C2%BE%19%24%C2%84%C2%80%C2%94%12J)%C3%B8%C2%BE%0F%C2%A5%C3%94%C3%83%C3%A2R%C2%A9%C2%84%2F%C3%B5z%C3%BD%5B%C2%92%247%03%C2%95R%C2%98N%C2%A7%C3%A0%C2%9C%C2%A3%C3%9Dn%C3%A3x%3Cb%C2%B9%5C%3E%24g%C2%8C%C2%99%C3%AF9%00%04A%C2%80%C3%85b%01)%25%C2%AA%C3%95%C3%AA%C3%83%5D0%C2%AA%C3%BC%23%C2%BB%C3%9D%0E%C2%B3%C3%99%0C%C2%9Cst%C2%BB%5D%00%40%14Ew%C2%8B%19c%7F%C3%BF%C2%9F%07A%C2%80%C3%A1p%08!%04%5C%C3%97E%C2%A5R%C2%B9%3B%C3%87%C2%A7r%C3%8E9%5C%C3%97%C2%BD%C3%BAL)%C2%85%C3%91h%C2%840%0C%C3%A18%0E%C2%9A%C3%8D%C3%A6%C2%BF%C2%93%C3%97j5%C2%B8%C2%AE%C2%8Br%C2%B9%7C3%C3%89d2%C2%81%C3%A7y%10B%C2%A0%C3%97%C3%ABA%08a%24%C3%BF%C3%A3%C2%81%C2%B3m%1B%C2%8DF%C3%83(%09%C3%B0%C2%BB%0B%C2%96e%C3%81q%C2%9C%C2%8B%C3%A7F%C2%B3%C3%BD%C3%9C%C3%A6%7B%C3%84%1F%09%C3%83%C3%908%C3%B6%C2%A2r%C3%87q%C3%809%C2%BF%08%C3%8Cd%C2%B6%C2%8F%C3%87cH)%2F%C3%B69%C2%93%C3%99%1E%C3%87%C3%B1%C2%AF%13%C3%BCl%C2%AE%1E%C2%B8%C3%BD~%0F%C3%9F%C3%B7%11E%11l%C3%9B%C3%86%C3%9B%C3%9B%C3%9B%5DIM%C2%BBts%C2%BC%C2%86a%C2%88%C3%95j%05)%C3%A5%5DrS%3E%1D2q%1C%C3%83%C3%B3%C2%BC%C3%97%C3%88%C2%9FI%1E%C3%B8q%C2%85%C3%89%C2%92%C2%B3%2F%C2%9F%C2%A6%C3%A9%C2%861%C2%96%C3%89%07%C3%A4r%C2%B9%C3%B3%5D%0Di%C2%9An%C2%88%C3%96%C3%BA%C2%9DR%3A%2F%14%0A%C2%99%C3%9FR%C2%BF%03%C3%8BC%C3%85%1D%2C%04%C2%BC%C2%BB%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%8CIDATH%C2%89%C3%AD%C2%97%C2%B1%C2%AE%C2%A2%40%14%C2%86%7F%C3%88db(%1C%0A%C2%A3%1D%C3%BB%04%24%C2%B7%C2%A1%C2%9B%070%C2%B1%C2%B3%C2%A4%C3%B4)x%04%C3%9E%C3%80%C3%8E%C3%92%C2%8Er%3AJ%3A(L%C3%AC(%C3%AD%24%16%C2%8E%05%C2%91%C2%91%C3%84-%5C%C2%8B%1B%C2%85%15%C3%A5%C3%8E6%C3%BB%C2%97\'%C3%A7%C3%A4%C2%9B%C3%B9g8%C3%8C1%C2%A6%C3%93%C3%A9%2FJ%C3%A9%C2%86%10bC%C2%93%C3%AA%C2%BA%3E*%C2%A5%C2%BE%08%C2%A5t%03%C3%80%3E%C2%9DNPJ%C3%BD8%C2%98R%0A%C3%8B%C2%B2lJ%C3%A9%C3%86%24%C2%84%C3%98eYj%01%03%C2%80R%0AeY%C2%82%10b%C2%9B%C3%B7%C2%80N%5D.%17%00%C2%80%C2%A9%C2%95%C3%BAG%C3%97%C3%AB%15%00%40%C3%9A%C2%92%C2%82%20x%C2%88%C2%85a%C3%98%5B%C3%BE%3F%C3%99%C3%B9%7Fxg%C2%B8%C3%A38%C2%BD%C3%81%C2%8D%C3%B9%7C~%3D%1C%0E%C2%ADIA%10%20%0CC%C2%B8%C2%AE%0B%C3%8E9%C2%96%C3%8B%C3%A5K%C3%B9m%1A%C2%8DF%C3%9Dv%C2%BE%C3%9Dn!%C2%A5%04%C3%A7%C2%BCKY%C2%A3%3A%C3%9B.%C2%84%C2%80%C3%A7y%C2%98L%26%C3%BA%C3%A1RJ%24I%C2%82%C3%99l%C2%A6%1F%0E%00i%C2%9A%C3%A2%7C%3E%7Fl%C3%BF%C3%9B%C2%9F%C2%9A%10%02%C2%9Cs0%C3%86%C3%B4%C3%83%C3%BB%C2%B0%C3%BF%C2%A3%26%C2%93%24%09%06%C2%83%01%3C%C3%8F%C3%93%0F%07%3E%C2%B3%C3%BFc%C3%B8~%C2%BFG%C2%9A%C2%A6o%C3%99%C3%9FKoO%C2%92%04%C2%8C1%C2%B8%C2%AE%C3%9B%C2%A9%C3%AEi%7B%5D%2C%16%18%C2%8F%C3%87%C2%8DEEQ%60%C2%B5Z%7D%C2%8B9%C2%8E%03%C3%9F%C3%B7%1Fr%C2%9B%C3%9Alc%7B%C2%8D%C2%A2%08UU%3D-%C2%AA%C2%AA%0AQ%14%3D%C3%84w%C2%BB%5D%C3%A3b%C2%9B%C3%B4%14.%C2%A5%C2%84%10%C3%A2i%C2%81%10%02R%C3%8A%C3%8E%C2%A0%C2%97%C3%A1%00%C2%90%C3%A79%C2%B2%2C%C3%BB%16%C3%8B%C2%B2%0Cy%C2%9E%C3%B7%02n%C2%85%03%40%1C%C3%87(%C2%8A%02%C3%80%C3%AD%C2%9C%C3%A38%C3%AE%0D%0C%C2%BC%C3%B0%3Fg%C2%8C%C3%81%C3%B7%7D%C2%AC%C3%97%C3%AB%C3%9E%C3%AC%06n%17%C2%AE%C3%B5%C3%B5%0A%C3%9C%C3%8E%C3%BFo%C2%8F%C2%87we%02%C2%B7%11F%C2%A7%C3%AE%3C%C2%B3%C2%AE%C3%AB%C2%A3eYZ%16%60%18%C3%86%7DVC%5D%C3%97G%C2%A2%C2%94%C3%BA%C2%A2%C2%94n%C2%86%C3%83%C2%A1%C3%B6)%C3%B57J%C2%AC%C3%82XK%20%C2%99R%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%017IDATH%C2%89%C3%AD%C3%97%C2%B1%C2%8A%C2%830%18%07%C3%B0%C2%BF%12%3E!C%C3%A3%C3%A4z%C2%9Bs%07%1F%C3%81%C2%A1%C2%90%C3%91%C2%ADc%C2%9F%C2%A2%C2%8F%C3%90%C2%A7ptt%C3%AB%7B%08y%03%C2%B7t*%1DB%C2%89%C2%81v9\'%7B%C2%AD%C3%8A5%3D8%C3%BFc%12%C3%B2K%C3%82%C2%97%40%C2%82%C3%8Df%C3%B3ED%0Dc%2C%C2%86%C2%A78%C3%A7%C3%8E%C3%96%C3%9A5%23%C2%A2%06%40%7C%C2%B9%5C%60%C2%AD%7D%3BLD%C3%A0%C2%9C%C3%87D%C3%94%C2%84%C2%8C%C2%B1%C3%98%18%C3%A3%05%06%00k-%C2%8C1%60%C2%8C%C3%85a%C3%9F%C3%A03%5D%C3%97%01%00B%C2%AF%C3%AAwn%C2%B7%C3%9B%C3%A7%C3%B0%3E%0B%C2%BE%C3%A0%7F%0B%17B%60%C2%B7%C3%9B%C2%BD%05g%C3%8F%3A%C3%934%C2%85%C2%94%12Q%14%C3%B9%C3%85%C3%B3%3CG%C2%96e%C2%B3%26%C3%9D%C3%AF%C3%B7%C2%83%C2%B6%C3%83%C3%A1%C3%B0%1A%17B%C2%A0(%0A%24I2%0B%C2%9E%C2%92%01%C2%BE%C3%9Dn!%C2%84%18%0C%7C%C2%B4%C2%9B_%C3%87%C3%AB%C2%BA%C2%86%C2%94r%C2%B0%C3%B3G%C3%87%C3%B6S%C3%86.tP%C3%ADZkTU%05%C2%A5%C3%94hln%1E%16%C3%9C%C3%B5z%C3%85%C3%B1xD%C3%9B%C2%B6%C3%88%C3%B3%7Cr%C2%B5%C2%8F%3D%C2%A5%C2%A7WM)%C2%85%C3%93%C3%A9%04)%C3%A5%24%7Cl%5E%3E2Zk%C2%94e%C3%B9%19%C3%BC%C2%9DY%C3%B0%05%C3%BFg8%11yE%7B%2Ft%C3%8E%C2%9D9%C3%A7%5E%16%10%04A%C3%BFW%C2%83s%C3%AE%C3%8C%C2%AC%C2%B5k%22jV%C2%AB%C2%95%C3%B7_%C3%AA%1DS%C2%B2f%C2%B2%C2%99%12(1%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%19IDATH%C2%89%C3%AD%C2%971%C2%AE%C2%84%20%10%C2%86%7F%C3%8DdH%2C%C3%84%C3%8A%C3%B6u%C3%94%5Bx%04%0A%13JN%C3%A1%0D8%C2%82%C2%A7%C3%B0L%1E%C3%85XP%20%C3%89ne%C2%B2%C2%9B%C2%B7%0D%C3%AE%C2%93W%C2%AC%7F%0530%1F%C2%99%C2%99%40(%C3%BA%C2%BE%C3%BFa%C3%A6%C2%99%C2%88%1AdR%C2%8Cq%09!%C3%9C%C2%88%C2%99g%00%C3%8D%C2%BA%C2%AE%08!%C2%9C%0EffTU%C3%950%C3%B3%5C%12Q%C3%A3%C2%BD%C3%8F%02%06%C2%80%10%02%C2%BC%C3%B7%20%C2%A2%C2%A6%C3%9C%0D9%C2%B5m%1B%00%C2%A0%3C%1A%40J%C2%89a%18%20%C2%A5L%C3%9E%7B%C2%BF%C3%9F%3F%C2%83%5Bk!%C2%A5%C2%84%C2%B5%C3%B6h%C2%88cp%C2%AD5%C3%9A%C2%B6%05%00%C2%B4m%0B%C2%ADu%1E%C2%B8R%0A%5D%C3%97%C2%BD%C3%98%C2%BA%C2%AE%C2%83R%C3%AA%5C%C2%B8%C2%94%12%C3%86%C2%98%C2%B7%3EcLr%C3%BD%C2%93%C3%A0%C3%96Z%08!%C3%9E%C3%BA%C2%84%10%C3%89%C3%B5%C2%A7%C2%94%C3%85%C3%934%C2%BD%C3%8C%C2%9Ds%18%C3%871%09%C3%B8%C2%AC%C3%83%C3%9D%C3%BE%17%C2%BA%C3%A0%17%C3%BC%C2%82_%C3%B0S%C2%95t%C2%BD%C3%AEr%C3%8E%C3%BD%1A%1F%C2%B9f%C2%BF7%C3%AD%C3%9F%0B%3F%C3%94p%C2%9F%C2%BC%C3%A1%C3%8F%C3%BA%C3%BF%C2%B43sV%C3%A8%C3%8E%2Bc%C2%8CKUUY%0EP%14%C3%85%C3%BEWC%C2%8Cq%C2%A1%10%C3%82%C2%8D%C2%99%C3%A7%C2%BA%C2%AE%C2%B3%C3%BFR%1F%078PT%C3%B5k%C3%91r%00%00%00%00IEND%C2%AEB%60%C2%82')
        },
        pressed: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%84IDATH%C2%89%C3%8D%C2%97%C2%B1%C2%8E%C2%B2%40%14%C2%85%C2%8F%C3%84d%C3%88%C3%90%00%C2%95S%C3%BD%C2%89O%C2%B0%C2%89%25%C2%95%C2%91%C3%92%C2%96%C3%92%C2%96%C2%92%C3%9E\'0ZR%C3%A2%23%60\'%C2%A5%C2%96%C2%94%18%2BK%5B%C2%AC%C3%90%06%C2%9Ci%C3%8C6K%C2%B2%C3%B9%C2%97%5D%07%5CqO9%C2%B9w%C2%BE9%C2%97%C3%89%09%C3%93%19%0E%C2%87%C3%BF(%C2%A5%7BB%C2%88%C2%8E%C2%96%24%C2%84%C2%B8%14E%C3%B1%C3%96%C2%A5%C2%94%C3%AE%3B%C2%9D%C2%8E~%3A%C2%9D%C2%90%C3%A7%C3%B9%C3%93%C3%81%C2%9A%C2%A6%C3%814M%C2%9DR%C2%BAW%08!z%C2%96e%C2%AD%C2%80%01%20%C3%8FsdY%06B%C2%88%C2%AE%C2%94%0Bm%C3%AAz%C2%BD%02%00%C2%94V%C2%A9%1F%C2%BA%C3%9Dn%C2%AF%C2%83%C2%97z)%C2%BC%5B%C2%B58%C2%9B%C3%8D%C2%BE%C2%ACM%C2%A7S%C3%A9Me%C3%BB%C3%BF%C2%9E%C3%B3*U%C2%B9i%0D%C3%BE%C3%A8%C3%98%C2%AB%C3%B4%C3%ABc%1F%0C%06%C3%92%C2%B5%C2%95%C3%8E%C3%AB%C2%B8%2Ce%18%06%1C%C3%87%C2%81%C2%AA%C2%AAX.%C2%978%1E%C2%8F%C3%8D%C3%A0ueY%16l%C3%9BF%1C%C3%87%C3%98l6%C3%92%7D%0F%C3%81%3F%C2%BB%0D%C2%82%00i%C2%9A%C3%96%C3%AAo%0C%C2%B7m%1B%C2%A3%C3%91%08%C3%9B%C3%AD%C2%B6%C2%96%C3%9B%C2%87%C3%A0%C2%8C18%C2%8E%03%C3%8E9%16%C2%8B%05%C3%8E%C3%A7s%230P%C3%B3%C2%B6%C3%9B%C2%B6%0D%C3%97u%C2%91%24%09%C2%82%20%C3%B8%16%C3%ACy%1E%0C%C3%83%C2%B8%C2%BB%C2%9F%C2%94%C3%B3~%C2%BF%C2%8F%C3%B1x%0C%C3%8E9%7C%C3%9F%C2%BF%C3%AB%C2%961%06%C3%8F%C3%B3%10%C2%86!%0E%C2%87C%3DxUH%C2%ACV%2B%24I%22sV%00%C2%80%C2%AA%C2%AA%C2%98L%26%C2%88%C3%A3%18Q%14U%C3%96H%C2%8F%C2%BD%0E%C3%B8%C2%B3%2C%C3%8B%C3%BA%C3%B63%C2%B4%C2%92%C3%AD%C2%8C1%C2%B8%C2%AE%C2%8B%C3%B9%7C%C3%9E%0C%C3%BEH%C2%B6%C2%A7i%C2%8A0%0C%C2%BF%C3%94%C3%BDJ%C3%82%C3%BD%C2%A4%C3%9Dn%C2%87%C3%B5z%0D%C3%8E%C2%B9%1C%C2%BCI%C2%B6%C3%BF%2F%C3%8E9%C2%A2(%C3%BA%C3%B1%C2%AE%3C%C3%85y9%C3%A6%7Bq%C3%BB%14%C2%B8%C3%AF%C3%BBRu%C2%AF%C3%BF%7B%C3%954%C2%ADUh%C3%89S%C2%84%10%17%C3%934%5B9%C2%80%C2%A2(%C3%A5%5B%0DB%C2%88K%C2%B7(%C2%8A7J%C3%A9%C2%BE%C3%97%C3%AB%C2%B5%C3%BEJ%7D%074%2B%C2%B5r%C3%85k%C3%A4%C2%92%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%91IDATH%C2%89%C3%AD%C2%971%C2%AF%C2%A2%40%10%C3%87%C3%BF%10%C2%935k%03TR%5D%22_%C3%A0%25%C2%96TFJ%2B%13%C2%BE%C2%82%C2%A5%C2%BD%C2%9F%C2%80%C3%98ZjI%C2%89%25%25%C2%96%C2%94%24Vt%C3%9A%C2%AE%15k%03%C2%B2%C2%8Dw%C3%85%1D%C2%97%C2%BC%7BJ%C2%9E%0B%C3%87k%C3%9E%C2%BF%C2%9C%C2%9D%C3%89ow2%3B%C2%BB%C2%A3L%26%C2%93%1F%C2%94%C3%92%23!DCG%12B%5C%C2%8B%C2%A2x%C3%ABQJ%C2%8F%C2%8A%C2%A2h%C2%97%C3%8B%05y%C2%9E%C3%BFw%C3%B0%600%C2%80a%18%1A%C2%A5%C3%B4%C2%A8%12B%C2%B4%2C%C3%8B%3A%01%03%40%C2%9E%C3%A7%C3%88%C2%B2%0C%C2%84%10M%C2%AD%0C%5D%C3%AAv%C2%BB%01%00%C3%94N%C2%A9%7Ft%C2%BF%C3%9F%01%00%C2%BD%3A\'%C3%8F%C3%B3%3E%C3%98V%C2%ABUk%C3%BE_r%C3%B2oxk%C3%B0%C3%91h%C3%B4rLm%C3%81U%C3%85%C3%A2y%5Em%C3%A1%00%C3%80l6%C3%83~%C2%BFG%C2%92%24%C2%9F%C3%B2%07Z%3A%C2%B9%C3%A38(%C3%8B%12I%C2%92%C2%BC%14%C3%97%18n%C2%9A%26l%C3%9BF%10%04%2F%C3%876%C2%86%C2%BB%C2%AE%C2%8B(%C2%8A%C3%809%C3%AF%16%5E%C2%A5%3B%C2%8Ec%C2%A9xi%C2%B8%C2%AE%C3%AB%C2%98N%C2%A7R%C3%A9n%0Cw%5D%17%C2%87%C3%83A*%C3%9D%C2%8D%C3%A0%C2%B6m%C2%A3%C3%9F%C3%AF%23%C2%8A%22i%C2%B0%14%5C%C3%97u8%C2%8E%C3%93(%C3%9D%C3%92p%C3%97u%11%C3%871%18c%C3%9D%C3%82%C3%87%C3%A3q%2B%C3%A9%C2%AE%C2%A4%C3%8C%C3%A7%C3%B3%C2%9F%C2%A7%C3%93%C3%A9%C3%A1%C3%A2%C2%A3%C3%B7y%C2%B7%C3%9B%C3%A1%7C%3E%C2%BF%C2%B3-%C2%97K%C2%98%C2%A6%C3%B9%14%C3%82%18%C3%83f%C2%B3yg%C2%B3%2C%C3%AB%C3%B5%C2%B4%C3%BF%0B%06%00%C3%9F%C3%B7Q%C2%96%C3%A5C%C3%BF%C2%B2%2C%C3%A1%C3%BB%C3%BE%C3%83%C2%B5Vz%3B%C3%A7%C3%BCi%01%06A%C3%B0%C3%B4%3A%C2%B6%C3%B6%C2%A4%C2%A6i%C3%BA%C2%A1%C3%93%C3%85q%C2%8C4M%C2%9F%C3%86%C2%B4%C3%BA%C2%99%08%C3%83%C3%B0%C3%AF-%60%C2%8C!%0C%C3%83Z%C3%BF%C3%9A%C2%82%C2%93%C2%91%C2%AE%C3%ABX%2C%16%C3%98n%C2%B7%C2%B5%C3%9D%C3%8F%C2%B2%C2%AC%C3%BA%C3%8F%C2%84%C2%8C8%C3%A7X%C2%AF%C3%97%C2%9F%C3%B2U%C2%81%C3%9F%23L%C2%97%C2%AAx%C2%AA%10%C3%A2j%18F\'%1BPU%C2%B5%C2%9A%C3%95%20%C2%84%C2%B8%C3%B6%C2%8A%C2%A2x%C2%A3%C2%94%1E%C2%87%C3%83a%C3%A7S%C3%AA%2F_%C2%B5%C3%82%C3%AE%C2%BFG%3C%C2%9D%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%011IDATH%C2%89%C3%AD%C2%97-%C2%8E%C2%84%40%10F%3F%3A%24M%1A%03(P%C3%ABQ%C3%A3Q%1C%04%C2%8B%C3%84s%02%C2%82%C3%A5%1AH%C2%B8%02%12%C3%82-%40%11%0C%3Fm%C3%88%C2%98%C3%85%C3%AC0%13%C2%98%C2%84%C2%9E%C3%9D%0DOV%C2%BE%C3%A4UWu%3Ai%C3%89u%C3%9D%2F%C3%86XM)%C3%95%20%08%C3%8Ey%3F%C2%8E%C3%A3Mf%C2%8C%C3%95%C2%92%24im%C3%9Bb%18%C2%86%C3%93%C3%85%C2%AA%C2%AA%C3%820%0C%C2%8D1V%13J%C2%A9%C3%96u%C2%9D%101%00%0C%C3%83%C2%80%C2%AE%C3%AB%40)%C3%95%C3%88Z%10%C3%894M%00%00%22%C3%94%C3%BA%C3%8D%C2%B2%2C%C2%9F%C2%93%C2%AF%5C%C3%B2K%C3%BE%C3%B7%C3%A5A%10%40%C3%97%C3%B5%C3%8F%C3%88-%C3%8BB%10%04%C2%B0m%C3%BBeN%C3%9E*FQ%C3%B4P%0B%C3%83%C3%B0P%03%C2%8A%C2%A2%C3%80%C3%B3%3C%14E%C2%81%3C%C3%8F73%C2%A7%C3%AF%C3%9Cq%C2%9C%C2%A7k%C3%98%3C%C3%B9%16%5B%C3%93%C3%98%C2%8BeY%C3%B0%7D%1Fq%1C%C2%BF\'%3F2%C3%B6%C2%9F%C2%8D6M%C2%834M%1Fr%C2%BB%C3%A5%C3%AFRU%15%C2%B2%2C%C3%83%3C%C3%8F%C3%BB%C3%A4G%2F%C3%97%16%C3%B3%3C%23%C3%8Fs%C2%94e%C3%B94s%C3%8A%C3%89%C3%9717M%C3%B32w%C2%8A%3CI%C2%92%5D%C2%B9%C3%BF%C3%B7%C2%BC%5E%C3%B2K%C3%BE%7B%C3%A5%C2%AA%C2%AA%0A%C2%95%C2%AE%3E%C3%829%C3%AF%0D%C3%83%10%C3%92%00!d%C3%BD%C2%AB%C2%81s%C3%9E%C3%8B%C3%A38%C3%9E%18c%C2%B5i%C2%9A%C3%82%7F%C2%A9w%C2%91xm%C2%87%C2%A8%C2%AC%C3%8C%C2%9C%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%019IDATH%C2%89%C3%AD%C2%971%C2%8E%C2%830%14D%07%0B%C3%89%C2%92i%10%15T%7B%C2%85%5C%C2%81%0B%C3%91s%02D%C3%8B%15%5C%C3%92%C3%BA%0A%C3%AE%C2%A9%C2%B8%02T%C2%86%06%C2%83%1B%C2%94%22AJ%C2%B4)%C3%96%5E%C3%85%C2%91v%C2%99%C3%B2%5B%C3%8C%C3%A3%C3%B3G6%0E%C3%B2%3C%C3%BFb%C2%8Cu%C2%94%C3%92%18%C2%9Ed%C2%8C%C2%99%C2%B5%C3%96%C2%97%C2%901%C3%96%05A%10%C2%8F%C3%A3%C2%88eY%C3%9E%0E%C2%8E%C2%A2%08I%C2%92%C3%84%C2%8C%C2%B1%C2%8EPJc%C2%A5%C2%94%170%00%2C%C3%8B%02%C2%A5%14(%C2%A519%0A%3E%C2%B5%C2%AE%2B%00%C2%80x%C2%A5%C3%9E%C2%B5%C3%AF%3B%00%20ty%C2%B8%C2%AA%C2%AAo%C2%B5%C2%B2%2C%C2%AD%7D%3E%C3%92%C3%B9%09%C3%BF(%C3%9C)pG%C2%B8%C2%AA%C2%AAr%0A%C3%9A%C2%A1%C3%BF%C3%BB%C3%99O%C3%B8%09%3F%C3%A1%7F%1Fn%C2%B5%C2%BD%16E%C2%81%2C%C3%8B%C2%9Ej%C2%8Fg%C3%BB0%0Ch%C2%9A%C3%A6%C3%87~V%C2%9Ds%C3%8E%C2%B1m%C3%9B%C3%8B%C2%B5m%C3%9B%C3%809%C2%B7%C2%B1%C2%B3%C2%83O%C3%93%C2%84%C2%B6m_%C2%AE%C2%B5m%C2%8Bi%C2%9A%C3%9E%07%07%C2%80%C2%BE%C3%AF!%C2%A5%7C%C2%AAI)%C3%91%C3%B7%C2%BD%C2%AD%C2%95%5B%C3%A0%C2%84%10%18%C2%86%01%C3%80m%C3%8EB%08%17%1B%C3%B7%C2%B4s%C3%8E1%C3%8F%C2%B3%C3%B5%C2%9C%1F%C3%A5%C3%B43%01%C3%9C%C3%A6_%C3%97%C2%B53%18%C2%B8w%1EE%C3%91%C2%AFLlu%C3%B0%C2%881fN%C2%92%C3%84%C3%8B%0B%10B%C2%8E%C2%BB%1A%C2%8C1s%C2%A8%C2%B5%C2%BE0%C3%86%C2%BA4M%C2%BD%C3%9FR%C2%AF%C2%B8ry%C3%B0%C3%94%C3%82%C2%8E%C2%9E%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%B9IDATH%C2%89%C3%8D%C2%97%C2%B1%C2%AE%C2%A2%40%14%C2%86%7FX%13n%C2%B0a%C2%A8%C2%B4%C3%9Ad%C2%AC%C2%B4%C2%BA%09%C2%A5%C2%95%C2%89%25%C2%95%C3%89%C2%94%C2%96R%C3%92%1A%C2%BB%C3%ADL%2C))my%05Z%0A%0B%C2%8C%2F%20%C2%8D%C2%89X1%16%C2%82%C2%92%18w%C2%8B%5D%C2%B3%26%C2%B2%C3%97a%C3%9D%C2%8B%C3%BB%C2%953%C3%83%C3%B9%C3%A6%1C%26%07F%C3%AA%C3%B5z_UU%5D)%C2%8A%C2%A2%C2%A1%22%C3%B2%3C%C3%9FgY%C3%B6%5ESUu%25I%C2%92%C2%B6%C3%9B%C3%AD%C2%90%C2%A6%C3%A9%C2%A7%C2%8B%C3%AB%C3%B5%3At%5D%C3%97TU%5D%C3%89%C2%8A%C2%A2hI%C2%92T%22%06%C2%804M%C2%91%24%09%14E%C3%91%C3%A4%C3%AB%40%C2%95%1C%C2%8FG%00%C2%80%5C%C2%A9%C3%B5%17%C2%97%C3%8B%C3%A5u%C3%B2%2B%2F%C2%95%C3%97%C2%8A%06%C2%A7%C3%93%C3%A9%C3%9D%C3%98d2%11%0E*%C3%BA%C3%BC%C3%BF%C2%97y%11E%C3%99T%26%7F%C2%B6%C3%ACE%08%C2%97%C3%9D0%0Ca%C2%B9(%C3%92%600%C3%B8%C2%BE%5E%C2%AF%3F%5CD)%05c%0C%C2%9Csx%C2%9E%07%C3%8E%C3%B9%C3%93%C3%A2V%C2%AB%C2%85%2F%C3%ADv%C3%BB%C3%9B%C2%A3%60%C2%9Cs%04A%00B%08%18c8%C2%9F%C3%8F%C3%98l6O%C3%89u%5D%2Fw%C3%9A%7D%C3%9F%C2%87%C3%AB%C2%BA0%0C%03%C2%96e%C2%81%10%C3%B2%C3%94%06%C2%842%C2%BF%C3%A5p8%60%C2%B1X%C2%80%10%C2%82%C3%A1p%08I%C2%92%10EQiq%C3%A9%C3%8Co%C3%B1%7D%1F%C2%B3%C3%99%0C%C2%94R%C3%98%C2%B6%C2%8Df%C2%B3Y%3A%C3%86C9!%04%C2%B6m%17%C3%8Eq%C3%8E%C3%A1%C2%BA.%C3%820%C2%84eY%C3%A8%C3%B7%C3%BB%C3%BFN%C3%9E%C3%A9t%C2%84%C2%B2%0A%C2%82%00%C2%8E%C3%A3%C2%80R%C2%8A%C3%B1x%0CJ%C2%A9%C2%90%C3%BC%C2%8FM%C3%864Mt%C2%BB%5D%C2%A1%20%C3%80%C3%AF*%18%C2%86%C2%81%C3%91ht7%2F%C3%94%C3%9B%C2%AFe.%23%C2%BE%25%0CC%C3%A1%C2%B5w%C2%99%5B%C2%96%05M%C2%BB%C3%BF%C2%97%C2%AC%C2%A4%C2%B7%C3%8F%C3%A7s0%C3%86%C3%AE%C3%9Es%25%C2%BD%3D%C2%8Ec%C2%B8%C2%AE%C2%8B%C3%A5r)%2C%C3%BB%5B%0A%0F%C3%9C%C3%A9t%C2%82%C3%A7y%C2%88%C2%A2%08%C2%A6i%C3%A2%C3%AD%C3%AD%C2%ADTP%C3%91*%7D%C3%B8I%0D%C3%83%10%C3%9B%C3%AD%16%C2%8C%C2%B1RrQ%1E6%C2%998%C2%8E%C3%A18%C3%8Ek%C3%A4%C2%9F%C2%89%0C%C3%BC%C2%BC%C3%82T%C3%89%C3%95\'%C3%A7y%C2%BE%C3%97u%C2%BD%C2%92%0D%C3%88%C2%B2%7C%C2%BD%C2%AB!%C3%8F%C3%B3%7D-%C3%8B%C2%B2wUUW%C2%8DF%C2%A3%C3%B2%5B%C3%AA%0F%C2%BE%07%C3%80%C2%A1%C3%B4%03%C3%B8%C3%BF%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%96IDATH%C2%89%C3%AD%C2%97%C2%B1%C3%8E%C2%A2%40%14%C2%85%0F%C3%84d%C3%8C%C3%98%00%C2%95V%C2%9B%C3%A0%0B%C3%BC%09%25%C2%95q%C2%9E%C3%80%C3%84G%C3%90%C3%92%C2%9E\'%20%C2%B6%C2%96%C2%B6%C2%94%C3%93Rb9%C2%A5%C2%89%15%C2%9D%C2%B6X16%204%C3%AE%16%C2%AC%C3%85%1F%C2%85%15eg%C2%9B%3D%C3%A5%C3%8D%C2%BD%C3%B9f%C3%8E%0C%C2%97%C2%B9%C3%9Ad2%C3%B9A)%3D%10B%0C(RY%C2%96%C2%97%3C%C3%8F%C2%BFz%C2%94%C3%92%C2%83%C2%A6i%C3%86%C3%B9%7CF%C2%96e%7F%1D%3C%18%0C%60Y%C2%96A)%3D%C3%A8%C2%84%10%23MS%25%60%00%C3%88%C2%B2%0Ci%C2%9A%C2%82%10b%C3%A8%C3%B7%C2%80J%5D%C2%AFW%00%C2%80%C2%AE%C2%94%C3%BA%5B%C2%B7%C3%9B%0D%00%C3%90kJ%C3%B2%7D%C3%BF!%C3%A6y%5Eg%C3%B9%C3%BFd%C3%A7%C3%BF%C3%A1%C2%AD%C3%A1%C2%B6mw%06%C3%97f%C2%B3%C3%99%C3%8F%C3%A3%C3%B1%C3%98%C2%98%C3%A4%C3%BB%3E%3C%C3%8F%C2%83%C3%A38%60%C2%8Ca%C2%BD%5E%C2%BF%C2%94%C3%9F%C2%A4%C3%B1x%C3%9Cn%C3%A7%C3%BB%C3%BD%1ERJ0%C3%86%C3%9A%C2%94%C3%95%C2%AA%C2%B5%C3%AD%C2%9Cs%C2%B8%C2%AE%C2%8B%C3%91h%C2%A4%1E.%C2%A5D%14E%C2%98%C3%8F%C3%A7%C3%AA%C3%A1%00%20%C2%84%40Q%14%1F%C3%9B%C3%BF%C3%B6%C2%A7%C3%869%C3%87t%3A%C2%85i%C2%9A%C3%AA%C3%A1RJ%C3%ACv%C2%BB%C2%8F%C3%AC%C3%BF%C2%A8%C3%89DQ%C2%84~%C2%BF%0F%C3%97u%C3%95%C3%83%C2%81%C3%8A~%C3%86%C3%98%5B%C3%B6%7F%0CO%C2%92%04B%C2%88%C2%B7%C3%AC%C3%AF%C2%A4%C2%B7GQ%04%C3%934%C3%A18N%C2%AB%C2%BA%C2%A7%C3%ADu%C2%B5Z56%C2%91%24I%C2%B0%C3%99l%C2%BE%C3%85l%C3%9B%C3%86b%C2%B1x%C3%88%C2%ADk%C2%B3%C2%B5%C3%AD5%08%02%14E%C3%B1%C2%B4%C2%A8(%0A%04A%C3%B0%10%3F%C2%9DN%C2%B5%C2%8B%C2%AD%C3%93S%C2%B8%C2%94%12%C2%9C%C3%B3%C2%A7%05%C2%9CsH)%5B%C2%83%5E%C2%86%03%40%1C%C3%87%10B%7C%C2%8B%09!%10%C3%87q\'%C3%A0F8%00%C2%84a%C2%88%24I%00T%C3%A7%1C%C2%86ag%60%C3%A0%C2%85%C3%BF%C2%B9i%C2%9AX.%C2%97%C3%98n%C2%B7%C2%9D%C3%99%0DT%17%C2%AE%C3%B1%C3%B5%0AT%C3%A7%C3%BF%C2%A7%C3%87%C3%83%C2%BB%C3%92%C2%81j%C2%84Q%C2%A9%3BO%2F%C3%8B%C3%B2bY%C2%96%C2%92%05%C3%A8%C2%BA~%C2%9F%C3%95P%C2%96%C3%A5%C2%A5%C2%97%C3%A7%C3%B9%17%C2%A5%C3%B40%1C%0E%C2%95O%C2%A9%C2%BF%00%C3%82I%C3%81%26%C3%95%C2%B7Y4%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%3AIDATH%C2%89%C3%AD%C3%97%C2%A1%C2%8E%C2%830%1C%06%C3%B0%C2%8F%C2%86%C2%A4%C2%A45%05%05%C3%AA%C3%BC%C3%94%C3%BC%C3%94%3C%C2%B6rvr%C2%9E\'%C3%80%23%C2%91%C2%B3%3C%0B%C2%84\'%40%0E%14%C2%99%C2%81%C2%ADf%C2%999%14%C3%9C%06%C3%A4%C3%A8.9%3E%C3%99%26%C3%BD%C2%B5%C3%8D%C2%BFmj%C3%AC%C3%B7%C3%BB%2F%C3%86XN)%15%C3%90%14%C2%A5%C3%94%C2%B5m%C3%9B%C2%AD%C3%89%18%C3%8B%0D%C3%83%10UU%C2%A1i%C2%9A%C3%85a%C3%8E9%1C%C3%87%11%C2%8C%C2%B1%C2%9CPJE%5D%C3%97Z%60%00h%C2%9A%06u%5D%C2%83R*H%C3%97%C2%A03%C2%B7%C3%9B%0D%00%40%C2%B4%C2%AA%C3%9Fy%3C%1E%C2%9F%C3%83%C2%BB%C2%AC%C3%B8%C2%8A%C3%BF-%C3%9C%C2%B6m%C2%9CN%C2%A7Ep%C3%B3U%C3%A7f%C2%B3%C2%81%C2%94%12%C2%96e%C3%A9%C3%85%7D%C3%9F%C3%87n%C2%B7%C2%9B5h%18%C2%86%C2%BD%C2%B6%20%08%C3%9E%C3%A3%C2%B6m%C3%A3p8%C3%80%C3%B3%C2%BCY%C3%B0%C2%94%C3%B4%C3%B0%C3%A3%C3%B1%08!%C3%BA%C2%AF%C3%AB%C3%90j~%1D%3F%C2%9F%C3%8F%C2%90R%C3%B6V%3E%C2%B4m%3Fe%C3%ACD%7B%C3%95%5E%C2%96%25%C3%A28F%C2%96e%C2%A3%C2%B1%C2%B9%19%2C%C2%B8%C3%BB%C3%BD%C2%8E%24IP%14%05%7C%C3%9F%C2%9F%5C%C3%ADcw%C3%A9%C3%A5QK%C3%93%14%C2%97%C3%8B%05R%C3%8AI%C3%B8%C3%98%C2%BC%C2%BDd%C3%8A%C2%B2D%14E%C2%9F%C3%81%C2%97%C3%8C%C2%8A%C2%AF%C3%B8%3F%C3%839%C3%A7Z%C3%91%C3%8E%23J%C2%A9%C2%AB%C3%A38Z%26%40%08%C3%A9%C3%BEjPJ%5D%C3%8D%C2%B6m%C2%B7%C2%8C%C2%B1%C3%9Cu%5D%C3%AD%C2%BF%C3%94\'%C2%BD%06g\'%C2%9E%C2%A8%3A%2C%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%1EIDATH%C2%89%C3%AD%C2%97%C2%B1%C2%8E%C2%84%20%10%C2%86%7F%C2%89%09%06%1B%C2%B0%C3%92%C3%AAz%C2%AA%C3%AD%C2%AD%C3%ACm)y%0E%C3%A3%13%C3%98%C3%BBd%3E%C2%86%15%C3%99%06%0C%C2%8D%C3%99%C3%8Ad7%C2%B7%0D%C3%AC%C3%89%15%C3%AB_%C3%81%0C%C3%8CGf%26%10%C2%B2%C2%AE%C3%AB~%18c%0B%C2%A5%C2%94%23%C2%91%C2%BC%C3%B7w%C3%A7%C3%9C-g%C2%8C-Y%C2%96%C3%B1u%5Da%C2%AD%3D%1D%5C%C2%96%25%C2%AA%C2%AA%C3%A2%C2%8C%C2%B1%C2%85PJ%C2%B91%26%09%18%00%C2%AC%C2%B50%C3%86%C2%80R%C3%8A%C3%89aH%C2%A9m%C3%9B%00%00%246%C2%80%10%02%C3%830%40%08%11%C2%BCw%C3%9F%C3%B7%C3%8F%C3%A0Zkp%C3%8E%C2%A1%C2%B5%C2%8E%0D%11%07%C3%AF%C3%BB%1EM%C3%93%00%00%C2%9A%C2%A6A%C3%9F%C3%B7i%C3%A0RJ%C2%B4m%C3%BBbk%C3%9B%16R%C3%8As%C3%A1B%08(%C2%A5%C3%9E%C3%BA%C2%94R%C3%81%C3%B5%0F%C2%82k%C2%ADQ%14%C3%85%5B_Q%14%C3%81%C3%B5%C3%8FC%16%C3%8F%C3%B3%C3%BC2%C2%9F%C2%A6%09%C3%A38%06%01%C2%9F%15%C3%9D%C3%AD%7F%C2%A1%0B~%C3%81%2F%C3%B8%05%3FUA%C3%97%C3%AB%C2%A1i%C2%9A~%C2%8Dc%C2%AE%C3%99%C3%AFM%C3%BB%C3%B7%C3%82%C2%A3%1A%C3%AE%C2%937%C3%BCY%C3%BF%C2%9F%C3%B6%C2%B2%2C%C2%93B%0F%1E%C3%B1%C3%9E%C3%9F%C2%AB%C2%AAJr%00B%C3%88%C3%B1W%C2%83%C3%B7%C3%BE%C2%9E%3B%C3%A7n%C2%8C%C2%B1%C2%A5%C2%AE%C3%AB%C3%A4%C2%BF%C3%94%07%240Q%C2%BFi%C3%AF%5C%C2%88%00%00%00%00IEND%C2%AEB%60%C2%82')
        },
        checked: {
            rowLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%89IDATH%C2%89%C3%8D%C2%97%C2%A1%C2%AF%C3%AA0%18%C3%85%C3%8F%5DH%C3%93%C2%A9-Yp%0B%C3%8F%C2%A1\'Q%24%60k%C3%91%C2%B3%248%C3%AC%02vv%C2%8E%3F%C2%A0%C2%B6%C3%98Y%C2%B0%C2%95%60%C2%91d%C2%8E%40%C2%98%C3%AA%C2%98%C3%A0%5E%C3%B1%C3%9E%5Enr%0Blw%C2%97q%C2%8F%5B%C3%B3%C2%B5%C2%BF%C2%9Eo%C3%8DI%C3%BB%C3%A6y%C3%9E%1FJ%C3%A9%C2%86%10b%C2%A3!%C3%A5y~%C3%8E%C2%B2%C3%8CkQJ7%00%C3%AC%C3%83%C3%A1%00%C2%A5%C3%94%C3%93%C3%81%C2%A6i%C3%82%C2%B2%2C%C2%9BR%C2%BA1%08!v%C2%9A%C2%A6%C2%8D%C2%80%01%40)%C2%854MA%08%C2%B1%C2%8Db%C2%A0I%5D.%17%00%C2%80%C3%91(%C3%B5%C2%9F%C2%AE%C3%97%C3%AB%C3%AB%C3%A0%C2%85%5E%0Ao%C3%A9%06%17%C2%8B%C3%85%C2%97%C2%B1%C3%89dRz%C3%91%C2%B2%C3%B3%7F%C2%9Fs%C2%9Dtn%1A%C2%83%C3%97m%C2%BBN%3F%C3%9E%C3%B6%5E%C2%AFW%C2%BA%C3%B6%C2%AD%C3%9F%C3%AF%C2%BF%C3%AF%C3%B7%C3%BB%C3%9AP%C3%87q%C3%A0%C3%BB%3EL%C3%93%C3%84r%C2%B9%C3%84n%C2%B7%C2%BB%5B%C3%9F%C3%A9t%C3%8A%C2%B7%C3%BD%C2%9E%C2%86%C3%83!%18cX%C2%ADV%C2%88%C3%A3%C2%B8%C3%B4%C2%BCZ%C3%B0%C3%8Fn%C2%A3(B%C2%92%24%C2%95%C3%A6%7F%1B%C3%8E%18%03c%0Cq%1CWr%5B%0B%C3%AE%C2%BA.%7C%C3%9F%C2%87R%0A%C3%B3%C3%B9%1C%C3%87%C3%A3%C3%B1%5B%60%C2%A0%C3%A2ig%C2%8Ca%3A%C2%9DBJ%C2%89(%C2%8An%C2%82%C2%83%20%C2%80%C3%A38%0F%C3%97%2B%C3%A5%C2%BC%C3%9B%C3%ADb4%1AA)%C2%850%0C%1F%C2%BAu%5D%17A%10%C2%80s%C2%8E%C3%ADv%5B%0D%C2%AE%0B%09%C3%8E9%C2%A4%C2%94e%C3%B6%0A%C3%A0%C3%AF%C2%8De%3C%1Ec%C2%BD%5EC%08%C2%A1%C2%AD)%C3%9D%C3%B6*%C3%A0%C3%8F%1A%0C%067%7F%C2%836d%C2%9E%C2%91%C3%A3%C2%A7%C3%93%09%C2%B3%C3%99%C3%AC%C3%BFw%C2%A5%C2%90%C2%A9%C2%93%C3%ADI%C2%92%C2%80s%C3%BE%C2%A5%C3%AEG%12%C3%AE%C2%9E%C2%A4%C2%94%10Bh%C3%AF%C2%89Zx%15%C2%97%C2%B7%C2%A4%C2%94%C2%82%10%C3%A2%C3%AEYy%C2%8A%C3%B3%C2%A2%C3%8D%C2%8F%C3%A2%C3%B6)%C3%B00%0CK%C3%95%C2%BD%C3%BE%C3%B6j%C2%9Af%C2%A3%C3%90%C2%82g%C3%A4y~%C2%B6%2C%C2%AB%C2%91%0D%18%C2%86Q%C2%BC%C3%95%C2%90%C3%A7%C3%B9%C2%B9%C2%95e%C2%99G)%C3%9D%C2%B4%C3%9B%C3%AD%C3%86_%C2%A9%1Fq%C3%AD%C2%BEB-%C2%8A%C3%81%C2%88%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnLR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C2%B5IDATH%C2%89%C3%AD%C2%97%2F%C2%AF%C2%A3%40%14%C3%85%C3%8F%23%0D%C2%99Q%C2%90%C2%90%3A%C3%82%C2%BAjdU%05%16%5B%C3%8D%07%C2%A8%C2%AB%C3%AD%07%C2%A8m%15%09v%2C%1A%0B%16Yle3%C2%AE!)j%C3%A8%C2%88%C3%AE%C2%9A%C3%A5%25%C3%9D-%C3%A4%15%C3%A6%C3%8D%C2%9A%3D%C3%B2%C3%8E%C2%BD%C3%B9%C3%8D%C2%9C%C3%9C%C3%B9%C3%B7%C3%A1%C3%BB%C3%BE%0FB%C3%88%C3%894M%1B%C2%9A%24%C2%A5%C2%BC%C2%B5m%C3%AB%C3%8F%08!\'%00%C3%B6%C3%B5z%C2%85%10%C3%A2%C3%9B%C3%81%C2%94RX%C2%96e%13BN%C2%86i%C2%9Av%C3%934Z%C3%80%00%20%C2%84%40%C3%9340M%C3%936%C2%BA%C2%80N%C3%9D%C3%AFw%00%C2%80%C2%A1%C2%95%C3%BA%5B%C2%8F%C3%87%03%000%1BJ%C2%8A%C3%A3%C3%B8%C2%AF%C3%98f%C2%B3Q%C2%96%C3%BFOV%C3%BE%1F%C2%AE%0C%C2%BEX%2C%C3%9E%C2%AE%19l%C2%B8%C2%AEY%C3%A28%1El%1C%00X%C2%AF%C3%97%60%C2%8C%C2%A1%2C%C3%8B%2F%C3%A5%03%C2%8AV%1E%C2%86!%C2%84%10(%C3%8B%C3%B2%C2%AD%C2%BA%C3%89p%C3%97u%11%04%01%18co%C3%97N%C2%86GQ%C2%84%2C%C3%8BP%C3%97%C2%B5%5Exgw%C2%9E%C3%A7%C2%A3%C3%AAG%C3%83%1D%C3%87A%18%C2%86%C2%A3%C3%AC%C2%9E%0C%C2%9Fb%C3%B7%24x%10%04%C2%A0%C2%94%22%C3%8B%C2%B2%C3%91%C3%A0Qp%15v%C2%8F%C2%86GQ%C2%84%3C%C3%8F%C3%819%C3%97%0B_.%C2%97J%C3%AC%C3%AE%C3%B4%C2%B1Z%C2%AD~%5E.%C2%97%C2%97%C2%83%C2%AF%C3%AE%C3%A7%C3%A3%C3%B1%C2%88%C3%B3%C3%B9%C3%BC%14%C3%9B%C3%ADvp%5D%C2%B7%17%C3%829%C3%87~%C2%BF%7F%C2%8Ay%C2%9E%C3%B7%C2%BE%C3%AD%7F%C2%82%01%20I%C2%92%C3%9E%C2%A7%C2%98%10%02I%C2%92%C2%BC%1CSr%C2%B6%C3%97u%C3%9D%C3%9B%C2%80%C2%8C%C2%B1%C3%9E%C3%AD%C2%A8%C3%ACJ%C2%AD%C2%AA%0AEQ%3C%C3%85%C2%8A%C2%A2%40UU%C2%BD5J%1F%13i%C2%9A~%C3%AE%02%C3%8E9%C3%924%1D%C3%8C%1Fl%C2%B81r%1C%07%C3%9B%C3%AD%16%C2%87%C3%83a%C3%B0%C3%B4%C3%B3%3CO%3D%C3%BC%C2%AB%C3%BA%C3%ACvJ%C2%A9Vp%C3%873%C2%A4%C2%947%C3%8B%C2%B2%C2%B4L%C3%800%C2%8C%C3%AE%C2%AF%06)%C3%A5m%C3%96%C2%B6%C2%ADO%089%C3%8D%C3%A7s%C3%AD%C2%BF%C3%94_%C3%90%C2%BE%C3%812%60%5Bi%C3%82%00%00%00%00IEND%C2%AEB%60%C2%82'),
            LtoR: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%3BIDATH%C2%89%C3%AD%C2%97%3Fj%C3%830%14%C3%86%C2%BF%C2%88%20%C2%A4I%06%C3%A3%C3%8D%C2%A4G%C2%88O%60%C3%B0i%029%C2%82%C3%B0%C3%AC%23%C3%B8%00Yu%C2%88%C2%AC%1E%7D%C2%86%C3%A0-%10%C2%ACI%C2%B2%C2%86%C2%B4K%5Dh%C2%A3%C2%84%C2%A4%C3%85J%5B%C3%BCmz%7C%C3%B0%7B%7F%C2%84%C3%84%5BdY%C3%B6%C3%82%18k)%C2%A5%11%02%C3%899%C3%97%5Bk%C2%B3%25c%C2%AC%05%10%1D%C2%8FG%18c%26%07s%C3%8E!%C2%84%C2%88%18c-%C2%A1%C2%94FZ%C3%AB%20%60%000%C3%86%40k%0DJiD%C3%86%40H%0D%C3%83%00%00%20A%C2%A9%C3%AF%3A%C2%9F%C3%8F%C3%8F%C2%83%C2%8F%C2%9A%C3%A13%C3%BC%C3%AF%C3%83%C2%A5%C2%94%C2%88%C3%A3%C3%B89%C3%B04M!%C2%A5%C3%84z%C2%BD%C2%BE%C3%A9%5B%C3%BA%C2%82u%5D_%C3%84%C2%B6%C3%9B%C3%ADC%09p%C3%8E%C2%B1%C3%99l%C2%B0%C3%9F%C3%AF%C2%A1%C2%94%C3%B2z%26%C2%9FyQ%14W%C3%87%C2%B0%C3%88%C3%B3%C3%BC%C3%B5p8%7C%0A%C3%BA*%C3%BF%C2%A9N%C2%A7%13%C3%8A%C2%B2%C3%BC8%C2%AFV%2B%7F%C3%9B%7Dz%C2%A4%C3%AD_%C2%93%C3%AF%C2%BA%0E%C2%BB%C3%9D%C3%AE%C3%82w7%C3%BC%C2%BBj%C2%9A%06J)%C3%AF%C3%8F%C3%A9%C2%85%3Fz%C2%B9%7C2%C3%86%40)%C2%85%C2%A6i%C2%AEz%26%C2%A9%7Cls%C3%97u7%7D%C2%93%C3%80%C2%AB%C2%AA%C2%BA%C3%8B%C3%B7%C3%BF%C2%9E%C3%97%19%3E%C3%83%7F%2F%C2%9Cs%1E%14%3A%C3%B2%C2%88s%C2%AE%17B%04I%C2%80%102%C3%AEjp%C3%8E%C3%B5Kkm%C3%86%18k%C2%93%24%09%C2%BE%C2%A5%C2%BE%013%C2%83r%C3%A8%C3%A39%C3%8D%C2%81%00%00%00%00IEND%C2%AEB%60%C2%82'),
            TtoB: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%017IDATH%C2%89%C3%AD%C2%971%C2%8A%C2%840%18%C2%85%C3%9F%04%09%C2%B1%C3%92%C3%86N%C3%9C%2Bx%C2%829%C2%90G%C3%B0%00%1E!%C2%906%C2%AD%C2%87%C3%B0%0Az%05%C2%B1%13%C3%85T%C3%91%14%C3%AE%16%C2%B3%C3%82%0C%C2%BB%0B%C2%9B%C3%80d%60%C3%97W%C3%BE!%C3%AF%C3%8B%C2%9F%3C%C2%A2%C2%B9%C3%A4y%C3%BE%C3%86%18k)%C2%A51%3C%C3%89%18%C2%B3%C2%AC%C3%AB%C2%9A%07%C2%8C%C2%B1%16%40%3C%C2%8E%23%C2%B4%C3%96O%07%C2%87a%C2%88(%C2%8Ab%C3%86XK(%C2%A5%C2%B1R%C3%8A%0B%18%00%C2%B4%C3%96PJ%C2%81R%1A%C2%93%C2%A3%C3%A0S%C3%9B%C2%B6%01%00%C2%88W%C3%AA%C2%A7%C3%B6%7D%07%00%04.%C2%939%C3%A7_jEQX%C3%BB%C2%BC%C2%A4%C3%B3%13%C3%BER%C2%B8S%C3%A0%C2%8Epq%C3%8E%C2%9D%C2%82v%C3%A8%C3%BFn%C3%BB%09%3F%C3%A1\'%C3%BC%C3%AF%C3%83%C2%AD%C2%AE%C3%97%C2%B2%2C%C2%91%C2%A6%C3%A9C%C3%AD%C3%BE%C3%9B%3E%0C%03%C2%AA%C2%AA%C3%BA%C2%B5%C2%9FU%C3%A7B%C2%88%1F%7F%C2%B9%C2%B4%C3%96%10B%C3%98%C3%98%C3%99%C3%81%C2%A7i%C2%82%C2%94%C3%B2%C3%9B1)%25%C2%A6iz%1E%1C%00%C2%BA%C2%AEC%C3%934%0F%C2%B5%C2%A6i%C3%90u%C2%9D%C2%AD%C2%95%5B%C3%A0%C3%AA%C2%BA%C3%860%0C%00n%C3%A7%5C%C3%97%C2%B5%C2%8B%C2%8D%7B%C3%9A%C2%85%10%C2%98%C3%A7%C3%99%C3%BA%C2%9C%C3%AFu%C2%B9%5E%C2%AF%C3%AF%7D%C3%9F%3B%1B%C2%B8*%C3%8B%C2%B2%5B%C3%A7a%18z%05%1F%3Cb%C2%8CY%C2%A2(%C3%B2%C2%B2%00B%C3%88%C3%B1V%C2%831f%09%C3%96u%C3%8D%19cm%C2%92%24%C3%9E_%C2%A9%1F%C3%B5tx%C2%AC%C2%BB%1B%C3%B2%1C%00%00%00%00IEND%C2%AEB%60%C2%82'),
            rowRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%BAIDATH%C2%89%C3%8D%C2%97!%C2%AF%C3%AA0%18%C2%86%C3%9F%C2%B3%C3%8B%C2%96N%C2%95%C3%90%C3%A0%C3%88pX%10%C2%88aH%C3%90%C3%A8%C3%99Z%12%14%C2%B3%C2%A8%C3%BB%0F%C3%A6%08z%3Fd%063%09%06%C3%81%04%09%C2%99%23M%18%C2%A6c!%C3%9Ck%C3%AE%C3%88%C2%B9%17%0E%C2%A7%C2%B0%C3%9C%C2%9D%C3%B3%C3%88%C2%AE%C3%AD%C3%93%C3%AFk%C3%BBe%7D%C3%ABt%3AMB%C3%88%C3%920%C2%8C*J%22%C3%8B%C2%B2C%C2%9A%C2%A6%C2%9D%0A!d%09%C2%A0%C2%BA%C3%9F%C3%AF!%C2%A5%C3%BC%C3%AFb%C3%934A)%C2%AD%12B%C2%96%C2%9Aa%18%C3%95%24IJ%11%03%C2%80%C2%94%12I%C2%92%C3%800%C2%8C%C2%AA%C2%967%C2%94%C3%89%C3%A9t%02%00h%C2%A5Z%C3%BFp%C2%B9%5C%C2%BEN%C2%9E%C3%B3%C2%A5%C3%B2%C3%8A%C2%BD%C3%86%C3%99lv%C3%936%1E%C2%8F%C2%95\'U%1D%C3%BF%C3%BD%22%C2%BF%C3%87%C2%BDhJ%C2%93%17M%C3%BB%3D%C2%94%C3%93n%C3%9B%C2%B6%C2%B2%5C%C2%95%C2%B7~%C2%BF%C3%BFk%C2%B7%C3%9B%3D%C3%AC%C3%94j%C2%B5%C3%809%C2%87%10%02%C2%BE%C3%AFC%08QXlY%16~4%C2%9B%C3%8D%C2%9FI%C2%92%3C%C3%AC(%C2%84%40%10%04%60%C2%8C%C2%81s%C2%8E%C3%B3%C3%B9%C2%8C%C3%ADv%5BHN)U%C2%93%C3%A7DQ%C2%84%C3%B5z%C2%8D%C3%A1p%C2%88n%C2%B7%C2%8B(%C2%8A%5E.%C3%8DO%C3%8B%01%C3%A0x%3Cb%C2%B1X%C2%801%C2%86%C3%91ht%5D%C3%94%2Br%C2%A5%3D%C3%BF%C2%88%7C%1BL%C3%93%C2%84%C3%AF%C3%BB%C2%88%C3%A3Xy%C2%AC%C3%92%C2%9E3%C3%86%C3%A0%C2%BA.%16%C2%8B%C3%85%C3%8D7)%25%C3%820%C2%84%C2%AE%C3%AB%C3%A0%C2%9CC%C3%97u%C3%A5%2CPJ%1F%C3%9F%C3%B3v%C2%BB%7D%C2%8D%C3%AC%11A%10%60%C2%B5Z%C2%81s%C2%8E%5E%C2%AF%07%C3%9F%C3%B7%C2%B1%C3%99l%3E%5D%C3%80%C2%87r%C3%87q0%18%0C%3E%C2%9D%20G%08%01%C3%8F%C3%B3%60%C3%9B6%26%C2%93%C3%89%C3%8Dw%C2%A5%C3%9A%C3%8E%18%C3%83t%3A%7DJ%C3%BC%C2%9E0%0C%C2%95%C3%BB%C3%9ED%C3%AE%C2%BA.j%C2%B5%C3%9AM%C3%87Rj%C3%BB%7C%3E%07%C3%A7%1C%C2%8DF%C3%A3%C2%AF%C3%B6Rj%7B%1C%C3%87%C3%B0%3C%C3%AF%C2%A9%C3%B4%C2%BD%C3%8A%C3%9D%03\'%C2%A5%C2%BC%C2%9EX%C3%87q%3E%3D%C3%AD%C3%BF%C2%A2%C2%9A%C2%A5%C2%87W-%0CC%C3%84q%0C%C3%8E%C3%B9SrU%0AU%C2%B8%22X%C2%96%C3%B5%0D%C3%BE%5E%C2%9F%C3%9D%C3%93%C2%A2%C3%A4%3E-%C3%8B%C2%B2%03%C2%A5%C2%B4%C2%94%05h%C2%9A%C2%96%C2%BF%C3%95%C2%90e%C3%99%C2%A1%C2%92%C2%A6i%C2%87%10%C2%B2%C2%AC%C3%97%C3%AB%C2%A5%C2%BFR%7F%03I*%C3%99%C3%B1.%C2%86i%C2%BE%00%00%00%00IEND%C2%AEB%60%C2%82'),
            columnRL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%C3%80IDATH%C2%89%C3%AD%C2%97%C2%A1%C2%8F%C2%830%18%C3%85%C3%9F%C2%91%C2%85%C2%B4%0A%C2%92fn%C3%9994%125%C2%81e%7F%C3%83%C3%BE%00%C3%94f%C2%97%C3%A9%C3%999%12l-%C2%BA%16%C3%AC%C3%A4%C2%B0%C2%93%0Bni2T%19bw%C3%A2n%C2%97%5C6%C3%88%C2%B8q%3DsO~%C3%B9%C2%9A_%C3%BBZ%1E%C3%AD%C2%8B%C3%AB%C2%BA%C2%AF%C2%84%C2%90%C2%9Di%C2%9A64%C2%A9%C2%AE%C3%ABSUU%C3%AE%C2%80%10%C2%B2%03%60%1F%C2%8FG(%C2%A5~%1DL)%C2%85eY6!dg%C2%98%C2%A6i%C2%97e%C2%A9%05%0C%00J)%C2%94e%09%C3%934m%C3%A3Z%C3%90%C2%A9%C3%B3%C3%B9%0C%000%C2%B4R%3Fu%C2%B9%5C%00%00%C2%83%C2%B6%C2%A6(%C2%8Anja%18%C3%B6%C3%96%C3%BF\'%2B%C3%BF%C2%87w%C2%86%3B%C2%8E%C3%93%1B%C3%BCe2%C2%99%C2%BC%1D%0E%C2%87%C3%96%C2%A6(%C2%8A%10%C2%86!%3C%C3%8F%C3%83t%3A%C3%85j%C2%B5z%C2%A8%C2%BFM%C3%A3%C3%B1%C2%B8%C3%9B%C3%8A%C2%B7%C3%9B-%C2%A4%C2%94%08%C2%82%C2%A0%C3%8B%C2%B0Fu%C2%B6%C2%9Ds%0E%C3%9F%C3%B71%1A%C2%8D%C3%B4%C3%83%C2%A5%C2%94%10B%606%C2%9B%C3%A9%C2%87%03%40%C2%9A%C2%A6PJ%3Dm%C3%BF%C2%8F%3F5%C3%8E9%C2%82%20%00cL%3F%C2%BC%0F%C3%BB%C2%9F%0A%19!%04(%C2%A5%C3%B0%7D_%3F%1Cx%C3%8E%C3%BE%C2%A7%C3%A1EQ%20M%C3%93%1F%C3%99%C3%9FK%C2%B6%0B!%C3%80%18%C2%83%C3%A7y%C2%9D%C3%86%C3%9D%C2%8D%C3%97%C3%A5r%C3%99%1A%22EQ%60%C2%BD%5E%7F%C2%AB9%C2%8E%C2%83%C3%B9%7C~%C3%93%C3%9B%14%C2%B3%C2%8D%C3%B1%1A%C3%87q%C3%A3%C3%95J)%C2%858%C2%8Eo%C3%AA%C3%BB%C3%BD%C2%BEq%C2%B2M%C2%BA%0B%C2%97R%C2%82s~w%00%C3%A7%1CR%C3%8A%C3%8E%C2%A0%C2%87%C3%A1%00%C2%90%C3%A79%C2%B2%2C%C3%BBV%C3%8B%C2%B2%0Cy%C2%9E%C3%B7%02n%C2%85%03%40%C2%92%24(%C2%8A%02%C3%80%C3%87%3E\'I%C3%92%1B%18x%C3%A0%7F%C3%8E%18%C3%83b%C2%B1%C3%80f%C2%B3%C3%A9%C3%8Dn%C3%A0%C3%A3%C3%80%3Dt%C2%99%C3%B8%0D%7D%C2%9DvJ%C2%A9V%C3%B0%C2%95g%C3%94u%7D%C2%B2%2CK%C3%8B%04%0C%C3%83%C2%B8%C2%BE%C3%95P%C3%97%C3%B5iPU%C2%95K%08%C3%99%0D%C2%87C%C3%AD%C2%AF%C3%94wP%C3%95%C2%BFp%C2%ACF%C2%BF%C3%B5%00%00%00%00IEND%C2%AEB%60%C2%82'),
            RtoL: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%3DIDATH%C2%89%C3%AD%C3%97%C2%B1%C2%8A%C2%830%1C%06%C3%B0%C2%AFAB%C2%9C%14%C2%83%C2%9B%C3%B4%1E%C2%A1%3E%C2%81%C3%A0C%C2%B8f%15%C3%9C%C2%BA%C3%B6%25%C3%9C%C2%8A%C2%B3O%C3%A3%C3%A8%C3%ACX%C3%9C%5Ct%C2%8Af%C3%B0n9\'%C3%9B%C2%9E%C2%963W%0E%C2%BF1%C2%86%C3%BC%C2%92%C3%B0O%24%07%C3%9F%C3%B7%3F%18c%25%C2%A5%C3%94%C2%86%C2%A6(%C2%A5%C3%9A%C2%BE%C3%AF%7D%C2%831V%02%C2%B0%C2%9B%C2%A6%C2%81%C2%94rs%C3%984MX%C2%96e3%C3%86JB)%C2%B5%C2%BB%C2%AE%C3%93%02%03%C2%80%C2%94%12%5D%C3%97%C2%81Rj%C2%93%C2%A9Ag%C2%86a%00%00%10%C2%AD%C3%AAw%C3%86q%C3%BC%3B%7C%C3%8A%C2%8E%C3%AF%C3%B8%7B%C3%A1%C2%9Cs%5C.%C2%97Mp%C3%A3%C3%99%C3%87%C3%93%C3%A9%04!%04L%C3%93%C3%94%C2%8BGQ%C2%840%0C_%1A%C3%B4z%C2%BD%C3%8E%C3%9A%C2%92%24%C3%B9%19%C3%A7%C2%9C%23%C2%8Ecx%C2%9E%C3%B7%12%C2%BC%263%C3%BC%7C%3E%C3%83q%C2%9CY%C3%87%7B%C2%AB%C3%B9u%3C%C3%8B2%08!f%2B%C2%BF%C2%B7m%C2%8F%C2%B2t%C2%A2%C2%B3j%C2%AF%C3%AB%1Ai%C2%9A%C2%A2(%C2%8A%C3%85%C3%98%C2%AB%C2%B9%5BpRJ%C3%A4y%C2%8E%C2%AA%C2%AA%10E%C3%91%C3%AAj_%C2%BAKO%C2%8FZQ%14%C2%A8%C3%AB%1AB%C2%88U%C3%B8%C3%92%1C%C2%82%20%C3%B8%C2%BC%C3%9Dn%C2%9B%0C%C3%BE%2C%C3%87%C3%A3%C3%B1%C3%8D%C2%AF%C3%97%1D%C3%9F%C3%B1%C3%BF%C2%85o%C3%B5%C2%BF~%C2%94%C3%89%23J%C2%A9%C3%96%C2%B2%2C-%13%20%C2%84Lo5(%C2%A5Z%C2%A3%C3%AF%7B%C2%9F1V%C2%BA%C2%AE%C2%AB%C3%BD%C2%95%C3%BA%05%C2%AF%C3%99g%C3%8E%C3%A8\'e%C2%80%00%00%00%00IEND%C2%AEB%60%C2%82'),
            BtoT: File.decode('%C2%89PNG%0D%0A%1A%0A%00%00%00%0DIHDR%00%00%00%1F%00%00%00%1F%08%06%00%00%00%1F%C2%AE%169%00%00%00%09pHYs%00%00%0B%12%00%00%0B%12%01%C3%92%C3%9D~%C3%BC%00%00%01%1DIDATH%C2%89%C3%AD%C2%971%C2%8E%C2%84%20%14%C2%86%C3%BF%25%C2%86%40%C2%85%C2%91%C3%98%C2%91%C3%AC%11%C3%B0%0A%1C%C2%82%C2%96%C2%96%C3%84s%18%C2%8F%C3%A0%C2%B9%C2%BC%C2%87%C2%8DV2%14f*%C2%93%C2%99%C3%ACl%01%C2%B3%C2%BA%C3%85%C3%B8W%C3%B0%C2%80%C3%B7%11%C3%9E%0B%C3%B0%C2%BE%C2%9A%C2%A6%C3%B9f%C2%8C%C2%8D%C2%94%C3%92%12\')%C3%868%C2%87%10%C2%9A%C2%8216%02(%C2%A7i%C3%82%C2%BA%C2%AE%C2%87%C2%839%C3%A7%10B%C2%94%C2%8C%C2%B1%C2%91PJ%C3%8BeYN%01%03%C3%80%C2%BA%C2%AEX%C2%96%05%C2%94%C3%92%C2%92%C3%AC%C2%863u%C2%BB%C3%9D%00%00%24%C3%97%C2%81%C2%94%12%5D%C3%97AJ%C2%99%C2%BCv%C3%9B%C2%B6%C3%B7%C3%A0%C3%9E%7BTU%05%C3%AF%7D%C2%AE%C2%8B%3C%C2%B8%C2%B5%16J)%00%C2%80R%0A%C3%96%C3%9As%C3%A0Zk%18c%C2%9El%C3%86%18h%C2%AD%C2%8F%C2%85K)%C3%A1%C2%9C%7B9%C3%A6%C2%9CK%C2%8E%7F%12%C3%9C%7B%0F%C3%8E%C3%B9%C3%8B1%C3%8Eyr%C3%BC%C2%8B%C2%94%C3%89%7D%C3%9F%3F%C3%B5%C2%87a%40%C3%9B%C2%B6I%C3%80Geg%C3%BB_%C3%A8%C2%82_%C3%B0%0B~%C3%81%0FU%C3%92%C3%B5%C2%BAk%18%C2%86%1F%C3%AD%C2%9Ck%C3%B6s%C2%8F%C3%BDs%C3%A1Y%09%C3%B7%C3%8E%1B%C3%BE%C2%A8%C3%BF%3F%C3%B6%C3%9F%C2%BEFGi%C3%A7%C2%91%18%C3%A3%2C%C2%848e%03%C2%84%C2%90%C2%BDVC%C2%8Cq.B%08%0Dcl%C2%AC%C3%AB%C3%BA%C3%B4*%C3%B5%0E%C3%82MR%C3%9C%C2%AC%C2%B7%C3%9D%C2%93%00%00%00%00IEND%C2%AEB%60%C2%82')
        }
    };

    var pref = app.preferences;
    var brightness = pref.getRealPreference('uiBrightness');

    if (brightness == 1) return lightest;
    if (0.5 < brightness && brightness < 1) return light;
    if (0 < brightness && brightness <= 0.5) return dark;
    if (brightness == 0) return darkest;
}


function localizeUI() {
    return {
        title: {
            en: 'Create Threaded Text',
            ja: 'スレッドテキストを作成'
        },
        order: {
            en: 'Order of Concatenation',
            ja: '連結順'
        },
        row: {
            en: 'Grid by Row (L → R)',
            ja: '横に配列（左 → 右）'
        },
        column: {
            en: 'Grid by Column (L → R)',
            ja: '縦に配列（左 → 右）'
        },
        rowRtoL: {
            en: 'Grid by Row (R → L)',
            ja: '横に配列（右 → 左）'
        },
        columnRtoL: {
            en: 'Grid by Column (R → L)',
            ja: '縦に配列（右 → 左）'
        },
        left: {
            en: 'Arrange by Row (L → R)',
            ja: '横一列（左 → 右）'
        },
        right: {
            en: 'Arrange by Row (R → L)',
            ja: '横一列（右 → 左）'
        },
        top: {
            en: 'Arrange by Column (T → B)',
            ja: '縦一列（上 → 下）'
        },
        bottom: {
            en: 'Arrange by Column (B → T)',
            ja: '縦一列（下 → 上）'
        },
        option: {
            en: 'Options',
            ja: 'オプション'
        },
        tolerance: {
            en: 'Alignment Position Tolerance:',
            ja: '整列位置の許容誤差:'
        },
        shape: {
            en: 'Convert Path to Type',
            ja: 'パスをテキストに変換'
        },
        area: {
            en: 'Area Type',
            ja: 'エリア内文字'
        },
        path: {
            en: 'Type on a Path',
            ja: 'パス上文字'
        },
        cancel: {
            en: 'Cancel',
            ja: 'キャンセル'
        },
        ok: {
            en: 'OK',
            ja: 'OK'
        }
    };
}
