#include <karm-base/array.h>
#include <karm-crypto/sha2.h>
#include <karm-test/macros.h>

namespace Karm::Crypto::Tests {

test$("crypto-sha224") {
    static constexpr Array<u8, SHA224_BYTES> EXPECTED_EMPTY = {
        0xd1, 0x4a, 0x02, 0x8c, 0x2a, 0x3a, 0x2b, 0xc9,
        0x47, 0x61, 0x02, 0xbb, 0x28, 0x82, 0x34, 0xc4,
        0x15, 0xa2, 0xb0, 0x1f, 0x82, 0x8e, 0xa6, 0x2a,
        0xc5, 0xb3, 0xe4, 0x2f
    };

    auto testCase = [&](Str data, Array<u8, SHA224_BYTES> const &expected) -> Res<> {
        auto sha = sha224(bytes(data));

        expectEq$(sha.len(), expected.len());
        for (usize idx = 0; idx < sha.len(); idx++) {
            expectEq$(sha[idx], expected[idx]);
        }
        return Ok();
    };

    try$(testCase("", EXPECTED_EMPTY));

    return Ok();
}

test$("crypto-sha256") {
    static constexpr Array<u8, SHA256_BYTES> EXPECTED_EMPTY = {
        0xe3, 0xb0, 0xc4, 0x42, 0x98, 0xfc, 0x1c, 0x14,
        0x9a, 0xfb, 0xf4, 0xc8, 0x99, 0x6f, 0xb9, 0x24,
        0x27, 0xae, 0x41, 0xe4, 0x64, 0x9b, 0x93, 0x4c,
        0xa4, 0x95, 0x99, 0x1b, 0x78, 0x52, 0xb8, 0x55
    };

    static constexpr Array<u8, SHA256_BYTES> EXPECTED_NIST_1 = {
        0xba, 0x78, 0x16, 0xbf, 0x8f, 0x01, 0xcf, 0xea,
        0x41, 0x41, 0x40, 0xde, 0x5d, 0xae, 0x22, 0x23,
        0xb0, 0x03, 0x61, 0xa3, 0x96, 0x17, 0x7a, 0x9c,
        0xb4, 0x10, 0xff, 0x61, 0xf2, 0x00, 0x15, 0xad
    };

    static constexpr Array<u8, SHA256_BYTES> EXPECTED_NIST_3 = {
        0xcd, 0xc7, 0x6e, 0x5c, 0x99, 0x14, 0xfb, 0x92,
        0x81, 0xa1, 0xc7, 0xe2, 0x84, 0xd7, 0x3e, 0x67,
        0xf1, 0x80, 0x9a, 0x48, 0xa4, 0x97, 0x20, 0x0e,
        0x04, 0x6d, 0x39, 0xcc, 0xc7, 0x11, 0x2c, 0xd0
    };

    auto testCase = [&](Str data, Array<u8, SHA256_BYTES> const &expected) -> Res<> {
        auto sha = sha256(bytes(data));

        expectEq$(sha.len(), expected.len());
        for (usize idx = 0; idx < sha.len(); idx++) {
            expectEq$(sha[idx], expected[idx]);
        }
        return Ok();
    };

    try$(testCase("", EXPECTED_EMPTY));
    try$(testCase("abc", EXPECTED_NIST_1));

    Array<u8, 1000000> nist3;

    for (usize idx = 0; idx < nist3.len(); idx++) {
        nist3[idx] = 'a';
    }
    auto sha = sha256(nist3.bytes());
    expectEq$(sha.len(), EXPECTED_NIST_3.len());
    for (usize idx = 0; idx < sha.len(); idx++) {
        expectEq$(sha[idx], EXPECTED_NIST_3[idx]);
    }
    return Ok();
}

test$("crypto-sha384") {
    static constexpr Array<u8, SHA384_BYTES> EXPECTED_EMPTY = {
        0x38, 0xb0, 0x60, 0xa7, 0x51, 0xac, 0x96, 0x38,
        0x4c, 0xd9, 0x32, 0x7e, 0xb1, 0xb1, 0xe3, 0x6a,
        0x21, 0xfd, 0xb7, 0x11, 0x14, 0xbe, 0x07, 0x43,
        0x4c, 0x0c, 0xc7, 0xbf, 0x63, 0xf6, 0xe1, 0xda,
        0x27, 0x4e, 0xde, 0xbf, 0xe7, 0x6f, 0x65, 0xfb,
        0xd5, 0x1a, 0xd2, 0xf1, 0x48, 0x98, 0xb9, 0x5b
    };

    static constexpr Array<u8, SHA384_BYTES> EXPECTED_NIST_2 = {
        0x09, 0x33, 0x0c, 0x33, 0xf7, 0x11, 0x47, 0xe8,
        0x3d, 0x19, 0x2f, 0xc7, 0x82, 0xcd, 0x1b, 0x47,
        0x53, 0x11, 0x1b, 0x17, 0x3b, 0x3b, 0x05, 0xd2,
        0x2f, 0xa0, 0x80, 0x86, 0xe3, 0xb0, 0xf7, 0x12,
        0xfc, 0xc7, 0xc7, 0x1a, 0x55, 0x7e, 0x2d, 0xb9,
        0x66, 0xc3, 0xe9, 0xfa, 0x91, 0x74, 0x60, 0x39
    };

    auto testCase = [&](Str data, Array<u8, SHA384_BYTES> const &expected) -> Res<> {
        auto sha = sha384(bytes(data));

        expectEq$(sha.len(), expected.len());
        for (usize idx = 0; idx < sha.len(); idx++) {
            expectEq$(sha[idx], expected[idx]);
        }
        return Ok();
    };

    try$(testCase("", EXPECTED_EMPTY));
    try$(testCase("abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu", EXPECTED_NIST_2));

    return Ok();
}

test$("crypto-sha512") {
    static constexpr Array<u8, SHA512_BYTES> EXPECTED_EMPTY = {
        0xcf, 0x83, 0xe1, 0x35, 0x7e, 0xef, 0xb8, 0xbd,
        0xf1, 0x54, 0x28, 0x50, 0xd6, 0x6d, 0x80, 0x07,
        0xd6, 0x20, 0xe4, 0x05, 0x0b, 0x57, 0x15, 0xdc,
        0x83, 0xf4, 0xa9, 0x21, 0xd3, 0x6c, 0xe9, 0xce,
        0x47, 0xd0, 0xd1, 0x3c, 0x5d, 0x85, 0xf2, 0xb0,
        0xff, 0x83, 0x18, 0xd2, 0x87, 0x7e, 0xec, 0x2f,
        0x63, 0xb9, 0x31, 0xbd, 0x47, 0x41, 0x7a, 0x81,
        0xa5, 0x38, 0x32, 0x7a, 0xf9, 0x27, 0xda, 0x3e
    };

    static constexpr Array<u8, SHA512_BYTES> EXPECTED_NIST_1 = {
        0xdd, 0xaf, 0x35, 0xa1, 0x93, 0x61, 0x7a, 0xba,
        0xcc, 0x41, 0x73, 0x49, 0xae, 0x20, 0x41, 0x31,
        0x12, 0xe6, 0xfa, 0x4e, 0x89, 0xa9, 0x7e, 0xa2,
        0x0a, 0x9e, 0xee, 0xe6, 0x4b, 0x55, 0xd3, 0x9a,
        0x21, 0x92, 0x99, 0x2a, 0x27, 0x4f, 0xc1, 0xa8,
        0x36, 0xba, 0x3c, 0x23, 0xa3, 0xfe, 0xeb, 0xbd,
        0x45, 0x4d, 0x44, 0x23, 0x64, 0x3c, 0xe8, 0x0e,
        0x2a, 0x9a, 0xc9, 0x4f, 0xa5, 0x4c, 0xa4, 0x9f
    };

    static constexpr Array<u8, SHA512_BYTES> EXPECTED_NIST_2 = {
        0x8e, 0x95, 0x9b, 0x75, 0xda, 0xe3, 0x13, 0xda,
        0x8c, 0xf4, 0xf7, 0x28, 0x14, 0xfc, 0x14, 0x3f,
        0x8f, 0x77, 0x79, 0xc6, 0xeb, 0x9f, 0x7f, 0xa1,
        0x72, 0x99, 0xae, 0xad, 0xb6, 0x88, 0x90, 0x18,
        0x50, 0x1d, 0x28, 0x9e, 0x49, 0x00, 0xf7, 0xe4,
        0x33, 0x1b, 0x99, 0xde, 0xc4, 0xb5, 0x43, 0x3a,
        0xc7, 0xd3, 0x29, 0xee, 0xb6, 0xdd, 0x26, 0x54,
        0x5e, 0x96, 0xe5, 0x5b, 0x87, 0x4b, 0xe9, 0x09
    };

    auto testCase = [&](Str data, Array<u8, SHA512_BYTES> const &expected) -> Res<> {
        auto sha = sha512(bytes(data));

        expectEq$(sha.len(), expected.len());
        for (usize idx = 0; idx < sha.len(); idx++) {
            expectEq$(sha[idx], expected[idx]);
        }
        return Ok();
    };

    try$(testCase("", EXPECTED_EMPTY));
    try$(testCase("abc", EXPECTED_NIST_1));
    try$(testCase("abcdefghbcdefghicdefghijdefghijkefghijklfghijklmghijklmnhijklmnoijklmnopjklmnopqklmnopqrlmnopqrsmnopqrstnopqrstu", EXPECTED_NIST_2));

    return Ok();
}
} // namespace Karm::Crypto::Tests