// I don't use the real useSafeAreaInsets() hook, since
// 1) the SafeAreaProvider forces you to render null on Web until it measures
// 2) you might not need to support it, unless you're doing landscape stuff
// 3) react-native-safe-area-context has a massive import on Web
// see: https://github.com/th3rdwave/react-native-safe-area-context/pull/189#issuecomment-815274313
// 4) most importantly, I think you can just use the env(safe-area-inset-bottom) CSS variable instead
// after all, safe area code is few-and-far-between, so if you have to write some platform-speciifc code for it,
// that is probably better than a massive bundle size for little benefit
import { useWindowDimensions } from "react-native";

import type { useSafeAreaInsets as nativeHook } from "./index";

// on Web, we don't use React Navigation, so we are going to avoid the safe area provider
// instead, we just have a no-op here
// for more, see: https://solito.dev/recipes/tree-shaking

// if you need safe area hooks yourself, you can implement this yourself
// however, you may be better off using the CSS selector for env(safe-area-inset-top) on Web

export const SafeAreaProvider = ({
  children,
}: {
  children: React.ReactElement;
}) => <>{children}</>;

const area = {
  bottom: 0,
  left: 0,
  right: 0,
  top: 0,

  // you could also use CSS env variables like below:
  // but you'll have to be sure to override the types for `useSafeArea`
  // and make sure to never add numbers and strings when you consue useSafeArea
  // just keep in mind that the env() doesn't work on older browsers I think

  // top: `env(safe-area-inset-top)`,
  // right: `env(safe-area-inset-right)`,
  // bottom: `env(safe-area-inset-bottom)`,
  // left: `env(safe-area-inset-left)`,
};

export function useSafeAreaInsets(): ReturnType<typeof nativeHook> {
  return area;
}

export function useSafeAreaFrame(): any {
  const windowDimensions = useWindowDimensions();
  return {
    ...area,
    width: windowDimensions.width,
    height: windowDimensions.height,
  };
}

export function SafeAreaView({ children }: { children: React.ReactElement }) {
  return <>{children}</>;
}
