import { Clothes, ClothesColor, Graphic} from '../avatar.enum';
import { ColorHelper } from './colors';

export class ClothesHelper {

  public static getClothesSvg(clothes: Clothes, color: ClothesColor, graphic: Graphic) {
    switch (clothes) {
      case Clothes.BLAZER_SHIRT:
        return ClothesHelper.getBlazerShirt();
      case Clothes.BLAZER_SWEATER:
        return ClothesHelper.getBlazerSweater();
      case Clothes.COLLAR_SWEATER:
        return ClothesHelper.getCollarSweater(color);
      case Clothes.GRAPHIC_SHIRT:
        return ClothesHelper.getGraphicShirt(color, graphic);
      case Clothes.HOODIE:
        return ClothesHelper.getHoodie(color);
      case Clothes.OVERALL:
        return ClothesHelper.getOveralls(color);
      case Clothes.SHIRT_CREWNECK:
        return ClothesHelper.getCrewNeckShirt(color);
      case Clothes.SHIRT_SCOOPNECK:
        return ClothesHelper.getScoopNeckShirt(color);
      case Clothes.SHIRT_VNECK:
        return ClothesHelper.getVNeckShirt(color);
      default:
        return '';
    }
  }

  private static getBlazerShirt() {
    return `
    <g
    id="Clothing/Blazer-+-Shirt"
    transform="translate(0.000000, 170.000000)">
    <defs>
      <path
        d="M133.960472,0.294916112 C170.936473,3.32499816 200,34.2942856 200,72.0517235 L200,81 L0,81 L0,72.0517235 C1.22536245e-14,33.9525631 29.591985,2.76498122 67.0454063,0.219526408 C67.0152598,0.593114549 67,0.969227185 67,1.34762511 C67,13.2107177 81.9984609,22.8276544 100.5,22.8276544 C119.001539,22.8276544 134,13.2107177 134,1.34762511 C134,0.994669088 133.986723,0.64370138 133.960472,0.294916112 Z"
    id="cloth_path"
      />
      </defs>
      <g id="Shirt" transform="translate(32.000000, 29.000000)">
    <mask id="cloth_mask" fill="white">
    <use href="#path1" />
      </mask>
      <use id="Clothes" fill="#E6E6E6" href="#cloth_path" />
    <g id="Color/Palette/Black" mask="url(#cloth_mask)" fill="#262E33">
    <g transform="translate(-32.000000, -29.000000)" id="pen-Color">
    <rect x="0" y="0" width="264" height="110" />
      </g>
      </g>
      <g
    id="Shadowy"
    opacity="0.599999964"
    mask="url(#cloth_mask)"
    fill-opacity="0.16"
    fill="#000000">
    <g transform="translate(60.000000, -25.000000)" id="Hola">
    <ellipse
      cx="40.5"
    cy="27.8476251"
    rx="39.6351047"
    ry="26.9138272"
      />
      </g>
      </g>
      </g>
      <g id="Blazer" transform="translate(32.000000, 28.000000)">
    <path
      d="M68.784807,1.12222847 C30.512317,2.80409739 -1.89486556e-14,34.3646437 -1.42108547e-14,73.0517235 L0,73.0517235 L0,82 L69.3616767,82 C65.9607412,69.9199941 64,55.7087296 64,40.5 C64,26.1729736 65.7399891,12.7311115 68.784807,1.12222847 Z M131.638323,82 L200,82 L200,73.0517235 C200,34.7067641 170.024954,3.36285166 132.228719,1.17384225 C135.265163,12.7709464 137,26.1942016 137,40.5 C137,55.7087296 135.039259,69.9199941 131.638323,82 Z"
    id="Saco"
    fill="#3A4C5A"
    />
    <path
      d="M149,58 L158.555853,50.83311 L158.555853,50.83311 C159.998897,49.7508275 161.987779,49.7682725 163.411616,50.8757011 L170,56 L149,58 Z"
    id="Pocket-hanky"
    fill="#E6E6E6"
    />
    <path
      d="M69,1.13686838e-13 C65,19.3333333 66.6666667,46.6666667 74,82 L58,82 L44,46 L50,37 L44,31 L63,1 C65.027659,0.369238637 67.027659,0.0359053037 69,1.13686838e-13 Z"
    id="Wing"
    fill="#2F4351"
    />
    <path
      d="M151,1.13686838e-13 C147,19.3333333 148.666667,46.6666667 156,82 L140,82 L126,46 L132,37 L126,31 L145,1 C147.027659,0.369238637 149.027659,0.0359053037 151,1.13686838e-13 Z"
    id="Wing"
    fill="#2F4351"
    transform="translate(141.000000, 41.000000) scale(-1, 1) translate(-141.000000, -41.000000) "
      />
      </g>
      </g>`;
  }

  private static getBlazerSweater() {
    return `
    <g
    id='Clothing/Blazer-+-Sweater'
    transform='translate(0.000000, 170.000000)'>
    <defs>
      <path
        d='M105.192402,29.0517235 L104,29.0517235 L104,29.0517235 C64.235498,29.0517235 32,61.2872215 32,101.051724 L32,110 L232,110 L232,101.051724 C232,61.2872215 199.764502,29.0517235 160,29.0517235 L160,29.0517235 L158.807598,29.0517235 C158.934638,30.0353144 159,31.0364513 159,32.0517235 C159,45.8588423 146.911688,57.0517235 132,57.0517235 C117.088312,57.0517235 105,45.8588423 105,32.0517235 C105,31.0364513 105.065362,30.0353144 105.192402,29.0517235 Z'
    id='blazer_path'
      />
      </defs>
      <mask id="blazer_mask" fill='white'>
    <use href='#blazer_path' />
      </mask>
      <use
    id='Clothes'
    fill='#E6E6E6'
    fill-rule='evenodd'
    href='#blazer_path'
    />
    <g
      id='Color/Palette/Black'
    mask='url(#blazer_mask)'
    fill-rule='evenodd'
    fill='#262E33'>
    <rect id='pen-Color' x='0' y='0' width='264' height='110' />
      </g>
      <g
    id='Blazer'
    stroke-width='1'
    fill-rule='evenodd'
    transform='translate(32.000000, 28.000000)'>
    <path
      d='M68.784807,1.12222847 C30.512317,2.80409739 1.24427139e-14,34.3646437 0,73.0517235 L0,82 L69.3616767,82 C65.9607412,69.9199941 64,55.7087296 64,40.5 C64,26.1729736 65.7399891,12.7311115 68.784807,1.12222847 Z M131.638323,82 L200,82 L200,73.0517235 C200,34.7067641 170.024954,3.36285166 132.228719,1.17384225 C135.265163,12.7709464 137,26.1942016 137,40.5 C137,55.7087296 135.039259,69.9199941 131.638323,82 Z'
    id='Saco'
    fill='#3A4C5A'
    />
    <path
      d='M149,58 L158.555853,50.83311 L158.555853,50.83311 C159.998897,49.7508275 161.987779,49.7682725 163.411616,50.8757011 L170,56 L149,58 Z'
    id='Pocket-hanky'
    fill='#E6E6E6'
    />
    <path
      d='M69,1.13686838e-13 C65,19.3333333 66.6666667,46.6666667 74,82 L58,82 L44,46 L50,37 L44,31 L63,1 C65.027659,0.369238637 67.027659,0.0359053037 69,1.13686838e-13 Z'
    id='Wing'
    fill='#2F4351'
    />
    <path
      d='M151,1.13686838e-13 C147,19.3333333 148.666667,46.6666667 156,82 L140,82 L126,46 L132,37 L126,31 L145,1 C147.027659,0.369238637 149.027659,0.0359053037 151,1.13686838e-13 Z'
    id='Wing'
    fill='#2F4351'
    transform='translate(141.000000, 41.000000) scale(-1, 1) translate(-141.000000, -41.000000) '
      />
      </g>
      <path
    d='M156,21.5390062 C162.772319,26.1359565 167,32.6563196 167,39.8878801 C167,47.2887711 162.572015,53.9447688 155.520105,58.5564942 L149.57933,53.8764929 L145,54.207887 L146,51.0567821 L145.922229,50.995516 C152.022491,47.8530505 156,42.7003578 156,36.8768102 L156,21.5390062 Z M108,21.5390062 C101.227681,26.1359565 97,32.6563196 97,39.8878801 C97,47.2887711 101.427985,53.9447688 108.479895,58.5564942 L114.42067,53.8764929 L119,54.207887 L118,51.0567821 L118.077771,50.995516 C111.977509,47.8530505 108,42.7003578 108,36.8768102 L108,21.5390062 Z'
    id='Collar'
    fill='#F2F2F2'
    fill-rule='evenodd'
      />
      </g>`;
  }

  private static getCollarSweater(color: ClothesColor) {
    return `
    <g
    id='Clothing/Collar-+-Sweater'
    transform='translate(0.000000, 170.000000)'>
    <defs>
      <path
        d='M105.192402,29.0517235 L104,29.0517235 L104,29.0517235 C64.235498,29.0517235 32,61.2872215 32,101.051724 L32,110 L232,110 L232,101.051724 C232,61.2872215 199.764502,29.0517235 160,29.0517235 L160,29.0517235 L158.807598,29.0517235 C158.934638,30.0353144 159,31.0364513 159,32.0517235 C159,45.8588423 146.911688,57.0517235 132,57.0517235 C117.088312,57.0517235 105,45.8588423 105,32.0517235 C105,31.0364513 105.065362,30.0353144 105.192402,29.0517235 Z'
    id="collar_path"
      />
      </defs>
      <mask id="collar_mask" fill='white'>
    <use href='#collar_path'/>
      </mask>
      <use
    id='Clothes'
    fill='#E6E6E6'
    fill-rule='evenodd'
    href='#collar_path'
      />${ ColorHelper.getClothesColor(color, 'collar_mask')}<path
          d='M156,22.2794906 C162.181647,26.8351858 166,33.1057265 166,40.027915 C166,47.2334941 161.862605,53.7329769 155.228997,58.3271669 L149.57933,53.8764929 L145,54.207887 L146,51.0567821 L145.922229,50.995516 C152.022491,47.8530505 156,42.7003578 156,36.8768102 L156,22.2794906 Z M108,21.5714994 C101.232748,26.1740081 97,32.7397769 97,40.027915 C97,47.4261549 101.361602,54.080035 108.308428,58.6915723 L114.42067,53.8764929 L119,54.207887 L118,51.0567821 L118.077771,50.995516 C111.977509,47.8530505 108,42.7003578 108,36.8768102 L108,21.5714994 Z'
          id='Collar'
          fill='#F2F2F2'
          fill-rule='evenodd'
        />
      </g>`;
  }

  private static getGraphicShirt(color: ClothesColor, g: Graphic) {
    return `
    <g
    id='Clothing/Graphic-Shirt'
    transform='translate(0.000000, 170.000000)'>
    <defs>
      <path
        d='M165.624032,29.2681342 C202.760022,32.1373245 232,63.1798426 232,101.051724 L232,110 L32,110 L32,101.051724 C32,62.8348009 61.7752018,31.5722494 99.3929298,29.1967444 C99.1342224,30.2735458 99,31.3767131 99,32.5 C99,44.3741221 113.998461,54 132.5,54 C151.001539,54 166,44.3741221 166,32.5 C166,31.4015235 165.871641,30.3222877 165.624025,29.2681336 Z'
    id="shirt_path"
      />
      </defs>
      <mask id="shirt_mask" fill='white'>
    <use href='#shirt_path' />
      </mask>
      <use
    id='Clothes'
    fill='#E6E6E6'
    fill-rule='evenodd'
    href='#shirt_path'
      />${ ColorHelper.getClothesColor(color, 'shirt_mask')}${this._getgraphic(g, 'shirt_mask')}</g>`;
  }

  private static getHoodie(color: ClothesColor) {
    return `
    <g id='Clothing/Hoodie' transform='translate(0.000000, 170.000000)'>
    <defs>
      <path
        d='M108,13.0708856 C90.0813006,15.075938 76.2798424,20.5518341 76.004203,34.6449676 C50.1464329,45.5680933 32,71.1646257 32,100.999485 L32,100.999485 L32,110 L232,110 L232,100.999485 C232,71.1646257 213.853567,45.5680933 187.995797,34.6449832 C187.720158,20.5518341 173.918699,15.075938 156,13.0708856 L156,32 L156,32 C156,45.254834 145.254834,56 132,56 L132,56 C118.745166,56 108,45.254834 108,32 L108,13.0708856 Z'
    id="hoodie_path"
      />
      </defs>
      <mask id="hoodie_mask" fill='white'>
    <use href='#hoodie_path' />
      </mask>
      <use
    id='Hoodie'
    fill='#B7C1DB'
    fill-rule='evenodd'
    href='#hoodie_path'
      />${ColorHelper.getClothesColor(color, 'hoodie_mask')}<path
          d='M102,61.7390531 L102,110 L95,110 L95,58.1502625 C97.2037542,59.4600576 99.5467694,60.6607878 102,61.7390531 Z M169,58.1502625 L169,98.5 C169,100.432997 167.432997,102 165.5,102 C163.567003,102 162,100.432997 162,98.5 L162,61.7390531 C164.453231,60.6607878 166.796246,59.4600576 169,58.1502625 Z'
          id='Straps'
          fill='#F4F4F4'
          fill-rule='evenodd'
          mask='url(#hoodie_mask)'
        />
        <path
          d='M90.9601329,12.7243537 C75.9093095,15.5711782 65.5,21.2428847 65.5,32.3076923 C65.5,52.0200095 98.5376807,68 132,68 C165.462319,68 198.5,52.0200095 198.5,32.3076923 C198.5,21.2428847 188.09069,15.5711782 173.039867,12.7243537 C182.124921,16.0744598 188,21.7060546 188,31.0769231 C188,51.4689754 160.178795,68 132,68 C103.821205,68 76,51.4689754 76,31.0769231 C76,21.7060546 81.8750795,16.0744598 90.9601329,12.7243537 Z'
          id='Shadow'
          fill-opacity='0.16'
          fill='#000000'
          fill-rule='evenodd'
          mask='url(#hoodie_mask)'
        />
      </g>`;
  }

  private static getOveralls(color: ClothesColor) {
    return `
    <g id='Clothing/Overall' transform='translate(0.000000, 170.000000)'>
    <defs>
      <path
        d='M94,29.6883435 L94,74 L170,74 L170,29.6883435 C179.362956,30.9893126 188.149952,34.0907916 196.00002,38.6318143 L196,110 L187,110 L77,110 L68,110 L68,38.6318027 C75.8500482,34.0907916 84.6370437,30.9893126 94,29.6883435 Z'
    id="overall_path"
      />
      </defs>
      <mask id="overall_mask" fill='white'>
    <use href='#overall_path' />
      </mask>
      <use
    id='Overall'
    fill='#B7C1DB'
    fill-rule='evenodd'
    href='#overall_path'
      />${ColorHelper.getClothesColor(color, 'overall_mask')}<circle
          id='Button'
          fill='#F4F4F4'
          fill-rule='evenodd'
          cx='81'
          cy='83'
          r='5'
        />
        <circle
          id='Button'
          fill='#F4F4F4'
          fill-rule='evenodd'
          cx='183'
          cy='83'
          r='5'
        />
      </g>`;
  }

  private static getCrewNeckShirt(color: ClothesColor) {
    return `
    <g
    id='Clothing/Shirt-Crew-Neck'
    transform='translate(0.000000, 170.000000)'>
    <defs>
      <path
        d='M165.960472,29.2949161 C202.936473,32.3249982 232,63.2942856 232,101.051724 L232,110 L32,110 L32,101.051724 C32,62.9525631 61.591985,31.7649812 99.0454063,29.2195264 C99.0152598,29.5931145 99,29.9692272 99,30.3476251 C99,42.2107177 113.998461,51.8276544 132.5,51.8276544 C151.001539,51.8276544 166,42.2107177 166,30.3476251 C166,29.9946691 165.986723,29.6437014 165.960472,29.2949161 Z'
    id="crew_path"
      />
      </defs>
      <mask id="crew_mask" fill='white'>
    <use href='#crew_path' />
      </mask>
      <use
    id='Clothes'
    fill='#E6E6E6'
    fill-rule='evenodd'
    href='#crew_path'
      />${ColorHelper.getClothesColor(color, 'crew_mask') }<g
          id='Shadowy'
          opacity='0.599999964'
          stroke-width='1'
          fill-rule='evenodd'
          mask='url(#crew_path)'
          fill-opacity='0.16'
          fill='#000000'>
          <g transform='translate(92.000000, 4.000000)' id='Hola'>
            <ellipse
              cx='40.5'
              cy='27.8476251'
              rx='39.6351047'
              ry='26.9138272'
            />
          </g>
        </g>
      </g>`;
  }

  private static getScoopNeckShirt(color: ClothesColor) {
    return `
    <g
    id='Clothing/Shirt-Scoop-Neck'
    transform='translate(0.000000, 170.000000)'>
    <defs>
      <path
        d='M181.544255,32.3304167 C210.784209,41.4878872 232,68.7921987 232,101.051724 L232,110 L32,110 L32,101.051724 C32,68.3969699 53.7388273,40.8195914 83.5340368,32.0020332 C83.182234,33.4201865 83,34.8712315 83,36.3476251 C83,52.6289957 105.161905,65.8276544 132.5,65.8276544 C159.838095,65.8276544 182,52.6289957 182,36.3476251 C182,34.9849859 181.844766,33.6439396 181.544255,32.3304167 Z'
    id="scoop_path"
      />
      </defs>
      <mask id="scoop_mask" fill='white'>
    <use href='#scoop_path' />
      </mask>
      <use
    id='Clothes'
    fill='#E6E6E6'
    fill-rule='evenodd'
    href='#scoop_path'
      />${ColorHelper.getClothesColor(color, 'scoop_mask')}</g>`;
  }

  private static getVNeckShirt(color: ClothesColor) {
    return `
    <g id='Clothing/Shirt-V-Neck' transform='translate(0.000000, 170.000000)'>
    <defs>
      <path
        d='M171.319631,29.9364358 C205.706337,35.3665707 232,65.13854 232,101.051724 L232,110 L32,110 L32,101.051724 C32,65.1380521 58.2943778,35.3657617 92.6817711,29.9362145 C93.5835973,35.0053598 96.116393,39.8238432 100.236125,43.5389794 L100.236125,43.5389794 L129.321203,69.7676333 C130.843316,71.1402598 133.156684,71.1402598 134.678797,69.7676333 L134.678797,69.7676333 L163.763875,43.5389794 C164.189462,43.1551884 164.601167,42.7562772 164.998197,42.3430127 C168.414164,38.7873666 170.517305,34.4520434 171.319628,29.9364354 Z'
    id="v_path"
      />
      </defs>
      <mask id="v_mask" fill='white'>
    <use href='#v_path' />
      </mask>
      <use
    id='Clothes'
    fill='#E6E6E6'
    fill-rule='evenodd'
    href='#v_path'
      />${ColorHelper.getClothesColor(color, 'v_mask') }</g>`;
  }

  private static _getgraphic(graphic: Graphic, maskId: String) {
    switch (graphic) {
      case Graphic.BAT:
        return `
    <g
    id='Clothing/Graphic/Bat'
    mask='url(#${maskId})'
    fill-rule='evenodd'
    fill='#FFFFFF'>
    <g transform='translate(77.000000, 58.000000)' id='Batman!'>
    <path d='M87.6852404,13.4031978 C86.29151,6.97182978 81.4755637,3.2534243 75.1844381,1.81688169 C72.6586144,1.2402574 64.2721154,-0.570361088 65.5784818,4.22465302 C66.1709196,6.39748644 65.8414841,8.18940114 64.0250388,9.99135202 C62.2550056,11.7463153 58.793658,12.1906445 57.0732221,9.97994727 C55.5989529,8.08493361 57.5000322,5.25883596 56.3515582,3.41309083 C55.9056373,2.69596 55.1361961,2.32963935 54.323983,2.68090573 C53.1827893,3.17450341 53.7902429,4.23560158 53.2510425,4.99880761 C52.4024277,6.20178089 52.005649,5.82542406 51.2416681,4.6261003 C50.7616206,3.87247426 51.2334777,3.05133209 49.9949096,2.72926188 C48.5615922,2.35655457 48.0924652,3.5572469 47.9627841,4.65164695 C47.8813352,5.33866923 48.2849392,6.44356164 48.2990449,7.14016392 C48.3263462,8.50417231 48.2084956,10.4630526 47.5741958,11.668763 C46.449383,13.8060136 44.8695487,13.1221847 43.1909749,11.7645629 C41.2107218,10.1633357 40.6296595,8.3800886 41.0127875,5.94768301 C41.4719041,3.03125972 41.3053663,0.239376333 37.7334484,2.06368053 C32.7127425,4.62792506 28.0537786,9.19894983 25.144827,13.9830154 C22.6999971,18.0025063 20.7757117,22.8759851 22.9416134,27.4556775 C25.1521073,32.1307136 28.8036539,36.1460988 33.9080837,37.3394921 C35.2294658,37.6483328 38.9997729,39.1501106 40.2533568,37.8905698 C42.1999383,35.9335142 37.7052371,34.1082977 36.8179454,32.9158167 C35.5598113,31.2256324 34.4805006,28.0012808 35.8614905,26.0944062 C37.6206031,23.6670187 39.4602546,25.0200786 40.9063125,26.7376343 C42.0415909,28.085676 43.6550967,31.5636693 45.4119342,28.759925 C46.621381,26.8288723 46.513086,23.6583511 49.812446,25.0433443 C54.5269225,27.0213845 54.6716193,36.1552226 55.5120438,40.3343801 C55.879246,42.1637024 57.5473544,44.3981215 58.9829469,41.8594236 C59.7719539,40.4634819 59.3460539,37.437573 59.2022672,35.9960123 C58.9055932,33.0111605 58.1225014,30.0760334 59.092607,27.1190092 C59.6536484,25.4082963 61.093336,22.9594679 63.2223811,24.2751201 C64.9469121,25.3407802 64.2284333,30.0910877 66.5745236,30.0824201 C68.6412307,30.075121 68.0774592,26.2910242 69.1777009,25.0433443 C70.8571848,23.1387506 74.0937518,24.0442879 75.0870635,26.2919365 C76.4243714,29.3169331 72.88476,31.4176885 73.0435625,34.0768205 C73.1982698,36.6684365 76.5376717,35.657063 77.9345873,34.9996931 C80.8057722,33.6493704 83.0526678,31.1467116 84.6097509,28.428731 C87.0172691,24.2253954 88.7354298,18.2068795 87.6852404,13.4031978' />
      </g>
      </g>`;
      case Graphic.CUMBIA:
        return `
    <g
    id='Clothing/Graphic/Cumbia'
    mask='url(#${maskId})'
    stroke-width='1'
    fill-rule='evenodd'>
    <g transform='translate(77.000000, 58.000000)' id='Group'>
    <g transform='translate(0.000000, 6.000000)'>
    <path
      d='M10.271566,24.1315073 C13.5453197,23.5731503 16.0029461,20.5783262 15.4519585,17.3443595 C14.9878717,14.6197774 13.7065483,17.0011951 12.4758861,18.1969085 C11.1405733,19.4948562 10.0263951,20.7706412 7.9400381,20.2508902 C4.33273358,19.3518711 3.08247252,14.8503409 4.09976568,11.7711556 C4.65593031,10.0878629 5.87660814,8.76060323 7.57838317,8.07498941 C9.42992328,7.32896437 10.7818767,8.16542752 12.3264908,9.1691833 C12.6145575,9.35649386 14.0556303,10.5386236 14.3204002,10.4181586 C14.7822683,10.2079704 14.435405,7.98955578 14.3651448,7.69393398 C14.0267867,6.27266155 13.0993526,5.11877135 11.7481388,4.45603516 C8.41484886,2.82099993 4.23436935,4.94468693 2.10881463,7.51091298 C-2.77648522,13.4083357 1.18803674,25.6753895 10.271566,24.1315073'
    id='Fill-1'
    fill='#FFFFFF'
    />
    <path
      d='M20.283037,5.0382162 C20.2819296,5.04100097 20.2826679,5.03926048 20.283037,5.0382162 M18.6886879,17.9031695 C19.1918363,21.3771733 21.6658646,24.4038731 25.6412168,24.2597611 C29.9192703,24.1048581 31.6970859,20.1616201 32.6494871,16.7739442 C33.6055798,13.3751292 34.7008413,9.09110464 33.3132148,5.68149867 C32.8931247,4.64939232 32.6284457,3.30051808 31.5966777,4.15126612 C30.3445285,5.18372056 30.188379,8.18953451 30.0824336,9.59271181 C29.8819863,12.2396382 29.3005786,19.5639383 25.9741881,20.5368682 C21.7954354,21.7593834 21.9264829,14.6895419 21.9733647,12.559887 C22.0147092,10.6641531 22.2203245,8.82550695 21.6322721,6.97572174 C21.3192348,5.99095653 21.0416357,4.53521664 20.1036312,5.34175641 C18.8141981,6.45044401 18.6506656,9.16977447 18.5602244,10.6690264 C18.4155185,13.0736776 18.3464879,15.5110498 18.6886879,17.9031695'
    id='Fill-5'
    fill='#FFFFFF'
    />
    <path
      d='M37.7827156,20.7457507 C37.9712859,21.1583285 38.4071734,22.1476129 38.8028084,22.4248624 C39.7471104,23.085889 38.7480505,23.1327334 39.6027816,22.3662202 C40.4161723,21.6371824 40.7331154,19.6485506 40.8571366,18.6953537 C41.2422552,15.7354812 40.7367418,12.5875371 40.7704668,9.60302793 C41.7938233,11.8206768 42.3544573,14.1868398 43.1555184,16.4766638 C43.7096249,18.0589637 44.5516638,21.2808179 46.809793,21.2301566 C49.2532288,21.1753312 49.3790632,18.0849884 49.7043469,16.4110816 C50.1652562,14.0411017 50.674396,11.6898595 51.3797214,9.37401083 C51.4718308,13.2836103 49.952752,20.3692605 53.4880823,23.2940864 C53.5018625,23.3072722 54.9204913,19.144366 54.954579,18.8882832 C55.1783249,17.1970267 55.0383477,15.4440049 55.0622816,13.7402565 C55.1130505,10.1433001 55.779574,5.73680294 54.762745,2.23422934 C54.4338349,1.10024771 53.7872564,-0.0302639607 52.3530342,0.000618647874 C50.5278914,0.0401761465 50.1202895,1.98612748 49.6550286,3.29915884 C48.3742011,6.91277103 47.2097796,10.5229133 46.2981148,14.2395832 C45.7498104,12.5594305 40.9593998,-2.18198369 37.5092887,3.33767535 C36.9540943,4.22633108 37.1937961,5.55601823 37.1709501,6.53385184 C37.1274339,8.41283302 37.0161049,10.2897322 36.9754898,12.1690604 C36.9138418,15.0116484 36.5653494,18.0756195 37.7827156,20.7457507'
    id='Fill-7'
    fill='#FFFFFF'
    />
    <path
      d='M62.0203329,7.70997634 C62.739914,7.57453437 67.7552707,5.98272311 67.5433251,7.57379827 C67.3203514,9.24915661 62.9098151,10.8829254 61.7267107,11.4497207 C61.7246429,10.2414163 61.4930536,9.00918877 61.0839813,7.88259125 C61.3958688,7.82517563 61.7081008,7.76739196 62.0203329,7.70997634 M67.7366608,7.0706755 C67.7370055,7.06957135 67.7101246,7.03497476 67.7366608,7.0706755 M67.8590034,15.4110304 C70.1314738,16.6318484 69.1461853,18.8279958 67.4326998,20.0186339 C66.7844565,20.4691256 60.9030521,21.8209691 60.9195942,21.6818466 C60.9202835,21.6855271 60.9192496,21.6869993 60.9164926,21.6866312 C61.0960433,19.9925024 60.6587116,16.6686533 61.9159109,15.6664563 C63.2203241,14.6267183 66.4163948,14.8593252 67.8590034,15.4110304 M67.9210363,7.41553727 C67.9299966,7.4383563 67.9220702,7.41811361 67.9210363,7.41553727 M57.9444294,24.2596607 C58.1677477,24.8109979 58.8039291,26.1720425 59.5100698,26.2047988 C60.3706034,26.2449162 60.3040904,25.1617484 60.4388396,24.49558 C63.8802848,26.2162084 68.9369967,24.4481017 71.346284,21.4654339 C74.1325933,18.014976 72.6941202,13.1953763 68.7674402,11.7202366 C70.879659,10.0319965 72.7933728,6.32096 70.0222271,4.23081061 C67.8262639,2.57459081 64.19424,2.48294665 61.8993688,3.92827446 C59.1637197,5.65111111 58.0495406,9.7614806 57.8096802,12.9340616 C57.553967,16.3156944 56.6513889,21.0591079 57.9444294,24.2596607'
    id='Fill-9'
    fill='#FFFFFF'
    />
    <path
      d='M76.0506048,15.8714133 C76.1237655,17.940757 75.9002189,20.1580872 76.3823293,22.1766928 C76.5533503,22.8940747 76.8153532,23.6882686 77.1423879,24.3390566 C77.7529985,25.5536025 77.4525393,25.3897603 78.1744543,24.6980999 C80.3464523,22.6164239 79.3828568,16.1236949 79.3306438,13.4500732 C79.2896863,11.3715683 79.3947376,9.17044617 78.8182061,7.16734396 C78.656252,6.60499548 77.6954704,3.81897451 77.1633356,3.88345432 C76.3469995,3.98140725 75.7873513,7.81319908 75.7401407,8.57603396 C75.592256,10.9723573 75.9658759,13.4712141 76.0506048,15.8714133'
    id='Fill-11'
    fill='#FFFFFF'
    />
    <path
      d='M94.7477626,16.430477 C93.1655372,16.2926323 91.1264105,16.5045376 89.6311966,16.9871715 C90.3280165,15.0708102 91.1056588,12.9329761 91.8694667,11.1973368 C92.343479,10.1066267 92.8411555,9.02761048 93.3628603,7.95745314 C94.6410916,10.6431228 95.3546586,13.7848497 96.19492,16.6232471 C95.7158108,16.535721 95.233425,16.4715824 94.7477626,16.430477 M100.188346,17.1547822 C99.4605809,14.375208 98.6093975,11.6225649 97.7622188,8.8759458 C97.2244951,7.13215631 96.6321618,4.96384744 95.1624324,3.70729846 C90.9986238,0.146013273 88.6354794,9.55701945 87.6142778,11.9390054 C86.6294827,14.2352371 85.3978516,16.5725741 84.764743,18.9878696 C84.4541957,20.1714211 84.3555342,21.4261983 84.5197274,22.6402245 C84.725424,24.1589974 84.5157227,24.3811791 85.8136135,23.5544648 C86.8140635,22.9176858 87.2280051,21.7617741 88.0409617,20.9917569 C88.1847673,20.8556839 88.2586724,20.3241488 88.4301469,20.2373314 C88.6194606,20.1413008 89.9348266,20.4896334 90.2461019,20.5112492 C92.4155638,20.6622052 94.958192,20.2933198 96.9605482,19.4573923 C97.1658807,20.2989895 98.5882817,25.4187356 99.9378698,25.2302178 C100.5353,25.1465897 100.903369,22.1660952 100.929946,21.6933833 C101.017322,20.1398833 100.57826,18.6444979 100.188346,17.1547822'
    id='Fill-13'
    fill='#FFFFFF'
    />
    <path
      d='M109.301672,3.43213272 C109.038492,2.22890734 108.487555,0.14016215 107.459721,1.31990649 C106.069159,2.9162614 106.369359,6.49285063 106.348516,8.49692176 C106.333584,9.94847693 104.797121,20.5594163 106.905363,20.3801066 C106.815459,20.3801066 107.750277,18.7118855 107.885912,18.4578633 C108.700028,16.9340861 109.016405,15.4885791 109.213945,13.735684 C109.578851,10.499214 109.997264,6.62484253 109.301672,3.43213272'
    id='Fill-15'
    fill='#FFFFFF'
    />
    <path
      d='M108.164017,24.3012978 C105.924782,21.573307 101.852706,24.9591601 103.122505,27.675428 C104.85153,31.3753862 110.454236,27.1147969 108.164017,24.3012978'
    id='Fill-17'
    fill='#FFFFFF'
    />
    <path
      d='M94.8984918,28.5446042 C92.0066895,27.8070876 88.6012921,28.3048186 85.6491092,28.3933947 C82.5678984,28.4853064 79.489563,28.6568994 76.4083523,28.7499229 C69.8412367,28.9481999 63.2849033,28.8418345 56.7170689,28.7877252 C44.2631983,28.6854365 31.7816533,29.4770624 19.334252,28.9556122 C16.661689,28.8440582 13.7889353,28.2429265 11.1361398,28.7473286 C10.409056,28.8859372 8.12717022,29.2902742 7.80729649,30.013337 C7.47088995,30.7730903 9.20934979,31.5680517 10.1384213,31.968312 C12.560117,33.0119536 15.4670145,32.8262773 18.0389431,32.9319015 C20.9688428,33.0527208 23.9249793,32.9900875 26.856676,32.9181889 C38.9324486,32.6213292 50.9456841,31.5750933 63.0376301,31.7529868 C70.0126746,31.8560167 76.974421,31.7911597 83.9494655,31.7544692 C87.2686047,31.7366799 90.9489497,32.2799961 94.2217253,31.6922065 C94.773777,31.5928827 97.9847343,30.8375767 98.0293009,29.8561978 C98.0497872,29.4033107 95.2233973,28.6276211 94.8984918,28.5446042'
    id='Fill-19'
    fill='#FFFFFF'
      />
      </g>
      </g>
      </g>`;
      case Graphic.DEER:
        return `
    <g
    id='Clothing/Graphic/Deer'
    mask='url(#${maskId})'
    fill-rule='evenodd'
    fill='#FFFFFF'>
    <g transform='translate(77.000000, 58.000000)' id='oh,-deer!'>
    <path d='M76.9598206,16.2351328 C77.3545748,17.6603319 74.7317873,18.9054413 73.8540158,19.6535687 C72.3474717,20.9370645 70.7886986,22.2879408 69.0287021,23.2010483 C68.2808959,23.5889965 67.4755973,23.9042554 66.6674645,24.1357992 C65.420446,24.4927115 65.5625576,24.2427912 65.0864232,25.3008688 C64.2491394,27.1617948 64.0321258,29.6352706 63.2754123,31.6133978 C61.5708838,36.0670427 57.5083567,46.4138243 51.0084754,41.8719723 C48.9403681,40.4271716 47.8107637,37.6249608 46.6471497,35.4361165 C45.312678,32.9242543 44.0348889,29.927661 43.2668389,27.1858884 C42.8923285,25.8501217 43.1299908,25.3952014 42.1376394,24.7708091 C41.514535,24.3787772 40.4582132,24.2775023 39.7537287,23.9908291 C38.2544724,23.3807295 36.9414591,22.5546042 35.6656945,21.5602318 C34.4279881,20.5948534 29.5561137,16.6096049 33.5340216,15.8696448 C35.3592032,15.5298839 37.7512114,15.6560692 39.6006855,15.7495851 C41.6283052,15.8520851 43.6741443,16.3196647 45.4774626,17.2854514 C47.3184343,18.2720648 46.7783296,17.9862083 48.6865108,17.4345052 C45.3653119,16.3090472 42.0384447,14.4370952 39.8031236,11.6536693 C38.5961879,10.1508806 35.8037572,4.00047299 37.9022302,2.58875006 C40.4225841,0.893212425 40.6331197,7.93508406 41.3416529,9.15569148 C42.9344356,11.8995059 45.8495437,12.9351233 48.3844732,14.454655 C46.6435058,12.7868862 45.3446632,10.9496454 44.7106272,8.58683697 C44.4357163,7.56224546 43.6846711,5.0981621 44.1919809,4.10133952 C44.7207491,3.06123008 45.9641237,2.99507471 46.6499838,3.88041325 C47.2678248,4.67754462 47.1123524,6.5115185 47.3241026,7.47240483 C47.6739156,9.06095045 48.2451958,10.282783 49.3059712,11.5127828 C50.8113007,13.2585495 52.7227209,14.6866071 54.9750468,15.2395353 C60.1015883,16.4977125 61.8947847,11.1317769 62.5826693,6.99461605 C62.7510977,5.98186717 62.5033135,4.30184746 63.6329179,3.81589134 C65.0001846,3.22743524 65.5293577,4.56156854 65.5690356,5.59718594 C65.6564888,7.91058207 64.5127138,11.4535697 63.4029483,13.4419061 C66.1929497,11.7100238 68.4064075,8.96375923 69.3015886,5.75399867 C69.6327773,4.56646894 69.3639395,1.9182123 70.4757294,1.25134983 C72.0154733,0.327216479 72.5049686,2.16772422 72.45112,3.27317229 C72.1324825,9.78825116 67.1525061,15.7132405 60.9627599,17.341806 C62.4810453,18.1005509 62.2814414,18.2716565 63.8114683,17.3650829 C64.6961227,16.840332 65.656084,16.4352324 66.6375037,16.1342663 C68.1076089,15.6834297 69.6931038,15.453111 71.2182722,15.3183501 C72.4377591,15.2105413 76.4921887,14.5012087 76.9598206,16.2351328 Z M58.8595294,24.6214062 C59.7193936,22.4825816 56.3639386,22.4048657 56.0375022,24.7163015 C55.622102,27.6609177 58.7932497,27.8924294 58.8595294,24.6214062 Z M56.9297708,33.6367531 C57.5886752,31.4414328 53.4036129,31.5246305 53.0344702,33.4664333 C52.5980925,35.7582944 56.4400387,35.1974948 56.9297708,33.6367531 Z M51.7024587,27 C54.0260737,27 54.9736078,23.9354052 52.6986912,23.1385385 C49.8083182,22.1255453 48.8835101,27 51.7024587,27 Z' />
      </g>
      </g>`;
      case Graphic.DIAMOND:
        return `
    <g
    id='Clothing/Graphic/Diamond'
    mask='url(#${maskId})'
    stroke-width='1'
    fill-rule='evenodd'>
    <g transform='translate(77.000000, 58.000000)' id='Diamond'>
    <g transform='translate(29.000000, 2.000000)'>
    <path
      d='M34.7804613,27.7365051 C32.3910217,30.3933795 29.9226048,32.9801662 27.4446301,35.5565142 C28.4300853,32.0044048 29.1846452,27.9785814 30.6489944,24.5994547 C31.0655115,23.6371144 30.7485963,23.5153306 31.576097,23.1057395 C32.1847753,22.8045111 33.6536519,23.1494823 34.3825567,23.1678741 C35.9404713,23.2066461 37.5335988,23.3597457 39.0789374,23.1186635 C37.6281703,24.6436945 36.1869609,26.1712109 34.7804613,27.7365051 M19.5162179,31.798118 C16.8174087,28.5164183 14.1321816,25.2302449 10.8578947,22.4814113 C12.146683,22.6176103 13.5536857,22.4376686 14.8163159,22.687201 C16.7595591,23.0714412 16.6534177,23.6778747 17.6358547,25.6458009 C19.0866218,28.5522078 20.5248129,31.434755 21.6858291,34.4689106 C20.9750337,33.5692019 20.2446197,32.6844056 19.5162179,31.798118 M10.4182378,19.266817 C11.3810563,17.6920783 11.5616476,16.294796 13.3373785,15.4865494 C15.3988361,14.5480688 18.6379102,14.7454082 20.8256308,14.5709343 C19.4553501,16.093977 17.7887788,17.6736865 17.2223558,19.6928116 C16.3480725,17.9749149 15.2599971,16.5189775 13.5521766,15.5223389 C13.3368755,15.4159645 14.7101745,20.0725781 14.8897597,20.3926954 C13.1743936,20.2967597 11.3684803,19.9850926 9.66518708,20.3012334 C9.916204,19.9562622 10.1672209,19.611291 10.4182378,19.266817 M26.8243819,14.2592673 C28.4154971,14.2945597 29.9940364,14.4183318 31.563521,14.6783029 C30.2968665,16.353451 29.9150592,18.3104416 29.4653415,20.3086895 C28.9869505,18.0509676 27.2564932,15.9473394 25.7312761,14.2607585 C26.095477,14.2602615 26.4596779,14.2597644 26.8243819,14.2592673 M23.3257392,15.3334498 C23.5601558,15.1748824 27.7826729,20.4190405 28.3923573,20.889275 C25.0798394,20.8813217 21.7743641,20.7192748 18.4658705,20.5736313 C20.3728949,19.0868752 21.4886374,16.8863171 23.3257392,15.3334498 M26.0124754,32.4289082 C25.6105465,33.7779741 25.2352788,35.1369815 24.797634,36.4756088 C23.2432407,31.7444337 20.9357965,27.288142 18.7898282,22.8020257 C22.1295103,22.8900083 25.465671,23.0107979 28.8058561,23.0575231 C27.9864041,26.2164449 26.9441054,29.3032906 26.0124754,32.4289082 M33.2416622,14.9656131 C34.5188805,16.093977 36.338879,18.4063774 36.9284418,19.1833081 C36.9867944,19.260355 37.0461531,19.3369048 37.1070209,19.4124604 C37.3177947,19.6744198 38.0733606,20.6680759 38.3701542,21.0567898 C36.1426934,20.8306199 33.8251885,20.9270528 31.5901821,20.9066727 C32.0881916,18.8527521 32.6938517,16.8952645 32.1737084,14.7787123 C32.5308668,14.8363732 32.886516,14.8985078 33.2416622,14.9656131 M40.8898811,20.5830758 C38.7323429,17.9068154 36.5199733,13.3247633 33.025858,12.5055811 C29.5538765,11.6918667 25.478247,12.0517501 21.9519372,12.2754346 C18.8773572,12.4702886 13.8690917,12.0721303 11.1572035,13.8690622 C9.76378291,14.7921334 8.73154499,16.8738902 8.2491297,18.4202955 C7.81450321,19.8131041 8.51523782,20.7048596 7.00259677,20.7048596 C6.90651615,20.7048596 9.50672951,24.3777585 9.81308082,24.66656 C12.0853121,26.8064748 14.2574385,28.8688456 16.2701013,31.2642577 C19.1047312,34.6379166 21.6747622,39.6022206 25.414864,41 C24.7292206,40.6609938 28.2067356,38.0334469 28.6242587,37.5811072 C30.7355173,35.2930636 32.8709217,33.0268914 34.9967685,30.7517718 C36.9349813,28.6779682 39.0472459,26.7209776 40.9271061,24.6004489 C42.361776,22.9824645 42.3658003,22.4262358 40.8898811,20.5830758'
    id='Fill-21'
    fill='#FFFFFF'
    />
    <path
      d='M6.99858316,11.9999257 C7.04159508,12.0142408 6.09038895,9.95483812 6.17394084,10.0812061 C5.69932656,9.36248835 5.12830624,8.8402333 4.43566602,8.35351923 C3.70792411,7.84261755 0.1270582,7.25421678 0,7 C1.49355183,9.65224611 4.16721231,11.1276907 6.99858316,11.9999257'
    id='Fill-23'
    fill='#FFFFFF'
    />
    <path
      d='M22.3553325,7.19614224 C22.5279973,7.84840006 22.7925782,8.44932951 23.1490753,8.99893058 C23.2186566,9.07737584 24.0076748,4.81518326 23.9999436,4.45201075 C23.9866286,3.8317121 23.6120919,0 22.814054,0 C21.4769753,0 22.1397162,6.46931298 22.3553325,7.19614224'
    id='Fill-25'
    fill='#FFFFFF'
    />
    <path
      d='M48.969198,5 C45.977552,5 39.2896081,10.8535875 40.0615528,13.9997703 C40.0673207,14.0103781 40.50905,13.6497123 40.4129174,13.7172166 C40.4321439,13.7032335 40.4508897,13.6887683 40.4691549,13.6738209 C41.7405084,12.649202 43.1873037,11.6544779 44.4605798,10.4683308 C45.3723974,9.61874094 46.4476403,8.85642448 47.2772645,7.92727594 C47.3618612,7.83276992 49.2628831,5 48.969198,5'
    id='Fill-27'
    fill='#FFFFFF'
      />
      </g>
      </g>
      </g>`;
      case Graphic.HOLA:
        return `
    <g
    id='Clothing/Graphic/Hola'
    mask='url(#${maskId})'
    fill-rule='evenodd'
    fill='#FFFFFF'>
    <g transform='translate(77.000000, 58.000000)' id='Hola'>
    <path d='M74.6371722,16.9479681 C75.917206,22.7098952 73.7328313,28.1917851 69.3072818,31.9519351 C67.2267705,33.7200766 64.8728066,35.0262382 62.2514759,35.7865716 C60.7921156,36.2102906 59.2674351,36.5090512 57.7541146,36.6572104 C56.61811,36.7683298 54.9700918,36.2570992 53.9631048,36.5456839 C52.8441403,36.8672382 51.4050658,38.4111058 50.2065809,38.9084974 C49.535121,39.1873134 48.8385068,39.4319389 48.1390525,39.6350472 C46.6326292,40.0734193 45.0742742,40.3689236 43.5256565,40.611514 C41.9242956,40.8622449 35.2433712,41.9140123 36.070626,38.9752504 C36.3550329,37.9641861 39.1816559,36.4707903 39.7950984,35.791049 C40.2486888,35.288773 42.9398027,32.427347 42.8140307,31.4960606 C42.7742705,31.2021844 41.0666121,29.8650885 40.6657648,29.4157265 C39.7573668,28.3973356 38.9041462,27.3581862 38.4205328,26.0646425 C37.5920608,23.8487672 37.6354724,21.0667121 37.9105478,18.7446017 C38.5986421,12.9407505 41.476791,9.08616941 46.5133487,6.19584415 C49.3679661,4.55795247 52.4720988,3.65678642 55.718232,3.26074551 C59.6739625,2.77841409 62.8279982,2.76294692 66.1848919,5.01097768 C70.6051671,7.97090519 73.4601902,11.6785482 74.6371722,16.9479681 Z M69.5056769,23.9358717 C70.5917785,23.6358901 69.1977385,20.8827342 68.427284,22.2161669 C68.1497742,22.6968702 68.8902058,24.1047895 69.5056769,23.9358717 Z M67.6848238,18.6143519 C67.7728641,18.973353 68.1185341,19.0836583 68.4325583,19.0405131 C69.182727,18.9371273 68.7125023,18.059976 68.7340052,17.6684125 C68.7741711,16.9194759 68.8407085,15.0292252 68.5619816,14.3474487 C68.0491567,13.0942011 67.5866405,13.3339422 67.2884393,14.5615469 C67.0210725,15.6617509 67.4109655,17.5092635 67.6848238,18.6143519 Z M66.6372652,22.892652 C67.4795315,22.3952604 65.2570196,16.7171816 64.7685376,15.9051553 C62.8770898,12.7604359 62.2652702,18.3363499 61.9751833,19.714149 C61.8498171,20.310856 60.4574,26.4960951 62.3135504,24.5020515 C63.3047144,23.4376661 61.755691,19.900162 64.5567538,20.7227711 C65.540615,21.0117629 64.8289892,20.9287287 65.4314774,21.8315228 C65.6517812,22.1616248 65.8887193,23.3326522 66.6372652,22.892652 Z M60.2947079,21.5987013 C61.0809854,20.2481733 57.6303713,20.8778498 57.2327696,20.9022717 C57.3268957,19.1027888 57.6782457,17.2792911 57.6733771,15.4798082 C57.67216,15.0052104 57.8380978,13.2887618 56.7146704,14.0987529 C56.2606742,14.4264127 56.4168749,15.9283561 56.3787376,16.3789391 C56.2886686,17.436405 55.4313908,21.0858425 56.2708171,21.9381649 C56.8660024,22.5421985 59.9640494,22.1587756 60.2947079,21.5987013 Z M55.4961584,19.1587406 C55.8337431,17.8148148 55.4118876,14.9057989 53.6512301,14.8794474 C53.0382604,14.8705068 53.3156518,15.7104604 53.164165,15.7984555 C52.0992431,16.4200683 51.6357538,16.3414844 51.0047261,17.4454234 C48.5177346,21.7957715 54.0705776,24.8897184 55.4961584,19.1587406 Z M49.4685836,22.9899323 C50.2301124,22.7799045 50.0146773,13.0115738 49.1375194,13.0062824 C47.6286618,12.9977348 49.0750392,17.8462849 48.5033854,18.49672 C48.3321733,18.6916877 47.4769241,18.5109661 47.2529689,18.5520762 C46.1129071,18.7637322 45.3623326,18.6444722 45.0126054,17.9464145 C44.7014213,17.3265067 45.3736926,13.0661159 43.7552917,15.1562188 C43.7698975,15.8937585 43.7816632,16.6317052 43.7901833,17.3692449 C44.1261161,18.0843979 44.0007498,18.5195137 43.4136789,18.6754065 C43.3844673,19.14512 43.8056005,19.5387187 43.8108748,19.6921693 C43.8360292,20.4630855 43.3268557,23.9232538 44.7111585,23.7563712 C45.5960249,23.6493221 45.0316741,20.2746303 45.0280226,19.7255458 C46.1705187,19.7532239 47.3673807,19.8806245 48.5033854,19.7336864 C48.5086597,20.1610687 48.2149214,23.3367225 49.4685836,22.9899323 Z M63.8368644,17 C64.2706568,17.6069046 64.6557203,18.2184508 65,18.8421816 C64.3342161,18.8990426 63.6679025,18.9518422 63,19 C63.2722458,18.332463 63.5439619,17.664926 63.8368644,17 Z M53.9230169,17 C54.0684325,17.4229772 54.1789882,20.2877939 52.5679624,19.9763934 C51.6476608,19.7985469 52.0614978,18.2121289 52.3219511,17.8848372 C52.8722396,17.1947521 53.1949428,17.4368397 53.9230169,17 Z' />
      </g>
      </g>`;
      case Graphic.PIZZA:
        return `
    <g
    id='Clothing/Graphic/Pizza'
    mask='url(#${maskId})'
    stroke-width='1'
    fill-rule='evenodd'>
    <g transform='translate(77.000000, 58.000000)' id='Group-2'>
    <g transform='translate(35.000000, 1.000000)'>
    <path
      d='M30.9332298,24.3911229 C26.8510636,25.7713262 22.3826909,27.7147062 20.0279438,31.464875 C18.1805823,34.4072634 18.056518,31.0097105 15.4771364,32.0578917 C12.9702973,33.0766532 13.6557418,37.2811459 12.7452506,39.3594362 C12.0367243,38.1679392 11.7020395,36.8427927 10.8096839,35.755105 C11.5033719,35.309607 12.1459503,34.7115469 12.6240715,34.0306915 C12.7802853,33.7116981 7.83173048,34.3791045 7.54774295,34.4602188 C6.13687312,34.8620076 4.80843799,35.231855 3.46021853,35.8900153 C3.8167486,35.3734898 4.16627171,34.8531817 4.51043656,34.3282505 C4.82039102,33.8562748 5.1253994,33.3800962 5.42463735,32.9009757 C5.63196061,32.2209608 6.00126805,32.2188594 6.53255966,32.8950918 C7.28147888,32.931236 7.71055147,33.2519105 8.49450544,33.0724504 C11.4098085,32.4054642 12.3380232,26.8489271 8.74469474,27.0851251 C10.3237313,24.2158656 11.7251212,21.3024766 12.9975007,18.2831768 C13.4999402,22.5742474 19.0774717,21.0961186 20.7166857,18.8076877 C22.8571026,15.8190684 20.1132637,10.1456931 16.2334747,10.6962614 C16.8142272,9.32068116 17.4036352,7.93375334 17.8825808,6.5182464 C18.9752527,8.1287638 20.8844403,8.90880562 22.421023,9.97463863 C24.031797,11.0917459 25.3940304,12.414791 26.5839752,13.9807586 C28.9704596,17.1210994 30.3767956,20.9881063 32.7987269,24.0334636 C32.1610946,24.0675064 31.539537,24.1868662 30.9332298,24.3911229 M39.9388082,20.0029673 C39.306534,14.2455357 34.7895249,8.40194655 30.4996233,4.82661475 C27.9499181,2.7020935 20.2587609,-2.55814545 17.1930146,1.48706068 C19.664819,2.55835735 22.3921709,2.81640998 24.8425422,3.97596569 C28.1630118,5.5482375 30.7646508,8.13506801 33.0114085,11.0312255 C36.1484608,15.075591 40.9816071,23.3139423 33.485408,25.4170292 C34.0030979,25.0791231 34.3035723,24.7080149 34.7132728,24.1086939 C33.706333,23.8762784 34.1131482,22.231298 33.7590912,21.1570594 C33.3473298,19.9063026 32.549362,18.6904293 31.874634,17.5745828 C30.8042195,15.803518 29.6278764,14.0408588 28.3707472,12.4038637 C26.692789,10.2179814 24.685092,8.49735037 22.3938196,7.00829423 C21.4874501,6.41948033 20.6190006,5.89749114 19.5774382,5.59320758 C18.2279822,5.19898385 18.6698322,5.67936522 18.3005247,4.44668058 C18.0692954,3.67420382 18.277443,2.68402141 18.0548693,1.85480669 C16.0817949,3.51323614 15.4400408,5.90967929 14.5555165,8.23635577 C13.5353871,10.9202712 12.4328231,13.5714048 11.3591111,16.2330453 C8.72367389,22.767997 5.32530354,28.6283975 1.46777192,34.479972 C0.796341323,35.4991538 -0.697375374,37.4299255 0.377573066,38.6537842 C1.33876162,39.7486166 2.7084141,38.9597489 3.76399039,38.5150914 C5.20041497,37.9107271 7.14917119,36.3506434 8.69894348,37.1680903 C10.6971605,38.2221554 9.75287108,42.1522045 12.9130052,41.9954397 C16.0265636,41.8407762 15.4404529,36.7154475 16.4395615,34.6775042 C18.7514425,36.3628316 20.615291,35.3171721 22.0187417,33.0665665 C23.6451783,30.4578814 25.7942508,29.4101204 28.5323192,28.1291035 C29.6521946,27.6050129 31.5663282,26.1899262 32.7327792,26.0138284 C33.6980895,25.8688314 34.6254798,26.8972594 35.915995,26.8506082 C39.0138909,26.7375526 40.3077034,22.5977832 39.9388082,20.0029673'
    id='Fill-29'
    fill='#FFFFFF'
    />
    <path
      d='M18.5372079,22.8336349 C16.1119296,22.3885123 12.7302873,22.7596628 12.5083803,25.9869094 C12.3721811,27.9648021 13.9174686,29.8351772 15.5900451,30.6510203 C20.7121513,33.1501581 24.6988403,23.9625691 18.5372079,22.8336349'
    id='Fill-31'
    fill='#FFFFFF'
    />
    <path
      d='M28.3362306,19.157604 C27.6856351,18.4518413 26.6326311,17.918164 25.6605301,18.1654572 C25.9719227,17.9508815 26.2293799,17.5642204 26.5251817,17.3262751 C23.3686979,15.0403008 21.3853086,20.8049502 23.1786599,23.0743534 C25.8964974,26.5139375 31.2158743,22.3078297 28.3362306,19.157604'
    id='Fill-33'
    fill='#FFFFFF'
      />
      </g>
      </g>
      </g>`;
      case Graphic.RESIST:
        return `
    <g
    id='Clothing/Graphic/Resist'
    mask='url(#${maskId})'
    stroke-width='1'
    fill-rule='evenodd'>
    <g transform='translate(77.000000, 58.000000)' id='Resist'>
    <g transform='translate(3.000000, 9.000000)'>
    <path
      d='M102.565245,21.0710109 C99.4833509,20.4054313 97.3728781,24.6059197 100.65303,25.85541 C103.46796,26.9271503 105.497722,21.7103865 102.565245,21.0710109'
    id='!'
    fill='#FFFFFF'
    />
    <path
      d='M101.191001,18 C104.847558,18 103.499462,12.0185653 103.512139,10.034892 C103.52569,7.89835492 105.057378,1.43936922 102.622588,0.300988793 C98.4060734,-1.67043963 99.5622709,6.62567562 99.5876242,8.26886181 C99.6156003,10.0850565 99.7546063,11.9885455 99.3594446,13.7770905 C99.009744,15.3574722 98.2303489,18 101.191001,18'
    id='H'
    fill='#FFFFFF'
    />
    <path
      d='M96.057932,1.97392328 C94.9756601,1.35173361 93.2567577,1.65427333 92.0711316,1.60022061 C90.7220251,1.53916824 89.3807782,1.40539746 88.0356015,1.30662485 C85.8592683,1.14718875 83.080101,0.743543202 80.9175222,1.23857286 C79.6922049,1.51933595 78.5800664,2.46389764 79.1565707,3.84671419 C79.777482,5.33646958 81.4602301,4.95187859 82.7370279,4.87682696 C83.3217848,4.84221766 84.7703158,4.58750876 85.3421043,4.78816493 C86.3390991,5.13853549 85.9154647,4.67967061 86.1430012,5.85522022 C86.491969,7.66229234 86.2825097,9.85512207 86.2675764,11.6929148 C86.2420326,14.8668599 86.2322081,18.0567485 86.1650082,21.2287493 C86.1386784,22.4645735 85.7244756,23.8606116 86.9218911,24.6768466 C87.9196719,25.3569777 89.1379155,24.9023904 89.6550795,23.8835548 C90.1694927,22.8705522 89.6806234,20.8169375 89.6259989,19.6876632 C89.5599779,18.3406226 89.4900272,17.0122476 89.5293254,15.6628738 C89.6330726,12.0775058 89.8110933,8.49563763 89.8971563,4.90910305 C90.8399197,4.95537841 91.8188373,4.92737987 92.7525621,5.05842857 C93.4359574,5.15409023 94.4184119,5.59351169 95.0754775,5.55890239 C96.9743656,5.45818543 97.7689748,2.97175996 96.057932,1.97392328'
    id='C'
    fill='#FFFFFF'
    />
    <path
      d='M67.7216119,8.81264097 C67.7100258,8.75972031 67.6397369,8.1691564 67.7216119,8.81264097 M67.6853088,8.53730013 C67.6887847,8.56222653 67.685695,8.5449698 67.6853088,8.53730013 M69.1169633,5.42725256 C72.533315,1.44593148 73.7004204,9.77441647 76.3625176,9.42199554 C80.623109,8.85827545 75.4217272,2.46944776 73.6942412,1.64649312 C70.1786354,-0.0289443454 67.0886252,1.72587411 65.4167525,4.9126183 C63.3188978,8.91234656 64.6466637,11.6208874 68.6751471,13.3572987 C70.1461943,13.9915796 75.714469,15.8848355 74.2051877,18.3157346 C73.4451402,19.5398125 70.6791544,19.6445033 69.5047111,19.402142 C67.1492591,18.916269 67.5161519,17.3159942 66.3699015,15.8288469 C65.3371947,14.4885737 63.3397528,14.8820273 63.028087,16.6100019 C62.7828481,17.9675319 64.198282,20.0348891 65.1433208,20.9897619 C67.3759602,23.2457927 71.1831493,23.4271802 74.0298516,22.3998291 C78.4155727,20.816811 78.9473743,16.6909168 75.8357368,13.4953526 C74.0870096,11.6995014 71.9026455,11.154572 69.7225296,10.0773681 C67.0781977,8.7712248 67.5725376,7.68673472 69.1169633,5.42725256'
    id='A'
    fill='#FFFFFF'
    />
    <path
      d='M58.749962,20.5676943 C58.1855671,15.7434329 58.0538346,10.8505093 57.9696387,6.00460433 C57.9426476,4.45075032 58.6758373,0.798596322 56.5169562,0.139962235 C53.6023258,-0.749287073 53.9890632,2.83233897 54.0502966,4.30372379 C54.255348,9.21679823 54.8922562,14.0929294 55.1210758,19.0000333 C55.193992,20.5568726 54.6924419,23.5739018 56.9520358,23.9463193 C59.7027058,24.4000865 58.9598476,21.8648117 58.749962,20.5676943'
    id='E'
    fill='#FFFFFF'
    />
    <path
      d='M49.4716454,4.67909741 C47.1120678,1.521124 42.3181606,1.01110068 39.382093,3.92183601 C37.3101733,5.97595969 36.00495,10.8410892 37.9661116,13.3252251 C40.0904751,16.016404 45.3211527,13.6574039 46.6913436,16.7080638 C48.3726748,20.4511421 43.9611374,21.8560762 41.6168232,19.3708027 C40.7738094,18.4774087 40.9581453,16.917382 39.7163241,16.3637507 C37.9539791,15.5788077 36.853443,17.2916519 37.2001979,18.7227508 C38.0479082,22.2235205 41.8524287,24.1130716 45.3011927,23.9947614 C49.0689244,23.8650751 50.7028996,21.0184245 50.4641631,17.5957699 C50.1342371,12.8603265 46.4792204,12.1186123 42.4747091,11.597213 C40.7761576,11.3761397 40.5534673,11.3966165 40.6583549,9.64547322 C40.7855505,7.52650643 42.0277631,5.07763608 44.6491681,5.58159221 C46.7551372,5.98657727 46.942604,9.15327227 49.1037564,9.30040167 C52.5998763,9.53853896 50.3604497,5.86712943 49.4716454,4.67909741'
    id='P'
    fill='#FFFFFF'
    />
    <path
      d='M31.7209696,20.4381695 C30.3835461,20.7580967 28.7579173,20.5355712 27.3944988,20.5112208 C26.3407598,20.4924897 22.8201338,20.9379153 22.1329619,20.2148947 C21.3727026,19.4150766 21.6202201,16.9680453 21.5881973,15.9288437 C21.5422351,14.4753101 21.1839563,14.2599024 22.4595946,13.9298603 C23.2085518,13.7361807 24.3674009,13.8332078 25.1415996,13.7976187 C26.6572206,13.727939 28.606845,14.008531 30.0716062,13.7099572 C31.4414292,13.4304891 32.5739067,11.958599 31.3223797,10.7058628 C30.4355361,9.81763391 28.7797682,10.2949024 27.6909924,10.3200021 C25.6573553,10.3664552 23.6222112,10.3555912 21.5881973,10.3915549 C21.5964855,8.82376157 21.5448723,7.2454788 21.6669357,5.68105705 C24.5143291,5.82154032 27.4788884,6.56329201 30.3255283,6.42655496 C31.766555,6.35724988 33.3654353,5.43530497 32.6368221,3.69630934 C32.0121888,2.20606276 30.1149312,2.40985716 28.792954,2.35066687 C27.1398233,2.27611708 25.4848088,2.23790563 23.8313013,2.18096308 C22.6068996,2.13900541 20.8260543,1.73628668 19.6721027,2.28847961 C17.3095724,3.41833976 18.121445,7.29867513 18.1873743,9.39955568 C18.2706336,12.0657409 18.2721406,14.6678658 18.3644416,17.3561538 C18.4484544,19.7885748 18.3271445,22.9972128 21.2159792,23.6808981 C24.1051906,24.364958 27.4611816,23.7142395 30.4084109,23.8565959 C31.6166129,23.9150369 33.2693669,24.2671816 33.8631075,22.8608504 C34.4319834,21.512211 33.1326106,20.086774 31.7209696,20.4381695'
    id='M'
    fill='#FFFFFF'
    />
    <path
      d='M8.41021607,5.87934286 C10.7314099,6.38056324 11.3520318,8.89406466 11.4306011,11.0286326 C11.4842397,12.4898438 11.613048,12.4018284 10.4273316,12.768689 C9.23708235,13.1371074 7.50704692,12.9377097 6.28808963,12.8851342 C3.74516418,12.7756991 4.0526421,12.6070678 4.00353629,9.93973422 C3.99220418,9.31505862 3.52947636,6.4362544 3.89625898,6.02967003 C4.37220759,5.5015777 7.73482232,5.8267673 8.41021607,5.87934286 M13.4870013,20.7181143 C12.1966517,19.2140637 10.9010138,17.7773878 9.45277018,16.4318428 C11.6107816,16.3722571 13.9557729,15.9594416 14.7229567,13.6130136 C15.3711534,11.6307201 14.8079476,8.6132722 14.05474,6.74080322 C13.0612917,4.27092005 10.9928039,2.62238402 8.42003723,2.26253349 C6.62389782,2.01172858 2.14242611,1.58956627 0.799948834,2.97327723 C-0.664537495,4.48161175 0.351952758,8.62222952 0.4422319,10.4701632 C0.602014649,13.7411422 0.486049391,16.987586 0.285471047,20.2550599 C0.220500284,21.3096868 -0.296621662,23.0423437 0.243919977,23.9918195 C0.843766325,25.0456674 2.30296434,25.3089347 3.21255502,24.5288691 C4.19693763,23.6845445 3.74176455,22.6513373 3.68132663,21.5678912 C3.58802559,19.8917044 3.75574082,18.1644997 3.85546339,16.4937651 C5.46348977,17.806207 7.10551249,19.0851563 8.61683821,20.5163799 C10.1096548,21.9300785 11.1809169,23.7184265 12.6076296,25.143419 C13.618076,26.1528699 15.4266807,26.5676327 15.9381366,24.6947742 C16.377067,23.0871303 14.3667507,21.7435326 13.4870013,20.7181143'
    id='I'
    fill='#FFFFFF'
      />
      </g>
      </g>
      </g>`;
      case Graphic.SELENA:
        return `
    <g
    id='Clothing/Graphic/Selena'
    mask='url(#${maskId})'
    stroke-width='1'
    fill-rule='evenodd'>
    <g transform='translate(77.000000, 58.000000)' id='Selena'>
    <g transform='translate(17.000000, 0.000000)'>
    <path
      d='M65.3033729,7.56759412 C65.1470447,8.49857487 63.3805357,12.656817 62.0741755,10.5184706 C61.5650892,9.68550605 62.4755312,7.05707864 62.7589611,6.30931546 C63.3169849,4.836648 63.7329539,4.67144048 65.0219821,3.97805378 C65.1307322,5.16706304 65.4331933,6.37685313 65.3033729,7.56759412 M62.3188632,10.8866375 C62.3487694,10.8987596 62.4772304,11.018942 62.3188632,10.8866375 M62.3110467,10.8804032 C62.3052694,10.8755544 62.3107069,10.8800569 62.3110467,10.8804032 M55.0298897,6.20471867 C55.040085,6.21337734 55.0829054,6.246973 55.0298897,6.20471867 M42.5545574,11.1605979 C42.3475925,12.3804321 42.0797954,13.5995735 41.6509123,14.760875 C41.48167,13.0294864 41.7542249,11.3046783 42.651413,9.81019097 C42.6412176,10.2618275 42.6089325,10.7120786 42.5545574,11.1605979 M42.544702,8.65858718 C42.5406238,8.64334792 42.3859948,8.41891506 42.544702,8.65858718 M25.9083201,18.8030909 C26.0609101,17.2701591 26.6763675,15.7860622 27.8141651,14.7591433 C27.5371923,16.3838571 26.7416175,17.4294787 25.9083201,18.8030909 M75.8371766,5.29694317 C75.8371766,5.93422166 73.7787416,7.63166832 73.2635382,8.10027581 C72.3850415,8.898952 71.4487714,9.53969396 70.1865909,9.35370562 C66.7252801,8.84422918 68.0968903,5.14766761 67.5456634,2.87251415 C67.1120225,1.08501726 66.6671667,1.75900853 65.4641191,1.64644576 C64.3456926,1.54219531 63.9460361,0.990464532 62.7824103,1.83312679 C60.685573,3.35151205 59.0159196,5.32915344 58.1231495,7.8135005 C57.7992782,6.11813191 56.8776214,3.30960407 54.6706746,3.71967892 C53.0387439,4.02273255 51.6868446,6.64596475 51.0078363,7.96277605 C50.95618,6.76545446 50.3403828,4.88721467 48.8328349,5.74476985 C47.7973303,6.3339061 47.8418499,9.12615564 47.615174,10.1548062 C47.1499276,12.2664839 46.236427,17.7723622 42.9708663,17.056463 C43.999574,15.1377006 46.2510403,9.08632573 44.3207265,7.1194211 C41.6091115,4.35730375 39.5459188,11.2748924 39.4354695,12.9584852 C39.355946,14.1644655 40.2405598,16.9030312 39.7637588,17.6511408 C39.1788873,18.568614 37.3362533,18.9641423 36.4608152,18.2922291 C35.0562401,17.2143972 35.6101858,15.4622277 35.9313383,14.093118 C36.3527448,12.2990405 36.5512137,10.495958 36.6650614,8.65616275 C36.7503623,7.27631629 37.951031,1.10787617 35.6370334,1.02406019 C33.5415555,0.948902892 33.3281335,6.16211798 33.1786022,7.58664321 C32.9944067,9.33812 32.8825981,11.1121094 32.94377,12.8746693 C33.0056216,14.6573173 33.3376491,15.8449412 32.6467463,17.5181435 C31.9065662,19.3108356 28.3834038,24.6889119 26.2743321,21.1180743 C28.5142437,20.1354878 31.1850774,15.5997275 29.9833892,13.0506135 C28.317474,9.51683506 24.5679755,14.9018383 24.024565,16.7073452 C23.1474277,19.6222016 23.5328108,23.4292478 26.8877504,24.3782386 C30.3810066,25.366713 33.0780083,22.0916557 34.309263,19.1975801 C35.3107832,20.7859275 37.1544368,21.1842266 38.8580746,20.7031506 C39.5350438,20.5116207 40.1861848,20.1822447 40.7496461,19.7555452 C41.8198148,18.9457859 41.1934823,19.2471078 42.3292409,19.2142048 C45.6264071,19.1196521 47.4252012,18.0612156 48.660534,14.6690931 C48.6459207,15.6031909 49.0041163,17.0287552 50.2588202,16.5178934 C51.2334927,16.1209797 51.2266958,14.5083881 51.4074928,13.6480621 C51.9774112,10.935126 52.8729,8.10651006 55.0057608,6.22584583 C56.8466956,8.22807786 55.4856205,11.6534497 56.6968244,13.7862546 C58.5635873,17.0737803 59.5035957,11.2537653 59.7962014,10.1319473 C59.8526155,15.0140547 65.6574903,13.2158211 66.9135536,9.97505214 C70.2130987,13.4901279 77.1479547,10.3238236 75.8371766,5.29694317'
    id='Bom-Bom'
    fill='#FFFFFF'
    />
    <path
      d='M69.433817,15.3488372 C65.8038135,15.3488372 61.8287631,16.3747039 58.3068219,17.1662556 C54.7602589,17.9637144 51.2314783,18.9518414 47.8046046,20.1368717 C43.9335128,21.4748303 39.9926593,22.6897243 36.2145831,24.2580599 C32.9385178,25.6183343 28.7066484,26.6107275 26.7027087,29.6738881 C26.6986051,29.6824206 28.8051356,29.5856097 28.9754363,29.5573869 C30.0242559,29.3834556 30.9267127,29.0139335 31.8924339,28.6010927 C33.6392127,27.8548288 35.3740227,27.0823111 37.115672,26.3248894 C41.0202768,24.6265975 45.0951823,23.2295678 49.1044296,21.7708418 C53.1003402,20.3176946 57.1991835,19.0860638 61.3588973,18.1455219 C64.7498643,17.3792395 68.5409353,17.2322183 71.6624305,15.8729284 C71.5444511,15.9172317 71.6210523,15.887368 71.8918919,15.7816965 C71.813239,15.8197645 69.6708017,15.3488372 69.433817,15.3488372'
    id='Bidi'
    fill='#FFFFFF'
    />
    <path
      d='M22.1170349,20.8250383 C19.8478706,19.8267659 16.6395075,20.267131 14.2823358,20.5862701 C12.1385566,20.8762747 8.78577979,21.5209156 7.44867865,21.6937129 C6.34297926,21.8346967 3.04321074,22.7348497 2.4393915,21.4968044 C1.79003946,20.1656628 5.15878677,16.0935417 5.89512684,15.1341152 C9.20984643,10.8135143 13.1946457,6.01370248 17.9749097,3.21110531 C17.5345328,5.16177188 17.1498826,7.12884748 16.7479027,9.08721625 C16.5824216,9.89460802 15.79851,14.2513757 17.2338124,14.5946261 C17.8233602,14.735275 19.0058537,11.8121216 19.104055,11.5505816 C19.9270433,9.36181652 20.1163103,6.84118695 20.4951839,4.54927932 C20.7119744,3.23789558 21.6953468,0.499260193 19.8009787,0.0471743805 C18.1723317,-0.341619419 15.9646707,1.77447706 14.8776601,2.66558843 C11.5962407,5.35767573 8.75146029,8.61302846 6.0905101,11.8864646 C4.78025293,13.4985691 3.4761121,15.1200503 2.37686886,16.8831849 C1.47300888,18.3332083 -0.712565331,21.3735691 0.232070287,23.1651685 C1.92969602,26.3860287 10.0715716,23.5626691 12.663543,23.133355 C14.708781,22.7941232 16.8162019,22.4729748 18.8913421,22.7566168 C21.0534703,23.0519796 20.9484731,24.0274803 20.2298024,25.953031 C18.7520254,29.9126329 15.4539558,33.3136578 12.5058772,36.2726431 C11.2071731,37.5759898 6.1251694,43.6918736 4.84549395,40.3641872 C3.97595347,38.1030884 4.74525384,35.3289559 5.74561617,33.2503658 C6.21419621,32.2765394 6.8346655,31.3887768 7.32329357,30.4256666 C7.45649398,30.163122 8.12453486,28.0011471 8.27030776,27.9529247 C4.4003599,29.303824 2.15328244,34.538308 1.93751135,38.2397187 C1.83591206,39.9810863 2.07376995,42.5640033 3.73741598,43.6084889 C5.40174162,44.6533095 6.96378836,43.3814414 8.28661802,42.4230195 C11.9289019,39.7838429 15.1780407,36.4246779 18.1519439,33.0879497 C19.7778725,31.2635323 26.6054133,22.8014905 22.1170349,20.8250383'
    id='Bidi'
    fill='#FFFFFF'
      />
      </g>
      </g>
      </g>`;
      case Graphic.BEAR:
        return `
    <g
    id='Clothing/Graphic/Bear'
    mask='url(#${maskId})'
    stroke-width='1'
    fill-rule='evenodd'>
    <defs>
      <polygon
        id="bear_path"
    points='47.8916289 0.430054064 47.8916289 41.9204613 0.45345556 41.9204613 0.45345556 0.430054064 47.8916289 0.430054064'
      />
      </defs>
      <g transform='translate(77.000000, 58.000000)' id='Group-76'>
    <g transform='translate(31.000000, 1.000000)'>
    <mask id="bear_mask" fill='white'>
    <use href='#bear_path' />
      </mask>
      <path
    d='M34.4465814,31.5861741 C33.968066,34.4016565 30.3675711,38.9307388 27.2995299,35.97048 C23.8326433,32.6262918 31.5245196,31.1602447 28.7810969,28.3763859 C27.5974268,27.1761741 24.3685608,27.83928 22.7791175,27.96528 C21.289633,28.0833741 18.9133856,28.4198682 19.4834474,30.5900329 C19.9159423,32.2398918 21.1926433,31.5342918 21.8715711,32.4815153 C22.4069938,33.2301035 22.1798598,35.9887624 21.3826639,36.7492094 C20.3796124,37.7053271 19.193468,37.4123153 17.7975093,36.7996094 C14.6057567,35.39928 13.1212206,31.4868565 15.2228289,28.5829271 C19.3780454,22.8417741 35.2897979,21.5244565 34.4465814,31.5861741 M14.9853031,14.7244094 C15.9695505,14.2811859 17.132932,14.5213271 17.7173443,15.5110447 C18.5189938,16.8669035 17.3595711,18.6842682 15.7424165,18.4668565 C14.0347052,18.2365976 13.1726845,15.5481035 14.9853031,14.7244094 M32.9472,16.3787153 C32.0911175,19.3854212 27.3109113,16.0012094 30.0112825,14.08848 C31.5626227,12.9900565 33.4583753,14.5801271 32.9472,16.3787153 M47.8267052,23.16888 C47.5911588,20.7887153 46.9270763,18.5231859 45.7389526,16.4385035 C45.1164371,15.3460094 44.3464577,14.3276329 43.5492619,13.35768 C43.2483959,12.9930212 42.6432,12.34968 42.5264165,12.1446212 C41.8173031,10.90488 41.888066,11.7834212 42.2789938,10.3598682 C42.8950763,8.11657412 44.2262103,6.28043294 43.1627876,3.87062118 C42.1735918,1.62930353 39.5078598,0.576832941 37.1791175,0.443915294 C36.0914474,0.382150588 35.0017979,0.525938824 33.9720247,0.892574118 C32.9368082,1.26069176 31.9708701,2.53255059 31.0717361,2.79393882 C30.3740041,2.99702118 28.3872,2.33391529 27.6305814,2.28549176 C26.2356124,2.19605647 24.8332206,2.23558588 23.4387464,2.31859765 C21.8087258,2.41544471 18.0632412,3.58205647 16.5257567,3.19565647 C15.5108289,2.94069176 14.3172619,1.41485647 13.1449732,1.03191529 C11.852932,0.610432941 10.4673649,0.467632941 9.17829278,0.954338824 C6.89260206,1.81805647 5.17845773,4.50605647 5.12501443,6.89560941 C5.09680825,8.15066824 5.78018969,9.01092706 5.91528247,10.1469035 C6.11816907,11.8619859 6.35223093,10.5026682 5.42835464,11.8718682 C4.9577567,12.5690682 4.13284948,13.1822682 3.60385979,13.8537741 C0.778787629,17.4405741 -0.278696907,22.73208 0.974746392,27.0733976 C4.13284948,38.0131624 14.6423753,42.4676329 25.4136742,41.86728 C34.6648082,41.3509271 48.9782103,34.1491624 47.8267052,23.16888'
    id='Fill-74'
    fill='#FFFFFF'
    mask='url(#bear_mask)'
      />
      </g>
      </g>
      </g>`;
      case Graphic.SKULLOUTLINE:
        return `
    <g
    id='Clothing/Graphic/Skull-Outline'
    mask='url(#${maskId})'
    stroke-width='1'
    fill-rule='evenodd'>
    <g transform='translate(77.000000, 58.000000)' id='Skull'>
    <g transform='translate(36.000000, 1.000000)'>
    <path
      d='M36.3353853,17.0393673 C36.1551375,18.0777567 35.8927571,19.1087846 35.3466862,20.0285256 C34.6274709,21.2401241 33.3213404,21.7355025 32.5643885,22.8466397 C31.3705799,24.5990841 32.9555173,27.1231756 31.7781352,28.6842237 C30.5119611,30.3630542 27.6359878,29.346749 26.5163205,31.5755188 C25.3358307,33.925968 27.0535122,37.0679834 25.5729054,39.4058749 C24.1038415,39.0369393 23.6514462,33.5197923 21.3841422,37.0389708 C19.9319488,39.2928559 20.9126566,40.513981 18.7354764,37.2741024 C17.9789685,36.1486754 17.1247893,35.1384326 15.6392989,35.8810671 C14.5955488,36.4024269 14.3806721,38.7178013 13.4297097,38.9672226 C11.0962556,39.5786511 11.0105713,33.3500473 10.2172146,32.1709255 C9.78124586,31.523123 9.3750224,31.1394645 8.59986813,30.8939404 C7.92815655,30.6808931 6.73124021,31.1087199 6.17495825,30.7934791 C5.13032022,30.2019696 4.98914092,28.23821 4.94962847,27.1963565 C4.8759311,25.2572791 5.52544468,23.2770647 4.98958488,21.3579064 C4.53896541,19.7461945 3.1027546,18.7545717 2.63348882,17.1753366 C0.0993647654,8.64045187 11.6836147,3.89495204 18.2986197,3.63037495 C26.0390634,3.32033043 37.3356273,7.85191686 36.3353853,17.0393673 M38.172936,11.7240091 C36.716747,8.28277481 33.5180147,5.54736808 30.2580159,3.71135026 C28.6748543,2.82018878 26.955841,2.16285976 25.1728973,1.77140691 C23.531577,1.41069869 21.6150014,1.64842836 20.0482663,1.19462018 C18.728373,0.811827785 17.9097107,-0.0477230086 16.3562944,0.00207464445 C14.238161,0.0696262433 12.0370069,1.17426809 10.1879133,2.08751375 C6.5279065,3.89538507 3.41663437,6.23630778 1.4618782,9.83126531 C-0.636277062,13.6895009 -0.437382953,17.1865952 1.81083082,20.7832848 C3.95648981,24.2167247 0.837226393,29.045365 3.97824385,32.3086268 C5.29902504,33.6808769 6.59938407,32.6801606 7.84957562,33.3409538 C8.81163708,33.8493229 8.77123671,36.8034063 9.03805671,37.6694524 C10.2385247,41.5623298 14.5524847,43.0692501 16.5436456,38.8602659 C17.4830651,41.2128802 21.1990108,43.6239527 22.9291232,40.5460247 C24.0106099,41.9455552 25.8832335,42.541395 27.3123409,41.3414881 C28.6557641,40.2130299 28.8040467,37.585879 28.8724166,35.9984164 C28.9261357,34.7604034 28.3747373,33.2339971 29.3261438,32.3402375 C30.3734455,31.3559761 32.5204364,31.7708122 33.6951548,30.5011885 C35.0359141,29.0518603 34.4849597,27.3552759 34.5879584,25.6288129 C34.6922891,23.8828639 35.0035051,24.3280116 36.2980926,23.0670485 C39.1833891,20.2567287 39.6717451,15.2600755 38.172936,11.7240091'
    id='Boo!'
    fill='#FFFFFF'
    />
    <path
      d='M14.4199416,28.1214878 C16.4409104,26.2992475 16.0109503,20.7236412 15.8441352,18.1620823 C15.5281264,13.3012213 12.4886458,14.7535905 10.6384079,17.7811955 C9.23849255,20.072024 5.8716471,23.7758196 7.38120621,26.6576793 C8.58337131,28.9535919 12.5610889,29.793746 14.4199416,28.1214878'
    id='Boo!'
    fill='#FFFFFF'
    />
    <path
      d='M27.7905585,19.2735023 C26.7563456,17.3532744 26.3690889,17.0619469 25.1393718,15.4887787 C24.3486011,14.4775769 23.2406052,12.6869687 21.7361817,13.0471169 C19.1541926,13.6660822 20.2075808,19.6585611 20.238428,21.4487472 C20.2617717,22.8065864 19.9608037,24.2138247 21.0892255,25.1768941 C22.2443259,26.1627631 24.1422588,26.073676 25.5295464,25.8676357 C29.7877025,25.2338929 29.5913646,22.6136348 27.7905585,19.2735023'
    id='Boo!'
    fill='#FFFFFF'
    />
    <path
      d='M19.2396436,31.8300192 C18.9563137,31.7944779 19.3174042,31.4678578 19.3591799,31.2433628 C19.5473771,31.8574625 19.6855263,31.8862554 19.2396436,31.8300192 M20.2848621,27.5191742 C17.6678865,24.7492021 12.711061,33.9166085 16.2049116,34.9450569 C17.0098167,35.1821488 17.604189,34.5815458 18.3598733,34.4767215 C19.4596896,34.3242088 20.3837174,34.9635023 21.3333895,33.9548492 C22.8244907,32.3712368 21.5203459,28.8566969 20.2848621,27.5191742'
    id='Boo!'
    fill='#FFFFFF'
      />
      </g>
      </g>
      </g>`;
      case Graphic.SKULL:
        return `
    <g
    id='Clothing/Graphic/Skull'
    mask='url(#${maskId})'
    fill-rule='evenodd'
    fill='#FFFFFF'>
    <g transform='translate(77.000000, 58.000000)' id='Fill-49'>
    <path d='M65.2820354,19.9288113 C64.9841617,22.7059411 59.588846,24.2025715 57.6102394,22.3766824 C56.6984983,21.5350479 56.6825881,19.8029182 56.4815012,18.6751632 C56.1014249,16.5420247 55.8256486,14.4172496 55.7306295,12.2519776 C55.6727342,10.9274596 55.3253621,9.86749314 56.7745135,9.67029008 C57.65797,9.55055964 58.5675014,10.137767 59.2896464,10.6026028 C61.736719,12.1758255 65.6201265,16.7414286 65.2820354,19.9288113 M52.8813831,14.0756657 C53.1659984,16.901216 54.2014853,21.8145656 51.9457767,24.1810024 C49.9296045,26.2960933 45.7863308,24.19905 45.1631825,21.7084809 C44.3897714,18.6188195 47.4383369,14.9274245 49.307782,12.8387447 C49.881874,12.1969544 51.151594,10.4256483 52.1442119,11.018578 C52.526056,11.2461539 52.8367463,13.6301981 52.8813831,14.0756657 M54.3212536,25.1062722 C54.9678252,23.5832306 61.2342228,28.1246236 58.2744891,30.2850536 C57.7918806,30.6376421 54.1148633,31.7513112 53.4099544,31.2274906 C51.9250051,30.1235056 53.8408548,26.2630794 54.3212536,25.1062722 M73.3250687,17.5267194 C72.8817937,2.05112066 53.065234,-2.31331777 42.4756895,6.50447654 C38.426551,9.87585667 36.113389,14.0039155 36.0073212,19.2826191 C35.9171635,23.7544627 36.6256081,27.9718792 40.0409914,31.0465744 C41.5219631,32.379896 42.5004386,33.1955596 43.2862243,35.0170469 C44.1095756,36.9234899 44.4852324,39.3524331 46.0280771,40.8495037 C46.8788292,41.6752915 48.1176128,42.3417322 49.2940816,41.8091079 C51.455655,40.8301355 50.7644465,37.8320326 51.4194152,36.1606486 C53.4559171,41.1294616 58.6302582,42.7141291 59.5694002,36.4097935 C60.6000257,38.2286397 63.2945899,40.610483 65.268335,38.6195243 C66.0806376,37.8003393 66.2030575,36.4705391 66.3409457,35.3929652 C66.5857855,33.4807998 66.1601884,32.7294032 67.6955199,31.4180909 C71.7349355,27.9683578 73.4691441,22.7464381 73.3250687,17.5267194' />
      </g>
      </g>`;
      default:
        return '';
    }
  }

}
