/*
 * This file is part of ShieldBlaze ExpressGateway. [www.shieldblaze.com]
 * Copyright (c) 2020-2022 ShieldBlaze
 *
 * ShieldBlaze ExpressGateway is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * ShieldBlaze ExpressGateway is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with ShieldBlaze ExpressGateway.  If not, see <https://www.gnu.org/licenses/>.
 */
package com.shieldblaze.expressgateway.configuration.tls;

import com.fasterxml.jackson.annotation.JsonIgnore;

/**
 * Configuration for TLS Server (Internet--to--ExpressGateway)
 */
public final class TlsServerConfiguration extends TlsConfiguration {

    @JsonIgnore
    private boolean validated;

    /**
     * This is the default implementation of {@link TlsClientConfiguration}
     * which is disabled by default.
     * </p>
     * <p>
     * To enable this, call {@link #enabled()}.
     */
    @JsonIgnore
    public static final TlsServerConfiguration DEFAULT = new TlsServerConfiguration();

    static {
        DEFAULT.ciphers = IntermediateCrypto.CIPHERS;
        DEFAULT.protocols = IntermediateCrypto.PROTOCOLS;
        DEFAULT.useStartTLS = false;
        DEFAULT.sessionTimeout = 43_200;
        DEFAULT.sessionCacheSize = 1_000_000;
        DEFAULT.validated = true;
    }

    @Override
    public TlsConfiguration validate() {
        super.validate();
        validated = true;
        return this;
    }

    @Override
    public boolean validated() {
        return validated;
    }

    public static TlsServerConfiguration copyFrom(TlsServerConfiguration from) {
        from.validate();

        TlsServerConfiguration configuration = new TlsServerConfiguration();
        configuration.certificateKeyPairMap.putAll(from.certificateKeyPairMap);

        configuration.ciphers = from.ciphers;
        configuration.protocols = from.protocols;
        configuration.mutualTLS = from.mutualTLS;
        configuration.useStartTLS = from.useStartTLS;
        configuration.sessionTimeout = from.sessionTimeout;
        configuration.sessionCacheSize = from.sessionCacheSize;
        configuration.acceptAllCerts = from.acceptAllCerts;

        configuration.validate();
        return configuration;
    }
}
