// SPDX-FileCopyrightText: Copyright 2024 shadPS4 Emulator Project
// SPDX-License-Identifier: GPL-2.0-or-later

#include <optional>
#include <xxhash.h>
#include "common/assert.h"
#include "video_core/buffer_cache/buffer_cache.h"
#include "video_core/page_manager.h"
#include "video_core/renderer_vulkan/vk_instance.h"
#include "video_core/renderer_vulkan/vk_scheduler.h"
#include "video_core/texture_cache/host_compatibility.h"
#include "video_core/texture_cache/texture_cache.h"
#include "video_core/texture_cache/tile_manager.h"

namespace VideoCore {

static constexpr u64 PageShift = 12;
static constexpr u64 NumFramesBeforeRemoval = 32;

TextureCache::TextureCache(const Vulkan::Instance& instance_, Vulkan::Scheduler& scheduler_,
                           BufferCache& buffer_cache_, PageManager& tracker_)
    : instance{instance_}, scheduler{scheduler_}, buffer_cache{buffer_cache_}, tracker{tracker_},
      tile_manager{instance, scheduler} {
    ImageInfo info{};
    info.pixel_format = vk::Format::eR8G8B8A8Unorm;
    info.type = vk::ImageType::e2D;
    info.tiling_idx = u32(AmdGpu::TilingMode::Texture_MicroTiled);
    info.num_bits = 32;
    info.UpdateSize();
    const ImageId null_id = slot_images.insert(instance, scheduler, info);
    ASSERT(null_id.index == NULL_IMAGE_ID.index);
    auto& img = slot_images[null_id];
    const vk::Image& null_image = img.image;
    Vulkan::SetObjectName(instance.GetDevice(), null_image, "Null Image");
    img.flags = ImageFlagBits::Empty;
    img.track_addr = img.info.guest_address;
    img.track_addr_end = img.info.guest_address + img.info.guest_size_bytes;

    ImageViewInfo view_info;
    const auto null_view_id =
        slot_image_views.insert(instance, view_info, slot_images[null_id], null_id);
    ASSERT(null_view_id.index == NULL_IMAGE_VIEW_ID.index);
    const vk::ImageView& null_image_view = slot_image_views[null_view_id].image_view.get();
    Vulkan::SetObjectName(instance.GetDevice(), null_image_view, "Null Image View");
}

TextureCache::~TextureCache() = default;

void TextureCache::MarkAsMaybeDirty(ImageId image_id, Image& image) {
    if (image.hash == 0) {
        // Initialize hash
        const u8* addr = std::bit_cast<u8*>(image.info.guest_address);
        image.hash = XXH3_64bits(addr, image.info.guest_size_bytes);
    }
    image.flags |= ImageFlagBits::MaybeCpuDirty;
    UntrackImage(image_id);
}

void TextureCache::InvalidateMemory(VAddr addr, size_t size) {
    std::scoped_lock lock{mutex};
    const auto end = addr + size;
    const auto pages_start = PageManager::GetPageAddr(addr);
    const auto pages_end = PageManager::GetNextPageAddr(addr + size - 1);
    ForEachImageInRegion(pages_start, pages_end - pages_start, [&](ImageId image_id, Image& image) {
        const auto image_begin = image.info.guest_address;
        const auto image_end = image.info.guest_address + image.info.guest_size_bytes;
        if (image_begin < end && addr < image_end) {
            // Start or end of the modified region is in the image, or the image is entirely within
            // the modified region, so the image was definitely accessed by this page fault.
            // Untrack the image, so that the range is unprotected and the guest can write freely.
            image.flags |= ImageFlagBits::CpuDirty;
            UntrackImage(image_id);
        } else if (pages_end < image_end) {
            // This page access may or may not modify the image.
            // We should not mark it as dirty now. If it really was modified
            // it will receive more invalidations on its other pages.
            // Remove tracking from this page only.
            UntrackImageHead(image_id);
        } else if (image_begin < pages_start) {
            // This page access does not modify the image but the page should be untracked.
            // We should not mark this image as dirty now. If it really was modified
            // it will receive more invalidations on its other pages.
            UntrackImageTail(image_id);
        } else {
            // Image begins and ends on this page so it can not receive any more invalidations.
            // We will check it's hash later to see if it really was modified.
            MarkAsMaybeDirty(image_id, image);
        }
    });
}

void TextureCache::InvalidateMemoryFromGPU(VAddr address, size_t max_size) {
    std::scoped_lock lock{mutex};
    ForEachImageInRegion(address, max_size, [&](ImageId image_id, Image& image) {
        // Only consider images that match base address.
        // TODO: Maybe also consider subresources
        if (image.info.guest_address != address) {
            return;
        }
        // Ensure image is reuploaded when accessed again.
        image.flags |= ImageFlagBits::GpuDirty;
    });
}

void TextureCache::UnmapMemory(VAddr cpu_addr, size_t size) {
    std::scoped_lock lk{mutex};

    boost::container::small_vector<ImageId, 16> deleted_images;
    ForEachImageInRegion(cpu_addr, size, [&](ImageId id, Image&) { deleted_images.push_back(id); });
    for (const ImageId id : deleted_images) {
        // TODO: Download image data back to host.
        FreeImage(id);
    }
}

ImageId TextureCache::ResolveDepthOverlap(const ImageInfo& requested_info, BindingType binding,
                                          ImageId cache_image_id) {
    const auto& cache_image = slot_images[cache_image_id];

    if (!cache_image.info.IsDepthStencil() && !requested_info.IsDepthStencil()) {
        return {};
    }

    const bool stencil_match = requested_info.HasStencil() == cache_image.info.HasStencil();
    const bool bpp_match = requested_info.num_bits == cache_image.info.num_bits;

    // If an image in the cache has less slices we need to expand it
    bool recreate = cache_image.info.resources < requested_info.resources;

    switch (binding) {
    case BindingType::Texture:
        // The guest requires a depth sampled texture, but cache can offer only Rxf. Need to
        // recreate the image.
        recreate |= requested_info.IsDepthStencil() && !cache_image.info.IsDepthStencil();
        break;
    case BindingType::Storage:
        // If the guest is going to use previously created depth as storage, the image needs to be
        // recreated. (TODO: Probably a case with linear rgba8 aliasing is legit)
        recreate |= cache_image.info.IsDepthStencil();
        break;
    case BindingType::RenderTarget:
        // Render target can have only Rxf format. If the cache contains only Dx[S8] we need to
        // re-create the image.
        ASSERT(!requested_info.IsDepthStencil());
        recreate |= cache_image.info.IsDepthStencil();
        break;
    case BindingType::DepthTarget:
        // The guest has requested previously allocated texture to be bound as a depth target.
        // In this case we need to convert Rx float to a Dx[S8] as requested
        recreate |= !cache_image.info.IsDepthStencil();

        // The guest is trying to bind a depth target and cache has it. Need to be sure that aspects
        // and bpp match
        recreate |= cache_image.info.IsDepthStencil() && !(stencil_match && bpp_match);
        break;
    default:
        break;
    }

    if (recreate) {
        auto new_info{requested_info};
        new_info.resources = std::max(requested_info.resources, cache_image.info.resources);
        new_info.UpdateSize();
        const auto new_image_id = slot_images.insert(instance, scheduler, new_info);
        RegisterImage(new_image_id);

        // Inherit image usage
        auto& new_image = GetImage(new_image_id);
        new_image.usage = cache_image.usage;

        // TODO: perform a depth copy here

        FreeImage(cache_image_id);
        return new_image_id;
    }

    // Will be handled by view
    return cache_image_id;
}

std::tuple<ImageId, int, int> TextureCache::ResolveOverlap(const ImageInfo& image_info,
                                                           BindingType binding,
                                                           ImageId cache_image_id,
                                                           ImageId merged_image_id) {
    auto& tex_cache_image = slot_images[cache_image_id];
    // We can assume it is safe to delete the image if it wasn't accessed in some number of frames.
    const bool safe_to_delete =
        scheduler.CurrentTick() - tex_cache_image.tick_accessed_last > NumFramesBeforeRemoval;

    if (image_info.guest_address == tex_cache_image.info.guest_address) { // Equal address
        if (image_info.size != tex_cache_image.info.size) {
            // Very likely this kind of overlap is caused by allocation from a pool.
            if (safe_to_delete) {
                FreeImage(cache_image_id);
            }
            return {merged_image_id, -1, -1};
        }

        if (const auto depth_image_id = ResolveDepthOverlap(image_info, binding, cache_image_id)) {
            return {depth_image_id, -1, -1};
        }

        if (image_info.pixel_format != tex_cache_image.info.pixel_format ||
            image_info.guest_size_bytes <= tex_cache_image.info.guest_size_bytes) {
            auto result_id = merged_image_id ? merged_image_id : cache_image_id;
            const auto& result_image = slot_images[result_id];
            return {
                IsVulkanFormatCompatible(image_info.pixel_format, result_image.info.pixel_format)
                    ? result_id
                    : ImageId{},
                -1, -1};
        }

        ImageId new_image_id{};
        if (image_info.type == tex_cache_image.info.type) {
            ASSERT(image_info.resources > tex_cache_image.info.resources);
            new_image_id = ExpandImage(image_info, cache_image_id);
        } else {
            UNREACHABLE();
        }
        return {new_image_id, -1, -1};
    }

    // Right overlap, the image requested is a possible subresource of the image from cache.
    if (image_info.guest_address > tex_cache_image.info.guest_address) {
        if (auto mip = image_info.IsMipOf(tex_cache_image.info); mip >= 0) {
            return {cache_image_id, mip, -1};
        }

        if (auto slice = image_info.IsSliceOf(tex_cache_image.info); slice >= 0) {
            return {cache_image_id, -1, slice};
        }

        // TODO: slice and mip

        if (safe_to_delete) {
            FreeImage(cache_image_id);
        }

        return {{}, -1, -1};
    } else {
        // Left overlap, the image from cache is a possible subresource of the image requested
        if (!merged_image_id) {
            // We need to have a larger, already allocated image to copy this one into
            return {{}, -1, -1};
        }

        if (auto mip = tex_cache_image.info.IsMipOf(image_info); mip >= 0) {
            if (tex_cache_image.binding.is_target) {
                // We have a larger image created and a separate one, representing a subres of it,
                // bound as render target. In this case we need to rebind render target.
                tex_cache_image.binding.needs_rebind = 1u;
                GetImage(merged_image_id).binding.is_target = 1u;

                FreeImage(cache_image_id);
                return {merged_image_id, -1, -1};
            }

            tex_cache_image.Transit(vk::ImageLayout::eTransferSrcOptimal,
                                    vk::AccessFlagBits2::eTransferRead, {});

            const auto num_mips_to_copy = tex_cache_image.info.resources.levels;
            ASSERT(num_mips_to_copy == 1);

            auto& merged_image = slot_images[merged_image_id];
            merged_image.CopyMip(tex_cache_image, mip);

            FreeImage(cache_image_id);
        }
    }

    return {merged_image_id, -1, -1};
}

ImageId TextureCache::ExpandImage(const ImageInfo& info, ImageId image_id) {
    const auto new_image_id = slot_images.insert(instance, scheduler, info);
    RegisterImage(new_image_id);

    auto& src_image = slot_images[image_id];
    auto& new_image = slot_images[new_image_id];

    src_image.Transit(vk::ImageLayout::eTransferSrcOptimal, vk::AccessFlagBits2::eTransferRead, {});
    new_image.CopyImage(src_image);

    if (src_image.binding.is_bound || src_image.binding.is_target) {
        src_image.binding.needs_rebind = 1u;
    }

    FreeImage(image_id);

    TrackImage(new_image_id);
    new_image.flags &= ~ImageFlagBits::Dirty;
    return new_image_id;
}

ImageId TextureCache::FindImage(BaseDesc& desc, FindFlags flags) {
    const auto& info = desc.info;

    if (info.guest_address == 0) [[unlikely]] {
        return NULL_IMAGE_ID;
    }

    std::scoped_lock lock{mutex};
    boost::container::small_vector<ImageId, 8> image_ids;
    ForEachImageInRegion(info.guest_address, info.guest_size_bytes,
                         [&](ImageId image_id, Image& image) { image_ids.push_back(image_id); });

    ImageId image_id{};

    // Check for a perfect match first
    for (const auto& cache_id : image_ids) {
        auto& cache_image = slot_images[cache_id];
        if (cache_image.info.guest_address != info.guest_address) {
            continue;
        }
        if (False(flags & FindFlags::RelaxSize) &&
            cache_image.info.guest_size_bytes != info.guest_size_bytes) {
            continue;
        }
        if (False(flags & FindFlags::RelaxDim) && cache_image.info.size != info.size) {
            continue;
        }
        if (False(flags & FindFlags::RelaxFmt) &&
            !IsVulkanFormatCompatible(info.pixel_format, cache_image.info.pixel_format)) {
            continue;
        }
        if (True(flags & FindFlags::ExactFmt) &&
            info.pixel_format != cache_image.info.pixel_format) {
            continue;
        }
        ASSERT((cache_image.info.type == info.type || info.size == Extent3D{1, 1, 1} ||
                True(flags & FindFlags::RelaxFmt)));
        image_id = cache_id;
    }

    if (True(flags & FindFlags::NoCreate) && !image_id) {
        return {};
    }

    // Try to resolve overlaps (if any)
    int view_mip{-1};
    int view_slice{-1};
    if (!image_id) {
        for (const auto& cache_id : image_ids) {
            view_mip = -1;
            view_slice = -1;

            const auto& merged_info = image_id ? slot_images[image_id].info : info;
            std::tie(image_id, view_mip, view_slice) =
                ResolveOverlap(merged_info, desc.type, cache_id, image_id);
        }
    }

    if (image_id) {
        Image& image_resolved = slot_images[image_id];
        if (True(flags & FindFlags::ExactFmt) &&
            info.pixel_format != image_resolved.info.pixel_format) {
            // Cannot reuse this image as we need the exact requested format.
            image_id = {};
        } else if (image_resolved.info.resources < info.resources) {
            // The image was clearly picked up wrong.
            FreeImage(image_id);
            image_id = {};
            LOG_WARNING(Render_Vulkan, "Image overlap resolve failed");
        }
    }
    // Create and register a new image
    if (!image_id) {
        image_id = slot_images.insert(instance, scheduler, info);
        RegisterImage(image_id);
    }

    if (view_mip > 0) {
        desc.view_info.range.base.level = view_mip;
    }

    Image& image = slot_images[image_id];
    image.tick_accessed_last = scheduler.CurrentTick();

    return image_id;
}

ImageView& TextureCache::RegisterImageView(ImageId image_id, const ImageViewInfo& view_info) {
    Image& image = slot_images[image_id];
    if (const ImageViewId view_id = image.FindView(view_info); view_id) {
        return slot_image_views[view_id];
    }

    const ImageViewId view_id = slot_image_views.insert(instance, view_info, image, image_id);
    image.image_view_infos.emplace_back(view_info);
    image.image_view_ids.emplace_back(view_id);
    return slot_image_views[view_id];
}

ImageView& TextureCache::FindTexture(ImageId image_id, const ImageViewInfo& view_info) {
    Image& image = slot_images[image_id];
    UpdateImage(image_id);
    return RegisterImageView(image_id, view_info);
}

ImageView& TextureCache::FindRenderTarget(BaseDesc& desc) {
    const ImageId image_id = FindImage(desc);
    Image& image = slot_images[image_id];
    image.flags |= ImageFlagBits::GpuModified;
    image.usage.render_target = 1u;
    UpdateImage(image_id);

    // Register meta data for this color buffer
    if (!(image.flags & ImageFlagBits::MetaRegistered)) {
        if (desc.info.meta_info.cmask_addr) {
            surface_metas.emplace(desc.info.meta_info.cmask_addr,
                                  MetaDataInfo{.type = MetaDataInfo::Type::CMask});
            image.info.meta_info.cmask_addr = desc.info.meta_info.cmask_addr;
            image.flags |= ImageFlagBits::MetaRegistered;
        }

        if (desc.info.meta_info.fmask_addr) {
            surface_metas.emplace(desc.info.meta_info.fmask_addr,
                                  MetaDataInfo{.type = MetaDataInfo::Type::FMask});
            image.info.meta_info.fmask_addr = desc.info.meta_info.fmask_addr;
            image.flags |= ImageFlagBits::MetaRegistered;
        }
    }

    return RegisterImageView(image_id, desc.view_info);
}

ImageView& TextureCache::FindDepthTarget(BaseDesc& desc) {
    const ImageId image_id = FindImage(desc);
    Image& image = slot_images[image_id];
    image.flags |= ImageFlagBits::GpuModified;
    image.flags &= ~ImageFlagBits::Dirty;
    image.usage.depth_target = 1u;
    image.usage.stencil = image.info.HasStencil();

    // Register meta data for this depth buffer
    if (!(image.flags & ImageFlagBits::MetaRegistered)) {
        if (desc.info.meta_info.htile_addr) {
            surface_metas.emplace(desc.info.meta_info.htile_addr,
                                  MetaDataInfo{.type = MetaDataInfo::Type::HTile});
            image.info.meta_info.htile_addr = desc.info.meta_info.htile_addr;
            image.flags |= ImageFlagBits::MetaRegistered;
        }
    }

    return RegisterImageView(image_id, desc.view_info);
}

void TextureCache::RefreshImage(Image& image, Vulkan::Scheduler* custom_scheduler /*= nullptr*/) {
    if (False(image.flags & ImageFlagBits::Dirty)) {
        return;
    }

    if (image.info.num_samples > 1) {
        return;
    }

    if (True(image.flags & ImageFlagBits::MaybeCpuDirty) &&
        False(image.flags & ImageFlagBits::CpuDirty)) {
        // The image size should be less than page size to be considered MaybeCpuDirty
        // So this calculation should be very uncommon and reasonably fast
        // For now we'll just check up to 64 first pixels
        const auto addr = std::bit_cast<u8*>(image.info.guest_address);
        const auto w = std::min(image.info.size.width, u32(8));
        const auto h = std::min(image.info.size.height, u32(8));
        const auto size = w * h * image.info.num_bits / 8;
        const u64 hash = XXH3_64bits(addr, size);
        if (image.hash == hash) {
            image.flags &= ~ImageFlagBits::MaybeCpuDirty;
            return;
        }
        image.hash = hash;
    }

    const auto& num_layers = image.info.resources.layers;
    const auto& num_mips = image.info.resources.levels;
    ASSERT(num_mips == image.info.mips_layout.size());

    const bool is_gpu_modified = True(image.flags & ImageFlagBits::GpuModified);
    const bool is_gpu_dirty = True(image.flags & ImageFlagBits::GpuDirty);

    boost::container::small_vector<vk::BufferImageCopy, 14> image_copy{};
    for (u32 m = 0; m < num_mips; m++) {
        const u32 width = std::max(image.info.size.width >> m, 1u);
        const u32 height = std::max(image.info.size.height >> m, 1u);
        const u32 depth =
            image.info.props.is_volume ? std::max(image.info.size.depth >> m, 1u) : 1u;
        const auto& mip = image.info.mips_layout[m];

        // Protect GPU modified resources from accidental CPU reuploads.
        if (is_gpu_modified && !is_gpu_dirty) {
            const u8* addr = std::bit_cast<u8*>(image.info.guest_address);
            const u64 hash = XXH3_64bits(addr + mip.offset, mip.size);
            if (image.mip_hashes[m] == hash) {
                continue;
            }
            image.mip_hashes[m] = hash;
        }

        image_copy.push_back({
            .bufferOffset = mip.offset * num_layers,
            .bufferRowLength = static_cast<u32>(mip.pitch),
            .bufferImageHeight = static_cast<u32>(mip.height),
            .imageSubresource{
                .aspectMask = image.aspect_mask & ~vk::ImageAspectFlagBits::eStencil,
                .mipLevel = m,
                .baseArrayLayer = 0,
                .layerCount = num_layers,
            },
            .imageOffset = {0, 0, 0},
            .imageExtent = {width, height, depth},
        });
    }

    if (image_copy.empty()) {
        image.flags &= ~ImageFlagBits::Dirty;
        return;
    }

    auto* sched_ptr = custom_scheduler ? custom_scheduler : &scheduler;
    sched_ptr->EndRendering();

    const auto cmdbuf = sched_ptr->CommandBuffer();
    image.Transit(vk::ImageLayout::eTransferDstOptimal, vk::AccessFlagBits2::eTransferWrite, {},
                  cmdbuf);

    const VAddr image_addr = image.info.guest_address;
    const size_t image_size = image.info.guest_size_bytes;
    const auto [vk_buffer, buf_offset] =
        buffer_cache.ObtainViewBuffer(image_addr, image_size, is_gpu_dirty);
    // The obtained buffer may be written by a shader so we need to emit a barrier to prevent RAW
    // hazard
    if (auto barrier = vk_buffer->GetBarrier(vk::AccessFlagBits2::eTransferRead,
                                             vk::PipelineStageFlagBits2::eTransfer)) {
        const auto dependencies = vk::DependencyInfo{
            .dependencyFlags = vk::DependencyFlagBits::eByRegion,
            .bufferMemoryBarrierCount = 1,
            .pBufferMemoryBarriers = &barrier.value(),
        };
        cmdbuf.pipelineBarrier2(dependencies);
    }

    const auto [buffer, offset] = tile_manager.TryDetile(vk_buffer->Handle(), buf_offset, image);
    for (auto& copy : image_copy) {
        copy.bufferOffset += offset;
    }

    cmdbuf.copyBufferToImage(buffer, image.image, vk::ImageLayout::eTransferDstOptimal, image_copy);
    image.flags &= ~ImageFlagBits::Dirty;
}

vk::Sampler TextureCache::GetSampler(const AmdGpu::Sampler& sampler) {
    const u64 hash = XXH3_64bits(&sampler, sizeof(sampler));
    const auto [it, new_sampler] = samplers.try_emplace(hash, instance, sampler);
    return it->second.Handle();
}

void TextureCache::RegisterImage(ImageId image_id) {
    Image& image = slot_images[image_id];
    ASSERT_MSG(False(image.flags & ImageFlagBits::Registered),
               "Trying to register an already registered image");
    image.flags |= ImageFlagBits::Registered;
    ForEachPage(image.info.guest_address, image.info.guest_size_bytes,
                [this, image_id](u64 page) { page_table[page].push_back(image_id); });
}

void TextureCache::UnregisterImage(ImageId image_id) {
    Image& image = slot_images[image_id];
    ASSERT_MSG(True(image.flags & ImageFlagBits::Registered),
               "Trying to unregister an already unregistered image");
    image.flags &= ~ImageFlagBits::Registered;
    ForEachPage(image.info.guest_address, image.info.guest_size_bytes, [this, image_id](u64 page) {
        const auto page_it = page_table.find(page);
        if (page_it == nullptr) {
            UNREACHABLE_MSG("Unregistering unregistered page=0x{:x}", page << PageShift);
            return;
        }
        auto& image_ids = *page_it;
        const auto vector_it = std::ranges::find(image_ids, image_id);
        if (vector_it == image_ids.end()) {
            ASSERT_MSG(false, "Unregistering unregistered image in page=0x{:x}", page << PageShift);
            return;
        }
        image_ids.erase(vector_it);
    });
}

void TextureCache::TrackImage(ImageId image_id) {
    auto& image = slot_images[image_id];
    const auto image_begin = image.info.guest_address;
    const auto image_end = image.info.guest_address + image.info.guest_size_bytes;
    if (image_begin == image.track_addr && image_end == image.track_addr_end) {
        return;
    }

    if (!image.IsTracked()) {
        // Re-track the whole image
        image.track_addr = image_begin;
        image.track_addr_end = image_end;
        tracker.UpdatePagesCachedCount(image_begin, image.info.guest_size_bytes, 1);
    } else {
        if (image_begin < image.track_addr) {
            TrackImageHead(image_id);
        }
        if (image.track_addr_end < image_end) {
            TrackImageTail(image_id);
        }
    }
}

void TextureCache::TrackImageHead(ImageId image_id) {
    auto& image = slot_images[image_id];
    const auto image_begin = image.info.guest_address;
    if (image_begin == image.track_addr) {
        return;
    }
    ASSERT(image.track_addr != 0 && image_begin < image.track_addr);
    const auto size = image.track_addr - image_begin;
    image.track_addr = image_begin;
    tracker.UpdatePagesCachedCount(image_begin, size, 1);
}

void TextureCache::TrackImageTail(ImageId image_id) {
    auto& image = slot_images[image_id];
    const auto image_end = image.info.guest_address + image.info.guest_size_bytes;
    if (image_end == image.track_addr_end) {
        return;
    }
    ASSERT(image.track_addr_end != 0 && image.track_addr_end < image_end);
    const auto addr = image.track_addr_end;
    const auto size = image_end - image.track_addr_end;
    image.track_addr_end = image_end;
    tracker.UpdatePagesCachedCount(addr, size, 1);
}

void TextureCache::UntrackImage(ImageId image_id) {
    auto& image = slot_images[image_id];
    if (!image.IsTracked()) {
        return;
    }
    const auto addr = image.track_addr;
    const auto size = image.track_addr_end - image.track_addr;
    image.track_addr = 0;
    image.track_addr_end = 0;
    if (size != 0) {
        tracker.UpdatePagesCachedCount(addr, size, -1);
    }
}

void TextureCache::UntrackImageHead(ImageId image_id) {
    auto& image = slot_images[image_id];
    const auto image_begin = image.info.guest_address;
    if (!image.IsTracked() || image_begin < image.track_addr) {
        return;
    }
    const auto addr = tracker.GetNextPageAddr(image_begin);
    const auto size = addr - image_begin;
    image.track_addr = addr;
    if (image.track_addr == image.track_addr_end) {
        // This image spans only 2 pages and both are modified,
        // but the image itself was not directly affected.
        // Cehck its hash later.
        MarkAsMaybeDirty(image_id, image);
    }
    tracker.UpdatePagesCachedCount(image_begin, size, -1);
}

void TextureCache::UntrackImageTail(ImageId image_id) {
    auto& image = slot_images[image_id];
    const auto image_end = image.info.guest_address + image.info.guest_size_bytes;
    if (!image.IsTracked() || image.track_addr_end < image_end) {
        return;
    }
    ASSERT(image.track_addr_end != 0);
    const auto addr = tracker.GetPageAddr(image_end);
    const auto size = image_end - addr;
    image.track_addr_end = addr;
    if (image.track_addr == image.track_addr_end) {
        // This image spans only 2 pages and both are modified,
        // but the image itself was not directly affected.
        // Cehck its hash later.
        MarkAsMaybeDirty(image_id, image);
    }
    tracker.UpdatePagesCachedCount(addr, size, -1);
}

void TextureCache::DeleteImage(ImageId image_id) {
    Image& image = slot_images[image_id];
    ASSERT_MSG(!image.IsTracked(), "Image was not untracked");
    ASSERT_MSG(False(image.flags & ImageFlagBits::Registered), "Image was not unregistered");

    // Remove any registered meta areas.
    const auto& meta_info = image.info.meta_info;
    if (meta_info.cmask_addr) {
        surface_metas.erase(meta_info.cmask_addr);
    }
    if (meta_info.fmask_addr) {
        surface_metas.erase(meta_info.fmask_addr);
    }
    if (meta_info.htile_addr) {
        surface_metas.erase(meta_info.htile_addr);
    }

    // Reclaim image and any image views it references.
    scheduler.DeferOperation([this, image_id] {
        Image& image = slot_images[image_id];
        for (const ImageViewId image_view_id : image.image_view_ids) {
            slot_image_views.erase(image_view_id);
        }
        slot_images.erase(image_id);
    });
}

} // namespace VideoCore
