//  SPDX-FileCopyrightText: Copyright 2024 shadPS4 Emulator Project
//  SPDX-License-Identifier: GPL-2.0-or-later

#include "common/logging/log.h"
#include "core/libraries/kernel/file_system.h"
#include "logger.h"

namespace Core::Devices {

Logger::Logger(std::string prefix, bool is_err) : prefix(std::move(prefix)), is_err(is_err) {}

Logger::~Logger() = default;

s64 Logger::write(const void* buf, size_t nbytes) {
    log(static_cast<const char*>(buf), nbytes);
    return nbytes;
}

size_t Logger::writev(const Libraries::Kernel::SceKernelIovec* iov, int iovcnt) {
    for (int i = 0; i < iovcnt; i++) {
        log(static_cast<const char*>(iov[i].iov_base), iov[i].iov_len);
    }
    return iovcnt;
}

s64 Logger::pwrite(const void* buf, size_t nbytes, u64 offset) {
    log(static_cast<const char*>(buf), nbytes);
    return nbytes;
}

s32 Logger::fsync() {
    log_flush();
    return 0;
}

void Logger::log(const char* buf, size_t nbytes) {
    std::scoped_lock lock{mtx};
    const char* end = buf + nbytes;
    for (const char* it = buf; it < end; ++it) {
        char c = *it;
        if (c == '\r') {
            continue;
        }
        if (c == '\n') {
            log_flush();
            continue;
        }
        buffer.push_back(c);
    }
}

void Logger::log_flush() {
    std::scoped_lock lock{mtx};
    if (buffer.empty()) {
        return;
    }
    if (is_err) {
        LOG_ERROR(Tty, "[{}] {}", prefix, std::string_view{buffer});
    } else {
        LOG_INFO(Tty, "[{}] {}", prefix, std::string_view{buffer});
    }
    buffer.clear();
}

} // namespace Core::Devices