/*
 * Copyright 2020, Data61, CSIRO (ABN 41 687 119 230)
 *
 * SPDX-License-Identifier: GPL-2.0-only
 */

#pragma once

#include <util.h>
#include <linker.h>
#include <api/types.h>
#include <api/syscall.h>
#include <plat/machine/hardware.h>

/* seL4 is always in the top of memory, so the high bits of pointers are always 1.
   The autogenerated unpacking code doesn't know that, however, so will try to
   conditionally sign extend (in 64-bit mode), which wastes cycles in the fast
   path. Instead, we can do the unpacking ourselves and explicitly set the high
   bits. */

static inline tcb_t *endpoint_ptr_get_epQueue_tail_fp(endpoint_t *ep_ptr)
{
    uint64_t ret = ep_ptr->words[0] & 0xfffffffffffcull;
    return unlikely(ret) ? TCB_PTR(ret | PPTR_BASE) : NULL;
}

static inline vspace_root_t *cap_vtable_cap_get_vspace_root_fp(cap_t vtable_cap)
{
    return PML4E_PTR(vtable_cap.words[1]);
}

static inline word_t cap_pml4_cap_get_capPML4MappedASID_fp(cap_t vtable_cap)
{
    return (uint32_t)vtable_cap.words[0];
}

static inline void FORCE_INLINE switchToThread_fp(tcb_t *thread, vspace_root_t *vroot, pde_t stored_hw_asid)
{
    word_t new_vroot = pptr_to_paddr(vroot);
    /* the asid is the 12-bit PCID */
    asid_t asid = (asid_t)(stored_hw_asid.words[0] & 0xfff);
    cr3_t next_cr3 = makeCR3(new_vroot, asid);
    if (likely(getCurrentUserCR3().words[0] != next_cr3.words[0])) {
        SMP_COND_STATEMENT(tlb_bitmap_set(vroot, getCurrentCPUIndex());)
        setCurrentUserCR3(next_cr3);
    }

#ifdef ENABLE_SMP_SUPPORT
    asm volatile("movq %[value], %%gs:%c[offset]"
                 :
                 : [value] "r"(&thread->tcbArch.tcbContext.registers[Error + 1]),
                 [offset] "i"(OFFSETOF(nodeInfo_t, currentThreadUserContext)));
#endif /* ENABLE_SMP_SUPPORT */

    if (config_set(CONFIG_KERNEL_X86_IBPB_ON_CONTEXT_SWITCH)) {
        x86_ibpb();
    }

    if (config_set(CONFIG_KERNEL_X86_RSB_ON_CONTEXT_SWITCH)) {
        x86_flush_rsb();
    }

#ifdef CONFIG_BENCHMARK_TRACK_UTILISATION
    benchmark_utilisation_switch(NODE_STATE(ksCurThread), thread);
#endif

    NODE_STATE(ksCurThread) = thread;
}

#ifndef CONFIG_KERNEL_MCS
static inline void thread_state_ptr_set_blockingIPCDiminish_np(thread_state_t *ts_ptr, word_t dim)
{
    ts_ptr->words[1] = (ts_ptr->words[1] & 1) | dim;
}

static inline void mdb_node_ptr_mset_mdbNext_mdbRevocable_mdbFirstBadged(
    mdb_node_t *node_ptr, word_t mdbNext,
    word_t mdbRevocable, word_t mdbFirstBadged)
{
    node_ptr->words[1] = mdbNext | (mdbRevocable << 1) | mdbFirstBadged;
}

static inline void mdb_node_ptr_set_mdbPrev_np(mdb_node_t *node_ptr, word_t mdbPrev)
{
    node_ptr->words[0] = mdbPrev;
}
#endif

static inline bool_t isValidVTableRoot_fp(cap_t vspace_root_cap)
{
    /* Check the cap is a pml4_cap, and that it is mapped. The fields are next
       to each other, so they can be read and checked in parallel */
    return (vspace_root_cap.words[0] >> (64 - 6)) == ((cap_pml4_cap << 1) | 0x1);
}

static inline void fastpath_copy_mrs(word_t length, tcb_t *src, tcb_t *dest)
{
    word_t i;
    register_t reg;

    /* assuming that length < n_msgRegisters */
    for (i = 0; i < length; i ++) {
        /* assuming that the message registers simply increment */
        reg = msgRegisters[0] + i;
        setRegister(dest, reg, getRegister(src, reg));
    }
}

/* This is an accelerated check that msgLength, which appears
   in the bottom of the msgInfo word, is <= 4 and that msgExtraCaps
   which appears above it is zero. We are assuming that n_msgRegisters == 4
   for this check to be useful. By masking out the bottom 3 bits, we are
   really checking that n + 3 <= MASK(3), i.e. n + 3 <= 7 or n <= 4. */
compile_assert(n_msgRegisters_eq_4, n_msgRegisters == 4)
static inline int
fastpath_mi_check(word_t msgInfo)
{
    return ((msgInfo & MASK(seL4_MsgLengthBits + seL4_MsgExtraCapBits))
            + 3) & ~MASK(3);
}

static inline void NORETURN FORCE_INLINE fastpath_restore(word_t badge, word_t msgInfo, tcb_t *cur_thread)
{
    if (config_set(CONFIG_SYSENTER) && config_set(CONFIG_HARDWARE_DEBUG_API)
        && ((getRegister(NODE_STATE(ksCurThread), FLAGS) & FLAGS_TF) != 0)) {
        /* If single stepping using sysenter we need to do a return using iret to avoid
         * a race condition in restoring the flags (which enables stepping and interrupts) and
         * calling sysexit. This case is handled in restore_user_context so we just go there
         */
        restore_user_context();
    }
    NODE_UNLOCK;
    c_exit_hook();
    lazyFPURestore(cur_thread);

    if (config_set(CONFIG_KERNEL_SKIM_WINDOW)) {
        /* see restore_user_context for a full explanation of why we do this */
        word_t *irqstack = x64KSIRQStack[CURRENT_CPU_INDEX()];
        irqstack[0] = 0;
        irqstack[1] = 0;
        irqstack[2] = 0;
        irqstack[3] = 0;
        irqstack[4] = 0;
        irqstack[5] = 0;
    }

#ifdef CONFIG_HARDWARE_DEBUG_API
    restore_user_debug_context(cur_thread);
#endif

#ifdef ENABLE_SMP_SUPPORT
#ifdef CONFIG_KERNEL_SKIM_WINDOW
    word_t next_cr3 = MODE_NODE_STATE(x64KSCurrentUserCR3);
#endif
    swapgs();
#endif /* ENABLE_SMP_SUPPORT */

    if (config_set(CONFIG_KERNEL_X86_IBRS_BASIC)) {
        x86_disable_ibrs();
    }

    if (config_set(CONFIG_SYSENTER)) {
        cur_thread->tcbArch.tcbContext.registers[FLAGS] &= ~FLAGS_IF;

#if defined(ENABLE_SMP_SUPPORT) && defined(CONFIG_KERNEL_SKIM_WINDOW)
        register word_t next_cr3_r11 asm("r11") = next_cr3;
#endif
        asm volatile(
            "movq %%rcx, %%rsp\n"
            "popq %%rax\n"
            "popq %%rbx\n"
            "popq %%rbp\n"
            "popq %%r12\n"
            "popq %%r13\n"
            "popq %%r14\n"
            // Skip RDX, we need to put NextIP into it
            "addq $8, %%rsp\n"
            "popq %%r10\n"
            "popq %%r8\n"
            "popq %%r9\n"
            "popq %%r15\n"
            // restore RFLAGS
            "popfq\n"
            // reset interrupt bit
            "orq %[IF], -8(%%rsp)\n"
            // Restore NextIP
            "popq %%rdx\n"
            // skip Error
            "addq $8, %%rsp\n"
            // restore RSP
            "popq %%rcx\n"
            // Skip FaultIP
            "addq $8, %%rsp\n"
#if defined(ENABLE_SMP_SUPPORT) && defined(CONFIG_KERNEL_SKIM_WINDOW)
            "popq %%rsp\n"
            "movq %%r11, %%cr3\n"
            "movq %%rsp, %%r11\n"
#else
            "popq %%r11\n"
#ifdef CONFIG_KERNEL_SKIM_WINDOW
            "movq (x64KSCurrentUserCR3), %%rsp\n"
            "movq %%rsp, %%cr3\n"
#endif /* CONFIG_KERNEL_SKIM_WINDOW */
#endif /* defined(ENABLE_SMP_SUPPORT) && defined(CONFIG_KERNEL_SKIM_WINDOW) */
            "sti\n"
            SYSEXITQ"\n"
            :
            : "c"(&cur_thread->tcbArch.tcbContext.registers[RAX]),
            "D"(badge),
            "S"(msgInfo),
#if defined(ENABLE_SMP_SUPPORT) && defined(CONFIG_KERNEL_SKIM_WINDOW)
            "r"(next_cr3_r11),
#endif
            [IF] "i"(FLAGS_IF)
            : "memory"
        );
    } else {
        asm volatile(
            // Set our stack pointer to the top of the tcb so we can efficiently pop
            "movq %0, %%rsp\n"
            "popq %%rax\n"
            "popq %%rbx\n"
            "popq %%rbp\n"
            "popq %%r12\n"
            "popq %%r13\n"
            "popq %%r14\n"
            "popq %%rdx\n"
            "popq %%r10\n"
            "popq %%r8\n"
            "popq %%r9\n"
            "popq %%r15\n"
            //restore RFLAGS
            "popq %%r11\n"
            // Restore NextIP
#if defined(ENABLE_SMP_SUPPORT) && defined(CONFIG_KERNEL_SKIM_WINDOW)
            "popq %%rsp\n"
            "movq %%rcx, %%cr3\n"
            "movq %%rsp, %%rcx\n"
#else
            "popq %%rcx\n"
#ifdef CONFIG_KERNEL_SKIM_WINDOW
            "movq (x64KSCurrentUserCR3), %%rsp\n"
            "movq %%rsp, %%cr3\n"
#endif /* CONFIG_KERNEL_SKIM_WINDOW */
#endif /* defined(ENABLE_SMP_SUPPORT) && defined(CONFIG_KERNEL_SKIM_WINDOW) */
            // clear RSP to not leak information to the user
            "xor %%rsp, %%rsp\n"
            // More register but we can ignore and are done restoring
            // enable interrupt disabled by sysenter
            "sysretq\n"
            :
            : "r"(&cur_thread->tcbArch.tcbContext.registers[RAX]),
            "D"(badge),
            "S"(msgInfo)
#if defined(ENABLE_SMP_SUPPORT) && defined(CONFIG_KERNEL_SKIM_WINDOW)
            , "c"(next_cr3)
#endif
            : "memory"
        );
    }
    UNREACHABLE();
}

