#!/bin/bash

echo Check for lua interpreter.
which lua > /dev/null || exit 1

echo Check for python interpreter.
which python3 > /dev/null || exit 1

# echo Load shared library in lua.
# ./load_shared_lib.lua || exit 1

# Valgrind always report the file descriptors stdin,
# stdout and stderr as leaked ones. Maybe the compiler
# or glibc does not close them for optimisation reasons.
# This function receive the full Valgrind output and
# checks if there are only the 3 standard file descriptors
# that are leaked.
function only_std_fd() {
    echo "$@" |grep -c "3 open at exit" > /dev/null
    LEAKED_MORE=$?
    if [ ${LEAKED_MORE} == "1" ]; then
        echo "$@" |grep -c "4 open at exit" > /dev/null
        LEAKED_MORE_THAN_4=$?
        if [ ${LEAKED_MORE_THAN_4} == "1" ]; then
            return 0
        else
            # We have exactly 4 open file descriptors.
            # If you run this script directly from bash
            # instead of from the Makefile you maybe have
            # a specition 4th file descriptor which
            # is used for bash completion scripts (tab).
            # This case  is handled like the 3 default fds.
            echo "$@" |grep -c "bash-completion" > /dev/null
            NOT_INHERITED_FROM_BASH=$?
            if [ ${NOT_INHERITED_FROM_BASH} == "1" ]; then
                return 0
            else
                return 1
            fi
        fi
    else
        return 1
    fi
}

# We want to check the debug and the release build.
TARGETS="debug release"
# List of continuous test binaries we want to check.
PROGRAMS="list tree priority_queue load_shared_lib async_call"
PROGRAMS="list tree priority_queue async_call"
# Valgrind has some problems to track file descriptors
# and to make a full memory leak check at the same time.
# So we run those 2 tests separate.
declare -a CHECKS=( "--leak-check=full --track-origins=yes --show-reachable=yes" "--track-fds=yes" "--tool=helgrind --free-is-write=yes" "--tool=drd --exclusive-threshold=50 --shared-threshold=50 --check-stack-var=yes --free-is-write=yes")
# "--tool=exp-sgcheck"
# If set to 1, we suppress some output (stdin, stdout, stderr) on the "track-fds" check
VALGRIND_SUPPRESS_STD_FD=1
# We want to silence Valgrind if there are no leaks/etc... (-q, --suppressions=valgrind_suppress).
# "--suppressions=valgrind_suppress" is a list of warnings, generated by the glibc, which
# we want to ignore because we are not responsible for this warnings.
# The default is, that Valgrinds returns the exit code of the binary as its own exit code but
# "--error-exitcode=1" gives us an error on memory leaks and other warnings even when the
# program has the exit code "0" which is what we want for automated testing.
VALGRIND_OPTIONS="-q --suppressions=valgrind_suppress --error-exitcode=1"

for BINARY in ${PROGRAMS}; do
    echo "module: ${BINARY}"
    for TARGET in ${TARGETS}; do
        echo "  build target: ${TARGET}"
        I=0
        while [ "${CHECKS[${I}]}" != "" ]; do
            # Build the Valgrind command we want to execute for
            # the current test.
            VALGRIND_PARAMS="${VALGRIND_OPTIONS} ${CHECKS[${I}]} ./test_${BINARY}_${TARGET}"
            echo "    runtime check: valgrind ${VALGRIND_PARAMS}"
            if [ "${VALGRIND_SUPPRESS_STD_FD}" == "1" -a "${CHECKS[${I}]}" == "--track-fds=yes" ]; then
                # On the ""--track-fds=yes" test we silence the
                # output and error code if only the stdin, stdout
                # and stderr are leaked.
                VALGRIND_OUTPUT=$(valgrind ${VALGRIND_PARAMS} 2>&1)
                only_std_fd "${VALGRIND_OUTPUT}"
                NO_FD_LEAK=$?
                if [ "${NO_FD_LEAK}" == "1" ]; then
                    #BINARY_OUTPUT=$(echo "${VALGRIND_OUTPUT}" |egrep -v "^==")
                    #echo "${BINARY_OUTPUT}"
                    true
                else
                    # Found additional file descriptor leaks.
                    echo "${VALGRIND_OUTPUT}"
                    exit 1
                fi
            else
                # All the other tests.
                valgrind ${VALGRIND_PARAMS} || exit 1
            fi
            let I=I+1
        done
    done
done
