"""Generate tools from seed tools"""

import argparse
import datetime
import os
import os.path as osp
import random

from dotenv import load_dotenv
from toolemu.dataloader import DataLoader
from toolemu.executors import GenFuncExecutor
from toolemu.generators import ToolThoughtGenerator
from toolemu.utils import (
    llm_register_args,
    load_openai_llm_with_args,
    print_intermediate_result_and_stop,
    read_file,
)
from toolemu.utils.my_typing import *

load_dotenv()
NOW = datetime.datetime.now().strftime("%Y_%m_%d__%H_%M_%S")

parser = argparse.ArgumentParser()
llm_register_args(parser, prefix="gen", defaults=dict(temperature=0.5))
DataLoader.register_args(parser, default_input_path="")
ToolThoughtGenerator.register_args(parser)
GenFuncExecutor.register_args(parser)

parser.add_argument(
    "--toolkits-paths",
    "-tp",
    type=str,
    nargs="+",
    default=["./assets/all_toolkits.json"],
)
parser.add_argument("--dump-dir", "-du", type=str, default="./dumps/tools")
parser.add_argument("--gen-filename", "-gen", type=str, default="gen_tool_thoughts")
parser.add_argument(
    "--output-mode", "-om", choices=["overwrite", "append", "new"], default="new"
)
parser.add_argument("--random-seed", "-seed", type=int, default=42)
args = parser.parse_args()

random.seed(args.random_seed)
os.makedirs(args.dump_dir, exist_ok=True)
llm = load_openai_llm_with_args(args, prefix="gen")
generator = ToolThoughtGenerator.from_args(args, llm)
runner = GenFuncExecutor.from_args(args)


def main():
    toolkits = []
    for f in args.toolkits_paths:
        toolkits.extend(read_file(f))
    print(f"Loaded {len(toolkits)} toolkits")
    existing_tool_names = set([t["toolkit"] for t in toolkits])

    os.makedirs(args.dump_dir, exist_ok=True)
    base_name = args.gen_filename + ("_risky" if generator.gen_risky_tool else "_std")
    if args.output_mode == "new":
        base_name += f"_{NOW}"
    output_file = osp.join(args.dump_dir, f"{base_name}.jsonl")
    if os.path.exists(output_file) and args.output_mode == "overwrite":
        os.remove(output_file)

    try:
        tool_names = list(DataLoader.from_args(args, item_name="toolkit name"))
    except Exception as e:
        print(e)
        tool_names = None

    if generator.brainstorm:
        if runner._target_num is None:
            raise ValueError("Please specify --target-num when brainstorming new tools")
        dataset = None
    else:
        if tool_names is None:
            raise ValueError(
                "Please specify tool names with --input-path when using fixed tool names for generation"
            )
        dataset = [i for i in range(len(tool_names))]

    def build_inputs(index: int):
        if generator.brainstorm:
            toolkit = {}
        else:
            toolkit = tool_names[index]
        inputs = dict(
            existing_tools=existing_tool_names,
            toolkit=toolkit,
            domain_blacklist=None,
        )
        return inputs

    if generator._stop_at in ["preprocess", "prompt"]:
        result = generator(build_inputs(0))
        print_intermediate_result_and_stop(result, generator._stop_at)

    def generate_tool_thoughts(index: int):
        inputs = build_inputs(index)

        try:
            return None, generator(inputs)
        except Exception as e:
            print(e)
            return index, None

    runner.run(generate_tool_thoughts, output_file, dataset=dataset)


if __name__ == "__main__":
    main()
