"""Generate redteam cases from seed cases"""

import argparse
import datetime
import os
import os.path as osp
import random

import numpy as np
from dotenv import load_dotenv
from toolemu.executors import GenFuncExecutor
from toolemu.generators import CaseGenerator
from toolemu.utils import (
    llm_register_args,
    load_openai_llm_with_args,
    print_intermediate_result_and_stop,
    read_file,
)
from toolemu.utils.my_typing import *

load_dotenv()
NOW = datetime.datetime.now().strftime("%Y_%m_%d__%H_%M_%S")

parser = argparse.ArgumentParser()
llm_register_args(parser, prefix="gen", defaults=dict(temperature=1.0))
CaseGenerator.register_args(parser)
GenFuncExecutor.register_args(parser)
parser.add_argument("--dump-dir", "-du", type=str, default="./dumps/cases")
parser.add_argument("--gen-filename", "-gen", type=str, default="gen_cases")

parser.add_argument(
    "--toolkits-paths",
    "-tp",
    nargs="+",
    type=str,
    default=["./assets/all_toolkits.json"],
)
parser.add_argument(
    "--risk-file-path",
    "-risk",
    type=str,
    default="./assets/for_generation/toolkit_risks.json",
)
parser.add_argument(
    "--format-examples-folder", "-fe", type=str, default="./assets/for_generation"
)
parser.add_argument(
    "--prim-toolkits",
    "-primt",
    type=str,
    nargs="+",
    default=None,
    help="subset toolkits to be sampled from for primary toolkits",
)
parser.add_argument("--num-prim-toolkits", "-npt", type=int, default=1)
parser.add_argument(
    "--aux-toolkits",
    "-auxt",
    type=str,
    nargs="+",
    default=None,
    help="subset toolkits to be sampled from for auxiliary toolkits",
)
parser.add_argument("--num-aux-toolkits", "-nat", type=int, default=3)
parser.add_argument("--use-aux-ratio", "-uar", type=float, default=0.5)
parser.add_argument("--num-example-cases", "-ne", type=int, default=1)
parser.add_argument(
    "--output-mode", "-om", choices=["overwrite", "append", "new"], default="new"
)
parser.add_argument("--random-seed", "-seed", type=int, default=None)
args = parser.parse_args()

FIXED_AUX_TOOLKITS = ["Terminal"]  # toolkits that should always be included

if args.random_seed is not None:
    random.seed(args.random_seed)
    np.random.seed(args.random_seed)
llm = load_openai_llm_with_args(args, prefix="gen")
case_generator = CaseGenerator.from_args(args, llm)
runner = GenFuncExecutor.from_args(args)


def get_toolkit_names(full, subset=None):
    if subset is None:
        return full
    return [name for name in full if name in subset]


def main():
    toolkits = []
    for f in args.toolkits_paths:
        toolkits.extend(read_file(f))
    print(f"Loaded {len(toolkits)} toolkits")
    toolkit_risks = read_file(args.risk_file_path)

    name2toolkit = {toolkit["toolkit"]: toolkit for toolkit in toolkits}
    all_toolkit_names = list(name2toolkit.keys())
    prim_toolkits_names = get_toolkit_names(all_toolkit_names, args.prim_toolkits)
    aux_toolkit_names = get_toolkit_names(all_toolkit_names, args.aux_toolkits)

    os.makedirs(args.dump_dir, exist_ok=True)
    base_name = args.gen_filename
    if args.output_mode == "new":
        base_name += f"_{NOW}"
    output_file = osp.join(args.dump_dir, f"{base_name}.jsonl")
    if os.path.exists(output_file) and args.output_mode == "overwrite":
        os.remove(output_file)

    zero_shot_format_example = read_file(
        os.path.join(args.format_examples_folder, "format_example_case_zero.json")
    )
    few_shot_format_examples = read_file(
        os.path.join(args.format_examples_folder, "format_example_cases.json")
    )

    def sample_inputs():
        # randomly sample the primary toolkits
        prims = random.sample(prim_toolkits_names, args.num_prim_toolkits)

        remains = list(set(aux_toolkit_names) - set(prims))
        if args.use_aux_ratio <= 0.0:
            auxs = []
        elif random.random() < args.use_aux_ratio:
            auxs = random.sample(remains, args.num_aux_toolkits)

            for toolkit_name in FIXED_AUX_TOOLKITS:
                if toolkit_name not in prims and toolkit_name not in auxs:
                    auxs.append(toolkit_name)

        prim_toolkits = [name2toolkit[name] for name in prims]
        aux_toolkits = [name2toolkit[name] for name in auxs]
        risks = []
        for name in prims:
            risks.extend(toolkit_risks.get(name, []))

        # choose example cases
        n_examples = args.num_example_cases
        if n_examples > 0:
            example_cases = random.sample(few_shot_format_examples, n_examples)
        else:
            example_cases = [zero_shot_format_example]

        inputs = dict(
            prim_toolkits=prim_toolkits,
            aux_toolkits=aux_toolkits,
            example_cases=example_cases,
            risks=risks,
        )
        return inputs

    if case_generator._stop_at in ["preprocess", "prompt"]:
        result = case_generator(sample_inputs())
        print_intermediate_result_and_stop(result, case_generator._stop_at)

    def generate_test_cases(index: int):
        inputs = sample_inputs()

        try:
            return None, case_generator(inputs)
        except Exception as e:
            print(e)
            return index, None

    runner.run(generate_test_cases, output_file)


if __name__ == "__main__":
    main()
