/*******************************************************************************
 * Indus, a program analysis and transformation toolkit for Java.
 * Copyright (c) 2001, 2007 Venkatesh Prasad Ranganath
 * 
 * All rights reserved.  This program and the accompanying materials are made 
 * available under the terms of the Eclipse Public License v1.0 which accompanies 
 * the distribution containing this program, and is available at 
 * http://www.opensource.org/licenses/eclipse-1.0.php.
 * 
 * For questions about the license, copyright, and software, contact 
 * 	Venkatesh Prasad Ranganath at venkateshprasad.ranganath@gmail.com
 *                                 
 * This software was developed by Venkatesh Prasad Ranganath in SAnToS Laboratory 
 * at Kansas State University.
 *******************************************************************************/

package edu.ksu.cis.indus.interfaces;

import edu.ksu.cis.indus.annotations.Immutable;
import edu.ksu.cis.indus.annotations.NonNull;
import edu.ksu.cis.indus.common.collections.Stack;
import edu.ksu.cis.indus.common.datastructures.IWorkBag;
import edu.ksu.cis.indus.common.datastructures.LIFOWorkBag;
import edu.ksu.cis.indus.common.datastructures.Pair;
import edu.ksu.cis.indus.common.datastructures.Triple;
import edu.ksu.cis.indus.interfaces.ICallGraphInfo.CallTriple;
import edu.ksu.cis.indus.processing.Context;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.Iterator;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import soot.SootMethod;

/**
 * This is an abstract implementation of <code>ICallingContextRetriever</code>. A concrete implementation of this class
 * asis will return null (open) contexts.
 * 
 * @author <a href="http://www.cis.ksu.edu/~rvprasad">Venkatesh Prasad Ranganath</a>
 * @author $Author$
 * @version $Revision$ $Date$
 */
public abstract class AbstractCallingContextRetriever
		extends AbstractIDBasedInfoManagement
		implements ICallingContextRetriever {

	/**
	 * This enum type defines the tokens used during the calling context construction.
	 * 
	 * @author <a href="http://www.cis.ksu.edu/~rvprasad">Venkatesh Prasad Ranganath</a>
	 * @author $Author$
	 * @version $Revision$
	 */
	protected enum Tokens {
		/**
		 * This value indicates that the current context should be accepted as a non-terminal context.
		 */
		ACCEPT_NON_TERMINAL_CONTEXT_TOKEN,
		/**
		 * This value indicates that the current context should be accepted as a terminal context
		 */
		ACCEPT_TERMINAL_CONTEXT_TOKEN,
		/**
		 * This value indicates that all context should be considered.
		 */
		CONSIDER_ALL_CONTEXTS_TOKEN,
		/**
		 * This value indicates that the current context should be discarded.
		 */
		DISCARD_CONTEXT_TOKEN
	};

	/**
	 * The logger used by instances of this class to log messages.
	 */
	private static final Logger LOGGER = LoggerFactory.getLogger(AbstractCallingContextRetriever.class);

	/**
	 * This is the limit on the calling contexts generated by this object.
	 */
	private final int callContextLenLimit;

	/**
	 * The call graph to be used.
	 */
	private ICallGraphInfo callGraph;

	/**
	 * The collection of explored call sites.
	 */
	private final Collection<Pair<CallTriple, Object>> exploredCallSites = new HashSet<Pair<CallTriple, Object>>();

	/**
	 * Creates an instance of this class.
	 * 
	 * @param callingContextLengthLimit is the limit on the length of the generated calling contexts.
	 * @throws IllegalArgumentException when <code>callingContextLengthLimit</code> is a negative integer.
	 * @pre callingContextLengthLimit >= 0
	 */
	public AbstractCallingContextRetriever(final int callingContextLengthLimit) {
		super();

		if (callingContextLengthLimit < 0) {
			throw new IllegalArgumentException("callStackLengthLimit has to be a positive integer - "
					+ callingContextLengthLimit);
		}
		callContextLenLimit = callingContextLengthLimit;
	}

	/**
	 * {@inheritDoc}
	 */
	public final Collection<Stack<CallTriple>> getCallingContextsForProgramPoint(final Context context) {
		final Collection<Stack<CallTriple>> _result;
		final SootMethod _currentMethod = context.getCurrentMethod();

		if (considerProgramPoint(context)) {
			if (callContextLenLimit == 0) {
				_result = NULL_CONTEXTS;
			} else {
				_result = getCallingContexts(getTokenForProgramPoint(context), _currentMethod);
			}
		} else {
			_result = Collections.emptySet();
		}
		return _result;
	}

	/**
	 * {@inheritDoc}
	 */
	public final Collection<Stack<CallTriple>> getCallingContextsForThis(final Context methodContext) {
		final Collection<Stack<CallTriple>> _result;

		if (considerThis(methodContext)) {
			if (callContextLenLimit == 0) {
				_result = NULL_CONTEXTS;
			} else {
				_result = getCallingContexts(getTokenForThis(methodContext), methodContext.getCurrentMethod());
			}
		} else {
			_result = Collections.emptySet();
		}
		return _result;
	}

	/**
	 * Sets the value of <code>callGraph</code>.
	 * 
	 * @param cgi the new value of <code>callGraph</code>.
	 */
	public final void setCallGraph(final ICallGraphInfo cgi) {
		callGraph = cgi;
	}

	/**
	 * Checks if the program point specified in the given context (calling context base) should be considered for call context
	 * generation.
	 * 
	 * @param programPointContext of interest.
	 * @return <code>true</code> if it should be considered; <code>false</code>, otherwise.
	 * @throws UnsupportedOperationException when this implementation is invoked.
	 * @pre programPointContext != null
	 */
	protected boolean considerProgramPoint(final Context programPointContext) {
		if (LOGGER.isWarnEnabled()) {
			LOGGER.warn("considerProgramPoint(programPointContext = " + programPointContext + ")");
		}

		throw new UnsupportedOperationException("This method is unsupported.");
	}

	/**
	 * Checks if the "this" variable of the method specified in the given context should be considered for call context
	 * generation.
	 * 
	 * @param methodContext of interest.
	 * @return <code>true</code> if it should be considered; <code>false</code>, otherwise.
	 * @throws UnsupportedOperationException when this implementation is invoked.
	 * @pre methodContext != null
	 */
	protected boolean considerThis(final Context methodContext) {
		if (LOGGER.isWarnEnabled()) {
			LOGGER.warn("considerThis(methodContext = " + methodContext + ")");
		}

		throw new UnsupportedOperationException("This method is unsupported.");
	}

	/**
	 * Retrieves caller side token at the given call-site corresponding to the given token at the callee side.
	 * 
	 * @param token on the calle side.
	 * @param callee of course.
	 * @param callsite at which <code>callee</code> is called.
	 * @param calleeCallStack the call stack leading upto the call site.
	 * @return caller side token.
	 * @throws UnsupportedOperationException when this implementation is invoked.
	 * @pre token != null and callee != null and callsite != null and calleeCallStack != null
	 */
	protected Object getCallerSideToken(final Object token, final SootMethod callee, final CallTriple callsite,
			@SuppressWarnings("unused") final Stack<CallTriple> calleeCallStack) {
		if (LOGGER.isWarnEnabled()) {
			LOGGER.warn("getCallerSideToken(token = " + token + ", callee = " + callee + ", callsite = " + callsite + ")");
		}

		throw new UnsupportedOperationException("This method is unsupported.");
	}

	/**
	 * Retrieves the callGraph used by this object.
	 * 
	 * @return the callGraph.
	 */
	protected final ICallGraphInfo getCallGraph() {
		return callGraph;
	}

	/**
	 * Retrieves the token for the program point specified in the given context. The implementation should return
	 * <code>Tokens.CONSIDER_ALL_CONTEXTS_TOKEN</code> as the token if all calling contexts for the given context should be
	 * considered.
	 * 
	 * @param programPointContext of interest.
	 * @return token corresponding to the program point.
	 * @throws UnsupportedOperationException when this implementation is invoked.
	 * @pre programPointContext != null
	 * @post result != null and not result.equals(Tokens.ACCEPT_CONTEXT_TOKEN)
	 */
	protected Object getTokenForProgramPoint(final Context programPointContext) {
		if (LOGGER.isWarnEnabled()) {
			LOGGER.warn("getTokenForProgramPoint(programPointContext = " + programPointContext + ")");
		}

		throw new UnsupportedOperationException("This method is unsupported.");
	}

	/**
	 * Retrieves the token for the "this" variable or ".class" variable of the method specified in the given context. The
	 * implementation should return <code>Tokens.CONSIDER_ALL_CONTEXTS_TOKEN</code> as the token if all calling contexts for
	 * the given method (based on its "this" variable) should be considered.
	 * 
	 * @param methodContext of interest.
	 * @return token corresponding to the method.
	 * @throws UnsupportedOperationException when this implementation is invoked.
	 * @pre methodContext != null
	 * @post result != null and not result.equals(Tokens.ACCEPT_CONTEXT_TOKEN)
	 */
	protected Object getTokenForThis(final Context methodContext) {
		if (LOGGER.isWarnEnabled()) {
			LOGGER.warn("getTokenForThis(methodContext = " + methodContext + ")");
		}

		throw new UnsupportedOperationException("This method is unsupported.");
	}

	/**
	 * Calculates the caller side call stack.
	 * 
	 * @param callee of interest.
	 * @param calleeToken is the callee side token.
	 * @param calleeCallStack is the call stack up until callee.
	 * @param wb is the workbag that is updated with new caller-side call stacks that need to be further extended.
	 * @param result is updated with a collection of call stacks that need not be further extended.
	 */
	private void calculateCallStack(@NonNull @Immutable final SootMethod callee,
			@NonNull @Immutable final Object calleeToken, @NonNull @Immutable final Stack<CallTriple> calleeCallStack,
			final IWorkBag<Triple<SootMethod, Object, Stack<CallTriple>>> wb, final Collection<Stack<CallTriple>> result) {
		if (calleeCallStack.size() == callContextLenLimit) {
			result.add(calleeCallStack);
		} else {
			final Collection<CallTriple> _callers = callGraph.getCallers(callee);
			final int _jEnd = _callers.size();

			if (_jEnd == 0 && calleeCallStack.size() < callContextLenLimit) {
				result.add(calleeCallStack);
			} else {
				final Iterator<CallTriple> _j = _callers.iterator();

				// For collection of call stacks associated to _currToken
				for (int _jIndex = 0; _jIndex < _jEnd; _jIndex++) {
					final CallTriple _callSite = _j.next();

					if (isCallSiteExplored(_callSite, calleeToken)) {

						final Object _callerToken = getCallerSideToken(calleeToken, callee, _callSite, calleeCallStack);

						// if there was a corresponding token on the caller side
						if (_callerToken == Tokens.DISCARD_CONTEXT_TOKEN) {
							if (LOGGER.isDebugEnabled()) {
								LOGGER.debug("Discarding context " + calleeToken + "   " + this.getClass());
							}
						} else if (_callerToken == Tokens.ACCEPT_TERMINAL_CONTEXT_TOKEN
								|| _callerToken == Tokens.ACCEPT_NON_TERMINAL_CONTEXT_TOKEN) {
							/*
							 * if we have reached the property-based "pivotal" point in the call chain then we decide to
							 * extend all call chains and add it to the resulting contexts.
							 */
							@SuppressWarnings("unchecked") final Stack<CallTriple> _callerStack = calleeCallStack.clone();
							_callerStack.push(_callSite);
							if (_callerToken == Tokens.ACCEPT_TERMINAL_CONTEXT_TOKEN) {
								_callerStack.push(null);
							}
							result.add(_callerStack);
						} else {
							@SuppressWarnings("unchecked") final Stack<CallTriple> _callerStack = calleeCallStack.clone();
							_callerStack.push(_callSite);
							wb.addWorkNoDuplicates(new Triple<SootMethod, Object, Stack<CallTriple>>(_callSite.getMethod(),
									_callerToken, _callerStack));
						}
					} else {
						/*
						 * We have come to a site from which we have already considered paths up the call chain. So, we mark
						 * this call chain as terminating by inserting a null token.
						 */
						@SuppressWarnings("unchecked") final Stack<CallTriple> _callerStack = calleeCallStack.clone();
						_callerStack.push(_callSite);
						_callerStack.push(null);
						result.add(_callerStack);
					}
				}
			}
		}
	}

	/**
	 * Retrieves the contexts based on the given token and method in which it occurs.
	 * 
	 * @param token is the seed token
	 * @param method where the calling context should start from.
	 * @return a collection of calling contexts.
	 * @pre token != null and method != null
	 * @post result != null and result.oclIsKindOf(Collection(Stack(CallTriple)))
	 */
	private Collection<Stack<CallTriple>> getCallingContexts(final Object token, final SootMethod method) {
		if (LOGGER.isDebugEnabled()) {
			LOGGER.debug("getCallingContexts(Object token = " + token + ", SootMethod method = " + method + ") - BEGIN");
		}

		final Collection<Stack<CallTriple>> _result;

		if (token == Tokens.CONSIDER_ALL_CONTEXTS_TOKEN) {
			_result = ICallingContextRetriever.NULL_CONTEXTS;
		} else if (token != Tokens.DISCARD_CONTEXT_TOKEN) {
			final IWorkBag<Triple<SootMethod, Object, Stack<CallTriple>>> _wb = new LIFOWorkBag<Triple<SootMethod, Object, Stack<CallTriple>>>();
			_wb.addWork(new Triple<SootMethod, Object, Stack<CallTriple>>(method, token, new Stack<CallTriple>()));
			_result = new HashSet<Stack<CallTriple>>();

			while (_wb.hasWork()) {
				final Triple<SootMethod, Object, Stack<CallTriple>> _triple = _wb.getWork();
				final SootMethod _callee = _triple.getFirst();
				final Object _calleeToken = _triple.getSecond();
				final Stack<CallTriple> _calleeCallStack = _triple.getThird();
				calculateCallStack(_callee, _calleeToken, _calleeCallStack, _wb, _result);
			}

			// Reverse the call stacks as they have been constructed bottom-up.
			final Iterator<Stack<CallTriple>> _j = _result.iterator();
			final int _jEnd = _result.size();

			for (int _jIndex = 0; _jIndex < _jEnd; _jIndex++) {
				final Stack<?> _stack = _j.next();

				if (_stack != null) {
					Collections.reverse(_stack);
				}
			}
		} else {
			_result = Collections.emptySet();
		}

		if (LOGGER.isDebugEnabled()) {
			LOGGER.debug("getCallingContexts() - END - return value = " + _result);
		}

		return _result;
	}

	/**
	 * Checks if the given call site and callee-side token have been explored.
	 * 
	 * @param callSite of interest.
	 * @param calleeToken of interest.
	 * @return <code>true</code> if they have been explored; <code>false</code>, otherwise.
	 */
	private boolean isCallSiteExplored(@NonNull @Immutable final CallTriple callSite,
			@NonNull @Immutable final Object calleeToken) {
		final Pair<CallTriple, Object> _pair = new Pair<CallTriple, Object>(callSite, calleeToken, true, false);
		return exploredCallSites.add(_pair);
	}
}

// End of File
