/**
 * Marlin 3D Printer Firmware
 * Copyright (C) 2016 MarlinFirmware [https://github.com/MarlinFirmware/Marlin]
 *
 * Based on Sprinter and grbl.
 * Copyright (C) 2011 Camiel Gubbels / Erik van der Zalm
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

/**
 * Standard Marlin Boot and Status Screen bitmaps
 *
 * Use the Marlin Bitmap Converter to make your own:
 * http://marlinfw.org/tools/u8glib/converter.html
 */

#include "MarlinConfig.h"


#if ENABLED(SHOW_BOOTSCREEN)

  //#define START_BMPHIGH // Costs 399 bytes more flash

  #if ENABLED(SHOW_CUSTOM_BOOTSCREEN)

    #if ENABLED(SHOW_BOOTSCREEN) && ENABLED(SHOW_CUSTOM_BOOTSCREEN)
      #if ENABLED(TM3D_BOOT)
        #include "_BootscreenTM3D.h"
      #elif ENABLED(TORNADO_BOOT) && ENABLED(TORNADO)
        #include "_BootscreenTORNADO.h"
      #elif ENABLED(TARANTULA_PRO_BOOT) && ENABLED(TARANTULA_PRO)
        #include "_BootscreenTARANTULAPRO.h"
      #elif ENABLED(ENDER_BOOT)
        #include "_BootscreenENDER.h"
      #elif ENABLED(TH3D_EZ300)
        #include "_BootscreenEZ300.h"
      #else
        #include "_Bootscreen.h"
      #endif  
    #endif

    #ifndef CUSTOM_BOOTSCREEN_TIMEOUT
      #define CUSTOM_BOOTSCREEN_TIMEOUT 2500
    #endif

  #endif

  #if ENABLED(START_BMPHIGH)

    #define START_BMPWIDTH      112

    const unsigned char start_bmp[] PROGMEM = {
      B00000001,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,
      B00001111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,
      B00011110,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000111,B11111111,B11111111,
      B00111000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000011,B11111111,B11111111,
      B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000001,B11111111,B11111111,
      B01100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,
      B01100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01111111,B11111111,
      B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01111000,B00000000,B00000000,B00111111,B11111111,
      B11000000,B00001111,B11000000,B11111100,B00000000,B00000000,B00000000,B00000000,B00000000,B01111000,B00011000,B00000000,B00011111,B11111111,
      B11000000,B00111111,B11100001,B11111111,B00000000,B00000000,B00000000,B00000000,B00000000,B01111000,B00111100,B00000000,B00001111,B11111111,
      B11000000,B01111111,B11110011,B11111111,B10000000,B00000000,B00000000,B00000000,B00000000,B01111000,B00111100,B00000000,B00000111,B11111111,
      B11000000,B11111111,B11111111,B11111111,B11000000,B00000000,B00000000,B00000000,B00000000,B01111000,B00111100,B00000000,B00000011,B11111111,
      B11000001,B11111000,B01111111,B10000111,B11100000,B00000000,B00000000,B00000000,B00000000,B01111000,B00000000,B00000000,B00000001,B11111111,
      B11000001,B11110000,B00111111,B00000011,B11100000,B00000000,B00000000,B00000000,B00000000,B01111000,B00000000,B00000000,B00000000,B11111111,
      B11000001,B11100000,B00011110,B00000001,B11100000,B00011111,B00000000,B00000011,B11100000,B01111000,B00111100,B00000011,B11110000,B01111111,
      B11000001,B11100000,B00011110,B00000001,B11100000,B01111111,B11000000,B00001111,B11111000,B01111000,B00111100,B00000111,B11111100,B00111111,
      B11000001,B11100000,B00011110,B00000001,B11100001,B11111111,B11100000,B00011111,B11111100,B01111000,B00111100,B00001111,B11111110,B00011111,
      B11000001,B11100000,B00011110,B00000001,B11100011,B11111111,B11110000,B00111111,B11111110,B01111000,B00111100,B00011111,B11111110,B00001111,
      B11000001,B11100000,B00011110,B00000001,B11100011,B11110011,B11111000,B00111111,B00111110,B01111000,B00111100,B00111111,B00111111,B00000111,
      B11000001,B11100000,B00011110,B00000001,B11100111,B11100000,B11111100,B01111100,B00011111,B01111000,B00111100,B00111110,B00011111,B00000111,
      B11000001,B11100000,B00011110,B00000001,B11100111,B11000000,B01111100,B01111100,B00001111,B01111000,B00111100,B00111100,B00001111,B00000011,
      B11000001,B11100000,B00011110,B00000001,B11100111,B10000000,B01111100,B01111000,B00001111,B01111000,B00111100,B00111100,B00001111,B00000011,
      B11000001,B11100000,B00011110,B00000001,B11100111,B10000000,B00111100,B01111000,B00000000,B01111000,B00111100,B00111100,B00001111,B00000011,
      B11000001,B11100000,B00011110,B00000001,B11100111,B10000000,B00111100,B01111000,B00000000,B01111000,B00111100,B00111100,B00001111,B00000011,
      B11000001,B11100000,B00011110,B00000001,B11100111,B10000000,B00111100,B01111000,B00000000,B01111000,B00111100,B00111100,B00001111,B00000011,
      B11000001,B11100000,B00011110,B00000001,B11100111,B11000000,B00111100,B01111000,B00000000,B01111000,B00111100,B00111100,B00001111,B00000011,
      B11000001,B11100000,B00011110,B00000001,B11100011,B11100000,B00111100,B01111000,B00000000,B01111100,B00111100,B00111100,B00001111,B00000011,
      B11000001,B11100000,B00011110,B00000001,B11100011,B11111111,B00111111,B11111000,B00000000,B01111111,B10111100,B00111100,B00001111,B00000011,
      B11000001,B11100000,B00011110,B00000001,B11100001,B11111111,B00111111,B11111000,B00000000,B00111111,B10111111,B11111100,B00001111,B00000011,
      B11000001,B11100000,B00011110,B00000001,B11100000,B11111111,B00111111,B11111000,B00000000,B00011111,B10111111,B11111100,B00001111,B00000011,
      B11000001,B11100000,B00011110,B00000001,B11100000,B01111111,B00111111,B11111000,B00000000,B00001111,B10111111,B11111100,B00001111,B00000011,
      B01100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000111,
      B01100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000110,
      B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00001110,
      B00111000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00011100,
      B00011110,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01111000,
      B00001111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11110000,
      B00000001,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,B10000000
    };

  #else

    #define START_BMPWIDTH      56

    const unsigned char start_bmp[] PROGMEM = {
      B00011111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111111,
      B01100000,B00000000,B00000000,B00000000,B00000000,B00000001,B11111111,
      B01000000,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,
      B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01111111,
      B10000011,B11001111,B00000000,B00000000,B00001100,B00110000,B00111111,
      B10000111,B11111111,B10000000,B00000000,B00001100,B00110000,B00011111,
      B10000110,B01111001,B10000000,B00000000,B00001100,B00000000,B00001111,
      B10001100,B00110000,B11000111,B10000011,B10001100,B00110000,B11100111,
      B10001100,B00110000,B11001111,B11000111,B11001100,B00110001,B11110011,
      B10001100,B00110000,B11011100,B11101100,B11101100,B00110011,B10111001,
      B10001100,B00110000,B11011000,B01101100,B01101100,B00110011,B00011001,
      B10001100,B00110000,B11010000,B01101100,B00001100,B00110011,B00011001,
      B10001100,B00110000,B11011000,B01101100,B00001100,B00110011,B00011001,
      B10001100,B00110000,B11011100,B01101100,B00001110,B00111011,B00011001,
      B10001100,B00110000,B11001111,B01111100,B00000111,B10011111,B00011001,
      B10001100,B00110000,B11000111,B01111100,B00000011,B10001111,B00011001,
      B01000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000010,
      B01100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000110,
      B00011111,B11111111,B11111111,B11111111,B11111111,B11111111,B11111000
    };

  #endif

  #ifndef START_BMP_BYTEWIDTH
    #define START_BMP_BYTEWIDTH ((START_BMPWIDTH + 7) / 8)
  #endif
  #ifndef START_BMPHEIGHT
    #define START_BMPHEIGHT (sizeof(start_bmp) / (START_BMP_BYTEWIDTH))
  #endif

  static_assert(sizeof(start_bmp) == (START_BMP_BYTEWIDTH) * (START_BMPHEIGHT), "Bootscreen (start_bmp) dimensions don't match data.");

#endif

#if ENABLED(CUSTOM_STATUS_SCREEN_IMAGE)

  // This file must define STATUS_SCREENWIDTH and status_screen[012]_bmp.
  // It can also define STATUS_SCREEN_X, STATUS_SCREEN_{BED,FAN}_TEXT_X and
  // STATUS_SCREEN_HOTEND_TEXT_X(i) to modify draw locations.
  #include "_Statusscreen.h"

#else // !CUSTOM_STATUS_SCREEN_IMAGE

  // Can also be overridden in Configuration.h
  // If you can afford it, try the 3-frame fan animation!
  #ifndef FAN_ANIM_FRAMES
    #define FAN_ANIM_FRAMES 2
  #endif

  #if HAS_HEATED_BED

    #define STATUS_SCREEN_X        (  8 + (HOTENDS ? 0 : 64))
    #define STATUS_SCREENWIDTH     (120 - (HOTENDS ? 0 : 64))

    #if HOTENDS == 0

      #if FAN_ANIM_FRAMES == 3
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00111110,B00000000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00111001,B00000001,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00000000,B00000000,B00000000,B00101111,B11000111,B11101000,
          B00000000,B00000000,B00000000,B00000000,B00100111,B11000111,B11001000,
          B00000000,B00100000,B10000010,B00000000,B00100001,B11111111,B00001000,
          B00000000,B00010000,B01000001,B00000000,B00100000,B01111100,B00001000,
          B00000000,B00010000,B01000001,B00000000,B00100000,B01111100,B00001000,
          B00000000,B00100000,B10000010,B00000000,B00100000,B01111100,B00001000,
          B00000000,B01000001,B00000100,B00000000,B00100001,B11111111,B00001000,
          B00000000,B10000010,B00001000,B00000000,B00100111,B11000111,B11001000,
          B00000000,B10000010,B00001000,B00000000,B00101111,B11000111,B11101000,
          B00000000,B01000001,B00000100,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00100000,B10000010,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00000000,B00000000,B00000000,B00111001,B00000001,B00111000,
          B00000000,B11111111,B11111111,B11000000,B00111110,B00000000,B11111000,
          B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11111000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00111110,B00110000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00111001,B11110000,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00110001,B11110000,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00110000,B11110000,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00100000,B11110000,B01101000,
          B00000000,B00000000,B00000000,B00000000,B00100000,B00110001,B11101000,
          B00000000,B00100000,B10000010,B00000000,B00100000,B00111001,B11101000,
          B00000000,B00010000,B01000001,B00000000,B00100000,B01111111,B11111000,
          B00000000,B00010000,B01000001,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00100000,B10000010,B00000000,B00111111,B11111100,B00001000,
          B00000000,B01000001,B00000100,B00000000,B00101111,B00111000,B00001000,
          B00000000,B10000010,B00001000,B00000000,B00101110,B00011000,B00001000,
          B00000000,B10000010,B00001000,B00000000,B00101100,B00011110,B00001000,
          B00000000,B01000001,B00000100,B00000000,B00110000,B00011110,B00011000,
          B00000000,B00100000,B10000010,B00000000,B00110000,B00011111,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00111000,B00011111,B00111000,
          B00000000,B11111111,B11111111,B11000000,B00111110,B00011000,B11111000,
          B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11111000
        };
        const unsigned char status_screen2_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00111110,B00011000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00111000,B00011111,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00110000,B00011111,B10011000,
          B00000000,B00000000,B00000000,B00000000,B00110100,B00011111,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00101110,B00011110,B00001000,
          B00000000,B00000000,B00000000,B00000000,B00101111,B00011100,B00001000,
          B00000000,B00100000,B10000010,B00000000,B00101111,B10111000,B00001000,
          B00000000,B00010000,B01000001,B00000000,B00111111,B11111100,B00001000,
          B00000000,B00010000,B01000001,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00100000,B10000010,B00000000,B00100000,B01111111,B11111000,
          B00000000,B01000001,B00000100,B00000000,B00100000,B00111011,B11101000,
          B00000000,B10000010,B00001000,B00000000,B00100000,B01110001,B11101000,
          B00000000,B10000010,B00001000,B00000000,B00100000,B11110000,B11101000,
          B00000000,B01000001,B00000100,B00000000,B00110001,B11110000,B01011000,
          B00000000,B00100000,B10000010,B00000000,B00110011,B11110000,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00111001,B11110000,B00111000,
          B00000000,B11111111,B11111111,B11000000,B00111110,B00110000,B11111000,
          B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11111000
        };
      #else
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B00000000,B00000000,B00000000,B00110000,B11111100,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00100000,B11111100,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00101100,B00000000,B11010000,
          B00000000,B00100000,B10000010,B00000000,B00101110,B00110001,B11010000,
          B00000000,B00010000,B01000001,B00000000,B00101111,B01111011,B11010000,
          B00000000,B00010000,B01000001,B00000000,B00101111,B01111011,B11010000,
          B00000000,B00100000,B10000010,B00000000,B00101110,B00110001,B11010000,
          B00000000,B01000001,B00000100,B00000000,B00101100,B00000000,B11010000,
          B00000000,B10000010,B00001000,B00000000,B00100000,B00110000,B00010000,
          B00000000,B10000010,B00001000,B00000000,B00100000,B01111000,B00010000,
          B00000000,B01000001,B00000100,B00000000,B00100000,B11111100,B00010000,
          B00000000,B00100000,B10000010,B00000000,B00110000,B11111100,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11110000,
          B00000000,B11111111,B11111111,B11000000,B00000000,B00000000,B00000000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B00000000,B00000000,B00000000,B00110001,B10000110,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00100011,B10000111,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00100111,B10000111,B10010000,
          B00000000,B00000000,B00000000,B00000000,B00101111,B10000111,B11010000,
          B00000000,B00000000,B00000000,B00000000,B00101111,B00000011,B11010000,
          B00000000,B00100000,B10000010,B00000000,B00100000,B00110000,B00010000,
          B00000000,B00010000,B01000001,B00000000,B00100000,B01111000,B00010000,
          B00000000,B00010000,B01000001,B00000000,B00100000,B01111000,B00010000,
          B00000000,B00100000,B10000010,B00000000,B00100000,B00110000,B00010000,
          B00000000,B01000001,B00000100,B00000000,B00101111,B00000011,B11010000,
          B00000000,B10000010,B00001000,B00000000,B00101111,B10000111,B11010000,
          B00000000,B10000010,B00001000,B00000000,B00100111,B10000111,B10010000,
          B00000000,B01000001,B00000100,B00000000,B00100011,B10000111,B00010000,
          B00000000,B00100000,B10000010,B00000000,B00110001,B10000110,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11110000,
          B00000000,B11111111,B11111111,B11000000,B00000000,B00000000,B00000000
        };
      #endif

    #elif HOTENDS == 1

      #if FAN_ANIM_FRAMES == 3
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00000000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B00000001,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B11000111,B11101000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B11000111,B11001000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00100001,B11111111,B00001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00100000,B01111100,B00001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00100000,B01111100,B00001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00100000,B01111100,B00001000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00100001,B11111111,B00001000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00100111,B11000111,B11001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00101111,B11000111,B11101000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00110111,B10000011,B11011000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00110111,B10000011,B11011000,
          B00011111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B00000001,B00111000,
          B00001111,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111110,B00000000,B11111000,
          B00000110,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11111000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00110000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B11110000,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B11110000,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B11110000,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11110000,B01101000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110001,B11101000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00100000,B00111001,B11101000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00100000,B01111111,B11111000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00111111,B11111111,B11111000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00111111,B11111100,B00001000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00101111,B00111000,B00001000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00101110,B00011000,B00001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00101100,B00011110,B00001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00110000,B00011110,B00011000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00110000,B00011111,B00011000,
          B00011111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00011111,B00111000,
          B00001111,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111110,B00011000,B11111000,
          B00000110,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11111000
        };
        const unsigned char status_screen2_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00011000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00011111,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B00011111,B10011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110100,B00011111,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101110,B00011110,B00001000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00011100,B00001000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00101111,B10111000,B00001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00111111,B11111100,B00001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00111111,B11111111,B11111000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00100000,B01111111,B11111000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00100000,B00111011,B11101000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00100000,B01110001,B11101000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00100000,B11110000,B11101000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00110001,B11110000,B01011000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00110011,B11110000,B00011000,
          B00011111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B11110000,B00111000,
          B00001111,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111110,B00110000,B11111000,
          B00000110,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11111000
        };
      #else
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B11111100,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11111100,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101100,B00000000,B11010000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00101110,B00110001,B11010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00101111,B01111011,B11010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00101111,B01111011,B11010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00101110,B00110001,B11010000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00101100,B00000000,B11010000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00100000,B00110000,B00010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00100000,B01111000,B00010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00100000,B11111100,B00010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00110000,B11111100,B00110000,
          B00011111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00001111,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11110000,
          B00000110,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00000000,B00000000,B00000000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B10000110,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100011,B10000111,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B10000111,B10010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B10000111,B11010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00000011,B11010000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00100000,B00110000,B00010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00100000,B01111000,B00010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00100000,B01111000,B00010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00100000,B00110000,B00010000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00101111,B00000011,B11010000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00101111,B10000111,B11010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00100111,B10000111,B10010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00100011,B10000111,B00010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00110001,B10000110,B00110000,
          B00011111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00001111,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11110000,
          B00000110,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00000000,B00000000,B00000000
        };
      #endif

    #elif HOTENDS == 2

      #if FAN_ANIM_FRAMES == 3
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00000000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B00000001,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B11000111,B11101000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B11000111,B11001000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00100001,B11111111,B00001000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00100000,B01111100,B00001000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00100000,B01111100,B00001000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00100000,B01111100,B00001000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00100001,B11111111,B00001000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00100111,B11000111,B11001000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00101111,B11000111,B11101000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00110111,B10000011,B11011000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00110111,B10000011,B11011000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B00000001,B00111000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111110,B00000000,B11111000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11111000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00110000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B11110000,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B11110000,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B11110000,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11110000,B01101000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110001,B11101000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00100000,B00111001,B11101000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00100000,B01111111,B11111000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00111111,B11111111,B11111000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00111111,B11111100,B00001000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00101111,B00111000,B00001000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00101110,B00011000,B00001000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00101100,B00011110,B00001000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00110000,B00011110,B00011000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00110000,B00011111,B00011000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00011111,B00111000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111110,B00011000,B11111000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11111000
        };
        const unsigned char status_screen2_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00011000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00011111,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B00011111,B10011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110100,B00011111,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101110,B00011110,B00001000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00011100,B00001000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00101111,B10111000,B00001000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00111111,B11111100,B00001000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00111111,B11111111,B11111000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00100000,B01111111,B11111000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00100000,B00111011,B11101000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00100000,B01110001,B11101000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00100000,B11110000,B11101000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00110001,B11110000,B01011000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00110011,B11110000,B00011000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B11110000,B00111000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111110,B00110000,B11111000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11111000
        };
      #else
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B11111100,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11111100,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101100,B00000000,B11010000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00101110,B00110001,B11010000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00101111,B01111011,B11010000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00101111,B01111011,B11010000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00101110,B00110001,B11010000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00101100,B00000000,B11010000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00100000,B00110000,B00010000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00100000,B01111000,B00010000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00100000,B11111100,B00010000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00110000,B11111100,B00110000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11110000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00000000,B00000000,B00000000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B10000110,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100011,B10000111,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B10000111,B10010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B10000111,B11010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00000011,B11010000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00100000,B00110000,B00010000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00100000,B01111000,B00010000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00010000,B01000001,B00000000,B00100000,B01111000,B00010000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00100000,B00110000,B00010000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00101111,B00000011,B11010000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00101111,B10000111,B11010000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00000000,B00000000,B00000000,B10000010,B00001000,B00000000,B00100111,B10000111,B10010000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00000000,B00000000,B00000000,B01000001,B00000100,B00000000,B00100011,B10000111,B00010000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00100000,B10000010,B00000000,B00110001,B10000110,B00110000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11110000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B11111111,B11111111,B11000000,B00000000,B00000000,B00000000
        };
      #endif

    #else // HOTENDS > 2

      #if FAN_ANIM_FRAMES == 3
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00000000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B00000001,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B11000111,B11101000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B11000111,B11001000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00100000,B10000010,B00000000,B00100001,B11111111,B00001000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00010000,B01000001,B00000000,B00100000,B01111100,B00001000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00010000,B01000001,B00000000,B00100000,B01111100,B00001000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00100000,B10000010,B00000000,B00100000,B01111100,B00001000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B01000001,B00000100,B00000000,B00100001,B11111111,B00001000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B10000010,B00001000,B00000000,B00100111,B11000111,B11001000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B10000010,B00001000,B00000000,B00101111,B11000111,B11101000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B01000001,B00000100,B00000000,B00110111,B10000011,B11011000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00100000,B10000010,B00000000,B00110111,B10000011,B11011000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00111001,B00000001,B00111000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B11111111,B11111111,B11000000,B00111110,B00000000,B11111000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11111000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00110000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B11110000,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B11110000,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B11110000,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11110000,B01101000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110001,B11101000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00100000,B10000010,B00000000,B00100000,B00111001,B11101000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00010000,B01000001,B00000000,B00100000,B01111111,B11111000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00010000,B01000001,B00000000,B00111111,B11111111,B11111000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00100000,B10000010,B00000000,B00111111,B11111100,B00001000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B01000001,B00000100,B00000000,B00101111,B00111000,B00001000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B10000010,B00001000,B00000000,B00101110,B00011000,B00001000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B10000010,B00001000,B00000000,B00101100,B00011110,B00001000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B01000001,B00000100,B00000000,B00110000,B00011110,B00011000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00100000,B10000010,B00000000,B00110000,B00011111,B00011000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00011111,B00111000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B11111111,B11111111,B11000000,B00111110,B00011000,B11111000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11111000
        };
        const unsigned char status_screen2_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00011000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00011111,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B00011111,B10011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110100,B00011111,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101110,B00011110,B00001000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00011100,B00001000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00100000,B10000010,B00000000,B00101111,B10111000,B00001000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00010000,B01000001,B00000000,B00111111,B11111100,B00001000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00010000,B01000001,B00000000,B00111111,B11111111,B11111000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00100000,B10000010,B00000000,B00100000,B01111111,B11111000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B01000001,B00000100,B00000000,B00100000,B00111011,B11101000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B10000010,B00001000,B00000000,B00100000,B01110001,B11101000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B10000010,B00001000,B00000000,B00100000,B11110000,B11101000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B01000001,B00000100,B00000000,B00110001,B11110000,B01011000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00100000,B10000010,B00000000,B00110011,B11110000,B00011000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00111001,B11110000,B00111000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B11111111,B11111111,B11000000,B00111110,B00110000,B11111000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11111000
        };
      #else
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B11111100,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11111100,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101100,B00000000,B11010000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00100000,B10000010,B00000000,B00101110,B00110001,B11010000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00010000,B01000001,B00000000,B00101111,B01111011,B11010000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00010000,B01000001,B00000000,B00101111,B01111011,B11010000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00100000,B10000010,B00000000,B00101110,B00110001,B11010000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B01000001,B00000100,B00000000,B00101100,B00000000,B11010000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B10000010,B00001000,B00000000,B00100000,B00110000,B00010000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B10000010,B00001000,B00000000,B00100000,B01111000,B00010000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B01000001,B00000100,B00000000,B00100000,B11111100,B00010000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00100000,B10000010,B00000000,B00110000,B11111100,B00110000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11110000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B11111111,B11111111,B11000000,B00000000,B00000000,B00000000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B10000110,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100011,B10000111,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B10000111,B10010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B10000111,B11010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00000011,B11010000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00100000,B10000010,B00000000,B00100000,B00110000,B00010000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00010000,B01000001,B00000000,B00100000,B01111000,B00010000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00010000,B01000001,B00000000,B00100000,B01111000,B00010000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00100000,B10000010,B00000000,B00100000,B00110000,B00010000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B01000001,B00000100,B00000000,B00101111,B00000011,B11010000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B10000010,B00001000,B00000000,B00101111,B10000111,B11010000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B10000010,B00001000,B00000000,B00100111,B10000111,B10010000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B01000001,B00000100,B00000000,B00100011,B10000111,B00010000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00100000,B10000010,B00000000,B00110001,B10000110,B00110000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B11111111,B11111111,B11000000,B00111111,B11111111,B11110000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B11111111,B11111111,B11000000,B00000000,B00000000,B00000000
        };
      #endif

    #endif // HOTENDS

  #else // !HAS_HEATED_BED

    #define STATUS_SCREEN_X        (  8 + (HOTENDS ? 0 : 96))
    #define STATUS_SCREENWIDTH     (120 - (HOTENDS ? 0 : 96))

    #if HOTENDS == 0

      #if FAN_ANIM_FRAMES == 3
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00111111,B11111111,B11111000,
          B00111110,B00000000,B11111000,
          B00111001,B00000001,B00111000,
          B00110111,B10000011,B11011000,
          B00110111,B10000011,B11011000,
          B00101111,B11000111,B11101000,
          B00100111,B11000111,B11001000,
          B00100001,B11111111,B00001000,
          B00100000,B01111100,B00001000,
          B00100000,B01111100,B00001000,
          B00100000,B01111100,B00001000,
          B00100001,B11111111,B00001000,
          B00100111,B11000111,B11001000,
          B00101111,B11000111,B11101000,
          B00110111,B10000011,B11011000,
          B00110111,B10000011,B11011000,
          B00111001,B00000001,B00111000,
          B00111110,B00000000,B11111000,
          B00111111,B11111111,B11111000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00111111,B11111111,B11111000,
          B00111110,B00110000,B11111000,
          B00111001,B11110000,B00111000,
          B00110001,B11110000,B00011000,
          B00110000,B11110000,B00011000,
          B00100000,B11110000,B01101000,
          B00100000,B00110001,B11101000,
          B00100000,B00111001,B11101000,
          B00100000,B01111111,B11111000,
          B00111111,B11111111,B11111000,
          B00111111,B11111100,B00001000,
          B00101111,B00111000,B00001000,
          B00101110,B00011000,B00001000,
          B00101100,B00011110,B00001000,
          B00110000,B00011110,B00011000,
          B00110000,B00011111,B00011000,
          B00111000,B00011111,B00111000,
          B00111110,B00011000,B11111000,
          B00111111,B11111111,B11111000
        };
        const unsigned char status_screen2_bmp[] PROGMEM = {
          B00111111,B11111111,B11111000,
          B00111110,B00011000,B11111000,
          B00111000,B00011111,B00111000,
          B00110000,B00011111,B10011000,
          B00110100,B00011111,B00011000,
          B00101110,B00011110,B00001000,
          B00101111,B00011100,B00001000,
          B00101111,B10111000,B00001000,
          B00111111,B11111100,B00001000,
          B00111111,B11111111,B11111000,
          B00100000,B01111111,B11111000,
          B00100000,B00111011,B11101000,
          B00100000,B01110001,B11101000,
          B00100000,B11110000,B11101000,
          B00110001,B11110000,B01011000,
          B00110011,B11110000,B00011000,
          B00111001,B11110000,B00111000,
          B00111110,B00110000,B11111000,
          B00111111,B11111111,B11111000
        };
      #else
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00111111,B11111111,B11110000,
          B00111000,B00000000,B01110000,
          B00110000,B11111100,B00110000,
          B00100000,B11111100,B00010000,
          B00100000,B01111000,B00010000,
          B00100000,B00110000,B00010000,
          B00101100,B00000000,B11010000,
          B00101110,B00110001,B11010000,
          B00101111,B01111011,B11010000,
          B00101111,B01111011,B11010000,
          B00101110,B00110001,B11010000,
          B00101100,B00000000,B11010000,
          B00100000,B00110000,B00010000,
          B00100000,B01111000,B00010000,
          B00100000,B11111100,B00010000,
          B00110000,B11111100,B00110000,
          B00111000,B00000000,B01110000,
          B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00111111,B11111111,B11110000,
          B00111000,B00000000,B01110000,
          B00110001,B10000110,B00110000,
          B00100011,B10000111,B00010000,
          B00100111,B10000111,B10010000,
          B00101111,B10000111,B11010000,
          B00101111,B00000011,B11010000,
          B00100000,B00110000,B00010000,
          B00100000,B01111000,B00010000,
          B00100000,B01111000,B00010000,
          B00100000,B00110000,B00010000,
          B00101111,B00000011,B11010000,
          B00101111,B10000111,B11010000,
          B00100111,B10000111,B10010000,
          B00100011,B10000111,B00010000,
          B00110001,B10000110,B00110000,
          B00111000,B00000000,B01110000,
          B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000
        };
      #endif

    #elif HOTENDS == 1

      #if FAN_ANIM_FRAMES == 3
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00000000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B00000001,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B11000111,B11101000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B11000111,B11001000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100001,B11111111,B00001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111100,B00001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111100,B00001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111100,B00001000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100001,B11111111,B00001000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B11000111,B11001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B11000111,B11101000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00011111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B00000001,B00111000,
          B00001111,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00000000,B11111000,
          B00000110,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00110000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B11110000,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B11110000,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B11110000,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11110000,B01101000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110001,B11101000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00111001,B11101000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111111,B11111000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111100,B00001000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00111000,B00001000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101110,B00011000,B00001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101100,B00011110,B00001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B00011110,B00011000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B00011111,B00011000,
          B00011111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00011111,B00111000,
          B00001111,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00011000,B11111000,
          B00000110,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000
        };
        const unsigned char status_screen2_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00011000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00011111,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B00011111,B10011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110100,B00011111,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101110,B00011110,B00001000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00011100,B00001000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B10111000,B00001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111100,B00001000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111111,B11111000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00111011,B11101000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01110001,B11101000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11110000,B11101000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B11110000,B01011000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110011,B11110000,B00011000,
          B00011111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B11110000,B00111000,
          B00001111,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00110000,B11111000,
          B00000110,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000
        };
      #else
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B11111100,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11111100,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101100,B00000000,B11010000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101110,B00110001,B11010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B01111011,B11010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B01111011,B11010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101110,B00110001,B11010000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101100,B00000000,B11010000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11111100,B00010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B11111100,B00110000,
          B00011111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00001111,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000110,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B10000110,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100011,B10000111,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B10000111,B10010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B10000111,B11010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00000011,B11010000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00000011,B11010000,
          B00111111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B10000111,B11010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B10000111,B10010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100011,B10000111,B00010000,
          B01111111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B10000110,B00110000,
          B00011111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00001111,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000110,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000
        };
      #endif

    #elif HOTENDS == 2

      #if FAN_ANIM_FRAMES == 3
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00000000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B00000001,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B11000111,B11101000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B11000111,B11001000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100001,B11111111,B00001000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111100,B00001000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111100,B00001000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111100,B00001000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100001,B11111111,B00001000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B11000111,B11001000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B11000111,B11101000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B00000001,B00111000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00000000,B11111000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00110000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B11110000,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B11110000,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B11110000,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11110000,B01101000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110001,B11101000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00111001,B11101000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111111,B11111000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111100,B00001000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00111000,B00001000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101110,B00011000,B00001000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101100,B00011110,B00001000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B00011110,B00011000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B00011111,B00011000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00011111,B00111000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00011000,B11111000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000
        };
        const unsigned char status_screen2_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00011000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00011111,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B00011111,B10011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110100,B00011111,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101110,B00011110,B00001000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00011100,B00001000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B10111000,B00001000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111100,B00001000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111111,B11111000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00111011,B11101000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01110001,B11101000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11110000,B11101000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B11110000,B01011000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110011,B11110000,B00011000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B11110000,B00111000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00110000,B11111000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000
        };
      #else
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B11111100,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11111100,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101100,B00000000,B11010000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101110,B00110001,B11010000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B01111011,B11010000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B01111011,B11010000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101110,B00110001,B11010000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101100,B00000000,B11010000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11111100,B00010000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B11111100,B00110000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B10000110,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100011,B10000111,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B10000111,B10010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B10000111,B11010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00000011,B11010000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00000011,B11010000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B10000111,B11010000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B10000111,B10010000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100011,B10000111,B00010000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B10000110,B00110000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000
        };
      #endif

    #else // HOTENDS > 2

      #if FAN_ANIM_FRAMES == 3
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00000000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B00000001,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B11000111,B11101000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B11000111,B11001000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00100001,B11111111,B00001000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111100,B00001000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111100,B00001000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111100,B00001000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B00000000,B00000000,B00000000,B00100001,B11111111,B00001000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B00000000,B00000000,B00000000,B00100111,B11000111,B11001000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00101111,B11000111,B11101000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00110111,B10000011,B11011000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00111001,B00000001,B00111000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00000000,B11111000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00110000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111001,B11110000,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B11110000,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B11110000,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11110000,B01101000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110001,B11101000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00100000,B00111001,B11101000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111111,B11111000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111100,B00001000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B00000000,B00000000,B00000000,B00101111,B00111000,B00001000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B00000000,B00000000,B00000000,B00101110,B00011000,B00001000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00101100,B00011110,B00001000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00110000,B00011110,B00011000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00110000,B00011111,B00011000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00011111,B00111000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00011000,B11111000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000
        };
        const unsigned char status_screen2_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00011000,B11111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00011111,B00111000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B00011111,B10011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110100,B00011111,B00011000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101110,B00011110,B00001000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00011100,B00001000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00101111,B10111000,B00001000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111100,B00001000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111111,B11111000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B00000000,B00000000,B00000000,B00100000,B00111011,B11101000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B00000000,B00000000,B00000000,B00100000,B01110001,B11101000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00100000,B11110000,B11101000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00110001,B11110000,B01011000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00110011,B11110000,B00011000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00111001,B11110000,B00111000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B00111110,B00110000,B11111000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11111000
        };
      #else
        const unsigned char status_screen0_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110000,B11111100,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B11111100,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101100,B00000000,B11010000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00101110,B00110001,B11010000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00101111,B01111011,B11010000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00101111,B01111011,B11010000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00101110,B00110001,B11010000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B00000000,B00000000,B00000000,B00101100,B00000000,B11010000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00100000,B11111100,B00010000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00110000,B11111100,B00110000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000
        };
        const unsigned char status_screen1_bmp[] PROGMEM = {
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00110001,B10000110,B00110000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100011,B10000111,B00010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00100111,B10000111,B10010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B10000111,B11010000,
          B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00101111,B00000011,B11010000,
          B00111111,B11000000,B00000000,B00011111,B11100000,B00000000,B00011111,B11100000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B01111101,B11100000,B00000000,B00111100,B11110000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B01111001,B11100000,B00000000,B00111011,B01110000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00100000,B01111000,B00010000,
          B01110101,B11100000,B00000000,B00111111,B01110000,B00000000,B00111111,B01110000,B00000000,B00000000,B00000000,B00000000,B00100000,B00110000,B00010000,
          B00111101,B11000000,B00000000,B00011110,B11100000,B00000000,B00011100,B11100000,B00000000,B00000000,B00000000,B00000000,B00101111,B00000011,B11010000,
          B00111101,B11000000,B00000000,B00011101,B11100000,B00000000,B00011111,B01100000,B00000000,B00000000,B00000000,B00000000,B00101111,B10000111,B11010000,
          B01111101,B11100000,B00000000,B00111011,B11110000,B00000000,B00111011,B01110000,B00000000,B00000000,B00000000,B00000000,B00100111,B10000111,B10010000,
          B01111101,B11100000,B00000000,B00111000,B01110000,B00000000,B00111100,B11110000,B00000000,B00000000,B00000000,B00000000,B00100011,B10000111,B00010000,
          B01111111,B11100000,B00000000,B00111111,B11110000,B00000000,B00111111,B11110000,B00000000,B00000000,B00000000,B00000000,B00110001,B10000110,B00110000,
          B00011111,B10000000,B00000000,B00001111,B11000000,B00000000,B00001111,B11000000,B00000000,B00000000,B00000000,B00000000,B00111000,B00000000,B01110000,
          B00001111,B00000000,B00000000,B00000111,B10000000,B00000000,B00000111,B10000000,B00000000,B00000000,B00000000,B00000000,B00111111,B11111111,B11110000,
          B00000110,B00000000,B00000000,B00000011,B00000000,B00000000,B00000011,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000,B00000000
        };
      #endif

    #endif // HOTENDS

  #endif // !HAS_HEATED_BED

#endif // !CUSTOM_STATUS_SCREEN_IMAGE

#if ENABLED(BABYSTEP_ZPROBE_GFX_OVERLAY) || ENABLED(MESH_EDIT_GFX_OVERLAY)

  const unsigned char cw_bmp[] PROGMEM = {
    B00000011,B11111000,B00000000,
    B00001111,B11111110,B00000000,
    B00011110,B00001111,B00000000,
    B00111000,B00000111,B00000000,
    B00111000,B00000011,B10000000,
    B01110000,B00000011,B10000000,
    B01110000,B00001111,B11100000,
    B01110000,B00000111,B11000000,
    B01110000,B00000011,B10000000,
    B01110000,B00000001,B00000000,
    B01110000,B00000000,B00000000,
    B00111000,B00000000,B00000000,
    B00111000,B00000111,B00000000,
    B00011110,B00001111,B00000000,
    B00001111,B11111110,B00000000,
    B00000011,B11111000,B00000000
  };

  const unsigned char ccw_bmp[] PROGMEM = {
    B00000000,B11111110,B00000000,
    B00000011,B11111111,B10000000,
    B00000111,B10000011,B11000000,
    B00001110,B00000001,B11000000,
    B00001110,B00000000,B11100000,
    B00011100,B00000000,B11100000,
    B01111111,B00000000,B11100000,
    B00111110,B00000000,B11100000,
    B00011100,B00000000,B11100000,
    B00001000,B00000000,B11100000,
    B00000000,B00000000,B11100000,
    B00000000,B00000001,B11000000,
    B00001110,B00000001,B11000000,
    B00001111,B00000111,B10000000,
    B00000111,B11111111,B00000000,
    B00000001,B11111100,B00000000
  };

  const unsigned char up_arrow_bmp[] PROGMEM = {
    B00000100,B00000000,
    B00001110,B00000000,
    B00011111,B00000000,
    B00111111,B10000000,
    B01111111,B11000000,
    B00001110,B00000000,
    B00001110,B00000000,
    B00001110,B00000000,
    B00001110,B00000000,
    B00001110,B00000000,
    B00001110,B00000000,
    B00001110,B00000000,
    B00001110,B00000000
  };

  const unsigned char down_arrow_bmp[] PROGMEM = {
    B00001110,B00000000,
    B00001110,B00000000,
    B00001110,B00000000,
    B00001110,B00000000,
    B00001110,B00000000,
    B00001110,B00000000,
    B00001110,B00000000,
    B00001110,B00000000,
    B01111111,B11000000,
    B00111111,B10000000,
    B00011111,B00000000,
    B00001110,B00000000,
    B00000100,B00000000
  };

  const unsigned char offset_bedline_bmp[] PROGMEM = {
    B11111111,B11111111,B11111111
  };

  const unsigned char nozzle_bmp[] PROGMEM = {
    B01111111,B10000000,
    B11111111,B11000000,
    B11111111,B11000000,
    B11111111,B11000000,
    B01111111,B10000000,
    B01111111,B10000000,
    B11111111,B11000000,
    B11111111,B11000000,
    B11111111,B11000000,
    B00111111,B00000000,
    B00011110,B00000000,
    B00001100,B00000000
  };

#endif // BABYSTEP_ZPROBE_GFX_OVERLAY || MESH_EDIT_GFX_OVERLAY

#ifndef CUSTOM_BOOTSCREEN_BMP_BYTEWIDTH
  #define CUSTOM_BOOTSCREEN_BMP_BYTEWIDTH ((CUSTOM_BOOTSCREEN_BMPWIDTH + 7) / 8)
#endif
#ifndef CUSTOM_BOOTSCREEN_BMPHEIGHT
  #define CUSTOM_BOOTSCREEN_BMPHEIGHT (sizeof(custom_start_bmp) / (CUSTOM_BOOTSCREEN_BMP_BYTEWIDTH))
#endif

#ifndef FAN_ANIM_FRAMES
  #define FAN_ANIM_FRAMES 2
#elif FAN_ANIM_FRAMES > 3
  #error "Only 3 fan animation frames currently supported."
#endif
#ifndef STATUS_SCREEN_X
  #define STATUS_SCREEN_X 0
#endif
#ifndef STATUS_SCREEN_Y
  #define STATUS_SCREEN_Y 1
#endif
#ifndef STATUS_BMP_BYTEWIDTH
  #define STATUS_BMP_BYTEWIDTH ((STATUS_SCREENWIDTH + 7) / 8)
#endif
#ifndef STATUS_SCREENHEIGHT
  #define STATUS_SCREENHEIGHT (sizeof(status_screen0_bmp) / (STATUS_BMP_BYTEWIDTH))
#endif
#ifndef STATUS_SCREEN_HOTEND_TEXT_X
  #define STATUS_SCREEN_HOTEND_TEXT_X(i) (5 + (i) * 25)
#endif
#ifndef STATUS_SCREEN_BED_TEXT_X
  #define STATUS_SCREEN_BED_TEXT_X 81
#endif
#ifndef STATUS_SCREEN_FAN_TEXT_X
  #define STATUS_SCREEN_FAN_TEXT_X 104
#endif
#ifndef STATUS_SCREEN_FAN_TEXT_Y
  #define STATUS_SCREEN_FAN_TEXT_Y (FAN_ANIM_FRAMES > 2 ? 28 : 27)
#endif

#define BMP_SIZE (STATUS_BMP_BYTEWIDTH) * (STATUS_SCREENHEIGHT)
static_assert(sizeof(status_screen0_bmp) == BMP_SIZE, "Status header (status_screen0_bmp) dimensions don't match data.");
#if FAN_ANIM_FRAMES > 1
  static_assert(sizeof(status_screen1_bmp) == BMP_SIZE, "Status header (status_screen1_bmp) dimensions don't match data.");
  #if FAN_ANIM_FRAMES > 2
    static_assert(sizeof(status_screen2_bmp) == BMP_SIZE, "Status header (status_screen2_bmp) dimensions don't match data.");
    #if FAN_ANIM_FRAMES > 3
      static_assert(sizeof(status_screen3_bmp) == BMP_SIZE, "Status header (status_screen3_bmp) dimensions don't match data.");
    #endif
  #endif
#endif
