# Copyright (c) RenChu Wang - All Rights Reserved

import abc
from collections import OrderedDict
from typing import Protocol

from numpy.typing import NDArray

from bocoel.corpora import Index


class Exam(Protocol):
    """
    Exams are designed to evaluate the performance of a particular index,
    using a particular set of results generated by the optimizer.
    """

    def run(self, index: Index, results: OrderedDict[int, float]) -> NDArray:
        """
        Run the exam on the given index and results.

        Parameters:
            index: The index to evaluate.
            results: The results generated by the optimizer.

        Returns:
            The scores for each entry in the index. The length must be the same as the results.
        """

        outcome = self._run(index=index, results=results)

        if len(outcome) != len(results):
            raise ValueError(
                f"Length of outcome ({len(outcome)}) must be the same as "
                f"the length of results ({len(results)})"
            )

        return outcome

    @abc.abstractmethod
    def _run(self, index: Index, results: OrderedDict[int, float]) -> NDArray:
        """
        Run the exam on the given index and results.

        Parameters:
            index: The index to evaluate.
            results: The results generated by the optimizer.

        Returns:
            The scores for each entry in the index. The length must be the same as the results.
        """

        ...
