// Copyright 2022 Redpanda Data, Inc.
//
// Use of this software is governed by the Business Source License
// included in the file licenses/BSL.md
//
// As of the Change Date specified in that file, in accordance with
// the Business Source License, use of this software will be governed
// by the Apache License, Version 2.0

package cloud

import (
	"fmt"

	"github.com/redpanda-data/redpanda/src/go/rpk/pkg/config"
	rpkos "github.com/redpanda-data/redpanda/src/go/rpk/pkg/os"
	"github.com/redpanda-data/redpanda/src/go/rpk/pkg/out"
	"github.com/spf13/afero"
	"github.com/spf13/cobra"
)

func newLogoutCommand(fs afero.Fs, p *config.Params) *cobra.Command {
	var clear bool
	var all bool
	cmd := &cobra.Command{
		Use:   "logout",
		Short: "Log out from Redpanda cloud",
		Long: `Log out from Redpanda cloud

This command deletes your cloud auth token. If you want to log out entirely and
switch to a different organization, you can use the --clear-credentials flag to
additionally clear your client ID and client secret.

You can use the --all flag to log out of all organizations you may be logged
into.
`,
		Args: cobra.ExactArgs(0),
		Run: func(_ *cobra.Command, _ []string) {
			if rpkos.IsRunningSudo() {
				out.Die("detected rpk is running with sudo; please execute this command without sudo to avoid saving the cloud configuration as a root owned file")
			}
			cfg, err := p.Load(fs)
			out.MaybeDie(err, "rpk unable to load config: %v", err)

			y, ok := cfg.ActualRpkYaml()
			if !ok {
				fmt.Println("You are not logged in.")
				return
			}
			if !all {
				a := y.CurrentAuth()
				if a == nil || a.AuthToken == "" && !clear {
					fmt.Println("You are not logged in.")
					return
				}
			}

			do := func(a *config.RpkCloudAuth) {
				if clear {
					a.ClientID = ""
					a.ClientSecret = ""
					a.AuthToken = ""
				} else {
					a.AuthToken = ""
				}
			}

			if all {
				for i := range y.CloudAuths {
					do(&y.CloudAuths[i])
				}
			} else {
				do(y.CurrentAuth())
			}

			err = y.Write(fs)
			out.MaybeDie(err, "unable to save the cloud configuration :%v", err)
			if all {
				fmt.Println("You are now logged out of all organizations you were logged into.")
			} else {
				fmt.Println("You are now logged out.")
			}
		},
	}

	cmd.Flags().BoolVarP(&clear, "clear-credentials", "c", false, "Clear the client ID and client secret in addition to the auth token")
	cmd.Flags().BoolVarP(&all, "all", "a", false, "Log out of all organizations you may be logged into, rather than just the current auth's organization")
	return cmd
}
