# Some considerations about the use of AWS S3 not included in the [ankitpokhrel/tus-php](https://github.com/ankitpokhrel/tus-php) documentation:

Note: version 2.x of the [ankitpokhrel/tus-php](https://github.com/ankitpokhrel/tus-php) project has been used for these considerations.

**S3 support does not work out of the box**, as the documentation says. This happens because the cache (`Cache\FileStore`) tries to save the cache to the local disk. This is not optimal for distributed systems (like kubernetes). We could initialize the `Cache\FileStore` and use the same bucket information. But this also doesn't work because AWS S3 doesn't implement the `touch()` or `flock()` functions used in `Cache\FileStore`. We could extend this class (`Cache\FileStore`) to rewrite the methods that use those functions with others compatible with AWS S3. But I think **the best solution would be to use `Cache\RedisStore` as a cache**.

```php
<?php

use Aws\S3\S3Client;

use TusPhp\Tus\Server;
use TusPhp\Events\TusEvent;
use TusPhp\File;

// S3 BUCKET
$bucketPath = "s3://my-bucket/my-prefix/";

// init AWS S3 Client
// and register stream wrapper
// ...
$s3Client->registerStreamWrapper();

$server = new Server('redis');
$server->setUploadDir($bucketPath);

// if u need use other cache you can use `setCache()` method
//$server->setCache($cache); 

// --> api path
// if api path is longer than /files, like /api/files and /api/files/{token}
// u need set api path to server and not use default

$fileToken = $server->getUploadKey();
$apiPath = null == $fileToken 
    ? $server->getRequest()->path() 
    : str_replace(sprintf("/%s", $fileToken), "". $server->getRequest()->path());

$server->setApiPath($apiPath);

// If you do not want to use the default uuid to identify the file, 
// you can modify it using the setUploadKey() method
$server->setUploadKey($token);

// Event Name	Dispatched
// tus-server.upload.created	after the upload is created during POST request.
// tus-server.upload.progress	after a chunk is uploaded during PATCH request.
// tus-server.upload.complete	after the upload is complete and checksum verification is done.
// tus-server.upload.merged	    after all partial uploads are merged during concatenation request.

$file = null;
// $server->event()->addListener('tus-server.upload.created', fn(TusEvent $event) => error_log("created") );
// $server->event()->addListener('tus-server.upload.progress', fn(TusEvent $event) => error_log("progress") );
$server->event()->addListener('tus-server.upload.complete', fn(TusEvent $event) => $file = $event->getFile());


// get http response
$server->getResponse()->createOnly(true);
$response = $server->serve();

if(null != $file) {
    $s3key = $file->getFilePath();
    $filename = $file->getName();
}
```
