/*
 * SonarQube Java Properties Analyzer
 * Copyright (C) 2015-2017 David RACODON
 * david.racodon@gmail.com
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 3 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */
package org.sonar.jproperties.parser;

import com.google.common.base.Charsets;
import org.junit.Test;
import org.sonar.plugins.jproperties.api.tree.PropertyTree;
import org.sonar.plugins.jproperties.api.tree.SeparatorTree;

import javax.annotation.Nullable;

import static org.fest.assertions.Assertions.assertThat;

public class PropertyTreeTest {

  @Test
  public void property() throws Exception {
    checkParsed("abc:def", "abc", SeparatorTree.Separator.COLON, "def");
    checkParsed("abc :def", "abc", SeparatorTree.Separator.COLON, "def");
    checkParsed("abc: def", "abc", SeparatorTree.Separator.COLON, "def");
    checkParsed("abc: def ", "abc", SeparatorTree.Separator.COLON, "def ");
    checkParsed("abc : def ", "abc", SeparatorTree.Separator.COLON, "def ");

    checkParsed("abc=def", "abc", SeparatorTree.Separator.EQUALS, "def");
    checkParsed("abc =def", "abc", SeparatorTree.Separator.EQUALS, "def");
    checkParsed("abc= def", "abc", SeparatorTree.Separator.EQUALS, "def");
    checkParsed("abc= def ", "abc", SeparatorTree.Separator.EQUALS, "def ");
    checkParsed("abc = def ", "abc", SeparatorTree.Separator.EQUALS, "def ");

    checkParsed("abc\\=abc = def", "abc\\=abc", SeparatorTree.Separator.EQUALS, "def");
    checkParsed("abc\\:abc = def", "abc\\:abc", SeparatorTree.Separator.EQUALS, "def");

    checkParsed("abc : d : ef  ", "abc", SeparatorTree.Separator.COLON, "d : ef  ");
    checkParsed("abc = d = ef  ", "abc", SeparatorTree.Separator.EQUALS, "d = ef  ");

    checkParsed("abc:", "abc", SeparatorTree.Separator.COLON, null);
    checkParsed("abc :", "abc", SeparatorTree.Separator.COLON, null);
    checkParsed("abc : ", "abc", SeparatorTree.Separator.COLON, null);
    checkParsed("abc=", "abc", SeparatorTree.Separator.EQUALS, null);
    checkParsed("abc =", "abc", SeparatorTree.Separator.EQUALS, null);
    checkParsed("abc = ", "abc", SeparatorTree.Separator.EQUALS, null);

    checkParsed("a!bc = def", "a!bc", SeparatorTree.Separator.EQUALS, "def");
    checkParsed("a#bc = def", "a#bc", SeparatorTree.Separator.EQUALS, "def");
    checkParsed("a#bc = #def", "a#bc", SeparatorTree.Separator.EQUALS, "#def");
    checkParsed("a#bc = !def", "a#bc", SeparatorTree.Separator.EQUALS, "!def");

    checkParsed("abc = blabla\\\nblabla...", "abc", SeparatorTree.Separator.EQUALS, "blabla\\\nblabla...");

    checkParsed(
      "java=javax.accessibility;uses\\:\\=\"javax.swing.text\";version\\=\"0.0.0.1_007_JavaSE\", javax.activity;version\\=\"0.0.0.1_007_JavaSE\", javax.annotation.processing;uses\\:\\=\"javax.tools,javax.lang.model,javax.lang.model.element,javax.lang.model.util\";version\\=\"0.0.0.1_007_JavaSE\", javax.annotation;version\\=\"0.0.0.1_007_JavaSE\", javax.crypto.interfaces;uses\\:\\=\"javax.crypto.spec,javax.crypto\";version\\=\"0.0.0.1_007_JavaSE\", javax.crypto.spec;uses\\:\\=\"javax.crypto\";version\\=\"0.0.0.1_007_JavaSE\", javax.crypto;uses\\:\\=\"javax.crypto.spec\";version\\=\"0.0.0.1_007_JavaSE\", javax.imageio.event;uses\\:\\=\"javax.imageio\";version\\=\"0.0.0.1_007_JavaSE\", javax.imageio.metadata;uses\\:\\=\"org.w3c.dom,javax.imageio\";version\\=\"0.0.0.1_007_JavaSE\", javax.imageio.plugins.bmp;uses\\:\\=\"javax.imageio\";version\\=\"0.0.0.1_007_JavaSE\", javax.imageio.plugins.jpeg;uses\\:\\=\"javax.imageio\";version\\=\"0.0.0.1_007_JavaSE\", javax.imageio.spi;uses\\:\\=\"javax.imageio.stream,javax.imageio,javax.imageio.metadata\";version\\=\"0.0.0.1_007_JavaSE\", javax.imageio.stream;uses\\:\\=\"javax.imageio\";version\\=\"0.0.0.1_007_JavaSE\", javax.imageio;uses\\:\\=\"javax.imageio.metadata,javax.imageio.stream,javax.imageio.spi,javax.imageio.event\";version\\=\"0.0.0.1_007_JavaSE\", javax.lang.model.element;uses\\:\\=\"javax.lang.model.type,javax.lang.model\";version\\=\"0.0.0.1_007_JavaSE\", javax.lang.model.type;uses\\:\\=\"javax.lang.model.element,javax.lang.model\";version\\=\"0.0.0.1_007_JavaSE\", javax.lang.model.util;uses\\:\\=\"javax.lang.model,javax.lang.model.element,javax.annotation.processing,javax.lang.model.type\";version\\=\"0.0.0.1_007_JavaSE\", javax.lang.model;version\\=\"0.0.0.1_007_JavaSE\", javax.management.loading;uses\\:\\=\"javax.management\";version\\=\"0.0.0.1_007_JavaSE\", javax.management.modelmbean;uses\\:\\=\"javax.management,javax.management.loading\";version\\=\"0.0.0.1_007_JavaSE\", javax.management.monitor;uses\\:\\=\"javax.management\";version\\=\"0.0.0.1_007_JavaSE\", javax.management.openmbean;uses\\:\\=\"javax.management\";version\\=\"0.0.0.1_007_JavaSE\", javax.management.relation;uses\\:\\=\"javax.management\";version\\=\"0.0.0.1_007_JavaSE\", javax.management.remote.rmi;uses\\:\\=\"javax.management.remote,javax.security.auth,javax.management,javax.management.loading,javax.naming,javax.rmi.ssl,org.omg.CORBA,org.omg.CORBA_2_3.portable,org.omg.CORBA.portable,javax.rmi.CORBA,javax.rmi\";version\\=\"0.0.0.1_007_JavaSE\", javax.management.remote;uses\\:\\=\"javax.security.auth,javax.management\";version\\=\"0.0.0.1_007_JavaSE\", javax.management.timer;uses\\:\\=\"javax.management\";version\\=\"0.0.0.1_007_JavaSE\", javax.management;uses\\:\\=\"javax.management.loading,javax.management.openmbean\";version\\=\"0.0.0.1_007_JavaSE\", javax.naming.directory;uses\\:\\=\"javax.naming\";version\\=\"0.0.0.1_007_JavaSE\", javax.naming.event;uses\\:\\=\"javax.naming,javax.naming.directory\";version\\=\"0.0.0.1_007_JavaSE\", javax.naming.ldap;uses\\:\\=\"javax.naming,javax.naming.directory,javax.spret.ssl,javax.naming.event\";version\\=\"0.0.0.1_007_JavaSE\", javax.naming.spi;uses\\:\\=\"javax.naming,javax.naming.directory\";version\\=\"0.0.0.1_007_JavaSE\", javax.naming;uses\\:\\=\"javax.naming.spi\";version\\=\"0.0.0.1_007_JavaSE\", javax.net.ssl;uses\\:\\=\"javax.security.cert,javax.security.auth.x500,javax.net\";version\\=\"0.0.0.1_007_JavaSE\", javax.net;version\\=\"0.0.0.1_007_JavaSE\", javax.print.attribute.standard;uses\\:\\=\"javax.print.attribute\";version\\=\"0.0.0.1_007_JavaSE\", javax.print.attribute;version\\=\"0.0.0.1_007_JavaSE\", javax.print.event;uses\\:\\=\"javax.print,javax.print.attribute\";version\\=\"0.0.0.1_007_JavaSE\", javax.print;uses\\:\\=\"javax.print.attribute,javax.print.event,javax.print.attribute.standard\";version\\=\"0.0.0.1_007_JavaSE\", javax.rmi.CORBA;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA_2_3.portable,org.omg.CORBA.portable,org.omg.SendingContext\";version\\=\"0.0.0.1_007_JavaSE\", javax.rmi.ssl;uses\\:\\=\"javax.net,javax.net.ssl\";version\\=\"0.0.0.1_007_JavaSE\", javax.rmi;uses\\:\\=\"org.omg.CORBA,javax.rmi.CORBA\";version\\=\"0.0.0.1_007_JavaSE\", javax.script;version\\=\"0.0.0.1_007_JavaSE\", javax.security.auth.callback;version\\=\"0.0.0.1_007_JavaSE\", javax.security.auth.kerberos;uses\\:\\=\"javax.security.auth,javax.crypto\";version\\=\"0.0.0.1_007_JavaSE\", javax.security.auth.login;uses\\:\\=\"javax.security.auth,javax.security.auth.callback\";version\\=\"0.0.0.1_007_JavaSE\", javax.security.auth.spi;uses\\:\\=\"javax.security.auth.callback,javax.security.auth.login,javax.security.auth\";version\\=\"0.0.0.1_007_JavaSE\", javax.security.auth.x500;uses\\:\\=\"javax.security.auth\";version\\=\"0.0.0.1_007_JavaSE\", javax.security.auth;version\\=\"0.0.0.1_007_JavaSE\", javax.security.cert;version\\=\"0.0.0.1_007_JavaSE\", javax.security.sasl;uses\\:\\=\"javax.security.auth.callback\";version\\=\"0.0.0.1_007_JavaSE\", javax.sound.midi.spi;uses\\:\\=\"javax.sound.midi\";version\\=\"0.0.0.1_007_JavaSE\", javax.sound.midi;uses\\:\\=\"javax.sound.midi.spi\";version\\=\"0.0.0.1_007_JavaSE\", javax.sound.sampled.spi;uses\\:\\=\"javax.sound.sampled\";version\\=\"0.0.0.1_007_JavaSE\", javax.sound.sampled;uses\\:\\=\"javax.sound.sampled.spi\";version\\=\"0.0.0.1_007_JavaSE\", javax.sql.rowset.serial;uses\\:\\=\"javax.sql.rowset\";version\\=\"0.0.0.1_007_JavaSE\", javax.sql.rowset.spi;uses\\:\\=\"javax.sql,javax.naming,javax.sql.rowset\";version\\=\"0.0.0.1_007_JavaSE\", javax.sql.rowset;uses\\:\\=\"javax.sql,javax.sql.rowset.serial,javax.sql.rowset.spi\";version\\=\"0.0.0.1_007_JavaSE\", javax.sql;uses\\:\\=\"javax.transaction.xa\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.border;uses\\:\\=\"javax.swing\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.colorchooser;uses\\:\\=\"javax.swing,javax.swing.border,javax.swing.event,javax.swing.text\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.event;uses\\:\\=\"javax.swing,javax.swing.text,javax.swing.table,javax.swing.tree,javax.swing.undo\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.filechooser;uses\\:\\=\"javax.swing\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.plaf.basic;uses\\:\\=\"javax.swing.border,javax.swing,javax.swing.plaf,javax.swing.text,javax.swing.event,javax.swing.colorchooser,javax.accessibility,javax.swing.filechooser,javax.swing.text.html,javax.sound.sampled,javax.swing.table,javax.swing.plaf.synth,javax.swing.tree\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.plaf.metal;uses\\:\\=\"javax.swing.plaf,javax.swing,javax.swing.border,javax.swing.text,javax.swing.plaf.basic,javax.swing.filechooser,javax.swing.event,javax.swing.tree\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.plaf.multi;uses\\:\\=\"javax.accessibility,javax.swing,javax.swing.plaf,javax.swing.filechooser,javax.swing.text,javax.swing.tree\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.plaf.nimbus;uses\\:\\=\"javax.swing,javax.swing.plaf,javax.swing.border,javax.swing.plaf.synth\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.plaf.synth;uses\\:\\=\"javax.swing,javax.swing.plaf,javax.swing.text,javax.swing.border,javax.swing.plaf.basic,javax.swing.colorchooser,javax.swing.event,javax.xml.parsers,org.xml.sax,org.xml.sax.helpers,javax.swing.table,javax.swing.tree\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.plaf;uses\\:\\=\"javax.swing,javax.swing.border,javax.accessibility,javax.swing.filechooser,javax.swing.text,javax.swing.tree\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.table;uses\\:\\=\"javax.swing.event,javax.swing.plaf,javax.swing.border,javax.swing,javax.accessibility\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.text.html.parser;uses\\:\\=\"javax.swing.text,javax.swing.text.html\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.text.html;uses\\:\\=\"javax.swing.event,javax.swing.text,javax.accessibility,javax.swing,javax.swing.plaf,javax.swing.border,javax.swing.undo\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.text.rtf;uses\\:\\=\"javax.swing.text\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.text;uses\\:\\=\"javax.swing.event,javax.swing.tree,javax.swing.undo,javax.swing,javax.swing.plaf,javax.swing.plaf.basic,javax.print,javax.print.attribute,javax.accessibility,javax.swing.text.html\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.tree;uses\\:\\=\"javax.swing.event,javax.swing,javax.swing.border,javax.swing.plaf,javax.swing.plaf.basic\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.undo;uses\\:\\=\"javax.swing,javax.swing.event\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing;uses\\:\\=\"javax.swing.event,javax.accessibility,javax.swing.text,javax.swing.plaf,javax.swing.border,javax.swing.tree,javax.swing.table,javax.swing.colorchooser,javax.swing.plaf.basic,javax.swing.text.html,javax.swing.filechooser,javax.print,javax.print.attribute,javax.swing.plaf.metal\";version\\=\"0.0.0.1_007_JavaSE\", javax.tools;uses\\:\\=\"javax.lang.model.element,javax.annotation.processing,javax.lang.model\";version\\=\"0.0.0.1_007_JavaSE\", org.ietf.jgss;version\\=\"0.0.0.1_007_JavaSE\", org.omg.CORBA.DynAnyPackage;uses\\:\\=\"org.omg.CORBA\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CORBA.ORBPackage;uses\\:\\=\"org.omg.CORBA\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CORBA.TypeCodePackage;uses\\:\\=\"org.omg.CORBA\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CORBA.portable;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA_2_3.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CORBA;uses\\:\\=\"org.omg.CORBA.portable,org.omg.CORBA.DynAnyPackage,org.omg.CORBA.ORBPackage,org.omg.CORBA_2_3.portable,org.omg.CORBA.TypeCodePackage\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CORBA_2_3.portable;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CORBA_2_3;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CosNaming.NamingContextExtPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CosNaming.NamingContextPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable,org.omg.CosNaming\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CosNaming;uses\\:\\=\"org.omg.CORBA.portable,org.omg.CORBA,org.omg.PortableServer,org.omg.CosNaming.NamingContextPackage,org.omg.CosNaming.NamingContextExtPackage\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.Dynamic;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.DynamicAny.DynAnyFactoryPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.DynamicAny.DynAnyPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.DynamicAny;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable,org.omg.DynamicAny.DynAnyFactoryPackage,org.omg.DynamicAny.DynAnyPackage\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.IOP.CodecFactoryPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.IOP.CodecPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.IOP;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable,org.omg.IOP.CodecFactoryPackage,org.omg.IOP.CodecPackage\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.Messaging;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableInterceptor.ORBInitInfoPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableInterceptor;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable,org.omg.IOP,org.omg.PortableInterceptor.ORBInitInfoPackage,org.omg.CORBA_2_3.portable,org.omg.Dynamic\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableServer.CurrentPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableServer.POAManagerPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableServer.POAPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableServer.ServantLocatorPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableServer.portable;uses\\:\\=\"org.omg.CORBA,org.omg.PortableServer\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableServer;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable,org.omg.PortableServer.CurrentPackage,org.omg.PortableServer.POAManagerPackage,org.omg.PortableServer.POAPackage,org.omg.PortableServer.portable,org.omg.CORBA_2_3,org.omg.PortableServer.ServantLocatorPackage\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.SendingContext;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.stub.java.rmi;uses\\:\\=\"javax.rmi.CORBA\";version\\=\"0.0.0.1_007_JavaSE\"",
      "java",
      SeparatorTree.Separator.EQUALS,
      "javax.accessibility;uses\\:\\=\"javax.swing.text\";version\\=\"0.0.0.1_007_JavaSE\", javax.activity;version\\=\"0.0.0.1_007_JavaSE\", javax.annotation.processing;uses\\:\\=\"javax.tools,javax.lang.model,javax.lang.model.element,javax.lang.model.util\";version\\=\"0.0.0.1_007_JavaSE\", javax.annotation;version\\=\"0.0.0.1_007_JavaSE\", javax.crypto.interfaces;uses\\:\\=\"javax.crypto.spec,javax.crypto\";version\\=\"0.0.0.1_007_JavaSE\", javax.crypto.spec;uses\\:\\=\"javax.crypto\";version\\=\"0.0.0.1_007_JavaSE\", javax.crypto;uses\\:\\=\"javax.crypto.spec\";version\\=\"0.0.0.1_007_JavaSE\", javax.imageio.event;uses\\:\\=\"javax.imageio\";version\\=\"0.0.0.1_007_JavaSE\", javax.imageio.metadata;uses\\:\\=\"org.w3c.dom,javax.imageio\";version\\=\"0.0.0.1_007_JavaSE\", javax.imageio.plugins.bmp;uses\\:\\=\"javax.imageio\";version\\=\"0.0.0.1_007_JavaSE\", javax.imageio.plugins.jpeg;uses\\:\\=\"javax.imageio\";version\\=\"0.0.0.1_007_JavaSE\", javax.imageio.spi;uses\\:\\=\"javax.imageio.stream,javax.imageio,javax.imageio.metadata\";version\\=\"0.0.0.1_007_JavaSE\", javax.imageio.stream;uses\\:\\=\"javax.imageio\";version\\=\"0.0.0.1_007_JavaSE\", javax.imageio;uses\\:\\=\"javax.imageio.metadata,javax.imageio.stream,javax.imageio.spi,javax.imageio.event\";version\\=\"0.0.0.1_007_JavaSE\", javax.lang.model.element;uses\\:\\=\"javax.lang.model.type,javax.lang.model\";version\\=\"0.0.0.1_007_JavaSE\", javax.lang.model.type;uses\\:\\=\"javax.lang.model.element,javax.lang.model\";version\\=\"0.0.0.1_007_JavaSE\", javax.lang.model.util;uses\\:\\=\"javax.lang.model,javax.lang.model.element,javax.annotation.processing,javax.lang.model.type\";version\\=\"0.0.0.1_007_JavaSE\", javax.lang.model;version\\=\"0.0.0.1_007_JavaSE\", javax.management.loading;uses\\:\\=\"javax.management\";version\\=\"0.0.0.1_007_JavaSE\", javax.management.modelmbean;uses\\:\\=\"javax.management,javax.management.loading\";version\\=\"0.0.0.1_007_JavaSE\", javax.management.monitor;uses\\:\\=\"javax.management\";version\\=\"0.0.0.1_007_JavaSE\", javax.management.openmbean;uses\\:\\=\"javax.management\";version\\=\"0.0.0.1_007_JavaSE\", javax.management.relation;uses\\:\\=\"javax.management\";version\\=\"0.0.0.1_007_JavaSE\", javax.management.remote.rmi;uses\\:\\=\"javax.management.remote,javax.security.auth,javax.management,javax.management.loading,javax.naming,javax.rmi.ssl,org.omg.CORBA,org.omg.CORBA_2_3.portable,org.omg.CORBA.portable,javax.rmi.CORBA,javax.rmi\";version\\=\"0.0.0.1_007_JavaSE\", javax.management.remote;uses\\:\\=\"javax.security.auth,javax.management\";version\\=\"0.0.0.1_007_JavaSE\", javax.management.timer;uses\\:\\=\"javax.management\";version\\=\"0.0.0.1_007_JavaSE\", javax.management;uses\\:\\=\"javax.management.loading,javax.management.openmbean\";version\\=\"0.0.0.1_007_JavaSE\", javax.naming.directory;uses\\:\\=\"javax.naming\";version\\=\"0.0.0.1_007_JavaSE\", javax.naming.event;uses\\:\\=\"javax.naming,javax.naming.directory\";version\\=\"0.0.0.1_007_JavaSE\", javax.naming.ldap;uses\\:\\=\"javax.naming,javax.naming.directory,javax.spret.ssl,javax.naming.event\";version\\=\"0.0.0.1_007_JavaSE\", javax.naming.spi;uses\\:\\=\"javax.naming,javax.naming.directory\";version\\=\"0.0.0.1_007_JavaSE\", javax.naming;uses\\:\\=\"javax.naming.spi\";version\\=\"0.0.0.1_007_JavaSE\", javax.net.ssl;uses\\:\\=\"javax.security.cert,javax.security.auth.x500,javax.net\";version\\=\"0.0.0.1_007_JavaSE\", javax.net;version\\=\"0.0.0.1_007_JavaSE\", javax.print.attribute.standard;uses\\:\\=\"javax.print.attribute\";version\\=\"0.0.0.1_007_JavaSE\", javax.print.attribute;version\\=\"0.0.0.1_007_JavaSE\", javax.print.event;uses\\:\\=\"javax.print,javax.print.attribute\";version\\=\"0.0.0.1_007_JavaSE\", javax.print;uses\\:\\=\"javax.print.attribute,javax.print.event,javax.print.attribute.standard\";version\\=\"0.0.0.1_007_JavaSE\", javax.rmi.CORBA;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA_2_3.portable,org.omg.CORBA.portable,org.omg.SendingContext\";version\\=\"0.0.0.1_007_JavaSE\", javax.rmi.ssl;uses\\:\\=\"javax.net,javax.net.ssl\";version\\=\"0.0.0.1_007_JavaSE\", javax.rmi;uses\\:\\=\"org.omg.CORBA,javax.rmi.CORBA\";version\\=\"0.0.0.1_007_JavaSE\", javax.script;version\\=\"0.0.0.1_007_JavaSE\", javax.security.auth.callback;version\\=\"0.0.0.1_007_JavaSE\", javax.security.auth.kerberos;uses\\:\\=\"javax.security.auth,javax.crypto\";version\\=\"0.0.0.1_007_JavaSE\", javax.security.auth.login;uses\\:\\=\"javax.security.auth,javax.security.auth.callback\";version\\=\"0.0.0.1_007_JavaSE\", javax.security.auth.spi;uses\\:\\=\"javax.security.auth.callback,javax.security.auth.login,javax.security.auth\";version\\=\"0.0.0.1_007_JavaSE\", javax.security.auth.x500;uses\\:\\=\"javax.security.auth\";version\\=\"0.0.0.1_007_JavaSE\", javax.security.auth;version\\=\"0.0.0.1_007_JavaSE\", javax.security.cert;version\\=\"0.0.0.1_007_JavaSE\", javax.security.sasl;uses\\:\\=\"javax.security.auth.callback\";version\\=\"0.0.0.1_007_JavaSE\", javax.sound.midi.spi;uses\\:\\=\"javax.sound.midi\";version\\=\"0.0.0.1_007_JavaSE\", javax.sound.midi;uses\\:\\=\"javax.sound.midi.spi\";version\\=\"0.0.0.1_007_JavaSE\", javax.sound.sampled.spi;uses\\:\\=\"javax.sound.sampled\";version\\=\"0.0.0.1_007_JavaSE\", javax.sound.sampled;uses\\:\\=\"javax.sound.sampled.spi\";version\\=\"0.0.0.1_007_JavaSE\", javax.sql.rowset.serial;uses\\:\\=\"javax.sql.rowset\";version\\=\"0.0.0.1_007_JavaSE\", javax.sql.rowset.spi;uses\\:\\=\"javax.sql,javax.naming,javax.sql.rowset\";version\\=\"0.0.0.1_007_JavaSE\", javax.sql.rowset;uses\\:\\=\"javax.sql,javax.sql.rowset.serial,javax.sql.rowset.spi\";version\\=\"0.0.0.1_007_JavaSE\", javax.sql;uses\\:\\=\"javax.transaction.xa\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.border;uses\\:\\=\"javax.swing\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.colorchooser;uses\\:\\=\"javax.swing,javax.swing.border,javax.swing.event,javax.swing.text\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.event;uses\\:\\=\"javax.swing,javax.swing.text,javax.swing.table,javax.swing.tree,javax.swing.undo\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.filechooser;uses\\:\\=\"javax.swing\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.plaf.basic;uses\\:\\=\"javax.swing.border,javax.swing,javax.swing.plaf,javax.swing.text,javax.swing.event,javax.swing.colorchooser,javax.accessibility,javax.swing.filechooser,javax.swing.text.html,javax.sound.sampled,javax.swing.table,javax.swing.plaf.synth,javax.swing.tree\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.plaf.metal;uses\\:\\=\"javax.swing.plaf,javax.swing,javax.swing.border,javax.swing.text,javax.swing.plaf.basic,javax.swing.filechooser,javax.swing.event,javax.swing.tree\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.plaf.multi;uses\\:\\=\"javax.accessibility,javax.swing,javax.swing.plaf,javax.swing.filechooser,javax.swing.text,javax.swing.tree\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.plaf.nimbus;uses\\:\\=\"javax.swing,javax.swing.plaf,javax.swing.border,javax.swing.plaf.synth\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.plaf.synth;uses\\:\\=\"javax.swing,javax.swing.plaf,javax.swing.text,javax.swing.border,javax.swing.plaf.basic,javax.swing.colorchooser,javax.swing.event,javax.xml.parsers,org.xml.sax,org.xml.sax.helpers,javax.swing.table,javax.swing.tree\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.plaf;uses\\:\\=\"javax.swing,javax.swing.border,javax.accessibility,javax.swing.filechooser,javax.swing.text,javax.swing.tree\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.table;uses\\:\\=\"javax.swing.event,javax.swing.plaf,javax.swing.border,javax.swing,javax.accessibility\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.text.html.parser;uses\\:\\=\"javax.swing.text,javax.swing.text.html\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.text.html;uses\\:\\=\"javax.swing.event,javax.swing.text,javax.accessibility,javax.swing,javax.swing.plaf,javax.swing.border,javax.swing.undo\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.text.rtf;uses\\:\\=\"javax.swing.text\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.text;uses\\:\\=\"javax.swing.event,javax.swing.tree,javax.swing.undo,javax.swing,javax.swing.plaf,javax.swing.plaf.basic,javax.print,javax.print.attribute,javax.accessibility,javax.swing.text.html\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.tree;uses\\:\\=\"javax.swing.event,javax.swing,javax.swing.border,javax.swing.plaf,javax.swing.plaf.basic\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing.undo;uses\\:\\=\"javax.swing,javax.swing.event\";version\\=\"0.0.0.1_007_JavaSE\", javax.swing;uses\\:\\=\"javax.swing.event,javax.accessibility,javax.swing.text,javax.swing.plaf,javax.swing.border,javax.swing.tree,javax.swing.table,javax.swing.colorchooser,javax.swing.plaf.basic,javax.swing.text.html,javax.swing.filechooser,javax.print,javax.print.attribute,javax.swing.plaf.metal\";version\\=\"0.0.0.1_007_JavaSE\", javax.tools;uses\\:\\=\"javax.lang.model.element,javax.annotation.processing,javax.lang.model\";version\\=\"0.0.0.1_007_JavaSE\", org.ietf.jgss;version\\=\"0.0.0.1_007_JavaSE\", org.omg.CORBA.DynAnyPackage;uses\\:\\=\"org.omg.CORBA\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CORBA.ORBPackage;uses\\:\\=\"org.omg.CORBA\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CORBA.TypeCodePackage;uses\\:\\=\"org.omg.CORBA\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CORBA.portable;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA_2_3.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CORBA;uses\\:\\=\"org.omg.CORBA.portable,org.omg.CORBA.DynAnyPackage,org.omg.CORBA.ORBPackage,org.omg.CORBA_2_3.portable,org.omg.CORBA.TypeCodePackage\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CORBA_2_3.portable;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CORBA_2_3;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CosNaming.NamingContextExtPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CosNaming.NamingContextPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable,org.omg.CosNaming\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.CosNaming;uses\\:\\=\"org.omg.CORBA.portable,org.omg.CORBA,org.omg.PortableServer,org.omg.CosNaming.NamingContextPackage,org.omg.CosNaming.NamingContextExtPackage\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.Dynamic;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.DynamicAny.DynAnyFactoryPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.DynamicAny.DynAnyPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.DynamicAny;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable,org.omg.DynamicAny.DynAnyFactoryPackage,org.omg.DynamicAny.DynAnyPackage\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.IOP.CodecFactoryPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.IOP.CodecPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.IOP;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable,org.omg.IOP.CodecFactoryPackage,org.omg.IOP.CodecPackage\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.Messaging;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableInterceptor.ORBInitInfoPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableInterceptor;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable,org.omg.IOP,org.omg.PortableInterceptor.ORBInitInfoPackage,org.omg.CORBA_2_3.portable,org.omg.Dynamic\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableServer.CurrentPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableServer.POAManagerPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableServer.POAPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableServer.ServantLocatorPackage;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableServer.portable;uses\\:\\=\"org.omg.CORBA,org.omg.PortableServer\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.PortableServer;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable,org.omg.PortableServer.CurrentPackage,org.omg.PortableServer.POAManagerPackage,org.omg.PortableServer.POAPackage,org.omg.PortableServer.portable,org.omg.CORBA_2_3,org.omg.PortableServer.ServantLocatorPackage\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.SendingContext;uses\\:\\=\"org.omg.CORBA,org.omg.CORBA.portable\";version\\=\"0.0.0.1_007_JavaSE\", org.omg.stub.java.rmi;uses\\:\\=\"javax.rmi.CORBA\";version\\=\"0.0.0.1_007_JavaSE\"");
  }

  private void checkParsed(String toParse, String key, SeparatorTree.Separator separator, @Nullable String value) {
    PropertyTree propertyTree = (PropertyTree) JavaPropertiesParserBuilder
      .createTestParser(Charsets.ISO_8859_1, JavaPropertiesLexicalGrammar.PROPERTY)
      .parse(toParse);

    assertPropertyTree(propertyTree, key, separator, value);
  }

  private void assertPropertyTree(PropertyTree propertyTree, String key, SeparatorTree.Separator separator, @Nullable String value) {
    assertThat(propertyTree.key().text()).isEqualTo(key);
    assertThat(propertyTree.separator().separator()).isEqualTo(separator);

    if (value == null) {
      assertThat(propertyTree.value()).isNull();
    } else {
      assertThat(propertyTree.value()).isNotNull();
      assertThat(propertyTree.value().text()).isEqualTo(value);
    }

  }

}
