#nullable disable
using System;
using System.Collections.Generic;
using System.Collections.Immutable;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Bencodex.Types;
using Libplanet.Action;
using Libplanet.Action.Loader;
using Libplanet.Action.State;
using Libplanet.Action.Tests.Common;
using Libplanet.Blockchain;
using Libplanet.Blockchain.Policies;
using Libplanet.Blockchain.Renderers.Debug;
using Libplanet.Crypto;
using Libplanet.Store;
using Libplanet.Store.Trie;
using Libplanet.Tests;
using Libplanet.Tests.Store;
using Libplanet.Types.Blocks;
using Libplanet.Types.Consensus;
using Libplanet.Types.Evidence;
using Libplanet.Types.Tx;
using Serilog;
using Serilog.Events;
using xRetry;
using Xunit;
using static Libplanet.Tests.TestUtils;

namespace Libplanet.Net.Tests
{
    public partial class SwarmTest
    {
        [Fact(Timeout = Timeout)]
        public async Task InitialBlockDownload()
        {
            var minerKey = new PrivateKey();

            Swarm minerSwarm = await CreateSwarm(minerKey).ConfigureAwait(false);
            Swarm receiverSwarm = await CreateSwarm().ConfigureAwait(false);

            BlockChain minerChain = minerSwarm.BlockChain;
            BlockChain receiverChain = receiverSwarm.BlockChain;

            foreach (int i in Enumerable.Range(0, 10))
            {
                Block block = minerChain.ProposeBlock(
                    minerKey, CreateBlockCommit(minerChain.Tip));
                minerChain.Append(block, TestUtils.CreateBlockCommit(block));
            }

            try
            {
                await StartAsync(minerSwarm);
                await receiverSwarm.AddPeersAsync(new[] { minerSwarm.AsPeer }, null);

                await receiverSwarm.PreloadAsync();

                Assert.Equal(minerChain.BlockHashes, receiverChain.BlockHashes);
            }
            finally
            {
                CleaningSwarm(minerSwarm);
                CleaningSwarm(receiverSwarm);
            }
        }

        [Fact(Timeout = Timeout)]
        public async Task InitialBlockDownloadStates()
        {
            var minerKey = new PrivateKey();

            Swarm minerSwarm = await CreateSwarm(minerKey).ConfigureAwait(false);
            Swarm receiverSwarm = await CreateSwarm().ConfigureAwait(false);

            BlockChain minerChain = minerSwarm.BlockChain;
            BlockChain receiverChain = receiverSwarm.BlockChain;

            var key = new PrivateKey();
            var address1 = key.Address;
            var address2 = new PrivateKey().Address;

            var action = DumbAction.Create(
                (address1, "foo"),
                transfer: (null, address2, 10));

            minerChain.MakeTransaction(key, new[] { action });
            var block = minerChain.ProposeBlock(
                minerKey, CreateBlockCommit(minerChain.Tip));
            minerChain.Append(block, TestUtils.CreateBlockCommit(block));

            minerChain.MakeTransaction(key, new[] { DumbAction.Create((address1, "bar")) });
            block = minerChain.ProposeBlock(
                minerKey, CreateBlockCommit(minerChain.Tip));
            minerChain.Append(block, TestUtils.CreateBlockCommit(block));

            minerChain.MakeTransaction(key, new[] { DumbAction.Create((address1, "baz")) });
            block = minerChain.ProposeBlock(
                minerKey, CreateBlockCommit(minerChain.Tip));
            minerChain.Append(block, TestUtils.CreateBlockCommit(block));

            try
            {
                await StartAsync(minerSwarm);
                await receiverSwarm.AddPeersAsync(new[] { minerSwarm.AsPeer }, null);

                await receiverSwarm.PreloadAsync();
                var state = receiverChain
                    .GetNextWorldState()
                    .GetAccountState(ReservedAddresses.LegacyAccount)
                    .GetState(address1);

                Assert.Equal((Text)"foo,bar,baz", state);
                Assert.Equal(minerChain.BlockHashes, receiverChain.BlockHashes);
            }
            finally
            {
                CleaningSwarm(minerSwarm);
                CleaningSwarm(receiverSwarm);
            }
        }

        [Fact(Timeout = Timeout)]
        public async Task Preload()
        {
            Swarm minerSwarm = await CreateSwarm().ConfigureAwait(false);
            Swarm receiverSwarm = await CreateSwarm().ConfigureAwait(false);

            BlockChain minerChain = minerSwarm.BlockChain;
            BlockChain receiverChain = receiverSwarm.BlockChain;

            var blocks = new List<Block>();
            foreach (int i in Enumerable.Range(0, 12))
            {
                Block block = minerChain.EvaluateAndSign(
                    ProposeNext(
                        previousBlock: i == 0 ? minerChain.Genesis : blocks[i - 1],
                        miner: GenesisProposer.PublicKey,
                        lastCommit: CreateBlockCommit(minerChain.Tip)),
                    GenesisProposer);
                blocks.Add(block);
                if (i != 11)
                {
                    minerChain.Append(blocks[i], TestUtils.CreateBlockCommit(blocks[i]));
                }
            }

            var actualStates = new List<BlockSyncState>();
            var progress = new ActionProgress<BlockSyncState>(state =>
            {
                _logger.Information("Received a progress event: {@State}", state);
                lock (actualStates)
                {
                    actualStates.Add(state);

                    if (actualStates.Count == 10)
                    {
                        minerChain.Append(blocks[11], CreateBlockCommit(blocks[11]));
                    }
                }
            });

            try
            {
                await StartAsync(minerSwarm);

                await receiverSwarm.AddPeersAsync(new[] { minerSwarm.AsPeer }, null);

                _logger.Verbose("Both chains before synchronization:");
                _logger.CompareBothChains(
                    LogEventLevel.Verbose,
                    "Miner chain",
                    minerChain,
                    "Receiver chain",
                    receiverChain
                );

                minerSwarm.FindNextHashesChunkSize = 3;
                await receiverSwarm.PreloadAsync(
                    TimeSpan.FromSeconds(1),
                    0,
                    progress);

                // Await 1 second to make sure all progresses is reported.
                await Task.Delay(1000);

                _logger.Verbose(
                    $"Both chains after synchronization ({nameof(receiverSwarm.PreloadAsync)}):"
                );
                _logger.CompareBothChains(
                    LogEventLevel.Verbose,
                    "Miner chain",
                    minerChain,
                    "Receiver chain",
                    receiverChain
                );

                Assert.Equal(minerChain.Tip.Hash, receiverChain.Tip.Hash);
            }
            finally
            {
                CleaningSwarm(minerSwarm);
                CleaningSwarm(receiverSwarm);
            }
        }

        [Fact(
            Skip = "Scenario is no more possible since validator set has moved to state.",
            Timeout = Timeout)]
        public async Task PreloadWithMaliciousPeer()
        {
            const int initialSharedTipHeight = 3;
            const int maliciousTipHeight = 5;
            const int honestTipHeight = 7;
            var policy = new NullBlockPolicy();
            var policyB = new NullBlockPolicy();
            var genesis = new MemoryStoreFixture(policy.PolicyActionsRegistry).GenesisBlock;

            var swarmA = await CreateSwarm(
                privateKey: new PrivateKey(),
                policy: policy,
                genesis: genesis).ConfigureAwait(false);
            var swarmB = await CreateSwarm(
                privateKey: new PrivateKey(),
                policy: policyB,
                genesis: genesis).ConfigureAwait(false);
            var swarmC = await CreateSwarm(
                privateKey: new PrivateKey(),
                policy: policy,
                genesis: genesis).ConfigureAwait(false);
            var chainA = swarmA.BlockChain;
            var chainB = swarmB.BlockChain;
            var chainC = swarmC.BlockChain;

            // Setup initial state where all chains share the same blockchain state.
            for (int i = 1; i <= initialSharedTipHeight; i++)
            {
                var block = chainA.ProposeBlock(
                    new PrivateKey(), TestUtils.CreateBlockCommit(chainA.Tip));
                chainA.Append(block, TestUtils.CreateBlockCommit(block));
                chainB.Append(block, TestUtils.CreateBlockCommit(block));
                chainC.Append(block, TestUtils.CreateBlockCommit(block));
            }

            // Setup malicious node to broadcast.
            for (int i = initialSharedTipHeight + 1; i < maliciousTipHeight; i++)
            {
                var block = chainB.ProposeBlock(
                    new PrivateKey(), TestUtils.CreateBlockCommit(chainB.Tip));
                chainB.Append(block, TestUtils.CreateBlockCommit(block));
                chainC.Append(block, TestUtils.CreateBlockCommit(block));
            }

            var specialBlock = chainB.ProposeBlock(
                new PrivateKey(), TestUtils.CreateBlockCommit(chainB.Tip));
            var invalidBlockCommit = new BlockCommit(
                maliciousTipHeight,
                0,
                specialBlock.Hash,
                ImmutableArray<Vote>.Empty
                    .Add(new VoteMetadata(
                        maliciousTipHeight,
                        0,
                        specialBlock.Hash,
                        DateTimeOffset.UtcNow,
                        TestUtils.PrivateKeys[0].PublicKey,
                        TestUtils.ValidatorSet[0].Power,
                        VoteFlag.PreCommit).Sign(TestUtils.PrivateKeys[0])));
            var validBlockCommit = TestUtils.CreateBlockCommit(specialBlock);
            chainB.Append(specialBlock, invalidBlockCommit);
            chainC.Append(specialBlock, validBlockCommit);

            // Setup honest node with higher tip
            for (int i = maliciousTipHeight + 1; i <= honestTipHeight; i++)
            {
                var block = chainC.ProposeBlock(
                    new PrivateKey(), TestUtils.CreateBlockCommit(chainC.Tip));
                chainC.Append(block, TestUtils.CreateBlockCommit(block));
            }

            Assert.Equal(initialSharedTipHeight, chainA.Tip.Index);
            Assert.Equal(maliciousTipHeight, chainB.Tip.Index);
            Assert.Equal(honestTipHeight, chainC.Tip.Index);

            try
            {
                await StartAsync(swarmA, millisecondsBroadcastBlockInterval: int.MaxValue);
                await StartAsync(swarmB, millisecondsBroadcastBlockInterval: int.MaxValue);
                await StartAsync(swarmC, millisecondsBroadcastBlockInterval: int.MaxValue);

                // Checks swarmB cannot make swarmA append a block with invalid block commit.
                await swarmA.AddPeersAsync(new[] { swarmB.AsPeer }, null);
                await swarmB.AddPeersAsync(new[] { swarmA.AsPeer }, null);

                try
                {
                    await swarmA.PreloadAsync();
                }
                catch (InvalidBlockCommitException)
                {
                }

                // Makes sure preload failed.
                Assert.Equal(initialSharedTipHeight, chainA.Tip.Index);

                // Checks swarmA can sync with an honest node with higher tip afterwards.
                await swarmA.AddPeersAsync(new[] { swarmC.AsPeer }, null);
                await swarmC.AddPeersAsync(new[] { swarmA.AsPeer }, null);

                await swarmA.PreloadAsync();

                Assert.Equal(chainC.Tip, chainA.Tip);
                Assert.Equal(
                    chainC.GetBlockCommit(chainC.Tip.Hash),
                    chainA.GetBlockCommit(chainA.Tip.Hash));
            }
            finally
            {
                CleaningSwarm(swarmA);
                CleaningSwarm(swarmB);
                CleaningSwarm(swarmC);
            }
        }

        [RetryFact(Timeout = Timeout)]
        public async Task NoRenderInPreload()
        {
            var policy = new BlockPolicy(
                new PolicyActionsRegistry(
                    endBlockActions: ImmutableArray.Create<IAction>(new MinerReward(1))));
            var renderer = new RecordingActionRenderer();
            var chain = MakeBlockChain(
                policy,
                new MemoryStore(),
                new TrieStateStore(new MemoryKeyValueStore()),
                new SingleActionLoader(typeof(DumbAction)),
                renderers: new[] { renderer });

            var senderKey = new PrivateKey();

            var receiver = await CreateSwarm(chain).ConfigureAwait(false);
            var sender = await CreateSwarm(
                MakeBlockChain(
                    policy,
                    new MemoryStore(),
                    new TrieStateStore(new MemoryKeyValueStore()),
                    new SingleActionLoader(typeof(DumbAction))
                ),
                senderKey
            ).ConfigureAwait(false);

            int renderCount = 0;

            var privKey = new PrivateKey();
            var addr = sender.Address;
            var item = "foo";

            const int iteration = 3;
            for (var i = 0; i < iteration; i++)
            {
                sender.BlockChain.MakeTransaction(
                    privKey, new[] { DumbAction.Create((addr, item)) });
                Block block = sender.BlockChain.ProposeBlock(
                    senderKey, CreateBlockCommit(sender.BlockChain.Tip));
                sender.BlockChain.Append(block, TestUtils.CreateBlockCommit(block));
            }

            renderer.RenderEventHandler += (_, a) =>
                renderCount += IsDumbAction(a) ? 1 : 0;

            await StartAsync(receiver);
            await StartAsync(sender);

            await BootstrapAsync(receiver, sender.AsPeer);
            await receiver.PreloadAsync();

            Assert.Equal(sender.BlockChain.Tip, receiver.BlockChain.Tip);
            Assert.Equal(sender.BlockChain.Count, receiver.BlockChain.Count);
            Assert.Equal(0, renderCount);

            CleaningSwarm(receiver);
            CleaningSwarm(sender);
        }

        [Fact(Timeout = Timeout)]
        public async Task PreloadWithFailedActions()
        {
            var policy = new BlockPolicy();
            var fx1 = new MemoryStoreFixture();
            var fx2 = new MemoryStoreFixture();
            var minerChain = MakeBlockChain(
                policy, fx1.Store, fx1.StateStore, new SingleActionLoader(typeof(ThrowException)));
            var receiverChain = MakeBlockChain(
                policy, fx2.Store, fx2.StateStore, new SingleActionLoader(typeof(ThrowException)));

            var minerKey = new PrivateKey();

            Swarm minerSwarm =
                await CreateSwarm(minerChain, minerKey).ConfigureAwait(false);
            Swarm receiverSwarm =
                await CreateSwarm(receiverChain).ConfigureAwait(false);

            foreach (var unused in Enumerable.Range(0, 10))
            {
                Block block = minerSwarm.BlockChain.ProposeBlock(
                    minerKey, CreateBlockCommit(minerSwarm.BlockChain.Tip));
                minerSwarm.BlockChain.Append(block, TestUtils.CreateBlockCommit(block));
            }

            try
            {
                await StartAsync(minerSwarm);

                await receiverSwarm.AddPeersAsync(new[] { minerSwarm.AsPeer }, null);
                await receiverSwarm.PreloadAsync();

                var action = new ThrowException { ThrowOnExecution = true };

                var chainId = receiverChain.Id;
                Transaction tx = Transaction.Create(
                    0,
                    new PrivateKey(),
                    minerSwarm.BlockChain.Genesis.Hash,
                    new[] { action }.ToPlainValues(),
                    null,
                    null,
                    DateTimeOffset.UtcNow);

                Block block = minerChain.ProposeBlock(
                    GenesisProposer,
                    new[] { tx }.ToImmutableList(),
                    CreateBlockCommit(minerChain.Tip),
                    ImmutableArray<EvidenceBase>.Empty);
                minerSwarm.BlockChain.Append(block, CreateBlockCommit(block), true);

                await receiverSwarm.PreloadAsync();

                // Preloading should succeed even if action throws exception.
                Assert.Equal(minerChain.Tip, receiverChain.Tip);
            }
            finally
            {
                CleaningSwarm(minerSwarm);
                CleaningSwarm(receiverSwarm);
            }
        }

        [Fact(Timeout = Timeout)]
        public async Task PreloadFromNominer()
        {
            var minerKey = new PrivateKey();
            Swarm minerSwarm = await CreateSwarm(minerKey).ConfigureAwait(false);
            Swarm receiverSwarm = await CreateSwarm().ConfigureAwait(false);
            var fxForNominers = new StoreFixture[2];
            var policy = new BlockPolicy(
                new PolicyActionsRegistry(
                    endBlockActions: ImmutableArray.Create<IAction>(new MinerReward(1))));
            fxForNominers[0] = new MemoryStoreFixture(policy.PolicyActionsRegistry);
            fxForNominers[1] = new MemoryStoreFixture(policy.PolicyActionsRegistry);
            var blockChainsForNominers = new[]
            {
                MakeBlockChain(
                    policy,
                    fxForNominers[0].Store,
                    fxForNominers[0].StateStore,
                    new SingleActionLoader(typeof(DumbAction))),
                MakeBlockChain(
                    policy,
                    fxForNominers[1].Store,
                    fxForNominers[1].StateStore,
                    new SingleActionLoader(typeof(DumbAction))),
            };
            var nominerSwarm0 =
                await CreateSwarm(blockChainsForNominers[0]).ConfigureAwait(false);
            var nominerSwarm1 =
                await CreateSwarm(blockChainsForNominers[1]).ConfigureAwait(false);

            BlockChain minerChain = minerSwarm.BlockChain;
            BlockChain receiverChain = receiverSwarm.BlockChain;

            foreach (int i in Enumerable.Range(0, 10))
            {
                Block block = minerChain.ProposeBlock(
                    minerKey, CreateBlockCommit(minerChain.Tip));
                minerChain.Append(block, CreateBlockCommit(block));
            }

            var actualStates = new List<BlockSyncState>();
            var progress = new ActionProgress<BlockSyncState>(state =>
            {
                lock (actualStates)
                {
                    actualStates.Add(state);
                }
            });

            try
            {
                await StartAsync(minerSwarm);
                await StartAsync(nominerSwarm0);
                await StartAsync(nominerSwarm1);
                minerSwarm.FindNextHashesChunkSize = 2;
                nominerSwarm0.FindNextHashesChunkSize = 2;
                nominerSwarm1.FindNextHashesChunkSize = 2;

                await nominerSwarm0.AddPeersAsync(new[] { minerSwarm.AsPeer }, null);
                await nominerSwarm0.PreloadAsync(TimeSpan.FromSeconds(1), 0);
                await nominerSwarm1.AddPeersAsync(new[] { nominerSwarm0.AsPeer }, null);
                await nominerSwarm1.PreloadAsync(TimeSpan.FromSeconds(1), 0);
                await receiverSwarm.AddPeersAsync(new[] { nominerSwarm1.AsPeer }, null);
                await receiverSwarm.PreloadAsync(TimeSpan.FromSeconds(1), 0, progress);

                // Await 1 second to make sure all progresses is reported.
                await Task.Delay(1000);

                Assert.Equal(minerChain.BlockHashes, receiverChain.BlockHashes);
            }
            finally
            {
                CleaningSwarm(minerSwarm);
                CleaningSwarm(nominerSwarm0);
                CleaningSwarm(nominerSwarm1);
                CleaningSwarm(receiverSwarm);

                fxForNominers[0].Dispose();
                fxForNominers[1].Dispose();
            }
        }

        [Theory(Timeout = Timeout)]
        [InlineData(4)]
        [InlineData(8)]
        [InlineData(16)]
        [InlineData(32)]
        public async Task PreloadRetryWithNextPeers(int blockCount)
        {
            var key0 = new PrivateKey();

            Swarm swarm0 = await CreateSwarm(key0).ConfigureAwait(false);
            Swarm swarm1 = await CreateSwarm().ConfigureAwait(false);
            Swarm receiverSwarm = await CreateSwarm().ConfigureAwait(false);

            receiverSwarm.Options.TimeoutOptions.GetBlockHashesTimeout
                = TimeSpan.FromMilliseconds(1000);

            swarm0.FindNextHashesChunkSize = blockCount / 2;
            swarm1.FindNextHashesChunkSize = blockCount / 2;

            for (int i = 0; i < blockCount; ++i)
            {
                var block = swarm0.BlockChain.ProposeBlock(
                    key0, CreateBlockCommit(swarm0.BlockChain.Tip));
                swarm0.BlockChain.Append(block, TestUtils.CreateBlockCommit(block));
                swarm1.BlockChain.Append(block, TestUtils.CreateBlockCommit(block));
            }

            await StartAsync(swarm0);
            await StartAsync(swarm1);

            Assert.Equal(swarm0.BlockChain.BlockHashes, swarm1.BlockChain.BlockHashes);

            await receiverSwarm.AddPeersAsync(new[] { swarm0.AsPeer, swarm1.AsPeer }, null);
            Assert.Equal(
                new[] { swarm0.AsPeer, swarm1.AsPeer }.ToImmutableHashSet(),
                receiverSwarm.Peers.ToImmutableHashSet());

            var startedStop = false;
            var shouldStopSwarm =
                swarm0.AsPeer.Equals(receiverSwarm.Peers.First()) ? swarm0 : swarm1;

            // FIXME: This relies on progress report to artificially stop preloading.
            async void Action(BlockSyncState state)
            {
                if (!startedStop)
                {
                    startedStop = true;
                    await shouldStopSwarm.StopAsync(TimeSpan.Zero);
                }
            }

            await receiverSwarm.PreloadAsync(
                TimeSpan.FromSeconds(1),
                0,
                progress: new ActionProgress<BlockSyncState>(Action));

            Assert.Equal(swarm1.BlockChain.BlockHashes, receiverSwarm.BlockChain.BlockHashes);
            Assert.Equal(swarm0.BlockChain.BlockHashes, receiverSwarm.BlockChain.BlockHashes);

            CleaningSwarm(swarm0);
            CleaningSwarm(swarm1);
            CleaningSwarm(receiverSwarm);
        }

        [RetryTheory(10, Timeout = Timeout)]
        [InlineData(0)]
        [InlineData(50)]
        [InlineData(100)]
        [InlineData(500)]
        [InlineData(1000)]
        [InlineData(2500)]
        [InlineData(5000)]
        public async Task PreloadAsyncCancellation(int cancelAfter)
        {
            Swarm minerSwarm = await CreateSwarm().ConfigureAwait(false);
            Swarm receiverSwarm = await CreateSwarm().ConfigureAwait(false);
            Log.Logger.Information("Miner:    {0}", minerSwarm.Address);
            Log.Logger.Information("Receiver: {0}", receiverSwarm.Address);

            BlockChain minerChain = minerSwarm.BlockChain;
            BlockChain receiverChain = receiverSwarm.BlockChain;

            Guid receiverChainId = receiverChain.Id;

            (Address address, IEnumerable<Block> blocks) =
                MakeFixtureBlocksForPreloadAsyncCancellationTest();

            var blockArray = blocks.ToArray();
            foreach (Block block in blockArray)
            {
                minerChain.Append(block, CreateBlockCommit(block));
            }

            receiverChain.Append(blockArray[0], CreateBlockCommit(blockArray[0]));

            Assert.NotNull(minerChain.Tip);

            minerSwarm.FindNextHashesChunkSize = 2;
            await StartAsync(minerSwarm);
            await receiverSwarm.AddPeersAsync(new[] { minerSwarm.AsPeer }, null);

            CancellationTokenSource cts = new CancellationTokenSource();
            cts.CancelAfter(cancelAfter);
            bool canceled = true;
            try
            {
                await receiverSwarm.PreloadAsync(
                    TimeSpan.FromSeconds(1),
                    0,
                    cancellationToken: cts.Token);
                canceled = false;
                Log.Logger.Debug(
                    "{MethodName}() normally finished", nameof(receiverSwarm.PreloadAsync));
            }
            catch (OperationCanceledException)
            {
                Log.Logger.Debug("{MethodName}() aborted", nameof(receiverSwarm.PreloadAsync));
            }
            catch (AggregateException ae) when (ae.InnerException is TaskCanceledException)
            {
                Log.Logger.Debug("{MethodName}() aborted", nameof(receiverSwarm.PreloadAsync));
            }

            cts.Dispose();

            Assert.InRange(receiverChain.Store.ListChainIds().Count(), 0, 1);

            if (canceled)
            {
                Assert.Equal(receiverChainId, receiverChain.Id);
                Assert.Contains(receiverChain.Tip, blockArray);
                Assert.NotEqual(blockArray.Last(), receiverChain.Tip);
                Assert.Equal(
                    (Text)string.Join(
                        ",",
                        Enumerable.Range(0, (int)receiverChain.Tip.Index).Select(i =>
                            string.Join(",", Enumerable.Range(0, 5).Select(j => $"Item{i}.{j}")))),
                    receiverChain
                        .GetNextWorldState()
                        .GetAccountState(ReservedAddresses.LegacyAccount)
                        .GetState(address));
            }
            else
            {
                Assert.Equal(minerChain.Tip, receiverChain.Tip);
                Assert.Equal(
                    (Text)string.Join(
                        ",",
                        Enumerable.Range(0, 20).Select(i =>
                            string.Join(",", Enumerable.Range(0, 5).Select(j => $"Item{i}.{j}")))),
                    receiverChain
                        .GetNextWorldState()
                        .GetAccountState(ReservedAddresses.LegacyAccount)
                        .GetState(address)
                );
            }

            CleaningSwarm(minerSwarm);
            CleaningSwarm(receiverSwarm);
        }

        [Fact(Timeout = Timeout)]
        public async Task GetDemandBlockHashes()
        {
            Swarm minerSwarm = await CreateSwarm().ConfigureAwait(false);
            Swarm receiverSwarm = await CreateSwarm().ConfigureAwait(false);
            Log.Logger.Information("Miner:    {0}", minerSwarm.Address);
            Log.Logger.Information("Receiver: {0}", receiverSwarm.Address);

            BlockChain minerChain = minerSwarm.BlockChain;
            BlockChain receiverChain = receiverSwarm.BlockChain;

            (_, IEnumerable<Block> blocks) =
                MakeFixtureBlocksForPreloadAsyncCancellationTest();

            foreach (Block block in blocks)
            {
                minerChain.Append(block, CreateBlockCommit(block));
            }

            const int FindNextHashesChunkSize = 3;
            minerSwarm.FindNextHashesChunkSize = FindNextHashesChunkSize;
            await StartAsync(minerSwarm);

            (BoundPeer, IBlockExcerpt)[] peersWithExcerpt =
            {
                (minerSwarm.AsPeer, minerChain.Tip.Header),
            };

            (var _, List<BlockHash> demands) = await receiverSwarm.GetDemandBlockHashes(
                receiverChain,
                peersWithExcerpt,
                cancellationToken: CancellationToken.None);

            IEnumerable<BlockHash> expectedBlocks = minerChain
                .IterateBlocks()
                .Where(b => b.Index >= receiverChain.Tip.Index)
                .Take(FindNextHashesChunkSize)
                .Select(b => b.Hash);
            Assert.Equal(expectedBlocks, demands);

            CleaningSwarm(minerSwarm);
            CleaningSwarm(receiverSwarm);
        }

        [Fact(Timeout = Timeout)]
        public async Task PreloadToTheHighestTipIndexChain()
        {
            var minerKey1 = new PrivateKey();
            Swarm minerSwarm1 = await CreateSwarm(minerKey1).ConfigureAwait(false);
            Swarm minerSwarm2 = await CreateSwarm().ConfigureAwait(false);
            Swarm receiverSwarm = await CreateSwarm().ConfigureAwait(false);
            BlockChain minerChain1 = minerSwarm1.BlockChain;
            BlockChain minerChain2 = minerSwarm2.BlockChain;
            BlockChain receiverChain = receiverSwarm.BlockChain;

            Block block1 = minerChain1.ProposeBlock(
                minerKey1, CreateBlockCommit(minerChain1.Tip));
            minerChain1.Append(block1, CreateBlockCommit(block1));
            minerChain2.Append(block1, CreateBlockCommit(block1));
            Block block2 = minerChain1.ProposeBlock(
                minerKey1, CreateBlockCommit(minerChain1.Tip));
            minerChain1.Append(block2, CreateBlockCommit(block2));

            Assert.True(minerChain1.Count > minerChain2.Count);

            try
            {
                await StartAsync(minerSwarm1);
                await StartAsync(minerSwarm2);
                await receiverSwarm.AddPeersAsync(
                    new[] { minerSwarm1.AsPeer, minerSwarm2.AsPeer },
                    null);
                await receiverSwarm.PreloadAsync(TimeSpan.FromSeconds(1), 0);
            }
            finally
            {
                CleaningSwarm(minerSwarm1);
                CleaningSwarm(minerSwarm2);
                CleaningSwarm(receiverSwarm);
            }

            Assert.Equal(minerChain1.Count, receiverChain.Count);
            Assert.Equal(minerChain1.Tip, receiverChain.Tip);
        }

        [Fact(Timeout = Timeout)]
        public async Task PreloadIgnorePeerWithDifferentGenesisBlock()
        {
            var key1 = new PrivateKey();
            var key2 = new PrivateKey();
            var policy = new BlockPolicy();

            BlockChain receiverChain = MakeBlockChain(
                policy,
                new MemoryStore(),
                new TrieStateStore(new MemoryKeyValueStore()),
                new SingleActionLoader(typeof(DumbAction)),
                privateKey: key1);
            BlockChain validSeedChain = MakeBlockChain(
                policy,
                new MemoryStore(),
                new TrieStateStore(new MemoryKeyValueStore()),
                new SingleActionLoader(typeof(DumbAction)),
                privateKey: key1,
                genesisBlock: receiverChain.Genesis);
            BlockChain invalidSeedChain = MakeBlockChain(
                policy,
                new MemoryStore(),
                new TrieStateStore(new MemoryKeyValueStore()),
                new SingleActionLoader(typeof(DumbAction)),
                privateKey: key2);
            Swarm receiverSwarm =
                await CreateSwarm(receiverChain).ConfigureAwait(false);
            Swarm validSeedSwarm =
                await CreateSwarm(validSeedChain).ConfigureAwait(false);
            Swarm invalidSeedSwarm =
                await CreateSwarm(invalidSeedChain).ConfigureAwait(false);

            Assert.Equal(receiverSwarm.BlockChain.Genesis, validSeedSwarm.BlockChain.Genesis);
            Assert.NotEqual(receiverSwarm.BlockChain.Genesis, invalidSeedSwarm.BlockChain.Genesis);

            for (int i = 0; i < 10; i++)
            {
                Block block = validSeedChain.ProposeBlock(
                    key1, CreateBlockCommit(validSeedChain.Tip));
                validSeedChain.Append(block, CreateBlockCommit(block));
            }

            for (int i = 0; i < 20; i++)
            {
                Block block = invalidSeedChain.ProposeBlock(
                    key1, CreateBlockCommit(invalidSeedChain.Tip));
                invalidSeedChain.Append(block, CreateBlockCommit(block));
            }

            try
            {
                await StartAsync(receiverSwarm);
                await StartAsync(validSeedSwarm);
                await StartAsync(invalidSeedSwarm);

                await receiverSwarm.AddPeersAsync(
                    new[] { validSeedSwarm.AsPeer, invalidSeedSwarm.AsPeer }, null);
                await receiverSwarm.PreloadAsync();

                Assert.Equal(receiverChain.Tip, validSeedChain.Tip);
            }
            finally
            {
                CleaningSwarm(receiverSwarm);
                CleaningSwarm(validSeedSwarm);
                CleaningSwarm(invalidSeedSwarm);
            }
        }

        [Fact(Timeout = Timeout)]
        public async Task UpdateTxExecution()
        {
            PrivateKey seedKey = new PrivateKey();
            var policy = new BlockPolicy(
                new PolicyActionsRegistry(
                    endBlockActions: ImmutableArray.Create<IAction>(new MinerReward(1))));
            var fx1 = new MemoryStoreFixture(policy.PolicyActionsRegistry);
            var fx2 = new MemoryStoreFixture(policy.PolicyActionsRegistry);
            var seedChain = MakeBlockChain(
                policy, fx1.Store, fx1.StateStore, new SingleActionLoader(typeof(DumbAction)));
            var receiverChain = MakeBlockChain(
                policy, fx2.Store, fx2.StateStore, new SingleActionLoader(typeof(DumbAction)));

            Swarm seed =
                await CreateSwarm(seedChain).ConfigureAwait(false);
            Swarm receiver =
                await CreateSwarm(receiverChain).ConfigureAwait(false);

            List<Transaction> transactions = new List<Transaction>();
            for (int i = 0; i < 10; i++)
            {
                var transaction = seedChain.MakeTransaction(
                    new PrivateKey(),
                    new[]
                    {
                        DumbAction.Create((default, $"Item{i}")),
                    });
                Block block = seedChain.ProposeBlock(
                    seedKey, CreateBlockCommit(seedChain.Tip));
                seedChain.Append(block, TestUtils.CreateBlockCommit(block));
                transactions.Add(transaction);
            }

            Assert.Equal(10, seedChain.Tip.Index);

            try
            {
                await StartAsync(seed);
                await StartAsync(receiver);

                await receiver.AddPeersAsync(
                    new[] { seed.AsPeer }, null);
                await receiver.PreloadAsync();

                Assert.Equal(receiverChain.Tip, seedChain.Tip);

                for (int i = 1; i < receiverChain.Count; i++)
                {
                    Assert.NotNull(fx2.Store.GetTxExecution(
                        receiverChain[i].Hash,
                        transactions[i - 1].Id));
                }
            }
            finally
            {
                CleaningSwarm(seed);
                CleaningSwarm(receiver);
            }
        }
    }
}
