/// \file waFileSystem.cpp
/// ļʵļ

#include <cstdio>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/file.h>
#include "waFileSystem.h"

using namespace std;

/// Web Application Library namaspace
namespace webapp {

/// \defgroup waFileSystem waFileSystemļ
	
/// \ingroup waFileSystem
/// \fn bool file_exist( const string &file )
/// ļĿ¼Ƿ
/// \param file ļ·
/// \retval true ļ
/// \retval false 
bool file_exist( const string &file ) {
	if ( access(file.c_str(),F_OK) == 0 )
		return true;
	else
		return false;
}

/// \ingroup waFileSystem
/// \fn bool is_link( const string &file )
/// ļǷΪ
/// \param file ļ·
/// \retval true ļΪ
/// \retval false ڻ߲Ƿ
bool is_link( const string &file ) {
	struct stat statbuf;
	
	if( lstat(file.c_str(),&statbuf) == 0 ) {
		if ( S_ISLNK(statbuf.st_mode) != 0 )
			return true;
	}
	return false;
}

/// \ingroup waFileSystem
/// \fn bool is_dir( const string &file )
/// ǷΪĿ¼
/// \param dile Ŀ¼·
/// \retval true Ŀ¼
/// \retval false ڻ߲Ŀ¼
bool is_dir( const string &file ) {
	struct stat statbuf;
	
	if( stat(file.c_str(),&statbuf) == 0 ) {
		if ( S_ISDIR(statbuf.st_mode) != 0 )
			return true;
	}
	return false;
}

/// \ingroup waFileSystem
/// \fn bool make_link( const string &srcfile, const string &destfile )
/// ,ļ벻
/// \param srcfile ԭļ
/// \param destfile 
/// \retval true ɹ
/// \retval false ɹ
bool make_link( const string &srcfile, const string &destfile ) {
	if ( symlink(srcfile.c_str(),destfile.c_str()) == 0 )
		return true;
	else
		return false;
}

/// \ingroup waFileSystem
/// \fn size_t file_size( const string &file )
/// ȡļС
/// \param file ļ·
/// \return ļ򷵻شС,򷵻-1
size_t file_size( const string &file ) {
	struct stat statbuf;
	
	if( stat(file.c_str(),&statbuf)==0 )
		return statbuf.st_size;
	else		
		return -1;
}

/// \ingroup waFileSystem
/// \fn time_t file_time( const string &file )
/// ȡļʱ
/// \param file ļ·
/// \return ļ򷵻ʱ,򷵻-1
time_t file_time( const string &file ) {
	struct stat statbuf;
	
	if( stat(file.c_str(),&statbuf)==0 )
		return statbuf.st_mtime;
	else		
		return -1;
}

/// \ingroup waFileSystem
/// \fn string file_path( const string &file )
/// ȡļ·
/// \param file ļ·
/// \return ȡļ·򷵻,򷵻ؿַ
string file_path( const string &file ) {
	size_t p;
	if ( (p=file.rfind("/")) != file.npos )
		return file.substr( 0, p );
	else if ( (p=file.rfind("\\")) != file.npos )
		return file.substr( 0, p );
	return string( "" );
}

/// \ingroup waFileSystem
/// \fn string file_name( const string &file )
/// ȡļ
/// \param file ļ·
/// \return ȡļ򷵻,򷵻ԭļ·
string file_name( const string &file ) {
	size_t p;
	if ( (p=file.rfind("/")) != file.npos )
		return file.substr( p+1 );
	else if ( (p=file.rfind("\\")) != file.npos )
		return file.substr( p+1 );
	return file;
}

/// \ingroup waFileSystem
/// \fn bool rename_file( const string &oldname, const string &newname )
/// ļĿ¼,ļԭļλͬһļϵͳ
/// \param oldname ԭļ
/// \param newname ļ
/// \retval true ɹ
/// \retval false ʧ
bool rename_file( const string &oldname, const string &newname ) {
	if ( rename(oldname.c_str(),newname.c_str()) != -1 )
		return true;
	else
		return false;
}

/// \ingroup waFileSystem
/// \fn bool copy_file( const string &srcfile, const string &destfile )
/// ļ	
/// \param srcfile ԭļ
/// \param destfile Ŀļ,ļΪ0666
/// \retval true ɹ
/// \retval false ʧ
bool copy_file( const string &srcfile, const string &destfile ) {
	FILE *src=NULL, *dest=NULL;
	if ( (src=fopen(srcfile.c_str(),"rb")) == NULL ) {
		return false;
	}
	if ( (dest=fopen(destfile.c_str(),"wb+")) == NULL ) {
		fclose( src );
		return false;
	}
	
	const int bufsize = 8192;
	char buf[bufsize];
	size_t n;
	while ( (n=fread(buf,1,bufsize,src)) >= 1 ) {
		if ( fwrite(buf,1,n,dest) != n ) {
			fclose( src );
			fclose( dest );
			return false;
		}
	}
	
	fclose( src );
	fclose( dest );

	//chmod to 0666
	mode_t mask = umask( 0 );
	chmod( destfile.c_str(), S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH );
	umask( mask );
	
	return true;
}

/// \ingroup waFileSystem
/// \fn bool delete_file( const string &file )
/// ɾļ
/// \param file ļ·
/// \retval true ɾɹ
/// \retval false ļڻɾʧ
bool delete_file( const string &file ) {
	if ( remove(file.c_str()) == 0 )
		return true;
	else
		return false;
}

/// \ingroup waFileSystem
/// \fn bool move_file( const string &srcfile, const string &destfile )
/// ƶļ
/// \param srcfile ԭļ
/// \param destfile ļ
/// \retval true ɹ
/// \retval false ʧ
bool move_file( const string &srcfile, const string &destfile ) {
	if ( rename_file(srcfile,destfile) )
		return true;

	// rename fail, copy and delete file		
	if ( copy_file(srcfile,destfile) ) {
		if ( delete_file(srcfile) )
			return true;
	}
	
	return false;
}

/// \ingroup waFileSystem
/// \fn vector<string> dir_files( const string &dir )
/// Ŀ¼ļб
/// \param dir ΪĿ¼·
/// \return ؽΪļĿ¼б,Ŀ¼ĵһַΪ '/',
/// ؽвǰһĿ¼ "/.", "/.."
vector<string> dir_files( const string &dir ) {
	vector<string> files;
	string file;
	DIR *pdir = NULL;
	dirent *pdirent = NULL;
	
	if ( (pdir=opendir(dir.c_str())) != NULL ) {
		while ( (pdirent=readdir(pdir)) != NULL ) {
			file = pdirent->d_name;
			if ( file!="." && file!=".." ) {
				if ( is_dir(dir+"/"+file) ) 
					file = "/"+file;
				files.push_back( file );
			}
		}
		closedir( pdir );
	}
	
	return files;
}

/// \ingroup waFileSystem
/// \fn bool make_dir( const string &dir, const size_t mode )
/// Ŀ¼
/// \param dir ҪĿ¼,ϲĿ¼Զ
/// \param mode Ŀ¼Ȩ,ĬΪS_IRWXU|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH(0755)
/// ɹtrue, 򷵻false.
/// \retval true ɹ
/// \retval false ʧ
bool make_dir( const string &dir, const mode_t mode ) {
	// check
	size_t len = dir.length();
	if ( len <= 0 )	return false;

	string tomake;
	char curchr;
	for( size_t i=0; i<len; ++i ) {
		// append
		curchr = dir[i];
		tomake += curchr;
		if ( curchr=='/' || i==(len-1) ) {
			// need to mkdir
			if ( !file_exist(tomake) && !is_dir(tomake) ) {
				// able to mkdir
				mode_t mask = umask( 0 );				
				if ( mkdir(tomake.c_str(),mode) == -1 ) {
					umask( mask );
					return false;
				}
				umask( mask );
			}
		}
	}
	
	return true;
}	

/// \ingroup waFileSystem
/// \fn bool copy_dir( const string &srcdir, const string &destdir )
/// Ŀ¼,Ŀ¼ʱΪݹ
/// \param srcdir ԭĿ¼
/// \param destdir ĿĿ¼
/// \retval true ɹ
/// \retval false ʧ
bool copy_dir( const string &srcdir, const string &destdir ) {	
	vector<string> files = dir_files( srcdir );
	string from;
	string to;
	
	// ĿĿ¼
	if ( !file_exist(destdir) )
		make_dir( destdir );
	
	for ( size_t i=0; i<files.size(); ++i ) {
		from = srcdir + "/" + files[i];
		to = destdir + "/" + files[i];
		
		// Ŀ¼,ݹ
		if ( files[i][0] == '/' ) {
			 if ( !copy_dir(from,to) )
				return false;
		}
		
		// ļ
		else if ( !copy_file(from,to) )
			return false;
	}
	
	return true;
}

/// \ingroup waFileSystem
/// \fn bool delete_dir( const string &dir )
/// ɾĿ¼,ɾĿ¼ʱΪݹ
/// \param dir ҪɾĿ¼
/// \retval true ɹ
/// \retval false ʧ
bool delete_dir( const string &dir ) {	
	vector<string> files = dir_files( dir );
	string todel;
	
	// ɾļ
	for ( size_t i=0; i<files.size(); ++i ) {
		todel = dir + "/" + files[i];
		
		// Ŀ¼,ݹ
		if ( files[i][0] == '/' ) {
			 if ( !delete_dir(todel) )
				return false;
		}
		
		// ļ
		else if ( !delete_file(todel) )
			return false;
	}
	
	// ɾĿ¼
	if ( rmdir(dir.c_str()) == 0 )
		return true;

	return false;
}
			
/// \ingroup waFileSystem
/// \fn bool move_dir( const string &srcdir, const string &destdir )
/// ƶĿ¼
/// \param srcdir ԭĿ¼
/// \param destdir ĿĿ¼
/// \retval true ɹ
/// \retval false ʧ
bool move_dir( const string &srcdir, const string &destdir ) {
	if ( rename_file(srcdir,destdir) )
		return true;

	// rename fail, copy and delete dir	
	if ( copy_dir(srcdir,destdir) ) {
		if ( delete_dir(srcdir) )
			return true;
	}

	return false;
}

/// \ingroup waFileSystem
/// \fn void lock_file( int fd, const int type )
/// ļļѱȴ
/// \param fd ļ
/// \param type ģʽѡF_WRLCKF_RDLCKF_UNLCK
void lock_file( int fd, const int type ) {
	struct flock lck;
	lck.l_start = 0;
	lck.l_len = 0;
	lck.l_whence = SEEK_SET;
	
	lck.l_type = type;
	if ( type==F_WRLCK || type==F_RDLCK )
		lck.l_type = type;
	else
		lck.l_type = F_UNLCK;
	
	while ( fcntl(fd,F_SETLKW,&lck)==-1 && errno==EINTR );
	return;
}

/// \ingroup waFileSystem
/// \fn bool is_locked( int fd )
/// жļ
/// \param fd ļ
/// \retval true ļѱ
/// \retval false ļδ
bool is_locked( int fd ) {
	struct flock lck;
	lck.l_start = 0;
	lck.l_len = 0;
	lck.l_whence = SEEK_SET;
	lck.l_type = F_WRLCK;

	while ( (fcntl(fd,F_GETLK,&lck))==-1 && errno==EINTR );

	if ( lck.l_type == F_UNLCK )
		return false;
	else
		return true;
}

/// \ingroup waFileSystem
/// \fn FILE* lock_open( const string &file, const char* mode, const int type )
/// ļļѱȴ
/// \param file ļ·
/// \param mode ļģʽfopen()ͬͬ
/// \param type ģʽѡF_WRLCKF_RDLCKF_UNLCK
/// \return ļʧܷNULL
FILE* lock_open( const string &file, const char* mode, const int type ) {
	FILE *fp;
	mode_t mask = umask( 0 );
	int fd = open( file.c_str(), O_CREAT|O_EXCL|O_RDWR, 
				   S_IRUSR|S_IWUSR|S_IRGRP|S_IWGRP|S_IROTH|S_IWOTH );
	if ( fd >= 0 ) {
		// file not exist, create success
		lock_file( fd, type );
		fp = fdopen( fd, mode );
	} else {
		// file exist
		fp = fopen( file.c_str(), mode );
		if ( fp == NULL )
			return NULL;
		lock_file( fileno(fp), type );
	}
	umask( mask );

	return fp;
}

} // namespace

