/* driver.c - Wrapper for standalone EFI filesystem drivers */
/*
 *  Copyright © 2014-2020 Pete Batard <pete@akeo.ie>
 *  Based on iPXE's efi_driver.c and efi_file.c:
 *  Copyright © 2011,2013 Michael Brown <mbrown@fensystems.co.uk>.
 *
 *  Note: This file has been relicensed from GPLv3+ to GPLv2+ by formal
 *  agreement of all of the contributors who applied changes on top of
 *  its original GPLv2+ source. The original source can be found at:
 *  https://github.com/ipxe/ipxe/blob/26029658063dcafcca746640a28a76d4f7f4a66e/src/interface/efi/efi_driver.c
 *
 *  This program is free software: you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation, either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "driver.h"

/* We'll try to instantiate a custom protocol as a mutex, so we need a GUID */
EFI_GUID *MutexGUID = NULL;

/* Keep a global copy of our ImageHanle */
EFI_HANDLE EfiImageHandle = NULL;

/* Handle for our custom protocol/mutex instance */
static EFI_HANDLE MutexHandle = NULL;

/* Custom protocol/mutex definition */
typedef struct {
	INTN Unused;
} EFI_MUTEX_PROTOCOL;
static EFI_MUTEX_PROTOCOL MutexProtocol = { 0 };


/* Return the driver name */
static EFI_STATUS EFIAPI
FSGetDriverName(EFI_COMPONENT_NAME_PROTOCOL *This,
		CHAR8 *Language, CHAR16 **DriverName)
{
	*DriverName = FullDriverName;
	return EFI_SUCCESS;
}

static EFI_STATUS EFIAPI
FSGetDriverName2(EFI_COMPONENT_NAME2_PROTOCOL *This,
		CHAR8 *Language, CHAR16 **DriverName)
{
	*DriverName = FullDriverName;
	return EFI_SUCCESS;
}

/* Return the controller name (unsupported for a filesystem) */
static EFI_STATUS EFIAPI
FSGetControllerName(EFI_COMPONENT_NAME_PROTOCOL *This,
		EFI_HANDLE ControllerHandle, EFI_HANDLE ChildHandle,
		CHAR8 *Language, CHAR16 **ControllerName)
{
	return EFI_UNSUPPORTED;
}

static EFI_STATUS EFIAPI
FSGetControllerName2(EFI_COMPONENT_NAME2_PROTOCOL *This,
		EFI_HANDLE ControllerHandle, EFI_HANDLE ChildHandle,
		CHAR8 *Language, CHAR16 **ControllerName)
{
	return EFI_UNSUPPORTED;
}

static VOID
FreeFsInstance(EFI_FS *Instance) {
	if (Instance == NULL)
		return;
	if (Instance->RootFile != NULL)
		FreePool(Instance->RootFile);
	FreePool(Instance);
}

/*
 * To check if our driver has a chance to apply to the controllers sent during
 * the supported detection phase, try to open the child protocols it is meant
 * to consume (here EFI_DISK_IO) in exclusive access.
 */
static EFI_STATUS EFIAPI
FSBindingSupported(EFI_DRIVER_BINDING_PROTOCOL *This,
		EFI_HANDLE ControllerHandle,
		EFI_DEVICE_PATH_PROTOCOL *RemainingDevicePath)
{
	EFI_STATUS Status;
	EFI_DISK_IO_PROTOCOL  *DiskIo;
	EFI_DISK_IO2_PROTOCOL *DiskIo2;

	/* Don't handle this unless we can get exclusive access to DiskIO through it */
	Status = BS->OpenProtocol(ControllerHandle,
			&gEfiDiskIo2ProtocolGuid, (VOID **) &DiskIo2,
			This->DriverBindingHandle, ControllerHandle,
			EFI_OPEN_PROTOCOL_BY_DRIVER);
	if (EFI_ERROR(Status))
		DiskIo2 = NULL;
	Status = BS->OpenProtocol(ControllerHandle,
			&gEfiDiskIoProtocolGuid, (VOID **) &DiskIo,
			This->DriverBindingHandle, ControllerHandle,
			EFI_OPEN_PROTOCOL_BY_DRIVER);
	if (EFI_ERROR(Status))
		return Status;

	PrintDebug(L"FSBindingSupported\n");

	/*
	 * The whole concept of BindingSupported is to hint at what we may
	 * actually support, but not check if the target is valid or
	 * initialize anything, so we must close all protocols we opened.
	 */
	BS->CloseProtocol(ControllerHandle, &gEfiDiskIo2ProtocolGuid,
		This->DriverBindingHandle, ControllerHandle);
	BS->CloseProtocol(ControllerHandle, &gEfiDiskIoProtocolGuid,
			This->DriverBindingHandle, ControllerHandle);

	return EFI_SUCCESS;
}

static EFI_STATUS EFIAPI
FSBindingStart(EFI_DRIVER_BINDING_PROTOCOL *This,
		EFI_HANDLE ControllerHandle,
		EFI_DEVICE_PATH *RemainingDevicePath)
{
	EFI_STATUS Status;
	EFI_FS *Instance;
	PrintDebug(L"FSBindingStart\n");

	/* Allocate a new instance of a filesystem */
	Instance = AllocateZeroPool(sizeof(EFI_FS));
	if (Instance == NULL) {
		Status = EFI_OUT_OF_RESOURCES;
		PrintStatusError(Status, L"Could not allocate a new file system instance");
		return Status;
	}
	/* Allocate the root file structure */
	Instance->RootFile = AllocateZeroPool(sizeof(EFI_GRUB_FILE));
	if (Instance->RootFile == NULL) {
		Status = EFI_OUT_OF_RESOURCES;
		PrintStatusError(Status, L"Could not allocate root file");
		goto error;
	}
	Instance->FileIoInterface.Revision = EFI_SIMPLE_FILE_SYSTEM_PROTOCOL_REVISION;
	Instance->FileIoInterface.OpenVolume = FileOpenVolume;

	/* Fill the device path for our instance */
	Instance->DevicePath = DevicePathFromHandle(ControllerHandle);
	if (Instance->DevicePath == NULL) {
		Status = EFI_NO_MAPPING;
		PrintStatusError(Status, L"Could not get Device Path");
		goto error;
	}

	/* Get access to the Block IO protocol for this controller */
	Status = BS->OpenProtocol(ControllerHandle,
			&gEfiBlockIo2ProtocolGuid, (VOID **) &Instance->BlockIo2,
			This->DriverBindingHandle, ControllerHandle,
			EFI_OPEN_PROTOCOL_GET_PROTOCOL);
	if (EFI_ERROR(Status))
		Instance->BlockIo2 = NULL;

	Status = BS->OpenProtocol(ControllerHandle,
			&gEfiBlockIoProtocolGuid, (VOID **) &Instance->BlockIo,
			This->DriverBindingHandle, ControllerHandle,
			/*
			 * EFI_OPEN_PROTOCOL_BY_DRIVER would return Access Denied here,
			 * because the disk driver has that protocol already open. So use
			 * EFI_OPEN_PROTOCOL_GET_PROTOCOL (which doesn't require us to close it).
			 */
			EFI_OPEN_PROTOCOL_GET_PROTOCOL);
	if (EFI_ERROR(Status)) {
		PrintStatusError(Status, L"Could not access BlockIO protocol");
		goto error;
	}

	/* Get exclusive access to the Disk IO protocol */
	Status = BS->OpenProtocol(ControllerHandle,
			&gEfiDiskIo2ProtocolGuid, (VOID**) &Instance->DiskIo2,
			This->DriverBindingHandle, ControllerHandle,
			EFI_OPEN_PROTOCOL_BY_DRIVER);
	if (EFI_ERROR(Status))
		Instance->DiskIo2 = NULL;

	Status = BS->OpenProtocol(ControllerHandle,
			&gEfiDiskIoProtocolGuid, (VOID**) &Instance->DiskIo,
			This->DriverBindingHandle, ControllerHandle,
			EFI_OPEN_PROTOCOL_BY_DRIVER);
	if (EFI_ERROR(Status)) {
		PrintStatusError(Status, L"Could not access the DiskIo protocol");
		goto error;
	}

	/* Go through GRUB target init */
	Status = GrubDeviceInit(Instance);
	if (EFI_ERROR(Status)) {
		PrintStatusError(Status, L"Could not init grub device");
		goto error;
	}

	Status = FSInstall(Instance, ControllerHandle);
	if (EFI_ERROR(Status))
		GrubDeviceExit(Instance);

error:
	if (EFI_ERROR(Status)) {
		/*
		 * Unless we close the DiskIO protocols, which we do on error,
		 * no other FS driver would be able to access this partition.
		 */
		if (Instance->DiskIo2 != NULL)
			BS->CloseProtocol(ControllerHandle, &gEfiDiskIo2ProtocolGuid,
				This->DriverBindingHandle, ControllerHandle);
		if (Instance->DiskIo != NULL)
			BS->CloseProtocol(ControllerHandle, &gEfiDiskIoProtocolGuid,
				This->DriverBindingHandle, ControllerHandle);
		FreeFsInstance(Instance);
	}
	return Status;
}

static EFI_STATUS EFIAPI
FSBindingStop(EFI_DRIVER_BINDING_PROTOCOL *This,
		EFI_HANDLE ControllerHandle, UINTN NumberOfChildren,
		EFI_HANDLE *ChildHandleBuffer)
{
	EFI_STATUS Status;
	EFI_FS *Instance;
	EFI_SIMPLE_FILE_SYSTEM_PROTOCOL *FileIoInterface;

	PrintDebug(L"FSBindingStop\n");

	/* Get a pointer back to our FS instance through its installed protocol */
	Status = BS->OpenProtocol(ControllerHandle,
			&gEfiSimpleFileSystemProtocolGuid, (VOID **) &FileIoInterface,
			This->DriverBindingHandle, ControllerHandle,
			EFI_OPEN_PROTOCOL_GET_PROTOCOL);
	if (EFI_ERROR(Status)) {
		PrintStatusError(Status, L"Could not locate our instance");
		return Status;
	}

	Instance = _CR(FileIoInterface, EFI_FS, FileIoInterface);
	FSUninstall(Instance, ControllerHandle);

	Status = GrubDeviceExit(Instance);
	if (EFI_ERROR(Status)) {
		PrintStatusError(Status, L"Could not destroy grub device");
	}

	BS->CloseProtocol(ControllerHandle, &gEfiDiskIo2ProtocolGuid,
			This->DriverBindingHandle, ControllerHandle);
	BS->CloseProtocol(ControllerHandle, &gEfiDiskIoProtocolGuid,
			This->DriverBindingHandle, ControllerHandle);

	FreeFsInstance(Instance);

	return EFI_SUCCESS;
}

/*
 * The platform determines whether it will support the older Component
 * Name Protocol or the current Component Name2 Protocol, or both.
 * Because of this, it is strongly recommended that you implement both
 * protocols in your driver.
 *
 * NB: From what I could see, the only difference between Name and Name2
 * is that Name uses ISO-639-2 ("eng") whereas Name2 uses RFC 4646 ("en")
 * See: http://www.loc.gov/standards/iso639-2/faq.html#6
 */
static EFI_COMPONENT_NAME_PROTOCOL FSComponentName = {
	.GetDriverName = FSGetDriverName,
	.GetControllerName = FSGetControllerName,
	.SupportedLanguages = (CHAR8 *) "eng"
};

static EFI_COMPONENT_NAME2_PROTOCOL FSComponentName2 = {
	.GetDriverName = FSGetDriverName2,
	.GetControllerName = FSGetControllerName2,
	.SupportedLanguages = (CHAR8 *) "en"
};

static EFI_DRIVER_BINDING_PROTOCOL FSDriverBinding = {
	.Supported = FSBindingSupported,
	.Start = FSBindingStart,
	.Stop = FSBindingStop,
	/* This field is used by the EFI boot service ConnectController() to determine the order
	 * that driver's Supported() service will be used when a controller needs to be started.
	 * EFI Driver Binding Protocol instances with higher Version values will be used before
	 * ones with lower Version values. The Version values of 0x0-0x0f and
	 * 0xfffffff0-0xffffffff are reserved for platform/OEM specific drivers. The Version
	 * values of 0x10-0xffffffef are reserved for IHV-developed drivers.
	 */
	.Version = 0x10,
	.ImageHandle = NULL,
	.DriverBindingHandle = NULL
};

/**
 * Set a filesystem GUID according to the filesystem name
 * We use a static ID for the first 8 bytes, and then roll the lowercase name
 * for the last 8 bytes (eg. exfat => {'e', 'x', 'f', 'a', 't', 'e', 'x', 'f'})
 */
EFI_GUID *
GetFSGuid(VOID)
{
	INTN i, j, k, Len = StrLen(ShortDriverName);
	static EFI_GUID Guid = { 0xEF1F5EF1, 0xF17E, 0x5857, { 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00 } };
	CHAR16 c;
	const CHAR16 *PlusName = L"plus";
	UINT8 Data4[12];	/* +4 so that we can also reduce something like "1234567plus" into "1234567+" */

	for (i = 0, j = 0, k = 0; j < ARRAYSIZE(Data4); i = (i+1)%Len, j++) {
		c = ShortDriverName[i];
		if ((c >= L'A') && (c <= L'Z'))
			c += L'a' - L'A';
		/* Convert any 'plus' that is part of the name to '+' */
		if (c == PlusName[k]) {
			if (++k == 4) {
				k = 0;
				j -= 3;
				Data4[j] = (UINT8) '+';
			} else {
				Data4[j] = (UINT8) c;
			}
		} else {
			k = 0;
			Data4[j] = (UINT8) c;
		}
	}
	CopyMem(Guid.Data4, Data4, 8);

	return &Guid;
}

/**
 * Uninstall EFI driver
 *
 * @v ImageHandle       Handle identifying the loaded image
 * @ret Status          EFI status code to return on exit
 */
EFI_STATUS EFIAPI
FSDriverUninstall(EFI_HANDLE ImageHandle)
{
	EFI_STATUS Status;
	UINTN NumHandles;
	EFI_HANDLE *Handles = NULL;
	UINTN i;

	/* Enumerate all handles */
	Status = BS->LocateHandleBuffer(AllHandles, NULL, NULL, &NumHandles, &Handles);

	/* Disconnect controllers linked to our driver. This action will trigger a call to BindingStop */
	if (Status == EFI_SUCCESS) {
		for (i=0; i<NumHandles; i++) {
			/* Make sure to filter on DriverBindingHandle,  else EVERYTHING gets disconnected! */
			Status = BS->DisconnectController(Handles[i], FSDriverBinding.DriverBindingHandle, NULL);
			if (Status == EFI_SUCCESS)
				PrintDebug(L"DisconnectController[%d]\n", i);
		}
	} else {
		PrintStatusError(Status, L"Unable to enumerate handles");
	}
	if (Handles != NULL)
		BS->FreePool(Handles);

	/* Now that all controllers are disconnected, we can safely remove our protocols */
	BS->UninstallMultipleProtocolInterfaces(ImageHandle,
			&gEfiDriverBindingProtocolGuid, &FSDriverBinding,
			&gEfiComponentNameProtocolGuid, &FSComponentName,
			&gEfiComponentName2ProtocolGuid, &FSComponentName2,
			NULL);

	/* Release the relevant GRUB module(s) */
	for (i = 0; GrubModuleExit[i] != NULL; i++)
		GrubModuleExit[i]();

	/* Uninstall our mutex (we're the only instance that can run this code) */
	BS->UninstallMultipleProtocolInterfaces(MutexHandle,
				MutexGUID, &MutexProtocol,
				NULL);

	PrintDebug(L"FS driver uninstalled.\n");
	return EFI_SUCCESS;
}

/**
 * Install EFI driver - Will be the entrypoint for our driver executable
 * http://wiki.phoenix.com/wiki/index.php/EFI_IMAGE_ENTRY_POINT
 *
 * @v ImageHandle       Handle identifying the loaded image
 * @v SystemTable       Pointers to EFI system calls
 * @ret Status          EFI status code to return on exit
 */
EFI_STATUS EFIAPI
FSDriverInstall(EFI_HANDLE ImageHandle, EFI_SYSTEM_TABLE* SystemTable)
{
	EFI_STATUS Status;
	EFI_LOADED_IMAGE_PROTOCOL *LoadedImage = NULL;
	VOID *Interface;
	UINTN i;

#if defined(_GNU_EFI)
	InitializeLib(ImageHandle, SystemTable);
#endif
	SetLogging();
	EfiImageHandle = ImageHandle;

	/* Prevent the driver from being loaded twice by detecting and trying to
	 * instantiate a custom protocol, which we use as a global mutex.
	 */
	MutexGUID = GetFSGuid();

	Status = BS->LocateProtocol(MutexGUID, NULL, &Interface);
	if (Status == EFI_SUCCESS) {
		PrintError(L"This driver has already been installed\n");
		return EFI_LOAD_ERROR;
	}
	/* The only valid status we expect is NOT FOUND here */
	if (Status != EFI_NOT_FOUND) {
		PrintStatusError(Status, L"Could not locate global mutex");
		return Status;
	}
	Status = BS->InstallMultipleProtocolInterfaces(&MutexHandle,
			MutexGUID, &MutexProtocol,
			NULL);
	if (EFI_ERROR(Status)) {
		PrintStatusError(Status, L"Could not install global mutex");
		return Status;
	}

	/* Grab a handle to this image, so that we can add an unload to our driver */
	Status = BS->OpenProtocol(ImageHandle, &gEfiLoadedImageProtocolGuid,
			(VOID **) &LoadedImage, ImageHandle,
			NULL, EFI_OPEN_PROTOCOL_GET_PROTOCOL);
	if (EFI_ERROR(Status)) {
		PrintStatusError(Status, L"Could not open loaded image protocol");
		return Status;
	}

	/* Configure driver binding protocol */
	FSDriverBinding.ImageHandle = ImageHandle;
	FSDriverBinding.DriverBindingHandle = ImageHandle;

	/* Install driver */
	Status = BS->InstallMultipleProtocolInterfaces(&FSDriverBinding.DriverBindingHandle,
			&gEfiDriverBindingProtocolGuid, &FSDriverBinding,
			&gEfiComponentNameProtocolGuid, &FSComponentName,
			&gEfiComponentName2ProtocolGuid, &FSComponentName2,
			NULL);
	if (EFI_ERROR(Status)) {
		PrintStatusError(Status, L"Could not bind driver");
		return Status;
	}

	/* Register the uninstall callback */
	LoadedImage->Unload = FSDriverUninstall;

	/* Initialize the relevant GRUB fs module(s) */
	for (i = 0; GrubModuleInit[i] != NULL; i++)
		GrubModuleInit[i]();

	InitializeListHead(&FsListHead);

	PrintDebug(L"FS driver installed.\n");
	return EFI_SUCCESS;
}
