# ES6 Shim <sup>[![Version Badge][npm-version-svg]][package-url]</sup>
Provides compatibility shims so that legacy JavaScript engines behave as
closely as possible to ECMAScript 6 (Harmony).

[![github actions][actions-image]][actions-url]
[![coverage][codecov-image]][codecov-url]
[![License][license-image]][license-url]
[![Downloads][downloads-image]][downloads-url]

[![npm badge][npm-badge-png]][package-url]

<!-- [![browser support](https://ci.testling.com/paulmillr/es6-shim.png)](https://ci.testling.com/paulmillr/es6-shim) -->

<!-- [![Sauce Test Status](https://saucelabs.com/browser-matrix/es6-shim.svg)](https://saucelabs.com/u/es6-shim) -->

[HTML version of the final ECMAScript 6 spec][spec-html-url]

## Installation
If you want to use it in browser:

* Just include `es6-shim` before your scripts.
* Include [es5-shim][es5-shim-url] especially if your browser doesn't support ECMAScript 5 - but every JS engine requires the `es5-shim` to correct broken implementations, so it's strongly recommended to always include it. Additionally, `es5-shim` should be loaded before `es6-shim`.

For `node.js`, `io.js`, or any `npm`-managed workflow (this is the recommended method):

    npm install es6-shim

Alternative methods:
* `component install paulmillr/es6-shim` if you’re using [component(1)](https://github.com/componentjs/component).
* `bower install es6-shim` if you’re using [Bower](http://bower.io/).

In both browser and node you may also want to include `unorm`; see the [`String.prototype.normalize`](#stringprototypenormalize) section for details.

## Safe shims

* `Map` (requires ES5 property descriptor support) ([a standalone shim is also available](https://npmjs.com/es-map))
* `Set` (requires ES5 property descriptor support) ([a standalone shim is also available](https://npmjs.com/es-set))
* `Promise`
* `String`:
    * `fromCodePoint()` ([a standalone shim is also available](https://npmjs.com/string.fromcodepoint))
    * `raw()` ([a stanadlone shim is also available](https://www.npmjs.com/package/string.raw))
* `String.prototype`:
    * `codePointAt()` ([a standalone shim is also available](https://npmjs.com/string.prototype.codepointat))
    * `endsWith()` ([a standalone shim is also available](https://npmjs.com/string.prototype.endswith))
    * `includes()` ([a standalone shim is also available](https://npmjs.com/string.prototype.includes))
    * `repeat()` ([a standalone shim is also available](https://npmjs.com/string.prototype.repeat))
    * `startsWith()` ([a standalone shim is also available](https://npmjs.com/string.prototype.startswith))
* `RegExp`:
    * `new RegExp`, when given a RegExp as the pattern, will no longer throw when given a "flags" string argument. (requires ES5)
* `RegExp.prototype`:
    * `flags` (requires ES5) ([a standalone shim is also available](https://npmjs.com/regexp.prototype.flags))
    * `[Symbol.match]` (requires native `Symbol`s)
    * `[Symbol.replace]` (requires native `Symbol`s)
    * `[Symbol.search]` (requires native `Symbol`s)
    * `[Symbol.split]` (requires native `Symbol`s)
    * `toString`
* `Number`:
    * binary and octal literals: `Number('0b1')` and `Number('0o7')`
    * `EPSILON` ([a standalone shim is also available](https://www.npmjs.com/package/es-constants))
    * `MAX_SAFE_INTEGER` ([a standalone shim is also available](https://www.npmjs.com/package/es-constants))
    * `MIN_SAFE_INTEGER` ([a standalone shim is also available](https://www.npmjs.com/package/es-constants))
    * `isNaN()` ([a standalone shim is also available](https://www.npmjs.com/package/is-nan))
    * `isInteger()` ([a standalone shim is also available](https://www.npmjs.com/package/number.isinteger))
    * `isSafeInteger()`([a standalone shim is also available](https://www.npmjs.com/package/number.issafeinteger))
    * `isFinite()` ([a standalone shim is also available](https://www.npmjs.com/package/number.isfinite))
    * `parseInt()` ([a standalone shim is also available](https://www.npmjs.com/package/parseint))
    * `parseFloat()`
* `Array`:
    * `from()` ([a standalone shim is also available](https://www.npmjs.com/package/array.from))
    * `of()` ([a standalone shim is also available](https://www.npmjs.com/package/array.of))
* `Array.prototype`:
    * `copyWithin()` ([a standalone shim is also available](https://npmjs.com/array.prototype.copywithin))
    * `entries()` ([a standalone shim is also available](https://npmjs.com/array.prototype.entries))
    * `fill()`
    * `find()` ([a standalone shim is also available](https://npmjs.com/array.prototype.find))
    * `findIndex()` ([a standalone shim is also available](https://npmjs.com/array.prototype.findindex))
    * `keys()` ([a standalone shim is also available](https://npmjs.com/array.prototype.keys))
    * `values()` ([a standalone shim is also available](https://npmjs.com/array.prototype.values))
    * `indexOf()` (ES6 errata) ([a standalone shim is also available](https://npmjs.com/array.prototype.indexof))
* `Object`:
    * `assign()` ([a standalone shim is also available](https://npmjs.com/object.assign))
    * `is()` ([a standalone shim is also available](https://npmjs.com/object-is))
    * `keys()` (in ES5, but no longer throws on non-object non-null/undefined values in ES6) ([a standalone shim is also available](https://npmjs.com/object-keys)
    * `setPrototypeOf()` (IE >= 11)
* `Function.prototype`:
    * `name` (es6-sham, covers IE 9-11) ([a standalone shim is also available](https://npmjs.com/function.prototype.name)
* `Math`:
    * `acosh()` ([a standalone shim is also available](https://npmjs.com/math.acosh))
    * `asinh()`
    * `atanh()` ([a standalone shim is also available](https://npmjs.com/math.atanh))
    * `cbrt()` ([a standalone shim is also available](https://npmjs.com/math.cbrt))
    * `clz32()` ([a standalone shim is also available](https://npmjs.com/math.clz32))
    * `cosh()`
    * `expm1()`
    * `fround()` ([a standalone shim is also available](https://npmjs.com/math.fround))
    * `hypot()`
    * `imul()` ([a standalone shim is also available](https://npmjs.com/math.imul))
    * `log10()` ([a standalone shim is also available](https://npmjs.com/math.log10))
    * `log1p()` ([a standalone shim is also available](https://npmjs.com/math.log1p))
    * `log2()`
    * `sign()` ([a standalone shim is also available](https://npmjs.com/math.sign))
    * `sinh()`
    * `tanh()`
    * `trunc()`

Math functions’ accuracy is 1e-11.

* `Reflect`
    * `apply()` ([a standalone shim is also available](https://npmjs.com/reflect.apply))
    * `construct()`
    * `defineProperty()`
    * `deleteProperty()`
    * `get()`
    * `getOwnPropertyDescriptor()`
    * `getPrototypeOf()` ([a standalone shim is also available](https://npmjs.com/reflect.getprototypeof))
    * `has()`
    * `isExtensible()`
    * `ownKeys()` ([a standalone shim is also available](https://npmjs.com/reflect.ownkeys))
    * `preventExtensions()`
    * `set()`
    * `setPrototypeOf()`

* `Symbol` (only if it already exists)
    * `match` (and corresponding `String#match`, `String#startsWith`, `String#endsWith`, `String#includes`, `RegExp` support)
    * `replace` (and corresponding `String#replace` support)
    * `search` (and corresponding `String#search` support)
    * `split` (and corresponding `String#split` support)

Well-known symbols will only be provided if the engine already has `Symbol` support.

* `String.prototype` Annex B HTML methods ([a standalone shim is also available](https://www.npmjs.com/package/es-string-html-methods))
    * `anchor()`
    * `big()`
    * `blink()`
    * `bold()`
    * `fixed()`
    * `fontcolor()`
    * `fontsize()`
    * `italics()`
    * `link()`
    * `small()`
    * `strike()`
    * `sub()`
    * `sup()`

These methods are part of "Annex B", which means that although they are a defacto standard, you shouldn't use them. None the less, the `es6-shim` provides them and normalizes their behavior across browsers.

## Subclassing
The `Map`, `Set`, and `Promise` implementations are subclassable.
You should use the following pattern to create a subclass in ES5 which will continue to work in ES6:
```javascript
require('es6-shim');

function MyPromise(exec) {
  var promise = new Promise(exec);
  Object.setPrototypeOf(promise, MyPromise.prototype);
  // ...
  return promise;
}
Object.setPrototypeOf(MyPromise, Promise);
MyPromise.prototype = Object.create(Promise.prototype, {
  constructor: { value: MyPromise }
});
```

## String.prototype.normalize
Including a proper shim for `String.prototype.normalize` would increase the size of this library by a factor of more than 4.
So instead we recommend that you install the [`unorm`](https://github.com/walling/unorm) package alongside `es6-shim` if you need `String.prototype.normalize`.
See https://github.com/paulmillr/es6-shim/issues/134 for more discussion.


## WeakMap shim
It is not possible to implement WeakMap in pure javascript.
The [es6-collections](https://github.com/WebReflection/es6-collections) implementation doesn't hold values strongly, which is critical for the collection. `es6-shim` decided to not include an incorrect shim.

`WeakMap` has very unusual use-cases, so you probably won't need it at all (use simple `Map` instead).

## Getting started

```javascript
require('es6-shim');
var assert = require('assert');

assert.equal(true, 'abc'.startsWith('a'));
assert.equal(false, 'abc'.endsWith('a'));
assert.equal(true, 'john alice'.includes('john'));
assert.equal('123'.repeat(2), '123123');

assert.equal(false, NaN === NaN);
assert.equal(true, Object.is(NaN, NaN));
assert.equal(true, -0 === 0);
assert.equal(false, Object.is(-0, 0));

var result = Object.assign({ a: 1 }, { b: 2 });
assert.deepEqual(result, { a: 1, b: 2 });

assert.equal(true, isNaN('a'));
assert.equal(false, Number.isNaN('a'));
assert.equal(true, Number.isNaN(NaN));

assert.equal(true, isFinite('123'));
assert.equal(false, Number.isFinite('123'));
assert.equal(false, Number.isFinite(Infinity));

// Tests if value is a number, finite,
// >= -9007199254740992 && <= 9007199254740992 and floor(value) === value
assert.equal(false, Number.isInteger(2.4));

assert.equal(1, Math.sign(400));
assert.equal(0, Math.sign(0));
assert.equal(-1, Math.sign(-400));

var found = [5, 10, 15, 10].find(function (item) { return item / 2 === 5; });
assert.equal(10, found);

var foundIndex = [5, 10, 15, 10].findIndex(function (item) { return item / 2 === 5; });
assert.equal(1, foundIndex);

// Replacement for `{}` key-value storage.
// Keys can be anything.
var map = new Map([['Bob', 42], ['Foo', 'bar']]);
map.set('John', 25);
map.set('Alice', 400);
map.set(['meh'], 555);
assert.equal(undefined, map.get(['meh'])); // undefined because you need to use exactly the same object.
map.delete('Alice');
map.keys();
map.values();
assert.equal(4, map.size);

// Useful for storing unique items.
var set = new Set([0, 1]);
set.add(2);
set.add(5);
assert.equal(true, set.has(0));
assert.equal(true, set.has(1));
assert.equal(true, set.has(2));
assert.equal(false, set.has(4));
assert.equal(true, set.has(5));
set.delete(5);
assert.equal(false, set.has(5));

// Promises, see
// http://www.slideshare.net/domenicdenicola/callbacks-promises-and-coroutines-oh-my-the-evolution-of-asynchronicity-in-javascript
// https://github.com/petkaantonov/bluebird/#what-are-promises-and-why-should-i-use-them
Promise.resolve(5).then(function (value) {
  assert.equal(value, 5);
  if (value) throw new Error('whoops!');
  // do some stuff
  return anotherPromise();
}).catch(function (e) {
  assert.equal(e.message, 'whoops!');
  assert.equal(true, e instanceof Error);
  // any errors thrown asynchronously end up here
});
```

## Caveats

 - `Object.setPrototypeOf` / `Reflect.setPrototypeOf`
   - Note that null objects (`Object.create(null)`, eg, an object with `null` as its `[[Prototype]]`) can not have their `[[Prototype]]` changed except via a native `Object.setPrototypeOf`.
 - Well-known `Symbol`s
   - In order to make them work cross-realm, these are created with the global `Symbol` registry via `Symbol.for`. This does not violate the spec, but it does mean that `Symbol.for('Symbol.search') === Symbol.search` will be `true`, which it would not by default in a fresh compliant realm.

## [License][license-url]

The project was initially based on es6-shim by Axel Rauschmayer.

[license-url]: https://github.com/paulmillr/es6-shim/blob/master/LICENSE
[spec-html-url]: http://www.ecma-international.org/ecma-262/6.0/
[es5-shim-url]: https://github.com/es-shims/es5-shim

[package-url]: https://npmjs.org/package/es6-shim
[npm-version-svg]: https://versionbadg.es/paulmillr/es6-shim.svg
[deps-svg]: https://david-dm.org/paulmillr/es6-shim.svg
[deps-url]: https://david-dm.org/paulmillr/es6-shim
[dev-deps-svg]: https://david-dm.org/paulmillr/es6-shim/dev-status.svg
[dev-deps-url]: https://david-dm.org/paulmillr/es6-shim#info=devDependencies
[npm-badge-png]: https://nodei.co/npm/es6-shim.png?downloads=true&stars=true
[license-image]: https://img.shields.io/npm/l/es6-shim.svg
[license-url]: LICENSE
[downloads-image]: https://img.shields.io/npm/dm/es6-shim.svg
[downloads-url]: https://npm-stat.com/charts.html?package=es6-shim
[codecov-image]: https://codecov.io/gh/paulmillr/es6-shim/branch/main/graphs/badge.svg
[codecov-url]: https://app.codecov.io/gh/paulmillr/es6-shim/
[actions-image]: https://img.shields.io/endpoint?url=https://github-actions-badge-u3jn4tfpocch.runkit.sh/paulmillr/es6-shim
[actions-url]: https://github.com/paulmillr/es6-shim/actions
