from __future__ import annotations

from typing import Callable, Optional, Protocol

from nicegui import ui


class Item(Protocol):
    title: str


dragged: Optional[card] = None


class column(ui.column):

    def __init__(self, name: str, on_drop: Optional[Callable[[Item, str], None]] = None) -> None:
        super().__init__()
        with self.classes('bg-blue-grey-2 w-60 p-4 rounded shadow-2'):
            ui.label(name).classes('text-bold ml-1')
        self.name = name
        self.on('dragover.prevent', self.highlight)
        self.on('dragleave', self.unhighlight)
        self.on('drop', self.move_card)
        self.on_drop = on_drop

    def highlight(self) -> None:
        self.classes(remove='bg-blue-grey-2', add='bg-blue-grey-3')

    def unhighlight(self) -> None:
        self.classes(remove='bg-blue-grey-3', add='bg-blue-grey-2')

    def move_card(self) -> None:
        global dragged  # pylint: disable=global-statement # noqa: PLW0603
        self.unhighlight()
        dragged.parent_slot.parent.remove(dragged)
        with self:
            card(dragged.item)
        self.on_drop(dragged.item, self.name)
        dragged = None


class card(ui.card):

    def __init__(self, item: Item) -> None:
        super().__init__()
        self.item = item
        with self.props('draggable').classes('w-full cursor-pointer bg-grey-1'):
            ui.label(item.title)
        self.on('dragstart', self.handle_dragstart)

    def handle_dragstart(self) -> None:
        global dragged  # pylint: disable=global-statement # noqa: PLW0603
        dragged = self


#!/usr/bin/env python3
from dataclasses import dataclass


@dataclass
class ToDo:
    title: str


def handle_drop(todo: ToDo, location: str):
    ui.notify(f'"{todo.title}" is now in {location}')


with ui.row():
    with column('Next', on_drop=handle_drop):
        card(ToDo('Simplify Layouting'))
        card(ToDo('Provide Deployment'))
    with column('Doing', on_drop=handle_drop):
        card(ToDo('Improve Documentation'))
    with column('Done', on_drop=handle_drop):
        card(ToDo('Invent NiceGUI'))
        card(ToDo('Test in own Projects'))
        card(ToDo('Publish as Open Source'))
        card(ToDo('Release Native-Mode'))

ui.run()