# Copyright (c) 2024, NVIDIA CORPORATION & AFFILIATES. All rights reserved.
#
# This work is licensed under a Creative Commons
# Attribution-NonCommercial-ShareAlike 4.0 International License.
# You should have received a copy of the license along with this
# work. If not, see http://creativecommons.org/licenses/by-nc-sa/4.0/

"""Improved diffusion model architecture proposed in the paper
"Analyzing and Improving the Training Dynamics of Diffusion Models"."""

# Modifications under MIT License
#
# Copyright (c) 2023 Christopher Friesen
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# 
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
# 
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.

from dataclasses import dataclass
from typing import Union, Optional, Literal

import torch

from modules.unets.unet import DualDiffusionUNet, DualDiffusionUNetConfig
from modules.mp_tools import MPConv, MPFourier, mp_cat, mp_silu, mp_sum, normalize, resample_1d
from modules.formats.format import DualDiffusionFormat

@dataclass
class UNetConfig(DualDiffusionUNetConfig):

    model_channels: int  = 256               # Base multiplier for the number of channels.
    logvar_channels: int = 128               # Number of channels for training uncertainty estimation.
    channel_mult: list[int]    = (1,2,3,4,5) # Per-resolution multipliers for the number of channels.
    channel_mult_noise: Optional[int] = None # Multiplier for noise embedding dimensionality.
    channel_mult_emb: Optional[int]   = None # Multiplier for final embedding dimensionality.
    channels_per_head: int    = 64           # Number of channels per attention head.
    num_layers_per_block: int = 2            # Number of resnet blocks per resolution.
    label_balance: float      = 0.5          # Balance between noise embedding (0) and class embedding (1).
    concat_balance: float     = 0.5          # Balance between skip connections (0) and main path (1).
    res_balance: float        = 0.3          # Balance between main branch (0) and residual branch (1).
    attn_balance: float       = 0.3          # Balance between main branch (0) and self-attention (1).
    attn_levels: list[int]    = (3,4)        # List of resolution levels to use self-attention.
    mlp_multiplier: int = 2                  # Multiplier for the number of channels in the MLP.
    mlp_groups: int     = 8                  # Number of groups for the MLPs.
    latents_height: int = 32                 # Expected height dim of input latents
    pos_channels: int = 0                    # Number of channels for positional encoding.
    pos_emb_bandwidth: float = 1.0           # MPFourier bandwidth for positional encoding.
    pos_balance: float = 0.333               # Balance between noise + class embedding embedding (0) and positional embedding (1).
    use_skips: bool = True                   # Use U-Net structured skip connections in the model.

class Block(torch.nn.Module):

    def __init__(self,
        level: int,                             # Resolution level.
        in_channels: int,                       # Number of input channels.
        out_channels: int,                      # Number of output channels.
        emb_channels: int,                      # Number of embedding channels.
        flavor: Literal["enc", "dec"] = "enc",
        resample_mode: Literal["keep", "up", "down"] = "keep",
        dropout: float         = 0.,       # Dropout probability.
        res_balance: float     = 0.3,      # Balance between main branch (0) and residual branch (1).
        attn_balance: float    = 0.3,      # Balance between main branch (0) and self-attention (1).
        clip_act: float        = 256,      # Clip output activations. None = do not clip.
        mlp_multiplier: int    = 2,        # Multiplier for the number of channels in the MLP.
        mlp_groups: int        = 8,        # Number of groups for the MLP.
        channels_per_head: int = 64,       # Number of channels per attention head.
        use_attention: bool    = False,    # Use self-attention in this block.
    ) -> None:
        super().__init__()

        self.level = level
        self.use_attention = use_attention
        self.num_heads = out_channels // channels_per_head
        self.out_channels = out_channels
        self.flavor = flavor
        self.resample_mode = resample_mode
        self.dropout = dropout
        self.res_balance = res_balance
        self.attn_balance = attn_balance
        self.clip_act = clip_act
        
        self.conv_res0 = MPConv(out_channels if flavor == "enc" else in_channels,
                                out_channels * mlp_multiplier, kernel=(1,3), groups=mlp_groups)
        self.conv_res1 = MPConv(out_channels * mlp_multiplier, out_channels, kernel=(1,3), groups=mlp_groups)
        self.conv_skip = MPConv(in_channels, out_channels, kernel=(1,1), groups=1)

        self.emb_gain = torch.nn.Parameter(torch.zeros([]))
        self.emb_linear = MPConv(emb_channels, out_channels * mlp_multiplier,
                                 kernel=(1,1), groups=mlp_groups) if emb_channels != 0 else None

        if self.use_attention:
            self.emb_gain_qk = torch.nn.Parameter(torch.zeros([]))
            self.emb_gain_v = torch.nn.Parameter(torch.zeros([]))
            self.emb_linear_qk = MPConv(emb_channels, out_channels, kernel=(1,1), groups=1) if emb_channels != 0 else None
            self.emb_linear_v = MPConv(emb_channels, out_channels, kernel=(1,1), groups=1) if emb_channels != 0 else None

            self.attn_qk = MPConv(out_channels, out_channels * 2, kernel=(1,1))
            self.attn_v = MPConv(out_channels, out_channels, kernel=(1,1))
            self.attn_proj = MPConv(out_channels, out_channels, kernel=(1,1))

    def forward(self, x: torch.Tensor, emb: torch.Tensor) -> torch.Tensor:
        
        x = resample_1d(x, mode=self.resample_mode)

        if self.flavor == "enc":
            if self.conv_skip is not None:
                x = self.conv_skip(x)
            x = normalize(x, dim=1) # pixel norm

        y = self.conv_res0(mp_silu(x))

        c = self.emb_linear(emb, gain=self.emb_gain) + 1.
        y = mp_silu(y * c)

        if self.dropout != 0 and self.training == True: # magnitude preserving fix for dropout
            y = torch.nn.functional.dropout(y, p=self.dropout) * (1. - self.dropout)**0.5

        y = self.conv_res1(y)

        if self.flavor == "dec" and self.conv_skip is not None:
            x = self.conv_skip(x)
        x = mp_sum(x, y, t=self.res_balance)
        
        if self.use_attention:

            c = self.emb_linear_qk(emb, gain=self.emb_gain_qk) + 1.

            qk = self.attn_qk(x * c)
            qk = qk.reshape(qk.shape[0], self.num_heads, -1, 2, y.shape[2] * y.shape[3])
            q, k = normalize(qk, dim=2).unbind(3)

            v = self.attn_v(x)
            v = v.reshape(v.shape[0], self.num_heads, -1, y.shape[2] * y.shape[3])
            v = normalize(v, dim=2)

            y = torch.nn.functional.scaled_dot_product_attention(q.transpose(-1, -2),
                                                                 k.transpose(-1, -2),
                                                                 v.transpose(-1, -2)).transpose(-1, -2)
            y = y.reshape(*x.shape)

            c = self.emb_linear_v(emb, gain=self.emb_gain_v) + 1.
            y = mp_silu(y * c)

            y = self.attn_proj(y)
            x = mp_sum(x, y, t=self.attn_balance)

        if self.clip_act is not None:
            x = x.clip_(-self.clip_act, self.clip_act)
        return x

class UNet(DualDiffusionUNet):

    def __init__(self, config: UNetConfig) -> None:
        super().__init__()
        self.config = config

        block_kwargs = {"dropout": config.dropout,
                        "mlp_multiplier": config.mlp_multiplier,
                        "mlp_groups": config.mlp_groups,
                        "res_balance": config.res_balance,
                        "attn_balance": config.attn_balance,
                        "channels_per_head": config.channels_per_head}

        cblock = [config.model_channels * x for x in config.channel_mult]
        cnoise = config.model_channels * config.channel_mult_noise if config.channel_mult_noise is not None else max(cblock)
        cemb = config.model_channels * config.channel_mult_emb if config.channel_mult_emb is not None else max(cblock)
        cpos = config.pos_channels

        self.num_levels = len(config.channel_mult)

        # Embedding.
        self.emb_fourier = MPFourier(cnoise)
        self.emb_noise = MPConv(cnoise, cemb, kernel=())
        self.emb_label = MPConv(config.label_dim, cemb, kernel=()) if config.label_dim != 0 else None
        self.emb_label_unconditional = MPConv(1, cemb, kernel=()) if config.label_dim != 0 else None

        # Training uncertainty estimation.
        self.logvar_fourier = MPFourier(config.logvar_channels)
        self.logvar_linear = MPConv(config.logvar_channels, 1, kernel=(), disable_weight_norm=True)

        # Position embedding
        if cpos != 0 and len(cblock) > 1:
            raise ValueError("Positional embedding is only supported for single resolution models")
        self.position_fourier = MPFourier(cpos, bandwidth=config.pos_emb_bandwidth) if cpos != 0 else None
        self.position_linear = MPConv(cpos, cemb, kernel=()) if cpos != 0 else None

        # Encoder.
        self.enc = torch.nn.ModuleDict()
        cout = config.in_channels * config.latents_height + 1 # 1 extra const channel

        for level, channels in enumerate(cblock):
            
            if level == 0:
                cin = cout
                cout = channels
                self.enc[f"conv_in"] = MPConv(cin, cout, kernel=(1,3))
            else:
                self.enc[f"block{level}_down"] = Block(level, cout, cout, cemb, use_attention=level in config.attn_levels,
                                                       flavor="enc", resample_mode="down", **block_kwargs)
            
            for idx in range(config.num_layers_per_block):
                cin = cout
                cout = channels
                self.enc[f"block{level}_layer{idx}"] = Block(level, cin, cout, cemb, use_attention=level in config.attn_levels,
                                                             flavor="enc", **block_kwargs)

        # Decoder.
        self.dec = torch.nn.ModuleDict()
        skips = [block.out_channels for block in self.enc.values()]
        
        for level, channels in reversed(list(enumerate(cblock))):
            
            if level == len(cblock) - 1:
                self.dec[f"block{level}_in0"] = Block(level, cout, cout, cemb, use_attention=True,
                                                      flavor="dec", **block_kwargs)
                self.dec[f"block{level}_in1"] = Block(level, cout, cout, cemb, use_attention=True,
                                                      flavor="dec", **block_kwargs)
            else:
                self.dec[f"block{level}_up"] = Block(level, cout, cout, cemb, use_attention=level in config.attn_levels,
                                                     flavor="dec", resample_mode="up", **block_kwargs)
            for idx in range(config.num_layers_per_block + 1):
                cin = cout + skips.pop() if config.use_skips == True else cout
                cout = channels
                self.dec[f"block{level}_layer{idx}"] = Block(level, cin, cout, cemb, use_attention=level in config.attn_levels,
                                                             flavor="dec", **block_kwargs)
                
        self.out_gain = torch.nn.Parameter(torch.zeros([]))
        self.conv_out = MPConv(cout, config.out_channels * config.latents_height, kernel=(1,3))

    def get_class_embeddings(self, class_labels: torch.Tensor, conditioning_mask: torch.Tensor) -> torch.Tensor:
        u_embedding = self.emb_label_unconditional(torch.ones(1, device=self.device, dtype=self.dtype))
        if self.config.label_dim != 0:
            c_embedding = self.emb_label(normalize(class_labels).to(device=self.device, dtype=self.dtype))
            return mp_sum(u_embedding, c_embedding, t=conditioning_mask.unsqueeze(1).to(self.device, self.dtype))
        else:
            return u_embedding
    
    def get_clap_embeddings(self, clap_embeddings: torch.Tensor,
        unconditional_clap_embedding: torch.Tensor, conditioning_mask: torch.Tensor) -> torch.Tensor:
        if self.config.label_dim != 0:
            clap_embeddings = self.emb_label(normalize(clap_embeddings).to(device=self.device, dtype=self.dtype))
            unconditional_clap_embedding = self.emb_label(normalize(unconditional_clap_embedding).to(device=self.device, dtype=self.dtype))
            return mp_sum(unconditional_clap_embedding, clap_embeddings, t=conditioning_mask.unsqueeze(1).to(self.device, self.dtype))
        else:
            return unconditional_clap_embedding
        
    def get_sigma_loss_logvar(self, sigma: Optional[torch.Tensor] = None, class_embeddings: Optional[torch.Tensor] = None) -> torch.Tensor:
        return self.logvar_linear(self.logvar_fourier(sigma.flatten().log() / 4)).view(-1, 1, 1, 1).float()
    
    def get_latent_shape(self, latent_shape: Union[torch.Size, tuple[int, int, int, int]]) -> torch.Size:
        return latent_shape[0:2] + ((latent_shape[2] // 2**(self.num_levels-1)) * 2**(self.num_levels-1),
                                    (latent_shape[3] // 2**(self.num_levels-1)) * 2**(self.num_levels-1))

    @torch.no_grad()
    def convert_to_inpainting(self) -> None:
        raise NotImplementedError("Inpainting is not supported in this model.")

    def forward(self, x_in: torch.Tensor,
                sigma: torch.Tensor,
                format: DualDiffusionFormat,
                class_embeddings: torch.Tensor,
                t_ranges: Optional[torch.Tensor] = None,
                x_ref: Optional[torch.Tensor] = None) -> torch.Tensor:

        with torch.no_grad():
            sigma = sigma.view(-1, 1, 1, 1)
            
            # Preconditioning weights.
            c_skip = self.config.sigma_data ** 2 / (sigma ** 2 + self.config.sigma_data ** 2)
            c_out = sigma * self.config.sigma_data / (sigma ** 2 + self.config.sigma_data ** 2).sqrt()
            c_in = 1 / (self.config.sigma_data ** 2 + sigma ** 2).sqrt()
            c_noise = (sigma.flatten().log() / 4).to(self.dtype)

            x = (c_in * x_in).to(self.dtype)
 
        # Embedding.
        emb = self.emb_noise(self.emb_fourier(c_noise))
        if self.config.label_dim != 0:
            emb = mp_sum(emb, class_embeddings.to(emb.dtype), t=self.config.label_balance)
        emb = emb.unsqueeze(2).unsqueeze(3)

        if self.position_fourier is not None:
            pos = torch.linspace(-1, 1, x.shape[3], device=self.device, dtype=self.dtype)
            pos_emb = self.position_linear(self.position_fourier(pos)).transpose(-1,-2).unsqueeze(0).unsqueeze(-2)
            emb = mp_sum(emb, pos_emb.to(emb.dtype), t=self.config.pos_balance)

        emb = mp_silu(emb).to(x.dtype, memory_format=self.memory_format)

        # Encoder.
        x = x.reshape(x.shape[0], x.shape[1]*self.config.latents_height, 1, x.shape[3])
        if self.memory_format == torch.channels_last:
            x = x.contiguous(memory_format=torch.channels_last)
        x = torch.cat((x, torch.ones_like(x[:, :1])), dim=1)

        skips = []
        for name, block in self.enc.items():
            x = block(x) if "conv" in name else block(x, emb)
            if self.config.use_skips == True: skips.append(x)

        # Decoder.
        for name, block in self.dec.items():
            if "layer" in name and self.config.use_skips == True:
                x = mp_cat(x, skips.pop(), t=self.config.concat_balance)
            x = block(x, emb)

        x = self.conv_out(x, gain=self.out_gain)
        x = x.reshape(x.shape[0], x.shape[1] // self.config.latents_height, self.config.latents_height, x.shape[3])
        if self.memory_format == torch.channels_last:
            x = x.contiguous(memory_format=torch.contiguous_format)
        D_x = c_skip * x_in + c_out * x.float()
        
        if self.config.inpainting == False and x_ref is not None:
            D_x = mp_sum(x_ref[:, :-1].float(), D_x, t=x_ref[:, -1:].float())

        return D_x
    
