"""
Visualize DVS Input

https://synsense.gitlab.io/sinabs-dynapcnn/getting_started/notebooks/visualize_speck_dvs_input.html

SPECK integrates a Dynamic Vision Sensor (DVS) on the chip itself.
Users can read, visualize and save the events that generated by the embedded DVS after making some modification on the hardware configuration.

How to visualize the DVS Sensor events on the speck2edevkit.
We show a way that build necessary objects from scratch.
"""
#######################################################################################################
import sinabs.backend.dynapcnn.io as sio
#######################################################################################################
# 1. Open devkit
#######################################################################################################
# list all the device we plugged to our PC
device_map = sio.get_device_map()
print(device_map)

# when open devkit, we just need to pass the device name to the 'open_device' function of samna.device
devkit = sio.open_device("speck2fdevkit:0")
#######################################################################################################
# 2. Build samna graph
#######################################################################################################
# Samna graph defines how data flow between the devkit and host machine.
# We can send the raw DVS input events to a VizEventStreamer node which can help us to viusalize them.
# Build a graph in samna to show dvs
import samna

samna_graph = samna.graph.EventFilterGraph()

_, _, streamer = samna_graph.EventFilterGraph([
    devkit.get_model_source_node(), # Specify the source of events to this graph as the devkit
    "Speck2fDvsToVizConverter", # Convert the events to visualizer events
    "VizEventStreamer" # Stream events to a visualizer via a streamer node
])
#######################################################################################################
# 3. Launch visualizer window
#######################################################################################################
# Any data or configuration happens over a user-specified tcp port.
# Specify the tcp port of the visualizer
visualizer_port = "tcp://0.0.0.0:40000"

# Launch visualizer
gui_process = sio.launch_visualizer(receiver_endpoint=visualizer_port, disjoint_process=True)

# 3.1 Setup the visualizer
# Visualizer configuration branch of the graph
visualizer_config, _ = samna_graph.sequential([
    samna.BasicSourceNode_ui_event(), # For generating UI commands
    streamer
])

# Connect to the visualizer
streamer.set_streamer_destination(visualizer_port)
if streamer.wait_for_receiver_count() == 0:
    raise Exception(f'Connecting to visualizer on {visualizer_port} fails.')

# Specify which plot is to be shown in the visualizer
plot1 = samna.ui.ActivityPlotConfiguration(image_width=128, image_height=128, title="DVS Layer", layout=[0, 0, 1, 1])
visualizer_config.write([
    samna.ui.VisualizerConfiguration(plots=[plot1])
])
#######################################################################################################
# 4. Launch the devkit and start visualization
#######################################################################################################
# 4.1 Start samna graph
samna_graph.start()

# 4.2 Enable the DVS array on speck
# The configuration of Speck2e devkit has an attribute named dvs_layer which is an instance of samna.speck2e.configuration.DvsLayerConfig.
#
# Parameters
# monitor_enable: monitor the pre-processing block of the devkit. The output events from the pre-processing block
#                 are Spike with the .layer attribute equals to 13. If user doesn't apply any filtering, cropping, mirroring
#                 etc. in the dvs_layer, collected Spikes should be the same as the raw DvsEvent.
# raw_monitor_enable: monitor the raw events from the dvs, the event type of the raw events is DvsEvent.
devkit_config = samna.speck2e.configuration.SpeckConfiguration()
# enable monitoring the inputs from the DVS sensor
devkit_config.dvs_layer.raw_monitor_enable = True
# apply this configuration
devkit.get_model().apply_configuration(devkit_config)
#######################################################################################################
# 5. Stop the devkit and visualization
#######################################################################################################
# Stopping the graph is how you can terminate the visualization process.
# This doesn't terminate the chip/device from generating events.
# Stop the graph
samna_graph.stop()

# If we need a sub-process to launch the visualizer, use that to terminate the visualizer.
if gui_process:
    gui_process.terminate()
    gui_process.join()
    