import { camelCase, groupBy, map } from "lodash-es";
/**
 * 色板类
 */
export class Palette {
  id: string
  limited: boolean
  platforms: string[]
  colors: Color[]
  constructor({ id, limited, platforms, colors }: { id: string, limited: boolean, platforms: string[], colors?: string[] }) {
    [this.id, this.limited, this.platforms] = [id, limited, platforms];
    if (colors) this.colors = colors.map(v => new Color(v));
  }
}

/**
 * 色板搜索结果类
 */
export class MatchedPalette extends Palette {
  match: number[];
  deltaE: number;
  constructor({ id, limited, platforms, colors }: Palette, matchedColors: MatchedPaletteColorInfo[]) {
    super({ id, limited, platforms });
    this.colors = colors;
    this.match = [matchedColors[0].index];
    this.deltaE = matchedColors[0].deltaE;
  }
}

/**
 * 颜色类
 */
export class Color {
  r: number
  g: number
  b: number
  constructor(rgb: string | number[]) {
    if (typeof rgb === "string") {
      rgb = rgb.toLowerCase();
      if (rgb[0] === '#') rgb = rgb.substr(1);
      this.r = (rgb.charCodeAt(0) - 48) % 39 * 16 + (rgb.charCodeAt(1) - 48) % 39;
      this.g = (rgb.charCodeAt(2) - 48) % 39 * 16 + (rgb.charCodeAt(3) - 48) % 39;
      this.b = (rgb.charCodeAt(4) - 48) % 39 * 16 + (rgb.charCodeAt(5) - 48) % 39;
    } else {
      [this.r, this.g, this.b] = rgb;
    }
  }
  deltaE(cc: Color) {
    return Math.sqrt((this.r - cc.r) ** 2 + (this.g - cc.g) ** 2 + (this.b - cc.b) ** 2);
  }
  toString() {
    return "#" + String.fromCharCode(this.r / 16 + (this.r / 16 < 10 ? 48 : 87)) + String.fromCharCode(this.r % 16 + (this.r % 16 < 10 ? 48 : 87))
      + String.fromCharCode(this.g / 16 + (this.g / 16 < 10 ? 48 : 87)) + String.fromCharCode(this.g % 16 + (this.g % 16 < 10 ? 48 : 87))
      + String.fromCharCode(this.b / 16 + (this.b / 16 < 10 ? 48 : 87)) + String.fromCharCode(this.b % 16 + (this.b % 16 < 10 ? 48 : 87));
  }
}

const _paletteData = [
  {
    name: "Agony",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "676767", "6f6f6f", "797979", "818287", "898b93",
      "454457", "454558", "48485b", "49485d", "4a495d",
      "5b5c7d", "616383", "696b8a", "737992", "7b8197",
      "5c5d7e", "5d5e7f", "606282", "626384", "656887",
      "4e4d64", "525272", "5b5c7d", "676988", "717591",
      "5a5b7b", "616383", "686b8a", "727692", "7a8197",
      "373646", "3b3a4b", "424153", "48485c", "4e4d64",
      "48485c", "4e4d64", "515273", "5a5b7b", "626384",
      "545a74", "545b74", "545b74", "545d74", "545e74",
      "a1a0a3", "a1a0a3", "9c9b9e", "969599", "919095",
      "75737c", "727079", "6c6a74", "66646f", "5f5d69",
      "53515e", "4d4b57", "47454f", "3d3b44", "36343b",
      "38373e", "35343b", "302f36", "2a292e", "252428",
      "8d8e8a", "83847f", "72736d", "5f605b", "4e4f4b",
      "777873", "767772", "73746f", "70716b", "6d6e68",
      "c0c0bf", "b8b8b7", "aeaeac", "a1a29f", "969794",
      "8a8b87", "838480", "7d7e79", "73746f", "6b6c66",
      "6c6c62", "787869", "858475", "929183", "9d9b8e",
    ],
  },
  {
    name: "Bastille",
    limited: true,
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "d92e2f", "df3330", "e43731", "e93c33", "ef4135",
      "d2272c", "d72c2e", "dd312f", "e23531", "e63a33",
      "c92029", "cf242b", "d5292d", "da2f2e", "df3330",
      "c11827", "c71d29", "cc222a", "d2272c", "d82c2d",
      "ba1225", "bf1726", "c41b28", "c9202a", "ce252b",
      "b60e23", "b91124", "bc1525", "c11927", "c61e28",
      "f4f3f6", "f9f8fa", "fdfdfd", "ffffff", "ffffff",
      "edebf1", "f3f2f5", "f8f7f9", "fcfcfd", "ffffff",
      "e6e3ea", "ebe9ef", "f1f0f4", "f6f6f8", "fbfbfb",
      "dedbe5", "e3e1ea", "eae8ee", "f0eff3", "f5f4f7",
      "d7d3df", "dcdae3", "e2e0e8", "e8e7ec", "eeedf2",
      "d1cdda", "d6d2de", "dbd8e2", "e1dee7", "e7e4eb",
      "003d80", "01458b", "004b95", "00519e", "0055a4",
      "013574", "013d80", "00448b", "014b96", "00519e",
      "012e68", "003674", "013d80", "00448b", "004c95",
      "01265d", "012e68", "003574", "003d80", "00458b",
      "002052", "01265d", "012e69", "013574", "003d80",
      "011a4a", "012052", "01265c", "012d68", "013574",
    ],
  },
  {
    name: "Beach",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "ffce25", "ffcf28", "fecf2b", "fed02f", "fdd133",
      "fed031", "fdd135", "fcd23a", "fdd23f", "fcd344",
      "fcd342", "fbd347", "fbd54d", "fad552", "fad758",
      "fad656", "f9d75c", "f9d762", "f8d968", "f7da6d",
      "f8d96b", "f7da71", "f6db77", "f6dc7d", "f5de83",
      "f6dd81", "f5de87", "f5e08c", "f5e092", "f5e197",
      "f4e095", "f4e29a", "f4e39f", "f4e4a3", "f5e6a9",
      "f4e4a6", "f5e6ac", "f6e7b2", "f7eab8", "f9ecbe",
      "f9ebbb", "fbedc1", "fdeec7", "fff1cd", "fff2d3",
      "fff2d0", "fff4d6", "fff6db", "fff8e1", "fffae6",
      "fff9e4", "fffbe9", "fffded", "fffdf1", "fffff5",
      "fffef3", "fffff6", "fffffa", "fffffc", "fffffe",
      "fffffd", "ffffff", "ffffff", "fbfefe", "f5fcfe",
      "f7fdfe", "effbfc", "e7f9fa", "def6f6", "d5f3f3",
      "d9f3f4", "cff0f0", "c3eceb", "b8e9e6", "ace4e1",
      "b0e6e3", "a5e2dd", "98ded7", "8cdad2", "80d5cc",
      "85d7ce", "79d3c8", "6dcfc3", "63cbbd", "58c7b8",
      "5cc9ba", "52c5b5", "49c2b0", "41bfac", "3abda9",
    ],
  },
  {
    name: "Classic",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "461011", "671718", "952024", "bb282b", "d12e33",
      "3d181e", "58222a", "84333f", "ae4858", "ca576d",
      "200d15", "3b1b25", "713844", "ae616f", "d48897",
      "260a16", "3e1024", "6e1c3d", "9d275b", "be317a",
      "0f1221", "161a34", "26305b", "3c498e", "5166b7",
      "081826", "0c2a44", "164777", "1e6ea8", "2490c6",
      "0e3345", "174961", "29728d", "3f9fb7", "55bfd2",
      "192e2a", "25453d", "3a695f", "54968b", "70bcb1",
      "112c18", "1a4626", "2c7940", "41a95e", "53c97c",
      "3a3b0b", "606412", "8f921f", "b8ba2c", "d1d037",
      "5a4629", "7f613a", "a98c50", "ceb56a", "e6d186",
      "7c411b", "925425", "ad7137", "c69650", "ddb66e",
      "9a3a1e", "b34f29", "ce723d", "e1a05a", "e8c381",
      "321709", "602911", "a44d1e", "d17426", "d79227",
      "291d1c", "4c3732", "8f685c", "c99f8e", "e4bfad",
      "282724", "5d5a53", "bebaaf", "f5f3eb", "ffffff",
      "202324", "474f4f", "96a4a4", "d6e1e1", "f3fbfa",
      "1d1e22", "3f434b", "8a8e9f", "c9cddd", "e7ebf7",
    ],
  },
  {
    name: "Classic Saturated",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "520406", "790507", "af060b", "dc080d", "f60a11",
      "461019", "641622", "962132", "c53148", "e43d5e",
      "240914", "421423", "7e2c3d", "c05065", "e5778d",
      "2d0415", "490623", "810a3a", "b80d59", "de127c",
      "0b0f25", "10163b", "1a2867", "2a3da1", "3a58ce",
      "01192d", "002b51", "00488d", "0074c7", "009cea",
      "023851", "075072", "137ea4", "25b0d2", "39d4ee",
      "15332d", "1e4c41", "307465", "46a595", "5fcdbd",
      "0b3215", "115022", "1b8a38", "2ac154", "39e474",
      "454600", "727700", "a8ac05", "d8da0c", "f4f315",
      "65481e", "8f642b", "bd933c", "e5c154", "fcde71",
      "923d06", "ab510d", "c8711d", "e19c36", "f6be55",
      "b62b02", "d2420a", "ee6a1d", "ffa13c", "ffca6a",
      "3c1400", "722300", "c24400", "f77100", "ff9c00",
      "2c1b19", "52332c", "9b6251", "d69981", "f0bba1",
      "292823", "5f5b51", "c1bbac", "f8f5e9", "ffffff",
      "1f2425", "455151", "93a7a7", "d4e4e4", "f1fdfc",
      "1c1d23", "3d424e", "858ba4", "c5cbe2", "e4eafb",
    ],
  },
  {
    name: "Conquest",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "bbdbcc", "afd2c3", "a1c8b7", "8ebaa8", "7cad99",
      "b5d6c8", "a7ccbc", "96c0af", "84b3a0", "70a490",
      "add1c2", "9dc5b4", "8bb8a6", "78a996", "669d89",
      "7e8a89", "798482", "737d7b", "6b7472", "636a67",
      "7b8786", "757f7e", "6f7876", "676f6c", "5f6663",
      "798483", "717b79", "69716f", "616865", "5a605d",
      "9faea3", "97a49a", "8b978e", "7f8981", "707973",
      "9aa89e", "909d94", "848f87", "767f79", "68706a",
      "95a298", "89958c", "7b857e", "6d756f", "606762",
      "87736d", "816f69", "776661", "6b5c58", "5f5350",
      "84706b", "7b6964", "6f605b", "645754", "584e4b",
      "7e6c67", "74635f", "685a56", "5b504d", "504845",
      "599a95", "55928e", "4d8480", "467571", "3d6562",
      "579590", "518a86", "497b77", "416c69", "385c5a",
      "548f8a", "4d827e", "44726f", "3c625f", "335351",
      "7d8d8c", "738382", "647271", "546160", "444e4e",
      "778786", "6b7978", "5c6969", "4a5555", "3a4343",
      "70807f", "616f6e", "515d5c", "3f4948", "303938",
    ],
  },
  {
    name: "Corpus",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "e2f4f7", "b2d4e2", "75adc8", "529abf", "4093bf",
      "65f3fa", "63e1fa", "60c7f9", "5cabfa", "5991f9",
      "75dcff", "4dcee6", "19bec7", "0da7ac", "119298",
      "44c377", "3fb770", "39a566", "33945a", "2d8450",
      "ecd557", "d1cd37", "afc40e", "67c40f", "19c820",
      "e5ff01", "daf301", "cae101", "b9cd01", "a8bb01",
      "f1d711", "e3ca10", "ccb60d", "b5a10b", "9f8d0a",
      "fff37b", "f5ea75", "e7db6e", "d6cc66", "c7be5e",
      "e8ba57", "e0aa4b", "d59238", "c87825", "bd6012",
      "f8ae10", "eea511", "e09711", "d08912", "c17b13",
      "dc491c", "cb431a", "b23b17", "973113", "7e2a0f",
      "ed6846", "e36140", "d45736", "c34a2d", "b34023",
      "fa6165", "f05357", "e23e43", "d2272c", "c31317",
      "bda488", "b49c81", "a79178", "98846d", "8a7863",
      "616161", "585858", "4a4a4b", "3c3b3b", "2e2e2d",
      "5a5d68", "535761", "484e58", "3d444d", "323a44",
      "788a7d", "6f8074", "617165", "526055", "435248",
      "43316f", "3c2c63", "322453", "271c41", "1c142f",
    ],
  },
  {
    name: "Darkness",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "538bac", "528aab", "5288aa", "5187a9", "5185a7",
      "5185a8", "5183a6", "5082a5", "4f7fa4", "4e7ca2",
      "4e7da2", "4e7ba1", "4d79a0", "4b769e", "4b739c",
      "4b759d", "4a729a", "496f99", "496b97", "476995",
      "486a95", "476794", "466491", "44618f", "435e8e",
      "445f8f", "435c8c", "42598a", "415688", "405386",
      "415487", "3f5184", "3e4e82", "3e4b81", "3c497e",
      "3d4a7f", "3c477c", "3b457a", "3a4278", "393f77",
      "394077", "383f75", "383c73", "373a71", "37386f",
      "363970", "35376e", "35356c", "35356b", "343368",
      "353369", "333267", "343165", "333063", "322f61",
      "333062", "322f60", "322e5e", "322e5c", "312d59",
      "312d5a", "312d59", "312d57", "302b53", "302c52",
      "302b53", "2f2b51", "2f2b4e", "2f2b4c", "2e2b4a",
      "2f2b4b", "2e2b49", "2e2b47", "2e2b45", "2d2b42",
      "2e2b44", "2e2b42", "2e2b40", "2d2b3e", "2d2b3c",
      "2d2b3c", "2d2b3b", "2c2b39", "2d2b38", "2d2b35",
      "2c2b36", "2c2c34", "2c2b33", "2c2c32", "2c2c30",
    ],
  },
  {
    name: "Daybreak",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "a0c6e8", "9bb7d9", "95a4c8", "8e91b5", "887ea3",
      "c491ba", "b888b1", "a57ca4", "916d95", "7e6188",
      "efaed8", "e6a8d2", "dba0ca", "ce98c1", "c390b9",
      "db97cf", "d692ce", "ce8bcb", "c584c8", "be7dc6",
      "ff84d2", "f985d0", "ef86ce", "e587cb", "db88c9",
      "ffa8df", "f5acda", "e7b1d4", "d7b7cd", "c8bcc6",
      "5f6a75", "7c858d", "9a9fa5", "b5b8bc", "cbccce",
      "495563", "656f7a", "828a92", "9fa5aa", "babcc0",
      "354353", "4e5a67", "6b747f", "889097", "a5aaaf",
      "243445", "394856", "545f6c", "717a84", "8e959c",
      "c05531", "b85f41", "b06953", "a87464", "a07e76",
      "ca481b", "a9421d", "863b20", "653423", "492e25",
      "db4c1a", "be461c", "9b3f1f", "793821", "593123",
      "e94f19", "d04a1b", "b0431d", "8e3c20", "6b3522",
      "b74724", "a14429", "89402e", "713c33", "593938",
      "3b5b76", "34536e", "294761", "1e3a54", "132e47",
      "6489a6", "5e839e", "557a93", "4a6f87", "41667c",
      "a4d1f2", "96c1e1", "80a9c8", "698eac", "537693",
    ],
  },
  {
    name: "Discord",
    limited: true,
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "7289da", "6c82d0", "6579c1", "5c6eaf", "51639d",
      "6e85d2", "677cc5", "5e71b3", "5465a2", "4c5b91",
      "697ec8", "6074b8", "5768a6", "4d5d95", "465487",
      "6276bc", "596baa", "4f6099", "48568a", "43507f",
      "ffffff", "fbfafb", "f5f5f5", "eeeeee", "e6e7e7",
      "fcfcfc", "f6f6f6", "eff0f0", "e8e8e8", "e0e0e0",
      "f7f7f8", "f1f1f1", "e9eaea", "e1e1e2", "dadada",
      "f2f2f2", "ebebeb", "e3e4e3", "dcdcdb", "d5d5d5",
      "99aab5", "95a6b1", "90a1ab", "8a99a3", "83929b",
      "97a7b1", "91a1ac", "8b9ba5", "85939d", "7e8b94",
      "92a3ad", "8d9ca6", "86959f", "7f8e96", "79858e",
      "8e9da8", "8896a0", "818e98", "7a878f", "748088",
      "2c2f33", "292b2f", "24262a", "1e2022", "18191c",
      "2a2d30", "26292c", "212226", "1b1c1e", "151618",
      "282a2e", "232428", "1c1f21", "17181a", "111314",
      "23272a", "212427", "1d2123", "181c1e", "151719",
      "222528", "1f2225", "1a1d20", "16181a", "111416",
      "202326", "1c1f22", "181b1c", "131617", "0f1113",
    ],
  },
  {
    name: "Dojo",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "63779c", "6a7fa7", "748cb7", "809ac9", "8ca8dc",
      "919c3e", "a0ac44", "b5c34e", "cbdb58", "e0f061",
      "4f442b", "665b3e", "887a5c", "af9e7b", "d0bf98",
      "23324f", "3c4f69", "5f7b91", "87aabd", "aad5e4",
      "23324f", "2f436a", "405c91", "5377bc", "658fe5",
      "344d47", "476861", "629085", "7fbcae", "9be4d3",
      "009974", "00a880", "00bf90", "00d9a3", "00efb4",
      "4d1a09", "691f07", "8e2604", "99310f", "993b1d",
      "a58c30", "b09633", "c0a339", "d3b23e", "e3c142",
      "a69b85", "b1a68f", "c3b79c", "d5c9ab", "e7d9ba",
      "5a39a7", "623eb4", "6c44c8", "774bdd", "8252f2",
      "4559a5", "4b5fb2", "5269c2", "5b73d6", "627de7",
      "790b12", "7c1319", "802024", "852d30", "89393a",
      "28566f", "306785", "3c80a5", "489bc9", "54b5e9",
      "2a4e95", "33589b", "4266a4", "5176ac", "5e86b5",
      "0e8aa3", "1795af", "23a6c0", "2fb8d3", "3cc8e4",
      "a6662e", "b16d32", "c17736", "d4833c", "e58d41",
      "a35c1e", "af6b2f", "c18147", "d39962", "e5b07b",
    ],
  },
  {
    name: "Easter",
    limited: true,
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "cda75e", "d0af5f", "d1b35f", "dabb62", "dac262",
      "cdb45e", "d0bd5f", "d1c15f", "dac962", "dad162",
      "b5af59", "bdb85b", "c5c06d", "cdc970", "d8d373",
      "87ac5f", "a2bf83", "aec793", "b6d099", "bedb9f",
      "61aa8a", "83bca4", "94c5af", "99ceb7", "9fd8c0",
      "56a3a9", "76b5bc", "87bfc3", "8cc7cc", "92d1d6",
      "6da5bc", "8db8cb", "9bc1d1", "a2cbdb", "a9d4e5",
      "9fa9bf", "b4bccd", "bec4d3", "c7cede", "d0d8e9",
      "bea5be", "cbb9cc", "d2c2d3", "dccbdd", "e8d5e9",
      "e4abc5", "e9bed2", "ecc6d7", "f4cfe2", "f5d6ea",
      "e8a1c2", "edb5d0", "efbfd6", "f1c4dd", "f3cce6",
      "cc81b4", "d79dc5", "dcaacc", "e6b0d5", "eeb7df",
      "b261a7", "c384ba", "cb94c3", "d59acc", "dfa0d6",
      "9754a0", "a963b1", "b477bb", "bc7bc3", "c57fcc",
      "8c5091", "9e54a2", "ab68ae", "b16bb4", "b96ebd",
      "99527c", "a95b92", "b470a1", "bc73a7", "c577af",
      "a0546e", "af5e88", "ba7397", "c2769d", "cb7aa3",
      "a85661", "b5617d", "bf768e", "c87a93", "d17e99",
    ],
  },
  {
    name: "Eminence",
    limited: true,
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "e6b0fe", "b7a1f6", "858be9", "4c6cd3", "234db9",
      "d6a3ec", "aa95e4", "7b80d8", "4865c7", "2249b1",
      "bd91d1", "9684ca", "6d73c0", "415cb4", "1f45a4",
      "a27cb3", "8070ac", "5f63a6", "39519f", "1d3f95",
      "886896", "6c5f91", "51558f", "32478c", "1a3988",
      "d0ccfe", "adb6fb", "8545fe", "6b21ff", "6c648a",
      "bdb9e7", "9aa3df", "793fe7", "5d1dde", "5c5676",
      "a09dc4", "8188bb", "6735c4", "4917ad", "48435d",
      "83809f", "686f98", "542b9f", "34107a", "353144",
      "8685ff", "54529c", "596a9b", "4f3768", "563d77",
      "7876e4", "4b498c", "4f5d8a", "45305c", "4d376a",
      "6261ba", "3e3d74", "424d72", "39284c", "3f2c58",
      "4d4c92", "32315d", "353f5c", "2e203d", "312344",
      "7d48d8", "9078ea", "939ef8", "92cbf6", "96eaf5",
      "7241c5", "846ed7", "8690e3", "85b9e0", "88d5e0",
      "6438ab", "735fbb", "747dc3", "729ec0", "75b7bf",
      "533090", "5f4f9c", "6066a1", "5c819c", "60959c",
      "452777", "4e417f", "4c5281", "49667c", "4b767c",
    ],
  },
  {
    name: "Eximus",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "697785", "5f6b79", "515c69", "434c57", "353d46",
      "e6d56f", "e5d16d", "e2c969", "e0c166", "dfba64",
      "13b1e9", "14a8d7", "139bbf", "118ca3", "11808c",
      "cb2f2f", "c43232", "ba3434", "b03838", "a53a3b",
      "7facb8", "749eaa", "688d97", "587882", "4c676f",
      "20ec14", "23da13", "2ac213", "30a613", "368d12",
      "decd9d", "d8c796", "cfc08d", "c5b982", "bcb179",
      "8ade6b", "83d563", "78c557", "6db64a", "63a93f",
      "ecde14", "ecd714", "eccd14", "ecc214", "ecb814",
      "f2f0e9", "e9f0e9", "dcefea", "ceeeeb", "c1edec",
      "6cb4f3", "67afee", "5fa7e6", "589fde", "5097d5",
      "ec1414", "db1414", "c21615", "a61716", "8d1718",
      "5b3f6e", "573969", "513361", "4a2a58", "442250",
      "a2bfa5", "96af99", "839886", "6e7e70", "5c665e",
      "5ad168", "59ce74", "57c883", "54c194", "52bca4",
      "f9f3ea", "f6ecdb", "f2e1c4", "ecd5aa", "e7ca94",
      "dcaa7f", "daa171", "d79561", "d1884e", "cd7d3d",
      "ec9d14", "ec9814", "ec9014", "ec8714", "ec7e14",
    ],
  },
  {
    name: "Fear",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "a3bbb8", "b6c7c6", "c7d3d2", "d2dadb", "d7dcde",
      "809597", "95a6a8", "a5b3b5", "afbcbd", "bdcbcd",
      "3c5155", "51666a", "63777b", "6f8286", "798d92",
      "2c3c40", "374b50", "40575d", "475e65", "4c666d",
      "253136", "324349", "405359", "4a5d64", "50656c",
      "1b2328", "2d373d", "3e4a50", "4c585e", "536067",
      "161c20", "2c3338", "3f474d", "4e565c", "555e64",
      "12161b", "292f34", "3b4248", "4a5257", "50585e",
      "090a0d", "1c1f23", "2c3035", "3b3f45", "3f4349",
      "050608", "17181c", "27292d", "34363b", "383b40",
      "0d1014", "1f2329", "2f363c", "3d444a", "414850",
      "425260", "4a5e6c", "516874", "586f7c", "5f7885",
      "607a8d", "607c8d", "607e8d", "607e8d", "60818d",
      "6d7f81", "869597", "98a4a6", "a4b0b1", "b1bebf",
      "8b938f", "a2a9a5", "b1b7b3", "babfbc", "c9cecc",
      "8f9188", "a6a79f", "b5b6af", "bcbdb8", "cbccc6",
      "9ec4d8", "99bed1", "93b6c8", "8cacbf", "85a4b5",
      "9dc2d6", "97bace", "91b3c6", "88a8ba", "82a1b1",
    ],
  },
  {
    name: "Fire",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "ffde93", "ffe3a4", "ffe9b6", "fff0c4", "fffcd1",
      "ffd457", "ffdb6c", "ffe384", "ffea8e", "fff79b",
      "ffc013", "ffcb20", "ffd633", "ffdf35", "ffea3f",
      "ffa510", "ffb41c", "ffc32e", "ffca30", "ffd738",
      "ff770a", "ff8914", "ff9e24", "ffa525", "ffb22d",
      "ff6208", "ff7411", "ff8b1f", "ff9120", "ff9f27",
      "e94300", "f55500", "ff6e07", "ff7412", "ff831f",
      "ba1a00", "c82700", "db3b00", "e33e00", "f44700",
      "a00b00", "b01500", "c32300", "cb2500", "d92d00",
      "9b1b00", "b32700", "d13800", "d93b00", "ec4400",
      "993200", "b74200", "d35c0a", "db5f0a", "e96b13",
      "9b4b00", "b2600a", "c67b21", "ce7f23", "dd8c2b",
      "7f3a00", "984e08", "ae681d", "b56b1d", "c47726",
      "501b00", "682c06", "824415", "874716", "95521c",
      "371000", "4f1c04", "6a3110", "6f3210", "7c3c17",
      "220c00", "351604", "4e2710", "512810", "5d3115",
      "080000", "150200", "2a0300", "460600", "570700",
      "121212", "2e2e2e", "606060", "9f9f9f", "c4c4c4",
    ],
  },
  {
    name: "Gamma",
    limited: true,
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "3e1718", "5c2223", "843033", "a73d3f", "bb4549",
      "381d22", "50292f", "783e46", "a05662", "ba6777",
      "1d1015", "361f26", "694048", "a46c76", "c9929d",
      "220e16", "381725", "632840", "8d385e", "ab457a",
      "11131e", "1b1e30", "2d3453", "485183", "5f6ea9",
      "0c1822", "14293c", "234769", "326c95", "3a88af",
      "16313d", "224657", "376c7f", "5095a6", "66b3c0",
      "1c2b28", "29403a", "41635b", "5e8d85", "7ab1a9",
      "15281a", "214029", "376e45", "509b65", "64b981",
      "333411", "56591e", "80822f", "a5a63f", "bcbb4d",
      "534530", "766044", "9c875c", "c0ae78", "d9ca94",
      "6f4429", "835634", "9d7248", "b69361", "cdb17d",
      "89432f", "a0583c", "b97751", "ce9f6d", "d9be8f",
      "2d190f", "552d1c", "915230", "b9763e", "bf8d40",
      "271f1e", "483835", "886c63", "c0a296", "dcc2b5",
      "272624", "5b5954", "bbb8b1", "f4f3ed", "ffffff",
      "202223", "484e4e", "98a2a2", "d8e0e0", "f4faf9",
      "1e1e21", "414349", "8d909c", "cccfda", "eaedf5",
    ],
  },
  {
    name: "Grineer",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "ffa339", "ff982b", "ff8d1e", "ff8312", "ff7b08",
      "ffca6c", "ffc261", "ffb956", "ffae48", "ffa33a",
      "b78427", "b37b23", "ae731f", "aa6c1b", "a86518",
      "c6a136", "c39b34", "c0932f", "bc8c2c", "b78327",
      "7e4226", "733d24", "693823", "613421", "5b3220",
      "aa542c", "a0502a", "954c29", "894627", "7e4126",
      "545d28", "4b5426", "454c24", "3f4622", "3a4121",
      "738030", "6c782e", "646f2d", "5b662a", "535d27",
      "779900", "6a8700", "5e7700", "546b00", "4c6100",
      "9bc700", "90ba00", "85aa00", "779800", "6a8800",
      "21afc4", "1ea2b8", "1c96ae", "188da6", "1786a0",
      "2fe1f1", "2cd6e7", "28c9dc", "25bcd0", "21aec4",
      "418c87", "3c7c75", "386e65", "346156", "31594d",
      "4fb6b7", "4baaa9", "479b99", "418c87", "3c7d75",
      "6b7066", "5f655e", "545b56", "4c5350", "454d4b",
      "9b9b88", "909180", "848778", "777b6f", "6b7066",
      "504941", "444039", "393631", "252521", "1f201d",
      "756a5e", "6d6358", "635a50", "585148", "4d4740",
    ],
  },
  {
    name: "Halloween",
    limited: true,
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "de0000", "c30000", "a70000", "8c0000", "760000",
      "eb0000", "d30000", "b80000", "9c0000", "830000",
      "f70000", "e20000", "c90000", "ad0000", "920000",
      "ff7909", "ff6908", "ff5707", "ff4606", "ff3806",
      "ff8209", "ff7309", "ff6208", "ff5007", "ff4006",
      "ff890a", "ff7c09", "ff6c08", "ff5b07", "ff4a06",
      "ffd300", "ffc100", "ffad00", "ff9c00", "ff8c00",
      "ffdc00", "ffcc00", "ffb900", "ffa600", "ff9500",
      "ffe300", "ffd600", "ffc400", "ffb100", "ffa000",
      "ebff67", "dbfd2b", "c8fb00", "b9fa00", "acf800",
      "f2ff87", "e3ff51", "d5fd11", "c3fa00", "b4fa00",
      "fbffa2", "edff72", "dffd37", "cafb00", "bbfa00",
      "a4d56c", "93c34a", "85b527", "7aa805", "6e9d00",
      "abe17c", "9ccd5f", "8cbd3c", "81af19", "74a300",
      "b3eb8b", "a6da71", "97c751", "8ab92e", "7cab0c",
      "729e61", "618b4e", "51783c", "43682d", "3c5c23",
      "7aab6c", "6d9a5c", "5e8649", "4d7338", "416429",
      "85b777", "78a769", "699557", "598145", "496f33",
    ],
  },
  {
    name: "Hatred",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "b09b8e", "b5a397", "b8a9a0", "bcb2a9", "beb9af",
      "a2846c", "a78d78", "ac9785", "af9d8b", "b1a592",
      "926847", "956e4e", "9a7959", "9a7c59", "9c825f",
      "915c44", "94664b", "987055", "987356", "9b7b5b",
      "904c41", "925747", "96614f", "966450", "976b55",
      "8f443f", "914d45", "955b4e", "955c4e", "966351",
      "823738", "883d39", "8f493f", "914e46", "95594e",
      "672b34", "6f2f36", "793335", "7e3536", "873939",
      "582531", "622933", "6c2e36", "712f36", "783339",
      "55242a", "632a2f", "743134", "783335", "833839",
      "542524", "652e2a", "783e37", "7b4039", "854841",
      "552d24", "653a2f", "764e41", "7a5044", "855a4c",
      "46241d", "563028", "684439", "6a443a", "764d43",
      "2c1413", "3b1f1c", "4d2e2a", "50312c", "593832",
      "1e0d0d", "2d1514", "3f2422", "412523", "4a2b29",
      "120807", "1e0f0e", "2f1d1b", "301d1b", "382321",
      "040203", "0b0405", "17090c", "261014", "301419",
      "0e0e0e", "242424", "4c4c4c", "7e7e7e", "9b9b9b",
    ],
  },
  {
    name: "Ice",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "78ffd4", "9bffe1", "bbffeb", "cefff9", "d5fdff",
      "54d7c3", "74e1d1", "8ce8dc", "9bece2", "a7fff5",
      "07948b", "1eaca5", "32bdb7", "42c7c2", "47d7d2",
      "007474", "019191", "03a6a6", "06b3b3", "06c1c1",
      "005f63", "047e82", "0b9599", "13a4a7", "15b1b4",
      "003b47", "0d5b67", "1b7580", "288690", "2b919c",
      "012a39", "134a59", "246473", "337684", "37808f",
      "012431", "134351", "235c6b", "316e7c", "357786",
      "000c18", "0f2635", "1d3c4d", "2a4d5f", "2d5367",
      "00060f", "0d1b29", "1b2e40", "283e51", "2b4358",
      "001525", "0a3144", "16495d", "225b6f", "256278",
      "0084ae", "029fc1", "07b2ce", "0dbdd6", "0ecce7",
      "00caff", "00d7ff", "00e0ff", "00e5ff", "00f7ff",
      "49b8aa", "69c9be", "82d5cc", "92dcd4", "9eeee5",
      "83b18f", "9ec4a7", "b1d1b9", "bdd8c4", "cce9d4",
      "b1ae7e", "c4c199", "d1cead", "d8d6b9", "e9e7c8",
      "bdc6be", "cdd4ce", "d8ded9", "dee3df", "f0f5f1",
      "c3d7e8", "d2e1ee", "dce8f2", "e2ecf4", "f4ffff",
    ],
  },
  {
    name: "Infested",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "1f8e00", "1b8600", "168000", "137a00", "0f7500",
      "2fa600", "2a9e00", "259600", "1f8e00", "1a8600",
      "41c100", "3cb800", "35af00", "2fa700", "2a9e00",
      "52da00", "4cd200", "46ca00", "41c100", "3bb800",
      "60f000", "5ce900", "57e200", "52da00", "4dd200",
      "68e40a", "7dc426", "999a4b", "b76d74", "d5409a",
      "de02c4", "e702c8", "f201cc", "f801cf", "ff00d2",
      "c904bb", "d503c0", "e002c4", "ea01c9", "f401cd",
      "9c07a7", "a906ae", "b605b3", "c205b9", "cf04bd",
      "86099f", "9208a4", "9f07a9", "ac06af", "b906b5",
      "620c8f", "6b0c93", "750b97", "80099c", "8c09a1",
      "530b88", "480b83", "3d0a7e", "310978", "250872",
      "00057c", "000574", "00066c", "000665", "000660",
      "000499", "00048f", "000585", "00057d", "000674",
      "0003b8", "0003ad", "0004a3", "000498", "00048f",
      "0002d5", "0002cb", "0003c1", "0003b7", "0004ad",
      "0001ed", "0001e6", "0001de", "0002d4", "0002cc",
      "0104ff", "0c1dff", "1a3fff", "2a64ff", "3884ff",
    ],
  },
  {
    name: "Jade",
    platforms: ["xbox"],
    colors: [
      "ffffff", "fcfdfc", "f9faf9", "f4f6f5", "eff2f1",
      "fafbfb", "f6f9f7", "f1f4f3", "ecf0ee", "e6ede9",
      "f4f6f5", "eff2f0", "e9efec", "e4ebe6", "dee6e2",
      "edf1ee", "e7ede9", "e0e8e5", "dbe4df", "d5e0da",
      "e3eae7", "dde6e2", "d8e1dd", "d2ddd8", "ced9d3",
      "dae4df", "d5dfda", "d0dbd6", "cbd8d2", "c7d4cd",
      "42ff45", "3ff942", "3cf13e", "37e839", "32dc34",
      "3ff841", "3bef3e", "36e639", "31db33", "2bcf2e",
      "39ed3c", "34e338", "2fd832", "29cc2c", "24c026",
      "33e137", "2fd631", "29ca2b", "23be25", "1db21f",
      "2dd330", "28c82a", "22bb24", "1caf1e", "17a419",
      "27c428", "21b923", "1cad1d", "15a218", "119813",
      "3a3a38", "373735", "333332", "2e2f2c", "292a28",
      "363635", "333230", "2e2e2c", "282827", "222221",
      "313130", "2c2c2b", "272725", "212120", "1b1b1a",
      "2b2b2a", "262625", "20201f", "1a1a19", "151413",
      "252523", "1e1f1e", "191918", "141313", "0d0d0d",
      "1e1e1c", "181817", "121212", "0d0c0c", "070808",
    ],
  },
  {
    name: "Ki'Teer",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "90ffdc", "81fad6", "6cf3cd", "53ebc2", "38e3b7",
      "88fcd9", "75f6d0", "5deec6", "43e6bb", "29ddb1",
      "7df9d4", "66f2cb", "4de9c0", "33e0b5", "1bd8ac",
      "688f91", "638989", "5e7f7e", "577572", "506964",
      "658c8d", "608383", "5a7977", "526e69", "4c635c",
      "638788", "5c7d7c", "55726f", "4e6762", "485c55",
      "87ba9c", "80af93", "759f87", "698d77", "5b7a68",
      "83b498", "79a68c", "6e957d", "61826e", "546f5f",
      "7dac91", "739c83", "668974", "597665", "4d6557",
      "9f6544", "966041", "87573c", "774e37", "664431",
      "9a6243", "8d5b3f", "7d5239", "6d4834", "5c3e2e",
      "925e40", "84553c", "744c36", "624230", "53392b",
      "00d3e2", "00c5d4", "00b1be", "009aa4", "00818a",
      "00ccda", "00bac7", "00a3af", "008b95", "00727b",
      "00c2cf", "00acb9", "0094a0", "007c85", "00666d",
      "609599", "57898c", "49777a", "396265", "2a4b4e",
      "5b8f92", "507f82", "406a6e", "305457", "213f41",
      "548589", "467376", "365e60", "274749", "183436",
    ],
  },
  {
    name: "Lotus",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "827dd3", "6b66c0", "504caa", "363395", "1e1c82",
      "8882d8", "7570c8", "5c57b3", "413e9e", "272489",
      "7e7c90", "6b6a7e", "565567", "3f3d51", "2c2a3e",
      "838296", "747286", "5f5d70", "49475a", "343246",
      "686683", "56546e", "403e57", "2d2b41", "201e33",
      "706e8c", "5f5d7a", "4b4963", "36344c", "252339",
      "44527e", "3d4b76", "35426b", "2b3860", "243158",
      "465480", "404e79", "384670", "2e3c64", "26345b",
      "3e4661", "373e5d", "2f3559", "282e53", "242951",
      "414963", "3b425f", "343a5b", "2b3256", "262b52",
      "a09dde", "8885be", "6c6a99", "4e4d72", "353551",
      "a7a3e6", "918fcb", "7775a7", "5a5981", "3f3e5d",
      "bf92c3", "aa81b0", "926b99", "7a5583", "654270",
      "c598c9", "b389b8", "9c74a3", "845e8c", "6d4a77",
      "6c79ae", "6370a1", "586591", "4c5880", "434e72",
      "6f7db3", "6775a8", "5c6997", "515d87", "465277",
      "98e0e6", "85c7d4", "6daabe", "558ca8", "417195",
      "9de7eb", "8dd2dc", "76b6c8", "5e97b0", "487b9c",
    ],
  },
  {
    name: "Matisse",
    limited: true,
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "e20074", "de0474", "df0876", "df0d78", "df137c",
      "df107b", "de177e", "e01d82", "e12385", "e12b88",
      "e02887", "e0308b", "e2388f", "e34093", "e34898",
      "e14595", "e34d9b", "e4569f", "e55fa3", "e668a9",
      "e564a6", "e56eac", "e577b0", "e680b5", "e789b9",
      "e686b7", "e78ebd", "e797c1", "e8a0c6", "e9a9ca",
      "eaa5c8", "eaaecc", "eab6d0", "ebbdd5", "ebc4d9",
      "ebc1d6", "ebc8da", "ebcfde", "ecd5e2", "eedbe5",
      "ecd8e2", "eddde5", "eee3e9", "eee7eb", "eeebec",
      "eee8eb", "efeced", "efefee", "efefee", "efefef",
      "efefef", "efefef", "efefee", "efefef", "efedee",
      "efedee", "efecee", "efeaec", "efe7eb", "efe3ea",
      "efe5eb", "eee2e8", "efdde6", "eed9e4", "eed5e2",
      "eed6e1", "eed2e0", "edccdc", "edc7da", "edc1d7",
      "edc3d9", "edbed6", "edb9d4", "ecb2d1", "ebadcd",
      "ecafcf", "eba9cb", "eba4c9", "eb9ec6", "ea97c3",
      "eb9ac4", "eb94c1", "ea8fbe", "ea89bb", "ea84b9",
      "e986ba", "ea81b7", "e97cb4", "e977b2", "e973b0",
    ],
  },
  {
    name: "Opal",
    platforms: ["switch"],
    colors: [
      "ff3c28", "f23926", "df3523", "c83020", "b12a1d",
      "f73a27", "e73725", "d23221", "ba2c1e", "a4271a",
      "ef3826", "db3323", "c32f1f", "ad291b", "982518",
      "0ab9e6", "0aaed9", "089fc6", "078daf", "067b99",
      "0ab1dd", "09a3cb", "0892b5", "067f9e", "056e89",
      "09a7d0", "0897bb", "0784a4", "06738e", "05647c",
      "828282", "7c7c7b", "737373", "686868", "5d5d5e",
      "7f7f7e", "777777", "6d6d6d", "616262", "575757",
      "7a7a7a", "717171", "666666", "5b5b5b", "525152",
      "e6ff00", "d7ef00", "c2d800", "aabc00", "92a200",
      "e1fa00", "d0e700", "b8cd00", "a0b200", "899800",
      "dbf300", "c7dc00", "afc200", "97a700", "818f00",
      "ff3278", "f22f72", "df2c69", "c8285e", "b12353",
      "f83175", "e72d6d", "d22963", "bb2457", "a4204c",
      "ee2f70", "db2b67", "c4275c", "ad2251", "981e47",
      "1edc00", "1cd100", "1ac000", "18ad00", "149900",
      "1ed600", "1bc700", "19b500", "16a100", "138e00",
      "1cce00", "1abd00", "17a900", "159500", "118300",
    ],
  },
  {
    name: "Orokin",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "605d55", "6a675d", "76746a", "858376", "939283",
      "6b685e", "78756b", "878477", "949385", "a19f91",
      "b8a48d", "baa690", "beab94", "c3b09b", "c8b4a1",
      "baa68f", "beab94", "c2b09a", "c7b4a0", "ccb8a5",
      "9a9233", "a69d3c", "b5ac49", "c8be56", "dad066",
      "a59c3a", "b3ab47", "c6bc55", "d8ce64", "eae073",
      "ffd10b", "fdd422", "fbd942", "f8dd67", "f6e38c",
      "fdd31e", "fbd83e", "f9dd62", "f6e288", "f4e7ac",
      "ffb90a", "fdbe1a", "fbc331", "f9c94a", "f7ce65",
      "febd17", "fcc22e", "f9c847", "f7ce62", "f5d37b",
      "eeeeda", "f0f0df", "f2f2e4", "f6f4ec", "f9f9f1",
      "f0f0de", "f2f1e4", "f4f4ea", "f8f8f0", "fbfbf7",
      "54c9db", "4ccedf", "41d2e4", "36d8e9", "2bddef",
      "4fcbdd", "45d0e1", "3ad4e8", "2fdced", "24e1f0",
      "0ab6e5", "17bbe8", "29c4eb", "3ecdef", "54d6f3",
      "16bbe7", "27c3ea", "3bccef", "52d4f2", "66def7",
      "3a85be", "2d9ac5", "36aed2", "2ec2dd", "26d8e6",
      "309eca", "34b1d4", "2dc7de", "16e7ee", "0efff9",
    ],
  },
  {
    name: "Obsidian",
    platforms: ["ps4"],
    colors: [
      "f8a6da", "f492d0", "f27dc6", "ee69bc", "eb58b3",
      "fdc5e9", "fbb7e2", "f7a6da", "f492d0", "f27dc6",
      "a3beef", "91afe7", "7fa0df", "6f92d8", "5f85d0",
      "bcd2fa", "aec7f4", "9cb8ec", "8baae4", "799bdc",
      "fc7873", "fc6964", "fb5a56", "fb4e49", "fa443f",
      "ff938e", "fd8682", "fd7873", "fc6964", "fb5a56",
      "80e2ee", "69dbe6", "51d4dd", "39cdd4", "24c7cd",
      "9febf7", "8ee6f1", "78e0ea", "61d9e2", "4bd2d9",
      "276def", "1e61eb", "1655e6", "0e4ae2", "0841de",
      "2d76f3", "256aee", "1c5eea", "1452e5", "0d48e1",
      "337ef6", "2b74f2", "2368ed", "1b5ce9", "1250e4",
      "3785f8", "317cf5", "2a72f1", "2166ec", "195ae8",
      "e7eaec", "d9dde1", "cad1d6", "bec6cd", "b6bfc6",
      "f1f3f4", "e3e7ea", "d5dade", "c7ced3", "bcc4cb",
      "f9fafa", "eef0f2", "e0e4e7", "d1d7db", "c4cbd1",
      "282a2d", "1c1d20", "111113", "070708", "000000",
      "303236", "25272a", "191a1c", "0e0f10", "050505",
      "36393e", "2d2f33", "222427", "161719", "0b0c0d",
    ],
  },
  {
    name: "Red/White/Blue",
    limited: true,
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "e11115", "feffff", "e11115", "ffffff", "e11115",
      "df1217", "fdfdff", "e01116", "fefeff", "e11115",
      "dc1318", "fafcff", "de1217", "fcfdfe", "df1117",
      "d8141b", "f8fafe", "da1419", "f9fbfe", "db1319",
      "d4161d", "f5f9fe", "d6151c", "f7f9fe", "d9141b",
      "d01721", "f2f6fd", "d2161f", "f5f8fd", "d5161d",
      "cc1924", "eff4fc", "ce1822", "f1f6fd", "cf1721",
      "c71b26", "ecf2fc", "c91924", "eff4fd", "cb1923",
      "c21c2a", "e9f1fc", "c51c28", "ebf2fc", "c71b26",
      "be1e2c", "e6eefc", "c01d2b", "e9f0fc", "c21c29",
      "ba202f", "e3ecfb", "bc1e2d", "e6eefb", "bd1e2d",
      "b72031", "e0ebfa", "b82030", "e3ecfb", "ba1f2f",
      "b42133", "dfe9fb", "b52132", "e0ebfb", "b62032",
      "36428d", "374083", "393e7b", "3b3c75", "3c3b6f",
      "344394", "36418b", "383f83", "393e7a", "3b3c73",
      "32449b", "344392", "36418a", "383f81", "3a3d78",
      "3145a1", "324399", "354291", "364087", "383e7f",
      "3046a6", "3245a0", "334397", "35418f", "374085",
    ],
  },
  {
    name: "Rollers",
    limited: true,
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "f6a2b0", "e597a3", "cc8692", "af737d", "96626b",
      "ffd585", "ebc57b", "cfad6c", "b0925b", "937b4c",
      "ff8078", "eb766f", "cf6961", "b05852", "934a45",
      "b0dcfa", "a3cce8", "91b6ce", "7d9cb1", "698496",
      "ef880e", "de7f0d", "c5700c", "a9610a", "915309",
      "d9cfbf", "c9c0b1", "b3ab9e", "9a9287", "827c73",
      "569115", "4f8513", "447310", "375e0d", "2d4c0b",
      "e8c941", "d6ba3c", "bca234", "9f8a2d", "857325",
      "c12c2c", "b32929", "9e2524", "891f1f", "741b1b",
      "e4ca6f", "d3bb66", "baa45a", "9d8b4d", "847540",
      "124285", "103d7b", "0f366c", "0c2e5c", "0b264c",
      "f2f2f1", "dedfde", "c2c2c1", "a3a3a1", "868685",
      "374766", "33425e", "2d3952", "263046", "20283b",
      "c1a475", "b2976c", "9c855f", "836f50", "6c5c42",
      "854912", "7b4411", "6c3b0e", "5c330d", "4d2a0b",
      "ff831f", "ed7a1d", "d16b19", "b35c16", "974d12",
      "7b5a21", "765823", "705326", "684f28", "614a2b",
      "985d30", "8b562d", "774c2a", "624025", "4e3621",
    ],
  },
  {
    name: "Rot",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "bed1c1", "b7c7ba", "adbbb3", "a3aea9", "939d9c",
      "aaabb8", "a0a3ae", "92969e", "858a8f", "7a8084",
      "bfbfaa", "b9b8a4", "b1b09f", "a9a697", "a09d90",
      "aeaa99", "aca695", "a69f90", "a1988a", "9e9385",
      "bdbf98", "bcbc97", "b6b595", "b1ae93", "aca791",
      "c6c7aa", "c3c3aa", "bcbcab", "b4b4a9", "aeaeaa",
      "727873", "8c908d", "a5a8a6", "bdbebd", "d0d1d0",
      "5f6660", "787d79", "909591", "aaadaa", "c0c2c1",
      "4e5650", "636a64", "7c827d", "969a97", "afb2af",
      "3f4840", "525a53", "696f6a", "828783", "9c9f9c",
      "88966f", "8a9675", "8e977e", "929886", "96998e",
      "829565", "74825c", "636e52", "545b49", "484b41",
      "8b9f6a", "7e8e62", "6e7b59", "5e664e", "4e5345",
      "92a76e", "859867", "77865e", "697355", "565e4a",
      "7e8c64", "768160", "6c745b", "636857", "5a5b52",
      "5d6d5f", "566657", "4c5b4d", "404f42", "354437",
      "859787", "7f9180", "768777", "6c7c6d", "637363",
      "c4dac6", "b6cbb7", "a0b4a2", "8a9c8c", "758777",
    ],
  },
  {
    name: "Shamrock",
    limited: true,
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "ffd393", "fec673", "feb851", "feab31", "fda015",
      "ffdaa3", "fece86", "fec165", "feb344", "fda625",
      "ffdfb1", "ffd598", "fec979", "febb58", "feae37",
      "ff7700", "ff6600", "ff5400", "ff4300", "ff3400",
      "ff8000", "ff7100", "ff5f00", "ff4d00", "ff3d00",
      "ff8800", "ff7a00", "ff6a00", "ff5800", "ff4600",
      "4bef6f", "3bdd5f", "2ac84d", "19b33c", "0ba22c",
      "54fa78", "47eb6b", "37d85b", "26c349", "16af37",
      "58ff7d", "51f776", "43e668", "33d256", "22bd44",
      "03e02f", "05c82a", "07ae25", "09951f", "0b801b",
      "02ed32", "03d72d", "06bd28", "08a323", "0a8c1e",
      "01f734", "02e430", "04cd2b", "06b326", "099920",
      "27aa41", "2b923f", "307a3d", "33663b", "36583a",
      "24bb43", "28a441", "2c8c3f", "31753d", "34623b",
      "21c844", "25b543", "299e40", "2d863e", "326f3c",
      "1b1f75", "151870", "0f116a", "090a64", "03045f",
      "20267a", "1c2076", "161a71", "10136b", "0a0b65",
      "1351b2", "1044a4", "0b3190", "071e7b", "030c69",
    ],
  },
  {
    name: "Smoke Colors",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "dddede", "e0e2e2", "e4e5e5", "eff0f0", "f9f9f9",
      "cbcdcd", "d2d3d3", "d8d9d9", "e4e5e5", "eff0f0",
      "bdc0c0", "c4c7c7", "cbcdcd", "d7d8d8", "e0e2e2",
      "aeb1b2", "b7b9b9", "bec1c1", "c9cccc", "d3d4d4",
      "a3a6a7", "a7aaab", "b1b3b3", "bcbebe", "c3c6c6",
      "9b9d9f", "a0a2a3", "a3a6a7", "adb0b1", "b5b7b7",
      "8c9091", "929596", "96999a", "9fa1a2", "a6a8aa",
      "767a7b", "7d8081", "808485", "898d8f", "929596",
      "656869", "6b6e70", "707475", "787d7e", "808485",
      "4d5154", "565a5b", "5b5f60", "626869", "696d6f",
      "404243", "464a4c", "4c4f51", "545758", "575b5c",
      "2f3132", "373b3c", "3d4041", "414446", "484c4d",
      "1f2222", "282b2c", "2e3031", "333536", "363a3b",
      "0f1111", "171a1a", "1c1f1f", "212324", "242729",
      "070b0b", "131515", "181a1a", "1b1e1e", "1e2020",
      "060707", "0f1111", "121515", "161818", "181b1b",
      "030404", "0b0c0c", "101111", "141515", "151616",
      "000000", "0a0b0b", "0d0f0f", "111212", "121414",
    ],
  },
  {
    name: "Spektaka",
    limited: true,
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "46a3c5", "70aa43", "f8981f", "df4b99", "f3661f",
      "47a3c1", "74aa42", "f79523", "e04c95", "ed6824",
      "4aa3bc", "7aa940", "f69328", "e04d90", "e76a2b",
      "4ba4b6", "81a73f", "f58f2d", "e14e8a", "df6d33",
      "4ea4ae", "88a73d", "f38b35", "e25083", "d5713c",
      "4fa4a7", "90a53b", "f2863b", "e4517c", "cb7445",
      "52a59f", "99a538", "f08143", "e55375", "c07850",
      "55a595", "a1a436", "ef7c4b", "e7556d", "b57c5b",
      "58a68d", "aaa233", "ed7754", "e75665", "a98066",
      "5ba784", "b5a131", "eb725c", "e9585c", "9d8472",
      "5ea77b", "bd9f2f", "ea6c64", "eb5a53", "91887d",
      "61a773", "c79e2c", "e8676d", "ec5c4b", "858d89",
      "64a86a", "d09d2a", "e66275", "ed5e43", "7a9094",
      "66a962", "d89c28", "e55d7c", "ee603b", "6f959e",
      "69a959", "e09b25", "e35883", "ef6234", "6498a8",
      "6baa52", "e79a24", "e2548a", "f1632e", "5b9bb1",
      "6daa4c", "ee9922", "e15190", "f16428", "539fb9",
      "6eaa47", "f49820", "e04e95", "f26623", "4ca1bf",
    ],
  },
  {
    name: "Storm",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "df4019", "e45b39", "e56647", "f86f4d", "ff896a",
      "e66332", "e9784e", "eb835a", "fe8d61", "ffa986",
      "efa55e", "f1b173", "f3b67d", "ffc58f", "ffd9b3",
      "f5c774", "f6cf87", "f7d390", "ffe1a7", "fff1c4",
      "deb76f", "e2c183", "e4c68c", "f6d697", "ffe7b5",
      "978b63", "a69c78", "aca282", "bab08c", "cdc59f",
      "5d6755", "747c6d", "7d8477", "878e81", "9ba394",
      "424a45", "5c635e", "676e69", "6f7772", "828a85",
      "121c21", "323a3f", "40484d", "454f54", "556065",
      "00020a", "1b2530", "2b333f", "2f3744", "3c4453",
      "100007", "351d28", "442d37", "49313c", "593f4c",
      "380001", "532124", "5f3133", "673537", "794345",
      "4d0000", "652020", "703030", "793434", "8c4242",
      "4f0a01", "672c24", "723b33", "7b4037", "8e4f45",
      "674800", "84651a", "8d702b", "98772e", "ad8c3c",
      "5d5c14", "737234", "7d7c42", "878647", "9a9957",
      "ae8b00", "b99b17", "bea227", "cdaf2a", "e0c236",
      "ffaa00", "ffb600", "ffc90c", "ffd90d", "ffeb13",
    ],
  },
  {
    name: "Tenno",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "dccc9a", "83684f", "919a9c", "01b36c", "3f7bc1",
      "6c6c6c", "567b8d", "d0d0d0", "f8cf6f", "064661",
      "bc9e65", "bababa", "7d7b78", "f7780f", "ff7201",
      "b3b3b3", "535353", "686868", "9a2b23", "0158ff",
      "ffffff", "457d9e", "5d6c6f", "c4b388", "0234b7",
      "b0bdb8", "7d6f65", "277883", "b54542", "ffffff",
      "617b91", "7bbdd7", "6f4f3b", "2efff1", "48dae1",
      "8398b0", "4f5461", "787880", "68373c", "ff8a01",
      "e9e4da", "7d8894", "a06f50", "8a8580", "014bbf",
      "808080", "41423e", "54bcb0", "31b891", "0158ff",
      "ece4da", "42534e", "1f2d3c", "e9fb10", "01b8ff",
      "9e8f65", "4b6575", "eacd75", "bababa", "0061f2",
      "728a9c", "e8e9d2", "524f4b", "aa8d9c", "460dae",
      "fa4d40", "948f8d", "65615d", "4085ca", "005dfd",
      "4e4947", "ec7546", "0cbdb0", "f7e8b7", "fd682e",
      "788588", "4e4b46", "c66550", "f8e4c2", "280dae",
      "41675c", "916f4f", "5d6153", "c18a3d", "f40000",
      "5d6572", "f6f2e4", "6cb3f7", "e87d96", "80b7e2",
    ],
  },
  {
    name: "Tenno II",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "94bcc1", "857572", "3f555d", "edeabc", "00f3ff",
      "545d83", "cdbcd5", "c16175", "fffad5", "ff2232",
      "2b2b2b", "c6c1b7", "2d6c83", "857865", "00ffed",
      "4b6172", "ccc1b0", "4c4536", "ffc600", "ffe900",
      "c65009", "4f3e32", "8a9191", "e9e2d5", "ff8300",
      "2d2c2c", "d7d7d7", "572323", "429ccd", "01e8ff",
      "884837", "6f6c78", "dde5e5", "b59c7b", "ff4800",
      "686154", "4b80b8", "d74c1c", "fafafa", "00dcff",
      "6c615d", "b75d4b", "c1b178", "530d06", "00ffff",
      "4a4a4c", "5a78a6", "e5e5e5", "e8511c", "01e8ff",
      "ddca9a", "948065", "4e85ae", "ac553e", "ffbf00",
      "80d9d2", "944347", "414248", "e6dede", "14fff9",
      "a09a9a", "726158", "37373d", "6f3156", "ac0d50",
      "eaf7fb", "313f45", "aa4c9e", "00c2ff", "fb3972",
      "5d2e3e", "544a48", "dcd6b7", "757b7d", "7bcab7",
      "cc7b46", "4a4a4a", "fffbfb", "cfa67d", "5d8ab7",
      "181e23", "968072", "462b35", "c9c9c9", "0094ff",
      "b1aca0", "535755", "652f2f", "6c6f6f", "72ffe5",
    ],
  },
  {
    name: "Transmission",
    limited: true,
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "00285f", "002c63", "003169", "00376f", "003e75",
      "003970", "004077", "00467f", "004e87", "00558f",
      "005189", "005892", "00609a", "0068a2", "0071ab",
      "006ba6", "0073ae", "007bb7", "0084bf", "008cc8",
      "0087c2", "008fca", "0097d2", "009fda", "00a5e0",
      "00a1dc", "00a8e2", "00aee8", "00b5ee", "00baf2",
      "00b7ef", "00bcf3", "06c1f7", "0ec6fa", "18cbfd",
      "11c8fb", "1ccefe", "28d2ff", "35d8ff", "42ddff",
      "38daff", "47dfff", "55e4ff", "64e8ff", "74edff",
      "69eaff", "79efff", "88f3ff", "98f6ff", "a7faff",
      "9cf8ff", "acfbff", "b9feff", "c7ffff", "d4ffff",
      "cbffff", "d7ffff", "e3ffff", "ecffff", "f4ffff",
      "efffff", "f6ffff", "fcffff", "fffeff", "fffbfd",
      "fffefe", "fffafd", "fff5fa", "feeff8", "f9e7f5",
      "fcecf7", "f8e5f4", "f2dcf1", "ebd3ee", "e4caea",
      "e9d0ed", "e1c6e8", "d9bce5", "d0b1e0", "c6a7dd",
      "cdaddf", "c3a3dc", "b998d8", "af8dd4", "a583d0",
      "ac8ad3", "a280cf", "9976cc", "906dc7", "8864c4",
    ],
  },
  {
    name: "Twilight",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "784712", "683b13", "562f13", "462314", "3b1b14",
      "a48b76", "846d5b", "614a3c", "412c21", "2b170e",
      "dccf7f", "c0ad6c", "9f8757", "836544", "6f4e37",
      "e7dbbb", "d1b589", "ba8b53", "a46521", "964b00",
      "d2b48c", "c6a479", "b38c5e", "9f713f", "8c5924",
      "f5deb3", "e0c99c", "c2aa7b", "9f8856", "816935",
      "c2b280", "b1a775", "999766", "7e8554", "667545",
      "e0a65d", "d78f4f", "ca703b", "bc4e26", "b03014",
      "cc7722", "c36522", "b54a21", "a72d21", "991220",
      "b791f8", "a77ee9", "9061d4", "7641bc", "5f24a7",
      "cd9fe3", "bd94dc", "a585d1", "8a74c5", "7265ba",
      "bb70e1", "a060c0", "814e9b", "663d7a", "533263",
      "b5aff8", "a198de", "8579bb", "685795", "4d3974",
      "aa5de3", "9054ca", "744aae", "5a4095", "483a84",
      "7e0ac7", "700ec1", "5c14b7", "461aad", "3120a3",
      "5f0ac7", "510db7", "3b129f", "241784", "0e1c6c",
      "9e16e1", "8610c0", "6b0a9b", "53047a", "430064",
      "b046e3", "8e34bb", "6a1f8e", "480d66", "32004a",
    ],
  },
  {
    name: "Twitch",
    limited: true,
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "f7f6fb", "f9f8fc", "fafafd", "fcfcfd", "fefefe",
      "f1eef7", "f3f0f8", "f5f3fa", "f7f5fb", "f9f7fb",
      "e9e4f2", "ece7f4", "eeeaf5", "f0ecf6", "f3eff8",
      "e1daed", "e3ddef", "e6e1f1", "e9e3f2", "ebe7f3",
      "d8cfe8", "dbd2ea", "ddd5ec", "dfd9ec", "e2dcef",
      "cec3e2", "d1c6e5", "d4cae6", "d7cee7", "dad1e9",
      "c4b7dd", "c7badf", "cabee0", "cdc1e2", "d0c5e3",
      "baa9d6", "bdaed8", "bfb1da", "c3b5dc", "c6b9dd",
      "af9dd0", "b2a1d2", "b5a4d4", "b8a8d6", "bbacd8",
      "a490cb", "a794cc", "ab97ce", "ae9bcf", "b1a0d2",
      "9a84c4", "9e87c7", "a08bc8", "a48fca", "a793cb",
      "9178bf", "937bc1", "967ec2", "9a82c4", "9c86c6",
      "876cba", "8a6fbb", "8c73bc", "8f76be", "9279bf",
      "7e61b4", "8064b6", "8368b7", "866ab9", "896ebb",
      "7657af", "785ab1", "7a5db2", "7d5fb4", "8063b5",
      "6f4eab", "7151ad", "7353ad", "7556af", "7759b0",
      "6947a8", "6b49a9", "6c4baa", "6e4dab", "704fac",
      "6441a5", "6542a6", "6745a6", "6847a8", "6a49a8",
    ],
  },
  {
    name: "Undying",
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "598075", "598075", "5f8379", "688a80", "729088",
      "598075", "5b8277", "63877c", "6d8d84", "78938b",
      "65a6ac", "62a3a9", "5d9fa5", "589aa0", "52969c",
      "64a5ab", "60a1a8", "5b9da3", "56989f", "50959b",
      "9e7855", "9b7656", "957556", "8f7358", "8a705a",
      "9d7855", "987656", "937458", "8e7159", "88705a",
      "786f54", "7c7357", "82795b", "897f60", "918665",
      "787878", "727374", "686c6e", "5e6467", "545b60",
      "767777", "6e7171", "64686c", "5a6065", "50585e",
      "737475", "6b6d6f", "606569", "565d62", "4c555c",
      "5a6c67", "5a6c67", "5f706b", "667671", "6f7d79",
      "5a6c67", "5c6e69", "62736e", "6b7a76", "74807c",
      "6c8a8d", "69878a", "658487", "607f82", "5b7b7e",
      "6b898c", "68868a", "638285", "5f7e81", "5a7a7d",
      "806e5e", "7e6d5e", "7a6b5d", "76695c", "73665c",
      "7f6e5e", "7c6c5d", "796a5d", "76685d", "71665c",
      "666155", "686457", "6e6a5c", "746f61", "7a7566",
      "676256", "6a6659", "716c5e", "767163", "7b7667",
    ],
  },
  {
    name: "Valentine",
    limited: true,
    platforms: ["pc", "ps4", "switch", "xbox"],
    colors: [
      "760303", "8b1717", "a22a2a", "b03b3b", "ca4545",
      "a50707", "b51c1c", "c53030", "cf4242", "ec4d4d",
      "bb0812", "c71e2a", "d43342", "db4555", "f95164",
      "e60c2f", "eb2349", "f03964", "f34b77", "ff588a",
      "ff0d43", "ff245c", "ff3b77", "ff4e89", "ff5c9f",
      "fe1c50", "fe3468", "fe4d82", "fe6093", "ff70a9",
      "f46388", "f67a9c", "f893b0", "f9a3bd", "ffbcd9",
      "f08ba2", "f39db2", "f6b1c3", "f8becd", "ffdaea",
      "e1768c", "e78b9e", "eda2b2", "f0b0be", "ffcada",
      "ae363b", "bc4f54", "cb6a6f", "d47d82", "f29197",
      "9a1e1e", "aa3737", "bc5151", "c76464", "e37474",
      "9a2b2b", "aa4444", "bc5f5f", "c77272", "e38484",
      "9a4647", "aa5f60", "bc7a7b", "c78c8d", "e3a2a3",
      "9a5757", "aa6f6f", "bc8989", "c79a9a", "e3b1b1",
      "7f4445", "935d5e", "a97879", "b78a8b", "d2a0a1",
      "290b0b", "422121", "5d3737", "704949", "825656",
      "0f0000", "260c0c", "3d1a1a", "502828", "5e2f2f",
      "1f0000", "380e0f", "521d1e", "652c2d", "753334",
    ],
  },
];

/**
 * 色板数据
 */
export const PaletteData = _paletteData.map(v => new Palette({ id: camelCase(v.name), limited: v.limited || false, platforms: v.platforms, colors: v.colors }));


/**
 * 色板颜色信息
 */
export interface PaletteColorInfo {
  id: string;
  index: number;
  color: Color;
}

export interface MatchedPaletteColorInfo extends PaletteColorInfo {
  deltaE: number;
}

/**
 * 颜色索引
 */
export class ColorHelper {
  public static instance = new ColorHelper();
  private _colorMap = new Map<string, PaletteColorInfo[]>();
  private _palette = new Map<string, Palette>();
  private _colors: Color[] = [];
  constructor() {
    PaletteData.forEach(v => {
      this._palette.set(v.id, v);
      v.colors.forEach((c, index) => {
        let ck = c.toString();
        if (this._colorMap.has(ck))
          this._colorMap.get(ck).push({ id: v.id, index, color: c });
        else {
          this._colorMap.set(ck, [{ id: v.id, index, color: c }]);
          this._colors.push(c);
        }
      });
    });
  }
  /**
   * 搜索颜色
   *
   * @param {(string | Color)} color 要搜索的颜色
   * @param {number} [deltaE=50] 容差
   * @memberof ColorHelper
   */
  static searchColor(color: string | Color, deltaE: number = 50) {
    let cc = typeof color === "string" ? new Color(color) : color;
    return this.instance._colors.filter(v => cc.deltaE(v) < deltaE)
      .sort((a, b) => cc.deltaE(a) - cc.deltaE(b))
      .map(v => this.instance._colorMap.get(v.toString())
        .map(m => ({ ...m, deltaE: cc.deltaE(v) } as MatchedPaletteColorInfo))
      )
      .reduce((a, b) => a.concat(b), []);
  }
  /**
   * 搜索包含指定颜色的色板
   *
   * @param {(string | Color)} color 要搜索的颜色
   * @param {number} [deltaE=50] 容差
   * @memberof ColorHelper
   */
  static searchPalette(color: string | Color, deltaE: number = 50) {
    let colors = this.searchColor(color, deltaE);
    let ps = groupBy(colors, v => v.id);
    return map(ps, (info, id) => new MatchedPalette(this.getPalette(id), info));
  }

  /**
   * 获取色板信息
   *
   * @param {string} id
   * @memberof ColorHelper
   */
  static getPalette(id: string) { return this.instance._palette.get(id); }
}
