<?php
/**
 * OrangeHRM is a comprehensive Human Resource Management (HRM) System that captures
 * all the essential functionalities required for any enterprise.
 * Copyright (C) 2006 OrangeHRM Inc., http://www.orangehrm.com
 *
 * OrangeHRM is free software: you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation, either
 * version 3 of the License, or (at your option) any later version.
 *
 * OrangeHRM is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with OrangeHRM.
 * If not, see <https://www.gnu.org/licenses/>.
 */

namespace OrangeHRM\Tests\CorporateBranding\Api\ValidationRules;

use OrangeHRM\CorporateBranding\Api\ValidationRules\ImageAspectRatio;
use OrangeHRM\Tests\Util\TestCase;

class ImageAspectRatioTest extends TestCase
{
    /**
     * @dataProvider dataProvider
     */
    public function testValidate(
        float $aspectRatio,
        ?string $fileName,
        ?string $fileType,
        ?string $base64FileContent,
        bool $valid
    ): void {
        $imageAspectRatioRule = new ImageAspectRatio($aspectRatio);
        $assertion = $valid ? 'assertTrue' : 'assertFalse';
        $this->$assertion(
            $imageAspectRatioRule->validate([
                'name' => $fileName,
                'type' => $fileType,
                'base64' => $base64FileContent,
            ])
        );
    }

    public function dataProvider(): array
    {
        return [
            [
                4 / 3, // 4:3
                'logo.svg',
                'image/svg+xml',
                '',
                true,
            ],
            [
                4 / 3, // 4:3
                null,
                'image/svg+xml',
                null,
                false,
            ],
            [
                4 / 3, // 4:3
                'logo.png',
                'image/png',
                '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',
                false, // 1:1 (400 x 400) image
            ],
            [
                1, // 1:1
                'logo.png',
                'image/png',
                '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',
                true, // 1:1 (400 x 400) image
            ],
            [
                16 / 9, // 16:9
                'logo.jpg',
                'image/jpeg',
                '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',
                true, // 16:9 (1920 x 1080) image
            ],
            [
                4 / 3, // 4:3
                'logo.jpg',
                'image/jpeg',
                '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',
                false, // 16:9 (1920 x 1080) image
            ],
        ];
    }
}
