<?php
/**
 * OrangeHRM is a comprehensive Human Resource Management (HRM) System that captures
 * all the essential functionalities required for any enterprise.
 * Copyright (C) 2006 OrangeHRM Inc., http://www.orangehrm.com
 *
 * OrangeHRM is free software: you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation, either
 * version 3 of the License, or (at your option) any later version.
 *
 * OrangeHRM is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with OrangeHRM.
 * If not, see <https://www.gnu.org/licenses/>.
 */

namespace OrangeHRM\Tests\Claim\Dao;

use DoctrineProxies\__CG__\OrangeHRM\Entity\User;
use OrangeHRM\Claim\Dao\ClaimDao;
use OrangeHRM\Claim\Dto\ClaimAttachmentSearchFilterParams;
use OrangeHRM\Config\Config;
use OrangeHRM\Core\Traits\Auth\AuthUserTrait;
use OrangeHRM\Core\Traits\ORM\EntityManagerHelperTrait;
use OrangeHRM\Core\Traits\Service\DateTimeHelperTrait;
use OrangeHRM\Entity\ClaimAttachment;
use OrangeHRM\Tests\Util\KernelTestCase;
use OrangeHRM\Tests\Util\TestDataService;

class ClaimDaoAttachmentTest extends KernelTestCase
{
    use AuthUserTrait;
    use EntityManagerHelperTrait;
    use DateTimeHelperTrait;

    private ClaimDao $claimDao;

    public function testGetPartialClaimAttachment()
    {
        $claimAttachment = $this->claimDao->getPartialClaimAttachment(1, 1);
        $this->assertEquals(1, $claimAttachment->getAttachId());
        $this->assertEquals("travel.jpeg", $claimAttachment->getFileName());
    }

    public function testGetClaimAttachmentCount()
    {
        $claimAttachmentSearchFilterParams = new ClaimAttachmentSearchFilterParams();
        $claimAttachmentSearchFilterParams->setRequestId(1);
        $count = $this->claimDao->getClaimAttachmentCount($claimAttachmentSearchFilterParams);
        $this->assertEquals(3, $count);
    }

    public function testGetClaimAttachmentList()
    {
        $claimAttachmentSearchFilterParams = new ClaimAttachmentSearchFilterParams();
        $claimAttachmentSearchFilterParams->setRequestId(1);
        $claimAttachmentList = $this->claimDao->getClaimAttachmentList($claimAttachmentSearchFilterParams);
        $this->assertEquals(3, count($claimAttachmentList));
        $this->assertEquals("travel.jpeg", $claimAttachmentList[0]->getFileName());
    }

    public function testGetNextAttachmentId()
    {
        $nextId = $this->claimDao->getNextAttachmentId(1);
        $this->assertEquals(4, $nextId);
    }

    public function testSaveClaimAttachment()
    {
        $claimAttachment = new ClaimAttachment();
        $claimAttachment->setFileName("test.jpeg");
        $claimAttachment->setRequestId(1);
        $claimAttachment->setAttachId(4);
        $claimAttachment->setSize(1234);
        $claimAttachment->setFileType("image/jpeg");
        $claimAttachment->setDescription("test");
        $claimAttachment->setUser($this->getReference(User::class, 1));
        $claimAttachment->setAttachment(
            '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'
        );
        $this->claimDao->saveClaimAttachment($claimAttachment);
        $result = $this->claimDao->getPartialClaimAttachment(1, 4);
        $this->assertEquals(4, $result->getAttachId());
        $this->assertEquals("test.jpeg", $result->getFileName());
    }

    public function testDeleteClaimAttachments()
    {
        $this->claimDao->deleteClaimAttachments(1, [1, 2]);
        $result = $this->claimDao->getPartialClaimAttachment(1, 1);
        $this->assertEquals(null, $result);
    }

    protected function setUp(): void
    {
        $this->claimDao = new ClaimDao();
        $attachmentFixture = Config::get(
            Config::PLUGINS_DIR
        ) . '/orangehrmClaimPlugin/test/fixtures/ClaimAttachment.yaml';
        TestDataService::populate($attachmentFixture);
    }

    public function testGetClaimAttachmentFile(): void
    {
        $claimAttachment = $this->claimDao->getClaimAttachmentFile(1, 1);
        $this->assertEquals('fuel bill', $claimAttachment->getDescription());
        $this->assertEquals('travel.jpeg', $claimAttachment->getFilename());
        $this->assertEquals('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', $claimAttachment->getDecorator()->getAttachment());

        $claimAttachment = $this->claimDao->getClaimAttachmentFile(1, 100);
        $this->assertNull($claimAttachment);
    }
}
