module Main exposing (main)

import Browser
import Browser.Events
import Html
import Html.Attributes
import Json.Decode
import Keyboard.Event
import Keyboard.Key
import Svg
import Svg.Attributes
import Time


worldSizeX : Int
worldSizeX =
    16


worldSizeY : Int
worldSizeY =
    12


type Event
    = KeyDown Keyboard.Event.KeyboardEvent
    | Tick


type SnakeDirection
    = Up
    | Right
    | Down
    | Left


type alias GameState =
    { snake : Snake
    , appleLocation : Location
    }


type alias Location =
    { x : Int, y : Int }


type alias Snake =
    { headDirection : SnakeDirection
    , segments : List Location
    }


initialState : GameState
initialState =
    { snake = { headDirection = Right, segments = [ { x = 4, y = 5 }, { x = 3, y = 5 } ] }
    , appleLocation = { x = 3, y = 2 }
    }


snakeDirectionFromKeyboardKey : List ( Keyboard.Key.Key, SnakeDirection )
snakeDirectionFromKeyboardKey =
    [ ( Keyboard.Key.W, Up )
    , ( Keyboard.Key.A, Left )
    , ( Keyboard.Key.S, Down )
    , ( Keyboard.Key.D, Right )
    , ( Keyboard.Key.Up, Up )
    , ( Keyboard.Key.Down, Down )
    , ( Keyboard.Key.Left, Left )
    , ( Keyboard.Key.Right, Right )
    ]


onKeyDown : Keyboard.Event.KeyboardEvent -> GameState -> GameState
onKeyDown keyboardEvent gameStateBefore =
    case snakeDirectionFromKeyboardKey |> listDictGet keyboardEvent.keyCode of
        Nothing ->
            gameStateBefore

        Just snakeDirection ->
            let
                snakeBefore =
                    gameStateBefore.snake
            in
            { gameStateBefore | snake = { snakeBefore | headDirection = snakeDirection } }


xyOffsetFromDirection : SnakeDirection -> { x : Int, y : Int }
xyOffsetFromDirection direction =
    case direction of
        Up ->
            { x = 0, y = -1 }

        Down ->
            { x = 0, y = 1 }

        Left ->
            { x = -1, y = 0 }

        Right ->
            { x = 1, y = 0 }


moveSnakeForwardOneStep : GameState -> GameState
moveSnakeForwardOneStep gameStateBefore =
    let
        snakeBefore =
            gameStateBefore.snake
    in
    case snakeBefore.segments |> List.head of
        Nothing ->
            gameStateBefore

        Just headLocationBefore ->
            let
                tailBefore =
                    snakeBefore.segments |> List.tail |> Maybe.withDefault []

                headMovement =
                    xyOffsetFromDirection snakeBefore.headDirection

                headLocationBeforeWrapping =
                    { x = headLocationBefore.x + headMovement.x
                    , y = headLocationBefore.y + headMovement.y
                    }

                headLocation =
                    { x = (headLocationBeforeWrapping.x + worldSizeX) |> modBy worldSizeX
                    , y = (headLocationBeforeWrapping.y + worldSizeY) |> modBy worldSizeY
                    }

                snakeEats =
                    headLocation == gameStateBefore.appleLocation

                snakeTailBeforeGrowth =
                    if 0 < (tailBefore |> List.length) then
                        [ headLocationBefore ] ++ (tailBefore |> List.reverse |> List.drop 1 |> List.reverse)

                    else
                        []

                snakeTailGrowth =
                    if snakeEats then
                        [ tailBefore |> List.reverse |> List.head |> Maybe.withDefault headLocationBefore ]

                    else
                        []

                snakeTail =
                    snakeTailBeforeGrowth ++ snakeTailGrowth

                snakeSegments =
                    [ headLocation ] ++ snakeTail

                appleLocation =
                    if not snakeEats then
                        gameStateBefore.appleLocation

                    else
                        let
                            cellsLocationsWithoutSnake =
                                List.range 0 (worldSizeX - 1)
                                    |> List.concatMap
                                        (\x ->
                                            List.range 0 (worldSizeY - 1)
                                                |> List.map (\y -> { x = x, y = y })
                                        )
                                    |> listRemoveSet snakeSegments
                        in
                        cellsLocationsWithoutSnake
                            |> List.drop (15485863 |> modBy ((cellsLocationsWithoutSnake |> List.length) - 1))
                            |> List.head
                            |> Maybe.withDefault { x = -1, y = -1 }
            in
            { gameStateBefore
                | snake = { snakeBefore | segments = snakeSegments }
                , appleLocation = appleLocation
            }


view : GameState -> Html.Html Event
view gameState =
    let
        cellSideLength =
            30

        svgRectAtCellLocation fill cellLocation =
            svgRectFrom_Fill_UpperLeft_Width_Height
                fill
                { x = cellLocation.x * cellSideLength + 1, y = cellLocation.y * cellSideLength + 1 }
                (cellSideLength - 2)
                (cellSideLength - 2)

        snakeView =
            gameState.snake.segments
                |> List.map (svgRectAtCellLocation "whitesmoke")
                |> Svg.g []

        appleView =
            svgRectAtCellLocation "red" gameState.appleLocation
    in
    Svg.svg
        [ Svg.Attributes.width (worldSizeX * cellSideLength |> String.fromInt)
        , Svg.Attributes.height (worldSizeY * cellSideLength |> String.fromInt)
        , Html.Attributes.style "background" "black"
        ]
        [ snakeView, appleView ]



{-
   The `update` and `subscriptions` functions look like they would be at least partially generated using conventions for special function names.
   When the user defines a function named `onKeyDown`, generate a value in the `Event` type, a subscription and the code in the `update` function.
-}


subscriptions : GameState -> Sub Event
subscriptions model =
    [ Browser.Events.onKeyDown (Keyboard.Event.decodeKeyboardEvent |> Json.Decode.map KeyDown)
    , Time.every 125 (always Tick)
    ]
        |> Sub.batch


update : Event -> GameState -> GameState
update event gameStateBefore =
    case event of
        KeyDown keyboardEvent ->
            onKeyDown keyboardEvent gameStateBefore

        Tick ->
            gameStateBefore |> moveSnakeForwardOneStep



-- In an environment for beginners, the `main` function might be included already, depending on the student using recommended names for the functions integrated here.


main : Program () GameState Event
main =
    Browser.element
        { init = always ( initialState, Cmd.none )
        , view = view
        , update = \msg model -> ( update msg model, Cmd.none )
        , subscriptions = subscriptions
        }



-- Below are parts which seem common enough to be included with a library.


listRemoveSet : List a -> List a -> List a
listRemoveSet elementsToRemove =
    List.filter (\element -> elementsToRemove |> List.member element |> not)


listDictGet : key -> List ( key, value ) -> Maybe value
listDictGet key =
    List.filterMap
        (\( candidateKey, candidateValue ) ->
            if key == candidateKey then
                Just candidateValue

            else
                Nothing
        )
        >> List.head


svgRectFrom_Fill_UpperLeft_Width_Height : String -> Location -> Int -> Int -> Svg.Svg a
svgRectFrom_Fill_UpperLeft_Width_Height fill upperLeftCorner width height =
    Svg.rect
        [ Svg.Attributes.fill fill
        , Svg.Attributes.x (upperLeftCorner.x |> String.fromInt)
        , Svg.Attributes.y (upperLeftCorner.y |> String.fromInt)
        , Svg.Attributes.width (width |> String.fromInt)
        , Svg.Attributes.height (height |> String.fromInt)
        ]
        []
