<?php
/**
 * sysPass
 *
 * @author    nuxsmin
 * @link      https://syspass.org
 * @copyright 2012-2018, Rubén Domínguez nuxsmin@$syspass.org
 *
 * This file is part of sysPass.
 *
 * sysPass is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * sysPass is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 *  along with sysPass.  If not, see <http://www.gnu.org/licenses/>.
 */

namespace SP\Tests\Services\Account;

use DI\DependencyException;
use DI\NotFoundException;
use SP\Core\Context\ContextException;
use SP\Core\Exceptions\ConstraintException;
use SP\Core\Exceptions\InvalidImageException;
use SP\Core\Exceptions\QueryException;
use SP\Core\Exceptions\SPException;
use SP\DataModel\FileData;
use SP\DataModel\FileExtData;
use SP\DataModel\ItemSearchData;
use SP\Repositories\NoSuchItemException;
use SP\Services\Account\AccountFileService;
use SP\Services\ServiceException;
use SP\Storage\Database\DatabaseConnectionData;
use SP\Tests\DatabaseTestCase;
use function SP\Tests\setupContext;

/**
 * Class AccountFileServiceTest
 *
 * @package SP\Tests\Services
 */
class AccountFileServiceTest extends DatabaseTestCase
{
    /**
     * @var AccountFileService
     */
    private static $service;

    /**
     * @throws NotFoundException
     * @throws ContextException
     * @throws DependencyException
     */
    public static function setUpBeforeClass()
    {
        $dic = setupContext();

        self::$dataset = 'syspass_accountFile.xml';

        // Datos de conexión a la BBDD
        self::$databaseConnectionData = $dic->get(DatabaseConnectionData::class);

        // Inicializar el servicio
        self::$service = $dic->get(AccountFileService::class);
    }

    /**
     * @throws ConstraintException
     * @throws QueryException
     * @throws SPException
     */
    public function testCreate()
    {
        $file = RESOURCE_DIR . DIRECTORY_SEPARATOR . 'imgs' . DIRECTORY_SEPARATOR . 'add.png';
        $image = file_get_contents($file);

        $data = new FileData();
        $data->setAccountId(2);
        $data->setName('app.png');
        $data->setType('image/png');
        $data->setExtension('PNG');
        $data->setContent($image);
        $data->setSize(filesize($file));

        $this->assertEquals(5, self::$service->create($data));

        $resultData = self::$service->getById(5);

        $this->assertInstanceOf(FileExtData::class, $resultData);
        $this->assertEquals($data->getName(), $resultData->getName());
        $this->assertEquals($data->getType(), $resultData->getType());
        $this->assertEquals($data->getExtension(), $resultData->getExtension());
        $this->assertEquals($data->getSize(), $resultData->getSize());
        $this->assertEquals($data->getAccountId(), $resultData->getAccountId());
        $this->assertEquals($data->getContent(), $resultData->getContent());
        $this->assertEquals($data->getThumb(), $resultData->getThumb());

        $data = new FileData();
        $data->setAccountId(2);
        $data->setName('aaa.txt');
        $data->setType('text/plain');
        $data->setExtension('txt');
        $data->setContent('');
        $data->setSize(0);

        $this->assertEquals(6, self::$service->create($data));

        $resultData = self::$service->getById(6);

        $this->assertInstanceOf(FileExtData::class, $resultData);
        $this->assertEquals($data->getName(), $resultData->getName());
        $this->assertEquals($data->getType(), $resultData->getType());
        $this->assertEquals($data->getExtension(), $resultData->getExtension());
        $this->assertEquals($data->getSize(), $resultData->getSize());
        $this->assertEquals($data->getAccountId(), $resultData->getAccountId());
        $this->assertEquals($data->getContent(), $resultData->getContent());
        $this->assertEquals('no_thumb', $resultData->getThumb());

        $this->assertEquals(5, $this->conn->getRowCount('AccountFile'));
    }

    /**
     * @throws ConstraintException
     * @throws QueryException
     * @throws SPException
     */
    public function testCreateInvalid()
    {
        $this->expectException(InvalidImageException::class);

        $data = new FileData();
        $data->setAccountId(2);
        $data->setName('app.png');
        $data->setType('image/png');
        $data->setExtension('PNG');
        $data->setContent('');
        $data->setSize(0);

        $this->assertEquals(6, self::$service->create($data));
    }

    /**
     * @throws ConstraintException
     * @throws QueryException
     */
    public function testSearch()
    {
        $itemSearchData = new ItemSearchData();
        $itemSearchData->setLimitCount(10);
        $itemSearchData->setSeachString('android');

        $result = self::$service->search($itemSearchData);
        /** @var FileExtData[] $data */
        $data = $result->getDataAsArray();

        $this->assertEquals(1, $result->getNumRows());
        $this->assertInstanceOf(FileExtData::class, $data[0]);
        $this->assertEquals('android.png', $data[0]->getName());
        $this->assertEquals('image/png', $data[0]->getType());
        $this->assertEquals('PNG', $data[0]->getExtension());
        $this->assertEquals('Google', $data[0]->getAccountName());
        $this->assertEquals('Google', $data[0]->getClientName());
        $this->assertEquals(4295, $data[0]->getSize());
        $this->assertEquals(1, $data[0]->getAccountId());

        $itemSearchData = new ItemSearchData();
        $itemSearchData->setLimitCount(10);
        $itemSearchData->setSeachString('');

        $result = self::$service->search($itemSearchData);
        $this->assertEquals(3, $result->getNumRows());
    }

    /**
     * @throws ConstraintException
     * @throws QueryException
     */
    public function testGetById()
    {
        $data = self::$service->getById(1);

        $this->assertInstanceOf(FileExtData::class, $data);
        $this->assertEquals('sysPass.xml', $data->getName());
        $this->assertEquals('text/xml', $data->getType());
        $this->assertEquals('XML', $data->getExtension());
        $this->assertEquals('Google', $data->getAccountName());
        $this->assertEquals('Google', $data->getClientName());
        $this->assertEquals(1312, $data->getSize());
        $this->assertEquals(1, $data->getAccountId());
        $this->assertEquals(pack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data->getContent());
        $this->assertEquals(pack('H*', '6E6F5F7468756D62'), $data->getThumb());

        $data = self::$service->getById(4);

        $this->assertInstanceOf(FileExtData::class, $data);
        $this->assertEquals('android.png', $data->getName());
        $this->assertEquals('image/png', $data->getType());
        $this->assertEquals('PNG', $data->getExtension());
        $this->assertEquals('Google', $data->getAccountName());
        $this->assertEquals('Google', $data->getClientName());
        $this->assertEquals(4295, $data->getSize());
        $this->assertEquals(1, $data->getAccountId());
        $this->assertEquals(pack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data->getContent());
        $this->assertEquals(pack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data->getThumb());

        $this->assertNull(self::$service->getById(10));
    }

    /**
     * @throws ConstraintException
     * @throws QueryException
     */
    public function testGetByIdBatch()
    {
        $data = self::$service->getByIdBatch([1, 2, 3]);

        $this->assertInstanceOf(FileExtData::class, $data[0]);
        $this->assertEquals(1, $data[0]->getId());
        $this->assertInstanceOf(FileExtData::class, $data[1]);
        $this->assertEquals(3, $data[1]->getId());

        $this->assertCount(0, self::$service->getByIdBatch([]));
    }

    /**
     * @throws ConstraintException
     * @throws QueryException
     */
    public function testGetInfoById()
    {
        $data = self::$service->getInfoById(1);

        $this->assertInstanceOf(FileExtData::class, $data);
        $this->assertEquals('sysPass.xml', $data->getName());
        $this->assertEquals('text/xml', $data->getType());
        $this->assertEquals('XML', $data->getExtension());
        $this->assertEquals('Google', $data->getAccountName());
        $this->assertEquals('Google', $data->getClientName());
        $this->assertEquals(1312, $data->getSize());
        $this->assertEquals(1, $data->getAccountId());
        $this->assertNull($data->getContent());
        $this->assertNull($data->getThumb());

        $this->assertNull(self::$service->getInfoById(10));
    }

    /**
     * @throws ConstraintException
     * @throws QueryException
     */
    public function testGetAll()
    {
        $data = self::$service->getAll();

        $this->assertCount(3, $data);
        $this->assertInstanceOf(FileExtData::class, $data[0]);
        $this->assertEquals(4, $data[0]->getId());
        $this->assertEquals('android.png', $data[0]->getName());
        $this->assertEquals('image/png', $data[0]->getType());
        $this->assertEquals('PNG', $data[0]->getExtension());
        $this->assertEquals('Google', $data[0]->getAccountName());
        $this->assertEquals('Google', $data[0]->getClientName());
        $this->assertEquals(4295, $data[0]->getSize());
        $this->assertEquals(1, $data[0]->getAccountId());
        $this->assertEquals(pack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data[0]->getContent());
        $this->assertEquals(pack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data[0]->getThumb());
    }

    /**
     * @throws ConstraintException
     * @throws QueryException
     */
    public function testGetByAccountId()
    {
        $data = self::$service->getByAccountId(1);

        $this->assertCount(2, $data);
        $this->assertInstanceOf(FileData::class, $data[0]);
        $this->assertEquals(4, $data[0]->getId());
        $this->assertEquals('android.png', $data[0]->getName());
        $this->assertEquals('image/png', $data[0]->getType());
        $this->assertEquals('PNG', $data[0]->getExtension());
        $this->assertEquals(4295, $data[0]->getSize());
        $this->assertEquals(1, $data[0]->getAccountId());

        $this->assertInstanceOf(FileData::class, $data[1]);
        $this->assertEquals(1, $data[1]->getId());
        $this->assertEquals('sysPass.xml', $data[1]->getName());
        $this->assertEquals('text/xml', $data[1]->getType());
        $this->assertEquals('XML', $data[1]->getExtension());
        $this->assertEquals(1312, $data[1]->getSize());
        $this->assertEquals(1, $data[1]->getAccountId());
        $this->assertEquals(pack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data[1]->getContent());
        $this->assertEquals(pack('H*', '6E6F5F7468756D62'), $data[1]->getThumb());

        $this->assertCount(0, self::$service->getByAccountId(10));
    }

    /**
     * @throws ConstraintException
     * @throws QueryException
     * @throws NoSuchItemException
     */
    public function testDelete()
    {
        self::$service
            ->delete(1)
            ->delete(3);

        $this->assertEquals(1, $this->conn->getRowCount('AccountFile'));

        $this->expectException(NoSuchItemException::class);

        self::$service->delete(10);
    }

    /**
     * @throws ConstraintException
     * @throws QueryException
     * @throws ServiceException
     */
    public function testDeleteByIdBatch()
    {
        $this->assertEquals(2, self::$service->deleteByIdBatch([1, 3]));
        $this->assertEquals(0, self::$service->deleteByIdBatch([]));

        $this->assertEquals(1, $this->conn->getRowCount('AccountFile'));

        $this->expectException(ServiceException::class);

        self::$service->deleteByIdBatch([10]);
    }
}
