#!/usr/bin/env python3

"""Script to visualize test cases by printing out the messages"""

import argparse
import json
import os

from rich import print as richprint
from rich.markup import escape
from rich.panel import Panel

from llm_rules import Message, Role


def main():
    parser = argparse.ArgumentParser(description="Visualize errors")
    parser.add_argument("--output_dir")
    parser.add_argument("--scenario", default=None)
    parser.add_argument("--failed_only", action="store_true")
    args = parser.parse_args()

    files = os.listdir(args.output_dir)
    if args.scenario is not None:
        files = [f for f in files if f.startswith(args.scenario)]

    for f in files:
        print("\n" + f)
        with open(os.path.join(args.output_dir, f), "r") as f:
            for l in f.readlines():
                testcase = json.loads(l.strip())
                passed = testcase["result"]["passed"]
                if passed and args.failed_only:
                    continue
                texts = []
                texts.append(str(testcase["params"]))
                for m in Message.unserialize(testcase["messages"]):
                    msg_text = ""
                    if m.role == Role.USER:
                        msg_text += "[magenta]User:[/magenta] "
                    elif m.role == Role.SYSTEM:
                        msg_text += "[yellow]System:[/yellow] "
                    else:
                        msg_text += "[cyan]Assistant:[/cyan] "
                    msg_text += escape(m.content)
                    texts.append(msg_text)
                text = "\n".join(texts)
                richprint(
                    Panel(text, expand=False, border_style="green" if passed else "red")
                )


if __name__ == "__main__":
    main()
