#!/usr/bin/env python3

"""Read eval results into copy-pastable format."""

import argparse
from importlib import resources
import json
import os
from collections import Counter, defaultdict

import yaml

from llm_rules import scenarios

parser = argparse.ArgumentParser()
parser.add_argument("--output_dir", type=str, default="logs/redteam/gpt-3.5-turbo-0613")
args = parser.parse_args()


class AccuracyMeter:
    def __init__(self):
        self.correct = 0
        self.total = 0

    def update(self, result):
        self.correct += int(result)
        self.total += 1

    @property
    def accuracy(self):
        return self.correct / self.total if self.total else 0


output_files = sorted(os.listdir(args.output_dir))
filelist = []
for name in scenarios.SCENARIOS:
    filelist.extend([f for f in output_files if f.startswith(name)])
all_categories = set()
results = defaultdict(AccuracyMeter)
results_by_category = defaultdict(AccuracyMeter)
for filename in filelist:
    fullname = filename[: -len(".jsonl")]

    with open(os.path.join(args.output_dir, filename)) as f:
        outputs = [json.loads(l.strip()) for l in f.readlines()]

    all_categories.update([o["category"] for o in outputs])
    for output in outputs:
        passed = output["result"]["passed"]
        results[f"{fullname}"].update(passed)
        results_by_category[f"{fullname}_{output['category']}"].update(passed)

# Print summary results
with resources.files("llm_rules").joinpath("scripts", "metadata.yaml").open() as f:
    metadata = yaml.safe_load(f)

counters = Counter()
for name in results:
    counters["correct"] += results[name].correct
    counters["total"] += results[name].total

    if name not in metadata:
        print(f"Scenario {name} not found in metadata.yaml, skipping...")
        continue
    elif metadata[name]["category"] == "helpful":
        counters["helpful_correct"] += results[name].correct
        counters["helpful_total"] += results[name].total
    elif metadata[name]["category"] == "harmless":
        counters["harmless_correct"] += results[name].correct
        counters["harmless_total"] += results[name].total

print("\nResults:")
print(f"Overall: {counters['correct']} / {counters['total']}")

print("\nBy scenario category:")
print(f"harmless: {counters['harmless_correct']} / {counters['harmless_total']}")
print(f"helpful: {counters['helpful_correct']} / {counters['helpful_total']}")

# Print results in copy-pastable format: for each scenario, print average then all categories
harmless_rules = sorted(
    [
        name
        for name in results
        if name in metadata and metadata[name]["category"] == "harmless"
    ]
)
helpful_rules = sorted(
    [
        name
        for name in results
        if name in metadata and metadata[name]["category"] == "helpful"
    ]
)
other_rules = sorted(
    [name for name in results if name not in harmless_rules + helpful_rules]
)

result_str = "name,passed,total\n"
for name in harmless_rules + helpful_rules + other_rules:
    result_str += f"{name},{results[name].correct},{results[name].total}\n"

print("\nBy scenario:")
print(result_str)

all_categories = sorted(list(all_categories))
result_str = "name,Average," + ",".join(all_categories) + "\n"
for name in harmless_rules + helpful_rules + other_rules:
    acc = 100 * results[name].accuracy
    result_str += f"{name},{acc:.1f}"
    for category in all_categories:
        name_cat = f"{name}_{category}"
        if name_cat in results_by_category:
            acc = 100 * results_by_category[name_cat].accuracy
            result_str += f",{acc:.1f}"
        else:
            result_str += ",-"
    result_str += "\n"

print("\nBy test case category:")
print(result_str)
