"""Script to evaluate test cases."""

import argparse
import concurrent.futures
import importlib_resources as resources
import json
import os
import random
import time
from collections import Counter, defaultdict
from contextlib import nullcontext
from dataclasses import asdict
from typing import List

import fuzzteaming.transforms as transforms
import fuzzteaming.transform_wrappers as transform_wrappers
from fuzzteaming.counter import DequeCounter
import yaml
from rich.panel import Panel
from rich import print as richprint
from rich.progress import track

from llm_rules import data
from llm_rules import Message, Role, models, scenarios


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--test_suite", type=str, default="redteam")
    parser.add_argument("--output_dir", type=str, default="logs/redteam")
    parser.add_argument(
        "--provider", type=str, default="openai", choices=models.PROVIDER_NAMES
    )
    parser.add_argument(
        "--model",
        type=str,
        default="gpt-3.5-turbo-0613",
    )
    parser.add_argument(
        "--model_name",
        type=str,
        default=None,
        help="Name of model to store outputs under, if different from --model",
    )
    parser.add_argument(
        "--fastchat_template",
        type=str,
        default=None,
        help="Name of conversation template (from FastChat)",
    )
    parser.add_argument(
        "--tokenizer_config",
        type=str,
        default=None,
        help="Name of tokenizer config file (in llm_rules.tokenizer_configs)",
    )
    parser.add_argument(
        "--temperature",
        type=float,
        default=0.0,
    )
    parser.add_argument(
        "--max_len", type=int, default=4096, help="Max length for vLLM models"
    )
    parser.add_argument(
        "--concurrency",
        type=int,
        default=5,
        help="Max number of concurrent API calls",
    )
    parser.add_argument(
        "--scenario",
        type=str,
        default=None,
    )
    parser.add_argument(
        "--fuzz_transform",
        type=str,
        default="legalese",
    )
    parser.add_argument(
        "--fuzz_provider",
        type=str,
        default="openai",
    )
    parser.add_argument(
        "--fuzz_model",
        type=str,
        default="gpt-3.5-turbo",
    )
    return parser.parse_args()


class AccuracyMeter:
    def __init__(self):
        self.correct = 0
        self.total = 0

    def update(self, result):
        self.correct += int(result)
        self.total += 1

    @property
    def accuracy(self):
        return self.correct / self.total if self.total else 0


def load_dataset(args: argparse.Namespace):
    dataset = defaultdict(dict)
    testsuite = getattr(data, args.test_suite)
    files = [
        f.name
        for f in resources.files(testsuite).iterdir()
        if f.name.endswith(".jsonl")
    ]
    files = sorted(files)

    if args.scenario:
        files = [f for f in files if f.startswith(args.scenario)]

    for file in files:
        print("Loading: {}".format(file))
        scenario_name = os.path.splitext(file)[0]
        behavior_name = ""
        if "_" in scenario_name:
            scenario_name, behavior_name = scenario_name.split("_")

        with resources.files(testsuite).joinpath(file).open() as f:
            testcases = [json.loads(line) for line in f.readlines()]

            for t in testcases:
                if "category" not in t:
                    t["category"] = "default"
                if "id" not in t:
                    t["id"] = None

            if len(testcases) == 0:
                print("\tNo testcases remaining, skipped")
                continue
            dataset[scenario_name][behavior_name] = testcases

    return dataset


def evaluate_single(
    scenario: scenarios.BaseScenario,
    test_messages: List[Message],
    model: models.BaseModel,
    counter: DequeCounter,
    args: argparse.Namespace,
):
    full_prompt = scenario.prompt

    messages = [
        Message(Role.SYSTEM, full_prompt),
    ]

    result = scenarios.Result(True)

    # Skip over prefilled assistant messages
    if any([m.role == Role.ASSISTANT for m in test_messages]):
        last_idx = max(
            [i for i, m in enumerate(test_messages) if m.role == Role.ASSISTANT]
        )
        messages = messages + test_messages[: last_idx + 1]
        test_messages = test_messages[last_idx + 1 :]

    transform = transforms.TRANSFORMS_BY_NAME[args.fuzz_transform](
        provider=args.fuzz_provider, model=args.fuzz_model, counter=counter
    )

    for i, m in enumerate(test_messages):
        # Fuzz message
        if i == 0:
            output = transform(full_prompt, m.content)
            if scenario.is_valid_user_message(Message(Role.USER, output)):
                m.content = output
        messages.append(m)

        response = models.call_with_retries(model, messages)
        content = models.concat_stream(response)
        messages.append(Message(Role.ASSISTANT, content))

        result = scenario.evaluate(messages, system_instructions=True)

        if not result.passed:
            break

    conv_str = ""
    for m in messages:
        conv_str += f"<|{m.role.name}|>: {m.content}\n\n"

    richprint(
        Panel(conv_str, expand=False, border_style="green" if result.passed else "red")
    )

    return result, messages


def main(args):
    if args.model_name is None:
        args.model_name = os.path.basename(args.model.rstrip("/"))

    with resources.files().joinpath("metadata.yaml").open() as f:
        metadata = yaml.safe_load(f)

    args.output_dir = os.path.join(args.output_dir, args.model_name)
    print(
        "Evaluating {} and saving outputs to {}".format(
            args.model_name, args.output_dir
        )
    )

    os.makedirs(args.output_dir, exist_ok=True)
    existing_files = [
        f.removesuffix(".jsonl")
        for f in os.listdir(args.output_dir)
        if f.endswith(".jsonl")
    ]

    model_kwargs = dict(
        model=args.model,
        fastchat_template=args.fastchat_template,
        tokenizer_config=args.tokenizer_config,
        temperature=args.temperature,
    )

    server_context = nullcontext()
    if args.provider == "transformers":
        args.concurrency = 1
    elif args.provider == "vllm":
        import torch

        from llm_rules.models import vLLMServer

        server_context = vLLMServer(
            args.model,
            model_max_len=args.max_len,
            num_devices=torch.cuda.device_count(),
        )
        model_kwargs["port"] = server_context.port
        del model_kwargs["model"]

    model = models.MODEL_BUILDERS[args.provider](**model_kwargs)

    dataset = load_dataset(args)
    all_categories = set()

    tasks = []
    for scenario_name in scenarios.SCENARIOS:
        if scenario_name not in dataset:
            continue

        for behavior_name in dataset[scenario_name]:
            fullname = (
                f"{scenario_name}_{behavior_name}" if behavior_name else scenario_name
            )
            if fullname in existing_files:
                print(f"Skipping {fullname}, already exists in output directory")
                continue

            output_file = os.path.join(args.output_dir, f"{fullname}.jsonl")

            for testcase in dataset[scenario_name][behavior_name]:
                all_categories.add(testcase["category"])
                scenario = scenarios.SCENARIOS[scenario_name](testcase["params"])
                messages = Message.unserialize(testcase["messages"])
                tasks.append(
                    (testcase, scenario_name, behavior_name, scenario, messages)
                )

    if len(tasks) == 0:
        print("No test cases to evaluate, exiting...")
        return

    counter = DequeCounter()

    print("\n=== Beginning evaluation ===")
    start = time.time()

    results = defaultdict(AccuracyMeter)
    results_by_category = defaultdict(AccuracyMeter)

    executor = concurrent.futures.ThreadPoolExecutor(max_workers=args.concurrency)
    with executor, server_context:
        threads = {}

        for testcase, scenario_name, behavior_name, scenario, messages in tasks:
            thread = executor.submit(
                evaluate_single,
                scenario,
                messages,
                model,
                counter,
                args,
            )
            threads[thread] = (scenario_name, behavior_name, testcase)

        for t in track(
            concurrent.futures.as_completed(threads),
            description="[cyan]Evaluating:",
            total=len(threads),
        ):
            result, messages = t.result()
            scenario_name, behavior_name, testcase = threads[t]
            fullname = (
                f"{scenario_name}_{behavior_name}" if behavior_name else scenario_name
            )
            category = testcase["category"]

            results[fullname].update(result.passed)
            results_by_category[f"{fullname}_{category}"].update(result.passed)

            # log outputs to file
            output_file = os.path.join(args.output_dir, f"{fullname}.jsonl")
            with open(output_file, "a") as f:
                sample_ = testcase.copy()
                sample_["messages"] = Message.serialize(messages)
                sample_["result"] = asdict(result)
                f.write(json.dumps(sample_, sort_keys=True) + "\n")

    print(f"=== Evaluation finished in {time.time() - start:.1f}s ===")

    counters = Counter()
    for name in results:
        counters["correct"] += results[name].correct
        counters["total"] += results[name].total

        if name not in metadata:
            print(f"Scenario {name} not found in metadata.yaml, skipping...")
            continue
        elif metadata[name]["category"] == "helpful":
            counters["helpful_correct"] += results[name].correct
            counters["helpful_total"] += results[name].total
        elif metadata[name]["category"] == "harmless":
            counters["harmless_correct"] += results[name].correct
            counters["harmless_total"] += results[name].total

    print("\nResults:")
    print(f"Overall: {counters['correct']} / {counters['total']}")

    print("\nBy scenario category:")
    print(f"harmless: {counters['harmless_correct']} / {counters['harmless_total']}")
    print(f"helpful: {counters['helpful_correct']} / {counters['helpful_total']}")

    print("\nFuzzteaming token usage:")
    print(f"Total prompt tokens: {counter.prompt_tokens}")
    print(f"Total completion tokens: {counter.completion_tokens}")


if __name__ == "__main__":
    args = parse_args()
    main(args)
