#!/usr/bin/env python3

"""Script to evaluate test cases."""

import argparse
import concurrent.futures
from importlib import resources
import json
import os
import time
from collections import Counter, defaultdict
from contextlib import nullcontext
from dataclasses import asdict
from typing import List

import yaml
from rich.progress import track

from llm_rules import data
from llm_rules import Message, Role, models, scenarios


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument("--test_suite", type=str, default="redteam")
    parser.add_argument("--output_dir", type=str, default="logs/redteam")
    parser.add_argument(
        "--provider", type=str, default="openai", choices=models.PROVIDER_NAMES
    )
    parser.add_argument(
        "--model",
        type=str,
        default="gpt-3.5-turbo-0613",
    )
    parser.add_argument(
        "--model_name",
        type=str,
        default=None,
        help="Name of model to store outputs under, if different from --model",
    )
    parser.add_argument(
        "--fastchat_template",
        type=str,
        default=None,
        help="Name of conversation template (from FastChat)",
    )
    parser.add_argument(
        "--tokenizer_config",
        type=str,
        default="auto",
        help="Name of tokenizer config file (in llm_rules.tokenizer_configs), or 'auto' to use the tokenizer's built-in template",
    )
    parser.add_argument(
        "--temperature",
        type=float,
        default=0.0,
    )
    parser.add_argument(
        "--max_len", type=int, default=4096, help="Max length for vLLM models"
    )
    parser.add_argument(
        "--scenario",
        type=str,
        default=None,
        help="Evaluate a single scenario, or all scenarios if None",
    )
    parser.add_argument(
        "--category",
        type=str,
        default=None,
        help="Evaluate a single testcase category, or all categories if None",
    )
    parser.add_argument(
        "--id",
        type=str,
        default=None,
        help="If a single scenario is specified, evaluate a single test case by ID",
    )
    parser.add_argument(
        "--system_instructions",
        action="store_true",
        default=False,
        help="Present instructions as a system message, if supported",
    )
    parser.add_argument(
        "--remove_precedence_reminders",
        action="store_true",
        help="Remove reminders about rule precedence in the prompts",
    )
    parser.add_argument(
        "--system_prompt",
        type=str,
        default=None,
        choices=models.PROMPTS.keys(),
        help="System message to model, if not using --system_instructions",
    )
    parser.add_argument(
        "--prefix_prompt",
        type=str,
        default=None,
        choices=models.PROMPTS.keys(),
        help="Prefix message to instruction prompt",
    )
    parser.add_argument(
        "--suffix_dir",
        type=str,
        default=None,
        help="Directory containing GCG attack output logs",
    )
    parser.add_argument(
        "--manual_suffix",
        action="store_true",
        default=False,
        help="Prompt user to manually specify adversarial suffix",
    )
    parser.add_argument(
        "--concurrency",
        type=int,
        default=5,
        help="Max number of concurrent API calls",
    )
    parser.add_argument(
        "--model_kwargs",
        action="append",
        nargs="*",
        default=[],
        help="Additional kwargs for the model in the format key=value. Repeat for multiple kwargs.",
        metavar="KEY=VALUE",
    )
    return parser.parse_args()


class AccuracyMeter:
    def __init__(self):
        self.correct = 0
        self.total = 0

    def update(self, result):
        self.correct += int(result)
        self.total += 1

    @property
    def accuracy(self):
        return self.correct / self.total if self.total else 0


def load_dataset(args: argparse.Namespace):
    dataset = defaultdict(dict)
    testsuite = getattr(data, args.test_suite)

    files = [
        f.name
        for f in resources.files(testsuite).iterdir()
        if f.name.endswith(".jsonl")
    ]
    files = sorted(files)

    if args.scenario:
        files = [f for f in files if f.startswith(args.scenario)]

    for file in files:
        print("Loading: {}".format(file))
        scenario_name = os.path.splitext(file)[0]
        behavior_name = ""
        if "_" in scenario_name:
            scenario_name, behavior_name = scenario_name.split("_")

        with resources.files(testsuite).joinpath(file).open() as f:
            testcases = [json.loads(line) for line in f.readlines()]

            for t in testcases:
                if "category" not in t:
                    t["category"] = "default"
                if "id" not in t:
                    t["id"] = None

            if args.category:
                print("\tFiltering by category: {}".format(args.category))
                testcases = [
                    t
                    for t in testcases
                    if t["category"].lower() == args.category.lower()
                ]

            if args.id:
                print("\tFiltering by id: {}".format(args.id))
                testcases = [t for t in testcases if t["id"] == args.id]

            if len(testcases) == 0:
                print("\tNo testcases remaining, skipped")
                continue
            dataset[scenario_name][behavior_name] = testcases

    return dataset


def evaluate_single(
    scenario: scenarios.BaseScenario,
    test_messages: List[Message],
    model: models.BaseModel,
    use_system_instructions: bool,
    remove_precedence_reminders: bool,
    system_prompt: str,
    prefix_prompt: str,
    suffix: str,
):
    full_prompt = scenario.prompt

    if remove_precedence_reminders:
        full_prompt = scenarios.utils.remove_precedence_reminders(full_prompt)

    if prefix_prompt:
        full_prompt = models.PROMPTS[prefix_prompt] + "\n" + full_prompt

    if use_system_instructions:
        messages = [Message(Role.SYSTEM, full_prompt)]
    else:
        system_prompt = models.PROMPTS[system_prompt]
        messages = [
            Message(Role.SYSTEM, system_prompt),
            Message(Role.USER, full_prompt),
            Message(Role.ASSISTANT, scenario.initial_response),
        ]

    result = scenarios.Result(True)

    # Skip over prefilled assistant messages
    if any([m.role == Role.ASSISTANT for m in test_messages]):
        last_idx = max(
            [i for i, m in enumerate(test_messages) if m.role == Role.ASSISTANT]
        )
        messages = messages + test_messages[: last_idx + 1]
        test_messages = test_messages[last_idx + 1 :]

    for i, m in enumerate(test_messages):
        if i == 0 and suffix:
            m.content += " " + suffix  # gcg suffixes are applied with a preceding space
        messages.append(m)

        response = models.call_with_retries(model, messages)
        content = models.concat_stream(response)
        messages.append(Message(Role.ASSISTANT, content))

        result = scenario.evaluate(messages, use_system_instructions)

        if not result.passed:
            return result, messages

    return result, messages


def main(args):
    if args.model_name is None:
        args.model_name = os.path.basename(args.model.rstrip("/"))

    with resources.files("llm_rules").joinpath("scripts", "metadata.yaml").open() as f:
        metadata = yaml.safe_load(f)

    args.output_dir = os.path.join(args.output_dir, args.model_name)
    print(
        "Evaluating {} and saving outputs to {}".format(
            args.model_name, args.output_dir
        )
    )

    os.makedirs(args.output_dir, exist_ok=True)
    existing_files = [
        f.removesuffix(".jsonl")
        for f in os.listdir(args.output_dir)
        if f.endswith(".jsonl")
    ]

    model_kwargs = dict(
        model=args.model,
        fastchat_template=args.fastchat_template,
        tokenizer_config=args.tokenizer_config,
        temperature=args.temperature,
    )

    server_context = nullcontext()
    if args.provider == "transformers":
        args.concurrency = 1
    elif args.provider == "torch_llms":
        args.concurrency = 1
        model_kwargs["max_len"] = args.max_len
    elif args.provider == "vllm":
        import torch

        from llm_rules.models import vLLMServer

        server_context = vLLMServer(
            args.model,
            model_max_len=args.max_len,
            num_devices=torch.cuda.device_count(),
        )
        model_kwargs["port"] = server_context.port

    if args.model_kwargs:
        for kwargs_list in args.model_kwargs:
            for kwarg in kwargs_list:
                if "=" not in kwarg:
                    print(
                        f"Warning: Skipping invalid model kwarg (missing '='): {kwarg}"
                    )
                    continue

                key, value = kwarg.split("=", 1)
                key = key.strip()
                value = value.strip()

                if not key:
                    print(f"Warning: Skipping invalid model kwarg (empty key): {kwarg}")
                    continue

                try:
                    import ast

                    value = ast.literal_eval(value)
                except (ValueError, SyntaxError):
                    pass

                model_kwargs[key] = value

    model = models.MODEL_BUILDERS[args.provider](**model_kwargs)

    dataset = load_dataset(args)
    all_categories = set()

    tasks = []
    for scenario_name in scenarios.SCENARIOS:
        if scenario_name not in dataset:
            continue

        for behavior_name in dataset[scenario_name]:
            fullname = (
                f"{scenario_name}_{behavior_name}" if behavior_name else scenario_name
            )
            if fullname in existing_files:
                print(f"Skipping {fullname}, already exists in output directory")
                continue

            output_file = os.path.join(args.output_dir, f"{fullname}.jsonl")

            suffix = ""
            if args.suffix_dir:
                suffix_file = os.path.join(args.suffix_dir, f"{fullname}.jsonl")
                if not os.path.exists(suffix_file):
                    print(f"Suffix file for {fullname} not found, skipping scenario")
                    continue
                with open(suffix_file) as f:
                    suffix = json.loads(f.readlines()[-1].strip())["suffix"]
            elif args.manual_suffix:
                suffix = input("Enter suffix: ")

            for testcase in dataset[scenario_name][behavior_name]:
                all_categories.add(testcase["category"])
                scenario = scenarios.SCENARIOS[scenario_name](testcase["params"])
                messages = Message.unserialize(testcase["messages"])
                tasks.append(
                    (testcase, scenario_name, behavior_name, scenario, messages, suffix)
                )

    if len(tasks) == 0:
        print("No test cases to evaluate, exiting...")
        return

    print("\n=== Beginning evaluation ===")
    start = time.time()

    results = defaultdict(AccuracyMeter)
    results_by_category = defaultdict(AccuracyMeter)

    executor = concurrent.futures.ThreadPoolExecutor(max_workers=args.concurrency)
    with executor, server_context:
        threads = {}

        for testcase, scenario_name, behavior_name, scenario, messages, suffix in tasks:
            thread = executor.submit(
                evaluate_single,
                scenario,
                messages,
                model,
                args.system_instructions and model.supports_system_message,
                args.remove_precedence_reminders,
                args.system_prompt,
                args.prefix_prompt,
                suffix,
            )
            threads[thread] = (scenario_name, behavior_name, testcase)

        for t in track(
            concurrent.futures.as_completed(threads),
            description="[cyan]Evaluating:",
            total=len(threads),
        ):
            result, messages = t.result()
            scenario_name, behavior_name, testcase = threads[t]
            fullname = (
                f"{scenario_name}_{behavior_name}" if behavior_name else scenario_name
            )
            category = testcase["category"]

            results[fullname].update(result.passed)
            results_by_category[f"{fullname}_{category}"].update(result.passed)

            # log outputs to file
            output_file = os.path.join(args.output_dir, f"{fullname}.jsonl")
            with open(output_file, "a") as f:
                sample_ = testcase.copy()
                sample_["messages"] = Message.serialize(messages)
                sample_["result"] = asdict(result)
                f.write(json.dumps(sample_, sort_keys=True) + "\n")

    print(f"=== Evaluation finished in {time.time() - start:.1f}s ===")

    counters = Counter()
    for name in results:
        counters["correct"] += results[name].correct
        counters["total"] += results[name].total

        if name not in metadata:
            print(f"Scenario {name} not found in metadata.yaml, skipping...")
            continue
        elif metadata[name]["category"] == "helpful":
            counters["helpful_correct"] += results[name].correct
            counters["helpful_total"] += results[name].total
        elif metadata[name]["category"] == "harmless":
            counters["harmless_correct"] += results[name].correct
            counters["harmless_total"] += results[name].total

    print("\nResults:")
    print(f"Overall: {counters['correct']} / {counters['total']}")

    print("\nBy scenario category:")
    print(f"harmless: {counters['harmless_correct']} / {counters['harmless_total']}")
    print(f"helpful: {counters['helpful_correct']} / {counters['helpful_total']}")


if __name__ == "__main__":
    args = parse_args()
    main(args)
