import json
import os
import random
import signal
import socket
import time
from functools import partial
from importlib import resources
from subprocess import Popen
from typing import List, Optional, Union

import requests
from transformers import AutoTokenizer

from llm_rules import Message, tokenizer_configs
from llm_rules.fastchat_templates import get_stop_strings_fastchat
from llm_rules.models.base import (
    BaseModel,
    build_prompt_fastchat,
    build_prompt_tokenizer,
    get_token_string,
)


class vLLMModel(BaseModel):
    """Model builder for vllm models. `model` should be a FastChat template name.

    Call with a list of `Message` objects to generate a response.
    """

    supports_system_message = True

    def __init__(
        self,
        port: int,
        fastchat_template: Optional[str] = None,
        tokenizer_config: Optional[str] = None,
        temperature: float = 0.0,
        stream: bool = False,
        top_p: float = 1.0,
        max_tokens: int = 100,
        stop: Union[list, None] = None,
        frequency_penalty: float = 0.0,
        presence_penalty: float = 0.0,
        model: Optional[str] = None,
        **kwargs,
    ):
        if fastchat_template is None and tokenizer_config is None:
            fastchat_template = "zero_shot"

        if fastchat_template is not None:
            self.apply_template_fn = partial(
                build_prompt_fastchat, template_name=fastchat_template
            )
            self.stop = get_stop_strings_fastchat(stop, fastchat_template)
        elif tokenizer_config == "auto":
            assert True
            tokenizer = AutoTokenizer.from_pretrained(model)

            def apply_template_fn(messages):
                messages_dict = Message.serialize(messages)
                return tokenizer.apply_chat_template(
                    messages_dict, add_generation_prompt=True, tokenize=False
                )

            self.apply_template_fn = apply_template_fn
            self.stop = [tokenizer.eos_token]
        else:
            with resources.files(tokenizer_configs).joinpath(
                tokenizer_config
            ).open() as file:
                config = json.load(file)

            self.apply_template_fn = partial(
                build_prompt_tokenizer,
                config=config,
                add_generation_prompt=True,
            )
            self.stop = [get_token_string(config["eos_token"])]

        self.port = port
        self.temperature = temperature
        self.stream = stream
        self.top_p = top_p
        self.max_tokens = max_tokens
        self.frequency_penalty = frequency_penalty
        self.presence_penalty = presence_penalty

    def __call__(self, messages: List[Message]):
        prompt = self.apply_template_fn(messages)
        request_dict = dict(
            prompt=prompt,
            temperature=self.temperature,
            top_p=self.top_p,
            n=1,
            max_tokens=self.max_tokens,
            stop=self.stop,
            frequency_penalty=self.frequency_penalty,
            presence_penalty=self.presence_penalty,
        )
        response = requests.post(
            f"http://localhost:{self.port}/generate", json=request_dict
        ).json()
        response = response["text"][0][len(prompt) :]
        return response


class vLLMServer:
    """Context manager for starting and stopping a local VLLM server. h/t thavens for the idea."""

    def __init__(
        self,
        model: str,
        model_max_len: int = 4096,
        num_devices: int = 1,
    ):
        self.model = model
        self.model_max_len = model_max_len
        self.num_devices = num_devices
        self.port = self.get_free_port()

    def get_free_port(self):
        with socket.socket(socket.AF_INET, socket.SOCK_STREAM) as s:
            s.bind(("", 0))  # Bind to a free port provided by the host.
            return s.getsockname()[1]

    def verify(self):
        while True:
            try:
                r = requests.get(f"http://localhost:{self.port}/health")
                if r.ok:
                    break
            except requests.exceptions.ConnectionError:
                pass
            print("Waiting for server to start...")
            time.sleep(5)

    def launch_ray(self):
        import ray

        # config different ports for ray head and ray workers to avoid conflict when running multiple jobs on one machine/cluster
        # docs: https://docs.ray.io/en/latest/cluster/vms/user-guides/community/slurm.html#slurm-networking-caveats
        num_cpus = min([os.cpu_count(), 12])

        os.environ["RAY_DEDUP_LOGS"] = "0"

        RAY_PORT = random.randint(0, 999) + 6000  # Random port in 6xxx zone
        RAY_MIN_PORT = random.randint(0, 489) * 100 + 10002
        RAY_MAX_PORT = RAY_MIN_PORT + 99  # Random 100 port ranges zone

        os.environ["RAY_ADDRESS"] = f"127.0.0.1:{RAY_PORT}"
        ray_start_command = (
            f"ray stop; mkdir -p /tmp/$USER/; TMPDIR=/tmp/$USER/ ray start --head --disable-usage-stats --include-dashboard=False "
            f"--num-cpus={num_cpus} --port {RAY_PORT} --min-worker-port={RAY_MIN_PORT} --max-worker-port={RAY_MAX_PORT}"
        )

        print(f"Starting Ray with command: {ray_start_command}")
        os.system(ray_start_command)
        ray.init(ignore_reinit_error=True)

    def __enter__(self):
        if self.num_devices > 1:
            self.launch_ray()

        vllm_command = (
            f"python -m vllm.entrypoints.api_server --disable-log-requests --disable-log-stats --trust-remote-code "
            f"--model {self.model} --max-model-len {self.model_max_len} --enforce-eager "
            f"--tensor-parallel-size {self.num_devices} --port {self.port}"
        )
        # self.p = Popen(vllm_command.split(), stdout=DEVNULL, stderr=STDOUT)
        self.p = Popen(vllm_command.split())
        self.verify()

    def __exit__(self, exc_type, exc_value, exc_tb):
        self.p.send_signal(signal.SIGTERM)
        if self.num_devices > 1:
            os.system("ray stop")

        while self.p.poll() is None:
            time.sleep(1)
