import json
from functools import partial
from importlib import resources
from typing import List, Optional

import torch
from transformers import AutoModelForCausalLM, AutoTokenizer

from llm_rules import Message, tokenizer_configs
from llm_rules.models.base import (
    BaseModel,
    build_prompt_fastchat,
    build_prompt_tokenizer,
)


class TransformersModel(BaseModel):
    """Model builder for HuggingFace Transformers model. `model` should be in the format model_name@checkpoint_path

    Call with a list of `Message` objects to generate a response.
    """

    supports_system_message = True

    def __init__(
        self,
        model: str,
        fastchat_template: Optional[str] = None,
        tokenizer_config: Optional[str] = None,
        temperature: float = 0.0,
        top_p: float = 1.0,
        max_tokens: int = 100,
        **kwargs,
    ):
        if fastchat_template is None and tokenizer_config is None:
            fastchat_template = "zero_shot"

        if fastchat_template is not None:
            self.apply_template_fn = partial(
                build_prompt_fastchat, template_name=fastchat_template
            )
        elif tokenizer_config is not None:
            with resources.files(tokenizer_configs).joinpath(
                tokenizer_config
            ).open() as file:
                config = json.load(file)

            self.apply_template_fn = partial(
                build_prompt_tokenizer,
                config=config,
                add_generation_prompt=True,
            )

        self.model = AutoModelForCausalLM.from_pretrained(
            model, torch_dtype=torch.float16, device_map="auto"
        )
        self.tokenizer = AutoTokenizer.from_pretrained(model)
        if not self.tokenizer.pad_token:
            self.tokenizer.pad_token = self.tokenizer.eos_token

        self.temperature = temperature
        self.top_p = top_p
        self.max_tokens = max_tokens

    def __call__(self, messages: List[Message]):
        prompt = self.apply_template_fn(messages)
        model_inputs = self.tokenizer(prompt, return_tensors="pt")
        model_inputs["input_ids"] = model_inputs["input_ids"].cuda()
        model_inputs["attention_mask"] = model_inputs["attention_mask"].cuda()
        prompt_len = model_inputs["attention_mask"].sum(dim=1)
        output = self.model.generate(
            **model_inputs,
            max_new_tokens=self.max_tokens,
            do_sample=self.temperature > 0,
            temperature=self.temperature,
            top_p=self.top_p,
        )
        response = self.tokenizer.decode(
            output[0][prompt_len:], skip_special_tokens=True
        ).strip()
        return [response]
