import os
from typing import List

from dotenv import load_dotenv
from openai import OpenAI

from llm_rules import Message
from llm_rules.models.base import BaseModel


class Response:
    """Response wrapper class for OpenAI API response object.

    Implements the iterator interface to enable simple iteration over streamed response chunks, such that
    `"".join(response)` returns the full completion.
    """

    def __init__(self, response, stream=False):
        self.response = response
        self.stream = stream
        self.complete = False
        if self.stream:
            self.response_iter = iter(self.response)
        else:
            self.response = self.response.choices[0].message.content

    def __iter__(self):
        return self

    def __next__(self):
        if self.complete:
            raise StopIteration

        if not self.stream:
            self.complete = True
            return self.response

        try:
            chunk = next(self.response_iter)
            content = chunk.choices[0].delta.content
            if content is None:
                self.complete = True
                raise StopIteration
            return content
        except StopIteration as e:
            self.complete = True
            raise e


class OpenAIModel(BaseModel):
    """Model builder for OpenAI API models.

    Call with a list of `Message` objects to generate a response.
    """

    supports_system_message = True

    def __init__(
        self,
        model: str,
        temperature: float = 0.0,
        stream: bool = False,
        top_p: float = 1.0,
        max_tokens: int = 100,
        stop=None,
        **kwargs,
    ):
        self.model = model
        self.temperature = temperature
        self.stream = stream
        self.top_p = top_p
        self.max_tokens = max_tokens
        self.stop = stop
        load_dotenv()
        self.client = OpenAI(api_key=os.getenv("OPENAI_API_KEY", ""))

    def __call__(self, messages: List[Message]):
        messages_dict = Message.serialize(messages)
        response = self.client.chat.completions.create(
            model=self.model,
            messages=messages_dict,
            temperature=self.temperature,
            top_p=self.top_p,
            n=1,
            stream=self.stream,
            max_tokens=self.max_tokens,
            stop=self.stop,
        )
        response = Response(response, stream=self.stream)

        return response
