import os
from typing import List

import vertexai
from dotenv import load_dotenv
from google.api_core import exceptions
from tenacity import retry, retry_if_exception_type, wait_random_exponential
from vertexai.generative_models import GenerativeModel, SafetySetting

from llm_rules import Message, Role
from llm_rules.models.base import BaseModel

SAFETY_SETTINGS = [
    SafetySetting(
        category=c,
        # BLOCK_NONE requires special permission from GCP
        threshold="BLOCK_ONLY_HIGH",
    )
    for c in [
        "HARM_CATEGORY_UNSPECIFIED",
        "HARM_CATEGORY_HATE_SPEECH",
        "HARM_CATEGORY_DANGEROUS_CONTENT",
        "HARM_CATEGORY_HARASSMENT",
        "HARM_CATEGORY_SEXUALLY_EXPLICIT",
    ]
]


class Response:
    """Response wrapper class for Gemini API response object.

    Implements the iterator interface to enable simple iteration over streamed response chunks, such that
    `"".join(response)` returns the full completion.
    """

    def __init__(self, response, stream=False):
        self.response = response
        self.stream = stream
        self.complete = False
        if self.stream:
            self.response_iter = iter(self.response)
        else:
            self.response = self.get_text(self.response)

    def get_text(self, response):
        if (
            len(response.candidates) > 0
            and len(response.candidates[0].content.parts) > 0
        ):
            return response.text
        else:
            return " "

    def __iter__(self):
        return self

    def __next__(self):
        if self.complete:
            raise StopIteration

        if not self.stream:
            self.complete = True
            return self.response

        try:
            chunk = next(self.response_iter)
            content = self.get_text(chunk)
            if content is None:
                self.complete = True
                raise StopIteration
            return content
        except StopIteration as e:
            self.complete = True
            raise e


class GoogleVertexModel(BaseModel):
    """Interface for interacting with the VertexAI API.

    Call with a list of `Message` objects to generate a response.
    """

    supports_system_message = True

    def __init__(
        self,
        model: str,
        temperature: float = 0.0,
        top_k: int = 40,
        top_p: float = 0.95,
        max_tokens: int = 100,
        stream: bool = False,
        **kwargs,
    ):
        self.model_name = model
        self.model = GenerativeModel(model)
        self.temperature = temperature
        self.top_k = top_k
        self.top_p = top_p
        self.max_tokens = max_tokens
        self.stream = stream
        load_dotenv()
        vertexai.init(
            project=os.getenv("GCP_PROJECT_ID", ""),
            location=os.getenv("GCP_PROJECT_ZONE", "us-central1"),
        )

    def encode(self, messages: List[Message]):
        encoded = []
        for m in messages:
            if m.role == Role.USER:
                encoded.append({"role": "user", "parts": [{"text": m.content}]})
            elif m.role == Role.ASSISTANT:
                encoded.append({"role": "model", "parts": [{"text": m.content}]})
        return encoded

    def __call__(self, messages: List[Message]):
        if messages[0].role == Role.SYSTEM:
            self.model._system_instruction = messages[0].content
        else:
            self.model._system_instruction = None

        response = self.model.generate_content(
            self.encode(messages),
            generation_config=dict(
                temperature=self.temperature,
                top_k=self.top_k,
                top_p=self.top_p,
                max_output_tokens=self.max_tokens,
            ),
            safety_settings=SAFETY_SETTINGS,
            stream=self.stream,
        )
        response = Response(response, stream=self.stream)

        return response


@retry(
    retry=retry_if_exception_type(
        (
            exceptions.ResourceExhausted,
            exceptions.ServerError,
            exceptions.TooManyRequests,
        )
    ),
    wait=wait_random_exponential(min=1, max=10),
)
def google_vertex_call_with_retries(model, messages):
    return model(messages)
