import os
from typing import List

import google.generativeai as genai
from dotenv import load_dotenv
from google.api_core import exceptions
from tenacity import retry, retry_if_exception_type, wait_random_exponential

from llm_rules import Message, Role
from llm_rules.models.base import BaseModel, build_prompt_fastchat

SAFETY_SETTINGS = [
    {
        "category": c,
        "threshold": "block_none",
    }
    for c in ["harassment", "hate", "sex", "danger"]
]

OLD_SAFETY_SETTINGS = [
    {
        "category": c,
        "threshold": "block_none",
    }
    for c in [
        "unspecified",
        "derogatory",
        "toxicity",
        "violence",
        "sex",
        "medical",
        "danger",
    ]
]


class Response:
    """Response wrapper class for Gemini API response object.

    Implements the iterator interface to enable simple iteration over streamed response chunks, such that
    `"".join(response)` returns the full completion.
    """

    def __init__(self, response, stream=False):
        self.response = response
        self.stream = stream
        self.complete = False
        if self.stream:
            self.response_iter = iter(self.response)
        else:
            self.response = self.get_text(self.response)

    def get_text(self, response):
        if (
            len(response.candidates) > 0
            and len(response.candidates[0].content.parts) > 0
        ):
            return response.text
        else:
            return ""

    def __iter__(self):
        return self

    def __next__(self):
        if self.complete:
            raise StopIteration

        if not self.stream:
            self.complete = True
            return self.response

        try:
            chunk = next(self.response_iter)
            content = self.get_text(chunk)
            if content is None:
                self.complete = True
                raise StopIteration
            return content
        except StopIteration as e:
            self.complete = True
            raise e


class GoogleModel(BaseModel):
    """Interface for interacting with the Gemini chat or PaLM text API. PaLM text refuses non-English text, while PaLM
    chat is unusable for our purposes because of overly strict safety filters.

    Call with a list of `Message` objects to generate a response. Streaming is not yet implemented.
    """

    supports_system_message = False

    def __init__(
        self,
        model: str,
        temperature: float = 0.0,
        top_k: int = 40,
        top_p: float = 0.95,
        max_tokens: int = 100,
        stream: bool = False,
        **kwargs,
    ):
        self.model_name = model
        if "gemini" in model:
            self.model = genai.GenerativeModel(model)
        self.temperature = temperature
        self.top_k = top_k
        self.top_p = top_p
        self.max_tokens = max_tokens
        self.stream = stream
        load_dotenv()
        genai.configure(api_key=os.getenv("GEMINI_API_KEY", ""))

    def encode(self, messages: List[Message]):
        encoded = []
        for m in messages:
            if m.role == Role.USER:
                encoded.append({"role": "user", "parts": [m.content]})
            elif m.role == Role.ASSISTANT:
                encoded.append({"role": "model", "parts": [m.content]})
        return encoded

    def __call__(self, messages: List[Message]):
        if "gemini" in self.model_name:
            response = self.model.generate_content(
                self.encode(messages),
                generation_config=dict(
                    temperature=self.temperature,
                    top_k=self.top_k,
                    top_p=self.top_p,
                    max_output_tokens=self.max_tokens,
                ),
                safety_settings=SAFETY_SETTINGS,
                stream=self.stream,
            )
            response = Response(response, stream=self.stream)
        else:
            prompt = (
                build_prompt_fastchat(messages)
                if len(messages) > 0
                else messages[0].content
            )
            response = genai.generate_text(
                model="models/" + self.model_name,
                prompt=prompt,
                temperature=self.temperature,
                max_output_tokens=self.max_tokens,
                top_k=self.top_k,
                top_p=self.top_p,
                safety_settings=OLD_SAFETY_SETTINGS,
            )
            response = response.result or ""
            response = [response]

        return response


@retry(
    retry=retry_if_exception_type(
        (
            exceptions.TooManyRequests,
            exceptions.ServerError,
        )
    ),
    wait=wait_random_exponential(min=1, max=10),
)
def google_call_with_retries(model, messages):
    return model(messages)
