import os
from typing import List

import anthropic
from dotenv import load_dotenv
from tenacity import retry, retry_if_exception_type, wait_random_exponential

from llm_rules import Message, Role
from llm_rules.models.base import BaseModel


class Response:
    """Response wrapper class for Anthropic Messages API response object.

    Implements the iterator interface to enable simple iteration over streamed response chunks, such that
    `"".join(response)` returns the full completion.
    """

    def __init__(self, response, stream=False):
        self.response = response
        self.stream = stream
        self.complete = False
        if self.stream:
            self.response_iter = iter(self.response)
        else:
            self.response = self.response.content[0].text

    def __iter__(self):
        return self

    def __next__(self):
        if self.complete:
            raise StopIteration

        if not self.stream:
            self.complete = True
            return self.response

        try:
            chunk = next(self.response_iter)
            while chunk.type != "content_block_delta":
                chunk = next(self.response_iter)
            delta = chunk.delta.text
            return delta
        except StopIteration as e:
            self.complete = True
            raise e


class AnthropicModel(BaseModel):
    """Interface for interacting with the Anthropic API.

    Call with a list of `Message` objects to generate a response.
    """

    supports_system_message = True

    def __init__(
        self,
        model: str,
        stream: bool = False,
        temperature: float = 0.0,
        top_k: int = -1,
        top_p: float = -1,
        max_tokens: int = 100,
        **kwargs,
    ):
        self.model = model
        self.stream = stream
        self.temperature = temperature
        self.top_k = top_k
        self.top_p = top_p
        self.max_tokens = max_tokens
        load_dotenv()
        self.client = anthropic.Anthropic(api_key=os.getenv("ANTHROPIC_API_KEY", ""))

    def __call__(self, messages: List[Message]):
        system = ""
        if messages[0].role == Role.SYSTEM:
            system = messages[0].content
            messages = messages[1:]

        messages_dict = Message.serialize(messages)
        response = self.client.messages.create(
            messages=messages_dict,
            model=self.model,
            system=system,
            max_tokens=self.max_tokens,
            temperature=self.temperature,
            top_p=self.top_p,
            top_k=self.top_k,
            stream=self.stream,
        )
        response = Response(response, stream=self.stream)

        return response


@retry(
    retry=retry_if_exception_type(
        (
            anthropic.RateLimitError,
            anthropic.InternalServerError,
        )
    ),
    wait=wait_random_exponential(min=1, max=10),
)
def anthropic_call_with_retries(model, messages):
    return model(messages)
