from dataclasses import dataclass
from enum import Enum
from typing import Dict, List, Union


class Role(Enum):
    USER = 1
    ASSISTANT = 2
    SYSTEM = 3


@dataclass
class Message:
    role: Role
    content: str

    def __str__(self):
        return f"[{self.role.name.title()}]: {self.content}"

    @staticmethod
    def serialize(messages, user_only=False) -> List[Dict[str, str]]:
        if not isinstance(messages, list):
            messages = [messages]
        if user_only:
            messages = [
                {"role": m.role.name, "content": m.content}
                for m in messages
                if m.role == Role.USER
            ]
        else:
            messages = [
                {"role": m.role.name.lower(), "content": m.content} for m in messages
            ]
        return messages

    @staticmethod
    def unserialize(messages: Union[dict, List[dict]]) -> List["Message"]:
        if not isinstance(messages, list):
            messages = [messages]
        objects = [Message(Role[m["role"].upper()], m["content"]) for m in messages]
        return objects
