import argparse
import json
import os
from dataclasses import asdict

import numpy as np
import torch
import torch.nn as nn
import utils
import yaml

from llm_rules import Message, Role, get_conv_template, models, scenarios


def parse_args():
    parser = argparse.ArgumentParser()
    parser.add_argument(
        "--model",
        type=str,
        default="/data/norman_mu/models/Llama-2-7b-chat-hf",
        help="Path to model checkpoint",
    )
    parser.add_argument(
        "--conv_template",
        type=str,
        default="llama-2",
        help="FastChat model conv template",
    )
    parser.add_argument(
        "--log_dir",
        type=str,
        default="../logs/gcg_attack",
    )
    parser.add_argument("--metadata_file", type=str, default="../metadata.yaml")
    parser.add_argument("--testcase_file", type=str, default="direct_request.yaml")
    parser.add_argument("--target_file", type=str, default="targets.yaml")
    parser.add_argument(
        "--num_steps", type=int, default=500, help="Number of attack steps"
    )
    parser.add_argument(
        "--scenario", type=str, default="Encryption", choices=scenarios.SCENARIOS.keys()
    )
    parser.add_argument(
        "--behavior", type=str, default="", help="Specify rule/behavior to target"
    )
    parser.add_argument(
        "--adv_suffix_init",
        type=str,
        default="! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ! ",
    )
    parser.add_argument(
        "--batch_size",
        type=int,
        default=512,
        help="Number of candidates to evaluate in each step",
    )
    parser.add_argument(
        "--topk", type=int, default=256, help="Top k tokens to sample from"
    )
    parser.add_argument("--allow_non_ascii", action="store_true")
    args = parser.parse_args()
    return args


def build_messages_and_target(
    scenario, content_template, message_template, target_template, args
):
    var_dict = asdict(scenario.p)
    first_message = content_template.format(**var_dict)
    if message_template is not None:
        first_message = message_template.format(**var_dict, content=first_message)
    target = " " + target_template.format(**var_dict)

    messages = [
        Message(Role.SYSTEM, ""),
        Message(Role.USER, scenario.prompt),
        Message(Role.ASSISTANT, scenario.initial_response),
        Message(Role.USER, first_message),
    ]

    return messages, target


def main(args):
    np.random.seed(42)
    torch.manual_seed(42)
    torch.cuda.manual_seed_all(42)

    model, tokenizer = utils.load_model_and_tokenizer(
        args.model,
        low_cpu_mem_usage=True,
        use_cache=False,
        device="cuda",
        attn_implementation="flash_attention_2",
    )
    embed_layer = model.get_input_embeddings()

    # ========== Setup testcase messages ========== #
    full_name = args.scenario + ("_" + args.behavior if args.behavior else "")
    log_dir = os.path.join(args.log_dir, args.model.split("/")[-1])
    os.makedirs(log_dir, exist_ok=True)
    log_file = os.path.join(log_dir, full_name + ".jsonl")
    if os.path.exists(log_file):
        os.remove(log_file)

    with open(args.metadata_file) as f:
        metadata = yaml.safe_load(f)

    with open(args.testcase_file) as f:
        testcases = yaml.safe_load(f)

    with open(args.target_file) as f:
        targets = yaml.safe_load(f)

    target_template = targets[full_name]

    message_template = metadata[full_name]["format"]
    content_template = testcases[full_name]["messages"][0]["content"]

    suffix_manager = utils.SuffixManager(
        tokenizer=tokenizer,
        use_system_instructions=False,
        conv_template=get_conv_template(args.conv_template),
    )

    not_allowed_tokens = (
        None if args.allow_non_ascii else utils.get_nonascii_toks(tokenizer)
    )
    adv_suffix = args.adv_suffix_init

    for i in range(args.num_steps):
        # ========== Compute coordinate gradients ========== #
        scenario = scenarios.SCENARIOS[args.scenario]()
        messages, target = build_messages_and_target(
            scenario, content_template, message_template, target_template, args
        )

        (
            dynamic_input_ids,
            optim_slice,
            target_slice,
            loss_slice,
        ) = suffix_manager.get_input_ids(messages, adv_suffix, target)

        dynamic_input_ids = dynamic_input_ids.to("cuda")
        optim_ids, target_ids = (
            dynamic_input_ids[optim_slice],
            dynamic_input_ids[target_slice],
        )
        optim_ids, target_ids = optim_ids.to("cuda"), target_ids.to("cuda")
        input_embeds = embed_layer(dynamic_input_ids).unsqueeze(0)
        input_embeds.requires_grad_()

        # forward pass
        outputs = model(inputs_embeds=input_embeds)
        logits = outputs.logits

        # compute loss and gradients
        loss_logits = logits[:, loss_slice].squeeze(0)
        loss = nn.functional.cross_entropy(loss_logits, target_ids)

        embed_grads = torch.autograd.grad(outputs=[loss], inputs=[input_embeds])[0]
        token_grads = embed_grads @ embed_layer.weight.t()
        token_grads = token_grads / token_grads.norm(dim=-1, keepdim=True)

        # ========== Sample and evaluate updates based on gradients. ========== #
        adv_suffix_ids = utils.sample_updates(
            optim_ids,
            token_grads.squeeze(0),
            args.batch_size,
            topk=args.topk,
            not_allowed_tokens=not_allowed_tokens,
        )

        # filter out suffixes that do not tokenize back to the same ids
        decoded = tokenizer.batch_decode(adv_suffix_ids, skip_special_tokens=True)
        encoded = tokenizer(
            decoded, add_special_tokens=False, return_tensors="pt", padding=True
        ).input_ids.to("cuda")
        same_ids = torch.all(encoded[:, : optim_ids.shape[0]] == adv_suffix_ids, dim=1)
        adv_suffix_ids = adv_suffix_ids[same_ids]

        # update suffixes
        batch_dynamic_input_ids = dynamic_input_ids.repeat(adv_suffix_ids.shape[0], 1)
        batch_dynamic_input_ids[:, optim_slice] = adv_suffix_ids
        input_embeds = embed_layer(batch_dynamic_input_ids)

        # evaluate new suffixes
        with torch.no_grad():
            outputs = model(inputs_embeds=input_embeds)

        logits = outputs.logits

        # compute loss
        loss_logits = logits[:, loss_slice].permute(
            0, 2, 1
        )  # move class dim after batch
        batch_target_ids = target_ids.unsqueeze(0).repeat(adv_suffix_ids.shape[0], 1)
        loss = nn.functional.cross_entropy(
            loss_logits, batch_target_ids, reduction="none"
        )
        loss = loss.mean(dim=1)

        # ========== Update the suffix with the best candidate ========== #
        idx = loss.argmin()
        adv_suffix = tokenizer.decode(adv_suffix_ids[idx], skip_special_tokens=True)
        current_loss = loss[idx]
        num_cands = same_ids.sum().item()

        print(
            f"\n({i}/{args.num_steps}) num_cands={num_cands} loss={current_loss} "
            f"\nsuffix={adv_suffix}"
        )

        with open(log_file, "a") as f:
            f.write(
                json.dumps(
                    {
                        "loss": current_loss.item(),
                        "suffix": adv_suffix,
                    }
                )
                + "\n"
            )

        del same_ids


if __name__ == "__main__":
    args = parse_args()
    main(args)
