import copy
import functools
import logging
import os
import sys
from dataclasses import dataclass, field
from typing import Literal

import datasets
import torch
import transformers
from transformers import AutoModelForCausalLM, AutoTokenizer, Trainer

from llm_rules import get_conv_template

datasets.disable_caching()
logger = logging.getLogger(__name__)


@dataclass
class ModelArguments:
    model_name_or_path: str = field(
        default=None,
        metadata={
            "help": "Name to a huggingface native pretrained model or path to a model on disk."
        },
    )


@dataclass
class DataArguments:
    dataset_path: str = field(default="data/justask")
    conv_template_name: str = field(
        default="mistral",
        metadata={"help": "Name of the conversation template."},
    )


@dataclass
class TrainingArguments(transformers.TrainingArguments):
    lora: bool = field(default=False)
    lora_rank: int = field(default=8)
    lora_alpha: float = field(default=8)
    flash_attn: bool = field(default=True)
    optim: str = field(default="adamw_torch")
    model_max_length: int = field(
        default=4096,
        metadata={
            "help": "Maximum sequence length. Sequences will be right padded to this length (and possibly truncated)."
            "Enforcing a consistent max length ensures memory usage is constant and predictable."
        },
    )
    padding: Literal["max_length", "longest"] = field(
        default="longest",
        metadata={
            "help": "Padding strategy. If 'max_length', pads to `model_max_length` always; this might lead to some "
            "redundant compute. If 'longest', pads to the longest sequence in the batch, capped by `model_max_length`."
        },
    )
    resume_from_checkpoint: bool = field(
        default=False, metadata={"help": "If True, loads from last check point."}
    )


def format_and_tokenize(example, tokenizer, template_name):
    conv = get_conv_template(template_name)
    for message in example["messages"]:
        if message["role"] == "user":
            role = conv.roles[0]
        else:
            role = conv.roles[1]
        conv.append_message(role, message["content"])

    input_ids = tokenizer(conv.get_prompt()).input_ids
    example["input_ids"] = input_ids
    example["labels"] = copy.deepcopy(input_ids)

    return example


@dataclass
class DataCollator(object):
    tokenizer: transformers.PreTrainedTokenizer

    def __call__(self, instances):
        input_ids, labels = tuple(
            [torch.as_tensor(instance[key]) for instance in instances]
            for key in ("input_ids", "labels")
        )
        input_ids = torch.nn.utils.rnn.pad_sequence(
            input_ids, batch_first=True, padding_value=self.tokenizer.pad_token_id
        )
        labels = torch.nn.utils.rnn.pad_sequence(
            labels, batch_first=True, padding_value=-100
        )
        # When sequences are right padded, `attention_mask` is only useful for T5 training.
        attention_mask = input_ids.ne(self.tokenizer.pad_token_id).long()
        return dict(
            input_ids=input_ids,
            labels=labels,
            attention_mask=attention_mask,
        )


def main():
    os.environ["WANDB_PROJECT"] = "rules"

    parser = transformers.HfArgumentParser(
        (ModelArguments, DataArguments, TrainingArguments)
    )
    model_args, data_args, training_args = parser.parse_args_into_dataclasses()
    training_args.fsdp_config["activation_checkpointing"] = True

    logging.basicConfig(
        format="%(asctime)s - %(levelname)s - %(name)s - %(message)s",
        datefmt="%Y-%m-%d %H:%M:%S",
        handlers=[logging.StreamHandler(sys.stdout)],
    )
    log_level = training_args.get_process_log_level()
    logger.setLevel(log_level)
    datasets.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.set_verbosity(log_level)
    transformers.utils.logging.enable_default_handler()
    transformers.utils.logging.enable_explicit_format()

    # Log on each process a small summary
    logger.warning(
        f"Process rank: {training_args.local_rank}, device: {training_args.device}, n_gpu: {training_args.n_gpu}"
        + f" distributed training: {bool(training_args.local_rank != -1)}, 16-bits training: {training_args.fp16}"
    )
    logger.info(f"Model parameters {model_args}")
    logger.info(f"Data parameters {data_args}")
    logger.info(f"Training/evaluation parameters {training_args}")

    # Check for last checkpoint
    last_checkpoint = None
    if os.path.isdir(training_args.output_dir):
        last_checkpoint = transformers.trainer_utils.get_last_checkpoint(
            training_args.output_dir
        )
    checkpoint = training_args.resume_from_checkpoint or last_checkpoint
    if checkpoint is not None:
        logger.info(f"Checkpoint detected, resuming training at {checkpoint}.")

    # Load model on CPU to prevent upfront OOM.
    model: transformers.PreTrainedModel = AutoModelForCausalLM.from_pretrained(
        model_args.model_name_or_path,
        attn_implementation="flash_attention_2",
        use_cache=False,
    )

    tokenizer = AutoTokenizer.from_pretrained(
        model_args.model_name_or_path,
    )
    tokenizer.pad_token = tokenizer.eos_token

    all_datasets = []
    for f in os.scandir(data_args.dataset_path):
        ds = datasets.load_dataset("json", data_files=f.path, split="train")
        ds = ds.remove_columns(["id", "category", "params"])
        all_datasets.append(ds)

    dataset = datasets.concatenate_datasets(all_datasets)

    _format_and_tokenize = functools.partial(
        format_and_tokenize,
        tokenizer=tokenizer,
        template_name=data_args.conv_template_name,
    )

    dataset = dataset.map(_format_and_tokenize, remove_columns=["messages"])

    trainer = Trainer(
        model=model,
        tokenizer=tokenizer,
        args=training_args,
        train_dataset=dataset,
        data_collator=DataCollator(tokenizer),
    )

    trainer.train(resume_from_checkpoint=checkpoint)

    if trainer.is_fsdp_enabled:
        trainer.accelerator.state.fsdp_plugin.set_state_dict_type("FULL_STATE_DICT")

    trainer.save_state()
    trainer.save_model(training_args.output_dir)


if __name__ == "__main__":
    main()
