<?php

namespace Nitrapi\Services;

use Nitrapi\Nitrapi;
use Nitrapi\Services\CloudServers\Apps\AppManager;
use Nitrapi\Services\CloudServers\CloudServer;
use Nitrapi\Common\NitrapiObject;
use Nitrapi\Services\TaskManager\TaskManager;

abstract class Service extends NitrapiObject
{
    protected $api;

    protected $id;
    protected $location_id;
    protected $comment;
    protected $status;
    protected $status_code;
    protected $user_id;
    protected $username;
    protected $delete_date;
    protected $suspend_date;
    protected $start_date;
    protected $auto_extension;
    protected $auto_extension_duration;
    protected $details;
    protected $readonly;
    protected $websocket_token;
    protected $roles;
    protected $arguments;
    protected $servicetype;
    protected $is_owner;

    protected static $ensureActiveService = true;

    const SERVICE_STATUS_INSTALLING = 'installing';
    const SERVICE_STATUS_ACTIVE = 'active';
    const SERVICE_STATUS_SUSPENDED = 'suspended';
    const SERVICE_STATUS_DELETED = 'deleted';
    const SERVICE_STATUS_ADMINLOCKED = 'adminlocked';
    const SERVICE_STATUS_ADMINLOCKED_SUSPENDED = 'adminlocked_suspended';

    public function __construct(Nitrapi $api, array &$data) {
        parent::__construct($api);
        $this->loadData($data);
    }

     /**
     * Return the name of the service as a string.
     *
     * @return string Name of the service
     */
    public function __toString() {
        return strtolower(substr(get_class($this), (int)strrpos(get_class($this), '\\') + 1));
    }

    /**
     * Return a list of arguments for the service (also called service args).
     * This arguments store additional information like the startgame or the
     * ordered amount.
     *
     * @return array list of arguments
     */
    public function getArguments() {
        return $this->arguments;
    }

    /**
     * Returns the current location id
     *
     * @return int
     */
    public function getLocationId() {
        return $this->location_id;
    }

    /**
     * Returns the service status
     *
     * @return mixed
     */
    public function getStatus() {
        return $this->status;
    }

    /**
     * Returns a numeric status code
     *
     * @return integer
     */
    public function getStatusCode(): int {
        return $this->status_code;
    }

    /**
     * Returns if the webinterface is locked
     *
     * @return mixed
     */
    public function isReadonly() {
        return $this->readonly;
    }

    /**
     * Some actions do not need the check for active state, like deleting
     * the service. For this case, the active state check can be disabled.
     * If enabled, a Exception will be thrown if the service is not active
     * any more.
     *
     * Example:
     * Service::forceAction(function() use ($nitrapi, $serviceId) {
     *     $nitrapi->getService($serviceId)->doDelete();
     * });
     *
     * @see Gameserver::refresh()
     * @param $fn
     * @return null|mixed The return value from the given lambda fn
     */
    public static function forceAction($fn) {
        $res = null;

        self::$ensureActiveService = false;
        $res = $fn();
        self::$ensureActiveService = true;

        return $res;
    }

    /**
     * Returns if the service is currently installing.
     * @return bool
     */
    public function isInstalling() {
        return $this->getStatus() === self::SERVICE_STATUS_INSTALLING;
    }

    /**
     * Returns if the service is currently active.
     * @return bool
     */
    public function isActive() {
        return $this->getStatus() === self::SERVICE_STATUS_ACTIVE;
    }

    /**
     * Returns if the service is currently suspended.
     * @return bool
     */
    public function isSuspended() {
        return $this->getStatus() === self::SERVICE_STATUS_SUSPENDED;
    }

    /**
     * Returns if the service is currently deleted.
     * @return bool
     */
    public function isDeleted() {
        return $this->getStatus() === self::SERVICE_STATUS_DELETED;
    }

    /**
     * Returns if the service is currently admin locked.
     * @return bool
     */
    public function isAdminLocked() {
        return $this->getStatus() === self::SERVICE_STATUS_ADMINLOCKED;
    }

    /**
     * Returns if the service is currently admin locked and suspended.
     * @return bool
     */
    public function isAdminLockedSuspended() {
        return $this->getStatus() === self::SERVICE_STATUS_ADMINLOCKED_SUSPENDED;
    }

    /**
     * Services are rented for a predefined duration. When this duration is
     * over, the service changes the state from active to suspended. To prevent
     * a service from going into suspended mode, the auto extension feature
     * can be enabled to renew the rental time if the duration is over. For this,
     * the account need sufficient money to renew the rental time.
     *
     * @return bool True if the service has the auto extension enabled
     */
    public function isAutoExtensionEnabled() {
        return (bool)$this->auto_extension;
    }

    /**
     * If the service is renewed from the auto extension feature, it will renew
     * for this amount of time. This duration can be set on the website.
     *
     * @return int The duration for the auto extension feature
     */
    public function getAutoExtensionDuration() {
        return $this->auto_extension_duration;
    }

    /**
     * Return the service comment
     *
     * @return mixed
     */
    public function getComment() {
        return $this->comment;
    }

    /**
     * Returns the suspend date
     *
     * @return \DateTime
     */
    public function getSuspendDate() {
        $datetime = new \DateTime();
        $datetime->setTimestamp(strtotime($this->suspend_date));
        return $datetime;
    }

    /**
     * Returns the delete date
     *
     * @return \DateTime
     */
    public function getDeleteDate() {
        $datetime = new \DateTime();
        $datetime->setTimestamp(strtotime($this->delete_date));
        return $datetime;
    }

    /**
     * Returns the start date
     *
     * @return \DateTime
     */
    public function getStartDate() {
        $datetime = new \DateTime();
        $datetime->setTimestamp(strtotime($this->start_date));
        return $datetime;
    }

    /**
     * Returns the service id
     *
     * @return int
     */
    public function getId() {
        return (int)$this->id;
    }

    /**
     * Returns the user id of the service
     *
     * @return int
     */
    public function getUserId() {
        return (int)$this->user_id;
    }

    /**
     * Returns the username
     *
     * @return string
     */
    public function getUsername() {
        return (string)$this->username;
    }

    /**
     * Returns the websocket token
     *
     * @return string
     */
    public function getWebsocketToken() {
        return (string)$this->websocket_token;
    }

    /**
     * Returns all service details
     *
     * @return array
     */
    public function getServiceDetails() {
        return (array)$this->details;
    }

    /**
     * Returns the roles of the service
     *
     * @return array
     */
    public function getRoles() {
        return (array)$this->roles;
    }

    /**
     * Returns the ddos history
     *
     * @return array
     */
    public function getDDoSHistory() {
        $url = "services/" . $this->getId() . "/ddos";
        return $this->getApi()->dataGet($url);
    }

    /**
     * Lists all changelog items, which are available for the service.
     * A changelog item consists of a message with additional information.
     * This items will be used to show game updates and other notification
     * type messages to the service. A changelog item has the following
     * attributes:
     *
     * category     Which category the item will be (default "Game")
     * created      When the item is created
     * status
     *      name    What status the item has (e.g. "Update")
     *      icon    Icon name to use for display purpose
     *      button_class CSS Class to use for display purpose
     * text         The actual message to display
     * game         The full game name
     * alert        If the message is an alert message
     *
     * If the first parameter is set to true, changelog items for all
     * games will be returned. With the second parameter you can
     * suppress all non alert messages.
     *
     * @param boolean true if you need items from all games
     * @param boolean true if only alerts should be shown
     *
     * @return array a list of changelog items
     */
    public function getChangelogItems($allGames = false, $onlyAlerts = false) {
        $changelogs = $this->getApi()->dataGet('/changelogs');
        if (!isset($changelogs['changelogs'])) return [];

        $filteredChangelogs = $changelogs['changelogs'];

        // Filter out all non current game related items
        if (!$allGames) {
            $details = $this->getServiceDetails();
            if (!isset($details['game'])) return [];
            foreach ($filteredChangelogs as $i => $changelog)
                if (!isset($changelog['game']) || $changelog['game'] != $details['game'])
                    unset($filteredChangelogs[$i]);
        }

        // Filter out all normal items (only alerts)
        if ($onlyAlerts)
            foreach ($filteredChangelogs as $i => $changelog)
                if (!$changelog['alert']) unset($filteredChangelogs[$i]);

        return $filteredChangelogs;
    }

    /**
     * Returns the last log entries. You can optionally
     * provide a page number.
     *
     * @param int $hours
     * @return array
     */
    public function getLogs($page = 1) {
        $url = "services/" . $this->getId() . "/logs";
        return $this->getApi()->dataGet($url, null, [
            'query' => [
                'page' => $page
            ]
        ]);
    }

    /**
     * Adds a new log entry to your service
     *
     * @param string $category
     * @param string $message
     * @param string $severity
     * @return boolean
     */
    public function addLog($category, $message, $severity = 'info') {
        $url = "services/" . $this->getId() . "/logs";
        $this->getApi()->dataPost($url, [
            'category' => $category,
            'message' => $message,
            'severity' => $severity,
        ]);
        return true;
    }

    /**
     * This can be used to force delete a suspended service.
     *
     * @return boolean
     */
    public function doDelete() {
        $url = "services/" . $this->getId();
        $this->getApi()->dataDelete($url);
        return true;
    }

    /**
     * Returns the sale price for the specified service.
     *
     * @return integer
     */
    public function getSalePrice() {
        $url = "services/" . $this->getId() . "/sale_price";
        return $this->getApi()->dataGet($url)['sale_price']['price'];
    }

    /**
     * This can be used to force suspend a service.
     * The sale price will be added as credit to your account.
     *
     * @return boolean
     */
    public function doCancel() {
        $url = "services/" . $this->getId() . "/cancel";
        $this->getApi()->dataPost($url);
        return true;
    }

    /**
     * Returns the task manager
     *
     * @return TaskManager
     */
    public function getTaskManager() {
        return new TaskManager($this);
    }

    /**
     * Returns the app manager.
     *
     * @return AppManager|bool
     */
    public function getAppManager() {
        if ($this instanceof CloudServer)
            return new AppManager($this);
        return false;
    }

    /**
     * @param array $data
     */
    protected function loadData(array $data) {
        $reflectionClass = new \ReflectionClass($this);
        $properties = $reflectionClass->getProperties();

        foreach ($properties as $property) {
            if (!isset($data[$property->getName()])) continue;
            if (!$property->isProtected()) continue;
            $value = $data[$property->getName()];
            if (empty($value)) continue;

            $property->setAccessible(true);
            $property->setValue($this, $value);
            $property->setAccessible(false);
        }
    }

    /**
     * Returns all available domains to register subdomains to
     *
     * @return array
     */
    public function getDomains(): array {
        return $this->getApi()->dataGet("services/" . $this->getId() . "/subdomain/domains");
    }

    /**
     * Returns the currently set subdomain
     *
     * @return array ["subdomain": "mysubdomain", "domain": "nitrado.net", "fqdn": "mysubdomain.nitrado.net", "domain_id": 5]
     */
    public function getSubdomain(): array {
        return $this->getApi()->dataGet("services/" . $this->getId() . "/subdomain");
    }

    /**
     * Deletes the currently set subdomain
     *
     * @return string
     */
    public function deleteSubdomain(): string {
        return $this->getApi()->dataDelete("services/" . $this->getId() . "/subdomain");
    }

    /**
     * Sets a new subdomain
     *
     * @return string
     */
    public function setSubdomain(string $subdomain, int $domain_id): string {
        $data = [
          "subdomain" => $subdomain,
          "domain_id" => $domain_id
        ];

        return $this->getApi()->dataPut("services/" . $this->getId() . "/subdomain", $data);
    }

    /**
     * Returns the servicetype
     *
     * @return integer
     */
    public function getServicetype(): int {
        return (int) $this->servicetype;
    }

    /**
     *  Returns is the service belongs to the calling user
     *
     * @return bool
     */
    public function isOwnService(): bool {
        return (bool) $this->is_owner;
    }

    public function getSupportAuthorization() {
        return null;
    }

    public function createSupportAuthorization() {
        throw new NitrapiErrorException("Support Authorizations aren't supported for this service type.");
    }

    public function deleteSupportAuthorization() {
        throw new NitrapiErrorException("Support Authorizations aren't supported for this service type.");
    }
}
