import { create } from 'zustand';

import { getItem, removeItem, setItem } from '@/utils/storage';

import { StorageEnum, ThemeColorPresets, ThemeLayout, ThemeMode } from '#/enum';

type SettingsType = {
  themeColorPresets: ThemeColorPresets;
  themeMode: ThemeMode;
  themeLayout: ThemeLayout;
  themeStretch: boolean;
  breadCrumb: boolean;
  multiTab: boolean;
  captchaSiteKey?: string;
  taskStatus?: boolean;
};
type SettingStore = {
  settings: SettingsType;
  // 使用 actions 命名空间来存放所有的 action
  actions: {
    setTaskStatus: (taskStatus: boolean) => void;
    setCaptchaSiteKey: (captchaSiteKey: string) => void;
    setSettings: (settings: SettingsType) => void;
    clearSettings: () => void;
  };
};

const useSettingStore = create<SettingStore>((set) => ({
  settings: getItem<SettingsType>(StorageEnum.Settings) || {
    themeColorPresets: ThemeColorPresets.Default,
    themeMode: ThemeMode.Light,
    themeLayout: ThemeLayout.Vertical,
    themeStretch: false,
    breadCrumb: true,
    multiTab: true,
    taskStatus: false,
  },
  actions: {
    setTaskStatus: (taskStatus) => {
      set((state) => ({
        settings: {
          ...state.settings,
          taskStatus,
        },
      }));
    },
    setCaptchaSiteKey: (captchaSiteKey) => {
      set((state) => ({
        settings: {
          ...state.settings,
          captchaSiteKey,
        },
      }));
    },
    setSettings: (settings) => {
      set({ settings });
      setItem(StorageEnum.Settings, settings);
    },
    clearSettings() {
      removeItem(StorageEnum.Settings);
    },
  },
}));

export const useCaptchaSiteKey = () => useSettingStore((state) => state.settings.captchaSiteKey);
export const useSettings = () => useSettingStore((state) => state.settings);
export const useSettingActions = () => useSettingStore((state) => state.actions);
