// [x86data.js]
// X86/X64 instruction-set data.
//
// [License]
// Public Domain.


// This file can be parsed as pure JSON, locate ${JSON:BEGIN} and ${JSON:END}
// marks and strip everything outside, a sample JS function that would do the job:
//
// function strip(s) {
//   return s.replace(/(^.*\$\{JSON:BEGIN\}\s+)|(\/\/\s*\$\{JSON:END\}\s*.*$)/g, "");
// }


// INSTRUCTIONS
// ============
//
// Each instruction definition consists of 5 strings:
//
//   [0] - Instruction name.
//   [1] - Instruction operands.
//   [2] - Instruction encoding.
//   [3] - Instruction opcode.
//   [4] - Instruction metadata - CPU features, FLAGS (read/write), and other metadata.
//
// The definition tries to match Intel and AMD instruction set manuals, but there
// are small differences to make the definition more informative and compact.


// OPERANDS
// ========
//
//   * "op"    - Explicit operand, must always be part of the instruction. If a fixed
//               register (like "cl") is used, it means that the instruction uses this
//               register implicitly, but it must be specified anyway.
//
//   * "<op>"  - Implicit operand - some assemblers allow implicit operands the be passed
//               explicitly for documenting purposes. And some assemblers like AsmJit's
//               Compiler infrastructure requires implicit operands to be passed explicitly
//               for register allocation purposes.
//
//   * "{op}"  - Optional operand. Mostly used by AVX_512:
//
//               - {k} mask selector.
//               - {z} zeroing.
//               - {1tox} broadcast.
//               - {er} embedded-rounding.
//               - {sae} suppress-all-exceptions.
//
//   * "?:Op"  - Each operand can provide metadata that can be used to describe which
//               operands are used as a destination, and which operands are source-only.
//               Each instruction in general assumes that the first operand is always
//               read/write and all following operands are read-only. However, this is
//               not correct for all instructions, thus, instructions that don't match
//               this assumption must provide additional information:
//
//               - "R:Op" - The operand is read-only.
//               - "w:Op" - The operand is write-only (does not zero-extend).
//               - "W:Op" - The operand is write-only (implicit zero-extend).
//               - "x:Op" - The operand is read/write (does not zero-extend).
//               - "X:Op" - The operand is read/write (implicit zero-extend).
//
//   * Op[A:B] - Optional bit-range that describes which bits are read and written.
//
//   * "~Op"   - Operand is commutative with other operands prefixed by "~". Commutativity
//               means that all operands marked by '~' can be swapped and the result of the
//               instruction would be the same.

// WHAT IS MISSING
// ===============
//
// Here is a list of missing instructions to keep track of it:
//
// [ ] call/jmp/ret (far)
// [ ] xlat/xlatb

// ~ modifications:
//  - uncommented call/jmp/ret far instructions, they seem to be alright

(function($export, $as) {
    "use strict";
    
    $export[$as] =
    // ${JSON:BEGIN}
    {
      "architectures": [
        "ANY",
        "X86",
        "X64"
      ],
    
      "extensions": [
        { "name": "3DNOW"               },
        { "name": "3DNOW2"              },
        { "name": "ADX"                 },
        { "name": "AESNI"               },
        { "name": "AVX"                 },
        { "name": "AVX2"                },
        { "name": "AVX512_4FMAPS"       },
        { "name": "AVX512_4VNNIW"       },
        { "name": "AVX512_BF16"         },
        { "name": "AVX512_BITALG"       },
        { "name": "AVX512_BW"           },
        { "name": "AVX512_CDI"          },
        { "name": "AVX512_DQ"           },
        { "name": "AVX512_ERI"          },
        { "name": "AVX512_F"            },
        { "name": "AVX512_IFMA"         },
        { "name": "AVX512_PFI"          },
        { "name": "AVX512_VBMI"         },
        { "name": "AVX512_VBMI2"        },
        { "name": "AVX512_VNNI"         },
        { "name": "AVX512_VL"           },
        { "name": "AVX512_VP2INTERSECT" },
        { "name": "AVX512_VPOPCNTDQ"    },
        { "name": "BMI"                 },
        { "name": "BMI2"                },
        { "name": "CLDEMOTE"            },
        { "name": "CLFLUSH"             },
        { "name": "CLFLUSHOPT"          },
        { "name": "CLWB"                },
        { "name": "CLZERO"              },
        { "name": "CMOV"                },
        { "name": "CMPXCHG8B"           },
        { "name": "CMPXCHG16B"          },
        { "name": "ENCLV"               },
        { "name": "ENQCMD"              },
        { "name": "F16C"                },
        { "name": "FMA"                 },
        { "name": "FMA4"                },
        { "name": "FSGSBASE"            },
        { "name": "FXSR"                },
        { "name": "GEODE"               },
        { "name": "HLE"                 },
        { "name": "GFNI"                },
        { "name": "I486"                },
        { "name": "LAHFSAHF"            },
        { "name": "LWP"                 },
        { "name": "LZCNT"               },
        { "name": "MMX"                 },
        { "name": "MMX2"                },
        { "name": "MONITOR"             },
        { "name": "MONITORX"            },
        { "name": "MOVBE"               },
        { "name": "MOVDIR64B"           },
        { "name": "MOVDIRI"             },
        { "name": "MPX"                 },
        { "name": "MSR"                 },
        { "name": "PCLMULQDQ"           },
        { "name": "PCOMMIT"             },
        { "name": "PCONFIG"             },
        { "name": "POPCNT"              },
        { "name": "PREFETCHW"           },
        { "name": "PREFETCHWT1"         },
        { "name": "RDPID"               },
        { "name": "RDRAND"              },
        { "name": "RDSEED"              },
        { "name": "RDTSC"               },
        { "name": "RDTSCP"              },
        { "name": "RTM"                 },
        { "name": "SHA"                 },
        { "name": "SKINIT"              },
        { "name": "SMAP"                },
        { "name": "SMX"                 },
        { "name": "SSE"                 },
        { "name": "SSE2"                },
        { "name": "SSE3"                },
        { "name": "SSE4_1"              },
        { "name": "SSE4_2"              },
        { "name": "SSE4A"               },
        { "name": "SSSE3"               },
        { "name": "SVM"                 },
        { "name": "TBM"                 },
        { "name": "TSX"                 },
        { "name": "VAES"                },
        { "name": "VPCLMULQDQ"          },
        { "name": "VMX"                 },
        { "name": "WAITPKG"             },
        { "name": "WBNOINVD"            },
        { "name": "XOP"                 },
        { "name": "XSAVE"               },
        { "name": "XSAVEC"              },
        { "name": "XSAVEOPT"            },
        { "name": "XSAVES"              }
      ],
    
      "attributes": [
        { "name": "Control"          , "type": "string"      , "doc": "Describes control flow." },
        { "name": "Volatile"         , "type": "flag"        , "doc": "Instruction can have side effects (hint for instruction scheduler)." },
        { "name": "Deprecated"       , "type": "flag"        , "doc": "Deprecated instruction." },
    
        { "name": "AltForm"          , "type": "flag"        , "doc": "Alternative form that is shorter, but has restrictions." },
        { "name": "Lock"             , "type": "flag"        , "doc": "Can be used with LOCK prefix." },
        { "name": "ImplicitLock"     , "type": "flag"        , "doc": "Instruction is always atomic, regardless of use of the LOCK prefix." },
        { "name": "XAcquire"         , "type": "flag"        , "doc": "A hint used to start lock elision on the instruction memory operand address." },
        { "name": "XRelease"         , "type": "flag"        , "doc": "A hint used to end lock elision on the instruction memory operand address." },
    
        { "name": "REP"              , "type": "flag"        , "doc": "Can be used with REP (REPE/REPZ) prefix." },
        { "name": "REPNE"            , "type": "flag"        , "doc": "Can be used with REPNE (REPNZ) prefix." },
        { "name": "RepIgnored"       , "type": "flag"        , "doc": "Rep prefix can be used, but has no effect." },
    
        { "name": "AliasOf"          , "type": "string"      , "doc": "Instruction is an alias to another instruction, must apply to all instructions within the same group." },
        { "name": "EncodeAs"         , "type": "string"      , "doc": "Similar to AliasOf, but doesn't apply to all instructions in the group." }
      ],
    
      "specialRegs": [
        { "name": "FLAGS.CF"         , "group": "FLAGS.CF"   , "doc": "Carry flag." },
        { "name": "FLAGS.PF"         , "group": "FLAGS.PF"   , "doc": "Parity flag." },
        { "name": "FLAGS.AF"         , "group": "FLAGS.AF"   , "doc": "Adjust flag." },
        { "name": "FLAGS.ZF"         , "group": "FLAGS.ZF"   , "doc": "Zero flag." },
        { "name": "FLAGS.SF"         , "group": "FLAGS.SF"   , "doc": "Sign flag." },
        { "name": "FLAGS.TF"         , "group": "FLAGS.TF"   , "doc": "Trap flag." },
        { "name": "FLAGS.IF"         , "group": "FLAGS.IF"   , "doc": "Interrupt enable flag." },
        { "name": "FLAGS.DF"         , "group": "FLAGS.DF"   , "doc": "Direction flag." },
        { "name": "FLAGS.OF"         , "group": "FLAGS.OF"   , "doc": "Overflow flag." },
        { "name": "FLAGS.AC"         , "group": "FLAGS.Other", "doc": "Alignment check flag." },
        { "name": "FLAGS.IOPL"       , "group": "FLAGS.Other", "doc": "I/O privilege level." },
        { "name": "FLAGS.NT"         , "group": "FLAGS.Other", "doc": "Nested task flag." },
        { "name": "FLAGS.RF"         , "group": "FLAGS.Other", "doc": "Resume flag." },
        { "name": "FLAGS.VM"         , "group": "FLAGS.Other", "doc": "Virtual 8086 mode flag." },
        { "name": "FLAGS.VIF"        , "group": "FLAGS.Other", "doc": "Virtual interrupt flag." },
        { "name": "FLAGS.VIP"        , "group": "FLAGS.Other", "doc": "Virtual interrupt pending." },
        { "name": "FLAGS.CPUID"      , "group": "FLAGS.Other", "doc": "CPUID instruction available." },
    
        { "name": "X87CW.INVALID_OP" , "group": "X87CW.EXC"  , "doc": "Invalid operation exception enable bit." },
        { "name": "X87CW.DENORMAL"   , "group": "X87CW.EXC"  , "doc": "Dernormalized exception enable bit." },
        { "name": "X87CW.ZERO_DIVIDE", "group": "X87CW.EXC"  , "doc": "Division by zero exception enable bit." },
        { "name": "X87CW.OVERFLOW"   , "group": "X87CW.EXC"  , "doc": "Overflow exception enable bit." },
        { "name": "X87CW.UNDERFLOW"  , "group": "X87CW.EXC"  , "doc": "Underflow exception enable bit." },
        { "name": "X87CW.PRECISION"  , "group": "X87CW.EXC"  , "doc": "Lost of precision exception enable bit." },
        { "name": "X87CW.PC"         , "group": "X87CW.PC"   , "doc": "Precision control." },
        { "name": "X87CW.RC"         , "group": "X87CW.RC"   , "doc": "Rounding control." },
    
        { "name": "X87SW.INVALID_OP" , "group": "X87SW.EXC"  , "doc": "Invalid operation exception flag." },
        { "name": "X87SW.DENORMAL"   , "group": "X87SW.EXC"  , "doc": "Dernormalized exception flag." },
        { "name": "X87SW.ZERO_DIVIDE", "group": "X87SW.EXC"  , "doc": "Division by zero exception flag." },
        { "name": "X87SW.OVERFLOW"   , "group": "X87SW.EXC"  , "doc": "Overflow exception flag." },
        { "name": "X87SW.UNDERFLOW"  , "group": "X87SW.EXC"  , "doc": "Underflow exception flag." },
        { "name": "X87SW.PRECISION"  , "group": "X87SW.EXC"  , "doc": "Lost of precision exception flag." },
        { "name": "X87SW.STACK_FAULT", "group": "X87SW.EXC"  , "doc": "Stack fault." },
        { "name": "X87SW.EF"         , "group": "X87SW.EXC"  , "doc": "Exception flag." },
        { "name": "X87SW.C0"         , "group": "X87SW.C0"   , "doc": "C0 condifion." },
        { "name": "X87SW.C1"         , "group": "X87SW.C1"   , "doc": "C1 condifion." },
        { "name": "X87SW.C2"         , "group": "X87SW.C2"   , "doc": "C2 condifion." },
        { "name": "X87SW.TOP"        , "group": "X87SW.TOP"  , "doc": "Top of the FPU stack." },
        { "name": "X87SW.C3"         , "group": "X87SW.C3"   , "doc": "C3 condifion." },
    
        { "name": "MSR"              , "group": "MSR"        , "doc": "MSR register." },
        { "name": "XCR"              , "group": "XCR"        , "doc": "XCR register." }
      ],
    
      "shortcuts": [
        { "name": "CF"               , "expand": "FLAGS.CF" },
        { "name": "PF"               , "expand": "FLAGS.PF" },
        { "name": "AC"               , "expand": "FLAGS.AC" },
        { "name": "AF"               , "expand": "FLAGS.AF" },
        { "name": "ZF"               , "expand": "FLAGS.ZF" },
        { "name": "SF"               , "expand": "FLAGS.SF" },
        { "name": "TF"               , "expand": "FLAGS.TF" },
        { "name": "IF"               , "expand": "FLAGS.IF" },
        { "name": "DF"               , "expand": "FLAGS.DF" },
        { "name": "OF"               , "expand": "FLAGS.OF" },
        { "name": "C0"               , "expand": "X87SW.C0" },
        { "name": "C1"               , "expand": "X87SW.C1" },
        { "name": "C2"               , "expand": "X87SW.C2" },
        { "name": "C3"               , "expand": "X87SW.C3" },
    
        { "name": "_ILock"           , "expand": "Lock|ImplicitLock" },
        { "name": "_XLock"           , "expand": "Lock|XAcquire|XRelease" },
        { "name": "_Rep"             , "expand": "REP|REPNE" },
        { "name": "_RepIgnored"      , "expand": "REP|REPNE|RepIgnored" }
      ],
    
      "registers": {
        "r8"  : { "kind": "gp"  , "any": "r8"   , "names": ["al", "cl", "dl", "bl", "spl", "bpl", "sil", "dil", "r8-15b"] },
        "r8hi": { "kind": "gp"                  , "names": ["ah", "ch", "dh", "bh"] },
        "r16" : { "kind": "gp"  , "any": "r16"  , "names": ["ax", "cx", "dx", "bx", "sp", "bp", "si", "di", "r8-15w"] },
        "r32" : { "kind": "gp"  , "any": "r32"  , "names": ["eax", "ecx", "edx", "ebx", "esp", "ebp", "esi", "edi", "r8-15d"] },
        "r64" : { "kind": "gp"  , "any": "r64"  , "names": ["rax", "rcx", "rdx", "rbx", "rsp", "rbp", "rsi", "rdi", "r8-15"] },
        "rxx" : { "kind": "gp"                  , "names": ["zax", "zcx", "zdx", "zbx", "zsp", "zbp", "zsi", "zdi"] },
        "sreg": { "kind": "sreg", "any": "sreg" , "names": ["es", "cs", "ss", "ds", "fs", "gs" ] },
        "creg": { "kind": "creg", "any": "creg" , "names": ["cr0-15"]  },
        "dreg": { "kind": "dreg", "any": "dreg" , "names": ["dr0-15"]  },
        "bnd" : { "kind": "bnd" , "any": "bnd"  , "names": ["bnd0-3"]  },
        "st"  : { "kind": "st"  , "any": "st(i)", "names": ["st(0-7)"] },
        "mm"  : { "kind": "mm"  , "any": "mm"   , "names": ["mm0-7"]   },
        "k"   : { "kind": "k"   , "any": "k"    , "names": ["k0-7"]    },
        "xmm" : { "kind": "vec" , "any": "xmm"  , "names": ["xmm0-31"] },
        "ymm" : { "kind": "vec" , "any": "ymm"  , "names": ["ymm0-31"] },
        "zmm" : { "kind": "vec" , "any": "zmm"  , "names": ["zmm0-31"] }
      },
    
      "instructions": [
        ["aaa"              , "x:<ax>"                                          , "NONE"    , "37"                           , "X86 Deprecated   OF=U SF=U ZF=U AF=W PF=U CF=W"],
        ["aas"              , "x:<ax>"                                          , "NONE"    , "3F"                           , "X86 Deprecated   OF=U SF=U ZF=U AF=W PF=U CF=W"],
        ["aad"              , "x:<ax>, ib/ub"                                   , "I"       , "D5 ib"                        , "X86 Deprecated   OF=U SF=W ZF=W AF=U PF=W CF=U"],
        ["aam"              , "x:<ax>, ib/ub"                                   , "I"       , "D4 ib"                        , "X86 Deprecated   OF=U SF=W ZF=W AF=U PF=W CF=U"],
    
        ["adc"              , "x:al, ib/ub"                                     , "I"       , "14 ib"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["adc"              , "x:ax, iw/uw"                                     , "I"       , "66 15 iw"                     , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["adc"              , "X:eax, id/ud"                                    , "I"       , "15 id"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["adc"              , "X:rax, id"                                       , "I"       , "REX.W 15 id"                  , "X64 AltForm      OF=W SF=W ZF=W AF=W PF=W CF=X"],
    
        ["adc"              , "x:r8/m8, ib/ub"                                  , "MI"      , "80 /2 ib"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["adc"              , "x:r16/m16, iw/uw"                                , "MI"      , "66 81 /2 iw"                  , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["adc"              , "X:r32/m32, id/ud"                                , "MI"      , "81 /2 id"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["adc"              , "X:r64/m64, id"                                   , "MI"      , "REX.W 81 /2 id"               , "X64 _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
    
        ["adc"              , "x:r16/m16, ib"                                   , "MI"      , "66 83 /2 ib"                  , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["adc"              , "X:r32/m32, ib"                                   , "MI"      , "83 /2 ib"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["adc"              , "X:r64/m64, ib"                                   , "MI"      , "REX.W 83 /2 ib"               , "X64 _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
    
        ["adc"              , "x:~r8/m8,~r8"                                    , "MR"      , "10 /r"                        , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["adc"              , "x:~r16/m16,~r16"                                 , "MR"      , "66 11 /r"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["adc"              , "X:~r32/m32,~r32"                                 , "MR"      , "11 /r"                        , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["adc"              , "X:~r64/m64,~r64"                                 , "MR"      , "REX.W 11 /r"                  , "X64 _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
    
        ["adc"              , "x:~r8,~r8/m8"                                    , "RM"      , "12 /r"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["adc"              , "x:~r16,~r16/m16"                                 , "RM"      , "66 13 /r"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["adc"              , "X:~r32,~r32/m32"                                 , "RM"      , "13 /r"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["adc"              , "X:~r64,~r64/m64"                                 , "RM"      , "REX.W 13 /r"                  , "X64              OF=W SF=W ZF=W AF=W PF=W CF=X"],
    
        ["add"              , "x:al, ib/ub"                                     , "I"       , "04 ib"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["add"              , "x:ax, iw/uw"                                     , "I"       , "66 05 iw"                     , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["add"              , "X:eax, id/ud"                                    , "I"       , "05 id"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["add"              , "X:rax, id"                                       , "I"       , "REX.W 05 id"                  , "X64 AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["add"              , "x:r8/m8, ib/ub"                                  , "MI"      , "80 /0 ib"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["add"              , "x:r16/m16, iw/uw"                                , "MI"      , "66 81 /0 iw"                  , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["add"              , "X:r32/m32, id/ud"                                , "MI"      , "81 /0 id"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["add"              , "X:r64/m64, id"                                   , "MI"      , "REX.W 81 /0 id"               , "X64 _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["add"              , "x:r16/m16, ib"                                   , "MI"      , "66 83 /0 ib"                  , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["add"              , "X:r32/m32, ib"                                   , "MI"      , "83 /0 ib"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["add"              , "X:r64/m64, ib"                                   , "MI"      , "REX.W 83 /0 ib"               , "X64 _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["add"              , "x:~r8/m8,~r8"                                    , "MR"      , "00 /r"                        , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["add"              , "x:~r16/m16,~r16"                                 , "MR"      , "66 01 /r"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["add"              , "X:~r32/m32,~r32"                                 , "MR"      , "01 /r"                        , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["add"              , "X:~r64/m64,~r64"                                 , "MR"      , "REX.W 01 /r"                  , "X64 _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["add"              , "x:~r8,~r8/m8"                                    , "RM"      , "02 /r"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["add"              , "x:~r16,~r16/m16"                                 , "RM"      , "66 03 /r"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["add"              , "X:~r32,~r32/m32"                                 , "RM"      , "03 /r"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["add"              , "X:~r64,~r64/m64"                                 , "RM"      , "REX.W 03 /r"                  , "X64              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["and"              , "x:al, ib/ub"                                     , "I"       , "24 ib"                        , "ANY AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "x:ax, iw/uw"                                     , "I"       , "66 25 iw"                     , "ANY AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "X:eax, id/ud"                                    , "I"       , "25 id"                        , "ANY AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "X:rax, ud"                                       , "I"       , "25 id"                        , "X64 AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "X:rax, id"                                       , "I"       , "REX.W 25 id"                  , "X64 AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["and"              , "x:r8/m8, ib/ub"                                  , "MI"      , "80 /4 ib"                     , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "x:r16/m16, iw/uw"                                , "MI"      , "66 81 /4 iw"                  , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "X:r32/m32, id/ud"                                , "MI"      , "81 /4 id"                     , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "X:r64, ud"                                       , "MI"      , "81 /4 id"                     , "X64 _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "X:r64/m64, id"                                   , "MI"      , "REX.W 81 /4 id"               , "X64 _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["and"              , "x:r16/m16, ib/ub"                                , "MI"      , "66 83 /4 ib"                  , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "X:r32/m32, ib/ub"                                , "MI"      , "83 /4 ib"                     , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "X:r64/m64, ib/ub"                                , "MI"      , "REX.W 83 /4 ib"               , "X64 _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["and"              , "x:~r8/m8,~r8"                                    , "MR"      , "20 /r"                        , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "x:~r16/m16,~r16"                                 , "MR"      , "66 21 /r"                     , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "X:~r32/m32,~r32"                                 , "MR"      , "21 /r"                        , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "X:~r64/m64,~r64"                                 , "MR"      , "REX.W 21 /r"                  , "X64 _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["and"              , "x:~r8,~r8/m8"                                    , "RM"      , "22 /r"                        , "ANY              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "x:~r16,~r16/m16"                                 , "RM"      , "66 23 /r"                     , "ANY              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "X:~r32,~r32/m32"                                 , "RM"      , "23 /r"                        , "ANY              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["and"              , "X:~r64,~r64/m64"                                 , "RM"      , "REX.W 23 /r"                  , "X64              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["arpl"             , "x:r16/m16, R:r16"                                , "MR"      , "63 /r"                        , "X86 ZF=W"],
    
        ["bndcl"            , "R:bnd, r32/m32"                                  , "RM"      , "F3 0F 1A /r"                  , "MPX X86"],
        ["bndcl"            , "R:bnd, r64/m64"                                  , "RM"      , "F3 0F 1A /r"                  , "MPX X64"],
        ["bndcn"            , "R:bnd, r32/m32"                                  , "RM"      , "F2 0F 1B /r"                  , "MPX X86"],
        ["bndcn"            , "R:bnd, r64/m64"                                  , "RM"      , "F2 0F 1B /r"                  , "MPX X64"],
        ["bndcu"            , "R:bnd, r32/m32"                                  , "RM"      , "F2 0F 1A /r"                  , "MPX X86"],
        ["bndcu"            , "R:bnd, r64/m64"                                  , "RM"      , "F2 0F 1A /r"                  , "MPX X64"],
        ["bndldx"           , "W:bnd, mib"                                      , "RM"      , "0F 1A /r"                     , "MPX"],
        ["bndmk"            , "W:bnd, mem"                                      , "RM"      , "F3 0F 1B /r"                  , "MPX"],
        ["bndmov"           , "W:bnd, bnd/mem"                                  , "RM"      , "66 0F 1A /r"                  , "MPX"],
        ["bndmov"           , "W:bnd/mem, bnd"                                  , "MR"      , "66 0F 1B /r"                  , "MPX"],
        ["bndstx"           , "W:mib, bnd"                                      , "MR"      , "0F 1B /r"                     , "MPX"],
    
        ["bound"            , "R:r16, R:m32"                                    , "RM"      , "66 62 /r"                     , "X86 Deprecated"],
        ["bound"            , "R:r32, R:m64"                                    , "RM"      , "62 /r"                        , "X86 Deprecated"],
    
        ["bsf"              , "w:r16, r16/m16"                                  , "RM"      , "66 0F BC /r"                  , "ANY              OF=U SF=U ZF=W AF=U PF=U CF=U"],
        ["bsf"              , "W:r32, r32/m32"                                  , "RM"      , "0F BC /r"                     , "ANY              OF=U SF=U ZF=W AF=U PF=U CF=U"],
        ["bsf"              , "W:r64, r64/m64"                                  , "RM"      , "REX.W 0F BC /r"               , "X64              OF=U SF=U ZF=W AF=U PF=U CF=U"],
    
        ["bsr"              , "w:r16, r16/m16"                                  , "RM"      , "66 0F BD /r"                  , "ANY              OF=U SF=U ZF=W AF=U PF=U CF=U"],
        ["bsr"              , "W:r32, r32/m32"                                  , "RM"      , "0F BD /r"                     , "ANY              OF=U SF=U ZF=W AF=U PF=U CF=U"],
        ["bsr"              , "W:r64, r64/m64"                                  , "RM"      , "REX.W 0F BD /r"               , "X64              OF=U SF=U ZF=W AF=U PF=U CF=U"],
    
        ["bswap"            , "X:r32"                                           , "O"       , "0F C8+r"                      , "ANY"],
        ["bswap"            , "X:r64"                                           , "O"       , "REX.W 0F C8+r"                , "X64"],
    
        ["bt"               , "R:r16/m16, ib/ub"                                , "MI"      , "66 0F BA /4 ib"               , "ANY              OF=U SF=U AF=U PF=U CF=W"],
        ["bt"               , "R:r32/m32, ib/ub"                                , "MI"      , "0F BA /4 ib"                  , "ANY              OF=U SF=U AF=U PF=U CF=W"],
        ["bt"               , "R:r64/m64, ib/ub"                                , "MI"      , "REX.W 0F BA /4 ib"            , "X64              OF=U SF=U AF=U PF=U CF=W"],
        ["bt"               , "R:r16/m16, r16"                                  , "MR"      , "66 0F A3 /r"                  , "ANY              OF=U SF=U AF=U PF=U CF=W"],
        ["bt"               , "R:r32/m32, r32"                                  , "MR"      , "0F A3 /r"                     , "ANY              OF=U SF=U AF=U PF=U CF=W"],
        ["bt"               , "R:r64/m64, r64"                                  , "MR"      , "REX.W 0F A3 /r"               , "X64              OF=U SF=U AF=U PF=U CF=W"],
    
        ["btc"              , "x:r16/m16, ib/ub"                                , "MI"      , "66 0F BA /7 ib"               , "ANY _XLock       OF=U SF=U AF=U PF=U CF=W"],
        ["btc"              , "X:r32/m32, ib/ub"                                , "MI"      , "0F BA /7 ib"                  , "ANY _XLock       OF=U SF=U AF=U PF=U CF=W"],
        ["btc"              , "X:r64/m64, ib/ub"                                , "MI"      , "REX.W 0F BA /7 ib"            , "X64 _XLock       OF=U SF=U AF=U PF=U CF=W"],
        ["btc"              , "x:r16/m16, r16"                                  , "MR"      , "66 0F BB /r"                  , "ANY _XLock       OF=U SF=U AF=U PF=U CF=W"],
        ["btc"              , "X:r32/m32, r32"                                  , "MR"      , "0F BB /r"                     , "ANY _XLock       OF=U SF=U AF=U PF=U CF=W"],
        ["btc"              , "X:r64/m64, r64"                                  , "MR"      , "REX.W 0F BB /r"               , "X64 _XLock       OF=U SF=U AF=U PF=U CF=W"],
    
        ["btr"              , "x:r16/m16, ib/ub"                                , "MI"      , "66 0F BA /6 ib"               , "ANY _XLock       OF=U SF=U AF=U PF=U CF=W"],
        ["btr"              , "X:r32/m32, ib/ub"                                , "MI"      , "0F BA /6 ib"                  , "ANY _XLock       OF=U SF=U AF=U PF=U CF=W"],
        ["btr"              , "X:r64/m64, ib/ub"                                , "MI"      , "REX.W 0F BA /6 ib"            , "X64 _XLock       OF=U SF=U AF=U PF=U CF=W"],
        ["btr"              , "x:r16/m16, r16"                                  , "MR"      , "66 0F B3 /r"                  , "ANY _XLock       OF=U SF=U AF=U PF=U CF=W"],
        ["btr"              , "X:r32/m32, r32"                                  , "MR"      , "0F B3 /r"                     , "ANY _XLock       OF=U SF=U AF=U PF=U CF=W"],
        ["btr"              , "X:r64/m64, r64"                                  , "MR"      , "REX.W 0F B3 /r"               , "X64 _XLock       OF=U SF=U AF=U PF=U CF=W"],
    
        ["bts"              , "x:r16/m16, ib/ub"                                , "MI"      , "66 0F BA /5 ib"               , "ANY _XLock       OF=U SF=U AF=U PF=U CF=W"],
        ["bts"              , "X:r32/m32, ib/ub"                                , "MI"      , "0F BA /5 ib"                  , "ANY _XLock       OF=U SF=U AF=U PF=U CF=W"],
        ["bts"              , "X:r64/m64, ib/ub"                                , "MI"      , "REX.W 0F BA /5 ib"            , "X64 _XLock       OF=U SF=U AF=U PF=U CF=W"],
        ["bts"              , "x:r16/m16, r16"                                  , "MR"      , "66 0F AB /r"                  , "ANY _XLock       OF=U SF=U AF=U PF=U CF=W"],
        ["bts"              , "X:r32/m32, r32"                                  , "MR"      , "0F AB /r"                     , "ANY _XLock       OF=U SF=U AF=U PF=U CF=W"],
        ["bts"              , "X:r64/m64, r64"                                  , "MR"      , "REX.W 0F AB /r"               , "X64 _XLock       OF=U SF=U AF=U PF=U CF=W"],
    
        ["call"             , "rel16"                                           , "D"       , "66 E8 cw"                     , "X86              Control=Call OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["call"             , "rel32"                                           , "D"       , "E8 cd"                        , "ANY              Control=Call OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["call"             , "R:r16/m16"                                       , "M"       , "66 FF /2"                     , "X86              Control=Call OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["call"             , "R:r32/m32"                                       , "M"       , "FF /2"                        , "X86              Control=Call OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["call"             , "R:r64/m64"                                       , "M"       , "FF /2"                        , "X64              Control=Call OF=U SF=U ZF=U AF=U PF=U CF=U"],
    
        ["call_far"         , "R:p16_16"                                        , "D"       , "66 9A id"                     , "X86              Control=Call OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["call_far"         , "R:p16_32"                                        , "D"       , "9A if"                        , "X86              Control=Call OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["call_far"         , "R:m16_16"                                        , "M"       , "66 FF /3"                     , "ANY              Control=Call OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["call_far"         , "R:m16_32"                                        , "M"       , "FF /3"                        , "ANY              Control=Call OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["call_far"         , "R:m16_64"                                        , "M"       , "REX.W FF /3"                  , "X64              Control=Call OF=U SF=U ZF=U AF=U PF=U CF=U"],
        
    
        ["cbw"              , "x:<ax>"                                          , "NONE"    , "66 98"                        , "ANY"],
        ["cwde"             , "X:<eax>"                                         , "NONE"    , "98"                           , "ANY"],
        ["cdqe"             , "X:<rax>"                                         , "NONE"    , "REX.W 98"                     , "X64"],
    
        ["cwd"              , "w:<dx>, <ax>"                                    , "NONE"    , "66 99"                        , "ANY"],
        ["cdq"              , "W:<edx>, <eax>"                                  , "NONE"    , "99"                           , "ANY"],
        ["cqo"              , "W:<rdx>, <rax>"                                  , "NONE"    , "REX.W 99"                     , "X64"],
    
        ["clac"             , ""                                                , "NONE"    , "0F 01 CA"                     , "SMAP             Volatile AC=0"],
        ["cldemote"         , "R:mem"                                           , "M"       , "0F 1C /0"                     , "CLDEMOTE         Volatile"],
        ["clflush"          , "R:mem"                                           , "M"       , "0F AE /7"                     , "CLFLUSH          Volatile"],
        ["clflushopt"       , "R:mem"                                           , "M"       , "66 0F AE /7"                  , "CLFLUSHOPT       Volatile"],
        ["cli"              , ""                                                , "NONE"    , "FA"                           , "ANY              Volatile IF=W"],
        ["clwb"             , "R:mem"                                           , "M"       , "66 0F AE /6"                  , "CLWB             Volatile"],
        ["clzero"           , "R:<ds:zax>"                                      , "NONE"    , "0F 01 FC"                     , "CLZERO           Volatile"],
    
        ["clc"              , ""                                                , "NONE"    , "F8"                           , "ANY              CF=0"],
        ["cld"              , ""                                                , "NONE"    , "FC"                           , "ANY              DF=0"],
        ["cmc"              , ""                                                , "NONE"    , "F5"                           , "ANY              CF=X"],
    
        ["cmovo"            , "x:r16, r16/m16"                                  , "RM"      , "66 0F 40 /r"                  , "CMOV             OF=R"],
        ["cmovo"            , "X:r32, r32/m32"                                  , "RM"      , "0F 40 /r"                     , "CMOV             OF=R"],
        ["cmovo"            , "X:r64, r64/m64"                                  , "RM"      , "REX.W 0F 40 /r"               , "CMOV X64         OF=R"],
        ["cmovno"           , "x:r16, r16/m16"                                  , "RM"      , "66 0F 41 /r"                  , "CMOV             OF=R"],
        ["cmovno"           , "X:r32, r32/m32"                                  , "RM"      , "0F 41 /r"                     , "CMOV             OF=R"],
        ["cmovno"           , "X:r64, r64/m64"                                  , "RM"      , "REX.W 0F 41 /r"               , "CMOV X64         OF=R"],
        ["cmovb/cmovnae/cmovc" , "x:r16, r16/m16"                               , "RM"      , "66 0F 42 /r"                  , "CMOV             CF=R"],
        ["cmovb/cmovnae/cmovc" , "X:r32, r32/m32"                               , "RM"      , "0F 42 /r"                     , "CMOV             CF=R"],
        ["cmovb/cmovnae/cmovc" , "X:r64, r64/m64"                               , "RM"      , "REX.W 0F 42 /r"               , "CMOV X64         CF=R"],
        ["cmovae/cmovnb/cmovnc", "x:r16, r16/m16"                               , "RM"      , "66 0F 43 /r"                  , "CMOV             CF=R"],
        ["cmovae/cmovnb/cmovnc", "X:r32, r32/m32"                               , "RM"      , "0F 43 /r"                     , "CMOV             CF=R"],
        ["cmovae/cmovnb/cmovnc", "X:r64, r64/m64"                               , "RM"      , "REX.W 0F 43 /r"               , "CMOV X64         CF=R"],
        ["cmove/cmovz"      , "x:r16, r16/m16"                                  , "RM"      , "66 0F 44 /r"                  , "CMOV             ZF=R"],
        ["cmove/cmovz"      , "X:r32, r32/m32"                                  , "RM"      , "0F 44 /r"                     , "CMOV             ZF=R"],
        ["cmove/cmovz"      , "X:r64, r64/m64"                                  , "RM"      , "REX.W 0F 44 /r"               , "CMOV X64         ZF=R"],
        ["cmovne/cmovnz"    , "x:r16, r16/m16"                                  , "RM"      , "66 0F 45 /r"                  , "CMOV             ZF=R"],
        ["cmovne/cmovnz"    , "X:r32, r32/m32"                                  , "RM"      , "0F 45 /r"                     , "CMOV             ZF=R"],
        ["cmovne/cmovnz"    , "X:r64, r64/m64"                                  , "RM"      , "REX.W 0F 45 /r"               , "CMOV X64         ZF=R"],
        ["cmovbe/cmovna"    , "x:r16, r16/m16"                                  , "RM"      , "66 0F 46 /r"                  , "CMOV             CF=R ZF=R"],
        ["cmovbe/cmovna"    , "X:r32, r32/m32"                                  , "RM"      , "0F 46 /r"                     , "CMOV             CF=R ZF=R"],
        ["cmovbe/cmovna"    , "X:r64, r64/m64"                                  , "RM"      , "REX.W 0F 46 /r"               , "CMOV X64         CF=R ZF=R"],
        ["cmova/cmovnbe"    , "x:r16, r16/m16"                                  , "RM"      , "66 0F 47 /r"                  , "CMOV             CF=R ZF=R"],
        ["cmova/cmovnbe"    , "X:r32, r32/m32"                                  , "RM"      , "0F 47 /r"                     , "CMOV             CF=R ZF=R"],
        ["cmova/cmovnbe"    , "X:r64, r64/m64"                                  , "RM"      , "REX.W 0F 47 /r"               , "CMOV X64         CF=R ZF=R"],
        ["cmovs"            , "x:r16, r16/m16"                                  , "RM"      , "66 0F 48 /r"                  , "CMOV             SF=R"],
        ["cmovs"            , "X:r32, r32/m32"                                  , "RM"      , "0F 48 /r"                     , "CMOV             SF=R"],
        ["cmovs"            , "X:r64, r64/m64"                                  , "RM"      , "REX.W 0F 48 /r"               , "CMOV X64         SF=R"],
        ["cmovns"           , "x:r16, r16/m16"                                  , "RM"      , "66 0F 49 /r"                  , "CMOV             SF=R"],
        ["cmovns"           , "X:r32, r32/m32"                                  , "RM"      , "0F 49 /r"                     , "CMOV             SF=R"],
        ["cmovns"           , "X:r64, r64/m64"                                  , "RM"      , "REX.W 0F 49 /r"               , "CMOV X64         SF=R"],
        ["cmovp/cmovpe"     , "x:r16, r16/m16"                                  , "RM"      , "66 0F 4A /r"                  , "CMOV             PF=R"],
        ["cmovp/cmovpe"     , "X:r32, r32/m32"                                  , "RM"      , "0F 4A /r"                     , "CMOV             PF=R"],
        ["cmovp/cmovpe"     , "X:r64, r64/m64"                                  , "RM"      , "REX.W 0F 4A /r"               , "CMOV X64         PF=R"],
        ["cmovnp/cmovpo"    , "x:r16, r16/m16"                                  , "RM"      , "66 0F 4B /r"                  , "CMOV             PF=R"],
        ["cmovnp/cmovpo"    , "X:r32, r32/m32"                                  , "RM"      , "0F 4B /r"                     , "CMOV             PF=R"],
        ["cmovnp/cmovpo"    , "X:r64, r64/m64"                                  , "RM"      , "REX.W 0F 4B /r"               , "CMOV X64         PF=R"],
        ["cmovl/cmovnge"    , "x:r16, r16/m16"                                  , "RM"      , "66 0F 4C /r"                  , "CMOV             SF=R OF=R"],
        ["cmovl/cmovnge"    , "X:r32, r32/m32"                                  , "RM"      , "0F 4C /r"                     , "CMOV             SF=R OF=R"],
        ["cmovl/cmovnge"    , "X:r64, r64/m64"                                  , "RM"      , "REX.W 0F 4C /r"               , "CMOV X64         SF=R OF=R"],
        ["cmovge/cmovnl"    , "x:r16, r16/m16"                                  , "RM"      , "66 0F 4D /r"                  , "CMOV             SF=R OF=R"],
        ["cmovge/cmovnl"    , "X:r32, r32/m32"                                  , "RM"      , "0F 4D /r"                     , "CMOV             SF=R OF=R"],
        ["cmovge/cmovnl"    , "X:r64, r64/m64"                                  , "RM"      , "REX.W 0F 4D /r"               , "CMOV X64         SF=R OF=R"],
        ["cmovle/cmovng"    , "x:r16, r16/m16"                                  , "RM"      , "66 0F 4E /r"                  , "CMOV             ZF=R SF=R OF=R"],
        ["cmovle/cmovng"    , "X:r32, r32/m32"                                  , "RM"      , "0F 4E /r"                     , "CMOV             ZF=R SF=R OF=R"],
        ["cmovle/cmovng"    , "X:r64, r64/m64"                                  , "RM"      , "REX.W 0F 4E /r"               , "CMOV X64         ZF=R SF=R OF=R"],
        ["cmovg/cmovnle"    , "x:r16, r16/m16"                                  , "RM"      , "66 0F 4F /r"                  , "CMOV             ZF=R SF=R OF=R"],
        ["cmovg/cmovnle"    , "X:r32, r32/m32"                                  , "RM"      , "0F 4F /r"                     , "CMOV             ZF=R SF=R OF=R"],
        ["cmovg/cmovnle"    , "X:r64, r64/m64"                                  , "RM"      , "REX.W 0F 4F /r"               , "CMOV X64         ZF=R SF=R OF=R"],
    
        ["cmp"              , "R:al, ib/ub"                                     , "I"       , "3C ib"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmp"              , "R:ax, iw/uw"                                     , "I"       , "66 3D iw"                     , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmp"              , "R:eax, id/ud"                                    , "I"       , "3D id"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmp"              , "R:rax, id"                                       , "I"       , "REX.W 3D id"                  , "X64 AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["cmp"              , "R:r8/m8, ib/ub"                                  , "MI"      , "80 /7 ib"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmp"              , "R:r16/m16, iw/uw"                                , "MI"      , "66 81 /7 iw"                  , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmp"              , "R:r32/m32, id/ud"                                , "MI"      , "81 /7 id"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmp"              , "R:r64/m64, id"                                   , "MI"      , "REX.W 81 /7 id"               , "X64              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["cmp"              , "R:r16/m16, ib"                                   , "MI"      , "66 83 /7 ib"                  , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmp"              , "R:r32/m32, ib"                                   , "MI"      , "83 /7 ib"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmp"              , "R:r64/m64, ib"                                   , "MI"      , "REX.W 83 /7 ib"               , "X64              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["cmp"              , "R:r8/m8, r8"                                     , "MR"      , "38 /r"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmp"              , "R:r16/m16, r16"                                  , "MR"      , "66 39 /r"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmp"              , "R:r32/m32, r32"                                  , "MR"      , "39 /r"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmp"              , "R:r64/m64, r64"                                  , "MR"      , "REX.W 39 /r"                  , "X64              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["cmp"              , "R:r8, r8/m8"                                     , "RM"      , "3A /r"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmp"              , "R:r16, r16/m16"                                  , "RM"      , "66 3B /r"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmp"              , "R:r32, r32/m32"                                  , "RM"      , "3B /r"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmp"              , "R:r64, r64/m64"                                  , "RM"      , "REX.W 3B /r"                  , "X64              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["cmpsb"            , "X:<ds:zsi>, X:<es:zdi>"                          , "NONE"    , "A6"                           , "ANY _Rep         OF=W SF=W ZF=W AF=W PF=W CF=W DF=R"],
        ["cmpsw"            , "X:<ds:zsi>, X:<es:zdi>"                          , "NONE"    , "66 A7"                        , "ANY _Rep         OF=W SF=W ZF=W AF=W PF=W CF=W DF=R"],
        ["cmpsd"            , "X:<ds:zsi>, X:<es:zdi>"                          , "NONE"    , "A7"                           , "ANY _Rep         OF=W SF=W ZF=W AF=W PF=W CF=W DF=R"],
        ["cmpsq"            , "X:<ds:zsi>, X:<es:zdi>"                          , "NONE"    , "REX.W A7"                     , "X64 _Rep         OF=W SF=W ZF=W AF=W PF=W CF=W DF=R"],
    
        ["cmpxchg"          , "x:r8/m8, r8, <al>"                               , "MR"      , "0F B0 /r"                     , "I486             _XLock Volatile OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmpxchg"          , "x:r16/m16, r16, <ax>"                            , "MR"      , "66 0F B1 /r"                  , "I486             _XLock Volatile OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmpxchg"          , "X:r32/m32, r32, <eax>"                           , "MR"      , "0F B1 /r"                     , "I486             _XLock Volatile OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmpxchg"          , "X:r64/m64, r64, <rax>"                           , "MR"      , "REX.W 0F B1 /r"               , "I486 X64         _XLock Volatile OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["cmpxchg8b"        , "X:m64, X:<edx>, X:<eax>, <ecx>, <ebx>"           , "M"       , "0F C7 /1"                     , "CMPXCHG8B        _XLock Volatile ZF=W"],
        ["cmpxchg16b"       , "X:m128, X:<rdx>, X:<rax>, <rcx>, <rbx>"          , "M"       , "REX.W 0F C7 /1"               , "CMPXCHG16B X64   _XLock Volatile ZF=W"],
    
        ["cpuid"            , "X:<eax>, W:<ebx>, X:<ecx>, W:<edx>"              , "NONE"    , "0F A2"                        , "I486             Volatile"],
    
        ["daa"              , "x:<ax>"                                          , "NONE"    , "27"                           , "X86 Deprecated   OF=U SF=W ZF=W AF=W PF=W CF=W"],
        ["das"              , "x:<ax>"                                          , "NONE"    , "2F"                           , "X86 Deprecated   OF=U SF=W ZF=W AF=W PF=W CF=W"],
    
        ["dec"              , "x:r16"                                           , "O"       , "66 48+r"                      , "X86              OF=W SF=W ZF=W AF=W PF=W"],
        ["dec"              , "X:r32"                                           , "O"       , "48+r"                         , "X86              OF=W SF=W ZF=W AF=W PF=W"],
        ["dec"              , "x:r8/m8"                                         , "M"       , "FE /1"                        , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W"],
        ["dec"              , "x:r16/m16"                                       , "M"       , "66 FF /1"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W"],
        ["dec"              , "X:r32/m32"                                       , "M"       , "FF /1"                        , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W"],
        ["dec"              , "X:r64/m64"                                       , "M"       , "REX.W FF /1"                  , "X64 _XLock       OF=W SF=W ZF=W AF=W PF=W"],
    
        ["div"              , "x:<ax>, r8/m8"                                   , "M"       , "F6 /6"                        , "ANY              OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["div"              , "x:<dx>, x:<ax>, r16/m16"                         , "M"       , "66 F7 /6"                     , "ANY              OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["div"              , "X:<edx>, X:<eax>, r32/m32"                       , "M"       , "F7 /6"                        , "ANY              OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["div"              , "X:<rdx>, X:<rax>, r64/m64"                       , "M"       , "REX.W F7 /6"                  , "X64              OF=U SF=U ZF=U AF=U PF=U CF=U"],
    
        ["idiv"             , "x:<ax>, r8/m8"                                   , "M"       , "F6 /7"                        , "ANY              OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["idiv"             , "x:<dx>, x:<ax>, r16/m16"                         , "M"       , "66 F7 /7"                     , "ANY              OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["idiv"             , "X:<edx>, X:<eax>, r32/m32"                       , "M"       , "F7 /7"                        , "ANY              OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["idiv"             , "X:<rdx>, X:<rax>, r64/m64"                       , "M"       , "REX.W F7 /7"                  , "X64              OF=U SF=U ZF=U AF=U PF=U CF=U"],
    
        ["imul"             , "x:<ax>, r8/m8"                                   , "M"       , "F6 /5"                        , "ANY              OF=W SF=W ZF=U AF=U PF=U CF=W"],
        ["imul"             , "w:<dx>, x:<ax>, r16/m16"                         , "M"       , "66 F7 /5"                     , "ANY              OF=W SF=W ZF=U AF=U PF=U CF=W"],
        ["imul"             , "W:<edx>, X:<eax>, r32/m32"                       , "M"       , "F7 /5"                        , "ANY              OF=W SF=W ZF=U AF=U PF=U CF=W"],
        ["imul"             , "W:<rdx>, X:<rax>, r64/m64"                       , "M"       , "REX.W F7 /5"                  , "X64              OF=W SF=W ZF=U AF=U PF=U CF=W"],
        ["imul"             , "x:~r16,~r16/m16"                                 , "RM"      , "66 0F AF /r"                  , "ANY              OF=W SF=W ZF=U AF=U PF=U CF=W"],
        ["imul"             , "X:~r32,~r32/m32"                                 , "RM"      , "0F AF /r"                     , "ANY              OF=W SF=W ZF=U AF=U PF=U CF=W"],
        ["imul"             , "X:~r64,~r64/m64"                                 , "RM"      , "REX.W 0F AF /r"               , "X64              OF=W SF=W ZF=U AF=U PF=U CF=W"],
        ["imul"             , "w:r16, r16/m16, ib"                              , "RMI"     , "66 6B /r ib"                  , "ANY              OF=W SF=W ZF=U AF=U PF=U CF=W"],
        ["imul"             , "W:r32, r32/m32, ib"                              , "RMI"     , "6B /r ib"                     , "ANY              OF=W SF=W ZF=U AF=U PF=U CF=W"],
        ["imul"             , "W:r64, r64/m64, ib"                              , "RMI"     , "REX.W 6B /r ib"               , "X64              OF=W SF=W ZF=U AF=U PF=U CF=W"],
        ["imul"             , "w:r16, r16/m16, iw/uw"                           , "RMI"     , "66 69 /r iw"                  , "ANY              OF=W SF=W ZF=U AF=U PF=U CF=W"],
        ["imul"             , "W:r32, r32/m32, id/ud"                           , "RMI"     , "69 /r id"                     , "ANY              OF=W SF=W ZF=U AF=U PF=U CF=W"],
        ["imul"             , "W:r64, r64/m64, id"                              , "RMI"     , "REX.W 69 /r id"               , "X64              OF=W SF=W ZF=U AF=U PF=U CF=W"],
    
        ["in"               , "w:al, ib/ub"                                     , "I"       , "E4 ib"                        , "ANY              Volatile"],
        ["in"               , "w:ax, ib/ub"                                     , "I"       , "66 E5 ib"                     , "ANY              Volatile"],
        ["in"               , "W:eax, ib/ub"                                    , "I"       , "E5 ib"                        , "ANY              Volatile"],
        ["in"               , "w:al, dx"                                        , "NONE"    , "EC"                           , "ANY              Volatile"],
        ["in"               , "w:ax, dx"                                        , "NONE"    , "66 ED"                        , "ANY              Volatile"],
        ["in"               , "W:eax, dx"                                       , "NONE"    , "ED"                           , "ANY              Volatile"],
        ["insb"             , "W:es:zdi, dx"                                    , "NONE"    , "6C"                           , "ANY _Rep         Volatile"],
        ["insw"             , "W:es:zdi, dx"                                    , "NONE"    , "66 6D"                        , "ANY _Rep         Volatile"],
        ["insd"             , "W:es:zdi, dx"                                    , "NONE"    , "6D"                           , "ANY _Rep         Volatile"],
    
        ["inc"              , "x:r16"                                           , "O"       , "66 40+r"                      , "X86              OF=W SF=W ZF=W AF=W PF=W"],
        ["inc"              , "X:r32"                                           , "O"       , "40+r"                         , "X86              OF=W SF=W ZF=W AF=W PF=W"],
        ["inc"              , "x:r8/m8"                                         , "M"       , "FE /0"                        , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W"],
        ["inc"              , "x:r16/m16"                                       , "M"       , "66 FF /0"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W"],
        ["inc"              , "X:r32/m32"                                       , "M"       , "FF /0"                        , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W"],
        ["inc"              , "X:r64/m64"                                       , "M"       , "REX.W FF /0"                  , "X64 _XLock       OF=W SF=W ZF=W AF=W PF=W"],
    
        ["int"              , "ib/ub"                                           , "I"       , "CD ib"                        , "ANY              Volatile"],
        ["int3"             , ""                                                , "NONE"    , "CC"                           , "ANY              Volatile"],
        ["into"             , ""                                                , "NONE"    , "CE"                           , "X86 Deprecated   Volatile OF=R"],
    
        ["iret/iretd"       , ""                                                , "NONE"    , "CF"                           , "ANY              Control=Return OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["iretw"            , ""                                                , "NONE"    , "66 CF"                        , "ANY              Control=Return OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["iretq"            , ""                                                , "NONE"    , "REX.W CF"                     , "X64              Control=Return OF=U SF=U ZF=U AF=U PF=U CF=U"],
    
        ["jo"               , "rel8"                                            , "D"       , "70 cb"                        , "ANY              Control=Branch OF=R"],
        ["jno"              , "rel8"                                            , "D"       , "71 cb"                        , "ANY              Control=Branch OF=R"],
        ["jb/jnae/jc"       , "rel8"                                            , "D"       , "72 cb"                        , "ANY              Control=Branch CF=R"],
        ["jae/jnb/jnc"      , "rel8"                                            , "D"       , "73 cb"                        , "ANY              Control=Branch CF=R"],
        ["je/jz"            , "rel8"                                            , "D"       , "74 cb"                        , "ANY              Control=Branch ZF=R"],
        ["jne/jnz"          , "rel8"                                            , "D"       , "75 cb"                        , "ANY              Control=Branch ZF=R"],
        ["jbe/jna"          , "rel8"                                            , "D"       , "76 cb"                        , "ANY              Control=Branch CF=R ZF=R"],
        ["ja/jnbe"          , "rel8"                                            , "D"       , "77 cb"                        , "ANY              Control=Branch CF=R ZF=R"],
        ["js"               , "rel8"                                            , "D"       , "78 cb"                        , "ANY              Control=Branch SF=R"],
        ["jns"              , "rel8"                                            , "D"       , "79 cb"                        , "ANY              Control=Branch SF=R"],
        ["jp/jpe"           , "rel8"                                            , "D"       , "7A cb"                        , "ANY              Control=Branch PF=R"],
        ["jnp/jpo"          , "rel8"                                            , "D"       , "7B cb"                        , "ANY              Control=Branch PF=R"],
        ["jl/jnge"          , "rel8"                                            , "D"       , "7C cb"                        , "ANY              Control=Branch SF=R OF=R"],
        ["jge/jnl"          , "rel8"                                            , "D"       , "7D cb"                        , "ANY              Control=Branch SF=R OF=R"],
        ["jle/jng"          , "rel8"                                            , "D"       , "7E cb"                        , "ANY              Control=Branch ZF=R SF=R OF=R"],
        ["jg/jnle"          , "rel8"                                            , "D"       , "7F cb"                        , "ANY              Control=Branch ZF=R SF=R OF=R"],
    
        ["jo"               , "rel16"                                           , "D"       , "66 0F 80 cw"                  , "X86              Control=Branch OF=R"],
        ["jo"               , "rel32"                                           , "D"       , "0F 80 cd"                     , "ANY              Control=Branch OF=R"],
        ["jno"              , "rel16"                                           , "D"       , "66 0F 81 cw"                  , "X86              Control=Branch OF=R"],
        ["jno"              , "rel32"                                           , "D"       , "0F 81 cd"                     , "ANY              Control=Branch OF=R"],
        ["jb/jnae/jc"       , "rel16"                                           , "D"       , "66 0F 82 cw"                  , "X86              Control=Branch CF=R"],
        ["jb/jnae/jc"       , "rel32"                                           , "D"       , "0F 82 cd"                     , "ANY              Control=Branch CF=R"],
        ["jae/jnb/jnc"      , "rel16"                                           , "D"       , "66 0F 83 cw"                  , "X86              Control=Branch CF=R"],
        ["jae/jnb/jnc"      , "rel32"                                           , "D"       , "0F 83 cd"                     , "ANY              Control=Branch CF=R"],
        ["je/jz"            , "rel16"                                           , "D"       , "66 0F 84 cw"                  , "X86              Control=Branch ZF=R"],
        ["je/jz"            , "rel32"                                           , "D"       , "0F 84 cd"                     , "ANY              Control=Branch ZF=R"],
        ["jne/jnz"          , "rel16"                                           , "D"       , "66 0F 85 cw"                  , "X86              Control=Branch ZF=R"],
        ["jne/jnz"          , "rel32"                                           , "D"       , "0F 85 cd"                     , "ANY              Control=Branch ZF=R"],
        ["jbe/jna"          , "rel16"                                           , "D"       , "66 0F 86 cw"                  , "X86              Control=Branch CF=R ZF=R"],
        ["jbe/jna"          , "rel32"                                           , "D"       , "0F 86 cd"                     , "ANY              Control=Branch CF=R ZF=R"],
        ["ja/jnbe"          , "rel16"                                           , "D"       , "66 0F 87 cw"                  , "X86              Control=Branch CF=R ZF=R"],
        ["ja/jnbe"          , "rel32"                                           , "D"       , "0F 87 cd"                     , "ANY              Control=Branch CF=R ZF=R"],
        ["js"               , "rel16"                                           , "D"       , "66 0F 88 cw"                  , "X86              Control=Branch SF=R"],
        ["js"               , "rel32"                                           , "D"       , "0F 88 cd"                     , "ANY              Control=Branch SF=R"],
        ["jns"              , "rel16"                                           , "D"       , "66 0F 89 cw"                  , "X86              Control=Branch SF=R"],
        ["jns"              , "rel32"                                           , "D"       , "0F 89 cd"                     , "ANY              Control=Branch SF=R"],
        ["jp/jpe"           , "rel16"                                           , "D"       , "66 0F 8A cw"                  , "X86              Control=Branch PF=R"],
        ["jp/jpe"           , "rel32"                                           , "D"       , "0F 8A cd"                     , "ANY              Control=Branch PF=R"],
        ["jnp/jpo"          , "rel16"                                           , "D"       , "66 0F 8B cw"                  , "X86              Control=Branch PF=R"],
        ["jnp/jpo"          , "rel32"                                           , "D"       , "0F 8B cd"                     , "ANY              Control=Branch PF=R"],
        ["jl/jnge"          , "rel16"                                           , "D"       , "66 0F 8C cw"                  , "X86              Control=Branch SF=R OF=R"],
        ["jl/jnge"          , "rel32"                                           , "D"       , "0F 8C cd"                     , "ANY              Control=Branch SF=R OF=R"],
        ["jge/jnl"          , "rel16"                                           , "D"       , "66 0F 8D cw"                  , "X86              Control=Branch SF=R OF=R"],
        ["jge/jnl"          , "rel32"                                           , "D"       , "0F 8D cd"                     , "ANY              Control=Branch SF=R OF=R"],
        ["jle/jng"          , "rel16"                                           , "D"       , "66 0F 8E cw"                  , "X86              Control=Branch ZF=R SF=R OF=R"],
        ["jle/jng"          , "rel32"                                           , "D"       , "0F 8E cd"                     , "ANY              Control=Branch ZF=R SF=R OF=R"],
        ["jg/jnle"          , "rel16"                                           , "D"       , "66 0F 8F cw"                  , "X86              Control=Branch ZF=R SF=R OF=R"],
        ["jg/jnle"          , "rel32"                                           , "D"       , "0F 8F cd"                     , "ANY              Control=Branch ZF=R SF=R OF=R"],
    
        ["jecxz"            , "R:<cx>, rel8"                                    , "D"       , "67 E3 cb"                     , "X86              Control=Branch"],
        ["jecxz"            , "R:<ecx>, rel8"                                   , "D"       , "E3 cb"                        , "X86              Control=Branch"],
        ["jecxz"            , "R:<ecx>, rel8"                                   , "D"       , "67 E3 cb"                     , "X64              Control=Branch"],
        ["jecxz"            , "R:<rcx>, rel8"                                   , "D"       , "E3 cb"                        , "X64              Control=Branch"],
    
        ["jmp"              , "rel8"                                            , "D"       , "EB cb"                        , "ANY              Control=Jump"],
        ["jmp"              , "rel16"                                           , "D"       , "66 E9 cw"                     , "X86              Control=Jump"],
        ["jmp"              , "rel32"                                           , "D"       , "E9 cd"                        , "ANY              Control=Jump"],
        ["jmp"              , "R:r32/m32"                                       , "D"       , "FF /4"                        , "X86              Control=Jump"],
        ["jmp"              , "R:r64/m64"                                       , "D"       , "FF /4"                        , "X64              Control=Jump"],
    
        ["jmp_far"          , "R:p16_16"                                        , "M"       , "66 EA id"                     , "X86              Control=Jump"],
        ["jmp_far"          , "R:p16_32"                                        , "M"       , "EA if"                        , "X86              Control=Jump"],
        ["jmp_far"          , "R:m16_16"                                        , "M"       , "66 FF /5"                     , "ANY              Control=Jump"],
        ["jmp_far"          , "R:m16_32"                                        , "M"       , "FF /5"                        , "ANY              Control=Jump"],
        ["jmp_far"          , "R:m16_64"                                        , "M"       , "REX.W FF /5"                  , "X64              Control=Jump"],
        
    
        ["lar"              , "w:r16, R:r16/m16"                                , "RM"      , "66 0F 02 /r"                  , "ANY              Volatile ZF=W"],
        ["lar"              , "W:r32, R:r32/m16"                                , "RM"      , "0F 02 /r"                     , "ANY              Volatile ZF=W"],
    
        ["lds"              , "x:r16, m16_16"                                   , "RM"      , "66 C5 /r"                     , "X86              Volatile"],
        ["lds"              , "X:r32, m16_32"                                   , "RM"      , "C5 /r"                        , "X86              Volatile"],
    
        ["lea"              , "w:r16, mem"                                      , "RM"      , "67 8D /r"                     , "ANY"],
        ["lea"              , "W:r32, mem"                                      , "RM"      , "8D /r"                        , "ANY"],
        ["lea"              , "W:r64, mem"                                      , "RM"      , "REX.W 8D /r"                  , "X64"],
    
        ["les"              , "x:r16, m16_16"                                   , "RM"      , "66 C4 /r"                     , "X86              Volatile"],
        ["les"              , "X:r32, m16_32"                                   , "RM"      , "C4 /r"                        , "X86              Volatile"],
    
        ["lfs"              , "x:r16, m16_16"                                   , "RM"      , "66 0F B4 /r"                  , "ANY              Volatile"],
        ["lfs"              , "X:r32, m16_32"                                   , "RM"      , "0F B4 /r"                     , "ANY              Volatile"],
        ["lfs"              , "X:r64, m16_64"                                   , "RM"      , "REX.W 0F B4 /r"               , "X64              Volatile"],
    
        ["lgs"              , "x:r16, m16_16"                                   , "RM"      , "66 0F B5 /r"                  , "ANY              Volatile"],
        ["lgs"              , "X:r32, m16_32"                                   , "RM"      , "0F B5 /r"                     , "ANY              Volatile"],
        ["lgs"              , "X:r64, m16_64"                                   , "RM"      , "REX.W 0F B5 /r"               , "X64              Volatile"],
    
        ["lodsb"            , "w:<al>, X:<ds:zsi>"                              , "NONE"    , "AC"                           , "ANY _Rep         DF=R"],
        ["lodsw"            , "w:<ax>, X:<ds:zsi>"                              , "NONE"    , "66 AD"                        , "ANY _Rep         DF=R"],
        ["lodsd"            , "W:<eax>, X:<ds:zsi>"                             , "NONE"    , "AD"                           , "ANY _Rep         DF=R"],
        ["lodsq"            , "W:<rax>, X:<ds:zsi>"                             , "NONE"    , "REX.W AD"                     , "X64 _Rep         DF=R"],
    
        ["loop"             , "x:<cx>, rel8"                                    , "D"       , "67 E2 cb"                     , "X86              Control=Branch"],
        ["loop"             , "X:<ecx>, rel8"                                   , "D"       , "E2 cb"                        , "X86              Control=Branch"],
        ["loop"             , "X:<ecx>, rel8"                                   , "D"       , "67 E2 cb"                     , "X64              Control=Branch"],
        ["loop"             , "X:<rcx>, rel8"                                   , "D"       , "E2 cb"                        , "X64              Control=Branch"],
    
        ["loope"            , "x:<cx>, rel8"                                    , "D"       , "67 E1 cb"                     , "X86              Control=Branch ZF=R"],
        ["loope"            , "X:<ecx>, rel8"                                   , "D"       , "E1 cb"                        , "X86              Control=Branch ZF=R"],
        ["loope"            , "X:<ecx>, rel8"                                   , "D"       , "67 E1 cb"                     , "X64              Control=Branch ZF=R"],
        ["loope"            , "X:<rcx>, rel8"                                   , "D"       , "E1 cb"                        , "X64              Control=Branch ZF=R"],
    
        ["loopne"           , "x:<cx>, rel8"                                    , "D"       , "67 E0 cb"                     , "X86              Control=Branch ZF=R"],
        ["loopne"           , "X:<ecx>, rel8"                                   , "D"       , "E0 cb"                        , "X86              Control=Branch ZF=R"],
        ["loopne"           , "X:<ecx>, rel8"                                   , "D"       , "67 E0 cb"                     , "X64              Control=Branch ZF=R"],
        ["loopne"           , "X:<rcx>, rel8"                                   , "D"       , "E0 cb"                        , "X64              Control=Branch ZF=R"],
    
        ["lsl"              , "w:r16, R:r16/m16"                                , "RM"      , "66 0F 03 /r"                  , "ANY              Volatile   ZF=W"],
        ["lsl"              , "W:r32, R:r32/m16"                                , "RM"      , "0F 03 /r"                     , "ANY              Volatile   ZF=W"],
        ["lsl"              , "W:r64, R:r32/m16"                                , "RM"      , "REX.W 0F 03 /r"               , "X64              Volatile   ZF=W"],
    
        ["lss"              , "x:r16, m16_16"                                   , "RM"      , "66 0F B2 /r"                  , "ANY              Volatile"],
        ["lss"              , "X:r32, m16_32"                                   , "RM"      , "0F B2 /r"                     , "ANY              Volatile"],
        ["lss"              , "X:r64, m16_64"                                   , "RM"      , "REX.W 0F B2 /r"               , "X64              Volatile"],
    
        ["mov"              , "w:r8/m8, r8"                                     , "MR"      , "88 /r"                        , "ANY XRelease"],
        ["mov"              , "w:r16/m16, r16"                                  , "MR"      , "66 89 /r"                     , "ANY XRelease"],
        ["mov"              , "W:r32/m32, r32"                                  , "MR"      , "89 /r"                        , "ANY XRelease"],
        ["mov"              , "W:r64/m64, r64"                                  , "MR"      , "REX.W 89 /r"                  , "X64 XRelease"],
    
        ["mov"              , "w:r8/m8, ib/ub"                                  , "MI"      , "C6 /0 ib"                     , "ANY XRelease"],
        ["mov"              , "w:r16/m16, iw/uw"                                , "MI"      , "66 C7 /0 iw"                  , "ANY XRelease"],
        ["mov"              , "W:r32/m32, id/ud"                                , "MI"      , "C7 /0 id"                     , "ANY XRelease"],
        ["mov"              , "W:r64/m64, id"                                   , "MI"      , "REX.W C7 /0 id"               , "X64 XRelease"],
    
        ["mov"              , "w:r8, ib/ub"                                     , "I"       , "B0+r ib"                      , "ANY"],
        ["mov"              , "w:r16, iw/uw"                                    , "I"       , "66 B8+r iw"                   , "ANY"],
        ["mov"              , "W:r32, id/ud"                                    , "I"       , "B8+r id"                      , "ANY"],
        ["mov"              , "W:r64, iq/uq"                                    , "I"       , "REX.W B8+r iq"                , "X64"],
    
        ["mov"              , "w:r8, r8/m8"                                     , "RM"      , "8A /r"                        , "ANY"],
        ["mov"              , "w:r16, r16/m16"                                  , "RM"      , "66 8B /r"                     , "ANY"],
        ["mov"              , "W:r32, r32/m32"                                  , "RM"      , "8B /r"                        , "ANY"],
        ["mov"              , "W:r64, r64/m64"                                  , "RM"      , "REX.W 8B /r"                  , "X64"],
    
        ["mov"              , "w:r16/m16, sreg"                                 , "MR"      , "66 8C /r"                     , "ANY"],
        ["mov"              , "W:r32/m16, sreg"                                 , "MR"      , "8C /r"                        , "ANY"],
        ["mov"              , "W:r64/m16, sreg"                                 , "MR"      , "REX.W 8C /r"                  , "X64"],
        ["mov"              , "W:sreg, r16/m16"                                 , "RM"      , "66 8E /r"                     , "ANY"],
        ["mov"              , "W:sreg, r32/m16"                                 , "RM"      , "8E /r"                        , "ANY"],
        ["mov"              , "W:sreg, r64/m16"                                 , "RM"      , "REX.W 8E /r"                  , "X64"],
    
        ["mov"              , "w:al, moff8"                                     , "NONE"    , "A0"                           , "ANY"],
        ["mov"              , "w:ax, moff16"                                    , "NONE"    , "66 A1"                        , "ANY"],
        ["mov"              , "W:eax, moff32"                                   , "NONE"    , "A1"                           , "ANY"],
        ["mov"              , "W:rax, moff64"                                   , "NONE"    , "REX.W A1"                     , "X64"],
    
        ["mov"              , "W:moff8, al"                                     , "NONE"    , "A2"                           , "ANY"],
        ["mov"              , "W:moff16, ax"                                    , "NONE"    , "66 A3"                        , "ANY"],
        ["mov"              , "W:moff32, eax"                                   , "NONE"    , "A3"                           , "ANY"],
        ["mov"              , "W:moff64, rax"                                   , "NONE"    , "REX.W A3"                     , "X64"],
    
        ["mov"              , "W:r32, creg"                                     , "MR"      , "0F 20 /r"                     , "X86              OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["mov"              , "W:r64, creg"                                     , "MR"      , "0F 20 /r"                     , "X64              OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["mov"              , "W:creg, r32"                                     , "RM"      , "0F 22 /r"                     , "X86              OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["mov"              , "W:creg, r64"                                     , "RM"      , "0F 22 /r"                     , "X64              OF=U SF=U ZF=U AF=U PF=U CF=U"],
    
        ["mov"              , "W:r32, dreg"                                     , "MR"      , "0F 21 /r"                     , "X86              OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["mov"              , "W:r64, dreg"                                     , "MR"      , "0F 21 /r"                     , "X64              OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["mov"              , "W:dreg, r32"                                     , "RM"      , "0F 23 /r"                     , "X86              OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["mov"              , "W:dreg, r64"                                     , "RM"      , "0F 23 /r"                     , "X64              OF=U SF=U ZF=U AF=U PF=U CF=U"],
    
        ["movsb"            , "X:<es:zdi>, X:<ds:zsi>"                          , "NONE"    , "A4"                           , "ANY _Rep         DF=R"],
        ["movsw"            , "X:<es:zdi>, X:<ds:zsi>"                          , "NONE"    , "66 A5"                        , "ANY _Rep         DF=R"],
        ["movsd"            , "X:<es:zdi>, X:<ds:zsi>"                          , "NONE"    , "A5"                           , "ANY _Rep         DF=R"],
        ["movsq"            , "X:<es:zdi>, X:<ds:zsi>"                          , "NONE"    , "REX.W A5"                     , "X64 _Rep         DF=R"],
    
        ["movsx"            , "w:r16, r8/m8"                                    , "RM"      , "66 0F BE /r"                  , "ANY"],
        ["movsx"            , "W:r32, r8/m8"                                    , "RM"      , "0F BE /r"                     , "ANY"],
        ["movsx"            , "W:r64, r8/m8"                                    , "RM"      , "REX.W 0F BE /r"               , "X64"],
        ["movsx"            , "W:r32, r16/m16"                                  , "RM"      , "0F BF /r"                     , "ANY"],
        ["movsx"            , "W:r64, r16/m16"                                  , "RM"      , "REX.W 0F BF /r"               , "X64"],
        ["movsxd"           , "W:r64, r32/m32"                                  , "RM"      , "REX.W 63 /r"                  , "X64"],
    
        ["movzx"            , "w:r16, r8/m8"                                    , "RM"      , "66 0F B6 /r"                  , "ANY"],
        ["movzx"            , "W:r32, r8/m8"                                    , "RM"      , "0F B6 /r"                     , "ANY"],
        ["movzx"            , "W:r64, r8/m8"                                    , "RM"      , "REX.W 0F B6 /r"               , "X64"],
        ["movzx"            , "W:r32, r16/m16"                                  , "RM"      , "0F B7 /r"                     , "ANY"],
        ["movzx"            , "W:r64, r16/m16"                                  , "RM"      , "REX.W 0F B7 /r"               , "X64"],
    
        ["mul"              , "x:<ax>, r8/m8"                                   , "M"       , "F6 /4"                        , "ANY              OF=W SF=U ZF=U AF=U PF=U CF=W"],
        ["mul"              , "w:<dx>, x:<ax>, r16/m16"                         , "M"       , "66 F7 /4"                     , "ANY              OF=W SF=U ZF=U AF=U PF=U CF=W"],
        ["mul"              , "W:<edx>, X:<eax>, r32/m32"                       , "M"       , "F7 /4"                        , "ANY              OF=W SF=U ZF=U AF=U PF=U CF=W"],
        ["mul"              , "W:<rdx>, X:<rax>, r64/m64"                       , "M"       , "REX.W F7 /4"                  , "X64              OF=W SF=U ZF=U AF=U PF=U CF=W"],
    
        ["neg"              , "x:r8/m8"                                         , "M"       , "F6 /3"                        , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["neg"              , "x:r16/m16"                                       , "M"       , "66 F7 /3"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["neg"              , "X:r32/m32"                                       , "M"       , "F7 /3"                        , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["neg"              , "X:r64/m64"                                       , "M"       , "REX.W F7 /3"                  , "X64 _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["nop"              , ""                                                , "NONE"    , "90"                           , ""],
        ["nop"              , "R:r16/m16"                                       , "M"       , "66 0F 1F /0"                  , ""],
        ["nop"              , "R:r32/m32"                                       , "M"       , "0F 1F /0"                     , ""],
    
        ["not"              , "x:r8/m8"                                         , "M"       , "F6 /2"                        , "ANY _XLock"],
        ["not"              , "x:r16/m16"                                       , "M"       , "66 F7 /2"                     , "ANY _XLock"],
        ["not"              , "X:r32/m32"                                       , "M"       , "F7 /2"                        , "ANY _XLock"],
        ["not"              , "X:r64/m64"                                       , "M"       , "REX.W F7 /2"                  , "X64 _XLock"],
    
        ["or"               , "x:al, ib/ub"                                     , "I"       , "0C ib"                        , "ANY AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["or"               , "x:ax, iw/uw"                                     , "I"       , "66 0D iw"                     , "ANY AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["or"               , "X:eax, id/ud"                                    , "I"       , "0D id"                        , "ANY AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["or"               , "X:rax, id"                                       , "I"       , "REX.W 0D id"                  , "X64 AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["or"               , "x:r8/m8, ib/ub"                                  , "MI"      , "80 /1 ib"                     , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["or"               , "x:r16/m16, iw/uw"                                , "MI"      , "66 81 /1 iw"                  , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["or"               , "X:r32/m32, id/ud"                                , "MI"      , "81 /1 id"                     , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["or"               , "X:r64/m64, id"                                   , "MI"      , "REX.W 81 /1 id"               , "X64 _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["or"               , "x:r16/m16, ib"                                   , "MI"      , "66 83 /1 ib"                  , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["or"               , "X:r32/m32, ib"                                   , "MI"      , "83 /1 ib"                     , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["or"               , "X:r64/m64, ib"                                   , "MI"      , "REX.W 83 /1 ib"               , "X64 _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["or"               , "x:~r8/m8,~r8"                                    , "MR"      , "08 /r"                        , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["or"               , "x:~r16/m16,~r16"                                 , "MR"      , "66 09 /r"                     , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["or"               , "X:~r32/m32,~r32"                                 , "MR"      , "09 /r"                        , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["or"               , "X:~r64/m64,~r64"                                 , "MR"      , "REX.W 09 /r"                  , "X64 _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["or"               , "x:~r8,~r8/m8"                                    , "RM"      , "0A /r"                        , "ANY              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["or"               , "x:~r16,~r16/m16"                                 , "RM"      , "66 0B /r"                     , "ANY              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["or"               , "X:~r32,~r32/m32"                                 , "RM"      , "0B /r"                        , "ANY              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["or"               , "X:~r64,~r64/m64"                                 , "RM"      , "REX.W 0B /r"                  , "X64              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["out"              , "ub, al"                                          , "I"       , "E6 ib"                        , "ANY              Volatile"],
        ["out"              , "ub, ax"                                          , "I"       , "66 E7 ib"                     , "ANY              Volatile"],
        ["out"              , "ub, eax"                                         , "I"       , "E7 ib"                        , "ANY              Volatile"],
        ["out"              , "R:dx, R:al"                                      , "NONE"    , "EE"                           , "ANY              Volatile"],
        ["out"              , "R:dx, R:ax"                                      , "NONE"    , "66 EF"                        , "ANY              Volatile"],
        ["out"              , "R:dx, R:eax"                                     , "NONE"    , "EF"                           , "ANY              Volatile"],
        ["outsb"            , "R:dx, R:ds:zsi"                                  , "NONE"    , "6E"                           , "ANY _Rep         Volatile"],
        ["outsw"            , "R:dx, R:ds:zsi"                                  , "NONE"    , "66 6F"                        , "ANY _Rep         Volatile"],
        ["outsd"            , "R:dx, R:ds:zsi"                                  , "NONE"    , "6F"                           , "ANY _Rep         Volatile"],
    
        ["pause"            , ""                                                , "NONE"    , "F3 90"                        , "ANY              Volatile"],
        ["pcommit"          , ""                                                , "NONE"    , "66 0F AE F8"                  , "PCOMMIT          Volatile"],
    
        ["pop"              , "w:r16/m16"                                       , "M"       , "66 8F /0"                     , "ANY"],
        ["pop"              , "W:r32/m32"                                       , "M"       , "8F /0"                        , "X86"],
        ["pop"              , "W:r64/m64"                                       , "M"       , "8F /0"                        , "X64"],
    
        ["pop"              , "w:r16"                                           , "O"       , "66 58+r"                      , "ANY"],
        ["pop"              , "W:r32"                                           , "O"       , "58+r"                         , "X86"],
        ["pop"              , "W:r64"                                           , "O"       , "58+r"                         , "X64"],
    
        ["pop"              , "W:ds"                                            , "NONE"    , "1F"                           , "X86"],
        ["pop"              , "W:es"                                            , "NONE"    , "07"                           , "X86"],
        ["pop"              , "W:ss"                                            , "NONE"    , "17"                           , "X86"],
        ["pop"              , "W:fs"                                            , "NONE"    , "0F A1"                        , "ANY"],
        ["pop"              , "W:gs"                                            , "NONE"    , "0F A9"                        , "ANY"],
    
        ["popa"             , ""                                                , "NONE"    , "66 61"                        , "X86 Deprecated"],
        ["popad"            , ""                                                , "NONE"    , "61"                           , "X86 Deprecated"],
        ["popf"             , ""                                                , "NONE"    , "66 9D"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W DF=W IF=W TF=W"],
        ["popfd"            , ""                                                , "NONE"    , "9D"                           , "X86              OF=W SF=W ZF=W AF=W PF=W CF=W DF=W IF=W TF=W"],
        ["popfq"            , ""                                                , "NONE"    , "9D"                           , "X64              OF=W SF=W ZF=W AF=W PF=W CF=W DF=W IF=W TF=W"],
    
        ["prefetchw"        , "R:mem"                                           , "M"       , "0F 0D /1"                     , "PREFETCHW        OF=U SF=U ZF=U AF=U PF=U CF=U"],
        ["prefetchwt1"      , "R:mem"                                           , "M"       , "0F 0D /2"                     , "PREFETCHWT1      OF=U SF=U ZF=U AF=U PF=U CF=U"],
    
        ["push"             , "R:r16/m16"                                       , "M"       , "66 FF /6"                     , "ANY"],
        ["push"             , "R:r32/m32"                                       , "M"       , "FF /6"                        , "X86"],
        ["push"             , "R:r64/m64"                                       , "M"       , "FF /6"                        , "X64"],
    
        ["push"             , "R:r16"                                           , "O"       , "66 50+r"                      , "ANY"],
        ["push"             , "R:r32"                                           , "O"       , "50+r"                         , "X86"],
        ["push"             , "R:r64"                                           , "O"       , "50+r"                         , "X64"],
    
        ["push"             , "ib"                                              , "I"       , "6A ib"                        , "ANY"],
        ["push"             , "iw"                                              , "I"       , "66 68 iw"                     , "ANY"],
        ["push"             , "id/ud"                                           , "I"       , "68 id"                        , "X86"],
        ["push"             , "id"                                              , "I"       , "68 id"                        , "X64"],
    
        ["push"             , "R:cs"                                            , "NONE"    , "0E"                           , "X86"],
        ["push"             , "R:ss"                                            , "NONE"    , "16"                           , "X86"],
        ["push"             , "R:ds"                                            , "NONE"    , "1E"                           , "X86"],
        ["push"             , "R:es"                                            , "NONE"    , "06"                           , "X86"],
        ["push"             , "R:fs"                                            , "NONE"    , "0F A0"                        , "ANY"],
        ["push"             , "R:gs"                                            , "NONE"    , "0F A8"                        , "ANY"],
    
        ["pusha"            , ""                                                , "NONE"    , "66 60"                        , "X86 Deprecated"],
        ["pushad"           , ""                                                , "NONE"    , "60"                           , "X86 Deprecated"],
        ["pushf"            , ""                                                , "NONE"    , "66 9C"                        , "ANY              OF=R SF=R ZF=R AF=R PF=R CF=R DF=R IF=R TF=R"],
        ["pushfd"           , ""                                                , "NONE"    , "9C"                           , "X86              OF=R SF=R ZF=R AF=R PF=R CF=R DF=R IF=R TF=R"],
        ["pushfq"           , ""                                                , "NONE"    , "9C"                           , "X64              OF=R SF=R ZF=R AF=R PF=R CF=R DF=R IF=R TF=R"],
    
        ["rcl"              , "x:r8/m8, 1"                                      , "M"       , "D0 /2"                        , "ANY AltForm      CF=X OF=X"],
        ["rcl"              , "x:r8/m8, cl"                                     , "M"       , "D2 /2"                        , "ANY              CF=X OF=X"],
        ["rcl"              , "x:r8/m8, ib/ub"                                  , "MI"      , "C0 /2 ib"                     , "ANY              CF=X OF=X"],
    
        ["rcl"              , "x:r16/m16, 1"                                    , "M"       , "66 D1 /2"                     , "ANY AltForm      CF=X OF=X"],
        ["rcl"              , "x:r16/m16, cl"                                   , "M"       , "66 D3 /2"                     , "ANY              CF=X OF=X"],
        ["rcl"              , "x:r16/m16, ib/ub"                                , "MI"      , "66 C1 /2 ib"                  , "ANY              CF=X OF=X"],
    
        ["rcl"              , "X:r32/m32, 1"                                    , "M"       , "D1 /2"                        , "ANY AltForm      CF=X OF=X"],
        ["rcl"              , "X:r32/m32, cl"                                   , "M"       , "D3 /2"                        , "ANY              CF=X OF=X"],
        ["rcl"              , "X:r32/m32, ib/ub"                                , "MI"      , "C1 /2 ib"                     , "ANY              CF=X OF=X"],
    
        ["rcl"              , "X:r64/m64, 1"                                    , "M"       , "REX.W D1 /2"                  , "X64 AltForm      CF=X OF=X"],
        ["rcl"              , "X:r64/m64, cl"                                   , "M"       , "REX.W D3 /2"                  , "X64              CF=X OF=X"],
        ["rcl"              , "X:r64/m64, ib/ub"                                , "MI"      , "REX.W C1 /2 ib"               , "X64              CF=X OF=X"],
    
        ["rcr"              , "x:r8/m8, 1"                                      , "M"       , "D0 /3"                        , "ANY AltForm      CF=X OF=X"],
        ["rcr"              , "x:r8/m8, cl"                                     , "M"       , "D2 /3"                        , "ANY              CF=X OF=X"],
        ["rcr"              , "x:r8/m8, ib/ub"                                  , "MI"      , "C0 /3 ib"                     , "ANY              CF=X OF=X"],
    
        ["rcr"              , "x:r16/m16, 1"                                    , "M"       , "66 D1 /3"                     , "ANY AltForm      CF=X OF=X"],
        ["rcr"              , "x:r16/m16, cl"                                   , "M"       , "66 D3 /3"                     , "ANY              CF=X OF=X"],
        ["rcr"              , "x:r16/m16, ib/ub"                                , "MI"      , "66 C1 /3 ib"                  , "ANY              CF=X OF=X"],
    
        ["rcr"              , "X:r32/m32, 1"                                    , "M"       , "D1 /3"                        , "ANY AltForm      CF=X OF=X"],
        ["rcr"              , "X:r32/m32, cl"                                   , "M"       , "D3 /3"                        , "ANY              CF=X OF=X"],
        ["rcr"              , "X:r32/m32, ib/ub"                                , "MI"      , "C1 /3 ib"                     , "ANY              CF=X OF=X"],
    
        ["rcr"              , "X:r64/m64, 1"                                    , "M"       , "REX.W D1 /3"                  , "X64 AltForm      CF=X OF=X"],
        ["rcr"              , "X:r64/m64, cl"                                   , "M"       , "REX.W D3 /3"                  , "X64              CF=X OF=X"],
        ["rcr"              , "X:r64/m64, ib/ub"                                , "MI"      , "REX.W C1 /3 ib"               , "X64              CF=X OF=X"],
    
        ["rdfsbase"         , "W:r32"                                           , "M"       , "F3 0F AE /0"                  , "FSGSBASE X64     Volatile"],
        ["rdfsbase"         , "W:r64"                                           , "M"       , "REX.W F3 0F AE /0"            , "FSGSBASE X64     Volatile"],
        ["rdgsbase"         , "W:r32"                                           , "M"       , "F3 0F AE /1"                  , "FSGSBASE X64     Volatile"],
        ["rdgsbase"         , "W:r64"                                           , "M"       , "REX.W F3 0F AE /1"            , "FSGSBASE X64     Volatile"],
    
        ["rdrand"           , "w:r16"                                           , "M"       , "66 0F C7 /6"                  , "RDRAND           Volatile OF=0 SF=0 ZF=0 AF=0 PF=0 CF=W"],
        ["rdrand"           , "W:r32"                                           , "M"       , "0F C7 /6"                     , "RDRAND           Volatile OF=0 SF=0 ZF=0 AF=0 PF=0 CF=W"],
        ["rdrand"           , "W:r64"                                           , "M"       , "REX.W 0F C7 /6"               , "RDRAND X64       Volatile OF=0 SF=0 ZF=0 AF=0 PF=0 CF=W"],
    
        ["rdseed"           , "w:r16"                                           , "M"       , "66 0F C7 /7"                  , "RDSEED           Volatile OF=0 SF=0 ZF=0 AF=0 PF=0 CF=W"],
        ["rdseed"           , "W:r32"                                           , "M"       , "0F C7 /7"                     , "RDSEED           Volatile OF=0 SF=0 ZF=0 AF=0 PF=0 CF=W"],
        ["rdseed"           , "W:r64"                                           , "M"       , "REX.W 0F C7 /7"               , "RDSEED X64       Volatile OF=0 SF=0 ZF=0 AF=0 PF=0 CF=W"],
    
        ["ret"              , ""                                                , "NONE"    , "C3"                           , "ANY _RepIgnored  Control=Return"],
        ["ret"              , "uw"                                              , "I"       , "C2 iw"                        , "ANY _RepIgnored  Control=Return"],
    
        ["ret_far"          , ""                                                , "NONE"    , "CB"                           , "ANY              Control=Return"],
        ["ret_far"          , "uw"                                              , "I"       , "CA iw"                        , "ANY              Control=Return"],
    
        ["rol"              , "x:r8/m8, 1"                                      , "M"       , "D0 /0"                        , "ANY AltForm      CF=W OF=W"],
        ["rol"              , "x:r8/m8, cl"                                     , "M"       , "D2 /0"                        , "ANY              CF=W OF=W"],
        ["rol"              , "x:r8/m8, ib/ub"                                  , "MI"      , "C0 /0 ib"                     , "ANY              CF=W OF=W"],
    
        ["rol"              , "x:r16/m16, 1"                                    , "M"       , "66 D1 /0"                     , "ANY AltForm      CF=W OF=W"],
        ["rol"              , "x:r16/m16, cl"                                   , "M"       , "66 D3 /0"                     , "ANY              CF=W OF=W"],
        ["rol"              , "x:r16/m16, ib/ub"                                , "MI"      , "66 C1 /0 ib"                  , "ANY              CF=W OF=W"],
    
        ["rol"              , "X:r32/m32, 1"                                    , "M"       , "D1 /0"                        , "ANY AltForm      CF=W OF=W"],
        ["rol"              , "X:r32/m32, cl"                                   , "M"       , "D3 /0"                        , "ANY              CF=W OF=W"],
        ["rol"              , "X:r32/m32, ib/ub"                                , "MI"      , "C1 /0 ib"                     , "ANY              CF=W OF=W"],
    
        ["rol"              , "X:r64/m64, 1"                                    , "M"       , "REX.W D1 /0"                  , "X64 AltForm      CF=W OF=W"],
        ["rol"              , "X:r64/m64, cl"                                   , "M"       , "REX.W D3 /0"                  , "X64              CF=W OF=W"],
        ["rol"              , "X:r64/m64, ib/ub"                                , "MI"      , "REX.W C1 /0 ib"               , "X64              CF=W OF=W"],
    
        ["ror"              , "x:r8/m8, 1"                                      , "M"       , "D0 /1"                        , "ANY AltForm      CF=W OF=W"],
        ["ror"              , "x:r8/m8, cl"                                     , "M"       , "D2 /1"                        , "ANY              CF=W OF=W"],
        ["ror"              , "x:r8/m8, ib/ub"                                  , "MI"      , "C0 /1 ib"                     , "ANY              CF=W OF=W"],
    
        ["ror"              , "x:r16/m16, 1"                                    , "M"       , "66 D1 /1"                     , "ANY AltForm      CF=W OF=W"],
        ["ror"              , "x:r16/m16, cl"                                   , "M"       , "66 D3 /1"                     , "ANY              CF=W OF=W"],
        ["ror"              , "x:r16/m16, ib/ub"                                , "MI"      , "66 C1 /1 ib"                  , "ANY              CF=W OF=W"],
    
        ["ror"              , "X:r32/m32, 1"                                    , "M"       , "D1 /1"                        , "ANY AltForm      CF=W OF=W"],
        ["ror"              , "X:r32/m32, cl"                                   , "M"       , "D3 /1"                        , "ANY              CF=W OF=W"],
        ["ror"              , "X:r32/m32, ib/ub"                                , "MI"      , "C1 /1 ib"                     , "ANY              CF=W OF=W"],
    
        ["ror"              , "X:r64/m64, 1"                                    , "M"       , "REX.W D1 /1"                  , "X64 AltForm      CF=W OF=W"],
        ["ror"              , "X:r64/m64, cl"                                   , "M"       , "REX.W D3 /1"                  , "X64              CF=W OF=W"],
        ["ror"              , "X:r64/m64, ib/ub"                                , "MI"      , "REX.W C1 /1 ib"               , "X64              CF=W OF=W"],
    
        ["rsm"              , ""                                                , "NONE"    , "0F AA"                        , "X86              Volatile OF=U SF=U ZF=U AF=U PF=U CF=U"],
    
        ["sar"              , "x:r8/m8, 1"                                      , "M"       , "D0 /7"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sar"              , "x:r8/m8, cl"                                     , "M"       , "D2 /7"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sar"              , "x:r8/m8, ib/ub"                                  , "MI"      , "C0 /7 ib"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["sar"              , "x:r16/m16, 1"                                    , "M"       , "66 D1 /7"                     , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sar"              , "x:r16/m16, cl"                                   , "M"       , "66 D3 /7"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sar"              , "x:r16/m16, ib/ub"                                , "MI"      , "66 C1 /7 ib"                  , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["sar"              , "X:r32/m32, 1"                                    , "M"       , "D1 /7"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sar"              , "X:r32/m32, cl"                                   , "M"       , "D3 /7"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sar"              , "X:r32/m32, ib/ub"                                , "MI"      , "C1 /7 ib"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["sar"              , "X:r64/m64, 1"                                    , "M"       , "REX.W D1 /7"                  , "X64 AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sar"              , "X:r64/m64, cl"                                   , "M"       , "REX.W D3 /7"                  , "X64              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sar"              , "X:r64/m64, ib/ub"                                , "MI"      , "REX.W C1 /7 ib"               , "X64              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["sbb"              , "x:al, ib/ub"                                     , "I"       , "1C ib"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["sbb"              , "x:ax, iw/uw"                                     , "I"       , "66 1D iw"                     , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["sbb"              , "X:eax, id/ud"                                    , "I"       , "1D id"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["sbb"              , "X:rax, id"                                       , "I"       , "REX.W 1D id"                  , "X64 AltForm      OF=W SF=W ZF=W AF=W PF=W CF=X"],
    
        ["sbb"              , "x:r8/m8, ib/ub"                                  , "MI"      , "80 /3 ib"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["sbb"              , "x:r16/m16, iw/uw"                                , "MI"      , "66 81 /3 iw"                  , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["sbb"              , "X:r32/m32, id/ud"                                , "MI"      , "81 /3 id"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["sbb"              , "X:r64/m64, id"                                   , "MI"      , "REX.W 81 /3 id"               , "X64 _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
    
        ["sbb"              , "x:r16/m16, ib"                                   , "MI"      , "66 83 /3 ib"                  , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["sbb"              , "X:r32/m32, ib"                                   , "MI"      , "83 /3 ib"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["sbb"              , "X:r64/m64, ib"                                   , "MI"      , "REX.W 83 /3 ib"               , "X64 _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
    
        ["sbb"              , "x:r8/m8, r8"                                     , "MR"      , "18 /r"                        , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["sbb"              , "x:r16/m16, r16"                                  , "MR"      , "66 19 /r"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["sbb"              , "X:r32/m32, r32"                                  , "MR"      , "19 /r"                        , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["sbb"              , "X:r64/m64, r64"                                  , "MR"      , "REX.W 19 /r"                  , "X64 _XLock       OF=W SF=W ZF=W AF=W PF=W CF=X"],
    
        ["sbb"              , "x:r8, r8/m8"                                     , "RM"      , "1A /r"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["sbb"              , "x:r16, r16/m16"                                  , "RM"      , "66 1B /r"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["sbb"              , "X:r32, r32/m32"                                  , "RM"      , "1B /r"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=X"],
        ["sbb"              , "X:r64, r64/m64"                                  , "RM"      , "REX.W 1B /r"                  , "X64              OF=W SF=W ZF=W AF=W PF=W CF=X"],
    
        ["scasb"            , "R:<al>, X:<es:zdi>"                              , "NONE"    , "AE"                           , "ANY _Rep         OF=W SF=W ZF=W AF=W PF=W CF=W DF=R"],
        ["scasw"            , "R:<ax>, X:<es:zdi>"                              , "NONE"    , "66 AF"                        , "ANY _Rep         OF=W SF=W ZF=W AF=W PF=W CF=W DF=R"],
        ["scasd"            , "R:<eax>, X:<es:zdi>"                             , "NONE"    , "AF"                           , "ANY _Rep         OF=W SF=W ZF=W AF=W PF=W CF=W DF=R"],
        ["scasq"            , "R:<rax>, X:<es:zdi>"                             , "NONE"    , "REX.W AF"                     , "X64 _Rep         OF=W SF=W ZF=W AF=W PF=W CF=W DF=R"],
    
        ["seto"             , "w:r8/m8"                                         , "M"       , "0F 90 /r"                     , "ANY              OF=R"],
        ["setno"            , "w:r8/m8"                                         , "M"       , "0F 91 /r"                     , "ANY              OF=R"],
        ["setb/setnae/setc" , "w:r8/m8"                                         , "M"       , "0F 92 /r"                     , "ANY              CF=R"],
        ["setae/setnb/setnc", "w:r8/m8"                                         , "M"       , "0F 93 /r"                     , "ANY              CF=R"],
        ["sete/setz"        , "w:r8/m8"                                         , "M"       , "0F 94 /r"                     , "ANY              ZF=R"],
        ["setne/setnz"      , "w:r8/m8"                                         , "M"       , "0F 95 /r"                     , "ANY              ZF=R"],
        ["setbe/setna"      , "w:r8/m8"                                         , "M"       , "0F 96 /r"                     , "ANY              CF=R ZF=R"],
        ["seta/setnbe"      , "w:r8/m8"                                         , "M"       , "0F 97 /r"                     , "ANY              CF=R ZF=R"],
        ["sets"             , "w:r8/m8"                                         , "M"       , "0F 98 /r"                     , "ANY              SF=R"],
        ["setns"            , "w:r8/m8"                                         , "M"       , "0F 99 /r"                     , "ANY              SF=R"],
        ["setp/setpe"       , "w:r8/m8"                                         , "M"       , "0F 9A /r"                     , "ANY              PF=R"],
        ["setnp/setpo"      , "w:r8/m8"                                         , "M"       , "0F 9B /r"                     , "ANY              PF=R"],
        ["setl/setnge"      , "w:r8/m8"                                         , "M"       , "0F 9C /r"                     , "ANY              SF=R OF=R"],
        ["setge/setnl"      , "w:r8/m8"                                         , "M"       , "0F 9D /r"                     , "ANY              SF=R OF=R"],
        ["setle/setng"      , "w:r8/m8"                                         , "M"       , "0F 9E /r"                     , "ANY              ZF=R SF=R OF=R"],
        ["setg/setnle"      , "w:r8/m8"                                         , "M"       , "0F 9F /r"                     , "ANY              ZF=R SF=R OF=R"],
    
        ["sgdt"             , "W:mem"                                           , "M"       , "0F 01 /0"                     , "Volatile"],
    
        ["shl/sal"          , "x:r8/m8, 1"                                      , "M"       , "D0 /4"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shl/sal"          , "x:r8/m8, cl"                                     , "M"       , "D2 /4"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shl/sal"          , "x:r8/m8, ib/ub"                                  , "MI"      , "C0 /4 ib"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["shl/sal"          , "x:r16/m16, 1"                                    , "M"       , "66 D1 /4"                     , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shl/sal"          , "x:r16/m16, cl"                                   , "M"       , "66 D3 /4"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shl/sal"          , "x:r16/m16, ib/ub"                                , "MI"      , "66 C1 /4 ib"                  , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["shl/sal"          , "X:r32/m32, 1"                                    , "M"       , "D1 /4"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shl/sal"          , "X:r32/m32, cl"                                   , "M"       , "D3 /4"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shl/sal"          , "X:r32/m32, ib/ub"                                , "MI"      , "C1 /4 ib"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["shl/sal"          , "X:r64/m64, 1"                                    , "M"       , "REX.W D1 /4"                  , "X64 AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shl/sal"          , "X:r64/m64, cl"                                   , "M"       , "REX.W D3 /4"                  , "X64              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shl/sal"          , "X:r64/m64, ib/ub"                                , "MI"      , "REX.W C1 /4 ib"               , "X64              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["shr"              , "x:r8/m8, 1"                                      , "M"       , "D0 /5"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shr"              , "x:r8/m8, cl"                                     , "M"       , "D2 /5"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shr"              , "x:r8/m8, ib/ub"                                  , "MI"      , "C0 /5 ib"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["shr"              , "x:r16/m16, 1"                                    , "M"       , "66 D1 /5"                     , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shr"              , "x:r16/m16, cl"                                   , "M"       , "66 D3 /5"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shr"              , "x:r16/m16, ib/ub"                                , "MI"      , "66 C1 /5 ib"                  , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["shr"              , "X:r32/m32, 1"                                    , "M"       , "D1 /5"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shr"              , "X:r32/m32, cl"                                   , "M"       , "D3 /5"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shr"              , "X:r32/m32, ib/ub"                                , "MI"      , "C1 /5 ib"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["shr"              , "X:r64/m64, 1"                                    , "M"       , "REX.W D1 /5"                  , "X64 AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shr"              , "X:r64/m64, cl"                                   , "M"       , "REX.W D3 /5"                  , "X64              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["shr"              , "X:r64/m64, ib/ub"                                , "MI"      , "REX.W C1 /5 ib"               , "X64              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["shld"             , "x:r16/m16, r16, cl"                              , "MR"      , "66 0F A5 /r"                  , "ANY              OF=W SF=W ZF=W AF=U PF=W CF=W"],
        ["shld"             , "x:r16/m16, r16, ib/ub"                           , "MRI"     , "66 0F A4 /r ib"               , "ANY              OF=W SF=W ZF=W AF=U PF=W CF=W"],
        ["shld"             , "X:r32/m32, r32, cl"                              , "MR"      , "0F A5 /r"                     , "ANY              OF=W SF=W ZF=W AF=U PF=W CF=W"],
        ["shld"             , "X:r32/m32, r32, ib/ub"                           , "MRI"     , "0F A4 /r ib"                  , "ANY              OF=W SF=W ZF=W AF=U PF=W CF=W"],
        ["shld"             , "X:r64/m64, r64, cl"                              , "MR"      , "REX.W 0F A5 /r"               , "X64              OF=W SF=W ZF=W AF=U PF=W CF=W"],
        ["shld"             , "X:r64/m64, r64, ib/ub"                           , "MRI"     , "REX.W 0F A4 /r ib"            , "X64              OF=W SF=W ZF=W AF=U PF=W CF=W"],
    
        ["shrd"             , "x:r16/m16, r16, cl"                              , "MR"      , "66 0F AD /r"                  , "ANY              OF=W SF=W ZF=W AF=U PF=W CF=W"],
        ["shrd"             , "x:r16/m16, r16, ib/ub"                           , "MRI"     , "66 0F AC /r ib"               , "ANY              OF=W SF=W ZF=W AF=U PF=W CF=W"],
        ["shrd"             , "X:r32/m32, r32, cl"                              , "MR"      , "0F AD /r"                     , "ANY              OF=W SF=W ZF=W AF=U PF=W CF=W"],
        ["shrd"             , "X:r32/m32, r32, ib/ub"                           , "MRI"     , "0F AC /r ib"                  , "ANY              OF=W SF=W ZF=W AF=U PF=W CF=W"],
        ["shrd"             , "X:r64/m64, r64, cl"                              , "MR"      , "REX.W 0F AD /r"               , "X64              OF=W SF=W ZF=W AF=U PF=W CF=W"],
        ["shrd"             , "X:r64/m64, r64, ib/ub"                           , "MRI"     , "REX.W 0F AC /r ib"            , "X64              OF=W SF=W ZF=W AF=U PF=W CF=W"],
    
        ["sidt"             , "W:mem"                                           , "M"       , "0F 01 /1"                     , "ANY Volatile"],
        ["sldt"             , "w:r16/m16"                                       , "M"       , "66 0F 00 /0"                  , "ANY Volatile"],
        ["sldt"             , "W:r32/m16"                                       , "M"       , "0F 00 /0"                     , "ANY Volatile"],
        ["sldt"             , "W:r64/m16"                                       , "M"       , "REX.W 0F 00 /0"               , "X64 Volatile"],
    
        ["smsw"             , "w:r16/m16"                                       , "M"       , "66 0F 01 /4"                  , "ANY Volatile"],
        ["smsw"             , "W:r32/m16"                                       , "M"       , "0F 01 /4"                     , "ANY Volatile"],
        ["smsw"             , "W:r64/m16"                                       , "M"       , "REX.W 0F 01 /4"               , "X64 Volatile"],
    
        ["stac"             , ""                                                , "NONE"    , "0F 01 CB"                     , "SMAP             Volatile AC=1"],
        ["stc"              , ""                                                , "NONE"    , "F9"                           , "ANY              CF=1"],
        ["std"              , ""                                                , "NONE"    , "FD"                           , "ANY              DF=1"],
        ["sti"              , ""                                                , "NONE"    , "FB"                           , "ANY              Volatile IF=1"],
    
        ["stosb"            , "X:<es:zdi>, R:<al>"                              , "NONE"    , "AA"                           , "ANY _Rep         DF=R"],
        ["stosw"            , "X:<es:zdi>, R:<ax>"                              , "NONE"    , "66 AB"                        , "ANY _Rep         DF=R"],
        ["stosd"            , "X:<es:zdi>, R:<eax>"                             , "NONE"    , "AB"                           , "ANY _Rep         DF=R"],
        ["stosq"            , "X:<es:zdi>, R:<rax>"                             , "NONE"    , "REX.W AB"                     , "X64 _Rep         DF=R"],
    
        ["str"              , "w:r16/m16"                                       , "M"       , "66 0F 00 /1"                  , "ANY              Volatile"],
        ["str"              , "W:r32/m16"                                       , "M"       , "0F 00 /1"                     , "ANY              Volatile"],
        ["str"              , "W:r64/m16"                                       , "M"       , "REX.W 0F 00 /1"               , "X64              Volatile"],
    
        ["sub"              , "x:al, ib/ub"                                     , "I"       , "2C ib"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sub"              , "x:ax, iw/uw"                                     , "I"       , "66 2D iw"                     , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sub"              , "X:eax, id/ud"                                    , "I"       , "2D id"                        , "ANY AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sub"              , "X:rax, id"                                       , "I"       , "REX.W 2D id"                  , "X64 AltForm      OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["sub"              , "x:r8/m8, ib/ub"                                  , "MI"      , "80 /5 ib"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sub"              , "x:r16/m16, iw/uw"                                , "MI"      , "66 81 /5 iw"                  , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sub"              , "X:r32/m32, id/ud"                                , "MI"      , "81 /5 id"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sub"              , "X:r64/m64, id"                                   , "MI"      , "REX.W 81 /5 id"               , "X64 _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["sub"              , "x:r16/m16, ib"                                   , "MI"      , "66 83 /5 ib"                  , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sub"              , "X:r32/m32, ib"                                   , "MI"      , "83 /5 ib"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sub"              , "X:r64/m64, ib"                                   , "MI"      , "REX.W 83 /5 ib"               , "X64 _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["sub"              , "x:r8/m8, r8"                                     , "MR"      , "28 /r"                        , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sub"              , "x:r16/m16, r16"                                  , "MR"      , "66 29 /r"                     , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sub"              , "X:r32/m32, r32"                                  , "MR"      , "29 /r"                        , "ANY _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sub"              , "X:r64/m64, r64"                                  , "MR"      , "REX.W 29 /r"                  , "X64 _XLock       OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["sub"              , "x:r8, r8/m8"                                     , "RM"      , "2A /r"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sub"              , "x:r16, r16/m16"                                  , "RM"      , "66 2B /r"                     , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sub"              , "X:r32, r32/m32"                                  , "RM"      , "2B /r"                        , "ANY              OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["sub"              , "X:r64, r64/m64"                                  , "RM"      , "REX.W 2B /r"                  , "X64              OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["test"             , "R:al, ib/ub"                                     , "I"       , "A8 ib"                        , "ANY AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["test"             , "R:ax, iw/uw"                                     , "I"       , "66 A9 iw"                     , "ANY AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["test"             , "R:eax, id/ud"                                    , "I"       , "A9 id"                        , "ANY AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["test"             , "R:rax, id"                                       , "I"       , "REX.W A9 id"                  , "X64 AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["test"             , "R:r8/m8, ib/ub"                                  , "MI"      , "F6 /0 ib"                     , "ANY              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["test"             , "R:r16/m16, iw/uw"                                , "MI"      , "66 F7 /0 iw"                  , "ANY              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["test"             , "R:r32/m32, id/ud"                                , "MI"      , "F7 /0 id"                     , "ANY              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["test"             , "R:r64/m64, id"                                   , "MI"      , "REX.W F7 /0 id"               , "X64              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["test"             , "R:~r8/m8,~r8"                                    , "MR"      , "84 /r"                        , "ANY              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["test"             , "R:~r16/m16,~r16"                                 , "MR"      , "66 85 /r"                     , "ANY              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["test"             , "R:~r32/m32,~r32"                                 , "MR"      , "85 /r"                        , "ANY              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["test"             , "R:~r64/m64,~r64"                                 , "MR"      , "REX.W 85 /r"                  , "X64              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["ud2"              , ""                                                , "NONE"    , "0F 0B"                        , "ANY"],
    
        ["verr"             , "R:r16/m16"                                       , "M"       , "0F 00 /4"                     , "ANY Volatile     ZF=W"],
        ["verw"             , "R:r16/m16"                                       , "M"       , "0F 00 /5"                     , "ANY Volatile     ZF=W"],
    
        ["wrfsbase"         , "R:r32"                                           , "M"       , "F3 0F AE /2"                  , "FSGSBASE X64     Volatile"],
        ["wrfsbase"         , "R:r64"                                           , "M"       , "REX.W F3 0F AE /2"            , "FSGSBASE X64     Volatile"],
        ["wrgsbase"         , "R:r32"                                           , "M"       , "F3 0F AE /3"                  , "FSGSBASE X64     Volatile"],
        ["wrgsbase"         , "R:r64"                                           , "M"       , "REX.W F3 0F AE /3"            , "FSGSBASE X64     Volatile"],
    
        ["xadd"             , "x:r8/m8, x:r8"                                   , "MR"      , "0F C0 /r"                     , "I486     _XLock  OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["xadd"             , "x:r16/m16, x:r16"                                , "MR"      , "66 0F C1 /r"                  , "I486     _XLock  OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["xadd"             , "X:r32/m32, X:r32"                                , "MR"      , "0F C1 /r"                     , "I486     _XLock  OF=W SF=W ZF=W AF=W PF=W CF=W"],
        ["xadd"             , "X:r64/m64, X:r64"                                , "MR"      , "REX.W 0F C1 /r"               , "I486 X64 _XLock  OF=W SF=W ZF=W AF=W PF=W CF=W"],
    
        ["xchg"             , "x:~ax, x:~r16"                                   , "O"       , "66 90+r"                      , "ANY AltForm"],
        ["xchg"             , "X:~eax, X:~r32"                                  , "O"       , "90+r"                         , "ANY AltForm"],
        ["xchg"             , "X:~rax, X:~r64"                                  , "O"       , "REX.W 90+r"                   , "X64 AltForm"],
    
        ["xchg"             , "x:~r16, x:~ax"                                   , "O"       , "66 90+r"                      , "ANY AltForm"],
        ["xchg"             , "X:~r32, X:~eax"                                  , "O"       , "90+r"                         , "ANY AltForm"],
        ["xchg"             , "X:~r64, X:~rax"                                  , "O"       , "REX.W 90+r"                   , "X64 AltForm"],
    
        ["xchg"             , "x:~r8/m8, x:~r8"                                 , "MR"      , "86 /r"                        , "ANY _ILock XAcquire"],
        ["xchg"             , "x:~r16/m16, x:~r16"                              , "MR"      , "66 87 /r"                     , "ANY _ILock XAcquire"],
        ["xchg"             , "X:~r32/m32, X:~r32"                              , "MR"      , "87 /r"                        , "ANY _ILock XAcquire"],
        ["xchg"             , "X:~r64/m64, X:~r64"                              , "MR"      , "REX.W 87 /r"                  , "X64 _ILock XAcquire"],
    
        ["xchg"             , "x:~r8, x:~r8/m8"                                 , "RM"      , "86 /r"                        , "ANY _ILock"],
        ["xchg"             , "x:~r16, x:~r16/m16"                              , "RM"      , "66 87 /r"                     , "ANY _ILock"],
        ["xchg"             , "X:~r32, X:~r32/m32"                              , "RM"      , "87 /r"                        , "ANY _ILock"],
        ["xchg"             , "X:~r64, X:~r64/m64"                              , "RM"      , "REX.W 87 /r"                  , "X64 _ILock"],
    
        ["xlatb"            , ""                                                , "NONE"    , "D7"                           , "ANY              Volatile"],
    
        ["xor"              , "x:al, ib/ub"                                     , "I"       , "34 ib"                        , "ANY AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["xor"              , "x:ax, iw/uw"                                     , "I"       , "66 35 iw"                     , "ANY AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["xor"              , "X:eax, id/ud"                                    , "I"       , "35 id"                        , "ANY AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["xor"              , "X:rax, id"                                       , "I"       , "REX.W 35 id"                  , "X64 AltForm      OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["xor"              , "x:r8/m8, ib/ub"                                  , "MI"      , "80 /6 ib"                     , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["xor"              , "x:r16/m16, iw/uw"                                , "MI"      , "66 81 /6 iw"                  , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["xor"              , "X:r32/m32, id/ud"                                , "MI"      , "81 /6 id"                     , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["xor"              , "X:r64/m64, id"                                   , "MI"      , "REX.W 81 /6 id"               , "X64 _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["xor"              , "x:r16/m16, ib"                                   , "MI"      , "66 83 /6 ib"                  , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["xor"              , "X:r32/m32, ib"                                   , "MI"      , "83 /6 ib"                     , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["xor"              , "X:r64/m64, ib"                                   , "MI"      , "REX.W 83 /6 ib"               , "X64 _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["xor"              , "x:~r8/m8, ~r8"                                   , "MR"      , "30 /r"                        , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["xor"              , "x:~r16/m16, ~r16"                                , "MR"      , "66 31 /r"                     , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["xor"              , "X:~r32/m32, ~r32"                                , "MR"      , "31 /r"                        , "ANY _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["xor"              , "X:~r64/m64, ~r64"                                , "MR"      , "REX.W 31 /r"                  , "X64 _XLock       OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["xor"              , "x:~r8, ~r8/m8"                                   , "RM"      , "32 /r"                        , "ANY              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["xor"              , "x:~r16, ~r16/m16"                                , "RM"      , "66 33 /r"                     , "ANY              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["xor"              , "X:~r32, ~r32/m32"                                , "RM"      , "33 /r"                        , "ANY              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
        ["xor"              , "X:~r64, ~r64/m64"                                , "RM"      , "REX.W 33 /r"                  , "X64              OF=0 SF=W ZF=W AF=U PF=W CF=0"],
    
        ["enter"            , "iw/uw, ib/ub"                                    , "II"      , "C8 iw ib"                     , "ANY              Volatile"],
        ["leave"            , ""                                                , "NONE"    , "C9"                           , "ANY              Volatile"],
    
        ["lahf"             , "w:<ah>"                                          , "NONE"    , "9F"                           , "LAHFSAHF         Volatile SF=R ZF=R AF=R PF=R CF=R"],
        ["sahf"             , "R:<ah>"                                          , "NONE"    , "9E"                           , "LAHFSAHF         Volatile SF=W ZF=W AF=W PF=W CF=W"],
    
        ["syscall"          , ""                                                , "NONE"    , "0F 05"                        , "X64              Volatile"],
        ["sysenter"         , ""                                                , "NONE"    , "0F 34"                        , "ANY              Volatile"],
        ["sysexit"          , ""                                                , "NONE"    , "0F 35"                        , "ANY              Volatile PRIVILEGE=L0"],
        ["sysexit64"        , ""                                                , "NONE"    , "REX.W 0F 35"                  , "ANY              Volatile PRIVILEGE=L0"],
        ["sysret"           , ""                                                , "NONE"    , "0F 07"                        , "X64              Volatile PRIVILEGE=L0"],
        ["sysret64"         , ""                                                , "NONE"    , "REX.W 0F 07"                  , "X64              Volatile PRIVILEGE=L0"],
    
        ["clts"             , ""                                                , "NONE"    , "0F 06"                        , "ANY              Volatile PRIVILEGE=L0"],
        ["hlt"              , ""                                                , "NONE"    , "F4"                           , "ANY              Volatile PRIVILEGE=L0"],
        ["invd"             , ""                                                , "NONE"    , "0F 08"                        , "I486             Volatile PRIVILEGE=L0"],
        ["invlpg"           , "R:mem"                                           , "M"       , "0F 01 /7"                     , "I486             Volatile PRIVILEGE=L0"],
        ["invpcid"          , "R:r32, R:m128"                                   , "RM"      , "66 0F 38 82 /r"               , "I486 X86         Volatile PRIVILEGE=L0"],
        ["invpcid"          , "R:r64, R:m128"                                   , "RM"      , "66 0F 38 82 /r"               , "I486 X64         Volatile PRIVILEGE=L0"],
        ["lgdt"             , "R:mem"                                           , "M"       , "0F 01 /2"                     , "ANY              Volatile PRIVILEGE=L0"],
        ["lidt"             , "R:mem"                                           , "M"       , "0F 01 /3"                     , "ANY              Volatile PRIVILEGE=L0"],
        ["lldt"             , "R:r16/m16"                                       , "M"       , "0F 00 /2"                     , "ANY              Volatile PRIVILEGE=L0"],
        ["lmsw"             , "R:r16/m16"                                       , "M"       , "0F 01 /6"                     , "ANY              Volatile PRIVILEGE=L0"],
        ["ltr"              , "R:r16/m16"                                       , "M"       , "0F 00 /3"                     , "ANY              Volatile PRIVILEGE=L0"],
        ["monitor"          , "R:<ds:zax>, R:<ecx>, R:<edx>"                    , "NONE"    , "0F 01 C8"                     , "MONITOR          Volatile PRIVILEGE=L0"],
        ["monitorx"         , "R:<ds:zax>, R:<ecx>, R:<edx>"                    , "NONE"    , "0F 01 FA"                     , "MONITORX         Volatile"],
        ["mwait"            , "R:<eax>, R:<ecx>"                                , "NONE"    , "0F 01 C9"                     , "MONITOR          Volatile PRIVILEGE=L0"],
        ["mwaitx"           , "R:<eax>, R:<ecx>, R:<ebx>"                       , "NONE"    , "0F 01 FB"                     , "MONITORX         Volatile"],
    
        ["rdpid"            , "W:r32"                                           , "R"       , "F3 0F C7 /7"                  , "RDPID X86"],
        ["rdpid"            , "W:r64"                                           , "R"       , "F3 0F C7 /7"                  , "RDPID X64"],
    
    
        ["rdpmc"            , "W:<edx>, W:<eax>, R:<ecx>"                       , "NONE"    , "0F 33"                        , "ANY              Volatile PRIVILEGE=L0"],
        ["rdmsr"            , "W:<edx>, W:<eax>, R:<ecx>"                       , "NONE"    , "0F 32"                        , "ANY              Volatile PRIVILEGE=L0 MSR=R"],
        ["wrmsr"            , "R:<edx>, R:<eax>, R:<ecx>"                       , "NONE"    , "0F 30"                        , "ANY              Volatile PRIVILEGE=L0 MSR=W"],
        ["wbinvd"           , ""                                                , "NONE"    , "0F 09"                        , "ANY              Volatile PRIVILEGE=L0"],
        ["swapgs"           , ""                                                , "NONE"    , "0F 01 F8"                     , "X64              Volatile PRIVILEGE=L0"],
        ["xgetbv"           , "W:<edx>, W:<eax>, R:<ecx>"                       , "NONE"    , "0F 01 D0"                     , "XSAVE            Volatile              XCR=R"],
        ["xsetbv"           , "R:<edx>, R:<eax>, R:<ecx>"                       , "NONE"    , "0F 01 D1"                     , "XSAVE            Volatile PRIVILEGE=L0 XCR=W"],
    
        ["rdtsc"            , "W:<edx>, W:<eax>"                                , "NONE"    , "0F 31"                        , "RDTSC            Volatile"],
        ["rdtscp"           , "W:<edx>, W:<eax>, W:<ecx>"                       , "NONE"    , "0F 01 F9"                     , "RDTSCP           Volatile"],
    
        ["fxrstor"          , "R:mem"                                           , "NONE"    , "0F AE /1"                     , "FXSR             Volatile C0=W C1=W C2=W C3=W"],
        ["fxrstor64"        , "R:mem"                                           , "NONE"    , "REX.W 0F AE /1"               , "FXSR X64         Volatile C0=W C1=W C2=W C3=W"],
        ["fxsave"           , "W:mem"                                           , "NONE"    , "0F AE /0"                     , "FXSR             Volatile"],
        ["fxsave64"         , "W:mem"                                           , "NONE"    , "REX.W 0F AE /0"               , "FXSR X64         Volatile"],
    
        ["xrstor"           , "R:mem, <edx>, <eax>"                             , "M"       , "0F AE /5"                     , "XSAVE            Volatile XCR=R"],
        ["xrstor64"         , "R:mem, <edx>, <eax>"                             , "M"       , "REX.W 0F AE /5"               , "XSAVE X64        Volatile XCR=R"],
        ["xrstors"          , "R:mem, <edx>, <eax>"                             , "M"       , "0F C7 /3"                     , "XSAVES           Volatile XCR=R"],
        ["xrstors64"        , "R:mem, <edx>, <eax>"                             , "M"       , "REX.W 0F C7 /3"               , "XSAVES X64       Volatile XCR=R"],
        ["xsave"            , "W:mem, <edx>, <eax>"                             , "M"       , "0F AE /4"                     , "XSAVE            Volatile XCR=R"],
        ["xsave64"          , "W:mem, <edx>, <eax>"                             , "M"       , "REX.W 0F AE /4"               , "XSAVE X64        Volatile XCR=R"],
        ["xsavec"           , "W:mem, <edx>, <eax>"                             , "M"       , "0F C7 /4"                     , "XSAVEC           Volatile XCR=R"],
        ["xsavec64"         , "W:mem, <edx>, <eax>"                             , "M"       , "REX.W 0F C7 /4"               , "XSAVEC X64       Volatile XCR=R"],
        ["xsaveopt"         , "W:mem, <edx>, <eax>"                             , "M"       , "0F AE /6"                     , "XSAVEOPT         Volatile XCR=R"],
        ["xsaveopt64"       , "W:mem, <edx>, <eax>"                             , "M"       , "REX.W 0F AE /6"               , "XSAVEOPT X64     Volatile XCR=R"],
        ["xsaves"           , "W:mem, <edx>, <eax>"                             , "M"       , "0F C7 /5"                     , "XSAVES           Volatile XCR=R"],
        ["xsaves64"         , "W:mem, <edx>, <eax>"                             , "M"       , "REX.W 0F C7 /5"               , "XSAVES X64       Volatile XCR=R"],
    
        ["xbegin"           , "rel16"                                           , "NONE"    , "66 C7 /7 cw"                  , "RTM              Volatile"],
        ["xbegin"           , "rel32"                                           , "NONE"    , "C7 /7 cd"                     , "RTM              Volatile"],
        ["xend"             , ""                                                , "NONE"    , "0F 01 D5"                     , "RTM              Volatile"],
        ["xtest"            , ""                                                , "NONE"    , "0F 01 D6"                     , "TSX              Volatile OF=0 SF=0 ZF=W AF=0 PF=0 CF=0"],
        ["xabort"           , "ib/ub"                                           , "I"       , "C6 /7 ib"                     , "RTM              Volatile"],
    
        ["adcx"             , "X:~r32, ~r32/m32"                                , "RM"      , "66 0F 38 F6 /r"               , "ADX              CF=X"],
        ["adcx"             , "X:~r64, ~r64/m64"                                , "RM"      , "REX.W 66 0F 38 F6 /r"         , "ADX X64          CF=X"],
        ["adox"             , "X:~r32, ~r32/m32"                                , "RM"      , "F3 0F 38 F6 /r"               , "ADX              OF=X"],
        ["adox"             , "X:~r64, ~r64/m64"                                , "RM"      , "REX.W F3 0F 38 F6 /r"         , "ADX X64          OF=X"],
        ["andn"             , "W:r32, r32, r32/m32"                             , "RVM"     , "VEX.LZ.0F38.W0 F2 /r"         , "BMI              OF=0 SF=W ZF=W AF=U PF=U CF=0"],
        ["andn"             , "W:r64, r64, r64/m64"                             , "RVM"     , "VEX.LZ.0F38.W1 F2 /r"         , "BMI X64          OF=0 SF=W ZF=W AF=U PF=U CF=0"],
        ["bextr"            , "W:r32, r32/m32, r32"                             , "RMV"     , "VEX.LZ.0F38.W0 F7 /r"         , "BMI              OF=0 SF=U ZF=W AF=U PF=U CF=0"],
        ["bextr"            , "W:r64, r64/m64, r64"                             , "RMV"     , "VEX.LZ.0F38.W1 F7 /r"         , "BMI X64          OF=0 SF=U ZF=W AF=U PF=U CF=0"],
        ["blsi"             , "W:r32, r32/m32"                                  , "VM"      , "VEX.LZ.0F38.W0 F3 /3"         , "BMI              OF=0 SF=W ZF=W AF=U PF=U CF=W"],
        ["blsi"             , "W:r64, r64/m64"                                  , "VM"      , "VEX.LZ.0F38.W1 F3 /3"         , "BMI X64          OF=0 SF=W ZF=W AF=U PF=U CF=W"],
        ["blsmsk"           , "W:r32, r32/m32"                                  , "VM"      , "VEX.LZ.0F38.W0 F3 /2"         , "BMI              OF=0 SF=W ZF=0 AF=U PF=U CF=W"],
        ["blsmsk"           , "W:r64, r64/m64"                                  , "VM"      , "VEX.LZ.0F38.W1 F3 /2"         , "BMI X64          OF=0 SF=W ZF=0 AF=U PF=U CF=W"],
        ["blsr"             , "W:r32, r32/m32"                                  , "VM"      , "VEX.LZ.0F38.W0 F3 /1"         , "BMI              OF=0 SF=W ZF=W AF=U PF=U CF=W"],
        ["blsr"             , "W:r64, r64/m64"                                  , "VM"      , "VEX.LZ.0F38.W1 F3 /1"         , "BMI X64          OF=0 SF=W ZF=W AF=U PF=U CF=W"],
        ["bzhi"             , "W:r32, r32/m32, r32"                             , "RMV"     , "VEX.LZ.0F38.W0 F5 /r"         , "BMI2             OF=0 SF=W ZF=W AF=U PF=U CF=W"],
        ["bzhi"             , "W:r64, r64/m64, r64"                             , "RMV"     , "VEX.LZ.0F38.W1 F5 /r"         , "BMI2 X64         OF=0 SF=W ZF=W AF=U PF=U CF=W"],
        ["lzcnt"            , "w:r16, r16/m16"                                  , "RM"      , "66 F3 0F BD /r"               , "LZCNT            OF=U SF=U ZF=W AF=U PF=U CF=W"],
        ["lzcnt"            , "W:r32, r32/m32"                                  , "RM"      , "F3 0F BD /r"                  , "LZCNT            OF=U SF=U ZF=W AF=U PF=U CF=W"],
        ["lzcnt"            , "W:r64, r64/m64"                                  , "RM"      , "REX.W F3 0F BD /r"            , "LZCNT X64        OF=U SF=U ZF=W AF=U PF=U CF=W"],
        ["mulx"             , "W:r32, W:r32, ~r32/m32, ~<edx>"                  , "RVM"     , "VEX.LZ.F2.0F38.W0 F6 /r"      , "BMI2"],
        ["mulx"             , "W:r64, W:r64, ~r64/m64, ~<rdx>"                  , "RVM"     , "VEX.LZ.F2.0F38.W1 F6 /r"      , "BMI2 X64"],
        ["pdep"             , "W:r32, r32, r32/m32"                             , "RVM"     , "VEX.LZ.F2.0F38.W0 F5 /r"      , "BMI2"],
        ["pdep"             , "W:r64, r64, r64/m64"                             , "RVM"     , "VEX.LZ.F2.0F38.W1 F5 /r"      , "BMI2 X64"],
        ["pext"             , "W:r32, r32, r32/m32"                             , "RVM"     , "VEX.LZ.F3.0F38.W0 F5 /r"      , "BMI2"],
        ["pext"             , "W:r64, r64, r64/m64"                             , "RVM"     , "VEX.LZ.F3.0F38.W1 F5 /r"      , "BMI2 X64"],
        ["popcnt"           , "w:r16, r16/m16"                                  , "RM"      , "66 F3 0F B8 /r"               , "POPCNT           OF=0 SF=0 ZF=W AF=0 PF=0 CF=0"],
        ["popcnt"           , "W:r32, r32/m32"                                  , "RM"      , "F3 0F B8 /r"                  , "POPCNT           OF=0 SF=0 ZF=W AF=0 PF=0 CF=0"],
        ["popcnt"           , "W:r64, r64/m64"                                  , "RM"      , "REX.W F3 0F B8 /r"            , "POPCNT X64       OF=0 SF=0 ZF=W AF=0 PF=0 CF=0"],
        ["rorx"             , "W:r32, r32/m32, ib/ub"                           , "RMI"     , "VEX.LZ.F2.0F3A.W0 F0 /r ib"   , "BMI2"],
        ["rorx"             , "W:r64, r64/m64, ib/ub"                           , "RMI"     , "VEX.LZ.F2.0F3A.W1 F0 /r ib"   , "BMI2 X64"],
        ["sarx"             , "W:r32, r32/m32, r32"                             , "RMV"     , "VEX.LZ.F3.0F38.W0 F7 /r"      , "BMI2"],
        ["sarx"             , "W:r64, r64/m64, r64"                             , "RMV"     , "VEX.LZ.F3.0F38.W1 F7 /r"      , "BMI2 X64"],
        ["shlx"             , "W:r32, r32/m32, r32"                             , "RMV"     , "VEX.LZ.66.0F38.W0 F7 /r"      , "BMI2"],
        ["shlx"             , "W:r64, r64/m64, r64"                             , "RMV"     , "VEX.LZ.66.0F38.W1 F7 /r"      , "BMI2 X64"],
        ["shrx"             , "W:r32, r32/m32, r32"                             , "RMV"     , "VEX.LZ.F2.0F38.W0 F7 /r"      , "BMI2"],
        ["shrx"             , "W:r64, r64/m64, r64"                             , "RMV"     , "VEX.LZ.F2.0F38.W1 F7 /r"      , "BMI2 X64"],
        ["tzcnt"            , "w:r16, r16/m16"                                  , "RM"      , "66 F3 0F BC /r"               , "BMI              OF=U SF=U ZF=W AF=U PF=U CF=W"],
        ["tzcnt"            , "W:r32, r32/m32"                                  , "RM"      , "F3 0F BC /r"                  , "BMI              OF=U SF=U ZF=W AF=U PF=U CF=W"],
        ["tzcnt"            , "W:r64, r64/m64"                                  , "RM"      , "REX.W F3 0F BC /r"            , "BMI X64          OF=U SF=U ZF=W AF=U PF=U CF=W"],
    
        ["crc32"            , "X:r32, r8/m8"                                    , "RM"      , "F2 0F 38 F0 /r"               , "SSE4_2"],
        ["crc32"            , "X:r32, r16/m16"                                  , "RM"      , "66 F2 0F 38 F1 /r"            , "SSE4_2"],
        ["crc32"            , "X:r32, r32/m32"                                  , "RM"      , "F2 0F 38 F1 /r"               , "SSE4_2"],
        ["crc32"            , "X:r64, r8/m8"                                    , "RM"      , "REX.W F2 0F 38 F0 /r"         , "SSE4_2 X64"],
        ["crc32"            , "X:r64, r64/m64"                                  , "RM"      , "REX.W F2 0F 38 F1 /r"         , "SSE4_2 X64"],
    
        ["movbe"            , "w:r16, m16"                                      , "RM"      , "66 0F 38 F0 /r"               , "MOVBE"],
        ["movbe"            , "W:r32, m32"                                      , "RM"      , "0F 38 F0 /r"                  , "MOVBE"],
        ["movbe"            , "W:r64, m64"                                      , "RM"      , "REX.W 0F 38 F0 /r"            , "MOVBE X64"],
        ["movbe"            , "W:m16, r16"                                      , "MR"      , "66 0F 38 F1 /r"               , "MOVBE"],
        ["movbe"            , "W:m32, r32"                                      , "MR"      , "0F 38 F1 /r"                  , "MOVBE"],
        ["movbe"            , "W:m64, r64"                                      , "MR"      , "REX.W 0F 38 F1 /r"            , "MOVBE X64"],
    
        ["blci"             , "W:r32, r32/m32"                                  , "VM"      , "XOP.LZ.M09.W0 02 /6"          , "TBM"],
        ["blci"             , "W:r64, r64/m64"                                  , "VM"      , "XOP.LZ.M09.W1 02 /6"          , "TBM X64"],
        ["blcic"            , "W:r32, r32/m32"                                  , "VM"      , "XOP.LZ.M09.W0 01 /5"          , "TBM"],
        ["blcic"            , "W:r64, r64/m64"                                  , "VM"      , "XOP.LZ.M09.W1 01 /5"          , "TBM X64"],
        ["blsic"            , "W:r32, r32/m32"                                  , "VM"      , "XOP.LZ.M09.W0 01 /6"          , "TBM"],
        ["blsic"            , "W:r64, r64/m64"                                  , "VM"      , "XOP.LZ.M09.W1 01 /6"          , "TBM X64"],
        ["blcfill"          , "W:r32, r32/m32"                                  , "VM"      , "XOP.LZ.M09.W0 01 /1"          , "TBM"],
        ["blcfill"          , "W:r64, r64/m64"                                  , "VM"      , "XOP.LZ.M09.W1 01 /1"          , "TBM X64"],
        ["blsfill"          , "W:r32, r32/m32"                                  , "VM"      , "XOP.LZ.M09.W0 01 /2"          , "TBM"],
        ["blsfill"          , "W:r64, r64/m64"                                  , "VM"      , "XOP.LZ.M09.W1 01 /2"          , "TBM X64"],
        ["blcmsk"           , "W:r32, r32/m32"                                  , "VM"      , "XOP.LZ.M09.W0 02 /1"          , "TBM"],
        ["blcmsk"           , "W:r64, r64/m64"                                  , "VM"      , "XOP.LZ.M09.W1 02 /1"          , "TBM X64"],
        ["blcs"             , "W:r32, r32/m32"                                  , "VM"      , "XOP.LZ.M09.W0 01 /3"          , "TBM"],
        ["blcs"             , "W:r64, r64/m64"                                  , "VM"      , "XOP.LZ.M09.W1 01 /3"          , "TBM X64"],
        ["tzmsk"            , "W:r32, r32/m32"                                  , "VM"      , "XOP.LZ.M09.W0 01 /4"          , "TBM"],
        ["tzmsk"            , "W:r64, r64/m64"                                  , "VM"      , "XOP.LZ.M09.W1 01 /4"          , "TBM X64"],
        ["t1mskc"           , "W:r32, r32/m32"                                  , "VM"      , "XOP.LZ.M09.W0 01 /7"          , "TBM"],
        ["t1mskc"           , "W:r64, r64/m64"                                  , "VM"      , "XOP.LZ.M09.W1 01 /7"          , "TBM X64"],
    
        ["llwpcb"           , "R:r32"                                           , "M"       , "XOP.L0.P0.M09.W0 12 /0"       , "LWP"],
        ["llwpcb"           , "R:r64"                                           , "M"       , "XOP.L0.P0.M09.W1 12 /0"       , "LWP X64"],
        ["lwpins"           , "R:r32, R:r32/m32, id/ud"                         , "VMI"     , "XOP.L0.P0.M0A.W0 12 /0 id"    , "LWP"],
        ["lwpins"           , "R:r64, R:r32/m32, id/ud"                         , "VMI"     , "XOP.L0.P0.M0A.W1 12 /0 id"    , "LWP X64"],
        ["lwpval"           , "R:r32, R:r32/m32, id/ud"                         , "VMI"     , "XOP.L0.P0.M0A.W0 12 /1 id"    , "LWP"],
        ["lwpval"           , "R:r64, R:r32/m32, id/ud"                         , "VMI"     , "XOP.L0.P0.M0A.W1 12 /1 id"    , "LWP X64"],
        ["slwpcb"           , "W:r32"                                           , "M"       , "XOP.L0.P0.M09.W0 12 /1"       , "LWP"],
        ["slwpcb"           , "W:r64"                                           , "M"       , "XOP.L0.P0.M09.W1 12 /1"       , "LWP X64"],
    
        ["enqcmd"           , "W:es:r32, m512"                                  , "RM"      , "F2 0F 38 F8 /r"               , "ENQCMD X86"],
        ["enqcmd"           , "W:es:r32, m512"                                  , "RM"      , "F2 0F 38 F8 /r"               , "ENQCMD X64"],
        ["enqcmds"          , "W:es:r32, m512"                                  , "RM"      , "F3 0F 38 F8 /r"               , "ENQCMD X86"],
        ["enqcmds"          , "W:es:r32, m512"                                  , "RM"      , "F3 0F 38 F8 /r"               , "ENQCMD X64"],
    
        ["movdiri"          , "W:m32, r32"                                      , "MR"      , "0F 38 F9 /r"                  , "MOVDIRI"],
        ["movdiri"          , "W:m64, r64"                                      , "MR"      , "REX.W 0F 38 F9 /r"            , "MOVDIRI X64"],
        ["movdir64b"        , "W:es:r32, m512"                                  , "RM"      , "66 0F 38 F8 /r"               , "MOVDIR64B"],
        ["movdir64b"        , "W:es:r64, m512"                                  , "RM"      , "66 0F 38 F8 /r"               , "MOVDIR64B X64"],
    
        ["wbnoinvd"         , ""                                                , "NONE"    , "F3 0F 09"                     , "WBNOINVD"],
    
        ["getsec"           , ""                                                , "NONE"    , "0F 37"                        , "SMX"],
        ["pconfig"          , ""                                                , "NONE"    , "0F 01 C5"                     , "PCONFIG          PRIVILEGE=L0"],
    
        ["tpause"           , "R:r32, <edx>, <eax>"                             , "M"       , "66 0F AE /6"                  , "WAITPKG          OF=0 SF=0 ZF=0 AF=0 PF=0 CF=W"],
        ["umonitor"         , "R:ds:r32"                                        , "M"       , "F3 0F AE /6"                  , "WAITPKG"],
        ["umonitor"         , "R:ds:r64"                                        , "M"       , "F3 0F AE /6"                  , "WAITPKG X64"],
        ["umwait"           , "R:r32, <edx>, <eax>"                             , "M"       , "F2 0F AE /6"                  , "WAITPKG          OF=0 SF=0 ZF=0 AF=0 PF=0 CF=W"],
    
        ["invept"           , "R:r32, R:m128"                                   , "RM"      , "66 0F 38 80 /r"               , "VMX X86          PRIVILEGE=L0"],
        ["invept"           , "R:r64, R:m128"                                   , "RM"      , "66 0F 38 80 /r"               , "VMX X64          PRIVILEGE=L0"],
        ["invvpid"          , "R:r32, R:m128"                                   , "RM"      , "66 0F 38 81 /r"               , "VMX X86          PRIVILEGE=L0"],
        ["invvpid"          , "R:r64, R:m128"                                   , "RM"      , "66 0F 38 81 /r"               , "VMX X64          PRIVILEGE=L0"],
        ["vmcall"           , ""                                                , "NONE"    , "0F 01 C1"                     , "VMX              PRIVILEGE=L0"],
        ["vmclear"          , "W:m64"                                           , "M"       , "66 0F C7 /6"                  , "VMX              PRIVILEGE=L0"],
        ["vmfunc"           , ""                                                , "NONE"    , "0F 01 D4"                     , "VMX"],
        ["vmlaunch"         , ""                                                , "NONE"    , "0F 01 C2"                     , "VMX              PRIVILEGE=L0"],
        ["vmptrld"          , "R:m64"                                           , "M"       , "0F C7 /6"                     , "VMX              PRIVILEGE=L0"],
        ["vmptrst"          , "W:m64"                                           , "M"       , "0F C7 /7"                     , "VMX              PRIVILEGE=L0"],
        ["vmread"           , "W:r32/m32, R:r32"                                , "MR"      , "0F 78 /r"                     , "VMX X86          PRIVILEGE=L0"],
        ["vmread"           , "W:r64/m64, R:r64"                                , "MR"      , "0F 78 /r"                     , "VMX X64          PRIVILEGE=L0"],
        ["vmresume"         , ""                                                , "NONE"    , "0F 01 C3"                     , "VMX              PRIVILEGE=L0"],
        ["vmwrite"          , "R:r32, R:r32/m32"                                , "RM"      , "0F 79 /r"                     , "VMX X86          PRIVILEGE=L0"],
        ["vmwrite"          , "R:r64, R:r64/m64"                                , "RM"      , "0F 79 /r"                     , "VMX X64          PRIVILEGE=L0"],
        ["vmxon"            , "R:m64"                                           , "M"       , "F3 0F C7 /6"                  , "VMX"],
    
        ["skinit"           , "X:<eax>"                                         , "NONE"    , "0F 01 DE"                     , "SKINIT           PRIVILEGE=L0"],
        ["stgi"             , ""                                                , "NONE"    , "0F 01 DC"                     , "SKINIT           PRIVILEGE=L0"],
        ["clgi"             , ""                                                , "NONE"    , "0F 01 DD"                     , "SVM              PRIVILEGE=L0"],
        ["invlpga"          , "R:<eax>, R:<ecx>"                                , "NONE"    , "0F 01 DF"                     , "SVM X86          PRIVILEGE=L0"],
        ["invlpga"          , "R:<eax>, R:<ecx>"                                , "NONE"    , "67 0F 01 DF"                  , "SVM X64          PRIVILEGE=L0"],
        ["invlpga"          , "R:<rax>, R:<ecx>"                                , "NONE"    , "0F 01 DF"                     , "SVM X64          PRIVILEGE=L0"],
        ["vmload"           , "R:<eax>"                                         , "NONE"    , "0F 01 DA"                     , "SVM X86          PRIVILEGE=L0"],
        ["vmload"           , "R:<rax>"                                         , "NONE"    , "0F 01 DA"                     , "SVM X64          PRIVILEGE=L0"],
        ["vmmcall"          , ""                                                , "NONE"    , "0F 01 D9"                     , "SVM"],
        ["vmrun"            , "X:<eax>"                                         , "NONE"    , "0F 01 D8"                     , "SVM X86          PRIVILEGE=L0"],
        ["vmrun"            , "X:<rax>"                                         , "NONE"    , "0F 01 D8"                     , "SVM X64          PRIVILEGE=L0"],
        ["vmsave"           , "R:<eax>"                                         , "NONE"    , "0F 01 DB"                     , "SVM X86          PRIVILEGE=L0"],
        ["vmsave"           , "R:<rax>"                                         , "NONE"    , "0F 01 DB"                     , "SVM X64          PRIVILEGE=L0"],
    
        ["f2xm1"            , ""                                                , "NONE"    , "D9 F0"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fabs"             , ""                                                , "NONE"    , "D9 E1"                        , "FPU              C0=U C1=0 C2=U C3=U"],
        ["fadd"             , "R:m32fp"                                         , "M"       , "D8 /0"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fadd"             , "R:m64fp"                                         , "M"       , "DC /0"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fadd"             , "st(0), st(i)"                                    , "O"       , "D8 C0+i"                      , "FPU              C0=U C1=W C2=U C3=U"],
        ["fadd"             , "st(i), st(0)"                                    , "O"       , "DC C0+i"                      , "FPU              C0=U C1=W C2=U C3=U"],
        ["faddp"            , ""                                                , "NONE"    , "DE C1"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["faddp"            , "st(i)"                                           , "O"       , "DE C0+i"                      , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fbld"             , "R:m80dec"                                        , "M"       , "DF /4"                        , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fbstp"            , "W:m80bcd"                                        , "M"       , "DF /6"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fchs"             , ""                                                , "NONE"    , "D9 E0"                        , "FPU              C0=U C1=0 C2=U C3=U"],
        ["fclex"            , ""                                                , "NONE"    , "9B DB E2"                     , "FPU              C0=U C1=U C2=U C3=U"],
        ["fcmovb"           , "st(i)"                                           , "O"       , "DA C0+i"                      , "FPU CMOV         C0=U C1=W C2=U C3=U CF=R"],
        ["fcmovbe"          , "st(i)"                                           , "O"       , "DA D0+i"                      , "FPU CMOV         C0=U C1=W C2=U C3=U CF=R ZF=R"],
        ["fcmove"           , "st(i)"                                           , "O"       , "DA C8+i"                      , "FPU CMOV         C0=U C1=W C2=U C3=U ZF=R"],
        ["fcmovnb"          , "st(i)"                                           , "O"       , "DB C0+i"                      , "FPU CMOV         C0=U C1=W C2=U C3=U CF=R"],
        ["fcmovnbe"         , "st(i)"                                           , "O"       , "DB D0+i"                      , "FPU CMOV         C0=U C1=W C2=U C3=U CF=R ZF=R"],
        ["fcmovne"          , "st(i)"                                           , "O"       , "DB C8+i"                      , "FPU CMOV         C0=U C1=W C2=U C3=U ZF=R"],
        ["fcmovnu"          , "st(i)"                                           , "O"       , "DB D8+i"                      , "FPU CMOV         C0=U C1=W C2=U C3=U PF=R"],
        ["fcmovu"           , "st(i)"                                           , "O"       , "DA D8+i"                      , "FPU CMOV         C0=U C1=W C2=U C3=U PF=R"],
        ["fcom"             , ""                                                , "NONE"    , "D8 D1"                        , "FPU              C0=W C1=0 C2=W C3=W"],
        ["fcom"             , "R:m32fp"                                         , "M"       , "D8 /2"                        , "FPU              C0=W C1=0 C2=W C3=W"],
        ["fcom"             , "R:m64fp"                                         , "M"       , "DC /2"                        , "FPU              C0=W C1=0 C2=W C3=W"],
        ["fcom"             , "R:st(i)"                                         , "O"       , "D8 D0+i"                      , "FPU              C0=W C1=0 C2=W C3=W"],
        ["fcomi"            , "R:st(i)"                                         , "O"       , "DB F0+i"                      , "FPU              C1=0 ZF=W PF=W CF=W"],
        ["fcomip"           , "R:st(i)"                                         , "O"       , "DF F0+i"                      , "FPU_POP          C1=0 ZF=W PF=W CF=W"],
        ["fcomp"            , ""                                                , "NONE"    , "D8 D9"                        , "FPU_POP          C0=W C1=W C2=W C3=W"],
        ["fcomp"            , "R:m32fp"                                         , "M"       , "D8 /3"                        , "FPU_POP          C0=W C1=W C2=W C3=W"],
        ["fcomp"            , "R:m64fp"                                         , "M"       , "DC /3"                        , "FPU_POP          C0=W C1=W C2=W C3=W"],
        ["fcomp"            , "R:st(i)"                                         , "O"       , "D8 D8+i"                      , "FPU_POP          C0=W C1=W C2=W C3=W"],
        ["fcompp"           , ""                                                , "NONE"    , "DE D9"                        , "FPU_POP=2        C0=W C1=W C2=W C3=W"],
        ["fcos"             , ""                                                , "NONE"    , "D9 FF"                        , "FPU              C0=U C1=W C2=W C3=U"],
        ["fdecstp"          , ""                                                , "NONE"    , "D9 F6"                        , "FPU_TOP=-1       C0=U C1=W C2=U C3=U"],
        ["fdiv"             , "R:m32fp"                                         , "M"       , "D8 /6"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fdiv"             , "R:m64fp"                                         , "M"       , "DC /6"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fdiv"             , "st(0), st(i)"                                    , "O"       , "D8 F0+i"                      , "FPU              C0=U C1=W C2=U C3=U"],
        ["fdiv"             , "st(i), st(0)"                                    , "O"       , "DC F8+i"                      , "FPU              C0=U C1=W C2=U C3=U"],
        ["fdivp"            , ""                                                , "NONE"    , "DE F9"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fdivp"            , "st(i)"                                           , "O"       , "DE F8+i"                      , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fdivr"            , "R:m32fp"                                         , "M"       , "D8 /7"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fdivr"            , "R:m64fp"                                         , "M"       , "DC /7"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fdivr"            , "st(0), st(i)"                                    , "O"       , "D8 F8+i"                      , "FPU              C0=U C1=W C2=U C3=U"],
        ["fdivr"            , "st(i), st(0)"                                    , "O"       , "DC F0+i"                      , "FPU              C0=U C1=W C2=U C3=U"],
        ["fdivrp"           , ""                                                , "NONE"    , "DE F1"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fdivrp"           , "st(i)"                                           , "O"       , "DE F0+i"                      , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["ffree"            , "st(i)"                                           , "O"       , "DD C0+i"                      , "FPU              C0=U C1=U C2=U C3=U"],
        ["fiadd"            , "R:m16int"                                        , "M"       , "DE /0"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fiadd"            , "R:m32int"                                        , "M"       , "DA /0"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["ficom"            , "R:m16int"                                        , "M"       , "DE /2"                        , "FPU              C0=W C1=0 C2=W C3=W"],
        ["ficom"            , "R:m32int"                                        , "M"       , "DA /2"                        , "FPU              C0=W C1=0 C2=W C3=W"],
        ["ficomp"           , "R:m16int"                                        , "M"       , "DE /3"                        , "FPU              C0=W C1=0 C2=W C3=W"],
        ["ficomp"           , "R:m32int"                                        , "M"       , "DA /3"                        , "FPU              C0=W C1=0 C2=W C3=W"],
        ["fidiv"            , "R:m16int"                                        , "M"       , "DE /6"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fidiv"            , "R:m32int"                                        , "M"       , "DA /6"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fidivr"           , "R:m16int"                                        , "M"       , "DE /7"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fidivr"           , "R:m32int"                                        , "M"       , "DA /7"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fild"             , "R:m16int"                                        , "M"       , "DF /0"                        , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fild"             , "R:m32int"                                        , "M"       , "DB /0"                        , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fild"             , "R:m64int"                                        , "M"       , "DF /5"                        , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fimul"            , "R:m16int"                                        , "M"       , "DE /1"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fimul"            , "R:m32int"                                        , "M"       , "DA /1"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fincstp"          , ""                                                , "NONE"    , "D9 F7"                        , "FPU_TOP=+1       C0=U C1=0 C2=U C3=U"],
        ["finit"            , ""                                                , "NONE"    , "9B DB E3"                     , "FPU              C0=0 C1=0 C2=0 C3=0"],
        ["fist"             , "W:m16int"                                        , "M"       , "DF /2"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fist"             , "W:m32int"                                        , "M"       , "DB /2"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fistp"            , "W:m16int"                                        , "M"       , "DF /3"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fistp"            , "W:m32int"                                        , "M"       , "DB /3"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fistp"            , "W:m64int"                                        , "M"       , "DF /7"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fisttp"           , "W:m16int"                                        , "M"       , "DF /1"                        , "FPU_POP SSE3     C0=U C1=0 C2=U C3=U"],
        ["fisttp"           , "W:m32int"                                        , "M"       , "DB /1"                        , "FPU_POP SSE3     C0=U C1=0 C2=U C3=U"],
        ["fisttp"           , "W:m64int"                                        , "M"       , "DD /1"                        , "FPU_POP SSE3     C0=U C1=0 C2=U C3=U"],
        ["fisub"            , "R:m16int"                                        , "M"       , "DE /4"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fisub"            , "R:m32int"                                        , "M"       , "DA /4"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fisubr"           , "R:m16int"                                        , "M"       , "DE /5"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fisubr"           , "R:m32int"                                        , "M"       , "DA /5"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fld"              , "R:m32fp"                                         , "M"       , "D9 /0"                        , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fld"              , "R:m64fp"                                         , "M"       , "DD /0"                        , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fld"              , "R:m80fp"                                         , "M"       , "DB /5"                        , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fld"              , "R:st(i)"                                         , "O"       , "D9 C0+i"                      , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fld1"             , ""                                                , "NONE"    , "D9 E8"                        , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fldcw"            , "R:m16"                                           , "M"       , "D9 /5"                        , "FPU              C0=U C1=U C2=U C3=U"],
        ["fldenv"           , "R:mem"                                           , "M"       , "D9 /4"                        , "FPU              C0=W C1=W C2=W C3=W"],
        ["fldl2e"           , ""                                                , "NONE"    , "D9 EA"                        , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fldl2t"           , ""                                                , "NONE"    , "D9 E9"                        , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fldlg2"           , ""                                                , "NONE"    , "D9 EC"                        , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fldln2"           , ""                                                , "NONE"    , "D9 ED"                        , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fldpi"            , ""                                                , "NONE"    , "D9 EB"                        , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fldz"             , ""                                                , "NONE"    , "D9 EE"                        , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fmul"             , "R:m32fp"                                         , "M"       , "D8 /1"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fmul"             , "R:m64fp"                                         , "M"       , "DC /1"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fmul"             , "st(0), st(i)"                                    , "O"       , "D8 C8+i"                      , "FPU              C0=U C1=W C2=U C3=U"],
        ["fmul"             , "st(i), st(0)"                                    , "O"       , "DC C8+i"                      , "FPU              C0=U C1=W C2=U C3=U"],
        ["fmulp"            , ""                                                , "NONE"    , "DE C9"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fmulp"            , "st(i)"                                           , "O"       , "DE C8+i"                      , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fnclex"           , ""                                                , "NONE"    , "DB E2"                        , "FPU              C0=U C1=U C2=U C3=U"],
        ["fninit"           , ""                                                , "NONE"    , "DB E3"                        , "FPU              C0=0 C1=0 C2=0 C3=0"],
        ["fnop"             , ""                                                , "NONE"    , "D9 D0"                        , "FPU              C0=U C1=U C2=U C3=U"],
        ["fnsave"           , "W:mem"                                           , "M"       , "DD /6"                        , "FPU              C0=W C1=W C2=W C3=W"],
        ["fnstcw"           , "W:m16"                                           , "M"       , "D9 /7"                        , "FPU              C0=U C1=U C2=U C3=U"],
        ["fnstenv"          , "W:mem"                                           , "M"       , "D9 /6"                        , "FPU              C0=U C1=U C2=U C3=U"],
        ["fnstsw"           , "w:ax"                                            , "NONE"    , "DF E0"                        , "FPU              C0=U C1=U C2=U C3=U"],
        ["fnstsw"           , "W:m16"                                           , "M"       , "DD /7"                        , "FPU              C0=U C1=U C2=U C3=U"],
        ["fpatan"           , ""                                                , "NONE"    , "D9 F3"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fprem"            , ""                                                , "NONE"    , "D9 F8"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fprem1"           , ""                                                , "NONE"    , "D9 F5"                        , "FPU              C0=W C1=W C2=W C3=W"],
        ["fptan"            , ""                                                , "NONE"    , "D9 F2"                        , "FPU_POP          C0=U C1=W C2=W C3=U"],
        ["frndint"          , ""                                                , "NONE"    , "D9 FC"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["frstor"           , "R:mem"                                           , "M"       , "DD /4"                        , "FPU              C0=W C1=W C2=W C3=W"],
        ["fsave"            , "W:mem"                                           , "M"       , "9B DD /6"                     , "FPU              C0=W C1=W C2=W C3=W"],
        ["fscale"           , ""                                                , "NONE"    , "D9 FD"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fsin"             , ""                                                , "NONE"    , "D9 FE"                        , "FPU              C0=U C1=W C2=W C3=U"],
        ["fsincos"          , ""                                                , "NONE"    , "D9 FB"                        , "FPU_PUSH         C0=U C1=W C2=W C3=U"],
        ["fsqrt"            , ""                                                , "NONE"    , "D9 FA"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fst"              , "W:m32fp"                                         , "M"       , "D9 /2"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fst"              , "W:m64fp"                                         , "M"       , "DD /2"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fst"              , "W:st(i)"                                         , "O"       , "DD D0+i"                      , "FPU              C0=U C1=W C2=U C3=U"],
        ["fstcw"            , "W:m16"                                           , "M"       , "9B D9 /7"                     , "FPU              C0=U C1=U C2=U C3=U"],
        ["fstenv"           , "W:mem"                                           , "M"       , "9B D9 /6"                     , "FPU              C0=U C1=U C2=U C3=U"],
        ["fstp"             , "W:m32fp"                                         , "M"       , "D9 /3"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fstp"             , "W:m64fp"                                         , "M"       , "DD /3"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fstp"             , "W:m80fp"                                         , "M"       , "DB /7"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fstp"             , "W:st(i)"                                         , "O"       , "DD D8+i"                      , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fstsw"            , "w:ax"                                            , "NONE"    , "9B DF E0"                     , "FPU              C0=U C1=U C2=U C3=U"],
        ["fstsw"            , "W:m16"                                           , "M"       , "9B DD /7"                     , "FPU              C0=U C1=U C2=U C3=U"],
        ["fsub"             , "R:m32fp"                                         , "M"       , "D8 /4"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fsub"             , "R:m64fp"                                         , "M"       , "DC /4"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fsub"             , "st(0), st(i)"                                    , "O"       , "D8 E0+i"                      , "FPU              C0=U C1=W C2=U C3=U"],
        ["fsub"             , "st(i), st(0)"                                    , "O"       , "DC E8+i"                      , "FPU              C0=U C1=W C2=U C3=U"],
        ["fsubp"            , ""                                                , "NONE"    , "DE E9"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fsubp"            , "st(i)"                                           , "O"       , "DE E8+i"                      , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fsubr"            , "R:m32fp"                                         , "M"       , "D8 /5"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fsubr"            , "R:m64fp"                                         , "M"       , "DC /5"                        , "FPU              C0=U C1=W C2=U C3=U"],
        ["fsubr"            , "st(0), st(i)"                                    , "O"       , "D8 E8+i"                      , "FPU              C0=U C1=W C2=U C3=U"],
        ["fsubr"            , "st(i), st(0)"                                    , "O"       , "DC E0+i"                      , "FPU              C0=U C1=W C2=U C3=U"],
        ["fsubrp"           , ""                                                , "NONE"    , "DE E1"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fsubrp"           , "st(i)"                                           , "O"       , "DE E0+i"                      , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["ftst"             , ""                                                , "NONE"    , "D9 E4"                        , "FPU              C0=W C1=0 C2=W C3=W"],
        ["fucom"            , ""                                                , "NONE"    , "DD E1"                        , "FPU              C0=W C1=0 C2=W C3=W"],
        ["fucom"            , "R:st(i)"                                         , "O"       , "DD E0+i"                      , "FPU              C0=W C1=0 C2=W C3=W"],
        ["fucomi"           , "R:st(i)"                                         , "O"       , "DB E8+i"                      , "FPU              C1=0 ZF=W PF=W CF=W"],
        ["fucomip"          , "R:st(i)"                                         , "O"       , "DF E8+i"                      , "FPU_POP          C1=0 ZF=W PF=W CF=W"],
        ["fucomp"           , ""                                                , "NONE"    , "DD E9"                        , "FPU_POP          C0=W C1=W C2=W C3=W"],
        ["fucomp"           , "R:st(i)"                                         , "O"       , "DD E8+i"                      , "FPU_POP          C0=W C1=W C2=W C3=W"],
        ["fucompp"          , ""                                                , "NONE"    , "DA E9"                        , "FPU_POP=2        C0=W C1=W C2=W C3=W"],
        ["fwait/wait"       , ""                                                , "NONE"    , "DB"                           , "FPU              C0=U C1=U C2=U C3=U"],
        ["fxam"             , ""                                                , "NONE"    , "D9 E5"                        , "FPU              C0=W C1=W C2=W C3=W"],
        ["fxch"             , ""                                                , "NONE"    , "D9 C9"                        , "FPU              C0=U C1=0 C2=U C3=U"],
        ["fxch"             , "st(i)"                                           , "O"       , "D9 C8+i"                      , "FPU              C0=U C1=0 C2=U C3=U"],
        ["fxtract"          , ""                                                , "NONE"    , "D9 F4"                        , "FPU_PUSH         C0=U C1=W C2=U C3=U"],
        ["fyl2x"            , ""                                                , "NONE"    , "D9 F1"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
        ["fyl2xp1"          , ""                                                , "NONE"    , "D9 F9"                        , "FPU_POP          C0=U C1=W C2=U C3=U"],
    
        ["addpd"            , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 58 /r"                  , "SSE2"],
        ["addps"            , "X:~xmm, ~xmm/m128"                               , "RM"      , "0F 58 /r"                     , "SSE"],
        ["addsd"            , "x:xmm[63:0], xmm[63:0]/m64"                      , "RM"      , "F2 0F 58 /r"                  , "SSE2"],
        ["addss"            , "x:xmm[31:0], xmm[31:0]/m32"                      , "RM"      , "F3 0F 58 /r"                  , "SSE"],
        ["addsubpd"         , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F D0 /r"                  , "SSE3"],
        ["addsubps"         , "X:xmm, xmm/m128"                                 , "RM"      , "F2 0F D0 /r"                  , "SSE3"],
        ["andnpd"           , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 55 /r"                  , "SSE2"],
        ["andnps"           , "X:xmm, xmm/m128"                                 , "RM"      , "0F 55 /r"                     , "SSE"],
        ["andpd"            , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 54 /r"                  , "SSE2"],
        ["andps"            , "X:~xmm, ~xmm/m128"                               , "RM"      , "0F 54 /r"                     , "SSE"],
        ["blendpd"          , "X:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F 3A 0D /r ib"            , "SSE4_1"],
        ["blendps"          , "X:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F 3A 0C /r ib"            , "SSE4_1"],
        ["blendvpd"         , "X:xmm, xmm/m128, <xmm0>"                         , "RM"      , "66 0F 38 15 /r"               , "SSE4_1"],
        ["blendvps"         , "X:xmm, xmm/m128, <xmm0>"                         , "RM"      , "66 0F 38 14 /r"               , "SSE4_1"],
        ["cmppd"            , "X:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F C2 /r ib"               , "SSE2"],
        ["cmpps"            , "X:xmm, xmm/m128, ib/ub"                          , "RMI"     , "0F C2 /r ib"                  , "SSE"],
        ["cmpsd"            , "x:xmm[63:0], xmm[63:0]/m64, ib/ub"               , "RMI"     , "F2 0F C2 /r ib"               , "SSE2"],
        ["cmpss"            , "x:xmm[31:0], xmm[31:0]/m32, ib/ub"               , "RMI"     , "F3 0F C2 /r ib"               , "SSE"],
        ["comisd"           , "R:xmm[63:0], xmm[63:0]/m64"                      , "RM"      , "66 0F 2F /r"                  , "SSE2 OF=0 SF=0 ZF=W AF=0 PF=W CF=W"],
        ["comiss"           , "R:xmm[31:0], xmm[31:0]/m32"                      , "RM"      , "0F 2F /r"                     , "SSE  OF=0 SF=0 ZF=W AF=0 PF=W CF=W"],
        ["cvtdq2pd"         , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "F3 0F E6 /r"                  , "SSE2"],
        ["cvtdq2ps"         , "W:xmm, xmm/m128"                                 , "RM"      , "0F 5B /r"                     , "SSE2"],
        ["cvtpd2dq"         , "W:xmm[63:0], xmm/m128"                           , "RM"      , "F2 0F E6 /r"                  , "SSE2"],
        ["cvtpd2pi"         , "W:mm, xmm/m128"                                  , "RM"      , "66 0F 2D /r"                  , "SSE2"],
        ["cvtpd2ps"         , "W:xmm[63:0], xmm/m128"                           , "RM"      , "66 0F 5A /r"                  , "SSE2"],
        ["cvtpi2pd"         , "W:xmm, R:mm[63:0]/m64"                           , "RM"      , "66 0F 2A /r"                  , "SSE2"],
        ["cvtpi2ps"         , "w:xmm[63:0], mm/m64"                             , "RM"      , "0F 2A /r"                     , "SSE"],
        ["cvtps2dq"         , "W:xmm, xmm/m128"                                 , "RM"      , "66 0F 5B /r"                  , "SSE2"],
        ["cvtps2pd"         , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "0F 5A /r"                     , "SSE2"],
        ["cvtps2pi"         , "W:mm, xmm[63:0]/m64"                             , "RM"      , "0F 2D /r"                     , "SSE"],
        ["cvtsd2si"         , "W:r32, xmm[63:0]/m64"                            , "RM"      , "F2 0F 2D /r"                  , "SSE2"],
        ["cvtsd2si"         , "W:r64, xmm[63:0]/m64"                            , "RM"      , "REX.W F2 0F 2D /r"            , "SSE2 X64"],
        ["cvtsd2ss"         , "w:xmm[31:0], xmm[63:0]/m64"                      , "RM"      , "F2 0F 5A /r"                  , "SSE2"],
        ["cvtsi2sd"         , "w:xmm[63:0], r32/m32"                            , "RM"      , "F2 0F 2A /r"                  , "SSE2"],
        ["cvtsi2sd"         , "w:xmm[63:0], r64/m64"                            , "RM"      , "REX.W F2 0F 2A /r"            , "SSE2 X64"],
        ["cvtsi2ss"         , "w:xmm[31:0], r32/m32"                            , "RM"      , "F3 0F 2A /r"                  , "SSE"],
        ["cvtsi2ss"         , "w:xmm[31:0], r64/m64"                            , "RM"      , "REX.W F3 0F 2A /r"            , "SSE X64"],
        ["cvtss2sd"         , "w:xmm[63:0], xmm[31:0]/m32"                      , "RM"      , "F3 0F 5A /r"                  , "SSE2"],
        ["cvtss2si"         , "W:r32, xmm[31:0]/m32"                            , "RM"      , "F3 0F 2D /r"                  , "SSE"],
        ["cvtss2si"         , "W:r64, xmm[31:0]/m32"                            , "RM"      , "REX.W F3 0F 2D /r"            , "SSE X64"],
        ["cvttpd2dq"        , "W:xmm[63:0], xmm/m128"                           , "RM"      , "66 0F E6 /r"                  , "SSE2"],
        ["cvttpd2pi"        , "W:mm, xmm/m128"                                  , "RM"      , "66 0F 2C /r"                  , "SSE2"],
        ["cvttps2dq"        , "W:xmm, xmm/m128"                                 , "RM"      , "F3 0F 5B /r"                  , "SSE2"],
        ["cvttps2pi"        , "W:mm, xmm[63:0]/m64"                             , "RM"      , "0F 2C /r"                     , "SSE"],
        ["cvttsd2si"        , "W:r32, xmm[63:0]/m64"                            , "RM"      , "F2 0F 2C /r"                  , "SSE2"],
        ["cvttsd2si"        , "W:r64, xmm[63:0]/m64"                            , "RM"      , "REX.W F2 0F 2C /r"            , "SSE2 X64"],
        ["cvttss2si"        , "W:r32, xmm[31:0]/m32"                            , "RM"      , "F3 0F 2C /r"                  , "SSE"],
        ["cvttss2si"        , "W:r64, xmm[31:0]/m32"                            , "RM"      , "REX.W F3 0F 2C /r"            , "SSE X64"],
        ["divpd"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 5E /r"                  , "SSE2"],
        ["divps"            , "X:xmm, xmm/m128"                                 , "RM"      , "0F 5E /r"                     , "SSE"],
        ["divsd"            , "x:xmm[63:0], xmm[63:0]/m64"                      , "RM"      , "F2 0F 5E /r"                  , "SSE2"],
        ["divss"            , "x:xmm[31:0], xmm[31:0]/m32"                      , "RM"      , "F3 0F 5E /r"                  , "SSE"],
        ["dppd"             , "X:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F 3A 41 /r ib"            , "SSE4_1"],
        ["dpps"             , "X:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F 3A 40 /r ib"            , "SSE4_1"],
        ["extractps"        , "W:r32/m32, xmm, ib/ub"                           , "MRI"     , "66 0F 3A 17 /r ib"            , "SSE4_1"],
        ["extrq"            , "X:xmm, ib/ub, ib/ub"                             , "RII"     , "66 0F 78 /0 ib ib"            , "SSE4A"],
        ["extrq"            , "X:xmm, xmm"                                      , "RM"      , "66 0F 79 /r"                  , "SSE4A"],
        ["haddpd"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 7C /r"                  , "SSE3"],
        ["haddps"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "F2 0F 7C /r"                  , "SSE3"],
        ["hsubpd"           , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 7D /r"                  , "SSE3"],
        ["hsubps"           , "X:xmm, xmm/m128"                                 , "RM"      , "F2 0F 7D /r"                  , "SSE3"],
        ["insertps"         , "X:xmm, xmm[31:0]/m32, ib/ub"                     , "RMI"     , "66 0F 3A 21 /r ib"            , "SSE4_1"],
        ["insertq"          , "X:xmm, xmm"                                      , "RM"      , "F2 0F 79 /r"                  , "SSE4A"],
        ["insertq"          , "X:xmm, xmm, ib/ub, ib/ub"                        , "RMII"    , "F2 0F 78 /r ib ib"            , "SSE4A"],
        ["lddqu"            , "W:xmm, m128"                                     , "RM"      , "F2 0F F0 /r"                  , "SSE3"],
        ["maskmovdqu"       , "R:xmm, xmm, X:<ds:zdi>"                          , "RM"      , "66 0F F7 /r"                  , "SSE2"],
        ["maskmovq"         , "R:mm, mm, X:<ds:zdi>"                            , "RM"      , "0F F7 /r"                     , "MMX2"],
        ["maxpd"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 5F /r"                  , "SSE2"],
        ["maxps"            , "X:xmm, xmm/m128"                                 , "RM"      , "0F 5F /r"                     , "SSE"],
        ["maxsd"            , "x:xmm[63:0], xmm[63:0]/m64"                      , "RM"      , "F2 0F 5F /r"                  , "SSE2"],
        ["maxss"            , "x:xmm[31:0], xmm[31:0]/m32"                      , "RM"      , "F3 0F 5F /r"                  , "SSE"],
        ["minpd"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 5D /r"                  , "SSE2"],
        ["minps"            , "X:xmm, xmm/m128"                                 , "RM"      , "0F 5D /r"                     , "SSE"],
        ["minsd"            , "x:xmm[63:0], xmm[63:0]/m64"                      , "RM"      , "F2 0F 5D /r"                  , "SSE2"],
        ["minss"            , "x:xmm[31:0], xmm[31:0]/m32"                      , "RM"      , "F3 0F 5D /r"                  , "SSE"],
        ["movapd"           , "W:xmm, xmm/m128"                                 , "RM"      , "66 0F 28 /r"                  , "SSE2"],
        ["movapd"           , "W:xmm/m128, xmm"                                 , "MR"      , "66 0F 29 /r"                  , "SSE2"],
        ["movaps"           , "W:xmm, xmm/m128"                                 , "RM"      , "0F 28 /r"                     , "SSE"],
        ["movaps"           , "W:xmm/m128, xmm"                                 , "MR"      , "0F 29 /r"                     , "SSE"],
        ["movd"             , "W:mm[31:0], R:r32[31:0]/m32"                     , "RM"      , "0F 6E /r"                     , "MMX"],
        ["movd"             , "W:r32[31:0]/m32, R:mm[31:0]"                     , "MR"      , "0F 7E /r"                     , "MMX"],
        ["movd"             , "W:r32[31:0]/m32, xmm[31:0]"                      , "MR"      , "66 0F 7E /r"                  , "SSE2"],
        ["movd"             , "W:xmm[31:0], R:r32[31:0]/m32"                    , "RM"      , "66 0F 6E /r"                  , "SSE2"],
        ["movddup"          , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "F2 0F 12 /r"                  , "SSE3"],
        ["movdq2q"          , "W:mm, xmm[63:0]"                                 , "RM"      , "F2 0F D6 /r"                  , "SSE2"],
        ["movdqa"           , "W:xmm, xmm/m128"                                 , "RM"      , "66 0F 6F /r"                  , "SSE2"],
        ["movdqa"           , "W:xmm/m128, xmm"                                 , "MR"      , "66 0F 7F /r"                  , "SSE2"],
        ["movdqu"           , "W:xmm, xmm/m128"                                 , "RM"      , "F3 0F 6F /r"                  , "SSE2"],
        ["movdqu"           , "W:xmm/m128, xmm"                                 , "MR"      , "F3 0F 7F /r"                  , "SSE2"],
        ["movhlps"          , "w:xmm[63:0], xmm[127:64]"                        , "RM"      , "0F 12 /r"                     , "SSE"],
        ["movhpd"           , "W:m64, xmm[127:64]"                              , "MR"      , "66 0F 17 /r"                  , "SSE2"],
        ["movhpd"           , "w:xmm[127:64], m64"                              , "RM"      , "66 0F 16 /r"                  , "SSE2"],
        ["movhps"           , "W:m64, xmm[127:64]"                              , "MR"      , "0F 17 /r"                     , "SSE"],
        ["movhps"           , "w:xmm[127:64], m64"                              , "RM"      , "0F 16 /r"                     , "SSE"],
        ["movlhps"          , "w:xmm[127:64], xmm[63:0]"                        , "RM"      , "0F 16 /r"                     , "SSE"],
        ["movlpd"           , "W:m64, xmm[63:0]"                                , "MR"      , "66 0F 13 /r"                  , "SSE2"],
        ["movlpd"           , "w:xmm[63:0], m64"                                , "RM"      , "66 0F 12 /r"                  , "SSE2"],
        ["movlps"           , "W:m64, xmm[63:0]"                                , "MR"      , "0F 13 /r"                     , "SSE"],
        ["movlps"           , "w:xmm[63:0], m64"                                , "RM"      , "0F 12 /r"                     , "SSE"],
        ["movmskpd"         , "W:r32[1:0], xmm"                                 , "RM"      , "66 0F 50 /r"                  , "SSE2"],
        ["movmskps"         , "W:r32[3:0], xmm"                                 , "RM"      , "0F 50 /r"                     , "SSE"],
        ["movntdq"          , "W:m128, xmm"                                     , "MR"      , "66 0F E7 /r"                  , "SSE2"],
        ["movntdqa"         , "W:xmm, m128"                                     , "RM"      , "66 0F 38 2A /r"               , "SSE4_1"],
        ["movnti"           , "W:m32, r32"                                      , "MR"      , "0F C3 /r"                     , "SSE2"],
        ["movnti"           , "W:m64, r64"                                      , "MR"      , "REX.W 0F C3 /r"               , "SSE2 X64"],
        ["movntpd"          , "W:m128, xmm"                                     , "MR"      , "66 0F 2B /r"                  , "SSE2"],
        ["movntps"          , "W:m128, xmm"                                     , "MR"      , "0F 2B /r"                     , "SSE"],
        ["movntq"           , "W:m64, mm"                                       , "MR"      , "0F E7 /r"                     , "MMX2"],
        ["movntsd"          , "W:m64, xmm[63:0]"                                , "RM"      , "F2 0F 2B /r"                  , "SSE4A"],
        ["movntss"          , "W:m32, xmm[31:0]"                                , "RM"      , "F3 0F 2B /r"                  , "SSE4A"],
        ["movq"             , "W:mm, mm/m64"                                    , "RM"      , "0F 6F /r"                     , "MMX"],
        ["movq"             , "W:mm, r64/m64"                                   , "RM"      , "REX.W 0F 6E /r"               , "MMX X64"],
        ["movq"             , "W:mm/m64, mm"                                    , "MR"      , "0F 7F /r"                     , "MMX"],
        ["movq"             , "W:r64/m64, mm"                                   , "MR"      , "REX.W 0F 7E /r"               , "MMX X64"],
        ["movq"             , "W:r64/m64, xmm[63:0]"                            , "MR"      , "REX.W 66 0F 7E /r"            , "SSE2 X64"],
        ["movq"             , "W:xmm[63:0], r64[63:0]/m64"                      , "RM"      , "REX.W 66 0F 6E /r"            , "SSE2 X64"],
        ["movq"             , "W:xmm[63:0], xmm[63:0]/m64"                      , "RM"      , "F3 0F 7E /r"                  , "SSE2"],
        ["movq"             , "W:xmm[63:0]/m64, xmm[63:0]"                      , "MR"      , "66 0F D6 /r"                  , "SSE2"],
        ["movq2dq"          , "W:xmm[63:0], mm"                                 , "RM"      , "F3 0F D6 /r"                  , "SSE2"],
        ["movsd"            , "w:xmm[63:0], xmm[63:0]"                          , "RM"      , "F2 0F 10 /r"                  , "SSE2"],
        ["movsd"            , "W:xmm[63:0], m64"                                , "RM"      , "F2 0F 10 /r"                  , "SSE2"],
        ["movsd"            , "W:m64, xmm[63:0]"                                , "MR"      , "F2 0F 11 /r"                  , "SSE2"],
        ["movshdup"         , "W:xmm, xmm/m128"                                 , "RM"      , "F3 0F 16 /r"                  , "SSE3"],
        ["movsldup"         , "W:xmm, xmm/m128"                                 , "RM"      , "F3 0F 12 /r"                  , "SSE3"],
        ["movss"            , "w:xmm[31:0], xmm[31:0]"                          , "RM"      , "F3 0F 10 /r"                  , "SSE"],
        ["movss"            , "W:xmm[31:0], m32"                                , "RM"      , "F3 0F 10 /r"                  , "SSE"],
        ["movss"            , "W:m32, xmm[31:0]"                                , "MR"      , "F3 0F 11 /r"                  , "SSE"],
        ["movupd"           , "W:xmm, xmm/m128"                                 , "RM"      , "66 0F 10 /r"                  , "SSE2"],
        ["movupd"           , "W:xmm/m128, xmm"                                 , "MR"      , "66 0F 11 /r"                  , "SSE2"],
        ["movups"           , "W:xmm, xmm/m128"                                 , "RM"      , "0F 10 /r"                     , "SSE"],
        ["movups"           , "W:xmm/m128, xmm"                                 , "MR"      , "0F 11 /r"                     , "SSE"],
        ["mpsadbw"          , "X:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F 3A 42 /r ib"            , "SSE4_1"],
        ["mulpd"            , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 59 /r"                  , "SSE2"],
        ["mulps"            , "X:~xmm, ~xmm/m128"                               , "RM"      , "0F 59 /r"                     , "SSE"],
        ["mulsd"            , "x:xmm[63:0], xmm[63:0]/m64"                      , "RM"      , "F2 0F 59 /r"                  , "SSE2"],
        ["mulss"            , "x:xmm[31:0], xmm[31:0]/m32"                      , "RM"      , "F3 0F 59 /r"                  , "SSE"],
        ["orpd"             , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 56 /r"                  , "SSE2"],
        ["orps"             , "X:~xmm, ~xmm/m128"                               , "RM"      , "0F 56 /r"                     , "SSE"],
        ["pabsb"            , "W:mm, mm/m64"                                    , "RM"      , "0F 38 1C /r"                  , "SSSE3"],
        ["pabsb"            , "W:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 1C /r"               , "SSSE3"],
        ["pabsd"            , "W:mm, mm/m64"                                    , "RM"      , "0F 38 1E /r"                  , "SSSE3"],
        ["pabsd"            , "W:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 1E /r"               , "SSSE3"],
        ["pabsw"            , "W:mm, mm/m64"                                    , "RM"      , "0F 38 1D /r"                  , "SSSE3"],
        ["pabsw"            , "W:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 1D /r"               , "SSSE3"],
        ["packssdw"         , "X:mm, mm/m64"                                    , "RM"      , "0F 6B /r"                     , "MMX"],
        ["packssdw"         , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 6B /r"                  , "SSE2"],
        ["packsswb"         , "X:mm, mm/m64"                                    , "RM"      , "0F 63 /r"                     , "MMX"],
        ["packsswb"         , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 63 /r"                  , "SSE2"],
        ["packusdw"         , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 2B /r"               , "SSE4_1"],
        ["packuswb"         , "X:mm, mm/m64"                                    , "RM"      , "0F 67 /r"                     , "MMX"],
        ["packuswb"         , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 67 /r"                  , "SSE2"],
        ["paddb"            , "X:~mm, ~mm/m64"                                  , "RM"      , "0F FC /r"                     , "MMX"],
        ["paddb"            , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F FC /r"                  , "SSE2"],
        ["paddd"            , "X:~mm, ~mm/m64"                                  , "RM"      , "0F FE /r"                     , "MMX"],
        ["paddd"            , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F FE /r"                  , "SSE2"],
        ["paddq"            , "X:~mm, ~mm/m64"                                  , "RM"      , "0F D4 /r"                     , "SSE2"],
        ["paddq"            , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F D4 /r"                  , "SSE2"],
        ["paddsb"           , "X:~mm, ~mm/m64"                                  , "RM"      , "0F EC /r"                     , "MMX"],
        ["paddsb"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F EC /r"                  , "SSE2"],
        ["paddsw"           , "X:~mm, ~mm/m64"                                  , "RM"      , "0F ED /r"                     , "MMX"],
        ["paddsw"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F ED /r"                  , "SSE2"],
        ["paddusb"          , "X:~mm, ~mm/m64"                                  , "RM"      , "0F DC /r"                     , "MMX"],
        ["paddusb"          , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F DC /r"                  , "SSE2"],
        ["paddusw"          , "X:~mm, ~mm/m64"                                  , "RM"      , "0F DD /r"                     , "MMX"],
        ["paddusw"          , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F DD /r"                  , "SSE2"],
        ["paddw"            , "X:~mm, ~mm/m64"                                  , "RM"      , "0F FD /r"                     , "MMX"],
        ["paddw"            , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F FD /r"                  , "SSE2"],
        ["palignr"          , "X:mm, mm/m64, ib/ub"                             , "RMI"     , "0F 3A 0F /r ib"               , "SSE3"],
        ["palignr"          , "X:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F 3A 0F /r ib"            , "SSE3"],
        ["pand"             , "X:~mm, ~mm/m64"                                  , "RM"      , "0F DB /r"                     , "MMX"],
        ["pand"             , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F DB /r"                  , "SSE2"],
        ["pandn"            , "X:mm, mm/m64"                                    , "RM"      , "0F DF /r"                     , "MMX"],
        ["pandn"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F DF /r"                  , "SSE2"],
        ["pavgb"            , "X:~mm, ~mm/m64"                                  , "RM"      , "0F E0 /r"                     , "MMX2"],
        ["pavgb"            , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F E0 /r"                  , "SSE2"],
        ["pavgw"            , "X:~mm, ~mm/m64"                                  , "RM"      , "0F E3 /r"                     , "MMX2"],
        ["pavgw"            , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F E3 /r"                  , "SSE2"],
        ["pblendvb"         , "X:xmm, xmm/m128, <xmm0>"                         , "RM"      , "66 0F E0 /r"                  , "SSE4_1"],
        ["pblendw"          , "X:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F 3A 0E /r ib"            , "SSE4_1"],
        ["pclmulqdq"        , "X:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F 3A 44 /r ib"            , "PCLMULQDQ"],
        ["pcmpeqb"          , "X:~mm, ~mm/m64"                                  , "RM"      , "0F 74 /r"                     , "MMX"],
        ["pcmpeqb"          , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 74 /r"                  , "SSE2"],
        ["pcmpeqd"          , "X:~mm, ~mm/m64"                                  , "RM"      , "0F 76 /r"                     , "MMX"],
        ["pcmpeqd"          , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 76 /r"                  , "SSE2"],
        ["pcmpeqq"          , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 29 /r"               , "SSE4_1"],
        ["pcmpeqw"          , "X:~mm, ~mm/m64"                                  , "RM"      , "0F 75 /r"                     , "MMX"],
        ["pcmpeqw"          , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 75 /r"                  , "SSE2"],
        ["pcmpestri"        , "R:xmm, xmm/m128, ib/ub, W:<ecx>,<eax>,<edx>"     , "RMI"     , "66 0F 3A 61 /r ib"            , "SSE4_2 OF=W SF=W ZF=W AF=0 PF=0 CF=W"],
        ["pcmpestrm"        , "R:xmm, xmm/m128, ib/ub, W:<xmm0>,<eax>,<edx>"    , "RMI"     , "66 0F 3A 60 /r ib"            , "SSE4_2 OF=W SF=W ZF=W AF=0 PF=0 CF=W"],
        ["pcmpgtb"          , "X:mm, mm/m64"                                    , "RM"      , "0F 64 /r"                     , "MMX"],
        ["pcmpgtb"          , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 64 /r"                  , "SSE2"],
        ["pcmpgtd"          , "X:mm, mm/m64"                                    , "RM"      , "0F 66 /r"                     , "MMX"],
        ["pcmpgtd"          , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 66 /r"                  , "SSE2"],
        ["pcmpgtq"          , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 37 /r"               , "SSE4_2"],
        ["pcmpgtw"          , "X:mm, mm/m64"                                    , "RM"      , "0F 65 /r"                     , "MMX"],
        ["pcmpgtw"          , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 65 /r"                  , "SSE2"],
        ["pcmpistri"        , "R:xmm, xmm/m128, ib/ub, W:<ecx>"                 , "RMI"     , "66 0F 3A 63 /r ib"            , "SSE4_2 OF=W SF=W ZF=W AF=0 PF=0 CF=W"],
        ["pcmpistrm"        , "R:xmm, xmm/m128, ib/ub, W:<xmm0>"                , "RMI"     , "66 0F 3A 62 /r ib"            , "SSE4_2 OF=W SF=W ZF=W AF=0 PF=0 CF=W"],
        ["pextrb"           , "W:r32[7:0]/m8, xmm, ib/ub"                       , "MRI"     , "66 0F 3A 14 /r ib"            , "SSE4_1"],
        ["pextrd"           , "W:r32[31:0]/m32, xmm, ib/ub"                     , "MRI"     , "66 0F 3A 16 /r ib"            , "SSE4_1"],
        ["pextrq"           , "W:r64/m64, xmm, ib/ub"                           , "MRI"     , "REX.W 66 0F 3A 16 /r ib"      , "SSE4_1 X64"],
        ["pextrw"           , "W:r32[15:0], mm, ib/ub"                          , "RMI"     , "0F C5 /r ib"                  , "MMX2"],
        ["pextrw"           , "W:r32[15:0], xmm, ib/ub"                         , "RMI"     , "66 0F C5 /r ib"               , "SSE2"],
        ["pextrw"           , "W:r32[15:0]/m16, xmm, ib/ub"                     , "MRI"     , "66 0F 3A 15 /r ib"            , "SSE4_1"],
        ["phaddd"           , "X:~mm, ~mm/m64"                                  , "RM"      , "0F 38 02 /r"                  , "SSSE3"],
        ["phaddd"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 02 /r"               , "SSSE3"],
        ["phaddsw"          , "X:~mm, ~mm/m64"                                  , "RM"      , "0F 38 03 /r"                  , "SSSE3"],
        ["phaddsw"          , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 03 /r"               , "SSSE3"],
        ["phaddw"           , "X:~mm, ~mm/m64"                                  , "RM"      , "0F 38 01 /r"                  , "SSSE3"],
        ["phaddw"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 01 /r"               , "SSSE3"],
        ["phminposuw"       , "W:xmm[18:0], xmm/m128"                           , "RM"      , "66 0F 38 41 /r"               , "SSE4_1"],
        ["phsubd"           , "X:mm, mm/m64"                                    , "RM"      , "0F 38 06 /r"                  , "SSSE3"],
        ["phsubd"           , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 06 /r"               , "SSSE3"],
        ["phsubsw"          , "X:mm, mm/m64"                                    , "RM"      , "0F 38 07 /r"                  , "SSSE3"],
        ["phsubsw"          , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 07 /r"               , "SSSE3"],
        ["phsubw"           , "X:mm, mm/m64"                                    , "RM"      , "0F 38 05 /r"                  , "SSSE3"],
        ["phsubw"           , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 05 /r"               , "SSSE3"],
        ["pinsrb"           , "X:xmm, r32[7:0]/m8, ib/ub"                       , "RMI"     , "66 0F 3A 20 /r ib"            , "SSE4_1"],
        ["pinsrd"           , "X:xmm, r32[31:0]/m32, ib/ub"                     , "RMI"     , "66 0F 3A 22 /r ib"            , "SSE4_1"],
        ["pinsrq"           , "X:xmm, r64/m64, ib/ub"                           , "RMI"     , "REX.W 66 0F 3A 22 /r ib"      , "SSE4_1 X64"],
        ["pinsrw"           , "X:mm, r32[15:0]/m16, ib/ub"                      , "RMI"     , "0F C4 /r ib"                  , "MMX2"],
        ["pinsrw"           , "X:xmm, r32[15:0]/m16, ib/ub"                     , "RMI"     , "66 0F C4 /r ib"               , "SSE2"],
        ["pmaddubsw"        , "X:~mm, ~mm/m64"                                  , "RM"      , "0F 38 04 /r"                  , "SSSE3"],
        ["pmaddubsw"        , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 04 /r"               , "SSSE3"],
        ["pmaddwd"          , "X:~mm, ~mm/m64"                                  , "RM"      , "0F F5 /r"                     , "MMX"],
        ["pmaddwd"          , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F F5 /r"                  , "SSE2"],
        ["pmaxsb"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 3C /r"               , "SSE4_1"],
        ["pmaxsd"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 3D /r"               , "SSE4_1"],
        ["pmaxsw"           , "X:~mm, ~mm/m64"                                  , "RM"      , "0F EE /r"                     , "MMX2"],
        ["pmaxsw"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F EE /r"                  , "SSE2"],
        ["pmaxub"           , "X:~mm, ~mm/m64"                                  , "RM"      , "0F DE /r"                     , "MMX2"],
        ["pmaxub"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F DE /r"                  , "SSE2"],
        ["pmaxud"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 3F /r"               , "SSE4_1"],
        ["pmaxuw"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 3E /r"               , "SSE4_1"],
        ["pminsb"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 38 /r"               , "SSE4_1"],
        ["pminsd"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 39 /r"               , "SSE4_1"],
        ["pminsw"           , "X:~mm, ~mm/m64"                                  , "RM"      , "0F EA /r"                     , "MMX2"],
        ["pminsw"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F EA /r"                  , "SSE2"],
        ["pminub"           , "X:~mm, ~mm/m64"                                  , "RM"      , "0F DA /r"                     , "MMX2"],
        ["pminub"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F DA /r"                  , "SSE2"],
        ["pminud"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 3B /r"               , "SSE4_1"],
        ["pminuw"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 3A /r"               , "SSE4_1"],
        ["pmovmskb"         , "W:r32[7:0], mm"                                  , "RM"      , "0F D7 /r"                     , "MMX2"],
        ["pmovmskb"         , "W:r32[15:0], xmm"                                , "RM"      , "66 0F D7 /r"                  , "SSE2"],
        ["pmovsxbd"         , "W:xmm, xmm[31:0]/m32"                            , "RM"      , "66 0F 38 21 /r"               , "SSE4_1"],
        ["pmovsxbq"         , "W:xmm, xmm[15:0]/m16"                            , "RM"      , "66 0F 38 22 /r"               , "SSE4_1"],
        ["pmovsxbw"         , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "66 0F 38 20 /r"               , "SSE4_1"],
        ["pmovsxdq"         , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "66 0F 38 25 /r"               , "SSE4_1"],
        ["pmovsxwd"         , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "66 0F 38 23 /r"               , "SSE4_1"],
        ["pmovsxwq"         , "W:xmm, xmm[31:0]/m32"                            , "RM"      , "66 0F 38 24 /r"               , "SSE4_1"],
        ["pmovzxbd"         , "W:xmm, xmm[31:0]/m32"                            , "RM"      , "66 0F 38 31 /r"               , "SSE4_1"],
        ["pmovzxbq"         , "W:xmm, xmm[15:0]/m16"                            , "RM"      , "66 0F 38 32 /r"               , "SSE4_1"],
        ["pmovzxbw"         , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "66 0F 38 30 /r"               , "SSE4_1"],
        ["pmovzxdq"         , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "66 0F 38 35 /r"               , "SSE4_1"],
        ["pmovzxwd"         , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "66 0F 38 33 /r"               , "SSE4_1"],
        ["pmovzxwq"         , "W:xmm, xmm[31:0]/m32"                            , "RM"      , "66 0F 38 34 /r"               , "SSE4_1"],
        ["pmuldq"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 28 /r"               , "SSE4_1"],
        ["pmulhrsw"         , "X:~mm, ~mm/m64"                                  , "RM"      , "0F 38 0B /r"                  , "SSSE3"],
        ["pmulhrsw"         , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 0B /r"               , "SSSE3"],
        ["pmulhuw"          , "X:~mm, ~mm/m64"                                  , "RM"      , "0F E4 /r"                     , "MMX2"],
        ["pmulhuw"          , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F E4 /r"                  , "SSE2"],
        ["pmulhw"           , "X:~mm, ~mm/m64"                                  , "RM"      , "0F E5 /r"                     , "MMX"],
        ["pmulhw"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F E5 /r"                  , "SSE2"],
        ["pmulld"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 40 /r"               , "SSE4_1"],
        ["pmullw"           , "X:~mm, ~mm/m64"                                  , "RM"      , "0F D5 /r"                     , "MMX"],
        ["pmullw"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F D5 /r"                  , "SSE2"],
        ["pmuludq"          , "X:~mm, ~mm/m64"                                  , "RM"      , "0F F4 /r"                     , "SSE2"],
        ["pmuludq"          , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F F4 /r"                  , "SSE2"],
        ["por"              , "X:~mm, ~mm/m64"                                  , "RM"      , "0F EB /r"                     , "MMX"],
        ["por"              , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F EB /r"                  , "SSE2"],
        ["prefetchnta"      , "R:mem"                                           , "M"       , "0F 18 /0"                     , "MMX2"],
        ["prefetcht0"       , "R:mem"                                           , "M"       , "0F 18 /1"                     , "MMX2"],
        ["prefetcht1"       , "R:mem"                                           , "M"       , "0F 18 /2"                     , "MMX2"],
        ["prefetcht2"       , "R:mem"                                           , "M"       , "0F 18 /3"                     , "MMX2"],
        ["psadbw"           , "X:~mm, ~mm/m64"                                  , "RM"      , "0F F6 /r"                     , "MMX2"],
        ["psadbw"           , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F F6 /r"                  , "SSE2"],
        ["pshufb"           , "X:mm, mm/m64"                                    , "RM"      , "0F 38 00 /r"                  , "SSSE3"],
        ["pshufb"           , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 00 /r"               , "SSSE3"],
        ["pshufd"           , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F 70 /r ib"               , "SSE2"],
        ["pshufhw"          , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "F3 0F 70 /r ib"               , "SSE2"],
        ["pshuflw"          , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "F2 0F 70 /r ib"               , "SSE2"],
        ["pshufw"           , "W:mm, mm/m64, ib/ub"                             , "RMI"     , "0F 70 /r ib"                  , "MMX2"],
        ["psignb"           , "X:mm, mm/m64"                                    , "RM"      , "0F 38 08 /r"                  , "SSSE3"],
        ["psignb"           , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 08 /r"               , "SSSE3"],
        ["psignd"           , "X:mm, mm/m64"                                    , "RM"      , "0F 38 0A /r"                  , "SSSE3"],
        ["psignd"           , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 0A /r"               , "SSSE3"],
        ["psignw"           , "X:mm, mm/m64"                                    , "RM"      , "0F 38 09 /r"                  , "SSSE3"],
        ["psignw"           , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 09 /r"               , "SSSE3"],
        ["pslld"            , "X:mm, ib/ub"                                     , "MI"      , "0F 72 /6 ib"                  , "MMX"],
        ["pslld"            , "X:mm, mm/m64"                                    , "RM"      , "0F F2 /r"                     , "MMX"],
        ["pslld"            , "X:xmm, ib/ub"                                    , "MI"      , "66 0F 72 /6 ib"               , "SSE2"],
        ["pslld"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F F2 /r"                  , "SSE2"],
        ["pslldq"           , "X:xmm, ib/ub"                                    , "MI"      , "66 0F 73 /7 ib"               , "SSE2"],
        ["psllq"            , "X:mm, ib/ub"                                     , "MI"      , "0F 73 /6 ib"                  , "MMX"],
        ["psllq"            , "X:mm, mm/m64"                                    , "RM"      , "0F F3 /r"                     , "MMX"],
        ["psllq"            , "X:xmm, ib/ub"                                    , "MI"      , "66 0F 73 /6 ib"               , "SSE2"],
        ["psllq"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F F3 /r"                  , "SSE2"],
        ["psllw"            , "X:mm, ib/ub"                                     , "MI"      , "0F 71 /6 ib"                  , "MMX"],
        ["psllw"            , "X:mm, mm/m64"                                    , "RM"      , "0F F1 /r"                     , "MMX"],
        ["psllw"            , "X:xmm, ib/ub"                                    , "MI"      , "66 0F 71 /6 ib"               , "SSE2"],
        ["psllw"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F F1 /r"                  , "SSE2"],
        ["psrad"            , "X:mm, ib/ub"                                     , "MI"      , "0F 72 /4 ib"                  , "MMX"],
        ["psrad"            , "X:mm, mm/m64"                                    , "RM"      , "0F E2 /r"                     , "MMX"],
        ["psrad"            , "X:xmm, ib/ub"                                    , "MI"      , "66 0F 72 /4 ib"               , "SSE2"],
        ["psrad"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F E2 /r"                  , "SSE2"],
        ["psraw"            , "X:mm, ib/ub"                                     , "MI"      , "0F 71 /4 ib"                  , "MMX"],
        ["psraw"            , "X:mm, mm/m64"                                    , "RM"      , "0F E1 /r"                     , "MMX"],
        ["psraw"            , "X:xmm, ib/ub"                                    , "MI"      , "66 0F 71 /4 ib"               , "SSE2"],
        ["psraw"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F E1 /r"                  , "SSE2"],
        ["psrld"            , "X:mm, ib/ub"                                     , "MI"      , "0F 72 /2 ib"                  , "MMX"],
        ["psrld"            , "X:mm, mm/m64"                                    , "RM"      , "0F D2 /r"                     , "MMX"],
        ["psrld"            , "X:xmm, ib/ub"                                    , "MI"      , "66 0F 72 /2 ib"               , "SSE2"],
        ["psrld"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F D2 /r"                  , "SSE2"],
        ["psrldq"           , "X:xmm, ib/ub"                                    , "MI"      , "66 0F 73 /3 ib"               , "SSE2"],
        ["psrlq"            , "X:mm, ib/ub"                                     , "MI"      , "0F 73 /2 ib"                  , "MMX"],
        ["psrlq"            , "X:mm, mm/m64"                                    , "RM"      , "0F D3 /r"                     , "MMX"],
        ["psrlq"            , "X:xmm, ib/ub"                                    , "MI"      , "66 0F 73 /2 ib"               , "SSE2"],
        ["psrlq"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F D3 /r"                  , "SSE2"],
        ["psrlw"            , "X:mm, ib/ub"                                     , "MI"      , "0F 71 /2 ib"                  , "MMX"],
        ["psrlw"            , "X:mm, mm/m64"                                    , "RM"      , "0F D1 /r"                     , "MMX"],
        ["psrlw"            , "X:xmm, ib/ub"                                    , "MI"      , "66 0F 71 /2 ib"               , "SSE2"],
        ["psrlw"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F D1 /r"                  , "SSE2"],
        ["psubb"            , "X:mm, mm/m64"                                    , "RM"      , "0F F8 /r"                     , "MMX"],
        ["psubb"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F F8 /r"                  , "SSE2"],
        ["psubd"            , "X:mm, mm/m64"                                    , "RM"      , "0F FA /r"                     , "MMX"],
        ["psubd"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F FA /r"                  , "SSE2"],
        ["psubq"            , "X:mm, mm/m64"                                    , "RM"      , "0F FB /r"                     , "SSE2"],
        ["psubq"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F FB /r"                  , "SSE2"],
        ["psubsb"           , "X:mm, mm/m64"                                    , "RM"      , "0F E8 /r"                     , "MMX"],
        ["psubsb"           , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F E8 /r"                  , "SSE2"],
        ["psubsw"           , "X:mm, mm/m64"                                    , "RM"      , "0F E9 /r"                     , "MMX"],
        ["psubsw"           , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F E9 /r"                  , "SSE2"],
        ["psubusb"          , "X:mm, mm/m64"                                    , "RM"      , "0F D8 /r"                     , "MMX"],
        ["psubusb"          , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F D8 /r"                  , "SSE2"],
        ["psubusw"          , "X:mm, mm/m64"                                    , "RM"      , "0F D9 /r"                     , "MMX"],
        ["psubusw"          , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F D9 /r"                  , "SSE2"],
        ["psubw"            , "X:mm, mm/m64"                                    , "RM"      , "0F F9 /r"                     , "MMX"],
        ["psubw"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F F9 /r"                  , "SSE2"],
        ["ptest"            , "R:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 38 17 /r"               , "SSE4_1 OF=0 SF=0 ZF=W AF=0 PF=0 CF=W"],
        ["punpckhbw"        , "X:mm, mm/m64"                                    , "RM"      , "0F 68 /r"                     , "MMX"],
        ["punpckhbw"        , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 68 /r"                  , "SSE2"],
        ["punpckhdq"        , "X:mm, mm/m64"                                    , "RM"      , "0F 6A /r"                     , "MMX"],
        ["punpckhdq"        , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 6A /r"                  , "SSE2"],
        ["punpckhqdq"       , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 6D /r"                  , "SSE2"],
        ["punpckhwd"        , "X:mm, mm/m64"                                    , "RM"      , "0F 69 /r"                     , "MMX"],
        ["punpckhwd"        , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 69 /r"                  , "SSE2"],
        ["punpcklbw"        , "X:mm, mm/m64"                                    , "RM"      , "0F 60 /r"                     , "MMX"],
        ["punpcklbw"        , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 60 /r"                  , "SSE2"],
        ["punpckldq"        , "X:mm, mm/m64"                                    , "RM"      , "0F 62 /r"                     , "MMX"],
        ["punpckldq"        , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 62 /r"                  , "SSE2"],
        ["punpcklqdq"       , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 6C /r"                  , "SSE2"],
        ["punpcklwd"        , "X:mm, mm/m64"                                    , "RM"      , "0F 61 /r"                     , "MMX"],
        ["punpcklwd"        , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 61 /r"                  , "SSE2"],
        ["pxor"             , "X:~mm, ~mm/m64"                                  , "RM"      , "0F EF /r"                     , "MMX"],
        ["pxor"             , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F EF /r"                  , "SSE2"],
        ["rcpps"            , "W:xmm, xmm/m128"                                 , "RM"      , "0F 53 /r"                     , "SSE"],
        ["rcpss"            , "w:xmm[31:0], xmm[31:0]/m32"                      , "RM"      , "F3 0F 53 /r"                  , "SSE"],
        ["roundpd"          , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F 3A 09 /r ib"            , "SSE4_1"],
        ["roundps"          , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F 3A 08 /r ib"            , "SSE4_1"],
        ["roundsd"          , "w:xmm[63:0], xmm[63:0]/m64, ib/ub"               , "RMI"     , "66 0F 3A 0B /r ib"            , "SSE4_1"],
        ["roundss"          , "w:xmm[31:0], xmm[31:0]/m32, ib/ub"               , "RMI"     , "66 0F 3A 0A /r ib"            , "SSE4_1"],
        ["rsqrtps"          , "W:xmm, xmm/m128"                                 , "RM"      , "0F 52 /r"                     , "SSE"],
        ["rsqrtss"          , "w:xmm[31:0], xmm[31:0]/m32"                      , "RM"      , "F3 0F 52 /r"                  , "SSE"],
        ["shufpd"           , "X:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F C6 /r ib"               , "SSE2"],
        ["shufps"           , "X:xmm, xmm/m128, ib/ub"                          , "RMI"     , "0F C6 /r ib"                  , "SSE"],
        ["sqrtpd"           , "W:xmm, xmm/m128"                                 , "RM"      , "66 0F 51 /r"                  , "SSE2"],
        ["sqrtps"           , "W:xmm, xmm/m128"                                 , "RM"      , "0F 51 /r"                     , "SSE"],
        ["sqrtsd"           , "w:xmm[63:0], xmm[63:0]/m64"                      , "RM"      , "F2 0F 51 /r"                  , "SSE2"],
        ["sqrtss"           , "w:xmm[31:0], xmm[31:0]/m32"                      , "RM"      , "F3 0F 51 /r"                  , "SSE"],
        ["subpd"            , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 5C /r"                  , "SSE2"],
        ["subps"            , "X:xmm, xmm/m128"                                 , "RM"      , "0F 5C /r"                     , "SSE"],
        ["subsd"            , "x:xmm[63:0], xmm[63:0]/m64"                      , "RM"      , "F2 0F 5C /r"                  , "SSE2"],
        ["subss"            , "x:xmm[31:0], xmm[31:0]/m32"                      , "RM"      , "F3 0F 5C /r"                  , "SSE"],
        ["ucomisd"          , "R:xmm[63:0], xmm[63:0]/m64"                      , "RM"      , "66 0F 2E /r"                  , "SSE2 OF=0 SF=0 ZF=W AF=0 PF=W CF=W"],
        ["ucomiss"          , "R:xmm[31:0], xmm[31:0]/m32"                      , "RM"      , "0F 2E /r"                     , "SSE  OF=0 SF=0 ZF=W AF=0 PF=W CF=W"],
        ["unpckhpd"         , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 15 /r"                  , "SSE2"],
        ["unpckhps"         , "X:xmm, xmm/m128"                                 , "RM"      , "0F 15 /r"                     , "SSE"],
        ["unpcklpd"         , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 14 /r"                  , "SSE2"],
        ["unpcklps"         , "X:xmm, xmm/m128"                                 , "RM"      , "0F 14 /r"                     , "SSE"],
        ["xorpd"            , "X:~xmm, ~xmm/m128"                               , "RM"      , "66 0F 57 /r"                  , "SSE2"],
        ["xorps"            , "X:~xmm, ~xmm/m128"                               , "RM"      , "0F 57 /r"                     , "SSE"],
    
        ["pavgusb"          , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r BF"                  , "3DNOW"],
        ["pf2id"            , "W:mm, mm/m64"                                    , "RM"      , "0F 0F /r 1D"                  , "3DNOW"],
        ["pf2iw"            , "W:mm, mm/m64"                                    , "RM"      , "0F 0F /r 1C"                  , "3DNOW2"],
        ["pfacc"            , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r AE"                  , "3DNOW"],
        ["pfadd"            , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r 9E"                  , "3DNOW"],
        ["pfcmpeq"          , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r B0"                  , "3DNOW"],
        ["pfcmpge"          , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r 90"                  , "3DNOW"],
        ["pfcmpgt"          , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r A0"                  , "3DNOW"],
        ["pfmax"            , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r A4"                  , "3DNOW"],
        ["pfmin"            , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r 94"                  , "3DNOW"],
        ["pfmul"            , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r B4"                  , "3DNOW"],
        ["pfnacc"           , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r 8A"                  , "3DNOW2"],
        ["pfpnacc"          , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r 8E"                  , "3DNOW2"],
        ["pfrcp"            , "W:mm, mm/m64"                                    , "RM"      , "0F 0F /r 96"                  , "3DNOW"],
        ["pfrcpit1"         , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r A6"                  , "3DNOW"],
        ["pfrcpit2"         , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r B6"                  , "3DNOW"],
        ["pfrcpv"           , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r 86"                  , "GEODE"],
        ["pfrsqit1"         , "W:mm, mm/m64"                                    , "RM"      , "0F 0F /r A7"                  , "3DNOW"],
        ["pfrsqrt"          , "W:mm, mm/m64"                                    , "RM"      , "0F 0F /r 97"                  , "3DNOW"],
        ["pfrsqrtv"         , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r 87"                  , "GEODE"],
        ["pfsub"            , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r 9A"                  , "3DNOW"],
        ["pfsubr"           , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r AA"                  , "3DNOW"],
        ["pi2fd"            , "W:mm, mm/m64"                                    , "RM"      , "0F 0F /r 0D"                  , "3DNOW"],
        ["pi2fw"            , "W:mm, mm/m64"                                    , "RM"      , "0F 0F /r 0C"                  , "3DNOW2"],
        ["pmulhrw"          , "X:mm, mm/m64"                                    , "RM"      , "0F 0F /r B7"                  , "3DNOW"],
        ["prefetch"         , "R:mem"                                           , "M"       , "0F 0D /0"                     , "3DNOW"],
        ["pswapd"           , "W:mm, mm/m64"                                    , "RM"      , "0F 0F /r BB"                  , "3DNOW2"],
    
        ["emms"             , ""                                                , "NONE"    , "0F 77"                        , "MMX Volatile"],
        ["femms"            , ""                                                , "NONE"    , "0F 0E"                        , "3DNOW Volatile"],
        ["lfence"           , ""                                                , "NONE"    , "0F AE /5"                     , "SSE2 Volatile"],
        ["mfence"           , ""                                                , "NONE"    , "0F AE /6"                     , "SSE2 Volatile"],
        ["sfence"           , ""                                                , "NONE"    , "0F AE /7"                     , "MMX2 Volatile"],
        ["ldmxcsr"          , "R:m32"                                           , "M"       , "0F AE /2"                     , "SSE"],
        ["stmxcsr"          , "W:m32"                                           , "M"       , "0F AE /3"                     , "SSE"],
    
        ["aesdec"           , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 DE /r"               , "AESNI"],
        ["aesdeclast"       , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 DF /r"               , "AESNI"],
        ["aesenc"           , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 DC /r"               , "AESNI"],
        ["aesenclast"       , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 DD /r"               , "AESNI"],
        ["aesimc"           , "W:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 DB /r"               , "AESNI"],
        ["aeskeygenassist"  , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F 3A DF /r ib"            , "AESNI"],
    
        ["sha1msg1"         , "xmm, xmm/m128"                                   , "RM"      , "0F 38 C9 /r"                  , "SHA"],
        ["sha1msg2"         , "xmm, xmm/m128"                                   , "RM"      , "0F 38 CA /r"                  , "SHA"],
        ["sha1nexte"        , "xmm, xmm/m128"                                   , "RM"      , "0F 38 C8 /r"                  , "SHA"],
        ["sha1rnds4"        , "xmm, xmm/m128, ib/ub"                            , "RMI"     , "0F 3A CC /r ib"               , "SHA"],
        ["sha256msg1"       , "xmm, xmm/m128"                                   , "RM"      , "0F 38 CC /r"                  , "SHA"],
        ["sha256msg2"       , "xmm, xmm/m128"                                   , "RM"      , "0F 38 CD /r"                  , "SHA"],
        ["sha256rnds2"      , "xmm, xmm/m128, <xmm0>"                           , "RM"      , "0F 38 CB /r"                  , "SHA"],
    
        ["gf2p8affineinvqb" , "X:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F 3A CF /r ib"            , "GFNI"],
        ["gf2p8affineqb"    , "X:xmm, xmm/m128, ib/ub"                          , "RMI"     , "66 0F 3A CE /r ib"            , "GFNI"],
        ["gf2p8mulb"        , "X:xmm, xmm/m128"                                 , "RM"      , "66 0F 38 CF /r"               , "GFNI"],
    
        ["vaddpd"           , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 58 /r"      , "AVX"],
        ["vaddpd"           , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 58 /r"      , "AVX"],
        ["vaddps"           , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.0F.WIG 58 /r"         , "AVX"],
        ["vaddps"           , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.0F.WIG 58 /r"         , "AVX"],
        ["vaddsd"           , "W:xmm, xmm, xmm[63:0]/m64"                       , "RVM"     , "VEX.LIG.F2.0F.WIG 58 /r"      , "AVX"],
        ["vaddss"           , "W:xmm, xmm, xmm[31:0]/m32"                       , "RVM"     , "VEX.LIG.F3.0F.WIG 58 /r"      , "AVX"],
        ["vaddsubpd"        , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG D0 /r"      , "AVX"],
        ["vaddsubpd"        , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG D0 /r"      , "AVX"],
        ["vaddsubps"        , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.F2.0F.WIG D0 /r"      , "AVX"],
        ["vaddsubps"        , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.F2.0F.WIG D0 /r"      , "AVX"],
        ["vaesdec"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG DE /r"    , "AVX AESNI"],
        ["vaesdec"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG DE /r"    , "VAES"],
        ["vaesdeclast"      , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG DF /r"    , "AVX AESNI"],
        ["vaesdeclast"      , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG DF /r"    , "VAES"],
        ["vaesenc"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG DC /r"    , "AVX AESNI"],
        ["vaesenc"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG DC /r"    , "VAES"],
        ["vaesenclast"      , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG DD /r"    , "AVX AESNI"],
        ["vaesenclast"      , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG DD /r"    , "VAES"],
        ["vaesimc"          , "W:xmm, xmm/m128"                                 , "RVM"     , "VEX.128.66.0F38.WIG DB /r"    , "AVX AESNI"],
        ["vaeskeygenassist" , "W:xmm, xmm/m128, ib/ub"                          , "RVMI"    , "VEX.128.66.0F3A.WIG DF /r ib" , "AVX AESNI"],
        ["vandnpd"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 55 /r"      , "AVX"],
        ["vandnpd"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 55 /r"      , "AVX"],
        ["vandnps"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.0F.WIG 55 /r"         , "AVX"],
        ["vandnps"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.0F.WIG 55 /r"         , "AVX"],
        ["vandpd"           , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 54 /r"      , "AVX"],
        ["vandpd"           , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 54 /r"      , "AVX"],
        ["vandps"           , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.0F.WIG 54 /r"         , "AVX"],
        ["vandps"           , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.0F.WIG 54 /r"         , "AVX"],
        ["vblendpd"         , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.128.66.0F3A.WIG 0D /r ib" , "AVX"],
        ["vblendpd"         , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.66.0F3A.WIG 0D /r ib" , "AVX"],
        ["vblendps"         , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.128.66.0F3A.WIG 0C /r ib" , "AVX"],
        ["vblendps"         , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.66.0F3A.WIG 0C /r ib" , "AVX"],
        ["vblendvpd"        , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "VEX.128.66.0F3A.W0 4B /r /is4", "AVX"],
        ["vblendvpd"        , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "VEX.256.66.0F3A.W0 4B /r /is4", "AVX"],
        ["vblendvps"        , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "VEX.128.66.0F3A.W0 4A /r /is4", "AVX"],
        ["vblendvps"        , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "VEX.256.66.0F3A.W0 4A /r /is4", "AVX"],
        ["vbroadcastf128"   , "W:ymm, m128"                                     , "RM"      , "VEX.256.66.0F38.W0 1A /r"     , "AVX"],
        ["vbroadcasti128"   , "W:ymm, m128"                                     , "RM"      , "VEX.256.66.0F38.W0 5A /r"     , "AVX2"],
        ["vbroadcastsd"     , "W:ymm, m64"                                      , "RM"      , "VEX.256.66.0F38.W0 19 /r"     , "AVX"],
        ["vbroadcastsd"     , "W:ymm, xmm[63:0]"                                , "RM"      , "VEX.256.66.0F38.W0 19 /r"     , "AVX2"],
        ["vbroadcastss"     , "W:xmm, m32"                                      , "RM"      , "VEX.128.66.0F38.W0 18 /r"     , "AVX"],
        ["vbroadcastss"     , "W:xmm, xmm[31:0]"                                , "RM"      , "VEX.128.66.0F38.W0 18 /r"     , "AVX2"],
        ["vbroadcastss"     , "W:ymm, m32"                                      , "RM"      , "VEX.256.66.0F38.W0 18 /r"     , "AVX"],
        ["vbroadcastss"     , "W:ymm, xmm[31:0]"                                , "RM"      , "VEX.256.66.0F38.W0 18 /r"     , "AVX2"],
        ["vcmppd"           , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.128.66.0F.WIG C2 /r ib"   , "AVX"],
        ["vcmppd"           , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.66.0F.WIG C2 /r ib"   , "AVX"],
        ["vcmpps"           , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.128.0F.WIG C2 /r ib"      , "AVX"],
        ["vcmpps"           , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.0F.WIG C2 /r ib"      , "AVX"],
        ["vcmpsd"           , "W:xmm, xmm, xmm[63:0]/m64, ib/ub"                , "RVMI"    , "VEX.LIG.F2.0F.WIG C2 /r ib"   , "AVX"],
        ["vcmpss"           , "W:xmm, xmm, xmm[31:0]/m32, ib/ub"                , "RVMI"    , "VEX.LIG.F3.0F.WIG C2 /r ib"   , "AVX"],
        ["vcomisd"          , "R:xmm[63:0], xmm[63:0]/m64"                      , "RM"      , "VEX.LIG.66.0F.WIG 2F /r"      , "AVX OF=0 SF=0 ZF=W AF=0 PF=W CF=W"],
        ["vcomiss"          , "R:xmm[31:0], xmm[31:0]/m32"                      , "RM"      , "VEX.LIG.0F.WIG 2F /r"         , "AVX OF=0 SF=0 ZF=W AF=0 PF=W CF=W"],
        ["vcvtdq2pd"        , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "VEX.128.F3.0F.WIG E6 /r"      , "AVX"],
        ["vcvtdq2pd"        , "W:ymm, xmm/m128"                                 , "RM"      , "VEX.256.F3.0F.WIG E6 /r"      , "AVX"],
        ["vcvtdq2ps"        , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.0F.WIG 5B /r"         , "AVX"],
        ["vcvtdq2ps"        , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.0F.WIG 5B /r"         , "AVX"],
        ["vcvtpd2dq"        , "W:xmm[63:0], xmm/m128"                           , "RM"      , "VEX.128.F2.0F.WIG E6 /r"      , "AVX"],
        ["vcvtpd2dq"        , "W:xmm, ymm/m256"                                 , "RM"      , "VEX.256.F2.0F.WIG E6 /r"      , "AVX"],
        ["vcvtpd2ps"        , "W:xmm[63:0], xmm/m128"                           , "RM"      , "VEX.128.66.0F.WIG 5A /r"      , "AVX"],
        ["vcvtpd2ps"        , "W:xmm, ymm/m256"                                 , "RM"      , "VEX.256.66.0F.WIG 5A /r"      , "AVX"],
        ["vcvtps2dq"        , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.66.0F.WIG 5B /r"      , "AVX"],
        ["vcvtps2dq"        , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.66.0F.WIG 5B /r"      , "AVX"],
        ["vcvtps2pd"        , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "VEX.128.0F.WIG 5A /r"         , "AVX"],
        ["vcvtps2pd"        , "W:ymm, xmm/m128"                                 , "RM"      , "VEX.256.0F.WIG 5A /r"         , "AVX"],
        ["vcvtsd2si"        , "W:r32, xmm[63:0]/m64"                            , "RM"      , "VEX.LIG.F2.0F.W0 2D /r"       , "AVX"],
        ["vcvtsd2si"        , "W:r64, xmm[63:0]/m64"                            , "RM"      , "VEX.LIG.F2.0F.W1 2D /r"       , "AVX X64"],
        ["vcvtsd2ss"        , "W:xmm, xmm[127:32], xmm[63:0]/m64"               , "RVM"     , "VEX.LIG.F2.0F.WIG 5A /r"      , "AVX"],
        ["vcvtsi2sd"        , "W:xmm, xmm[127:64], r32/m32"                     , "RVM"     , "VEX.LIG.F2.0F.W0 2A /r"       , "AVX"],
        ["vcvtsi2sd"        , "W:xmm, xmm[127:64], r64/m64"                     , "RVM"     , "VEX.LIG.F2.0F.W1 2A /r"       , "AVX X64"],
        ["vcvtsi2ss"        , "W:xmm, xmm[127:32], r32/m32"                     , "RVM"     , "VEX.LIG.F3.0F.W0 2A /r"       , "AVX"],
        ["vcvtsi2ss"        , "W:xmm, xmm[127:32], r64/m64"                     , "RVM"     , "VEX.LIG.F3.0F.W1 2A /r"       , "AVX X64"],
        ["vcvtss2sd"        , "W:xmm, xmm[127:64], xmm[31:0]/m32"               , "RVM"     , "VEX.LIG.F3.0F.WIG 5A /r"      , "AVX"],
        ["vcvtss2si"        , "W:r32, xmm[31:0]/m32"                            , "RM"      , "VEX.LIG.F3.0F.W0 2D /r"       , "AVX"],
        ["vcvtss2si"        , "W:r64, xmm[31:0]/m32"                            , "RM"      , "VEX.LIG.F3.0F.W1 2D /r"       , "AVX X64"],
        ["vcvttpd2dq"       , "W:xmm[63:0], xmm/m128"                           , "RM"      , "VEX.128.66.0F.WIG E6 /r"      , "AVX"],
        ["vcvttpd2dq"       , "W:xmm, ymm/m256"                                 , "RM"      , "VEX.256.66.0F.WIG E6 /r"      , "AVX"],
        ["vcvttps2dq"       , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.F3.0F.WIG 5B /r"      , "AVX"],
        ["vcvttps2dq"       , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.F3.0F.WIG 5B /r"      , "AVX"],
        ["vcvttsd2si"       , "W:r32, xmm[63:0]/m64"                            , "RM"      , "VEX.LIG.F2.0F.W0 2C /r"       , "AVX"],
        ["vcvttsd2si"       , "W:r64, xmm[63:0]/m64"                            , "RM"      , "VEX.LIG.F2.0F.W1 2C /r"       , "AVX X64"],
        ["vcvttss2si"       , "W:r32, xmm[31:0]/m32"                            , "RM"      , "VEX.LIG.F3.0F.W0 2C /r"       , "AVX"],
        ["vcvttss2si"       , "W:r64, xmm[31:0]/m32"                            , "RM"      , "VEX.LIG.F3.0F.W1 2C /r"       , "AVX X64"],
        ["vdivpd"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 5E /r"      , "AVX"],
        ["vdivpd"           , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 5E /r"      , "AVX"],
        ["vdivps"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.0F.WIG 5E /r"         , "AVX"],
        ["vdivps"           , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.0F.WIG 5E /r"         , "AVX"],
        ["vdivsd"           , "W:xmm, xmm, xmm[63:0]/m64"                       , "RVM"     , "VEX.LIG.F2.0F.WIG 5E /r"      , "AVX"],
        ["vdivss"           , "W:xmm, xmm, xmm[31:0]/m32"                       , "RVM"     , "VEX.LIG.F3.0F.WIG 5E /r"      , "AVX"],
        ["vdppd"            , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.128.66.0F3A.WIG 41 /r ib" , "AVX"],
        ["vdpps"            , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.128.66.0F3A.WIG 40 /r ib" , "AVX"],
        ["vdpps"            , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.66.0F3A.WIG 40 /r ib" , "AVX"],
        ["vextractf128"     , "W:xmm/m128, ymm, ib/ub"                          , "MRI"     , "VEX.256.66.0F3A.W0 19 /r ib"  , "AVX"],
        ["vextracti128"     , "W:xmm/m128, ymm, ib/ub"                          , "MRI"     , "VEX.256.66.0F3A.W0 39 /r ib"  , "AVX2"],
        ["vextractps"       , "W:r32[31:0]/m32, xmm, ib/ub"                     , "MRI"     , "VEX.128.66.0F3A.WIG 17 /r ib" , "AVX"],
        ["vgatherdpd"       , "X:xmm, vm32x, X:xmm"                             , "RMV"     , "VEX.128.66.0F38.W1 92 /r"     , "AVX2"],
        ["vgatherdpd"       , "X:ymm, vm32x, X:ymm"                             , "RMV"     , "VEX.256.66.0F38.W1 92 /r"     , "AVX2"],
        ["vgatherdps"       , "X:xmm, vm32x, X:xmm"                             , "RMV"     , "VEX.128.66.0F38.W0 92 /r"     , "AVX2"],
        ["vgatherdps"       , "X:ymm, vm32y, X:ymm"                             , "RMV"     , "VEX.256.66.0F38.W0 92 /r"     , "AVX2"],
        ["vgatherqpd"       , "X:xmm, vm64x, X:xmm"                             , "RMV"     , "VEX.128.66.0F38.W1 93 /r"     , "AVX2"],
        ["vgatherqpd"       , "X:ymm, vm64y, X:ymm"                             , "RMV"     , "VEX.256.66.0F38.W1 93 /r"     , "AVX2"],
        ["vgatherqps"       , "X:xmm, vm64x, X:xmm"                             , "RMV"     , "VEX.128.66.0F38.W0 93 /r"     , "AVX2"],
        ["vgatherqps"       , "X:xmm, vm64y, X:xmm"                             , "RMV"     , "VEX.256.66.0F38.W0 93 /r"     , "AVX2"],
        ["vgf2p8affineinvqb", "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.128.66.0F3A.W1 CF /r ib"  , "AVX GFNI"],
        ["vgf2p8affineinvqb", "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.66.0F3A.W1 CF /r ib"  , "AVX GFNI"],
        ["vgf2p8affineqb"   , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.128.66.0F3A.W1 CE /r ib"  , "AVX GFNI"],
        ["vgf2p8affineqb"   , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.66.0F3A.W1 CE /r ib"  , "AVX GFNI"],
        ["vgf2p8mulb"       , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 CF /r"     , "AVX GFNI"],
        ["vgf2p8mulb"       , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 CF /r"     , "AVX GFNI"],
        ["vhaddpd"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 7C /r"      , "AVX"],
        ["vhaddpd"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 7C /r"      , "AVX"],
        ["vhaddps"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.F2.0F.WIG 7C /r"      , "AVX"],
        ["vhaddps"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.F2.0F.WIG 7C /r"      , "AVX"],
        ["vhsubpd"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 7D /r"      , "AVX"],
        ["vhsubpd"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 7D /r"      , "AVX"],
        ["vhsubps"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.F2.0F.WIG 7D /r"      , "AVX"],
        ["vhsubps"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.F2.0F.WIG 7D /r"      , "AVX"],
        ["vinsertf128"      , "W:ymm, ymm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.256.66.0F3A.W0 18 /r ib"  , "AVX"],
        ["vinserti128"      , "W:ymm, ymm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.256.66.0F3A.W0 38 /r ib"  , "AVX2"],
        ["vinsertps"        , "W:xmm, xmm, xmm[31:0]/m32, ib/ub"                , "RVMI"    , "VEX.128.66.0F3A.WIG 21 /r ib" , "AVX"],
        ["vlddqu"           , "W:xmm, m128"                                     , "RM"      , "VEX.128.F2.0F.WIG F0 /r"      , "AVX"],
        ["vlddqu"           , "W:ymm, m256"                                     , "RM"      , "VEX.256.F2.0F.WIG F0 /r"      , "AVX"],
        ["vmaskmovdqu"      , "R:xmm, xmm, X:<ds:zdi>"                          , "RM"      , "VEX.128.66.0F.WIG F7 /r"      , "AVX"],
        ["vmaskmovpd"       , "X:m128, xmm, xmm"                                , "MVR"     , "VEX.128.66.0F38.W0 2F /r"     , "AVX"],
        ["vmaskmovpd"       , "X:m256, ymm, ymm"                                , "MVR"     , "VEX.256.66.0F38.W0 2F /r"     , "AVX"],
        ["vmaskmovpd"       , "W:xmm, xmm, m128"                                , "RVM"     , "VEX.128.66.0F38.W0 2D /r"     , "AVX"],
        ["vmaskmovpd"       , "W:ymm, ymm, m256"                                , "RVM"     , "VEX.256.66.0F38.W0 2D /r"     , "AVX"],
        ["vmaskmovps"       , "X:m128, xmm, xmm"                                , "MVR"     , "VEX.128.66.0F38.W0 2E /r"     , "AVX"],
        ["vmaskmovps"       , "X:m256, ymm, ymm"                                , "MVR"     , "VEX.256.66.0F38.W0 2E /r"     , "AVX"],
        ["vmaskmovps"       , "W:xmm, xmm, m128"                                , "RVM"     , "VEX.128.66.0F38.W0 2C /r"     , "AVX"],
        ["vmaskmovps"       , "W:ymm, ymm, m256"                                , "RVM"     , "VEX.256.66.0F38.W0 2C /r"     , "AVX"],
        ["vmaxpd"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 5F /r"      , "AVX"],
        ["vmaxpd"           , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 5F /r"      , "AVX"],
        ["vmaxps"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.0F.WIG 5F /r"         , "AVX"],
        ["vmaxps"           , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.0F.WIG 5F /r"         , "AVX"],
        ["vmaxsd"           , "W:xmm, xmm, xmm[63:0]/m64"                       , "RVM"     , "VEX.LIG.F2.0F.WIG 5F /r"      , "AVX"],
        ["vmaxss"           , "W:xmm, xmm, xmm[31:0]/m32"                       , "RVM"     , "VEX.LIG.F3.0F.WIG 5F /r"      , "AVX"],
        ["vminpd"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 5D /r"      , "AVX"],
        ["vminpd"           , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 5D /r"      , "AVX"],
        ["vminps"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.0F.WIG 5D /r"         , "AVX"],
        ["vminps"           , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.0F.WIG 5D /r"         , "AVX"],
        ["vminsd"           , "W:xmm, xmm, xmm[63:0]/m64"                       , "RVM"     , "VEX.LIG.F2.0F.WIG 5D /r"      , "AVX"],
        ["vminss"           , "W:xmm, xmm, xmm[31:0]/m32"                       , "RVM"     , "VEX.LIG.F3.0F.WIG 5D /r"      , "AVX"],
        ["vmovapd"          , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.66.0F.WIG 28 /r"      , "AVX"],
        ["vmovapd"          , "W:xmm/m128, xmm"                                 , "MR"      , "VEX.128.66.0F.WIG 29 /r"      , "AVX"],
        ["vmovapd"          , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.66.0F.WIG 28 /r"      , "AVX"],
        ["vmovapd"          , "W:ymm/m256, ymm"                                 , "MR"      , "VEX.256.66.0F.WIG 29 /r"      , "AVX"],
        ["vmovaps"          , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.0F.WIG 28 /r"         , "AVX"],
        ["vmovaps"          , "W:xmm/m128, xmm"                                 , "MR"      , "VEX.128.0F.WIG 29 /r"         , "AVX"],
        ["vmovaps"          , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.0F.WIG 28 /r"         , "AVX"],
        ["vmovaps"          , "W:ymm/m256, ymm"                                 , "MR"      , "VEX.256.0F.WIG 29 /r"         , "AVX"],
        ["vmovd"            , "W:r32[31:0]/m32, xmm[31:0]"                      , "MR"      , "VEX.128.66.0F.W0 7E /r"       , "AVX"],
        ["vmovd"            , "W:xmm[31:0], r32[31:0]/m32"                      , "RM"      , "VEX.128.66.0F.W0 6E /r"       , "AVX"],
        ["vmovddup"         , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "VEX.128.F2.0F.WIG 12 /r"      , "AVX"],
        ["vmovddup"         , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.F2.0F.WIG 12 /r"      , "AVX"],
        ["vmovdqa"          , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.66.0F.WIG 6F /r"      , "AVX"],
        ["vmovdqa"          , "W:xmm/m128, xmm"                                 , "MR"      , "VEX.128.66.0F.WIG 7F /r"      , "AVX"],
        ["vmovdqa"          , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.66.0F.WIG 6F /r"      , "AVX"],
        ["vmovdqa"          , "W:ymm/m256, ymm"                                 , "MR"      , "VEX.256.66.0F.WIG 7F /r"      , "AVX"],
        ["vmovdqu"          , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.F3.0F.WIG 6F /r"      , "AVX"],
        ["vmovdqu"          , "W:xmm/m128, xmm"                                 , "MR"      , "VEX.128.F3.0F.WIG 7F /r"      , "AVX"],
        ["vmovdqu"          , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.F3.0F.WIG 6F /r"      , "AVX"],
        ["vmovdqu"          , "W:ymm/m256, ymm"                                 , "MR"      , "VEX.256.F3.0F.WIG 7F /r"      , "AVX"],
        ["vmovhlps"         , "W:xmm, xmm[127:64], xmm[127:64]"                 , "RVM"     , "VEX.128.0F.WIG 12 /r"         , "AVX"],
        ["vmovhpd"          , "W:m64, xmm[127:64]"                              , "MR"      , "VEX.128.66.0F.WIG 17 /r"      , "AVX"],
        ["vmovhpd"          , "W:xmm, xmm[63:0], m64"                           , "RVM"     , "VEX.128.66.0F.WIG 16 /r"      , "AVX"],
        ["vmovhps"          , "W:m64, xmm[127:64]"                              , "MR"      , "VEX.128.0F.WIG 17 /r"         , "AVX"],
        ["vmovhps"          , "W:xmm, xmm[63:0], m64"                           , "RVM"     , "VEX.128.0F.WIG 16 /r"         , "AVX"],
        ["vmovlhps"         , "W:xmm, xmm[63:0], xmm[63:0]"                     , "RVM"     , "VEX.128.0F.WIG 16 /r"         , "AVX"],
        ["vmovlpd"          , "W:m64, xmm[63:0]"                                , "MR"      , "VEX.128.66.0F.WIG 13 /r"      , "AVX"],
        ["vmovlpd"          , "W:xmm, xmm[127:64], m64"                         , "RVM"     , "VEX.128.66.0F.WIG 12 /r"      , "AVX"],
        ["vmovlps"          , "W:m64, xmm[63:0]"                                , "MR"      , "VEX.128.0F.WIG 13 /r"         , "AVX"],
        ["vmovlps"          , "W:xmm, xmm[127:64], m64"                         , "RVM"     , "VEX.128.0F.WIG 12 /r"         , "AVX"],
        ["vmovmskpd"        , "W:r32[1:0], xmm"                                 , "RM"      , "VEX.128.66.0F.WIG 50 /r"      , "AVX"],
        ["vmovmskpd"        , "W:r32[3:0], ymm"                                 , "RM"      , "VEX.256.66.0F.WIG 50 /r"      , "AVX"],
        ["vmovmskps"        , "W:r32[3:0], xmm"                                 , "RM"      , "VEX.128.0F.WIG 50 /r"         , "AVX"],
        ["vmovmskps"        , "W:r32[7:0], ymm"                                 , "RM"      , "VEX.256.0F.WIG 50 /r"         , "AVX"],
        ["vmovntdq"         , "W:m128, xmm"                                     , "MR"      , "VEX.128.66.0F.WIG E7 /r"      , "AVX"],
        ["vmovntdq"         , "W:m256, ymm"                                     , "MR"      , "VEX.256.66.0F.WIG E7 /r"      , "AVX"],
        ["vmovntdqa"        , "W:xmm, m128"                                     , "RM"      , "VEX.128.66.0F38.WIG 2A /r"    , "AVX"],
        ["vmovntdqa"        , "W:ymm, m256"                                     , "RM"      , "VEX.256.66.0F38.WIG 2A /r"    , "AVX2"],
        ["vmovntpd"         , "W:m128, xmm"                                     , "MR"      , "VEX.128.66.0F.WIG 2B /r"      , "AVX"],
        ["vmovntpd"         , "W:m256, ymm"                                     , "MR"      , "VEX.256.66.0F.WIG 2B /r"      , "AVX"],
        ["vmovntps"         , "W:m128, xmm"                                     , "MR"      , "VEX.128.0F.WIG 2B /r"         , "AVX"],
        ["vmovntps"         , "W:m256, ymm"                                     , "MR"      , "VEX.256.0F.WIG 2B /r"         , "AVX"],
        ["vmovq"            , "W:r64/m64, xmm[63:0]"                            , "MR"      , "VEX.128.66.0F.W1 7E /r"       , "AVX X64"],
        ["vmovq"            , "W:xmm[63:0], xmm[63:0]/m64"                      , "RM"      , "VEX.128.F3.0F.WIG 7E /r"      , "AVX"],
        ["vmovq"            , "W:xmm[63:0], r64/m64"                            , "RM"      , "VEX.128.66.0F.W1 6E /r"       , "AVX X64"],
        ["vmovq"            , "W:xmm[63:0]/m64, xmm[63:0]"                      , "MR"      , "VEX.128.66.0F.WIG D6 /r"      , "AVX"],
        ["vmovsd"           , "W:m64, xmm[63:0]"                                , "MR"      , "VEX.LIG.F2.0F.WIG 11 /r"      , "AVX"],
        ["vmovsd"           , "W:xmm[63:0], m64"                                , "RM"      , "VEX.LIG.F2.0F.WIG 10 /r"      , "AVX"],
        ["vmovsd"           , "W:xmm, xmm[127:64], xmm[63:0]"                   , "MVR"     , "VEX.LIG.F2.0F.WIG 11 /r"      , "AVX"],
        ["vmovsd"           , "W:xmm, xmm[127:64], xmm[63:0]"                   , "RVM"     , "VEX.LIG.F2.0F.WIG 10 /r"      , "AVX"],
        ["vmovshdup"        , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.F3.0F.WIG 16 /r"      , "AVX"],
        ["vmovshdup"        , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.F3.0F.WIG 16 /r"      , "AVX"],
        ["vmovsldup"        , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.F3.0F.WIG 12 /r"      , "AVX"],
        ["vmovsldup"        , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.F3.0F.WIG 12 /r"      , "AVX"],
        ["vmovss"           , "W:m32, xmm[31:0]"                                , "MR"      , "VEX.LIG.F3.0F.WIG 11 /r"      , "AVX"],
        ["vmovss"           , "W:xmm[31:0], m32"                                , "RM"      , "VEX.LIG.F3.0F.WIG 10 /r"      , "AVX"],
        ["vmovss"           , "W:xmm, xmm[127:32], xmm[31:0]"                   , "MVR"     , "VEX.LIG.F3.0F.WIG 11 /r"      , "AVX"],
        ["vmovss"           , "W:xmm, xmm[127:32], xmm[31:0]"                   , "RVM"     , "VEX.LIG.F3.0F.WIG 10 /r"      , "AVX"],
        ["vmovupd"          , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.0F.WIG 10 /r"         , "AVX"],
        ["vmovupd"          , "W:xmm/m128, xmm"                                 , "MR"      , "VEX.128.0F.WIG 11 /r"         , "AVX"],
        ["vmovupd"          , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.0F.WIG 10 /r"         , "AVX"],
        ["vmovupd"          , "W:ymm/m256, ymm"                                 , "MR"      , "VEX.256.0F.WIG 11 /r"         , "AVX"],
        ["vmovups"          , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.66.0F.WIG 10 /r"      , "AVX"],
        ["vmovups"          , "W:xmm/m128, xmm"                                 , "MR"      , "VEX.128.66.0F.WIG 11 /r"      , "AVX"],
        ["vmovups"          , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.66.0F.WIG 10 /r"      , "AVX"],
        ["vmovups"          , "W:ymm/m256, ymm"                                 , "MR"      , "VEX.256.66.0F.WIG 11 /r"      , "AVX"],
        ["vmpsadbw"         , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.128.66.0F3A.WIG 42 /r ib" , "AVX"],
        ["vmpsadbw"         , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.66.0F3A.WIG 42 /r ib" , "AVX2"],
        ["vmulpd"           , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 59 /r"      , "AVX"],
        ["vmulpd"           , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 59 /r"      , "AVX"],
        ["vmulps"           , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.0F.WIG 59 /r"         , "AVX"],
        ["vmulps"           , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.0F.WIG 59 /r"         , "AVX"],
        ["vmulsd"           , "W:xmm, xmm, xmm[63:0]/m64"                       , "RVM"     , "VEX.LIG.F2.0F.WIG 59 /r"      , "AVX"],
        ["vmulss"           , "W:xmm, xmm, xmm[31:0]/m32"                       , "RVM"     , "VEX.LIG.F3.0F.WIG 59 /r"      , "AVX"],
        ["vorpd"            , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 56 /r"      , "AVX"],
        ["vorpd"            , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 56 /r"      , "AVX"],
        ["vorps"            , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.0F.WIG 56 /r"         , "AVX"],
        ["vorps"            , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.0F.WIG 56 /r"         , "AVX"],
        ["vpabsb"           , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.66.0F38.WIG 1C /r"    , "AVX"],
        ["vpabsb"           , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.66.0F38.WIG 1C /r"    , "AVX2"],
        ["vpabsd"           , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.66.0F38.WIG 1E /r"    , "AVX"],
        ["vpabsd"           , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.66.0F38.WIG 1E /r"    , "AVX2"],
        ["vpabsw"           , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.66.0F38.WIG 1D /r"    , "AVX"],
        ["vpabsw"           , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.66.0F38.WIG 1D /r"    , "AVX2"],
        ["vpackssdw"        , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 6B /r"      , "AVX"],
        ["vpackssdw"        , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 6B /r"      , "AVX2"],
        ["vpacksswb"        , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 63 /r"      , "AVX"],
        ["vpacksswb"        , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 63 /r"      , "AVX2"],
        ["vpackusdw"        , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 2B /r"    , "AVX"],
        ["vpackusdw"        , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 2B /r"    , "AVX2"],
        ["vpackuswb"        , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 67 /r"      , "AVX"],
        ["vpackuswb"        , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 67 /r"      , "AVX2"],
        ["vpaddb"           , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG FC /r"      , "AVX"],
        ["vpaddb"           , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG FC /r"      , "AVX2"],
        ["vpaddd"           , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG FE /r"      , "AVX"],
        ["vpaddd"           , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG FE /r"      , "AVX2"],
        ["vpaddq"           , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG D4 /r"      , "AVX"],
        ["vpaddq"           , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG D4 /r"      , "AVX2"],
        ["vpaddsb"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG EC /r"      , "AVX"],
        ["vpaddsb"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG EC /r"      , "AVX2"],
        ["vpaddsw"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG ED /r"      , "AVX"],
        ["vpaddsw"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG ED /r"      , "AVX2"],
        ["vpaddusb"         , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG DC /r"      , "AVX"],
        ["vpaddusb"         , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG DC /r"      , "AVX2"],
        ["vpaddusw"         , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG DD /r"      , "AVX"],
        ["vpaddusw"         , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG DD /r"      , "AVX2"],
        ["vpaddw"           , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG FD /r"      , "AVX"],
        ["vpaddw"           , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG FD /r"      , "AVX2"],
        ["vpalignr"         , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.128.66.0F3A.WIG 0F /r ib" , "AVX"],
        ["vpalignr"         , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.66.0F3A.WIG 0F /r ib" , "AVX2"],
        ["vpand"            , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG DB /r"      , "AVX"],
        ["vpand"            , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG DB /r"      , "AVX2"],
        ["vpandn"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG DF /r"      , "AVX"],
        ["vpandn"           , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG DF /r"      , "AVX2"],
        ["vpavgb"           , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG E0 /r"      , "AVX"],
        ["vpavgb"           , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG E0 /r"      , "AVX2"],
        ["vpavgw"           , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG E3 /r"      , "AVX"],
        ["vpavgw"           , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG E3 /r"      , "AVX2"],
        ["vpblendd"         , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.128.66.0F3A.W0 02 /r ib"  , "AVX2"],
        ["vpblendd"         , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.66.0F3A.W0 02 /r ib"  , "AVX2"],
        ["vpblendvb"        , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "VEX.128.66.0F3A.W0 4C /r /is4", "AVX"],
        ["vpblendvb"        , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "VEX.256.66.0F3A.W0 4C /r /is4", "AVX2"],
        ["vpblendw"         , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.128.66.0F3A.WIG 0E /r ib" , "AVX"],
        ["vpblendw"         , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.66.0F3A.WIG 0E /r ib" , "AVX2"],
        ["vpbroadcastb"     , "W:xmm, xmm[7:0]/m8"                              , "RM"      , "VEX.128.66.0F38.W0 78 /r"     , "AVX2"],
        ["vpbroadcastb"     , "W:ymm, xmm[7:0]/m8"                              , "RM"      , "VEX.256.66.0F38.W0 78 /r"     , "AVX2"],
        ["vpbroadcastd"     , "W:xmm, xmm[31:0]/m32"                            , "RM"      , "VEX.128.66.0F38.W0 58 /r"     , "AVX2"],
        ["vpbroadcastd"     , "W:ymm, xmm[31:0]/m32"                            , "RM"      , "VEX.256.66.0F38.W0 58 /r"     , "AVX2"],
        ["vpbroadcastq"     , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "VEX.128.66.0F38.W0 59 /r"     , "AVX2"],
        ["vpbroadcastq"     , "W:ymm, xmm[63:0]/m64"                            , "RM"      , "VEX.256.66.0F38.W0 59 /r"     , "AVX2"],
        ["vpbroadcastw"     , "W:xmm, xmm[15:0]/m16"                            , "RM"      , "VEX.128.66.0F38.W0 79 /r"     , "AVX2"],
        ["vpbroadcastw"     , "W:ymm, xmm[15:0]/m16"                            , "RM"      , "VEX.256.66.0F38.W0 79 /r"     , "AVX2"],
        ["vpclmulqdq"       , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.128.66.0F3A.WIG 44 /r ib" , "AVX PCLMULQDQ"],
        ["vpclmulqdq"       , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.66.0F3A.WIG 44 /r ib" , "VPCLMULQDQ"],
        ["vpcmpeqb"         , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 74 /r"      , "AVX"],
        ["vpcmpeqb"         , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 74 /r"      , "AVX2"],
        ["vpcmpeqd"         , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 76 /r"      , "AVX"],
        ["vpcmpeqd"         , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 76 /r"      , "AVX2"],
        ["vpcmpeqq"         , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 29 /r"    , "AVX"],
        ["vpcmpeqq"         , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 29 /r"    , "AVX2"],
        ["vpcmpeqw"         , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 75 /r"      , "AVX"],
        ["vpcmpeqw"         , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 75 /r"      , "AVX2"],
        ["vpcmpestri"       , "R:xmm, xmm/m128, ib/ub, W:<ecx>,<eax>,<edx>"     , "RMI"     , "VEX.128.66.0F3A.WIG 61 /r ib" , "AVX OF=W SF=W ZF=W AF=0 PF=0 CF=W"],
        ["vpcmpestrm"       , "R:xmm, xmm/m128, ib/ub, W:<xmm0>,<eax>,<edx>"    , "RMI"     , "VEX.128.66.0F3A.WIG 60 /r ib" , "AVX OF=W SF=W ZF=W AF=0 PF=0 CF=W"],
        ["vpcmpgtb"         , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 64 /r"      , "AVX"],
        ["vpcmpgtb"         , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 64 /r"      , "AVX2"],
        ["vpcmpgtd"         , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 66 /r"      , "AVX"],
        ["vpcmpgtd"         , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 66 /r"      , "AVX2"],
        ["vpcmpgtq"         , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 37 /r"    , "AVX"],
        ["vpcmpgtq"         , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 37 /r"    , "AVX2"],
        ["vpcmpgtw"         , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 65 /r"      , "AVX"],
        ["vpcmpgtw"         , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 65 /r"      , "AVX2"],
        ["vpcmpistri"       , "R:xmm, xmm/m128, ib/ub, W:<ecx>"                 , "RMI"     , "VEX.128.66.0F3A.WIG 63 /r ib" , "AVX OF=W SF=W ZF=W AF=0 PF=0 CF=W"],
        ["vpcmpistrm"       , "R:xmm, xmm/m128, ib/ub, W:<xmm0>"                , "RMI"     , "VEX.128.66.0F3A.WIG 62 /r ib" , "AVX OF=W SF=W ZF=W AF=0 PF=0 CF=W"],
        ["vperm2f128"       , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.66.0F3A.W0 06 /r ib"  , "AVX"],
        ["vperm2i128"       , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.66.0F3A.W0 46 /r ib"  , "AVX2"],
        ["vpermd"           , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 36 /r"     , "AVX2"],
        ["vpermilpd"        , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 0D /r"     , "AVX"],
        ["vpermilpd"        , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "VEX.128.66.0F3A.W0 05 /r ib"  , "AVX"],
        ["vpermilpd"        , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 0D /r"     , "AVX"],
        ["vpermilpd"        , "W:ymm, ymm/m256, ib/ub"                          , "RMI"     , "VEX.256.66.0F3A.W0 05 /r ib"  , "AVX"],
        ["vpermilps"        , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 0C /r"     , "AVX"],
        ["vpermilps"        , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "VEX.128.66.0F3A.W0 04 /r ib"  , "AVX"],
        ["vpermilps"        , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 0C /r"     , "AVX"],
        ["vpermilps"        , "W:ymm, ymm/m256, ib/ub"                          , "RMI"     , "VEX.256.66.0F3A.W0 04 /r ib"  , "AVX"],
        ["vpermpd"          , "W:ymm, ymm/m256, ib/ub"                          , "RMI"     , "VEX.256.66.0F3A.W1 01 /r ib"  , "AVX2"],
        ["vpermps"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 16 /r"     , "AVX2"],
        ["vpermq"           , "W:ymm, ymm/m256, ib/ub"                          , "RMI"     , "VEX.256.66.0F3A.W1 00 /r ib"  , "AVX2"],
        ["vpextrb"          , "W:r32[7:0]/m8, xmm, ib/ub"                       , "MRI"     , "VEX.128.66.0F3A.W0 14 /r ib"  , "AVX"],
        ["vpextrd"          , "W:r32/m32, xmm, ib/ub"                           , "MRI"     , "VEX.128.66.0F3A.W0 16 /r ib"  , "AVX"],
        ["vpextrq"          , "W:r64/m64, xmm, ib/ub"                           , "MRI"     , "VEX.128.66.0F3A.W1 16 /r ib"  , "AVX X64"],
        ["vpextrw"          , "W:r32[15:0], xmm, ib/ub"                         , "RMI"     , "VEX.128.66.0F.W0 C5 /r ib"    , "AVX"],
        ["vpextrw"          , "W:r32[15:0]/m16, xmm, ib/ub"                     , "MRI"     , "VEX.128.66.0F3A.W0 15 /r ib"  , "AVX"],
        ["vpgatherdd"       , "X:xmm, vm32x, X:xmm"                             , "RMV"     , "VEX.128.66.0F38.W0 90 /r"     , "AVX2"],
        ["vpgatherdd"       , "X:ymm, vm32y, X:ymm"                             , "RMV"     , "VEX.256.66.0F38.W0 90 /r"     , "AVX2"],
        ["vpgatherdq"       , "X:xmm, vm32x, X:xmm"                             , "RMV"     , "VEX.128.66.0F38.W1 90 /r"     , "AVX2"],
        ["vpgatherdq"       , "X:ymm, vm32x, X:ymm"                             , "RMV"     , "VEX.256.66.0F38.W1 90 /r"     , "AVX2"],
        ["vpgatherqd"       , "X:xmm, vm64x, X:xmm"                             , "RMV"     , "VEX.128.66.0F38.W0 91 /r"     , "AVX2"],
        ["vpgatherqd"       , "X:xmm, vm64y, X:xmm"                             , "RMV"     , "VEX.256.66.0F38.W0 91 /r"     , "AVX2"],
        ["vpgatherqq"       , "X:xmm, vm64x, X:xmm"                             , "RMV"     , "VEX.128.66.0F38.W1 91 /r"     , "AVX2"],
        ["vpgatherqq"       , "X:ymm, vm64y, X:ymm"                             , "RMV"     , "VEX.256.66.0F38.W1 91 /r"     , "AVX2"],
        ["vphaddd"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 02 /r"    , "AVX"],
        ["vphaddd"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 02 /r"    , "AVX2"],
        ["vphaddsw"         , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 03 /r"    , "AVX"],
        ["vphaddsw"         , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 03 /r"    , "AVX2"],
        ["vphaddw"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 01 /r"    , "AVX"],
        ["vphaddw"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 01 /r"    , "AVX2"],
        ["vphminposuw"      , "W:xmm[18:0], xmm/m128"                           , "RM"      , "VEX.128.66.0F38.WIG 41 /r"    , "AVX"],
        ["vphsubd"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 06 /r"    , "AVX"],
        ["vphsubd"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 06 /r"    , "AVX2"],
        ["vphsubsw"         , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 07 /r"    , "AVX"],
        ["vphsubsw"         , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 07 /r"    , "AVX2"],
        ["vphsubw"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 05 /r"    , "AVX"],
        ["vphsubw"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 05 /r"    , "AVX2"],
        ["vpinsrb"          , "W:xmm, xmm, r32[7:0]/m8, ib/ub"                  , "RVMI"    , "VEX.128.66.0F3A.W0 20 /r ib"  , "AVX"],
        ["vpinsrd"          , "W:xmm, xmm, r32/m32, ib/ub"                      , "RVMI"    , "VEX.128.66.0F3A.W0 22 /r ib"  , "AVX"],
        ["vpinsrq"          , "W:xmm, xmm, r64/m64, ib/ub"                      , "RVMI"    , "VEX.128.66.0F3A.W1 22 /r ib"  , "AVX X64"],
        ["vpinsrw"          , "W:xmm, xmm, r32[15:0]/m16, ib/ub"                , "RVMI"    , "VEX.128.66.0F.W0 C4 /r ib"    , "AVX"],
        ["vpmaddubsw"       , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 04 /r"    , "AVX"],
        ["vpmaddubsw"       , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 04 /r"    , "AVX2"],
        ["vpmaddwd"         , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG F5 /r"      , "AVX"],
        ["vpmaddwd"         , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG F5 /r"      , "AVX2"],
        ["vpmaskmovd"       , "X:m128, xmm, xmm"                                , "MVR"     , "VEX.128.66.0F38.W0 8E /r"     , "AVX2"],
        ["vpmaskmovd"       , "X:m256, ymm, ymm"                                , "MVR"     , "VEX.256.66.0F38.W0 8E /r"     , "AVX2"],
        ["vpmaskmovd"       , "W:xmm, xmm, m128"                                , "RVM"     , "VEX.128.66.0F38.W0 8C /r"     , "AVX2"],
        ["vpmaskmovd"       , "W:ymm, ymm, m256"                                , "RVM"     , "VEX.256.66.0F38.W0 8C /r"     , "AVX2"],
        ["vpmaskmovq"       , "X:m128, xmm, xmm"                                , "MVR"     , "VEX.128.66.0F38.W1 8E /r"     , "AVX2"],
        ["vpmaskmovq"       , "X:m256, ymm, ymm"                                , "MVR"     , "VEX.256.66.0F38.W1 8E /r"     , "AVX2"],
        ["vpmaskmovq"       , "W:xmm, xmm, m128"                                , "RVM"     , "VEX.128.66.0F38.W1 8C /r"     , "AVX2"],
        ["vpmaskmovq"       , "W:ymm, ymm, m256"                                , "RVM"     , "VEX.256.66.0F38.W1 8C /r"     , "AVX2"],
        ["vpmaxsb"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 3C /r"    , "AVX"],
        ["vpmaxsb"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 3C /r"    , "AVX2"],
        ["vpmaxsd"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 3D /r"    , "AVX"],
        ["vpmaxsd"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 3D /r"    , "AVX2"],
        ["vpmaxsw"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG EE /r"      , "AVX"],
        ["vpmaxsw"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG EE /r"      , "AVX2"],
        ["vpmaxub"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG DE /r"      , "AVX"],
        ["vpmaxub"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG DE /r"      , "AVX2"],
        ["vpmaxud"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 3F /r"    , "AVX"],
        ["vpmaxud"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 3F /r"    , "AVX2"],
        ["vpmaxuw"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 3E /r"    , "AVX"],
        ["vpmaxuw"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 3E /r"    , "AVX2"],
        ["vpminsb"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 38 /r"    , "AVX"],
        ["vpminsb"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 38 /r"    , "AVX2"],
        ["vpminsd"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 39 /r"    , "AVX"],
        ["vpminsd"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 39 /r"    , "AVX2"],
        ["vpminsw"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG EA /r"      , "AVX"],
        ["vpminsw"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG EA /r"      , "AVX2"],
        ["vpminub"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG DA /r"      , "AVX"],
        ["vpminub"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG DA /r"      , "AVX2"],
        ["vpminud"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 3B /r"    , "AVX"],
        ["vpminud"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 3B /r"    , "AVX2"],
        ["vpminuw"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 3A /r"    , "AVX"],
        ["vpminuw"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 3A /r"    , "AVX2"],
        ["vpmovmskb"        , "W:r32[15:0], xmm"                                , "RVM"     , "VEX.128.66.0F.WIG D7 /r"      , "AVX"],
        ["vpmovmskb"        , "W:r32[31:0], ymm"                                , "RVM"     , "VEX.256.66.0F.WIG D7 /r"      , "AVX2"],
        ["vpmovsxbd"        , "W:xmm, xmm[31:0]/m32"                            , "RM"      , "VEX.128.66.0F38.WIG 21 /r"    , "AVX"],
        ["vpmovsxbd"        , "W:ymm, xmm[63:0]/m64"                            , "RM"      , "VEX.256.66.0F38.WIG 21 /r"    , "AVX2"],
        ["vpmovsxbq"        , "W:xmm, xmm[15:0]/m16"                            , "RM"      , "VEX.128.66.0F38.WIG 22 /r"    , "AVX"],
        ["vpmovsxbq"        , "W:ymm, xmm[31:0]/m32"                            , "RM"      , "VEX.256.66.0F38.WIG 22 /r"    , "AVX2"],
        ["vpmovsxbw"        , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "VEX.128.66.0F38.WIG 20 /r"    , "AVX"],
        ["vpmovsxbw"        , "W:ymm, xmm/m128"                                 , "RM"      , "VEX.256.66.0F38.WIG 20 /r"    , "AVX2"],
        ["vpmovsxdq"        , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "VEX.128.66.0F38.WIG 25 /r"    , "AVX"],
        ["vpmovsxdq"        , "W:ymm, xmm/m128"                                 , "RM"      , "VEX.256.66.0F38.WIG 25 /r"    , "AVX2"],
        ["vpmovsxwd"        , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "VEX.128.66.0F38.WIG 23 /r"    , "AVX"],
        ["vpmovsxwd"        , "W:ymm, xmm/m128"                                 , "RM"      , "VEX.256.66.0F38.WIG 23 /r"    , "AVX2"],
        ["vpmovsxwq"        , "W:xmm, xmm[31:0]/m32"                            , "RM"      , "VEX.128.66.0F38.WIG 24 /r"    , "AVX"],
        ["vpmovsxwq"        , "W:ymm, xmm[63:0]/m64"                            , "RM"      , "VEX.256.66.0F38.WIG 24 /r"    , "AVX2"],
        ["vpmovzxbd"        , "W:xmm, xmm[31:0]/m32"                            , "RM"      , "VEX.128.66.0F38.WIG 31 /r"    , "AVX"],
        ["vpmovzxbd"        , "W:ymm, xmm[63:0]/m64"                            , "RM"      , "VEX.256.66.0F38.WIG 31 /r"    , "AVX2"],
        ["vpmovzxbq"        , "W:xmm, xmm[15:0]/m16"                            , "RM"      , "VEX.128.66.0F38.WIG 32 /r"    , "AVX"],
        ["vpmovzxbq"        , "W:ymm, xmm[31:0]/m32"                            , "RM"      , "VEX.256.66.0F38.WIG 32 /r"    , "AVX2"],
        ["vpmovzxbw"        , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "VEX.128.66.0F38.WIG 30 /r"    , "AVX"],
        ["vpmovzxbw"        , "W:ymm, xmm/m128"                                 , "RM"      , "VEX.256.66.0F38.WIG 30 /r"    , "AVX2"],
        ["vpmovzxdq"        , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "VEX.128.66.0F38.WIG 35 /r"    , "AVX"],
        ["vpmovzxdq"        , "W:ymm, xmm/m128"                                 , "RM"      , "VEX.256.66.0F38.WIG 35 /r"    , "AVX2"],
        ["vpmovzxwd"        , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "VEX.128.66.0F38.WIG 33 /r"    , "AVX"],
        ["vpmovzxwd"        , "W:ymm, xmm/m128"                                 , "RM"      , "VEX.256.66.0F38.WIG 33 /r"    , "AVX2"],
        ["vpmovzxwq"        , "W:xmm, xmm[31:0]/m32"                            , "RM"      , "VEX.128.66.0F38.WIG 34 /r"    , "AVX"],
        ["vpmovzxwq"        , "W:ymm, xmm[63:0]/m64"                            , "RM"      , "VEX.256.66.0F38.WIG 34 /r"    , "AVX2"],
        ["vpmuldq"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 28 /r"    , "AVX"],
        ["vpmuldq"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 28 /r"    , "AVX2"],
        ["vpmulhrsw"        , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 0B /r"    , "AVX"],
        ["vpmulhrsw"        , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 0B /r"    , "AVX2"],
        ["vpmulhuw"         , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG E4 /r"      , "AVX"],
        ["vpmulhuw"         , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG E4 /r"      , "AVX2"],
        ["vpmulhw"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG E5 /r"      , "AVX"],
        ["vpmulhw"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG E5 /r"      , "AVX2"],
        ["vpmulld"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 40 /r"    , "AVX"],
        ["vpmulld"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 40 /r"    , "AVX2"],
        ["vpmullw"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG D5 /r"      , "AVX"],
        ["vpmullw"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG D5 /r"      , "AVX2"],
        ["vpmuludq"         , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG F4 /r"      , "AVX"],
        ["vpmuludq"         , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG F4 /r"      , "AVX2"],
        ["vpor"             , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG EB /r"      , "AVX"],
        ["vpor"             , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG EB /r"      , "AVX2"],
        ["vpsadbw"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG F6 /r"      , "AVX"],
        ["vpsadbw"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG F6 /r"      , "AVX2"],
        ["vpshufb"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 00 /r"    , "AVX"],
        ["vpshufb"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 00 /r"    , "AVX2"],
        ["vpshufd"          , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "VEX.128.66.0F.WIG 70 /r ib"   , "AVX"],
        ["vpshufd"          , "W:ymm, ymm/m256, ib/ub"                          , "RMI"     , "VEX.256.66.0F.WIG 70 /r ib"   , "AVX2"],
        ["vpshufhw"         , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "VEX.128.F3.0F.WIG 70 /r ib"   , "AVX"],
        ["vpshufhw"         , "W:ymm, ymm/m256, ib/ub"                          , "RMI"     , "VEX.256.F3.0F.WIG 70 /r ib"   , "AVX2"],
        ["vpshuflw"         , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "VEX.128.F2.0F.WIG 70 /r ib"   , "AVX"],
        ["vpshuflw"         , "W:ymm, ymm/m256, ib/ub"                          , "RMI"     , "VEX.256.F2.0F.WIG 70 /r ib"   , "AVX2"],
        ["vpsignb"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 08 /r"    , "AVX"],
        ["vpsignb"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 08 /r"    , "AVX2"],
        ["vpsignd"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 0A /r"    , "AVX"],
        ["vpsignd"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 0A /r"    , "AVX2"],
        ["vpsignw"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.WIG 09 /r"    , "AVX"],
        ["vpsignw"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.WIG 09 /r"    , "AVX2"],
        ["vpslld"           , "W:xmm, xmm, ib/ub"                               , "VMI"     , "VEX.128.66.0F.WIG 72 /6 ib"   , "AVX"],
        ["vpslld"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG F2 /r"      , "AVX"],
        ["vpslld"           , "W:ymm, ymm, ib/ub"                               , "VMI"     , "VEX.256.66.0F.WIG 72 /6 ib"   , "AVX2"],
        ["vpslld"           , "W:ymm, ymm, xmm/m128"                            , "RVM"     , "VEX.256.66.0F.WIG F2 /r"      , "AVX2"],
        ["vpslldq"          , "W:xmm, xmm, ib/ub"                               , "VMI"     , "VEX.128.66.0F.WIG 73 /7 ib"   , "AVX"],
        ["vpslldq"          , "W:ymm, ymm, ib/ub"                               , "VMI"     , "VEX.256.66.0F.WIG 73 /7 ib"   , "AVX2"],
        ["vpsllq"           , "W:xmm, xmm, ib/ub"                               , "VMI"     , "VEX.128.66.0F.WIG 73 /6 ib"   , "AVX"],
        ["vpsllq"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG F3 /r"      , "AVX"],
        ["vpsllq"           , "W:ymm, ymm, ib/ub"                               , "VMI"     , "VEX.256.66.0F.WIG 73 /6 ib"   , "AVX2"],
        ["vpsllq"           , "W:ymm, ymm, xmm/m128"                            , "RVM"     , "VEX.256.66.0F.WIG F3 /r"      , "AVX2"],
        ["vpsllvd"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 47 /r"     , "AVX2"],
        ["vpsllvd"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 47 /r"     , "AVX2"],
        ["vpsllvq"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 47 /r"     , "AVX2"],
        ["vpsllvq"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 47 /r"     , "AVX2"],
        ["vpsllw"           , "W:xmm, xmm, ib/ub"                               , "VMI"     , "VEX.128.66.0F.WIG 71 /6 ib"   , "AVX"],
        ["vpsllw"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG F1 /r"      , "AVX"],
        ["vpsllw"           , "W:ymm, ymm, ib/ub"                               , "VMI"     , "VEX.256.66.0F.WIG 71 /6 ib"   , "AVX2"],
        ["vpsllw"           , "W:ymm, ymm, xmm/m128"                            , "RVM"     , "VEX.256.66.0F.WIG F1 /r"      , "AVX2"],
        ["vpsrad"           , "W:xmm, xmm, ib/ub"                               , "VMI"     , "VEX.128.66.0F.WIG 72 /4 ib"   , "AVX"],
        ["vpsrad"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG E2 /r"      , "AVX"],
        ["vpsrad"           , "W:ymm, ymm, ib/ub"                               , "VMI"     , "VEX.256.66.0F.WIG 72 /4 ib"   , "AVX2"],
        ["vpsrad"           , "W:ymm, ymm, xmm/m128"                            , "RVM"     , "VEX.256.66.0F.WIG E2 /r"      , "AVX2"],
        ["vpsravd"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 46 /r"     , "AVX2"],
        ["vpsravd"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 46 /r"     , "AVX2"],
        ["vpsraw"           , "W:xmm, xmm, ib/ub"                               , "VMI"     , "VEX.128.66.0F.WIG 71 /4 ib"   , "AVX"],
        ["vpsraw"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG E1 /r"      , "AVX"],
        ["vpsraw"           , "W:ymm, ymm, ib/ub"                               , "VMI"     , "VEX.256.66.0F.WIG 71 /4 ib"   , "AVX2"],
        ["vpsraw"           , "W:ymm, ymm, xmm/m128"                            , "RVM"     , "VEX.256.66.0F.WIG E1 /r"      , "AVX2"],
        ["vpsrld"           , "W:xmm, xmm, ib/ub"                               , "VMI"     , "VEX.128.66.0F.WIG 72 /2 ib"   , "AVX"],
        ["vpsrld"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG D2 /r"      , "AVX"],
        ["vpsrld"           , "W:ymm, ymm, ib/ub"                               , "VMI"     , "VEX.256.66.0F.WIG 72 /2 ib"   , "AVX2"],
        ["vpsrld"           , "W:ymm, ymm, xmm/m128"                            , "RVM"     , "VEX.256.66.0F.WIG D2 /r"      , "AVX2"],
        ["vpsrldq"          , "W:xmm, xmm, ib/ub"                               , "VMI"     , "VEX.128.66.0F.WIG 73 /3 ib"   , "AVX"],
        ["vpsrldq"          , "W:ymm, ymm, ib/ub"                               , "VMI"     , "VEX.256.66.0F.WIG 73 /3 ib"   , "AVX2"],
        ["vpsrlq"           , "W:xmm, xmm, ib/ub"                               , "VMI"     , "VEX.128.66.0F.WIG 73 /2 ib"   , "AVX"],
        ["vpsrlq"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG D3 /r"      , "AVX"],
        ["vpsrlq"           , "W:ymm, ymm, ib/ub"                               , "VMI"     , "VEX.256.66.0F.WIG 73 /2 ib"   , "AVX2"],
        ["vpsrlq"           , "W:ymm, ymm, xmm/m128"                            , "RVM"     , "VEX.256.66.0F.WIG D3 /r"      , "AVX2"],
        ["vpsrlvd"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 45 /r"     , "AVX2"],
        ["vpsrlvd"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 45 /r"     , "AVX2"],
        ["vpsrlvq"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 45 /r"     , "AVX2"],
        ["vpsrlvq"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 45 /r"     , "AVX2"],
        ["vpsrlw"           , "W:xmm, xmm, ib/ub"                               , "VMI"     , "VEX.128.66.0F.WIG 71 /2 ib"   , "AVX"],
        ["vpsrlw"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG D1 /r"      , "AVX"],
        ["vpsrlw"           , "W:ymm, ymm, ib/ub"                               , "VMI"     , "VEX.256.66.0F.WIG 71 /2 ib"   , "AVX2"],
        ["vpsrlw"           , "W:ymm, ymm, xmm/m128"                            , "RVM"     , "VEX.256.66.0F.WIG D1 /r"      , "AVX2"],
        ["vpsubb"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG F8 /r"      , "AVX"],
        ["vpsubb"           , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG F8 /r"      , "AVX2"],
        ["vpsubd"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG FA /r"      , "AVX"],
        ["vpsubd"           , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG FA /r"      , "AVX2"],
        ["vpsubq"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG FB /r"      , "AVX"],
        ["vpsubq"           , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG FB /r"      , "AVX2"],
        ["vpsubsb"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG E8 /r"      , "AVX"],
        ["vpsubsb"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG E8 /r"      , "AVX2"],
        ["vpsubsw"          , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG E9 /r"      , "AVX"],
        ["vpsubsw"          , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG E9 /r"      , "AVX2"],
        ["vpsubusb"         , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG D8 /r"      , "AVX"],
        ["vpsubusb"         , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG D8 /r"      , "AVX2"],
        ["vpsubusw"         , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG D9 /r"      , "AVX"],
        ["vpsubusw"         , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG D9 /r"      , "AVX2"],
        ["vpsubw"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG F9 /r"      , "AVX"],
        ["vpsubw"           , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG F9 /r"      , "AVX2"],
        ["vptest"           , "R:~xmm, ~xmm/m128"                               , "RM"      , "VEX.128.66.0F38.WIG 17 /r"    , "AVX OF=0 SF=0 ZF=W AF=0 PF=0 CF=W"],
        ["vptest"           , "R:~ymm, ~ymm/m256"                               , "RM"      , "VEX.256.66.0F38.WIG 17 /r"    , "AVX OF=0 SF=0 ZF=W AF=0 PF=0 CF=W"],
        ["vpunpckhbw"       , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 68 /r"      , "AVX"],
        ["vpunpckhbw"       , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 68 /r"      , "AVX2"],
        ["vpunpckhdq"       , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 6A /r"      , "AVX"],
        ["vpunpckhdq"       , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 6A /r"      , "AVX2"],
        ["vpunpckhqdq"      , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 6D /r"      , "AVX"],
        ["vpunpckhqdq"      , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 6D /r"      , "AVX2"],
        ["vpunpckhwd"       , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 69 /r"      , "AVX"],
        ["vpunpckhwd"       , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 69 /r"      , "AVX2"],
        ["vpunpcklbw"       , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 60 /r"      , "AVX"],
        ["vpunpcklbw"       , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 60 /r"      , "AVX2"],
        ["vpunpckldq"       , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 62 /r"      , "AVX"],
        ["vpunpckldq"       , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 62 /r"      , "AVX2"],
        ["vpunpcklqdq"      , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 6C /r"      , "AVX"],
        ["vpunpcklqdq"      , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 6C /r"      , "AVX2"],
        ["vpunpcklwd"       , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 61 /r"      , "AVX"],
        ["vpunpcklwd"       , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 61 /r"      , "AVX2"],
        ["vpxor"            , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG EF /r"      , "AVX"],
        ["vpxor"            , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG EF /r"      , "AVX2"],
        ["vrcpps"           , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.0F.WIG 53 /r"         , "AVX"],
        ["vrcpps"           , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.0F.WIG 53 /r"         , "AVX"],
        ["vrcpss"           , "W:xmm, xmm[127:32], xmm[31:0]/m32"               , "RVM"     , "VEX.LIG.F3.0F.WIG 53 /r"      , "AVX"],
        ["vroundpd"         , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "VEX.128.66.0F3A.WIG 09 /r ib" , "AVX"],
        ["vroundpd"         , "W:ymm, ymm/m256, ib/ub"                          , "RMI"     , "VEX.256.66.0F3A.WIG 09 /r ib" , "AVX"],
        ["vroundps"         , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "VEX.128.66.0F3A.WIG 08 /r ib" , "AVX"],
        ["vroundps"         , "W:ymm, ymm/m256, ib/ub"                          , "RMI"     , "VEX.256.66.0F3A.WIG 08 /r ib" , "AVX"],
        ["vroundsd"         , "W:xmm, xmm[127:64], xmm[63:0]/m64, ib/ub"        , "RVMI"    , "VEX.LIG.66.0F3A.WIG 0B /r ib" , "AVX"],
        ["vroundss"         , "W:xmm, xmm[127:32], xmm[31:0]/m32, ib/ub"        , "RVMI"    , "VEX.LIG.66.0F3A.WIG 0A /r ib" , "AVX"],
        ["vrsqrtps"         , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.0F.WIG 52 /r"         , "AVX"],
        ["vrsqrtps"         , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.0F.WIG 52 /r"         , "AVX"],
        ["vrsqrtss"         , "W:xmm, xmm[127:32], xmm[31:0]/m32"               , "RVM"     , "VEX.LIG.F3.0F.WIG 52 /r"      , "AVX"],
        ["vshufpd"          , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.128.66.0F.WIG C6 /r ib"   , "AVX"],
        ["vshufpd"          , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.66.0F.WIG C6 /r ib"   , "AVX"],
        ["vshufps"          , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "VEX.128.0F.WIG C6 /r ib"      , "AVX"],
        ["vshufps"          , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI"    , "VEX.256.0F.WIG C6 /r ib"      , "AVX"],
        ["vsqrtpd"          , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.66.0F.WIG 51 /r"      , "AVX"],
        ["vsqrtpd"          , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.66.0F.WIG 51 /r"      , "AVX"],
        ["vsqrtps"          , "W:xmm, xmm/m128"                                 , "RM"      , "VEX.128.0F.WIG 51 /r"         , "AVX"],
        ["vsqrtps"          , "W:ymm, ymm/m256"                                 , "RM"      , "VEX.256.0F.WIG 51 /r"         , "AVX"],
        ["vsqrtsd"          , "W:xmm, xmm[127:64], xmm[63:0]/m64"               , "RVM"     , "VEX.LIG.F2.0F.WIG 51 /r"      , "AVX"],
        ["vsqrtss"          , "W:xmm, xmm[127:32], xmm[31:0]/m32"               , "RVM"     , "VEX.LIG.F3.0F.WIG 51 /r"      , "AVX"],
        ["vsubpd"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 5C /r"      , "AVX"],
        ["vsubpd"           , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 5C /r"      , "AVX"],
        ["vsubps"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.0F.WIG 5C /r"         , "AVX"],
        ["vsubps"           , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.0F.WIG 5C /r"         , "AVX"],
        ["vsubsd"           , "W:xmm, xmm, xmm[63:0]/m64"                       , "RVM"     , "VEX.LIG.F2.0F.WIG 5C /r"      , "AVX"],
        ["vsubss"           , "W:xmm, xmm, xmm[31:0]/m32"                       , "RVM"     , "VEX.LIG.F3.0F.WIG 5C /r"      , "AVX"],
        ["vtestpd"          , "R:~xmm, ~xmm/m128"                               , "RM"      , "VEX.128.66.0F38.W0 0F /r"     , "AVX OF=0 SF=0 ZF=W AF=0 PF=0 CF=W"],
        ["vtestpd"          , "R:~ymm, ~ymm/m256"                               , "RM"      , "VEX.256.66.0F38.W0 0F /r"     , "AVX OF=0 SF=0 ZF=W AF=0 PF=0 CF=W"],
        ["vtestps"          , "R:~xmm, ~xmm/m128"                               , "RM"      , "VEX.128.66.0F38.W0 0E /r"     , "AVX OF=0 SF=0 ZF=W AF=0 PF=0 CF=W"],
        ["vtestps"          , "R:~ymm, ~ymm/m256"                               , "RM"      , "VEX.256.66.0F38.W0 0E /r"     , "AVX OF=0 SF=0 ZF=W AF=0 PF=0 CF=W"],
        ["vucomisd"         , "R:xmm[63:0], xmm[63:0]/m64"                      , "RM"      , "VEX.LIG.66.0F.WIG 2E /r"      , "AVX OF=0 SF=0 ZF=W AF=0 PF=W CF=W"],
        ["vucomiss"         , "R:xmm[31:0], xmm[31:0]/m32"                      , "RM"      , "VEX.LIG.0F.WIG 2E /r"         , "AVX OF=0 SF=0 ZF=W AF=0 PF=W CF=W"],
        ["vunpckhpd"        , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 15 /r"      , "AVX"],
        ["vunpckhpd"        , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 15 /r"      , "AVX"],
        ["vunpckhps"        , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.0F.WIG 15 /r"         , "AVX"],
        ["vunpckhps"        , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.0F.WIG 15 /r"         , "AVX"],
        ["vunpcklpd"        , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 14 /r"      , "AVX"],
        ["vunpcklpd"        , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 14 /r"      , "AVX"],
        ["vunpcklps"        , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.0F.WIG 14 /r"         , "AVX"],
        ["vunpcklps"        , "W:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.0F.WIG 14 /r"         , "AVX"],
        ["vxorpd"           , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.66.0F.WIG 57 /r"      , "AVX"],
        ["vxorpd"           , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.66.0F.WIG 57 /r"      , "AVX"],
        ["vxorps"           , "W:xmm,~xmm,~xmm/m128"                            , "RVM"     , "VEX.128.0F.WIG 57 /r"         , "AVX"],
        ["vxorps"           , "W:ymm,~ymm,~ymm/m256"                            , "RVM"     , "VEX.256.0F.WIG 57 /r"         , "AVX"],
    
        ["vldmxcsr"         , "R:m32"                                           , "M"       , "VEX.LZ.0F.WIG AE /2"          , "AVX"],
        ["vstmxcsr"         , "W:m32"                                           , "M"       , "VEX.LZ.0F.WIG AE /3"          , "AVX"],
        ["vzeroall"         , ""                                                , "NONE"    , "VEX.256.0F.WIG 77"            , "AVX Volatile"],
        ["vzeroupper"       , ""                                                , "NONE"    , "VEX.128.0F.WIG 77"            , "AVX Volatile"],
    
        ["vcvtph2ps"        , "W:xmm, xmm[63:0]/m64"                            , "RM"      , "VEX.128.66.0F38.W0 13 /r"     , "F16C"],
        ["vcvtph2ps"        , "W:ymm, xmm/m128"                                 , "RM"      , "VEX.256.66.0F38.W0 13 /r"     , "F16C"],
        ["vcvtps2ph"        , "W:xmm[63:0]/m64, xmm, ib/ub"                     , "MRI"     , "VEX.128.66.0F3A.W0 1D /r ib"  , "F16C"],
        ["vcvtps2ph"        , "W:xmm/m128, ymm, ib/ub"                          , "MRI"     , "VEX.256.66.0F3A.W0 1D /r ib"  , "F16C"],
    
        ["vfmadd132pd"      , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 98 /r"     , "FMA"],
        ["vfmadd132pd"      , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 98 /r"     , "FMA"],
        ["vfmadd132ps"      , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 98 /r"     , "FMA"],
        ["vfmadd132ps"      , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 98 /r"     , "FMA"],
        ["vfmadd132sd"      , "x:xmm[63:0], xmm[63:0], xmm[63:0]/m64"           , "RVM"     , "VEX.LIG.66.0F38.W1 99 /r"     , "FMA"],
        ["vfmadd132ss"      , "x:xmm[31:0], xmm[31:0], xmm[31:0]/m32"           , "RVM"     , "VEX.LIG.66.0F38.W0 99 /r"     , "FMA"],
        ["vfmadd213pd"      , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 A8 /r"     , "FMA"],
        ["vfmadd213pd"      , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 A8 /r"     , "FMA"],
        ["vfmadd213ps"      , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 A8 /r"     , "FMA"],
        ["vfmadd213ps"      , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 A8 /r"     , "FMA"],
        ["vfmadd213sd"      , "x:xmm[63:0], xmm[63:0], xmm[63:0]/m64"           , "RVM"     , "VEX.LIG.66.0F38.W1 A9 /r"     , "FMA"],
        ["vfmadd213ss"      , "x:xmm[31:0], xmm[31:0], xmm[31:0]/m32"           , "RVM"     , "VEX.LIG.66.0F38.W0 A9 /r"     , "FMA"],
        ["vfmadd231pd"      , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 B8 /r"     , "FMA"],
        ["vfmadd231pd"      , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 B8 /r"     , "FMA"],
        ["vfmadd231ps"      , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 B8 /r"     , "FMA"],
        ["vfmadd231ps"      , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 B8 /r"     , "FMA"],
        ["vfmadd231sd"      , "x:xmm[63:0], xmm[63:0], xmm[63:0]/m64"           , "RVM"     , "VEX.LIG.66.0F38.W1 B9 /r"     , "FMA"],
        ["vfmadd231ss"      , "x:xmm[31:0], xmm[31:0], xmm[31:0]/m32"           , "RVM"     , "VEX.LIG.66.0F38.W0 B9 /r"     , "FMA"],
        ["vfmaddsub132pd"   , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 96 /r"     , "FMA"],
        ["vfmaddsub132pd"   , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 96 /r"     , "FMA"],
        ["vfmaddsub132ps"   , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 96 /r"     , "FMA"],
        ["vfmaddsub132ps"   , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 96 /r"     , "FMA"],
        ["vfmaddsub213pd"   , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 A6 /r"     , "FMA"],
        ["vfmaddsub213pd"   , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 A6 /r"     , "FMA"],
        ["vfmaddsub213ps"   , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 A6 /r"     , "FMA"],
        ["vfmaddsub213ps"   , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 A6 /r"     , "FMA"],
        ["vfmaddsub231pd"   , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 B6 /r"     , "FMA"],
        ["vfmaddsub231pd"   , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 B6 /r"     , "FMA"],
        ["vfmaddsub231ps"   , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 B6 /r"     , "FMA"],
        ["vfmaddsub231ps"   , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 B6 /r"     , "FMA"],
        ["vfmsub132pd"      , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 9A /r"     , "FMA"],
        ["vfmsub132pd"      , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 9A /r"     , "FMA"],
        ["vfmsub132ps"      , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 9A /r"     , "FMA"],
        ["vfmsub132ps"      , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 9A /r"     , "FMA"],
        ["vfmsub132sd"      , "x:xmm[63:0], xmm[63:0], xmm[63:0]/m64"           , "RVM"     , "VEX.LIG.66.0F38.W1 9B /r"     , "FMA"],
        ["vfmsub132ss"      , "x:xmm[31:0], xmm[31:0], xmm[31:0]/m32"           , "RVM"     , "VEX.LIG.66.0F38.W0 9B /r"     , "FMA"],
        ["vfmsub213pd"      , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 AA /r"     , "FMA"],
        ["vfmsub213pd"      , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 AA /r"     , "FMA"],
        ["vfmsub213ps"      , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 AA /r"     , "FMA"],
        ["vfmsub213ps"      , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 AA /r"     , "FMA"],
        ["vfmsub213sd"      , "x:xmm[63:0], xmm[63:0], xmm[63:0]/m64"           , "RVM"     , "VEX.LIG.66.0F38.W1 AB /r"     , "FMA"],
        ["vfmsub213ss"      , "x:xmm[31:0], xmm[31:0], xmm[31:0]/m32"           , "RVM"     , "VEX.LIG.66.0F38.W0 AB /r"     , "FMA"],
        ["vfmsub231pd"      , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 BA /r"     , "FMA"],
        ["vfmsub231pd"      , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 BA /r"     , "FMA"],
        ["vfmsub231ps"      , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 BA /r"     , "FMA"],
        ["vfmsub231ps"      , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 BA /r"     , "FMA"],
        ["vfmsub231sd"      , "x:xmm[63:0], xmm[63:0], xmm[63:0]/m64"           , "RVM"     , "VEX.LIG.66.0F38.W1 BB /r"     , "FMA"],
        ["vfmsub231ss"      , "x:xmm[31:0], xmm[31:0], xmm[31:0]/m32"           , "RVM"     , "VEX.LIG.66.0F38.W0 BB /r"     , "FMA"],
        ["vfmsubadd132pd"   , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 97 /r"     , "FMA"],
        ["vfmsubadd132pd"   , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 97 /r"     , "FMA"],
        ["vfmsubadd132ps"   , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 97 /r"     , "FMA"],
        ["vfmsubadd132ps"   , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 97 /r"     , "FMA"],
        ["vfmsubadd213pd"   , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 A7 /r"     , "FMA"],
        ["vfmsubadd213pd"   , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 A7 /r"     , "FMA"],
        ["vfmsubadd213ps"   , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 A7 /r"     , "FMA"],
        ["vfmsubadd213ps"   , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 A7 /r"     , "FMA"],
        ["vfmsubadd231pd"   , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 B7 /r"     , "FMA"],
        ["vfmsubadd231pd"   , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 B7 /r"     , "FMA"],
        ["vfmsubadd231ps"   , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 B7 /r"     , "FMA"],
        ["vfmsubadd231ps"   , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 B7 /r"     , "FMA"],
        ["vfnmadd132pd"     , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 9C /r"     , "FMA"],
        ["vfnmadd132pd"     , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 9C /r"     , "FMA"],
        ["vfnmadd132ps"     , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 9C /r"     , "FMA"],
        ["vfnmadd132ps"     , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 9C /r"     , "FMA"],
        ["vfnmadd132sd"     , "x:xmm[63:0], xmm[63:0], xmm[63:0]/m64"           , "RVM"     , "VEX.LIG.66.0F38.W1 9D /r"     , "FMA"],
        ["vfnmadd132ss"     , "x:xmm[31:0], xmm[31:0], xmm[31:0]/m32"           , "RVM"     , "VEX.LIG.66.0F38.W0 9D /r"     , "FMA"],
        ["vfnmadd213pd"     , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 AC /r"     , "FMA"],
        ["vfnmadd213pd"     , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 AC /r"     , "FMA"],
        ["vfnmadd213ps"     , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 AC /r"     , "FMA"],
        ["vfnmadd213ps"     , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 AC /r"     , "FMA"],
        ["vfnmadd213sd"     , "x:xmm[63:0], xmm[63:0], xmm[63:0]/m64"           , "RVM"     , "VEX.LIG.66.0F38.W1 AD /r"     , "FMA"],
        ["vfnmadd213ss"     , "x:xmm[31:0], xmm[31:0], xmm[31:0]/m32"           , "RVM"     , "VEX.LIG.66.0F38.W0 AD /r"     , "FMA"],
        ["vfnmadd231pd"     , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 BC /r"     , "FMA"],
        ["vfnmadd231pd"     , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 BC /r"     , "FMA"],
        ["vfnmadd231ps"     , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 BC /r"     , "FMA"],
        ["vfnmadd231ps"     , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 BC /r"     , "FMA"],
        ["vfnmadd231sd"     , "x:xmm[63:0], xmm[63:0], xmm[63:0]/m64"           , "RVM"     , "VEX.LIG.66.0F38.W1 BD /r"     , "FMA"],
        ["vfnmadd231ss"     , "x:xmm[31:0], xmm[31:0], xmm[31:0]/m32"           , "RVM"     , "VEX.LIG.66.0F38.W0 BD /r"     , "FMA"],
        ["vfnmsub132pd"     , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 9E /r"     , "FMA"],
        ["vfnmsub132pd"     , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 9E /r"     , "FMA"],
        ["vfnmsub132ps"     , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 9E /r"     , "FMA"],
        ["vfnmsub132ps"     , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 9E /r"     , "FMA"],
        ["vfnmsub132sd"     , "x:xmm[63:0], xmm[63:0], xmm[63:0]/m64"           , "RVM"     , "VEX.LIG.66.0F38.W1 9F /r"     , "FMA"],
        ["vfnmsub132ss"     , "x:xmm[31:0], xmm[31:0], xmm[31:0]/m32"           , "RVM"     , "VEX.LIG.66.0F38.W0 9F /r"     , "FMA"],
        ["vfnmsub213pd"     , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 AE /r"     , "FMA"],
        ["vfnmsub213pd"     , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 AE /r"     , "FMA"],
        ["vfnmsub213ps"     , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 AE /r"     , "FMA"],
        ["vfnmsub213ps"     , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 AE /r"     , "FMA"],
        ["vfnmsub213sd"     , "x:xmm[63:0], xmm[63:0], xmm[63:0]/m64"           , "RVM"     , "VEX.LIG.66.0F38.W1 AF /r"     , "FMA"],
        ["vfnmsub213ss"     , "x:xmm[31:0], xmm[31:0], xmm[31:0]/m32"           , "RVM"     , "VEX.LIG.66.0F38.W0 AF /r"     , "FMA"],
        ["vfnmsub231pd"     , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W1 BE /r"     , "FMA"],
        ["vfnmsub231pd"     , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W1 BE /r"     , "FMA"],
        ["vfnmsub231ps"     , "X:xmm, xmm, xmm/m128"                            , "RVM"     , "VEX.128.66.0F38.W0 BE /r"     , "FMA"],
        ["vfnmsub231ps"     , "X:ymm, ymm, ymm/m256"                            , "RVM"     , "VEX.256.66.0F38.W0 BE /r"     , "FMA"],
        ["vfnmsub231sd"     , "x:xmm[63:0], xmm[63:0], xmm[63:0]/m64"           , "RVM"     , "VEX.LIG.66.0F38.W1 BF /r"     , "FMA"],
        ["vfnmsub231ss"     , "x:xmm[31:0], xmm[31:0], xmm[31:0]/m32"           , "RVM"     , "VEX.LIG.66.0F38.W0 BF /r"     , "FMA"],
    
        ["vfmaddpd"         , "W:xmm, xmm, xmm, xmm/m128"                       , "RVSM"    , "VEX.128.66.0F3A.W1 69 /r /is4", "FMA4"],
        ["vfmaddpd"         , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "VEX.128.66.0F3A.W0 69 /r /is4", "FMA4"],
        ["vfmaddpd"         , "W:ymm, ymm, ymm, ymm/m256"                       , "RVSM"    , "VEX.256.66.0F3A.W1 69 /r /is4", "FMA4"],
        ["vfmaddpd"         , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "VEX.256.66.0F3A.W0 69 /r /is4", "FMA4"],
        ["vfmaddps"         , "W:xmm, xmm, xmm, xmm/m128"                       , "RVSM"    , "VEX.128.66.0F3A.W1 68 /r /is4", "FMA4"],
        ["vfmaddps"         , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "VEX.128.66.0F3A.W0 68 /r /is4", "FMA4"],
        ["vfmaddps"         , "W:ymm, ymm, ymm, ymm/m256"                       , "RVSM"    , "VEX.256.66.0F3A.W1 68 /r /is4", "FMA4"],
        ["vfmaddps"         , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "VEX.256.66.0F3A.W0 68 /r /is4", "FMA4"],
        ["vfmaddsd"         , "W:xmm[63:0], xmm[63:0], xmm[63:0], xmm[63:0]/m64", "RVSM"    , "VEX.128.66.0F3A.W1 6b /r /is4", "FMA4"],
        ["vfmaddsd"         , "W:xmm[63:0], xmm[63:0], xmm[63:0]/m64, xmm[63:0]", "RVMS"    , "VEX.128.66.0F3A.W0 6b /r /is4", "FMA4"],
        ["vfmaddss"         , "W:xmm[31:0], xmm[31:0], xmm[31:0], xmm[31:0]/m32", "RVSM"    , "VEX.128.66.0F3A.W1 6a /r /is4", "FMA4"],
        ["vfmaddss"         , "W:xmm[31:0], xmm[31:0], xmm[31:0]/m32, xmm[31:0]", "RVMS"    , "VEX.128.66.0F3A.W0 6a /r /is4", "FMA4"],
        ["vfmaddsubpd"      , "W:xmm, xmm, xmm, xmm/m128"                       , "RVSM"    , "VEX.128.66.0F3A.W1 5D /r /is4", "FMA4"],
        ["vfmaddsubpd"      , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "VEX.128.66.0F3A.W0 5D /r /is4", "FMA4"],
        ["vfmaddsubpd"      , "W:ymm, ymm, ymm, ymm/m256"                       , "RVSM"    , "VEX.256.66.0F3A.W1 5D /r /is4", "FMA4"],
        ["vfmaddsubpd"      , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "VEX.256.66.0F3A.W0 5D /r /is4", "FMA4"],
        ["vfmaddsubps"      , "W:xmm, xmm, xmm, xmm/m128"                       , "RVSM"    , "VEX.128.66.0F3A.W1 5C /r /is4", "FMA4"],
        ["vfmaddsubps"      , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "VEX.128.66.0F3A.W0 5C /r /is4", "FMA4"],
        ["vfmaddsubps"      , "W:ymm, ymm, ymm, ymm/m256"                       , "RVSM"    , "VEX.256.66.0F3A.W1 5C /r /is4", "FMA4"],
        ["vfmaddsubps"      , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "VEX.256.66.0F3A.W0 5C /r /is4", "FMA4"],
        ["vfmsubaddpd"      , "W:xmm, xmm, xmm, xmm/m128"                       , "RVSM"    , "VEX.128.66.0F3A.W1 5F /r /is4", "FMA4"],
        ["vfmsubaddpd"      , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "VEX.128.66.0F3A.W0 5F /r /is4", "FMA4"],
        ["vfmsubaddpd"      , "W:ymm, ymm, ymm, ymm/m256"                       , "RVSM"    , "VEX.256.66.0F3A.W1 5F /r /is4", "FMA4"],
        ["vfmsubaddpd"      , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "VEX.256.66.0F3A.W0 5F /r /is4", "FMA4"],
        ["vfmsubaddps"      , "W:xmm, xmm, xmm, xmm/m128"                       , "RVSM"    , "VEX.128.66.0F3A.W1 5E /r /is4", "FMA4"],
        ["vfmsubaddps"      , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "VEX.128.66.0F3A.W0 5E /r /is4", "FMA4"],
        ["vfmsubaddps"      , "W:ymm, ymm, ymm, ymm/m256"                       , "RVSM"    , "VEX.256.66.0F3A.W1 5E /r /is4", "FMA4"],
        ["vfmsubaddps"      , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "VEX.256.66.0F3A.W0 5E /r /is4", "FMA4"],
        ["vfmsubpd"         , "W:xmm, xmm, xmm, xmm/m128"                       , "RVSM"    , "VEX.128.66.0F3A.W1 6D /r /is4", "FMA4"],
        ["vfmsubpd"         , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "VEX.128.66.0F3A.W0 6D /r /is4", "FMA4"],
        ["vfmsubpd"         , "W:ymm, ymm, ymm, ymm/m256"                       , "RVSM"    , "VEX.256.66.0F3A.W1 6D /r /is4", "FMA4"],
        ["vfmsubpd"         , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "VEX.256.66.0F3A.W0 6D /r /is4", "FMA4"],
        ["vfmsubps"         , "W:xmm, xmm, xmm, xmm/m128"                       , "RVSM"    , "VEX.128.66.0F3A.W1 6C /r /is4", "FMA4"],
        ["vfmsubps"         , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "VEX.128.66.0F3A.W0 6C /r /is4", "FMA4"],
        ["vfmsubps"         , "W:ymm, ymm, ymm, ymm/m256"                       , "RVSM"    , "VEX.256.66.0F3A.W1 6C /r /is4", "FMA4"],
        ["vfmsubps"         , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "VEX.256.66.0F3A.W0 6C /r /is4", "FMA4"],
        ["vfmsubsd"         , "W:xmm[63:0], xmm[63:0], xmm[63:0], xmm[63:0]/m64", "RVSM"    , "VEX.128.66.0F3A.W1 6F /r /is4", "FMA4"],
        ["vfmsubsd"         , "W:xmm[63:0], xmm[63:0], xmm[63:0]/m64, xmm[63:0]", "RVMS"    , "VEX.128.66.0F3A.W0 6F /r /is4", "FMA4"],
        ["vfmsubss"         , "W:xmm[31:0], xmm[31:0], xmm[31:0], xmm[31:0]/m32", "RVSM"    , "VEX.128.66.0F3A.W1 6E /r /is4", "FMA4"],
        ["vfmsubss"         , "W:xmm[31:0], xmm[31:0], xmm[31:0]/m32, xmm[31:0]", "RVMS"    , "VEX.128.66.0F3A.W0 6E /r /is4", "FMA4"],
        ["vfnmaddpd"        , "W:xmm, xmm, xmm, xmm/m128"                       , "RVSM"    , "VEX.128.66.0F3A.W1 79 /r /is4", "FMA4"],
        ["vfnmaddpd"        , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "VEX.128.66.0F3A.W0 79 /r /is4", "FMA4"],
        ["vfnmaddpd"        , "W:ymm, ymm, ymm, ymm/m256"                       , "RVSM"    , "VEX.256.66.0F3A.W1 79 /r /is4", "FMA4"],
        ["vfnmaddpd"        , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "VEX.256.66.0F3A.W0 79 /r /is4", "FMA4"],
        ["vfnmaddps"        , "W:xmm, xmm, xmm, xmm/m128"                       , "RVSM"    , "VEX.128.66.0F3A.W1 78 /r /is4", "FMA4"],
        ["vfnmaddps"        , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "VEX.128.66.0F3A.W0 78 /r /is4", "FMA4"],
        ["vfnmaddps"        , "W:ymm, ymm, ymm, ymm/m256"                       , "RVSM"    , "VEX.256.66.0F3A.W1 78 /r /is4", "FMA4"],
        ["vfnmaddps"        , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "VEX.256.66.0F3A.W0 78 /r /is4", "FMA4"],
        ["vfnmaddsd"        , "W:xmm[63:0], xmm[63:0], xmm[63:0], xmm[63:0]/m64", "RVSM"    , "VEX.128.66.0F3A.W1 7B /r /is4", "FMA4"],
        ["vfnmaddsd"        , "W:xmm[63:0], xmm[63:0], xmm[63:0]/m64, xmm[63:0]", "RVMS"    , "VEX.128.66.0F3A.W0 7B /r /is4", "FMA4"],
        ["vfnmaddss"        , "W:xmm[31:0], xmm[31:0], xmm[31:0], xmm[31:0]/m32", "RVSM"    , "VEX.128.66.0F3A.W1 7A /r /is4", "FMA4"],
        ["vfnmaddss"        , "W:xmm[31:0], xmm[31:0], xmm[31:0]/m32, xmm[31:0]", "RVMS"    , "VEX.128.66.0F3A.W0 7A /r /is4", "FMA4"],
        ["vfnmsubpd"        , "W:xmm, xmm, xmm, xmm/m128"                       , "RVSM"    , "VEX.128.66.0F3A.W1 7D /r /is4", "FMA4"],
        ["vfnmsubpd"        , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "VEX.128.66.0F3A.W0 7D /r /is4", "FMA4"],
        ["vfnmsubpd"        , "W:ymm, ymm, ymm, ymm/m256"                       , "RVSM"    , "VEX.256.66.0F3A.W1 7D /r /is4", "FMA4"],
        ["vfnmsubpd"        , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "VEX.256.66.0F3A.W0 7D /r /is4", "FMA4"],
        ["vfnmsubps"        , "W:xmm, xmm, xmm, xmm/m128"                       , "RVSM"    , "VEX.128.66.0F3A.W1 7C /r /is4", "FMA4"],
        ["vfnmsubps"        , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "VEX.128.66.0F3A.W0 7C /r /is4", "FMA4"],
        ["vfnmsubps"        , "W:ymm, ymm, ymm, ymm/m256"                       , "RVSM"    , "VEX.256.66.0F3A.W1 7C /r /is4", "FMA4"],
        ["vfnmsubps"        , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "VEX.256.66.0F3A.W0 7C /r /is4", "FMA4"],
        ["vfnmsubsd"        , "W:xmm[63:0], xmm[63:0], xmm[63:0], xmm[63:0]/m64", "RVSM"    , "VEX.128.66.0F3A.W1 7F /r /is4", "FMA4"],
        ["vfnmsubsd"        , "W:xmm[63:0], xmm[63:0], xmm[63:0]/m64, xmm[63:0]", "RVMS"    , "VEX.128.66.0F3A.W0 7F /r /is4", "FMA4"],
        ["vfnmsubss"        , "W:xmm[31:0], xmm[31:0], xmm[31:0], xmm[31:0]/m32", "RVSM"    , "VEX.128.66.0F3A.W1 7E /r /is4", "FMA4"],
        ["vfnmsubss"        , "W:xmm[31:0], xmm[31:0], xmm[31:0]/m32, xmm[31:0]", "RVMS"    , "VEX.128.66.0F3A.W0 7E /r /is4", "FMA4"],
    
        ["vfrczpd"          , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 81 /r"       , "XOP"],
        ["vfrczpd"          , "W:ymm, ymm/m256"                                 , "RM"      , "XOP.L1.P0.M09.W0 81 /r"       , "XOP"],
        ["vfrczps"          , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 80 /r"       , "XOP"],
        ["vfrczps"          , "W:ymm, ymm/m256"                                 , "RM"      , "XOP.L1.P0.M09.W0 80 /r"       , "XOP"],
        ["vfrczsd"          , "W:xmm[63:0], xmm[63:0]/m64"                      , "RM"      , "XOP.L0.P0.M09.W0 83 /r"       , "XOP"],
        ["vfrczss"          , "W:xmm[31:0], xmm[31:0]/m32"                      , "RM"      , "XOP.L0.P0.M09.W0 82 /r"       , "XOP"],
        ["vpcmov"           , "W:xmm, xmm, xmm, xmm/m128"                       , "RVSM"    , "XOP.L0.P0.M08.W1 A2 /r /is4"  , "XOP"],
        ["vpcmov"           , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "XOP.L0.P0.M08.W0 A2 /r /is4"  , "XOP"],
        ["vpcmov"           , "W:ymm, ymm, ymm, ymm/m256"                       , "RVSM"    , "XOP.L1.P0.M08.W1 A2 /r /is4"  , "XOP"],
        ["vpcmov"           , "W:ymm, ymm, ymm/m256, ymm"                       , "RVMS"    , "XOP.L1.P0.M08.W0 A2 /r /is4"  , "XOP"],
        ["vpcomb"           , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "XOP.L0.P0.M08.W0 CC /r ib"    , "XOP"],
        ["vpcomd"           , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "XOP.L0.P0.M08.W0 CE /r ib"    , "XOP"],
        ["vpcomq"           , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "XOP.L0.P0.M08.W0 CF /r ib"    , "XOP"],
        ["vpcomub"          , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "XOP.L0.P0.M08.W0 EC /r ib"    , "XOP"],
        ["vpcomud"          , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "XOP.L0.P0.M08.W0 EE /r ib"    , "XOP"],
        ["vpcomuq"          , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "XOP.L0.P0.M08.W0 EF /r ib"    , "XOP"],
        ["vpcomuw"          , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "XOP.L0.P0.M08.W0 ED /r ib"    , "XOP"],
        ["vpcomw"           , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI"    , "XOP.L0.P0.M08.W0 CD /r ib"    , "XOP"],
        ["vpermil2pd"       , "W:xmm, xmm, xmm/m128, xmm, i4/u4"                , "RVMSI"   , "VEX.L0.66.0F3A.W0 49 /r /is4" , "XOP"],
        ["vpermil2pd"       , "W:xmm, xmm, xmm, xmm/m128, i4/u4"                , "RVSMI"   , "VEX.L0.66.0F3A.W1 49 /r /is4" , "XOP"],
        ["vpermil2pd"       , "W:ymm, ymm, ymm/m256, ymm, i4/u4"                , "RVMSI"   , "VEX.L1.66.0F3A.W0 49 /r /is4" , "XOP"],
        ["vpermil2pd"       , "W:ymm, ymm, ymm, ymm/m256, i4/u4"                , "RVSMI"   , "VEX.L1.66.0F3A.W1 49 /r /is4" , "XOP"],
        ["vpermil2ps"       , "W:xmm, xmm, xmm/m128, xmm, i4/u4"                , "RVMSI"   , "VEX.L0.66.0F3A.W0 48 /r /is4" , "XOP"],
        ["vpermil2ps"       , "W:xmm, xmm, xmm, xmm/m128, i4/u4"                , "RVSMI"   , "VEX.L0.66.0F3A.W1 48 /r /is4" , "XOP"],
        ["vpermil2ps"       , "W:ymm, ymm, ymm/m256, ymm, i4/u4"                , "RVMSI"   , "VEX.L1.66.0F3A.W0 48 /r /is4" , "XOP"],
        ["vpermil2ps"       , "W:ymm, ymm, ymm, ymm/m256, i4/u4"                , "RVSMI"   , "VEX.L1.66.0F3A.W1 48 /r /is4" , "XOP"],
        ["vphaddbd"         , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 C2 /r"       , "XOP"],
        ["vphaddbq"         , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 C3 /r"       , "XOP"],
        ["vphaddbw"         , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 C1 /r"       , "XOP"],
        ["vphadddq"         , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 CB /r"       , "XOP"],
        ["vphaddubd"        , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 D2 /r"       , "XOP"],
        ["vphaddubq"        , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 D3 /r"       , "XOP"],
        ["vphaddubw"        , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 D1 /r"       , "XOP"],
        ["vphaddudq"        , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 DB /r"       , "XOP"],
        ["vphadduwd"        , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 D6 /r"       , "XOP"],
        ["vphadduwq"        , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 D7 /r"       , "XOP"],
        ["vphaddwd"         , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 C6 /r"       , "XOP"],
        ["vphaddwq"         , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 C7 /r"       , "XOP"],
        ["vphsubbw"         , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 E1 /r"       , "XOP"],
        ["vphsubdq"         , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 E3 /r"       , "XOP"],
        ["vphsubwd"         , "W:xmm, xmm/m128"                                 , "RM"      , "XOP.L0.P0.M09.W0 E2 /r"       , "XOP"],
        ["vpmacsdd"         , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "XOP.L0.P0.M08.W0 9E /r /is4"  , "XOP"],
        ["vpmacsdqh"        , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "XOP.L0.P0.M08.W0 9F /r /is4"  , "XOP"],
        ["vpmacsdql"        , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "XOP.L0.P0.M08.W0 97 /r /is4"  , "XOP"],
        ["vpmacssdd"        , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "XOP.L0.P0.M08.W0 8E /r /is4"  , "XOP"],
        ["vpmacssdqh"       , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "XOP.L0.P0.M08.W0 8F /r /is4"  , "XOP"],
        ["vpmacssdql"       , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "XOP.L0.P0.M08.W0 87 /r /is4"  , "XOP"],
        ["vpmacsswd"        , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "XOP.L0.P0.M08.W0 86 /r /is4"  , "XOP"],
        ["vpmacssww"        , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "XOP.L0.P0.M08.W0 85 /r /is4"  , "XOP"],
        ["vpmacswd"         , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "XOP.L0.P0.M08.W0 96 /r /is4"  , "XOP"],
        ["vpmacsww"         , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "XOP.L0.P0.M08.W0 95 /r /is4"  , "XOP"],
        ["vpmadcsswd"       , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "XOP.L0.P0.M08.W0 A6 /r /is4"  , "XOP"],
        ["vpmadcswd"        , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "XOP.L0.P0.M08.W0 B6 /r /is4"  , "XOP"],
        ["vpperm"           , "W:xmm, xmm, xmm, xmm/m128"                       , "RVSM"    , "XOP.L0.P0.M08.W1 A3 /r /is4"  , "XOP"],
        ["vpperm"           , "W:xmm, xmm, xmm/m128, xmm"                       , "RVMS"    , "XOP.L0.P0.M08.W0 A3 /r /is4"  , "XOP"],
        ["vprotb"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "XOP.L0.P0.M09.W1 90 /r"       , "XOP"],
        ["vprotb"           , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "XOP.L0.P0.M08.W0 C0 /r ib"    , "XOP"],
        ["vprotb"           , "W:xmm, xmm/m128, xmm"                            , "RMV"     , "XOP.L0.P0.M09.W0 90 /r"       , "XOP"],
        ["vprotd"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "XOP.L0.P0.M09.W1 92 /r"       , "XOP"],
        ["vprotd"           , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "XOP.L0.P0.M08.W0 C2 /r ib"    , "XOP"],
        ["vprotd"           , "W:xmm, xmm/m128, xmm"                            , "RMV"     , "XOP.L0.P0.M09.W0 92 /r"       , "XOP"],
        ["vprotq"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "XOP.L0.P0.M09.W1 93 /r"       , "XOP"],
        ["vprotq"           , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "XOP.L0.P0.M08.W0 C3 /r ib"    , "XOP"],
        ["vprotq"           , "W:xmm, xmm/m128, xmm"                            , "RMV"     , "XOP.L0.P0.M09.W0 93 /r"       , "XOP"],
        ["vprotw"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "XOP.L0.P0.M09.W1 91 /r"       , "XOP"],
        ["vprotw"           , "W:xmm, xmm/m128, ib/ub"                          , "RMI"     , "XOP.L0.P0.M08.W0 C1 /r ib"    , "XOP"],
        ["vprotw"           , "W:xmm, xmm/m128, xmm"                            , "RMV"     , "XOP.L0.P0.M09.W0 91 /r"       , "XOP"],
        ["vpshab"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "XOP.L0.P0.M09.W1 98 /r"       , "XOP"],
        ["vpshab"           , "W:xmm, xmm/m128, xmm"                            , "RMV"     , "XOP.L0.P0.M09.W0 98 /r"       , "XOP"],
        ["vpshad"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "XOP.L0.P0.M09.W1 9A /r"       , "XOP"],
        ["vpshad"           , "W:xmm, xmm/m128, xmm"                            , "RMV"     , "XOP.L0.P0.M09.W0 9A /r"       , "XOP"],
        ["vpshaq"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "XOP.L0.P0.M09.W1 9B /r"       , "XOP"],
        ["vpshaq"           , "W:xmm, xmm/m128, xmm"                            , "RMV"     , "XOP.L0.P0.M09.W0 9B /r"       , "XOP"],
        ["vpshaw"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "XOP.L0.P0.M09.W1 99 /r"       , "XOP"],
        ["vpshaw"           , "W:xmm, xmm/m128, xmm"                            , "RMV"     , "XOP.L0.P0.M09.W0 99 /r"       , "XOP"],
        ["vpshlb"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "XOP.L0.P0.M09.W1 94 /r"       , "XOP"],
        ["vpshlb"           , "W:xmm, xmm/m128, xmm"                            , "RMV"     , "XOP.L0.P0.M09.W0 94 /r"       , "XOP"],
        ["vpshld"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "XOP.L0.P0.M09.W1 96 /r"       , "XOP"],
        ["vpshld"           , "W:xmm, xmm/m128, xmm"                            , "RMV"     , "XOP.L0.P0.M09.W0 96 /r"       , "XOP"],
        ["vpshlq"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "XOP.L0.P0.M09.W1 97 /r"       , "XOP"],
        ["vpshlq"           , "W:xmm, xmm/m128, xmm"                            , "RMV"     , "XOP.L0.P0.M09.W0 97 /r"       , "XOP"],
        ["vpshlw"           , "W:xmm, xmm, xmm/m128"                            , "RVM"     , "XOP.L0.P0.M09.W1 95 /r"       , "XOP"],
        ["vpshlw"           , "W:xmm, xmm/m128, xmm"                            , "RMV"     , "XOP.L0.P0.M09.W0 95 /r"       , "XOP"],
    
        ["kaddb"            , "W:k[7:0] ,~k[7:0] ,~k[7:0]"                      , "RVM"     , "VEX.L1.66.0F.W0 4A /r"        , "AVX512_DQ"],
        ["kaddd"            , "W:k[31:0],~k[31:0],~k[31:0]"                     , "RVM"     , "VEX.L1.66.0F.W1 4A /r"        , "AVX512_BW"],
        ["kaddq"            , "W:k[63:0],~k[63:0],~k[63:0]"                     , "RVM"     , "VEX.L1.0F.W1 4A /r"           , "AVX512_BW"],
        ["kaddw"            , "W:k[15:0],~k[15:0],~k[15:0]"                     , "RVM"     , "VEX.L1.0F.W0 4A /r"           , "AVX512_DQ"],
        ["kandb"            , "W:k[7:0] ,~k[7:0] ,~k[7:0]"                      , "RVM"     , "VEX.L1.66.0F.W0 41 /r"        , "AVX512_DQ"],
        ["kandd"            , "W:k[31:0],~k[31:0],~k[31:0]"                     , "RVM"     , "VEX.L1.66.0F.W1 41 /r"        , "AVX512_BW"],
        ["kandnb"           , "W:k[7:0] , k[7:0] , k[7:0]"                      , "RVM"     , "VEX.L1.66.0F.W0 42 /r"        , "AVX512_DQ"],
        ["kandnd"           , "W:k[31:0], k[31:0], k[31:0]"                     , "RVM"     , "VEX.L1.66.0F.W1 42 /r"        , "AVX512_BW"],
        ["kandnq"           , "W:k[63:0], k[63:0], k[63:0]"                     , "RVM"     , "VEX.L1.0F.W1 42 /r"           , "AVX512_BW"],
        ["kandnw"           , "W:k[15:0], k[15:0], k[15:0]"                     , "RVM"     , "VEX.L1.0F.W0 42 /r"           , "AVX512_F"],
        ["kandq"            , "W:k[63:0],~k[63:0],~k[63:0]"                     , "RVM"     , "VEX.L1.0F.W1 41 /r"           , "AVX512_BW"],
        ["kandw"            , "W:k[15:0],~k[15:0],~k[15:0]"                     , "RVM"     , "VEX.L1.0F.W0 41 /r"           , "AVX512_F"],
        ["kmovb"            , "W:k[7:0], k[7:0]/m8"                             , "RM"      , "VEX.L0.66.0F.W0 90 /r"        , "AVX512_DQ"],
        ["kmovb"            , "W:k[7:0], r32[7:0]"                              , "RM"      , "VEX.L0.66.0F.W0 92 /r"        , "AVX512_DQ"],
        ["kmovb"            , "W:m8, k[7:0]"                                    , "MR"      , "VEX.L0.66.0F.W0 91 /r"        , "AVX512_DQ"],
        ["kmovb"            , "W:r32[7:0], k[7:0]"                              , "RM"      , "VEX.L0.66.0F.W0 93 /r"        , "AVX512_DQ"],
        ["kmovd"            , "W:k[31:0], k[31:0]/m32"                          , "RM"      , "VEX.L0.66.0F.W1 90 /r"        , "AVX512_BW"],
        ["kmovd"            , "W:k[31:0], r32[31:0]"                            , "RM"      , "VEX.L0.F2.0F.W0 92 /r"        , "AVX512_BW"],
        ["kmovd"            , "W:m32, k[31:0]"                                  , "MR"      , "VEX.L0.66.0F.W1 91 /r"        , "AVX512_BW"],
        ["kmovd"            , "W:r32[31:0], k[31:0]"                            , "RM"      , "VEX.L0.F2.0F.W0 93 /r"        , "AVX512_BW"],
        ["kmovq"            , "W:k[63:0], k[63:0]/m64"                          , "RM"      , "VEX.L0.0F.W1 90 /r"           , "AVX512_BW"],
        ["kmovq"            , "W:k[63:0], r64"                                  , "RM"      , "VEX.L0.F2.0F.W1 92 /r"        , "AVX512_BW X64"],
        ["kmovq"            , "W:m64, k[63:0]"                                  , "MR"      , "VEX.L0.0F.W1 91 /r"           , "AVX512_BW"],
        ["kmovq"            , "W:r64, k[63:0]"                                  , "RM"      , "VEX.L0.F2.0F.W1 93 /r"        , "AVX512_BW X64"],
        ["kmovw"            , "W:k[15:0], k[15:0]/m16"                          , "RM"      , "VEX.L0.0F.W0 90 /r"           , "AVX512_F"],
        ["kmovw"            , "W:k[15:0], r32[15:0]"                            , "RM"      , "VEX.L0.0F.W0 92 /r"           , "AVX512_F"],
        ["kmovw"            , "W:m16, k[15:0]"                                  , "MR"      , "VEX.L0.0F.W0 91 /r"           , "AVX512_F"],
        ["kmovw"            , "W:r32[15:0], k[15:0]"                            , "RM"      , "VEX.L0.0F.W0 93 /r"           , "AVX512_F"],
        ["knotb"            , "W:k[7:0] , k[7:0]"                               , "RM"      , "VEX.L0.66.0F.W0 44 /r"        , "AVX512_DQ"],
        ["knotd"            , "W:k[31:0], k[31:0]"                              , "RM"      , "VEX.L0.66.0F.W1 44 /r"        , "AVX512_BW"],
        ["knotq"            , "W:k[63:0] , k[63:0]"                             , "RM"      , "VEX.L0.0F.W1 44 /r"           , "AVX512_BW"],
        ["knotw"            , "W:k[15:0], k[15:0]"                              , "RM"      , "VEX.L0.0F.W0 44 /r"           , "AVX512_F"],
        ["korb"             , "W:k[7:0] , ~k[7:0] , ~k[7:0]"                    , "RVM"     , "VEX.L1.66.0F.W0 45 /r"        , "AVX512_DQ"],
        ["kord"             , "W:k[31:0], ~k[31:0], ~k[31:0]"                   , "RVM"     , "VEX.L1.66.0F.W1 45 /r"        , "AVX512_BW"],
        ["korq"             , "W:k[63:0], ~k[63:0], ~k[63:0]"                   , "RVM"     , "VEX.L1.0F.W1 45 /r"           , "AVX512_BW"],
        ["kortestb"         , "R:~k[7:0] , ~k[7:0]"                             , "RM"      , "VEX.L0.66.0F.W0 98 /r"        , "AVX512_DQ OF=0 SF=0 ZF=W AF=0 PF=0 CF=W"],
        ["kortestd"         , "R:~k[31:0], ~k[31:0]"                            , "RM"      , "VEX.L0.66.0F.W1 98 /r"        , "AVX512_BW OF=0 SF=0 ZF=W AF=0 PF=0 CF=W"],
        ["kortestq"         , "R:~k[63:0], ~k[63:0]"                            , "RM"      , "VEX.L0.0F.W1 98 /r"           , "AVX512_BW OF=0 SF=0 ZF=W AF=0 PF=0 CF=W"],
        ["kortestw"         , "R:~k[15:0], ~k[15:0]"                            , "RM"      , "VEX.L0.0F.W0 98 /r"           , "AVX512_F  OF=0 SF=0 ZF=W AF=0 PF=0 CF=W"],
        ["korw"             , "W:k[15:0], ~k[15:0], ~k[15:0]"                   , "RVM"     , "VEX.L1.0F.W0 45 /r"           , "AVX512_F"],
        ["kshiftlb"         , "W:k[7:0] , k[7:0] , ib/ub"                       , "RMI"     , "VEX.L0.66.0F3A.W0 32 /r ib"   , "AVX512_DQ"],
        ["kshiftld"         , "W:k[31:0], k[31:0], ib/ub"                       , "RMI"     , "VEX.L0.66.0F3A.W0 33 /r ib"   , "AVX512_BW"],
        ["kshiftlq"         , "W:k[63:0], k[63:0] , ib/ub"                      , "RMI"     , "VEX.L0.66.0F3A.W1 33 /r ib"   , "AVX512_BW"],
        ["kshiftlw"         , "W:k[15:0], k[15:0], ib/ub"                       , "RMI"     , "VEX.L0.66.0F3A.W1 32 /r ib"   , "AVX512_F"],
        ["kshiftrb"         , "W:k[7:0] , k[7:0] , ib/ub"                       , "RMI"     , "VEX.L0.66.0F3A.W0 30 /r ib"   , "AVX512_DQ"],
        ["kshiftrd"         , "W:k[31:0], k[31:0], ib/ub"                       , "RMI"     , "VEX.L0.66.0F3A.W0 31 /r ib"   , "AVX512_BW"],
        ["kshiftrq"         , "W:k[63:0], k[63:0], ib/ub"                       , "RMI"     , "VEX.L0.66.0F3A.W1 31 /r ib"   , "AVX512_BW"],
        ["kshiftrw"         , "W:k[15:0], k[15:0], ib/ub"                       , "RMI"     , "VEX.L0.66.0F3A.W1 30 /r ib"   , "AVX512_F"],
        ["ktestb"           , "R:~k[7:0] , ~k[7:0]"                             , "RM"      , "VEX.L0.66.0F.W0 99 /r"        , "AVX512_DQ OF=0 SF=0 ZF=W AF=0 PF=0 CF=W"],
        ["ktestd"           , "R:~k[31:0], ~k[31:0]"                            , "RM"      , "VEX.L0.66.0F.W1 99 /r"        , "AVX512_BW OF=0 SF=0 ZF=W AF=0 PF=0 CF=W"],
        ["ktestq"           , "R:~k[63:0] , ~k[63:0]"                           , "RM"      , "VEX.L0.0F.W1 99 /r"           , "AVX512_BW OF=0 SF=0 ZF=W AF=0 PF=0 CF=W"],
        ["ktestw"           , "R:~k[15:0], ~k[15:0]"                            , "RM"      , "VEX.L0.0F.W0 99 /r"           , "AVX512_DQ OF=0 SF=0 ZF=W AF=0 PF=0 CF=W"],
        ["kunpckbw"         , "W:k[15:0], k[7:0] , k[7:0]"                      , "RVM"     , "VEX.L1.66.0F.W0 4B /r"        , "AVX512_F"],
        ["kunpckdq"         , "W:k[63:0], k[31:0], k[31:0]"                     , "RVM"     , "VEX.L1.0F.W1 4B /r"           , "AVX512_BW"],
        ["kunpckwd"         , "W:k[31:0], k[15:0], k[15:0]"                     , "RVM"     , "VEX.L1.0F.W0 4B /r"           , "AVX512_BW"],
        ["kxnorb"           , "W:k[7:0] , k[7:0] , k[7:0]"                      , "RVM"     , "VEX.L1.66.0F.W0 46 /r"        , "AVX512_DQ"],
        ["kxnord"           , "W:k[31:0], k[31:0], k[31:0]"                     , "RVM"     , "VEX.L1.66.0F.W1 46 /r"        , "AVX512_BW"],
        ["kxnorq"           , "W:k[63:0], k[63:0], k[63:0]"                     , "RVM"     , "VEX.L1.0F.W1 46 /r"           , "AVX512_BW"],
        ["kxnorw"           , "W:k[15:0], k[15:0], k[15:0]"                     , "RVM"     , "VEX.L1.0F.W0 46 /r"           , "AVX512_F"],
        ["kxorb"            , "W:k[7:0] ,~k[7:0] ,~k[7:0]"                      , "RVM"     , "VEX.L1.66.0F.W0 47 /r"        , "AVX512_DQ"],
        ["kxord"            , "W:k[31:0],~k[31:0],~k[31:0]"                     , "RVM"     , "VEX.L1.66.0F.W1 47 /r"        , "AVX512_BW"],
        ["kxorq"            , "W:k[63:0],~k[63:0],~k[63:0]"                     , "RVM"     , "VEX.L1.0F.W1 47 /r"           , "AVX512_BW"],
        ["kxorw"            , "W:k[15:0],~k[15:0],~k[15:0]"                     , "RVM"     , "VEX.L1.0F.W0 47 /r"           , "AVX512_F"],
    
        ["v4fmaddps"        , "X:zmm {kz}, zmm, zmm, zmm, zmm, m128"            , "RM-T1_4X", "EVEX.512.F2.0F38.W0 9A /r"    , "AVX512_4FMAPS"],
        ["v4fmaddss"        , "X:xmm {kz}, xmm, xmm, xmm, xmm, m128"            , "RM-T1_4X", "EVEX.LIG.F2.0F38.W0 9B /r"    , "AVX512_4FMAPS"],
        ["v4fnmaddps"       , "X:zmm {kz}, zmm, zmm, zmm, zmm, m128"            , "RM-T1_4X", "EVEX.512.F2.0F38.W0 AA /r"    , "AVX512_4FMAPS"],
        ["v4fnmaddss"       , "X:xmm {kz}, xmm, xmm, xmm, xmm, m128"            , "RM-T1_4X", "EVEX.LIG.F2.0F38.W0 AB /r"    , "AVX512_4FMAPS"],
        ["vaddpd"           , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 58 /r"      , "AVX512_F-VL"],
        ["vaddpd"           , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 58 /r"      , "AVX512_F-VL"],
        ["vaddpd"           , "W:zmm {kz},~zmm,~zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F.W1 58 /r"      , "AVX512_F"],
        ["vaddps"           , "W:xmm {kz},~xmm,~xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.0F.W0 58 /r"         , "AVX512_F-VL"],
        ["vaddps"           , "W:ymm {kz},~ymm,~ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.0F.W0 58 /r"         , "AVX512_F-VL"],
        ["vaddps"           , "W:zmm {kz},~zmm,~zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.0F.W0 58 /r"         , "AVX512_F"],
        ["vaddsd"           , "W:xmm {kz}, xmm, xmm[63:0]/m64 {er}"             , "RVM-T1S" , "EVEX.LIG.F2.0F.W1 58 /r"      , "AVX512_F"],
        ["vaddss"           , "W:xmm {kz}, xmm, xmm[31:0]/m32 {er}"             , "RVM-T1S" , "EVEX.LIG.F3.0F.W0 58 /r"      , "AVX512_F"],
        ["vaesdec"          , "W:xmm, xmm, xmm/m128"                            , "RVM-FVM" , "EVEX.128.66.0F38.WIG DE /r"   , "AVX512_F-VL VAES"],
        ["vaesdec"          , "W:ymm, ymm, ymm/m256"                            , "RVM-FVM" , "EVEX.256.66.0F38.WIG DE /r"   , "AVX512_F-VL VAES"],
        ["vaesdec"          , "W:zmm, zmm, zmm/m512"                            , "RVM-FVM" , "EVEX.512.66.0F38.WIG DE /r"   , "AVX512_F    VAES"],
        ["vaesdeclast"      , "W:xmm, xmm, xmm/m128"                            , "RVM-FVM" , "EVEX.128.66.0F38.WIG DF /r"   , "AVX512_F-VL VAES"],
        ["vaesdeclast"      , "W:ymm, ymm, ymm/m256"                            , "RVM-FVM" , "EVEX.256.66.0F38.WIG DF /r"   , "AVX512_F-VL VAES"],
        ["vaesdeclast"      , "W:zmm, zmm, zmm/m512"                            , "RVM-FVM" , "EVEX.512.66.0F38.WIG DF /r"   , "AVX512_F    VAES"],
        ["vaesenc"          , "W:xmm, xmm, xmm/m128"                            , "RVM-FVM" , "EVEX.128.66.0F38.WIG DC /r"   , "AVX512_F-VL VAES"],
        ["vaesenc"          , "W:ymm, ymm, ymm/m256"                            , "RVM-FVM" , "EVEX.256.66.0F38.WIG DC /r"   , "AVX512_F-VL VAES"],
        ["vaesenc"          , "W:zmm, zmm, zmm/m512"                            , "RVM-FVM" , "EVEX.512.66.0F38.WIG DC /r"   , "AVX512_F    VAES"],
        ["vaesenclast"      , "W:xmm, xmm, xmm/m128"                            , "RVM-FVM" , "EVEX.128.66.0F38.WIG DD /r"   , "AVX512_F-VL VAES"],
        ["vaesenclast"      , "W:ymm, ymm, ymm/m256"                            , "RVM-FVM" , "EVEX.256.66.0F38.WIG DD /r"   , "AVX512_F-VL VAES"],
        ["vaesenclast"      , "W:zmm, zmm, zmm/m512"                            , "RVM-FVM" , "EVEX.512.66.0F38.WIG DD /r"   , "AVX512_F    VAES"],
        ["valignd"          , "W:xmm {kz}, xmm, xmm/m128/b32, ib/ub"            , "RVMI-FV" , "EVEX.128.66.0F3A.W0 03 /r ib" , "AVX512_F-VL"],
        ["valignd"          , "W:ymm {kz}, ymm, ymm/m256/b32, ib/ub"            , "RVMI-FV" , "EVEX.256.66.0F3A.W0 03 /r ib" , "AVX512_F-VL"],
        ["valignd"          , "W:zmm {kz}, zmm, zmm/m512/b32, ib/ub"            , "RVMI-FV" , "EVEX.512.66.0F3A.W0 03 /r ib" , "AVX512_F"],
        ["valignq"          , "W:xmm {kz}, xmm, xmm/m128/b64, ib/ub"            , "RVMI-FV" , "EVEX.128.66.0F3A.W1 03 /r ib" , "AVX512_F-VL"],
        ["valignq"          , "W:ymm {kz}, ymm, ymm/m256/b64, ib/ub"            , "RVMI-FV" , "EVEX.256.66.0F3A.W1 03 /r ib" , "AVX512_F-VL"],
        ["valignq"          , "W:zmm {kz}, zmm, zmm/m512/b64, ib/ub"            , "RVMI-FV" , "EVEX.512.66.0F3A.W1 03 /r ib" , "AVX512_F"],
        ["vandnpd"          , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 55 /r"      , "AVX512_DQ-VL"],
        ["vandnpd"          , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 55 /r"      , "AVX512_DQ-VL"],
        ["vandnpd"          , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F.W1 55 /r"      , "AVX512_DQ"],
        ["vandnps"          , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.W0 55 /r"         , "AVX512_DQ-VL"],
        ["vandnps"          , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.W0 55 /r"         , "AVX512_DQ-VL"],
        ["vandnps"          , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.W0 55 /r"         , "AVX512_DQ"],
        ["vandpd"           , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 54 /r"      , "AVX512_DQ-VL"],
        ["vandpd"           , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 54 /r"      , "AVX512_DQ-VL"],
        ["vandpd"           , "W:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F.W1 54 /r"      , "AVX512_DQ"],
        ["vandps"           , "W:xmm {kz},~xmm,~xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.0F.W0 54 /r"         , "AVX512_DQ-VL"],
        ["vandps"           , "W:ymm {kz},~ymm,~ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.0F.W0 54 /r"         , "AVX512_DQ-VL"],
        ["vandps"           , "W:zmm {kz},~zmm,~zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.0F.W0 54 /r"         , "AVX512_DQ"],
        ["vblendmb"         , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.W0 66 /r"    , "AVX512_BW-VL"],
        ["vblendmb"         , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.W0 66 /r"    , "AVX512_BW-VL"],
        ["vblendmb"         , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.W0 66 /r"    , "AVX512_BW"],
        ["vblendmd"         , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 64 /r"    , "AVX512_F-VL"],
        ["vblendmd"         , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 64 /r"    , "AVX512_F-VL"],
        ["vblendmd"         , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 64 /r"    , "AVX512_F"],
        ["vblendmpd"        , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 65 /r"    , "AVX512_F-VL"],
        ["vblendmpd"        , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 65 /r"    , "AVX512_F-VL"],
        ["vblendmpd"        , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 65 /r"    , "AVX512_F"],
        ["vblendmps"        , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 65 /r"    , "AVX512_F-VL"],
        ["vblendmps"        , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 65 /r"    , "AVX512_F-VL"],
        ["vblendmps"        , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 65 /r"    , "AVX512_F"],
        ["vblendmq"         , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 64 /r"    , "AVX512_F-VL"],
        ["vblendmq"         , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 64 /r"    , "AVX512_F-VL"],
        ["vblendmq"         , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 64 /r"    , "AVX512_F"],
        ["vblendmw"         , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.W1 66 /r"    , "AVX512_BW-VL"],
        ["vblendmw"         , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.W1 66 /r"    , "AVX512_BW-VL"],
        ["vblendmw"         , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.W1 66 /r"    , "AVX512_BW"],
        ["vbroadcastf32x2"  , "W:ymm {kz}, xmm[63:0]/m64"                       , "RM-T2"   , "EVEX.256.66.0F38.W0 19 /r"    , "AVX512_DQ-VL"],
        ["vbroadcastf32x2"  , "W:zmm {kz}, xmm[63:0]/m64"                       , "RM-T2"   , "EVEX.512.66.0F38.W0 19 /r"    , "AVX512_DQ"],
        ["vbroadcastf32x4"  , "W:ymm {kz}, m128"                                , "RM-T4"   , "EVEX.256.66.0F38.W0 1A /r"    , "AVX512_F"],
        ["vbroadcastf32x4"  , "W:zmm {kz}, m128"                                , "RM-T4"   , "EVEX.512.66.0F38.W0 1A /r"    , "AVX512_F"],
        ["vbroadcastf32x8"  , "W:zmm {kz}, m256"                                , "RM-T8"   , "EVEX.512.66.0F38.W0 1B /r"    , "AVX512_DQ"],
        ["vbroadcastf64x2"  , "W:ymm {kz}, m128"                                , "RM-T2"   , "EVEX.256.66.0F38.W1 1A /r"    , "AVX512_DQ-VL"],
        ["vbroadcastf64x2"  , "W:zmm {kz}, m128"                                , "RM-T2"   , "EVEX.512.66.0F38.W1 1A /r"    , "AVX512_DQ"],
        ["vbroadcastf64x4"  , "W:zmm {kz}, m256"                                , "RM-T4"   , "EVEX.512.66.0F38.W1 1B /r"    , "AVX512_F"],
        ["vbroadcasti32x2"  , "W:xmm {kz}, xmm[63:0]/m64"                       , "RM-T2"   , "EVEX.128.66.0F38.W0 59 /r"    , "AVX512_DQ-VL"],
        ["vbroadcasti32x2"  , "W:ymm {kz}, xmm[63:0]/m64"                       , "RM-T2"   , "EVEX.256.66.0F38.W0 59 /r"    , "AVX512_DQ-VL"],
        ["vbroadcasti32x2"  , "W:zmm {kz}, xmm[63:0]/m64"                       , "RM-T2"   , "EVEX.512.66.0F38.W0 59 /r"    , "AVX512_DQ"],
        ["vbroadcasti32x4"  , "W:ymm {kz}, m128"                                , "RM-T4"   , "EVEX.256.66.0F38.W0 5A /r"    , "AVX512_F-VL"],
        ["vbroadcasti32x4"  , "W:zmm {kz}, m128"                                , "RM-T4"   , "EVEX.512.66.0F38.W0 5A /r"    , "AVX512_F"],
        ["vbroadcasti32x8"  , "W:zmm {kz}, m256"                                , "RM-T8"   , "EVEX.512.66.0F38.W0 5B /r"    , "AVX512_DQ"],
        ["vbroadcasti64x2"  , "W:ymm {kz}, m128"                                , "RM-T2"   , "EVEX.256.66.0F38.W1 5A /r"    , "AVX512_DQ-VL"],
        ["vbroadcasti64x2"  , "W:zmm {kz}, m128"                                , "RM-T2"   , "EVEX.512.66.0F38.W1 5A /r"    , "AVX512_DQ"],
        ["vbroadcasti64x4"  , "W:zmm {kz}, m256"                                , "RM-T4"   , "EVEX.512.66.0F38.W1 5B /r"    , "AVX512_F"],
        ["vbroadcastsd"     , "W:ymm {kz}, xmm[63:0]/m64"                       , "RM-T1S"  , "EVEX.256.66.0F38.W1 19 /r"    , "AVX512_F-VL"],
        ["vbroadcastsd"     , "W:zmm {kz}, xmm[63:0]/m64"                       , "RM-T1S"  , "EVEX.512.66.0F38.W1 19 /r"    , "AVX512_F"],
        ["vbroadcastss"     , "W:xmm {kz}, xmm[31:0]/m32"                       , "RM-T1S"  , "EVEX.128.66.0F38.W0 18 /r"    , "AVX512_F-VL"],
        ["vbroadcastss"     , "W:ymm {kz}, xmm[31:0]/m32"                       , "RM-T1S"  , "EVEX.256.66.0F38.W0 18 /r"    , "AVX512_F-VL"],
        ["vbroadcastss"     , "W:zmm {kz}, xmm[31:0]/m32"                       , "RM-T1S"  , "EVEX.512.66.0F38.W0 18 /r"    , "AVX512_F"],
        ["vcmppd"           , "W:k {kz}, xmm, xmm/m128/b64, ib/ub"              , "RVMI-FV" , "EVEX.128.66.0F.W1 C2 /r ib"   , "AVX512_F-VL"],
        ["vcmppd"           , "W:k {kz}, ymm, ymm/m256/b64, ib/ub"              , "RVMI-FV" , "EVEX.256.66.0F.W1 C2 /r ib"   , "AVX512_F-VL"],
        ["vcmppd"           , "W:k {kz}, zmm, zmm/m512/b64, ib/ub {sae}"        , "RVMI-FV" , "EVEX.512.66.0F.W1 C2 /r ib"   , "AVX512_F"],
        ["vcmpps"           , "W:k {kz}, xmm, xmm/m128/b32, ib/ub"              , "RVMI-FV" , "EVEX.128.0F.W0 C2 /r ib"      , "AVX512_F-VL"],
        ["vcmpps"           , "W:k {kz}, ymm, ymm/m256/b32, ib/ub"              , "RVMI-FV" , "EVEX.256.0F.W0 C2 /r ib"      , "AVX512_F-VL"],
        ["vcmpps"           , "W:k {kz}, zmm, zmm/m512/b32, ib/ub {sae}"        , "RVMI-FV" , "EVEX.512.0F.W0 C2 /r ib"      , "AVX512_F"],
        ["vcmpsd"           , "W:k {kz}, xmm, xmm[63:0]/m64, ib/ub {sae}"       , "RVMI-T1S", "EVEX.LIG.F2.0F.W1 C2 /r ib"   , "AVX512_F"],
        ["vcmpss"           , "W:k {kz}, xmm, xmm[31:0]/m32, ib/ub {sae}"       , "RVMI-T1S", "EVEX.LIG.F3.0F.W0 C2 /r ib"   , "AVX512_F"],
        ["vcomisd"          , "R:xmm[63:0], xmm[63:0]/m64 {sae}"                , "RM-T1S"  , "EVEX.LIG.66.0F.W1 2F /r"      , "AVX512_F OF=0 SF=0 ZF=W AF=0 PF=W CF=W"],
        ["vcomiss"          , "R:xmm[31:0], xmm[31:0]/m32 {sae}"                , "RM-T1S"  , "EVEX.LIG.0F.W0 2F /r"         , "AVX512_F OF=0 SF=0 ZF=W AF=0 PF=W CF=W"],
        ["vcompresspd"      , "W:xmm/m128 {kz}, xmm"                            , "MR-T1S"  , "EVEX.128.66.0F38.W1 8A /r"    , "AVX512_F-VL"],
        ["vcompresspd"      , "W:ymm/m256 {kz}, ymm"                            , "MR-T1S"  , "EVEX.256.66.0F38.W1 8A /r"    , "AVX512_F-VL"],
        ["vcompresspd"      , "W:zmm/m512 {kz}, zmm"                            , "MR-T1S"  , "EVEX.512.66.0F38.W1 8A /r"    , "AVX512_F"],
        ["vcompressps"      , "W:xmm/m128 {kz}, xmm"                            , "MR-T1S"  , "EVEX.128.66.0F38.W0 8A /r"    , "AVX512_F-VL"],
        ["vcompressps"      , "W:ymm/m256 {kz}, ymm"                            , "MR-T1S"  , "EVEX.256.66.0F38.W0 8A /r"    , "AVX512_F-VL"],
        ["vcompressps"      , "W:zmm/m512 {kz}, zmm"                            , "MR-T1S"  , "EVEX.512.66.0F38.W0 8A /r"    , "AVX512_F"],
        ["vcvtdq2pd"        , "W:xmm {kz}, xmm[63:0]/m64/b32"                   , "RM-HV"   , "EVEX.128.F3.0F.W0 E6 /r"      , "AVX512_F-VL"],
        ["vcvtdq2pd"        , "W:ymm {kz}, xmm/m128/b32"                        , "RM-HV"   , "EVEX.256.F3.0F.W0 E6 /r"      , "AVX512_F-VL"],
        ["vcvtdq2pd"        , "W:zmm {kz}, ymm/m256/b32"                        , "RM-HV"   , "EVEX.512.F3.0F.W0 E6 /r"      , "AVX512_F"],
        ["vcvtdq2ps"        , "W:xmm {kz}, xmm/m128/b32"                        , "RM-FV"   , "EVEX.128.0F.W0 5B /r"         , "AVX512_F-VL"],
        ["vcvtdq2ps"        , "W:ymm {kz}, ymm/m256/b32"                        , "RM-FV"   , "EVEX.256.0F.W0 5B /r"         , "AVX512_F-VL"],
        ["vcvtdq2ps"        , "W:zmm {kz}, zmm/m512/b32 {er}"                   , "RM-FV"   , "EVEX.512.0F.W0 5B /r"         , "AVX512_F"],
        ["vcvtne2ps2bf16"   , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.F2.0F38.W0 72 /r"    , "AVX512_BF16-VL"],
        ["vcvtne2ps2bf16"   , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.F2.0F38.W0 72 /r"    , "AVX512_BF16-VL"],
        ["vcvtne2ps2bf16"   , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.F2.0F38.W0 72 /r"    , "AVX512_BF16-VL"],
        ["vcvtneps2bf16"    , "W:xmm {kz}, xmm/m128/b32"                        , "RM-FV"   , "EVEX.128.F3.0F38.W0 72 /r"    , "AVX512_BF16-VL"],
        ["vcvtneps2bf16"    , "W:xmm {kz}, ymm/m256/b32"                        , "RM-FV"   , "EVEX.256.F3.0F38.W0 72 /r"    , "AVX512_BF16-VL"],
        ["vcvtneps2bf16"    , "W:ymm {kz}, zmm/m512/b32"                        , "RM-FV"   , "EVEX.512.F3.0F38.W0 72 /r"    , "AVX512_BF16-VL"],
        ["vcvtpd2ps"        , "W:xmm[63:0] {kz}, xmm/m128/b64"                  , "RM-FV"   , "EVEX.128.66.0F.W1 5A /r"      , "AVX512_F-VL"],
        ["vcvtpd2ps"        , "W:xmm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.66.0F.W1 5A /r"      , "AVX512_F-VL"],
        ["vcvtpd2ps"        , "W:ymm {kz}, zmm/m512/b64 {er}"                   , "RM-FV"   , "EVEX.512.66.0F.W1 5A /r"      , "AVX512_F"],
        ["vcvtpd2dq"        , "W:xmm[63:0] {kz}, xmm/m128/b64"                  , "RM-FV"   , "EVEX.128.F2.0F.W1 E6 /r"      , "AVX512_F-VL"],
        ["vcvtpd2dq"        , "W:xmm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.F2.0F.W1 E6 /r"      , "AVX512_F-VL"],
        ["vcvtpd2dq"        , "W:ymm {kz}, zmm/m512/b64 {er}"                   , "RM-FV"   , "EVEX.512.F2.0F.W1 E6 /r"      , "AVX512_F"],
        ["vcvtpd2qq"        , "W:xmm {kz}, xmm/m128/b64"                        , "RM-FV"   , "EVEX.128.66.0F.W1 7B /r"      , "AVX512_DQ-VL"],
        ["vcvtpd2qq"        , "W:ymm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.66.0F.W1 7B /r"      , "AVX512_DQ-VL"],
        ["vcvtpd2qq"        , "W:zmm {kz}, zmm/m512/b64 {er}"                   , "RM-FV"   , "EVEX.512.66.0F.W1 7B /r"      , "AVX512_DQ"],
        ["vcvtpd2udq"       , "W:xmm[63:0] {kz}, xmm/m128/b64"                  , "RM-FV"   , "EVEX.128.0F.W1 79 /r"         , "AVX512_F-VL"],
        ["vcvtpd2udq"       , "W:xmm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.0F.W1 79 /r"         , "AVX512_F-VL"],
        ["vcvtpd2udq"       , "W:ymm {kz}, zmm/m512/b64 {er}"                   , "RM-FV"   , "EVEX.512.0F.W1 79 /r"         , "AVX512_F"],
        ["vcvtpd2uqq"       , "W:xmm {kz}, xmm/m128/b64"                        , "RM-FV"   , "EVEX.128.66.0F.W1 79 /r"      , "AVX512_DQ-VL"],
        ["vcvtpd2uqq"       , "W:ymm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.66.0F.W1 79 /r"      , "AVX512_DQ-VL"],
        ["vcvtpd2uqq"       , "W:zmm {kz}, zmm/m512/b64 {er}"                   , "RM-FV"   , "EVEX.512.66.0F.W1 79 /r"      , "AVX512_DQ"],
        ["vcvtph2ps"        , "W:xmm {kz}, xmm[63:0]/m64"                       , "RM-HVM"  , "EVEX.128.66.0F38.W0 13 /r"    , "AVX512_F-VL"],
        ["vcvtph2ps"        , "W:ymm {kz}, xmm/m128"                            , "RM-HVM"  , "EVEX.256.66.0F38.W0 13 /r"    , "AVX512_F-VL"],
        ["vcvtph2ps"        , "W:zmm {kz}, ymm/m256 {sae}"                      , "RM-HVM"  , "EVEX.512.66.0F38.W0 13 /r"    , "AVX512_F"],
        ["vcvtps2dq"        , "W:xmm {kz}, xmm/m128/b32"                        , "RM-FV"   , "EVEX.128.66.0F.W0 5B /r"      , "AVX512_F-VL"],
        ["vcvtps2dq"        , "W:ymm {kz}, ymm/m256/b32"                        , "RM-FV"   , "EVEX.256.66.0F.W0 5B /r"      , "AVX512_F-VL"],
        ["vcvtps2dq"        , "W:zmm {kz}, zmm/m512/b32 {er}"                   , "RM-FV"   , "EVEX.512.66.0F.W0 5B /r"      , "AVX512_F"],
        ["vcvtps2pd"        , "W:xmm {kz}, xmm[63:0]/m64/b32"                   , "RM-HV"   , "EVEX.128.0F.W0 5A /r"         , "AVX512_F-VL"],
        ["vcvtps2pd"        , "W:ymm {kz}, xmm/m128/b32"                        , "RM-HV"   , "EVEX.256.0F.W0 5A /r"         , "AVX512_F-VL"],
        ["vcvtps2pd"        , "W:zmm {kz}, ymm/m256/b32 {er}"                   , "RM-HV"   , "EVEX.512.0F.W0 5A /r"         , "AVX512_F"],
        ["vcvtps2ph"        , "W:xmm[63:0]/m64  {kz}, xmm, ib/ub"               , "MRI-HVM" , "EVEX.128.66.0F3A.W0 1D /r ib" , "AVX512_F-VL"],
        ["vcvtps2ph"        , "W:xmm/m128 {kz}, ymm, ib/ub"                     , "MRI-HVM" , "EVEX.256.66.0F3A.W0 1D /r ib" , "AVX512_F-VL"],
        ["vcvtps2ph"        , "W:ymm/m256 {kz}, zmm, ib/ub {sae}"               , "MRI-HVM" , "EVEX.512.66.0F3A.W0 1D /r ib" , "AVX512_F"],
        ["vcvtps2qq"        , "W:xmm {kz}, xmm[63:0]/m64/b32"                   , "RM-HV"   , "EVEX.128.66.0F.W0 7B /r"      , "AVX512_DQ-VL"],
        ["vcvtps2qq"        , "W:ymm {kz}, xmm/m128/b32"                        , "RM-HV"   , "EVEX.256.66.0F.W0 7B /r"      , "AVX512_DQ-VL"],
        ["vcvtps2qq"        , "W:zmm {kz}, ymm/m256/b32 {er}"                   , "RM-HV"   , "EVEX.512.66.0F.W0 7B /r"      , "AVX512_DQ"],
        ["vcvtps2udq"       , "W:xmm {kz}, xmm/m128/b32"                        , "RM-FV"   , "EVEX.128.0F.W0 79 /r"         , "AVX512_F-VL"],
        ["vcvtps2udq"       , "W:ymm {kz}, ymm/m256/b32"                        , "RM-FV"   , "EVEX.256.0F.W0 79 /r"         , "AVX512_F-VL"],
        ["vcvtps2udq"       , "W:zmm {kz}, zmm/m512/b32 {er}"                   , "RM-FV"   , "EVEX.512.0F.W0 79 /r"         , "AVX512_F"],
        ["vcvtps2uqq"       , "W:xmm {kz}, xmm[63:0]/m64/b32"                   , "RM-HV"   , "EVEX.128.66.0F.W0 79 /r"      , "AVX512_DQ-VL"],
        ["vcvtps2uqq"       , "W:ymm {kz}, xmm/m128/b32"                        , "RM-HV"   , "EVEX.256.66.0F.W0 79 /r"      , "AVX512_DQ-VL"],
        ["vcvtps2uqq"       , "W:zmm {kz}, ymm/m256/b32 {er}"                   , "RM-HV"   , "EVEX.512.66.0F.W0 79 /r"      , "AVX512_DQ"],
        ["vcvtqq2pd"        , "W:xmm {kz}, xmm/m128/b64"                        , "RM-FV"   , "EVEX.128.F3.0F.W1 E6 /r"      , "AVX512_DQ-VL"],
        ["vcvtqq2pd"        , "W:ymm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.F3.0F.W1 E6 /r"      , "AVX512_DQ-VL"],
        ["vcvtqq2pd"        , "W:zmm {kz}, zmm/m512/b64 {er}"                   , "RM-FV"   , "EVEX.512.F3.0F.W1 E6 /r"      , "AVX512_DQ"],
        ["vcvtqq2ps"        , "W:xmm[63:0] {kz}, xmm/m128/b64"                  , "RM-FV"   , "EVEX.128.0F.W1 5B /r"         , "AVX512_DQ-VL"],
        ["vcvtqq2ps"        , "W:xmm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.0F.W1 5B /r"         , "AVX512_DQ-VL"],
        ["vcvtqq2ps"        , "W:ymm {kz}, zmm/m512/b64 {er}"                   , "RM-FV"   , "EVEX.512.0F.W1 5B /r"         , "AVX512_DQ"],
        ["vcvtsd2si"        , "W:r32, xmm[63:0]/m64 {er}"                       , "RM-T1F"  , "EVEX.LIG.F2.0F.W0 2D /r"      , "AVX512_F"],
        ["vcvtsd2si"        , "W:r64, xmm[63:0]/m64 {er}"                       , "RM-T1F"  , "EVEX.LIG.F2.0F.W1 2D /r"      , "AVX512_F X64"],
        ["vcvtsd2ss"        , "W:xmm {kz}, xmm[127:32], xmm[63:0]/m64 {er}"     , "RVM-T1S" , "EVEX.LIG.F2.0F.W1 5A /r"      , "AVX512_F"],
        ["vcvtsd2usi"       , "W:r32, xmm[63:0]/m64 {er}"                       , "RM-T1F"  , "EVEX.LIG.F2.0F.W0 79 /r"      , "AVX512_F"],
        ["vcvtsd2usi"       , "W:r64, xmm[63:0]/m64 {er}"                       , "RM-T1F"  , "EVEX.LIG.F2.0F.W1 79 /r"      , "AVX512_F X64"],
        ["vcvtsi2sd"        , "W:xmm, xmm[127:64], r32/m32 {er}"                , "RVM-T1S" , "EVEX.LIG.F2.0F.W0 2A /r"      , "AVX512_F"],
        ["vcvtsi2sd"        , "W:xmm, xmm[127:64], r64/m64 {er}"                , "RVM-T1S" , "EVEX.LIG.F2.0F.W1 2A /r"      , "AVX512_F X64"],
        ["vcvtsi2ss"        , "W:xmm, xmm[127:32], r32/m32 {er}"                , "RVM-T1S" , "EVEX.LIG.F3.0F.W0 2A /r"      , "AVX512_F"],
        ["vcvtsi2ss"        , "W:xmm, xmm[127:32], r64/m64 {er}"                , "RVM-T1S" , "EVEX.LIG.F3.0F.W1 2A /r"      , "AVX512_F X64"],
        ["vcvtss2sd"        , "W:xmm {kz}, xmm[127:64], xmm[31:0]/m32 {sae}"    , "RVM-T1S" , "EVEX.LIG.F3.0F.W0 5A /r"      , "AVX512_F"],
        ["vcvtss2si"        , "W:r32, xmm[31:0]/m32 {er}"                       , "RM-T1F"  , "EVEX.LIG.F3.0F.W0 2D /r"      , "AVX512_F"],
        ["vcvtss2si"        , "W:r64, xmm[31:0]/m32 {er}"                       , "RM-T1F"  , "EVEX.LIG.F3.0F.W1 2D /r"      , "AVX512_F X64"],
        ["vcvtss2usi"       , "W:r32, xmm[31:0]/m32 {er}"                       , "RM-T1F"  , "EVEX.LIG.F3.0F.W0 79 /r"      , "AVX512_F"],
        ["vcvtss2usi"       , "W:r64, xmm[31:0]/m32 {er}"                       , "RM-T1F"  , "EVEX.LIG.F3.0F.W1 79 /r"      , "AVX512_F X64"],
        ["vcvttpd2dq"       , "W:xmm[63:0] {kz}, xmm/m128/b64"                  , "RM-FV"   , "EVEX.128.66.0F.W1 E6 /r"      , "AVX512_F-VL"],
        ["vcvttpd2dq"       , "W:xmm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.66.0F.W1 E6 /r"      , "AVX512_F-VL"],
        ["vcvttpd2dq"       , "W:ymm {kz}, zmm/m512/b64 {sae}"                  , "RM-FV"   , "EVEX.512.66.0F.W1 E6 /r"      , "AVX512_F"],
        ["vcvttpd2qq"       , "W:xmm {kz}, xmm/m128/b64"                        , "RM-FV"   , "EVEX.128.66.0F.W1 7A /r"      , "AVX512_F-VL"],
        ["vcvttpd2qq"       , "W:ymm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.66.0F.W1 7A /r"      , "AVX512_F-VL"],
        ["vcvttpd2qq"       , "W:zmm {kz}, zmm/m512/b64 {sae}"                  , "RM-FV"   , "EVEX.512.66.0F.W1 7A /r"      , "AVX512_F"],
        ["vcvttpd2udq"      , "W:xmm[63:0] {kz}, xmm/m128/b64"                  , "RM-FV"   , "EVEX.128.0F.W1 78 /r"         , "AVX512_F-VL"],
        ["vcvttpd2udq"      , "W:xmm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.0F.W1 78 /r"         , "AVX512_F-VL"],
        ["vcvttpd2udq"      , "W:ymm {kz}, zmm/m512/b64 {sae}"                  , "RM-FV"   , "EVEX.512.0F.W1 78 /r"         , "AVX512_F"],
        ["vcvttpd2uqq"      , "W:xmm {kz}, xmm/m128/b64"                        , "RM-FV"   , "EVEX.128.66.0F.W1 78 /r"      , "AVX512_DQ-VL"],
        ["vcvttpd2uqq"      , "W:ymm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.66.0F.W1 78 /r"      , "AVX512_DQ-VL"],
        ["vcvttpd2uqq"      , "W:zmm {kz}, zmm/m512/b64 {sae}"                  , "RM-FV"   , "EVEX.512.66.0F.W1 78 /r"      , "AVX512_DQ"],
        ["vcvttps2dq"       , "W:xmm {kz}, xmm/m128/b32"                        , "RM-FV"   , "EVEX.128.F3.0F.W0 5B /r"      , "AVX512_F-VL"],
        ["vcvttps2dq"       , "W:ymm {kz}, ymm/m256/b32"                        , "RM-FV"   , "EVEX.256.F3.0F.W0 5B /r"      , "AVX512_F-VL"],
        ["vcvttps2dq"       , "W:zmm {kz}, zmm/m512/b32 {sae}"                  , "RM-FV"   , "EVEX.512.F3.0F.W0 5B /r"      , "AVX512_F"],
        ["vcvttps2qq"       , "W:xmm {kz}, xmm[63:0]/m64/b32"                   , "RM-HV"   , "EVEX.128.66.0F.W0 7A /r"      , "AVX512_DQ-VL"],
        ["vcvttps2qq"       , "W:ymm {kz}, xmm/m128/b32"                        , "RM-HV"   , "EVEX.256.66.0F.W0 7A /r"      , "AVX512_DQ-VL"],
        ["vcvttps2qq"       , "W:zmm {kz}, ymm/m256/b32 {sae}"                  , "RM-HV"   , "EVEX.512.66.0F.W0 7A /r"      , "AVX512_DQ"],
        ["vcvttps2udq"      , "W:xmm {kz}, xmm/m128/b32"                        , "RM-FV"   , "EVEX.128.0F.W0 78 /r"         , "AVX512_F-VL"],
        ["vcvttps2udq"      , "W:ymm {kz}, ymm/m256/b32"                        , "RM-FV"   , "EVEX.256.0F.W0 78 /r"         , "AVX512_F-VL"],
        ["vcvttps2udq"      , "W:zmm {kz}, zmm/m512/b32 {sae}"                  , "RM-FV"   , "EVEX.512.0F.W0 78 /r"         , "AVX512_F"],
        ["vcvttps2uqq"      , "W:xmm {kz}, xmm[63:0]/m64/b32"                   , "RM-HV"   , "EVEX.128.66.0F.W0 78 /r"      , "AVX512_DQ-VL"],
        ["vcvttps2uqq"      , "W:ymm {kz}, xmm/m128/b32"                        , "RM-HV"   , "EVEX.256.66.0F.W0 78 /r"      , "AVX512_DQ-VL"],
        ["vcvttps2uqq"      , "W:zmm {kz}, ymm/m256/b32 {sae}"                  , "RM-HV"   , "EVEX.512.66.0F.W0 78 /r"      , "AVX512_DQ"],
        ["vcvttsd2si"       , "W:r32, xmm[63:0]/m64 {sae}"                      , "RM-T1F"  , "EVEX.LIG.F2.0F.W0 2C /r"      , "AVX512_F"],
        ["vcvttsd2si"       , "W:r64, xmm[63:0]/m64 {sae}"                      , "RM-T1F"  , "EVEX.LIG.F2.0F.W1 2C /r"      , "AVX512_F X64"],
        ["vcvttsd2usi"      , "W:r32, xmm[63:0]/m64 {sae}"                      , "RM-T1F"  , "EVEX.LIG.F2.0F.W0 78 /r"      , "AVX512_F"],
        ["vcvttsd2usi"      , "W:r64, xmm[63:0]/m64 {sae}"                      , "RM-T1F"  , "EVEX.LIG.F2.0F.W1 78 /r"      , "AVX512_F X64"],
        ["vcvttss2si"       , "W:r32, xmm[31:0]/m32 {sae}"                      , "RM-T1F"  , "EVEX.LIG.F3.0F.W0 2C /r"      , "AVX512_F"],
        ["vcvttss2si"       , "W:r64, xmm[31:0]/m32 {sae}"                      , "RM-T1F"  , "EVEX.LIG.F3.0F.W1 2C /r"      , "AVX512_F X64"],
        ["vcvttss2usi"      , "W:r32, xmm[31:0]/m32 {sae}"                      , "RM-T1F"  , "EVEX.LIG.F3.0F.W0 78 /r"      , "AVX512_F"],
        ["vcvttss2usi"      , "W:r64, xmm[31:0]/m32 {sae}"                      , "RM-T1F"  , "EVEX.LIG.F3.0F.W1 78 /r"      , "AVX512_F X64"],
        ["vcvtudq2pd"       , "W:xmm {kz}, xmm[63:0]/m64/b32"                   , "RM-HV"   , "EVEX.128.F3.0F.W0 7A /r"      , "AVX512_F-VL"],
        ["vcvtudq2pd"       , "W:ymm {kz}, xmm/m128/b32"                        , "RM-HV"   , "EVEX.256.F3.0F.W0 7A /r"      , "AVX512_F-VL"],
        ["vcvtudq2pd"       , "W:zmm {kz}, ymm/m256/b32"                        , "RM-HV"   , "EVEX.512.F3.0F.W0 7A /r"      , "AVX512_F"],
        ["vcvtudq2ps"       , "W:xmm {kz}, xmm/m128/b32"                        , "RM-FV"   , "EVEX.128.F2.0F.W0 7A /r"      , "AVX512_F-VL"],
        ["vcvtudq2ps"       , "W:ymm {kz}, ymm/m256/b32"                        , "RM-FV"   , "EVEX.256.F2.0F.W0 7A /r"      , "AVX512_F-VL"],
        ["vcvtudq2ps"       , "W:zmm {kz}, zmm/m512/b32 {er}"                   , "RM-FV"   , "EVEX.512.F2.0F.W0 7A /r"      , "AVX512_F"],
        ["vcvtuqq2pd"       , "W:xmm {kz}, xmm/m128/b64"                        , "RM-FV"   , "EVEX.128.F3.0F.W1 7A /r"      , "AVX512_DQ-VL"],
        ["vcvtuqq2pd"       , "W:ymm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.F3.0F.W1 7A /r"      , "AVX512_DQ-VL"],
        ["vcvtuqq2pd"       , "W:zmm {kz}, zmm/m512/b64 {er}"                   , "RM-FV"   , "EVEX.512.F3.0F.W1 7A /r"      , "AVX512_DQ"],
        ["vcvtuqq2ps"       , "W:xmm[63:0] {kz}, xmm/m128/b64"                  , "RM-FV"   , "EVEX.128.F2.0F.W1 7A /r"      , "AVX512_DQ-VL"],
        ["vcvtuqq2ps"       , "W:xmm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.F2.0F.W1 7A /r"      , "AVX512_DQ-VL"],
        ["vcvtuqq2ps"       , "W:ymm {kz}, zmm/m512/b64 {er}"                   , "RM-FV"   , "EVEX.512.F2.0F.W1 7A /r"      , "AVX512_DQ"],
        ["vcvtusi2sd"       , "W:xmm, xmm[127:64], r32/m32 {er}"                , "RVM-T1S" , "EVEX.LIG.F2.0F.W0 7B /r"      , "AVX512_F"],
        ["vcvtusi2sd"       , "W:xmm, xmm[127:64], r64/m64 {er}"                , "RVM-T1S" , "EVEX.LIG.F2.0F.W1 7B /r"      , "AVX512_F X64"],
        ["vcvtusi2ss"       , "W:xmm, xmm[127:32], r32/m32 {er}"                , "RVM-T1S" , "EVEX.LIG.F3.0F.W0 7B /r"      , "AVX512_F"],
        ["vcvtusi2ss"       , "W:xmm, xmm[127:32], r64/m64 {er}"                , "RVM-T1S" , "EVEX.LIG.F3.0F.W1 7B /r"      , "AVX512_F X64"],
        ["vdbpsadbw"        , "W:xmm {kz}, xmm, xmm/m128, ib/ub"                , "RVMI-FVM", "EVEX.128.66.0F3A.W0 42 /r ib" , "AVX512_BW-VL"],
        ["vdbpsadbw"        , "W:ymm {kz}, ymm, ymm/m256, ib/ub"                , "RVMI-FVM", "EVEX.256.66.0F3A.W0 42 /r ib" , "AVX512_BW-VL"],
        ["vdbpsadbw"        , "W:zmm {kz}, zmm, zmm/m512, ib/ub"                , "RVMI-FVM", "EVEX.512.66.0F3A.W0 42 /r ib" , "AVX512_BW"],
        ["vdivpd"           , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 5E /r"      , "AVX512_F-VL"],
        ["vdivpd"           , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 5E /r"      , "AVX512_F-VL"],
        ["vdivpd"           , "W:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F.W1 5E /r"      , "AVX512_F"],
        ["vdivps"           , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.0F.W0 5E /r"         , "AVX512_F-VL"],
        ["vdivps"           , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.0F.W0 5E /r"         , "AVX512_F-VL"],
        ["vdivps"           , "W:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.0F.W0 5E /r"         , "AVX512_F"],
        ["vdivsd"           , "W:xmm {kz}, xmm, xmm[63:0]/m64 {er}"             , "RVM-T1S" , "EVEX.LIG.F2.0F.W1 5E /r"      , "AVX512_F"],
        ["vdivss"           , "W:xmm {kz}, xmm, xmm[31:0]/m32 {er}"             , "RVM-T1S" , "EVEX.LIG.F3.0F.W0 5E /r"      , "AVX512_F"],
        ["vdpbf16ps"        , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.F3.0F38.W0 52 /r"    , "AVX512_BF16-VL"],
        ["vdpbf16ps"        , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.F3.0F38.W0 52 /r"    , "AVX512_BF16-VL"],
        ["vdpbf16ps"        , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.F3.0F38.W0 52 /r"    , "AVX512_BF16-VL"],
        ["vexp2pd"          , "W:zmm {kz}, zmm/m512/b64 {sae}"                  , "RM-FV"   , "EVEX.512.66.0F38.W1 C8 /r"    , "AVX512_ERI"],
        ["vexp2ps"          , "W:zmm {kz}, zmm/m512/b32 {sae}"                  , "RM-FV"   , "EVEX.512.66.0F38.W0 C8 /r"    , "AVX512_ERI"],
        ["vexpandpd"        , "W:xmm {kz}, xmm/m128"                            , "RM-T1S"  , "EVEX.128.66.0F38.W1 88 /r"    , "AVX512_F-VL"],
        ["vexpandpd"        , "W:ymm {kz}, ymm/m256"                            , "RM-T1S"  , "EVEX.256.66.0F38.W1 88 /r"    , "AVX512_F-VL"],
        ["vexpandpd"        , "W:zmm {kz}, zmm/m512"                            , "RM-T1S"  , "EVEX.512.66.0F38.W1 88 /r"    , "AVX512_F"],
        ["vexpandps"        , "W:xmm {kz}, xmm/m128"                            , "RM-T1S"  , "EVEX.128.66.0F38.W0 88 /r"    , "AVX512_F-VL"],
        ["vexpandps"        , "W:ymm {kz}, ymm/m256"                            , "RM-T1S"  , "EVEX.256.66.0F38.W0 88 /r"    , "AVX512_F-VL"],
        ["vexpandps"        , "W:zmm {kz}, zmm/m512"                            , "RM-T1S"  , "EVEX.512.66.0F38.W0 88 /r"    , "AVX512_F"],
        ["vextractf32x4"    , "W:xmm/m128 {kz}, ymm, ib/ub"                     , "MRI-T4"  , "EVEX.256.66.0F3A.W0 19 /r ib" , "AVX512_F-VL"],
        ["vextractf32x4"    , "W:xmm/m128 {kz}, zmm, ib/ub"                     , "MRI-T4"  , "EVEX.512.66.0F3A.W0 19 /r ib" , "AVX512_F"],
        ["vextractf32x8"    , "W:ymm/m256 {kz}, zmm, ib/ub"                     , "MRI-T8"  , "EVEX.512.66.0F3A.W0 1B /r ib" , "AVX512_DQ"],
        ["vextractf64x2"    , "W:xmm/m128 {kz}, ymm, ib/ub"                     , "MRI-T2"  , "EVEX.256.66.0F3A.W1 19 /r ib" , "AVX512_DQ-VL"],
        ["vextractf64x2"    , "W:xmm/m128 {kz}, zmm, ib/ub"                     , "MRI-T2"  , "EVEX.512.66.0F3A.W1 19 /r ib" , "AVX512_DQ"],
        ["vextractf64x4"    , "W:ymm/m256 {kz}, zmm, ib/ub"                     , "MRI-T4"  , "EVEX.512.66.0F3A.W1 1B /r ib" , "AVX512_F"],
        ["vextracti32x4"    , "W:xmm/m128 {kz}, ymm, ib/ub"                     , "MRI-T4"  , "EVEX.256.66.0F3A.W0 39 /r ib" , "AVX512_F-VL"],
        ["vextracti32x4"    , "W:xmm/m128 {kz}, zmm, ib/ub"                     , "MRI-T4"  , "EVEX.512.66.0F3A.W0 39 /r ib" , "AVX512_F"],
        ["vextracti32x8"    , "W:ymm/m256 {kz}, zmm, ib/ub"                     , "MRI-T8"  , "EVEX.512.66.0F3A.W0 3B /r ib" , "AVX512_DQ"],
        ["vextracti64x2"    , "W:xmm/m128 {kz}, ymm, ib/ub"                     , "MRI-T2"  , "EVEX.256.66.0F3A.W1 39 /r ib" , "AVX512_DQ-VL"],
        ["vextracti64x2"    , "W:xmm/m128 {kz}, zmm, ib/ub"                     , "MRI-T2"  , "EVEX.512.66.0F3A.W1 39 /r ib" , "AVX512_DQ"],
        ["vextracti64x4"    , "W:ymm/m256 {kz}, zmm, ib/ub"                     , "MRI-T4"  , "EVEX.512.66.0F3A.W1 3B /r ib" , "AVX512_F"],
        ["vextractps"       , "W:r32[31:0]/m32, xmm, ib/ub"                     , "MRI-T1S" , "EVEX.128.66.0F3A.WIG 17 /r ib", "AVX512_F"],
        ["vfixupimmpd"      , "X:xmm {kz}, xmm, xmm/m128/b64, ib/ub"            , "RVMI-FV" , "EVEX.128.66.0F3A.W1 54 /r ib" , "AVX512_F-VL"],
        ["vfixupimmpd"      , "X:ymm {kz}, ymm, ymm/m256/b64, ib/ub"            , "RVMI-FV" , "EVEX.256.66.0F3A.W1 54 /r ib" , "AVX512_F-VL"],
        ["vfixupimmpd"      , "X:zmm {kz}, zmm, zmm/m512/b64, ib/ub {sae}"      , "RVMI-FV" , "EVEX.512.66.0F3A.W1 54 /r ib" , "AVX512_F"],
        ["vfixupimmps"      , "X:xmm {kz}, xmm, xmm/m128/b32, ib/ub"            , "RVMI-FV" , "EVEX.128.66.0F3A.W0 54 /r ib" , "AVX512_F-VL"],
        ["vfixupimmps"      , "X:ymm {kz}, ymm, ymm/m256/b32, ib/ub"            , "RVMI-FV" , "EVEX.256.66.0F3A.W0 54 /r ib" , "AVX512_F-VL"],
        ["vfixupimmps"      , "X:zmm {kz}, zmm, zmm/m512/b32, ib/ub {sae}"      , "RVMI-FV" , "EVEX.512.66.0F3A.W0 54 /r ib" , "AVX512_F"],
        ["vfixupimmsd"      , "X:xmm {kz},xmm[127:64],xmm[63:0]/m64,ib/ub {sae}", "RVMI-T1S", "EVEX.LIG.66.0F3A.W1 55 /r ib" , "AVX512_F"],
        ["vfixupimmss"      , "X:xmm {kz},xmm[127:32],xmm[31:0]/m32,ib/ub {sae}", "RVMI-T1S", "EVEX.LIG.66.0F3A.W0 55 /r ib" , "AVX512_F"],
        ["vfmadd132pd"      , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 98 /r"    , "AVX512_F-VL"],
        ["vfmadd132pd"      , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 98 /r"    , "AVX512_F-VL"],
        ["vfmadd132pd"      , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 98 /r"    , "AVX512_F"],
        ["vfmadd132ps"      , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 98 /r"    , "AVX512_F-VL"],
        ["vfmadd132ps"      , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 98 /r"    , "AVX512_F-VL"],
        ["vfmadd132ps"      , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 98 /r"    , "AVX512_F"],
        ["vfmadd132sd"      , "x:xmm[63:0] {kz}, xmm[63:0], xmm[63:0]/m64 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 99 /r"    , "AVX512_F"],
        ["vfmadd132ss"      , "x:xmm[31:0] {kz}, xmm[31:0], xmm[31:0]/m32 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 99 /r"    , "AVX512_F"],
        ["vfmadd213pd"      , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 A8 /r"    , "AVX512_F-VL"],
        ["vfmadd213pd"      , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 A8 /r"    , "AVX512_F-VL"],
        ["vfmadd213pd"      , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 A8 /r"    , "AVX512_F"],
        ["vfmadd213ps"      , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 A8 /r"    , "AVX512_F-VL"],
        ["vfmadd213ps"      , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 A8 /r"    , "AVX512_F-VL"],
        ["vfmadd213ps"      , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 A8 /r"    , "AVX512_F"],
        ["vfmadd213sd"      , "x:xmm[63:0] {kz}, xmm[63:0], xmm[63:0]/m64 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 A9 /r"    , "AVX512_F"],
        ["vfmadd213ss"      , "x:xmm[31:0] {kz}, xmm[31:0], xmm[31:0]/m32 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 A9 /r"    , "AVX512_F"],
        ["vfmadd231pd"      , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 B8 /r"    , "AVX512_F-VL"],
        ["vfmadd231pd"      , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 B8 /r"    , "AVX512_F-VL"],
        ["vfmadd231pd"      , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 B8 /r"    , "AVX512_F"],
        ["vfmadd231ps"      , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 B8 /r"    , "AVX512_F-VL"],
        ["vfmadd231ps"      , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 B8 /r"    , "AVX512_F-VL"],
        ["vfmadd231ps"      , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 B8 /r"    , "AVX512_F"],
        ["vfmadd231sd"      , "x:xmm[63:0] {kz}, xmm[63:0], xmm[63:0]/m64 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 B9 /r"    , "AVX512_F"],
        ["vfmadd231ss"      , "x:xmm[31:0] {kz}, xmm[31:0], xmm[31:0]/m32 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 B9 /r"    , "AVX512_F"],
        ["vfmaddsub132pd"   , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 96 /r"    , "AVX512_F-VL"],
        ["vfmaddsub132pd"   , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 96 /r"    , "AVX512_F-VL"],
        ["vfmaddsub132pd"   , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 96 /r"    , "AVX512_F"],
        ["vfmaddsub132ps"   , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 96 /r"    , "AVX512_F-VL"],
        ["vfmaddsub132ps"   , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 96 /r"    , "AVX512_F-VL"],
        ["vfmaddsub132ps"   , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 96 /r"    , "AVX512_F"],
        ["vfmaddsub213pd"   , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 A6 /r"    , "AVX512_F-VL"],
        ["vfmaddsub213pd"   , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 A6 /r"    , "AVX512_F-VL"],
        ["vfmaddsub213pd"   , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 A6 /r"    , "AVX512_F"],
        ["vfmaddsub213ps"   , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 A6 /r"    , "AVX512_F-VL"],
        ["vfmaddsub213ps"   , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 A6 /r"    , "AVX512_F-VL"],
        ["vfmaddsub213ps"   , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 A6 /r"    , "AVX512_F"],
        ["vfmaddsub231pd"   , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 B6 /r"    , "AVX512_F-VL"],
        ["vfmaddsub231pd"   , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 B6 /r"    , "AVX512_F-VL"],
        ["vfmaddsub231pd"   , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 B6 /r"    , "AVX512_F"],
        ["vfmaddsub231ps"   , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 B6 /r"    , "AVX512_F-VL"],
        ["vfmaddsub231ps"   , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 B6 /r"    , "AVX512_F-VL"],
        ["vfmaddsub231ps"   , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 B6 /r"    , "AVX512_F"],
        ["vfmsub132pd"      , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 9A /r"    , "AVX512_F-VL"],
        ["vfmsub132pd"      , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 9A /r"    , "AVX512_F-VL"],
        ["vfmsub132pd"      , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 9A /r"    , "AVX512_F"],
        ["vfmsub132ps"      , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 9A /r"    , "AVX512_F-VL"],
        ["vfmsub132ps"      , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 9A /r"    , "AVX512_F-VL"],
        ["vfmsub132ps"      , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 9A /r"    , "AVX512_F"],
        ["vfmsub132sd"      , "x:xmm[63:0] {kz}, xmm[63:0], xmm[63:0]/m64 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 9B /r"    , "AVX512_F"],
        ["vfmsub132ss"      , "x:xmm[31:0] {kz}, xmm[31:0], xmm[31:0]/m32 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 9B /r"    , "AVX512_F"],
        ["vfmsub213pd"      , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 AA /r"    , "AVX512_F-VL"],
        ["vfmsub213pd"      , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 AA /r"    , "AVX512_F-VL"],
        ["vfmsub213pd"      , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 AA /r"    , "AVX512_F"],
        ["vfmsub213ps"      , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 AA /r"    , "AVX512_F-VL"],
        ["vfmsub213ps"      , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 AA /r"    , "AVX512_F-VL"],
        ["vfmsub213ps"      , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 AA /r"    , "AVX512_F"],
        ["vfmsub213sd"      , "x:xmm[63:0] {kz}, xmm[63:0], xmm[63:0]/m64 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 AB /r"    , "AVX512_F"],
        ["vfmsub213ss"      , "x:xmm[31:0] {kz}, xmm[31:0], xmm[31:0]/m32 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 AB /r"    , "AVX512_F"],
        ["vfmsub231pd"      , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 BA /r"    , "AVX512_F-VL"],
        ["vfmsub231pd"      , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 BA /r"    , "AVX512_F-VL"],
        ["vfmsub231pd"      , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 BA /r"    , "AVX512_F"],
        ["vfmsub231ps"      , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 BA /r"    , "AVX512_F-VL"],
        ["vfmsub231ps"      , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 BA /r"    , "AVX512_F-VL"],
        ["vfmsub231ps"      , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 BA /r"    , "AVX512_F"],
        ["vfmsub231sd"      , "x:xmm[63:0] {kz}, xmm[63:0], xmm[63:0]/m64 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 BB /r"    , "AVX512_F"],
        ["vfmsub231ss"      , "x:xmm[31:0] {kz}, xmm[31:0], xmm[31:0]/m32 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 BB /r"    , "AVX512_F"],
        ["vfmsubadd132pd"   , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 97 /r"    , "AVX512_F-VL"],
        ["vfmsubadd132pd"   , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 97 /r"    , "AVX512_F-VL"],
        ["vfmsubadd132pd"   , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 97 /r"    , "AVX512_F"],
        ["vfmsubadd132ps"   , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 97 /r"    , "AVX512_F-VL"],
        ["vfmsubadd132ps"   , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 97 /r"    , "AVX512_F-VL"],
        ["vfmsubadd132ps"   , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 97 /r"    , "AVX512_F"],
        ["vfmsubadd213pd"   , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 A7 /r"    , "AVX512_F-VL"],
        ["vfmsubadd213pd"   , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 A7 /r"    , "AVX512_F-VL"],
        ["vfmsubadd213pd"   , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 A7 /r"    , "AVX512_F"],
        ["vfmsubadd213ps"   , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 A7 /r"    , "AVX512_F-VL"],
        ["vfmsubadd213ps"   , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 A7 /r"    , "AVX512_F-VL"],
        ["vfmsubadd213ps"   , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 A7 /r"    , "AVX512_F"],
        ["vfmsubadd231pd"   , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 B7 /r"    , "AVX512_F-VL"],
        ["vfmsubadd231pd"   , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 B7 /r"    , "AVX512_F-VL"],
        ["vfmsubadd231pd"   , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 B7 /r"    , "AVX512_F"],
        ["vfmsubadd231ps"   , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 B7 /r"    , "AVX512_F-VL"],
        ["vfmsubadd231ps"   , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 B7 /r"    , "AVX512_F-VL"],
        ["vfmsubadd231ps"   , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 B7 /r"    , "AVX512_F"],
        ["vfnmadd132pd"     , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 9C /r"    , "AVX512_F-VL"],
        ["vfnmadd132pd"     , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 9C /r"    , "AVX512_F-VL"],
        ["vfnmadd132pd"     , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 9C /r"    , "AVX512_F"],
        ["vfnmadd132ps"     , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 9C /r"    , "AVX512_F-VL"],
        ["vfnmadd132ps"     , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 9C /r"    , "AVX512_F-VL"],
        ["vfnmadd132ps"     , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 9C /r"    , "AVX512_F"],
        ["vfnmadd132sd"     , "x:xmm[63:0] {kz}, xmm[63:0], xmm[63:0]/m64 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 9D /r"    , "AVX512_F"],
        ["vfnmadd132ss"     , "x:xmm[31:0] {kz}, xmm[31:0], xmm[31:0]/m32 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 9D /r"    , "AVX512_F"],
        ["vfnmadd213pd"     , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 AC /r"    , "AVX512_F-VL"],
        ["vfnmadd213pd"     , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 AC /r"    , "AVX512_F-VL"],
        ["vfnmadd213pd"     , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 AC /r"    , "AVX512_F"],
        ["vfnmadd213ps"     , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 AC /r"    , "AVX512_F-VL"],
        ["vfnmadd213ps"     , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 AC /r"    , "AVX512_F-VL"],
        ["vfnmadd213ps"     , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 AC /r"    , "AVX512_F"],
        ["vfnmadd213sd"     , "x:xmm[63:0] {kz}, xmm[63:0], xmm[63:0]/m64 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 AD /r"    , "AVX512_F"],
        ["vfnmadd213ss"     , "x:xmm[31:0] {kz}, xmm[31:0], xmm[31:0]/m32 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 AD /r"    , "AVX512_F"],
        ["vfnmadd231pd"     , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 BC /r"    , "AVX512_F-VL"],
        ["vfnmadd231pd"     , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 BC /r"    , "AVX512_F-VL"],
        ["vfnmadd231pd"     , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 BC /r"    , "AVX512_F"],
        ["vfnmadd231ps"     , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 BC /r"    , "AVX512_F-VL"],
        ["vfnmadd231ps"     , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 BC /r"    , "AVX512_F-VL"],
        ["vfnmadd231ps"     , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 BC /r"    , "AVX512_F"],
        ["vfnmadd231sd"     , "x:xmm[63:0] {kz}, xmm[63:0], xmm[63:0]/m64 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 BD /r"    , "AVX512_F"],
        ["vfnmadd231ss"     , "x:xmm[31:0] {kz}, xmm[31:0], xmm[31:0]/m32 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 BD /r"    , "AVX512_F"],
        ["vfnmsub132pd"     , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 9E /r"    , "AVX512_F-VL"],
        ["vfnmsub132pd"     , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 9E /r"    , "AVX512_F-VL"],
        ["vfnmsub132pd"     , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 9E /r"    , "AVX512_F"],
        ["vfnmsub132ps"     , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 9E /r"    , "AVX512_F-VL"],
        ["vfnmsub132ps"     , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 9E /r"    , "AVX512_F-VL"],
        ["vfnmsub132ps"     , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 9E /r"    , "AVX512_F"],
        ["vfnmsub132sd"     , "x:xmm[63:0] {kz}, xmm[63:0], xmm[63:0]/m64 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 9F /r"    , "AVX512_F"],
        ["vfnmsub132ss"     , "x:xmm[31:0] {kz}, xmm[31:0], xmm[31:0]/m32 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 9F /r"    , "AVX512_F"],
        ["vfnmsub213pd"     , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 AE /r"    , "AVX512_F-VL"],
        ["vfnmsub213pd"     , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 AE /r"    , "AVX512_F-VL"],
        ["vfnmsub213pd"     , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 AE /r"    , "AVX512_F"],
        ["vfnmsub213ps"     , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 AE /r"    , "AVX512_F-VL"],
        ["vfnmsub213ps"     , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 AE /r"    , "AVX512_F-VL"],
        ["vfnmsub213ps"     , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 AE /r"    , "AVX512_F"],
        ["vfnmsub213sd"     , "x:xmm[63:0] {kz}, xmm[63:0], xmm[63:0]/m64 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 AF /r"    , "AVX512_F"],
        ["vfnmsub213ss"     , "x:xmm[31:0] {kz}, xmm[31:0], xmm[31:0]/m32 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 AF /r"    , "AVX512_F"],
        ["vfnmsub231pd"     , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 BE /r"    , "AVX512_F-VL"],
        ["vfnmsub231pd"     , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 BE /r"    , "AVX512_F-VL"],
        ["vfnmsub231pd"     , "X:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 BE /r"    , "AVX512_F"],
        ["vfnmsub231ps"     , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 BE /r"    , "AVX512_F-VL"],
        ["vfnmsub231ps"     , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 BE /r"    , "AVX512_F-VL"],
        ["vfnmsub231ps"     , "X:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 BE /r"    , "AVX512_F"],
        ["vfnmsub231sd"     , "X:xmm[63:0] {kz}, xmm[63:0], xmm[63:0]/m64 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 BF /r"    , "AVX512_F"],
        ["vfnmsub231ss"     , "X:xmm[31:0] {kz}, xmm[31:0], xmm[31:0]/m32 {er}" , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 BF /r"    , "AVX512_F"],
        ["vfpclasspd"       , "W:k {k}, xmm/m128/b64, ib/ub"                    , "RMI-FV"  , "EVEX.128.66.0F3A.W1 66 /r ib" , "AVX512_DQ-VL"],
        ["vfpclasspd"       , "W:k {k}, ymm/m256/b64, ib/ub"                    , "RMI-FV"  , "EVEX.256.66.0F3A.W1 66 /r ib" , "AVX512_DQ-VL"],
        ["vfpclasspd"       , "W:k {k}, zmm/m512/b64, ib/ub"                    , "RMI-FV"  , "EVEX.512.66.0F3A.W1 66 /r ib" , "AVX512_DQ"],
        ["vfpclassps"       , "W:k {k}, xmm/m128/b32, ib/ub"                    , "RMI-FV"  , "EVEX.128.66.0F3A.W0 66 /r ib" , "AVX512_DQ-VL"],
        ["vfpclassps"       , "W:k {k}, ymm/m256/b32, ib/ub"                    , "RMI-FV"  , "EVEX.256.66.0F3A.W0 66 /r ib" , "AVX512_DQ-VL"],
        ["vfpclassps"       , "W:k {k}, zmm/m512/b32, ib/ub"                    , "RMI-FV"  , "EVEX.512.66.0F3A.W0 66 /r ib" , "AVX512_DQ"],
        ["vfpclasssd"       , "W:k {k}, xmm[63:0]/m64, ib/ub"                   , "RMI-T1S" , "EVEX.LIG.66.0F3A.W1 67 /r ib" , "AVX512_DQ"],
        ["vfpclassss"       , "W:k {k}, xmm[31:0]/m32, ib/ub"                   , "RMI-T1S" , "EVEX.LIG.66.0F3A.W0 67 /r ib" , "AVX512_DQ"],
        ["vgatherdpd"       , "X:xmm {k}, vm32x"                                , "RM-T1S"  , "EVEX.128.66.0F38.W1 92 /r"    , "AVX512_F-VL"],
        ["vgatherdpd"       , "X:ymm {k}, vm32y"                                , "RM-T1S"  , "EVEX.256.66.0F38.W1 92 /r"    , "AVX512_F-VL"],
        ["vgatherdpd"       , "X:zmm {k}, vm32z"                                , "RM-T1S"  , "EVEX.512.66.0F38.W1 92 /r"    , "AVX512_F"],
        ["vgatherdps"       , "X:xmm {k}, vm32x"                                , "RM-T1S"  , "EVEX.128.66.0F38.W0 92 /r"    , "AVX512_F-VL"],
        ["vgatherdps"       , "X:ymm {k}, vm32y"                                , "RM-T1S"  , "EVEX.256.66.0F38.W0 92 /r"    , "AVX512_F-VL"],
        ["vgatherdps"       , "X:zmm {k}, vm32z"                                , "RM-T1S"  , "EVEX.512.66.0F38.W0 92 /r"    , "AVX512_F"],
        ["vgatherpf0dpd"    , "R:vm32y {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W1 C6 /1"    , "AVX512_PFI"],
        ["vgatherpf0dps"    , "R:vm32z {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W0 C6 /1"    , "AVX512_PFI"],
        ["vgatherpf0qpd"    , "R:vm64z {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W1 C7 /1"    , "AVX512_PFI"],
        ["vgatherpf0qps"    , "R:vm64z {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W0 C7 /1"    , "AVX512_PFI"],
        ["vgatherpf1dpd"    , "R:vm32y {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W1 C6 /2"    , "AVX512_PFI"],
        ["vgatherpf1dps"    , "R:vm32z {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W0 C6 /2"    , "AVX512_PFI"],
        ["vgatherpf1qpd"    , "R:vm64z {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W1 C7 /2"    , "AVX512_PFI"],
        ["vgatherpf1qps"    , "R:vm64z {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W0 C7 /2"    , "AVX512_PFI"],
        ["vgatherqpd"       , "X:xmm {k}, vm64x"                                , "RM-T1S"  , "EVEX.128.66.0F38.W1 93 /r"    , "AVX512_F-VL"],
        ["vgatherqpd"       , "X:ymm {k}, vm64y"                                , "RM-T1S"  , "EVEX.256.66.0F38.W1 93 /r"    , "AVX512_F-VL"],
        ["vgatherqpd"       , "X:zmm {k}, vm64z"                                , "RM-T1S"  , "EVEX.512.66.0F38.W1 93 /r"    , "AVX512_F"],
        ["vgatherqps"       , "X:xmm {k}, vm64x"                                , "RM-T1S"  , "EVEX.128.66.0F38.W0 93 /r"    , "AVX512_F-VL"],
        ["vgatherqps"       , "X:ymm {k}, vm64y"                                , "RM-T1S"  , "EVEX.256.66.0F38.W0 93 /r"    , "AVX512_F-VL"],
        ["vgatherqps"       , "X:zmm {k}, vm64z"                                , "RM-T1S"  , "EVEX.512.66.0F38.W0 93 /r"    , "AVX512_F"],
        ["vgetexppd"        , "W:xmm {kz}, xmm/m128/b64"                        , "RM-FV"   , "EVEX.128.66.0F38.W1 42 /r"    , "AVX512_F-VL"],
        ["vgetexppd"        , "W:ymm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.66.0F38.W1 42 /r"    , "AVX512_F-VL"],
        ["vgetexppd"        , "W:zmm {kz}, zmm/m512/b64 {sae}"                  , "RM-FV"   , "EVEX.512.66.0F38.W1 42 /r"    , "AVX512_F"],
        ["vgetexpps"        , "W:xmm {kz}, xmm/m128/b32"                        , "RM-FV"   , "EVEX.128.66.0F38.W0 42 /r"    , "AVX512_F-VL"],
        ["vgetexpps"        , "W:ymm {kz}, ymm/m256/b32"                        , "RM-FV"   , "EVEX.256.66.0F38.W0 42 /r"    , "AVX512_F-VL"],
        ["vgetexpps"        , "W:zmm {kz}, zmm/m512/b32 {sae}"                  , "RM-FV"   , "EVEX.512.66.0F38.W0 42 /r"    , "AVX512_F"],
        ["vgetexpsd"        , "W:xmm {kz}, xmm[127:64], xmm[63:0]/m64 {sae}"    , "RM-T1S"  , "EVEX.LIG.66.0F38.W1 43 /r"    , "AVX512_F"],
        ["vgetexpss"        , "W:xmm {kz}, xmm[127:32], xmm[31:0]/m32 {sae}"    , "RM-T1S"  , "EVEX.LIG.66.0F38.W0 43 /r"    , "AVX512_F"],
        ["vgetmantpd"       , "W:xmm {kz}, xmm/m128/b64, ib/ub"                 , "RMI-FV"  , "EVEX.128.66.0F3A.W1 26 /r ib" , "AVX512_F-VL"],
        ["vgetmantpd"       , "W:ymm {kz}, ymm/m256/b64, ib/ub"                 , "RMI-FV"  , "EVEX.256.66.0F3A.W1 26 /r ib" , "AVX512_F-VL"],
        ["vgetmantpd"       , "W:zmm {kz}, zmm/m512/b64, ib/ub {sae}"           , "RMI-FV"  , "EVEX.512.66.0F3A.W1 26 /r ib" , "AVX512_F"],
        ["vgetmantps"       , "W:xmm {kz}, xmm/m128/b32, ib/ub"                 , "RMI-FV"  , "EVEX.128.66.0F3A.W0 26 /r ib" , "AVX512_F-VL"],
        ["vgetmantps"       , "W:ymm {kz}, ymm/m256/b32, ib/ub"                 , "RMI-FV"  , "EVEX.256.66.0F3A.W0 26 /r ib" , "AVX512_F-VL"],
        ["vgetmantps"       , "W:zmm {kz}, zmm/m512/b32, ib/ub {sae}"           , "RMI-FV"  , "EVEX.512.66.0F3A.W0 26 /r ib" , "AVX512_F"],
        ["vgetmantsd"       , "W:xmm {kz},xmm[127:64],xmm[63:0]/m64,ib/ub {sae}", "RMI-T1S" , "EVEX.LIG.66.0F3A.W1 27 /r ib" , "AVX512_F"],
        ["vgetmantss"       , "W:xmm {kz},xmm[127:32],xmm[31:0]/m32,ib/ub {sae}", "RMI-T1S" , "EVEX.LIG.66.0F3A.W0 27 /r ib" , "AVX512_F"],
        ["vgf2p8affineinvqb", "W:xmm {kz}, xmm, xmm/m128, ib/ub"                , "RVMI-FVM", "EVEX.128.66.0F3A.W1 CF /r ib" , "AVX512_F-VL GFNI"],
        ["vgf2p8affineinvqb", "W:ymm {kz}, ymm, ymm/m256, ib/ub"                , "RVMI-FVM", "EVEX.256.66.0F3A.W1 CF /r ib" , "AVX512_F-VL GFNI"],
        ["vgf2p8affineinvqb", "W:zmm {kz}, zmm, zmm/m512, ib/ub"                , "RVMI-FVM", "EVEX.512.66.0F3A.W1 CF /r ib" , "AVX512_F    GFNI"],
        ["vgf2p8affineqb"   , "W:xmm {kz}, xmm, xmm/m128, ib/ub"                , "RVMI-FVM", "EVEX.128.66.0F3A.W1 CE /r ib" , "AVX512_F-VL GFNI"],
        ["vgf2p8affineqb"   , "W:ymm {kz}, ymm, ymm/m256, ib/ub"                , "RVMI-FVM", "EVEX.256.66.0F3A.W1 CE /r ib" , "AVX512_F-VL GFNI"],
        ["vgf2p8affineqb"   , "W:zmm {kz}, zmm, zmm/m512, ib/ub"                , "RVMI-FVM", "EVEX.512.66.0F3A.W1 CE /r ib" , "AVX512_F    GFNI"],
        ["vgf2p8mulb"       , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.W0 CF /r"    , "AVX512_F-VL GFNI"],
        ["vgf2p8mulb"       , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.W0 CF /r"    , "AVX512_F-VL GFNI"],
        ["vgf2p8mulb"       , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.W0 CF /r"    , "AVX512_F    GFNI"],
        ["vinsertf32x4"     , "W:ymm {kz}, ymm, xmm/m128, ib/ub"                , "RVMI-T4" , "EVEX.256.66.0F3A.W0 18 /r ib" , "AVX512_F-VL"],
        ["vinsertf32x4"     , "W:zmm {kz}, zmm, xmm/m128, ib/ub"                , "RVMI-T4" , "EVEX.512.66.0F3A.W0 18 /r ib" , "AVX512_F"],
        ["vinsertf32x8"     , "W:zmm {kz}, zmm, ymm/m256, ib/ub"                , "RVMI-T8" , "EVEX.512.66.0F3A.W0 1A /r ib" , "AVX512_DQ"],
        ["vinsertf64x2"     , "W:ymm {kz}, ymm, xmm/m128, ib/ub"                , "RVMI-T2" , "EVEX.256.66.0F3A.W1 18 /r ib" , "AVX512_DQ-VL"],
        ["vinsertf64x2"     , "W:zmm {kz}, zmm, xmm/m128, ib/ub"                , "RVMI-T2" , "EVEX.512.66.0F3A.W1 18 /r ib" , "AVX512_DQ"],
        ["vinsertf64x4"     , "W:zmm {kz}, zmm, ymm/m256, ib/ub"                , "RVMI-T4" , "EVEX.512.66.0F3A.W1 1A /r ib" , "AVX512_F"],
        ["vinserti32x4"     , "W:ymm {kz}, ymm, xmm/m128, ib/ub"                , "RVMI-T4" , "EVEX.256.66.0F3A.W0 38 /r ib" , "AVX512_F-VL"],
        ["vinserti32x4"     , "W:zmm {kz}, zmm, xmm/m128, ib/ub"                , "RVMI-T4" , "EVEX.512.66.0F3A.W0 38 /r ib" , "AVX512_F"],
        ["vinserti32x8"     , "W:zmm {kz}, zmm, ymm/m256, ib/ub"                , "RVMI-T8" , "EVEX.512.66.0F3A.W0 3A /r ib" , "AVX512_DQ"],
        ["vinserti64x2"     , "W:ymm {kz}, ymm, xmm/m128, ib/ub"                , "RVMI-T2" , "EVEX.256.66.0F3A.W1 38 /r ib" , "AVX512_DQ-VL"],
        ["vinserti64x2"     , "W:zmm {kz}, zmm, xmm/m128, ib/ub"                , "RVMI-T2" , "EVEX.512.66.0F3A.W1 38 /r ib" , "AVX512_DQ"],
        ["vinserti64x4"     , "W:zmm {kz}, zmm, ymm/m256, ib/ub"                , "RVMI-T4" , "EVEX.512.66.0F3A.W1 3A /r ib" , "AVX512_F"],
        ["vinsertps"        , "W:xmm, xmm, xmm[31:0]/m32, ib/ub"                , "RVMI-T1S", "EVEX.128.66.0F3A.W0 21 /r ib" , "AVX512_F"],
        ["vmaxpd"           , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 5F /r"      , "AVX512_F-VL"],
        ["vmaxpd"           , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 5F /r"      , "AVX512_F-VL"],
        ["vmaxpd"           , "W:zmm {kz}, zmm, zmm/m512/b64 {sae}"             , "RVM-FV"  , "EVEX.512.66.0F.W1 5F /r"      , "AVX512_F"],
        ["vmaxps"           , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.0F.W0 5F /r"         , "AVX512_F-VL"],
        ["vmaxps"           , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.0F.W0 5F /r"         , "AVX512_F-VL"],
        ["vmaxps"           , "W:zmm {kz}, zmm, zmm/m512/b32 {sae}"             , "RVM-FV"  , "EVEX.512.0F.W0 5F /r"         , "AVX512_F"],
        ["vmaxsd"           , "W:xmm {kz}, xmm, xmm[63:0]/m64 {sae}"            , "RVM-T1S" , "EVEX.LIG.F2.0F.W1 5F /r"      , "AVX512_F-VL"],
        ["vmaxss"           , "W:xmm {kz}, xmm, xmm[31:0]/m32 {sae}"            , "RVM-T1S" , "EVEX.LIG.F3.0F.W0 5F /r"      , "AVX512_F-VL"],
        ["vminpd"           , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 5D /r"      , "AVX512_F-VL"],
        ["vminpd"           , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 5D /r"      , "AVX512_F-VL"],
        ["vminpd"           , "W:zmm {kz}, zmm, zmm/m512/b64 {sae}"             , "RVM-FV"  , "EVEX.512.66.0F.W1 5D /r"      , "AVX512_F"],
        ["vminps"           , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.0F.W0 5D /r"         , "AVX512_F-VL"],
        ["vminps"           , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.0F.W0 5D /r"         , "AVX512_F-VL"],
        ["vminps"           , "W:zmm {kz}, zmm, zmm/m512/b32 {sae}"             , "RVM-FV"  , "EVEX.512.0F.W0 5D /r"         , "AVX512_F"],
        ["vminsd"           , "W:xmm {kz}, xmm, xmm[63:0]/m64 {sae}"            , "RVM-T1S" , "EVEX.LIG.F2.0F.W1 5D /r"      , "AVX512_F-VL"],
        ["vminss"           , "W:xmm {kz}, xmm, xmm[31:0]/m32 {sae}"            , "RVM-T1S" , "EVEX.LIG.F3.0F.W0 5D /r"      , "AVX512_F-VL"],
        ["vmovapd"          , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.66.0F.W1 28 /r"      , "AVX512_F-VL"],
        ["vmovapd"          , "W:xmm/m128 {kz}, xmm"                            , "MR-FVM"  , "EVEX.128.66.0F.W1 29 /r"      , "AVX512_F-VL"],
        ["vmovapd"          , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.66.0F.W1 28 /r"      , "AVX512_F-VL"],
        ["vmovapd"          , "W:ymm/m256 {kz}, ymm"                            , "MR-FVM"  , "EVEX.256.66.0F.W1 29 /r"      , "AVX512_F-VL"],
        ["vmovapd"          , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.66.0F.W1 28 /r"      , "AVX512_F"],
        ["vmovapd"          , "W:zmm/m512 {kz}, zmm"                            , "MR-FVM"  , "EVEX.512.66.0F.W1 29 /r"      , "AVX512_F"],
        ["vmovaps"          , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.0F.W0 28 /r"         , "AVX512_F-VL"],
        ["vmovaps"          , "W:xmm/m128 {kz}, xmm"                            , "MR-FVM"  , "EVEX.128.0F.W0 29 /r"         , "AVX512_F-VL"],
        ["vmovaps"          , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.0F.W0 28 /r"         , "AVX512_F-VL"],
        ["vmovaps"          , "W:ymm/m256 {kz}, ymm"                            , "MR-FVM"  , "EVEX.256.0F.W0 29 /r"         , "AVX512_F-VL"],
        ["vmovaps"          , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.0F.W0 28 /r"         , "AVX512_F"],
        ["vmovaps"          , "W:zmm/m512 {kz}, zmm"                            , "MR-FVM"  , "EVEX.512.0F.W0 29 /r"         , "AVX512_F"],
        ["vmovd"            , "W:r32/m32, xmm[31:0]"                            , "MR-T1S"  , "EVEX.128.66.0F.W0 7E /r"      , "AVX512_F"],
        ["vmovd"            , "W:xmm[31:0], r32/m32"                            , "RM-T1S"  , "EVEX.128.66.0F.W0 6E /r"      , "AVX512_F"],
        ["vmovddup"         , "W:xmm {kz}, xmm[63:0]/m64"                       , "RM-DUP"  , "EVEX.128.F2.0F.W1 12 /r"      , "AVX512_F-VL"],
        ["vmovddup"         , "W:ymm {kz}, ymm/m256"                            , "RM-DUP"  , "EVEX.256.F2.0F.W1 12 /r"      , "AVX512_F-VL"],
        ["vmovddup"         , "W:zmm {kz}, zmm/m512"                            , "RM-DUP"  , "EVEX.512.F2.0F.W1 12 /r"      , "AVX512_F"],
        ["vmovdqa32"        , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.66.0F.W0 6F /r"      , "AVX512_F-VL"],
        ["vmovdqa32"        , "W:xmm/m128 {kz}, xmm"                            , "MR-FVM"  , "EVEX.128.66.0F.W0 7F /r"      , "AVX512_F-VL"],
        ["vmovdqa32"        , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.66.0F.W0 6F /r"      , "AVX512_F-VL"],
        ["vmovdqa32"        , "W:ymm/m256 {kz}, ymm"                            , "MR-FVM"  , "EVEX.256.66.0F.W0 7F /r"      , "AVX512_F-VL"],
        ["vmovdqa32"        , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.66.0F.W0 6F /r"      , "AVX512_F"],
        ["vmovdqa32"        , "W:zmm/m512 {kz}, zmm"                            , "MR-FVM"  , "EVEX.512.66.0F.W0 7F /r"      , "AVX512_F"],
        ["vmovdqa64"        , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.66.0F.W1 6F /r"      , "AVX512_F-VL"],
        ["vmovdqa64"        , "W:xmm/m128 {kz}, xmm"                            , "MR-FVM"  , "EVEX.128.66.0F.W1 7F /r"      , "AVX512_F-VL"],
        ["vmovdqa64"        , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.66.0F.W1 6F /r"      , "AVX512_F-VL"],
        ["vmovdqa64"        , "W:ymm/m256 {kz}, ymm"                            , "MR-FVM"  , "EVEX.256.66.0F.W1 7F /r"      , "AVX512_F-VL"],
        ["vmovdqa64"        , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.66.0F.W1 6F /r"      , "AVX512_F"],
        ["vmovdqa64"        , "W:zmm/m512 {kz}, zmm"                            , "MR-FVM"  , "EVEX.512.66.0F.W1 7F /r"      , "AVX512_F"],
        ["vmovdqu16"        , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.F2.0F.W1 6F /r"      , "AVX512_BW-VL"],
        ["vmovdqu16"        , "W:xmm/m128 {kz}, xmm"                            , "MR-FVM"  , "EVEX.128.F2.0F.W1 7F /r"      , "AVX512_BW-VL"],
        ["vmovdqu16"        , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.F2.0F.W1 6F /r"      , "AVX512_BW-VL"],
        ["vmovdqu16"        , "W:ymm/m256 {kz}, ymm"                            , "MR-FVM"  , "EVEX.256.F2.0F.W1 7F /r"      , "AVX512_BW-VL"],
        ["vmovdqu16"        , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.F2.0F.W1 6F /r"      , "AVX512_BW"],
        ["vmovdqu16"        , "W:zmm/m512 {kz}, zmm"                            , "MR-FVM"  , "EVEX.512.F2.0F.W1 7F /r"      , "AVX512_BW"],
        ["vmovdqu32"        , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.F3.0F.W0 6F /r"      , "AVX512_F-VL"],
        ["vmovdqu32"        , "W:xmm/m128 {kz}, xmm"                            , "MR-FVM"  , "EVEX.128.F3.0F.W0 7F /r"      , "AVX512_F-VL"],
        ["vmovdqu32"        , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.F3.0F.W0 6F /r"      , "AVX512_F-VL"],
        ["vmovdqu32"        , "W:ymm/m256 {kz}, ymm"                            , "MR-FVM"  , "EVEX.256.F3.0F.W0 7F /r"      , "AVX512_F-VL"],
        ["vmovdqu32"        , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.F3.0F.W0 6F /r"      , "AVX512_F"],
        ["vmovdqu32"        , "W:zmm/m512 {kz}, zmm"                            , "MR-FVM"  , "EVEX.512.F3.0F.W0 7F /r"      , "AVX512_F"],
        ["vmovdqu64"        , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.F3.0F.W1 6F /r"      , "AVX512_F-VL"],
        ["vmovdqu64"        , "W:xmm/m128 {kz}, xmm"                            , "MR-FVM"  , "EVEX.128.F3.0F.W1 7F /r"      , "AVX512_F-VL"],
        ["vmovdqu64"        , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.F3.0F.W1 6F /r"      , "AVX512_F-VL"],
        ["vmovdqu64"        , "W:ymm/m256 {kz}, ymm"                            , "MR-FVM"  , "EVEX.256.F3.0F.W1 7F /r"      , "AVX512_F-VL"],
        ["vmovdqu64"        , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.F3.0F.W1 6F /r"      , "AVX512_F"],
        ["vmovdqu64"        , "W:zmm/m512 {kz}, zmm"                            , "MR-FVM"  , "EVEX.512.F3.0F.W1 7F /r"      , "AVX512_F"],
        ["vmovdqu8"         , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.F2.0F.W0 6F /r"      , "AVX512_BW-VL"],
        ["vmovdqu8"         , "W:xmm/m128 {kz}, xmm"                            , "MR-FVM"  , "EVEX.128.F2.0F.W0 7F /r"      , "AVX512_BW-VL"],
        ["vmovdqu8"         , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.F2.0F.W0 6F /r"      , "AVX512_BW-VL"],
        ["vmovdqu8"         , "W:ymm/m256 {kz}, ymm"                            , "MR-FVM"  , "EVEX.256.F2.0F.W0 7F /r"      , "AVX512_BW-VL"],
        ["vmovdqu8"         , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.F2.0F.W0 6F /r"      , "AVX512_BW"],
        ["vmovdqu8"         , "W:zmm/m512 {kz}, zmm"                            , "MR-FVM"  , "EVEX.512.F2.0F.W0 7F /r"      , "AVX512_BW"],
        ["vmovhlps"         , "W:xmm, xmm[127:64], xmm[127:64]"                 , "RVM"     , "EVEX.128.0F.W0 12 /r"         , "AVX512_F"],
        ["vmovhpd"          , "W:m64, xmm[127:64]"                              , "MR-T1S"  , "EVEX.128.66.0F.W1 17 /r"      , "AVX512_F"],
        ["vmovhpd"          , "W:xmm, xmm[63:0], m64"                           , "RVM-T1S" , "EVEX.128.66.0F.W1 16 /r"      , "AVX512_F"],
        ["vmovhps"          , "W:m64, xmm[127:64]"                              , "MR-T2"   , "EVEX.128.0F.W0 17 /r"         , "AVX512_F"],
        ["vmovhps"          , "W:xmm, xmm[63:0], m64"                           , "RVM-T2"  , "EVEX.128.0F.W0 16 /r"         , "AVX512_F"],
        ["vmovlhps"         , "W:xmm, xmm[63:0], xmm[63:0]"                     , "RVM"     , "EVEX.128.0F.W0 16 /r"         , "AVX512_F"],
        ["vmovlpd"          , "W:m64, xmm[63:0]"                                , "MR-T1S"  , "EVEX.128.66.0F.W1 13 /r"      , "AVX512_F"],
        ["vmovlpd"          , "W:xmm, xmm[127:64], m64"                         , "RVM-T1S" , "EVEX.128.66.0F.W1 12 /r"      , "AVX512_F"],
        ["vmovlps"          , "W:m64, xmm[63:0]"                                , "MR-T2"   , "EVEX.128.0F.W0 13 /r"         , "AVX512_F"],
        ["vmovlps"          , "W:xmm, xmm[127:64], m64"                         , "RVM-T2"  , "EVEX.128.0F.W0 12 /r"         , "AVX512_F"],
        ["vmovntdq"         , "W:m128, xmm"                                     , "MR-FVM"  , "EVEX.128.66.0F.W0 E7 /r"      , "AVX512_F-VL"],
        ["vmovntdq"         , "W:m256, ymm"                                     , "MR-FVM"  , "EVEX.256.66.0F.W0 E7 /r"      , "AVX512_F-VL"],
        ["vmovntdq"         , "W:m512, zmm"                                     , "MR-FVM"  , "EVEX.512.66.0F.W0 E7 /r"      , "AVX512_F"],
        ["vmovntdqa"        , "W:xmm, m128"                                     , "RM-FVM"  , "EVEX.128.66.0F38.W0 2A /r"    , "AVX512_F-VL"],
        ["vmovntdqa"        , "W:ymm, m256"                                     , "RM-FVM"  , "EVEX.256.66.0F38.W0 2A /r"    , "AVX512_F-VL"],
        ["vmovntdqa"        , "W:zmm, m512"                                     , "RM-FVM"  , "EVEX.512.66.0F38.W0 2A /r"    , "AVX512_F"],
        ["vmovntpd"         , "W:m128, xmm"                                     , "MR-FVM"  , "EVEX.128.66.0F.W1 2B /r"      , "AVX512_F-VL"],
        ["vmovntpd"         , "W:m256, ymm"                                     , "MR-FVM"  , "EVEX.256.66.0F.W1 2B /r"      , "AVX512_F-VL"],
        ["vmovntpd"         , "W:m512, zmm"                                     , "MR-FVM"  , "EVEX.512.66.0F.W1 2B /r"      , "AVX512_F"],
        ["vmovntps"         , "W:m128, xmm"                                     , "MR-FVM"  , "EVEX.128.66.0F.W0 2B /r"      , "AVX512_F-VL"],
        ["vmovntps"         , "W:m256, ymm"                                     , "MR-FVM"  , "EVEX.256.66.0F.W0 2B /r"      , "AVX512_F-VL"],
        ["vmovntps"         , "W:m512, zmm"                                     , "MR-FVM"  , "EVEX.512.0F.W0 2B /r"         , "AVX512_F"],
        ["vmovq"            , "W:r64/m64, xmm[63:0]"                            , "MR-T1S"  , "EVEX.128.66.0F.W1 7E /r"      , "AVX512_F X64"],
        ["vmovq"            , "W:xmm[63:0], r64/m64"                            , "RM-T1S"  , "EVEX.128.66.0F.W1 6E /r"      , "AVX512_F X64"],
        ["vmovq"            , "W:xmm[63:0], xmm[63:0]/m64"                      , "RM-T1S"  , "EVEX.128.F3.0F.W1 7E /r"      , "AVX512_F"],
        ["vmovq"            , "W:xmm[63:0]/m64, xmm[63:0]"                      , "MR-T1S"  , "EVEX.128.66.0F.W1 D6 /r"      , "AVX512_F"],
        ["vmovsd"           , "W:m64, xmm[63:0]"                                , "MR-T1S"  , "EVEX.LIG.F2.0F.W1 11 /r"      , "AVX512_F"],
        ["vmovsd"           , "W:xmm[63:0] {kz}, m64"                           , "MR-T1S"  , "EVEX.LIG.F2.0F.W1 10 /r"      , "AVX512_F"],
        ["vmovsd"           , "W:xmm {kz}, xmm[127:64], xmm[63:0]"              , "MVR"     , "EVEX.LIG.F2.0F.W1 11 /r"      , "AVX512_F"],
        ["vmovsd"           , "W:xmm {kz}, xmm[127:64], xmm[63:0]"              , "RVM"     , "EVEX.LIG.F2.0F.W1 10 /r"      , "AVX512_F"],
        ["vmovshdup"        , "W:xmm {kz}, xmm/m128"                            , "RVM-FVM" , "EVEX.128.F3.0F.W0 16 /r"      , "AVX512_F-VL"],
        ["vmovshdup"        , "W:ymm {kz}, ymm/m256"                            , "RVM-FVM" , "EVEX.256.F3.0F.W0 16 /r"      , "AVX512_F-VL"],
        ["vmovshdup"        , "W:zmm {kz}, zmm/m512"                            , "RVM-FVM" , "EVEX.512.F3.0F.W0 16 /r"      , "AVX512_F"],
        ["vmovsldup"        , "W:xmm {kz}, xmm/m128"                            , "RVM-FVM" , "EVEX.128.F3.0F.W0 12 /r"      , "AVX512_F-VL"],
        ["vmovsldup"        , "W:ymm {kz}, ymm/m256"                            , "RVM-FVM" , "EVEX.256.F3.0F.W0 12 /r"      , "AVX512_F-VL"],
        ["vmovsldup"        , "W:zmm {kz}, zmm/m512"                            , "RVM-FVM" , "EVEX.512.F3.0F.W0 12 /r"      , "AVX512_F"],
        ["vmovss"           , "W:m32, xmm[31:0]"                                , "MR-T1S"  , "EVEX.LIG.F3.0F.W0 11 /r"      , "AVX512_F"],
        ["vmovss"           , "W:xmm[31:0] {kz}, m32"                           , "MR-T1S"  , "EVEX.LIG.F3.0F.W0 10 /r"      , "AVX512_F"],
        ["vmovss"           , "W:xmm {kz}, xmm[127:32], xmm[31:0]"              , "MVR"     , "EVEX.LIG.F3.0F.W0 11 /r"      , "AVX512_F"],
        ["vmovss"           , "W:xmm {kz}, xmm[127:32], xmm[31:0]"              , "RVM"     , "EVEX.LIG.F3.0F.W0 10 /r"      , "AVX512_F"],
        ["vmovupd"          , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.66.0F.W1 10 /r"      , "AVX512_F-VL"],
        ["vmovupd"          , "W:xmm/m128 {kz}, xmm"                            , "MR-FVM"  , "EVEX.128.66.0F.W1 11 /r"      , "AVX512_F-VL"],
        ["vmovupd"          , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.66.0F.W1 11 /r"      , "AVX512_F-VL"],
        ["vmovupd"          , "W:ymm/m256 {kz}, ymm"                            , "MR-FVM"  , "EVEX.256.66.0F.W1 10 /r"      , "AVX512_F-VL"],
        ["vmovupd"          , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.66.0F.W1 10 /r"      , "AVX512_F"],
        ["vmovupd"          , "W:zmm/m512 {kz}, zmm"                            , "MR-FVM"  , "EVEX.512.66.0F.W1 11 /r"      , "AVX512_F"],
        ["vmovups"          , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.0F.W0 10 /r"         , "AVX512_F-VL"],
        ["vmovups"          , "W:xmm/m128 {kz}, xmm"                            , "MR-FVM"  , "EVEX.128.0F.W0 11 /r"         , "AVX512_F-VL"],
        ["vmovups"          , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.0F.W0 10 /r"         , "AVX512_F-VL"],
        ["vmovups"          , "W:ymm/m256 {kz}, ymm"                            , "MR-FVM"  , "EVEX.256.0F.W0 11 /r"         , "AVX512_F-VL"],
        ["vmovups"          , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.0F.W0 10 /r"         , "AVX512_F"],
        ["vmovups"          , "W:zmm/m512 {kz}, zmm"                            , "MR-FVM"  , "EVEX.512.0F.W0 11 /r"         , "AVX512_F"],
        ["vmulpd"           , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 59 /r"      , "AVX512_F-VL"],
        ["vmulpd"           , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 59 /r"      , "AVX512_F-VL"],
        ["vmulpd"           , "W:zmm {kz},~zmm,~zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F.W1 59 /r"      , "AVX512_F"],
        ["vmulps"           , "W:xmm {kz},~xmm,~xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.0F.W0 59 /r"         , "AVX512_F-VL"],
        ["vmulps"           , "W:ymm {kz},~ymm,~ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.0F.W0 59 /r"         , "AVX512_F-VL"],
        ["vmulps"           , "W:zmm {kz},~zmm,~zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.0F.W0 59 /r"         , "AVX512_F"],
        ["vmulsd"           , "W:xmm {kz}, xmm, xmm[63:0]/m64 {er}"             , "RVM-T1S" , "EVEX.LIG.F2.0F.W1 59 /r"      , "AVX512_F"],
        ["vmulss"           , "W:xmm {kz}, xmm, xmm[31:0]/m32 {er}"             , "RVM-T1S" , "EVEX.LIG.F3.0F.W0 59 /r"      , "AVX512_F"],
        ["vorpd"            , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 56 /r"      , "AVX512_DQ-VL"],
        ["vorpd"            , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 56 /r"      , "AVX512_DQ-VL"],
        ["vorpd"            , "W:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F.W1 56 /r"      , "AVX512_DQ"],
        ["vorps"            , "W:xmm {kz},~xmm,~xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.0F.W0 56 /r"         , "AVX512_DQ-VL"],
        ["vorps"            , "W:ymm {kz},~ymm,~ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.0F.W0 56 /r"         , "AVX512_DQ-VL"],
        ["vorps"            , "W:zmm {kz},~zmm,~zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.0F.W0 56 /r"         , "AVX512_DQ"],
        ["vp4dpwssd"        , "W:zmm {kz}, zmm, zmm, zmm, zmm, m128"            , "RM-T1_4X", "EVEX.512.F2.0F38.W0 52 /r"    , "AVX512_4VNNIW"],
        ["vp4dpwssds"       , "W:zmm {kz}, zmm, zmm, zmm, zmm, m128"            , "RM-T1_4X", "EVEX.512.F2.0F38.W0 53 /r"    , "AVX512_4VNNIW"],
        ["vpabsb"           , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.66.0F38 1C /r"       , "AVX512_BW-VL"],
        ["vpabsb"           , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.66.0F38 1C /r"       , "AVX512_BW-VL"],
        ["vpabsb"           , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.66.0F38 1C /r"       , "AVX512_BW"],
        ["vpabsd"           , "W:xmm {kz}, xmm/m128"                            , "RM-FV"   , "EVEX.128.66.0F38.W0 1E /r"    , "AVX512_F-VL"],
        ["vpabsd"           , "W:ymm {kz}, ymm/m256"                            , "RM-FV"   , "EVEX.256.66.0F38.W0 1E /r"    , "AVX512_F-VL"],
        ["vpabsd"           , "W:zmm {kz}, zmm/m512"                            , "RM-FV"   , "EVEX.512.66.0F38.W0 1E /r"    , "AVX512_F"],
        ["vpabsq"           , "W:xmm {kz}, xmm/m128"                            , "RM-FV"   , "EVEX.128.66.0F38.W1 1F /r"    , "AVX512_F-VL"],
        ["vpabsq"           , "W:ymm {kz}, ymm/m256"                            , "RM-FV"   , "EVEX.256.66.0F38.W1 1F /r"    , "AVX512_F-VL"],
        ["vpabsq"           , "W:zmm {kz}, zmm/m512"                            , "RM-FV"   , "EVEX.512.66.0F38.W1 1F /r"    , "AVX512_F"],
        ["vpabsw"           , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.66.0F38 1D /r"       , "AVX512_BW-VL"],
        ["vpabsw"           , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.66.0F38 1D /r"       , "AVX512_BW-VL"],
        ["vpabsw"           , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.66.0F38 1D /r"       , "AVX512_BW"],
        ["vpackssdw"        , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F.W0 6B /r"      , "AVX512_BW-VL"],
        ["vpackssdw"        , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F.W0 6B /r"      , "AVX512_BW-VL"],
        ["vpackssdw"        , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F.W0 6B /r"      , "AVX512_BW"],
        ["vpacksswb"        , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG 63 /r"     , "AVX512_BW-VL"],
        ["vpacksswb"        , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG 63 /r"     , "AVX512_BW-VL"],
        ["vpacksswb"        , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG 63 /r"     , "AVX512_BW"],
        ["vpackusdw"        , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 2B /r"    , "AVX512_BW-VL"],
        ["vpackusdw"        , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 2B /r"    , "AVX512_BW-VL"],
        ["vpackusdw"        , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 2B /r"    , "AVX512_BW"],
        ["vpackuswb"        , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG 67 /r"     , "AVX512_BW-VL"],
        ["vpackuswb"        , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG 67 /r"     , "AVX512_BW-VL"],
        ["vpackuswb"        , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG 67 /r"     , "AVX512_BW"],
        ["vpaddb"           , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG FC /r"     , "AVX512_BW-VL"],
        ["vpaddb"           , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG FC /r"     , "AVX512_BW-VL"],
        ["vpaddb"           , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG FC /r"     , "AVX512_BW"],
        ["vpaddd"           , "W:xmm {kz},~xmm,~xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F.W0 FE /r"      , "AVX512_F-VL"],
        ["vpaddd"           , "W:ymm {kz},~ymm,~ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F.W0 FE /r"      , "AVX512_F-VL"],
        ["vpaddd"           , "W:zmm {kz},~zmm,~zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F.W0 FE /r"      , "AVX512_F"],
        ["vpaddq"           , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 D4 /r"      , "AVX512_F-VL"],
        ["vpaddq"           , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 D4 /r"      , "AVX512_F-VL"],
        ["vpaddq"           , "W:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F.W1 D4 /r"      , "AVX512_F"],
        ["vpaddsb"          , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG EC /r"     , "AVX512_BW-VL"],
        ["vpaddsb"          , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG EC /r"     , "AVX512_BW-VL"],
        ["vpaddsb"          , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG EC /r"     , "AVX512_BW"],
        ["vpaddsw"          , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG ED /r"     , "AVX512_BW-VL"],
        ["vpaddsw"          , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG ED /r"     , "AVX512_BW-VL"],
        ["vpaddsw"          , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG ED /r"     , "AVX512_BW"],
        ["vpaddusb"         , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG DC /r"     , "AVX512_BW-VL"],
        ["vpaddusb"         , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG DC /r"     , "AVX512_BW-VL"],
        ["vpaddusb"         , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG DC /r"     , "AVX512_BW"],
        ["vpaddusw"         , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG DD /r"     , "AVX512_BW-VL"],
        ["vpaddusw"         , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG DD /r"     , "AVX512_BW-VL"],
        ["vpaddusw"         , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG DD /r"     , "AVX512_BW"],
        ["vpaddw"           , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG FD /r"     , "AVX512_BW-VL"],
        ["vpaddw"           , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG FD /r"     , "AVX512_BW-VL"],
        ["vpaddw"           , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG FD /r"     , "AVX512_BW"],
        ["vpalignr"         , "W:xmm {kz}, xmm, xmm/m128, ib/ub"                , "RVMI-FVM", "EVEX.128.66.0F3A.WIG 0F /r ib", "AVX512_BW-VL"],
        ["vpalignr"         , "W:ymm {kz}, ymm, ymm/m256, ib/ub"                , "RVMI-FVM", "EVEX.256.66.0F3A.WIG 0F /r ib", "AVX512_BW-VL"],
        ["vpalignr"         , "W:zmm {kz}, zmm, zmm/m512, ib/ub"                , "RVMI-FVM", "EVEX.512.66.0F3A.WIG 0F /r ib", "AVX512_BW"],
        ["vpandd"           , "W:xmm {kz},~xmm,~xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F.W0 DB /r"      , "AVX512_F-VL"],
        ["vpandd"           , "W:ymm {kz},~ymm,~ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F.W0 DB /r"      , "AVX512_F-VL"],
        ["vpandd"           , "W:zmm {kz},~zmm,~zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F.W0 DB /r"      , "AVX512_F"],
        ["vpandnd"          , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F.W0 DF /r"      , "AVX512_F-VL"],
        ["vpandnd"          , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F.W0 DF /r"      , "AVX512_F-VL"],
        ["vpandnd"          , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F.W0 DF /r"      , "AVX512_F"],
        ["vpandnq"          , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 DF /r"      , "AVX512_F-VL"],
        ["vpandnq"          , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 DF /r"      , "AVX512_F-VL"],
        ["vpandnq"          , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F.W1 DF /r"      , "AVX512_F"],
        ["vpandq"           , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 DB /r"      , "AVX512_F-VL"],
        ["vpandq"           , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 DB /r"      , "AVX512_F-VL"],
        ["vpandq"           , "W:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F.W1 DB /r"      , "AVX512_F"],
        ["vpavgb"           , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG E0 /r"     , "AVX512_BW-VL"],
        ["vpavgb"           , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG E0 /r"     , "AVX512_BW-VL"],
        ["vpavgb"           , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG E0 /r"     , "AVX512_BW"],
        ["vpavgw"           , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG E3 /r"     , "AVX512_BW-VL"],
        ["vpavgw"           , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG E3 /r"     , "AVX512_BW-VL"],
        ["vpavgw"           , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG E3 /r"     , "AVX512_BW"],
        ["vpbroadcastb"     , "W:xmm {kz}, r32[7:0]"                            , "RM-T1S"  , "EVEX.128.66.0F38.W0 7A /r"    , "AVX512_BW-VL"],
        ["vpbroadcastb"     , "W:xmm {kz}, xmm[7:0]/m8"                         , "RM-T1S"  , "EVEX.128.66.0F38.W0 78 /r"    , "AVX512_BW-VL"],
        ["vpbroadcastb"     , "W:ymm {kz}, r32[7:0]"                            , "RM-T1S"  , "EVEX.256.66.0F38.W0 7A /r"    , "AVX512_BW-VL"],
        ["vpbroadcastb"     , "W:ymm {kz}, xmm[7:0]/m8"                         , "RM-T1S"  , "EVEX.256.66.0F38.W0 78 /r"    , "AVX512_BW-VL"],
        ["vpbroadcastb"     , "W:zmm {kz}, r32[7:0]"                            , "RM-T1S"  , "EVEX.512.66.0F38.W0 7A /r"    , "AVX512_BW"],
        ["vpbroadcastb"     , "W:zmm {kz}, xmm[7:0]/m8"                         , "RM-T1S"  , "EVEX.512.66.0F38.W0 78 /r"    , "AVX512_BW"],
        ["vpbroadcastd"     , "W:xmm {kz}, r32[31:0]"                           , "RM-T1S"  , "EVEX.128.66.0F38.W0 7C /r"    , "AVX512_F-VL"],
        ["vpbroadcastd"     , "W:xmm {kz}, xmm[31:0]/m32"                       , "RM-T1S"  , "EVEX.128.66.0F38.W0 58 /r"    , "AVX512_F-VL"],
        ["vpbroadcastd"     , "W:ymm {kz}, r32[31:0]"                           , "RM-T1S"  , "EVEX.256.66.0F38.W0 7C /r"    , "AVX512_F-VL"],
        ["vpbroadcastd"     , "W:ymm {kz}, xmm[31:0]/m32"                       , "RM-T1S"  , "EVEX.256.66.0F38.W0 58 /r"    , "AVX512_F-VL"],
        ["vpbroadcastd"     , "W:zmm {kz}, r32[31:0]"                           , "RM-T1S"  , "EVEX.512.66.0F38.W0 7C /r"    , "AVX512_F"],
        ["vpbroadcastd"     , "W:zmm {kz}, xmm[31:0]/m32"                       , "RM-T1S"  , "EVEX.512.66.0F38.W0 58 /r"    , "AVX512_F"],
        ["vpbroadcastmb2d"  , "W:xmm, k"                                        , "RM"      , "EVEX.128.F3.0F38.W0 3A /r"    , "AVX512_CDI-VL"],
        ["vpbroadcastmb2d"  , "W:ymm, k"                                        , "RM"      , "EVEX.256.F3.0F38.W0 3A /r"    , "AVX512_CDI-VL"],
        ["vpbroadcastmb2d"  , "W:zmm, k"                                        , "RM"      , "EVEX.512.F3.0F38.W0 3A /r"    , "AVX512_CDI"],
        ["vpbroadcastmb2q"  , "W:xmm, k"                                        , "RM"      , "EVEX.128.F3.0F38.W1 2A /r"    , "AVX512_CDI-VL"],
        ["vpbroadcastmb2q"  , "W:ymm, k"                                        , "RM"      , "EVEX.256.F3.0F38.W1 2A /r"    , "AVX512_CDI-VL"],
        ["vpbroadcastmb2q"  , "W:zmm, k"                                        , "RM"      , "EVEX.512.F3.0F38.W1 2A /r"    , "AVX512_CDI"],
        ["vpbroadcastq"     , "W:xmm {kz}, r64"                                 , "RM-T1S"  , "EVEX.128.66.0F38.W1 7C /r"    , "AVX512_F-VL X64"],
        ["vpbroadcastq"     , "W:xmm {kz}, xmm[63:0]/m64"                       , "RM-T1S"  , "EVEX.128.66.0F38.W1 59 /r"    , "AVX512_F-VL"],
        ["vpbroadcastq"     , "W:ymm {kz}, r64"                                 , "RM-T1S"  , "EVEX.256.66.0F38.W1 7C /r"    , "AVX512_F-VL X64"],
        ["vpbroadcastq"     , "W:ymm {kz}, xmm[63:0]/m64"                       , "RM-T1S"  , "EVEX.256.66.0F38.W1 59 /r"    , "AVX512_F-VL"],
        ["vpbroadcastq"     , "W:zmm {kz}, r64"                                 , "RM-T1S"  , "EVEX.512.66.0F38.W1 7C /r"    , "AVX512_F X64"],
        ["vpbroadcastq"     , "W:zmm {kz}, xmm[63:0]/m64"                       , "RM-T1S"  , "EVEX.512.66.0F38.W1 59 /r"    , "AVX512_F"],
        ["vpbroadcastw"     , "W:xmm {kz}, r32[15:0]"                           , "RM-T1S"  , "EVEX.128.66.0F38.W0 7B /r"    , "AVX512_BW-VL"],
        ["vpbroadcastw"     , "W:xmm {kz}, xmm[15:0]/m16"                       , "RM-T1S"  , "EVEX.128.66.0F38.W0 79 /r"    , "AVX512_BW-VL"],
        ["vpbroadcastw"     , "W:ymm {kz}, r32[15:0]"                           , "RM-T1S"  , "EVEX.256.66.0F38.W0 7B /r"    , "AVX512_BW-VL"],
        ["vpbroadcastw"     , "W:ymm {kz}, xmm[15:0]/m16"                       , "RM-T1S"  , "EVEX.256.66.0F38.W0 79 /r"    , "AVX512_BW-VL"],
        ["vpbroadcastw"     , "W:zmm {kz}, r32[15:0]"                           , "RM-T1S"  , "EVEX.512.66.0F38.W0 7B /r"    , "AVX512_BW"],
        ["vpbroadcastw"     , "W:zmm {kz}, xmm[15:0]/m16"                       , "RM-T1S"  , "EVEX.512.66.0F38.W0 79 /r"    , "AVX512_BW"],
        ["vpclmulqdq"       , "W:xmm, xmm, xmm/m128, ib/ub"                     , "RVMI-FVM", "EVEX.128.66.0F3A.WIG 44 /r ib", "AVX512_F-VL VPCLMULQDQ"],
        ["vpclmulqdq"       , "W:ymm, ymm, ymm/m256, ib/ub"                     , "RVMI-FVM", "EVEX.256.66.0F3A.WIG 44 /r ib", "AVX512_F-VL VPCLMULQDQ"],
        ["vpclmulqdq"       , "W:zmm, zmm, zmm/m512, ib/ub"                     , "RVMI-FVM", "EVEX.512.66.0F3A.WIG 44 /r ib", "AVX512_F    VPCLMULQDQ"],
        ["vpcmpb"           , "W:k {k}, xmm, xmm/m128, ib/ub"                   , "RVMI-FVM", "EVEX.128.66.0F3A.W0 3F /r ib" , "AVX512_BW-VL"],
        ["vpcmpb"           , "W:k {k}, ymm, ymm/m256, ib/ub"                   , "RVMI-FVM", "EVEX.256.66.0F3A.W0 3F /r ib" , "AVX512_BW-VL"],
        ["vpcmpb"           , "W:k {k}, zmm, zmm/m512, ib/ub"                   , "RVMI-FVM", "EVEX.512.66.0F3A.W0 3F /r ib" , "AVX512_BW"],
        ["vpcmpd"           , "W:k {k}, xmm, xmm/m128/b32, ib/ub"               , "RVMI-FV" , "EVEX.128.66.0F3A.W0 1F /r ib" , "AVX512_F-VL"],
        ["vpcmpd"           , "W:k {k}, ymm, ymm/m256/b32, ib/ub"               , "RVMI-FV" , "EVEX.256.66.0F3A.W0 1F /r ib" , "AVX512_F-VL"],
        ["vpcmpd"           , "W:k {k}, zmm, zmm/m512/b32, ib/ub"               , "RVMI-FV" , "EVEX.512.66.0F3A.W0 1F /r ib" , "AVX512_F"],
        ["vpcmpeqb"         , "W:k {k},~xmm,~xmm/m128"                          , "RVM-FV"  , "EVEX.128.66.0F.WIG 74 /r"     , "AVX512_BW-VL"],
        ["vpcmpeqb"         , "W:k {k},~ymm,~ymm/m256"                          , "RVM-FV"  , "EVEX.256.66.0F.WIG 74 /r"     , "AVX512_BW-VL"],
        ["vpcmpeqb"         , "W:k {k},~zmm,~zmm/m512"                          , "RVM-FV"  , "EVEX.512.66.0F.WIG 74 /r"     , "AVX512_BW"],
        ["vpcmpeqd"         , "W:k {k},~xmm,~xmm/m128/b32"                      , "RVM-FVM" , "EVEX.128.66.0F.W0 76 /r"      , "AVX512_F-VL"],
        ["vpcmpeqd"         , "W:k {k},~ymm,~ymm/m256/b32"                      , "RVM-FVM" , "EVEX.256.66.0F.W0 76 /r"      , "AVX512_F-VL"],
        ["vpcmpeqd"         , "W:k {k},~zmm,~zmm/m512/b32"                      , "RVM-FVM" , "EVEX.512.66.0F.W0 76 /r"      , "AVX512_F"],
        ["vpcmpeqq"         , "W:k {k},~xmm,~xmm/m128/b64"                      , "RVM-FVM" , "EVEX.128.66.0F38.W1 29 /r"    , "AVX512_F-VL"],
        ["vpcmpeqq"         , "W:k {k},~ymm,~ymm/m256/b64"                      , "RVM-FVM" , "EVEX.256.66.0F38.W1 29 /r"    , "AVX512_F-VL"],
        ["vpcmpeqq"         , "W:k {k},~zmm,~zmm/m512/b64"                      , "RVM-FVM" , "EVEX.512.66.0F38.W1 29 /r"    , "AVX512_F"],
        ["vpcmpeqw"         , "W:k {k},~xmm,~xmm/m128"                          , "RVM-FV"  , "EVEX.128.66.0F.WIG 75 /r"     , "AVX512_BW-VL"],
        ["vpcmpeqw"         , "W:k {k},~ymm,~ymm/m256"                          , "RVM-FV"  , "EVEX.256.66.0F.WIG 75 /r"     , "AVX512_BW-VL"],
        ["vpcmpeqw"         , "W:k {k},~zmm,~zmm/m512"                          , "RVM-FV"  , "EVEX.512.66.0F.WIG 75 /r"     , "AVX512_BW"],
        ["vpcmpgtb"         , "W:k {k}, xmm, xmm/m128"                          , "RVM-FV"  , "EVEX.128.66.0F.WIG 64 /r"     , "AVX512_BW-VL"],
        ["vpcmpgtb"         , "W:k {k}, ymm, ymm/m256"                          , "RVM-FV"  , "EVEX.256.66.0F.WIG 64 /r"     , "AVX512_BW-VL"],
        ["vpcmpgtb"         , "W:k {k}, zmm, zmm/m512"                          , "RVM-FV"  , "EVEX.512.66.0F.WIG 64 /r"     , "AVX512_BW"],
        ["vpcmpgtd"         , "W:k {k}, xmm, xmm/m128/b32"                      , "RVM-FVM" , "EVEX.128.66.0F.W0 66 /r"      , "AVX512_F-VL"],
        ["vpcmpgtd"         , "W:k {k}, ymm, ymm/m256/b32"                      , "RVM-FVM" , "EVEX.256.66.0F.W0 66 /r"      , "AVX512_F-VL"],
        ["vpcmpgtd"         , "W:k {k}, zmm, zmm/m512/b32"                      , "RVM-FVM" , "EVEX.512.66.0F.W0 66 /r"      , "AVX512_F"],
        ["vpcmpgtq"         , "W:k {k}, xmm, xmm/m128/b64"                      , "RVM-FVM" , "EVEX.128.66.0F38.W1 37 /r"    , "AVX512_F-VL"],
        ["vpcmpgtq"         , "W:k {k}, ymm, ymm/m256/b64"                      , "RVM-FVM" , "EVEX.256.66.0F38.W1 37 /r"    , "AVX512_F-VL"],
        ["vpcmpgtq"         , "W:k {k}, zmm, zmm/m512/b64"                      , "RVM-FVM" , "EVEX.512.66.0F38.W1 37 /r"    , "AVX512_F"],
        ["vpcmpgtw"         , "W:k {k}, xmm, xmm/m128"                          , "RVM-FV"  , "EVEX.128.66.0F.WIG 65 /r"     , "AVX512_BW-VL"],
        ["vpcmpgtw"         , "W:k {k}, ymm, ymm/m256"                          , "RVM-FV"  , "EVEX.256.66.0F.WIG 65 /r"     , "AVX512_BW-VL"],
        ["vpcmpgtw"         , "W:k {k}, zmm, zmm/m512"                          , "RVM-FV"  , "EVEX.512.66.0F.WIG 65 /r"     , "AVX512_BW"],
        ["vpcmpq"           , "W:k {k}, xmm, xmm/m128/b64, ib/ub"               , "RVMI-FV" , "EVEX.128.66.0F3A.W1 1F /r ib" , "AVX512_F-VL"],
        ["vpcmpq"           , "W:k {k}, ymm, ymm/m256/b64, ib/ub"               , "RVMI-FV" , "EVEX.256.66.0F3A.W1 1F /r ib" , "AVX512_F-VL"],
        ["vpcmpq"           , "W:k {k}, zmm, zmm/m512/b64, ib/ub"               , "RVMI-FV" , "EVEX.512.66.0F3A.W1 1F /r ib" , "AVX512_F"],
        ["vpcmpub"          , "W:k {k}, xmm, xmm/m128, ib/ub"                   , "RVMI-FVM", "EVEX.128.66.0F3A.W0 3E /r ib" , "AVX512_BW-VL"],
        ["vpcmpub"          , "W:k {k}, ymm, ymm/m256, ib/ub"                   , "RVMI-FVM", "EVEX.256.66.0F3A.W0 3E /r ib" , "AVX512_BW-VL"],
        ["vpcmpub"          , "W:k {k}, zmm, zmm/m512, ib/ub"                   , "RVMI-FVM", "EVEX.512.66.0F3A.W0 3E /r ib" , "AVX512_BW"],
        ["vpcmpud"          , "W:k {k}, xmm, xmm/m128/b32, ib/ub"               , "RVMI-FV" , "EVEX.128.66.0F3A.W0 1E /r ib" , "AVX512_F-VL"],
        ["vpcmpud"          , "W:k {k}, ymm, ymm/m256/b32, ib/ub"               , "RVMI-FV" , "EVEX.256.66.0F3A.W0 1E /r ib" , "AVX512_F-VL"],
        ["vpcmpud"          , "W:k {k}, zmm, zmm/m512/b32, ib/ub"               , "RVMI-FV" , "EVEX.512.66.0F3A.W0 1E /r ib" , "AVX512_F"],
        ["vpcmpuq"          , "W:k {k}, xmm, xmm/m128/b64, ib/ub"               , "RVMI-FV" , "EVEX.128.66.0F3A.W1 1E /r ib" , "AVX512_F-VL"],
        ["vpcmpuq"          , "W:k {k}, ymm, ymm/m256/b64, ib/ub"               , "RVMI-FV" , "EVEX.256.66.0F3A.W1 1E /r ib" , "AVX512_F-VL"],
        ["vpcmpuq"          , "W:k {k}, zmm, zmm/m512/b64, ib/ub"               , "RVMI-FV" , "EVEX.512.66.0F3A.W1 1E /r ib" , "AVX512_F"],
        ["vpcmpuw"          , "W:k {k}, xmm, xmm/m128/b64, ib/ub"               , "RVMI-FVM", "EVEX.128.66.0F3A.W1 3E /r ib" , "AVX512_BW-VL"],
        ["vpcmpuw"          , "W:k {k}, ymm, ymm/m256/b64, ib/ub"               , "RVMI-FVM", "EVEX.256.66.0F3A.W1 3E /r ib" , "AVX512_BW-VL"],
        ["vpcmpuw"          , "W:k {k}, zmm, zmm/m512/b64, ib/ub"               , "RVMI-FVM", "EVEX.512.66.0F3A.W1 3E /r ib" , "AVX512_BW"],
        ["vpcmpw"           , "W:k {k}, xmm, xmm/m128/b64, ib/ub"               , "RVMI-FVM", "EVEX.128.66.0F3A.W1 3F /r ib" , "AVX512_BW-VL"],
        ["vpcmpw"           , "W:k {k}, ymm, ymm/m256/b64, ib/ub"               , "RVMI-FVM", "EVEX.256.66.0F3A.W1 3F /r ib" , "AVX512_BW-VL"],
        ["vpcmpw"           , "W:k {k}, zmm, zmm/m512/b64, ib/ub"               , "RVMI-FVM", "EVEX.512.66.0F3A.W1 3F /r ib" , "AVX512_BW"],
        ["vpcompressb"      , "W:xmm/m128 {kz}, xmm"                            , "RVM-T1S" , "EVEX.128.66.0F38.W0 63 /r"    , "AVX512_VBMI2-VL"],
        ["vpcompressb"      , "W:ymm/m256 {kz}, ymm"                            , "RVM-T1S" , "EVEX.256.66.0F38.W0 63 /r"    , "AVX512_VBMI2-VL"],
        ["vpcompressb"      , "W:zmm/m512 {kz}, zmm"                            , "RVM-T1S" , "EVEX.512.66.0F38.W0 63 /r"    , "AVX512_VBMI2"],
        ["vpcompressw"      , "W:xmm/m128 {kz}, xmm"                            , "RVM-T1S" , "EVEX.128.66.0F38.W1 63 /r"    , "AVX512_VBMI2-VL"],
        ["vpcompressw"      , "W:ymm/m256 {kz}, ymm"                            , "RVM-T1S" , "EVEX.256.66.0F38.W1 63 /r"    , "AVX512_VBMI2-VL"],
        ["vpcompressw"      , "W:zmm/m512 {kz}, zmm"                            , "RVM-T1S" , "EVEX.512.66.0F38.W1 63 /r"    , "AVX512_VBMI2"],
        ["vpcompressd"      , "W:xmm/m128 {kz}, xmm"                            , "MR-T1S"  , "EVEX.128.66.0F38.W0 8B /r"    , "AVX512_F-VL"],
        ["vpcompressd"      , "W:ymm/m256 {kz}, ymm"                            , "MR-T1S"  , "EVEX.256.66.0F38.W0 8B /r"    , "AVX512_F-VL"],
        ["vpcompressd"      , "W:zmm/m512 {kz}, zmm"                            , "MR-T1S"  , "EVEX.512.66.0F38.W0 8B /r"    , "AVX512_F"],
        ["vpcompressq"      , "W:xmm/m128 {kz}, xmm"                            , "MR-T1S"  , "EVEX.128.66.0F38.W1 8B /r"    , "AVX512_F-VL"],
        ["vpcompressq"      , "W:ymm/m256 {kz}, ymm"                            , "MR-T1S"  , "EVEX.256.66.0F38.W1 8B /r"    , "AVX512_F-VL"],
        ["vpcompressq"      , "W:zmm/m512 {kz}, zmm"                            , "MR-T1S"  , "EVEX.512.66.0F38.W1 8B /r"    , "AVX512_F"],
        ["vpconflictd"      , "W:xmm {kz}, xmm/m128/b32"                        , "RM-FV"   , "EVEX.128.66.0F38.W0 C4 /r"    , "AVX512_CDI-VL"],
        ["vpconflictd"      , "W:ymm {kz}, ymm/m256/b32"                        , "RM-FV"   , "EVEX.256.66.0F38.W0 C4 /r"    , "AVX512_CDI-VL"],
        ["vpconflictd"      , "W:zmm {kz}, zmm/m512/b32"                        , "RM-FV"   , "EVEX.512.66.0F38.W0 C4 /r"    , "AVX512_CDI"],
        ["vpconflictq"      , "W:xmm {kz}, xmm/m128/b32"                        , "RM-FV"   , "EVEX.128.66.0F38.W1 C4 /r"    , "AVX512_CDI-VL"],
        ["vpconflictq"      , "W:ymm {kz}, ymm/m256/b32"                        , "RM-FV"   , "EVEX.256.66.0F38.W1 C4 /r"    , "AVX512_CDI-VL"],
        ["vpconflictq"      , "W:zmm {kz}, zmm/m512/b32"                        , "RM-FV"   , "EVEX.512.66.0F38.W1 C4 /r"    , "AVX512_CDI"],
        ["vpdpbusd"         , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 50 /r"    , "AVX512_VNNI-VL"],
        ["vpdpbusd"         , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 50 /r"    , "AVX512_VNNI-VL"],
        ["vpdpbusd"         , "X:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 50 /r"    , "AVX512_VNNI"],
        ["vpdpbusds"        , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 51 /r"    , "AVX512_VNNI-VL"],
        ["vpdpbusds"        , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 51 /r"    , "AVX512_VNNI-VL"],
        ["vpdpbusds"        , "X:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 51 /r"    , "AVX512_VNNI"],
        ["vpdpwssd"         , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 52 /r"    , "AVX512_VNNI-VL"],
        ["vpdpwssd"         , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 52 /r"    , "AVX512_VNNI-VL"],
        ["vpdpwssd"         , "X:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 52 /r"    , "AVX512_VNNI"],
        ["vpdpwssds"        , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 53 /r"    , "AVX512_VNNI-VL"],
        ["vpdpwssds"        , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 53 /r"    , "AVX512_VNNI-VL"],
        ["vpdpwssds"        , "X:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 53 /r"    , "AVX512_VNNI"],
        ["vpexpandb"        , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.66.0F38.W0 62 /r"    , "AVX512_VBMI2-VL"],
        ["vpexpandb"        , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.66.0F38.W0 62 /r"    , "AVX512_VBMI2-VL"],
        ["vpexpandb"        , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.66.0F38.W0 62 /r"    , "AVX512_VBMI2"],
        ["vpexpandw"        , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.66.0F38.W1 62 /r"    , "AVX512_VBMI2-VL"],
        ["vpexpandw"        , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.66.0F38.W1 62 /r"    , "AVX512_VBMI2-VL"],
        ["vpexpandw"        , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.66.0F38.W1 62 /r"    , "AVX512_VBMI2"],
        ["vpermb"           , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.W0 8D /r"    , "AVX512_VBMI-VL"],
        ["vpermb"           , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.W0 8D /r"    , "AVX512_VBMI-VL"],
        ["vpermb"           , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.W0 8D /r"    , "AVX512_VBMI"],
        ["vpermd"           , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 36 /r"    , "AVX512_F-VL"],
        ["vpermd"           , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 36 /r"    , "AVX512_F"],
        ["vpermi2b"         , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.W0 75 /r"    , "AVX512_VBMI-VL"],
        ["vpermi2b"         , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.W0 75 /r"    , "AVX512_VBMI-VL"],
        ["vpermi2b"         , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.W0 75 /r"    , "AVX512_VBMI"],
        ["vpermi2d"         , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 76 /r"    , "AVX512_F-VL"],
        ["vpermi2d"         , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 76 /r"    , "AVX512_F-VL"],
        ["vpermi2d"         , "X:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 76 /r"    , "AVX512_F"],
        ["vpermi2pd"        , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 77 /r"    , "AVX512_F-VL"],
        ["vpermi2pd"        , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 77 /r"    , "AVX512_F-VL"],
        ["vpermi2pd"        , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 77 /r"    , "AVX512_F"],
        ["vpermi2ps"        , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 77 /r"    , "AVX512_F-VL"],
        ["vpermi2ps"        , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 77 /r"    , "AVX512_F-VL"],
        ["vpermi2ps"        , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 77 /r"    , "AVX512_F"],
        ["vpermi2q"         , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 76 /r"    , "AVX512_F-VL"],
        ["vpermi2q"         , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 76 /r"    , "AVX512_F-VL"],
        ["vpermi2q"         , "X:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 76 /r"    , "AVX512_F"],
        ["vpermi2w"         , "X:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.W1 75 /r"    , "AVX512_BW-VL"],
        ["vpermi2w"         , "X:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.W1 75 /r"    , "AVX512_BW-VL"],
        ["vpermi2w"         , "X:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.W1 75 /r"    , "AVX512_BW"],
        ["vpermilpd"        , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 0D /r"    , "AVX512_F-VL"],
        ["vpermilpd"        , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 0D /r"    , "AVX512_F-VL"],
        ["vpermilpd"        , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 0D /r"    , "AVX512_F"],
        ["vpermilpd"        , "W:xmm {kz}, xmm/m128/b64, ib/ub"                 , "RMI-FV"  , "EVEX.128.66.0F3A.W1 05 /r ib" , "AVX512_F-VL"],
        ["vpermilpd"        , "W:ymm {kz}, ymm/m256/b64, ib/ub"                 , "RMI-FV"  , "EVEX.256.66.0F3A.W1 05 /r ib" , "AVX512_F-VL"],
        ["vpermilpd"        , "W:zmm {kz}, zmm/m512/b64, ib/ub"                 , "RMI-FV"  , "EVEX.512.66.0F3A.W1 05 /r ib" , "AVX512_F"],
        ["vpermilps"        , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 0C /r"    , "AVX512_F-VL"],
        ["vpermilps"        , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 0C /r"    , "AVX512_F-VL"],
        ["vpermilps"        , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 0C /r"    , "AVX512_F"],
        ["vpermilps"        , "W:xmm {kz}, xmm/m128/b64, ib/ub"                 , "RMI-FV"  , "EVEX.128.66.0F3A.W0 04 /r ib" , "AVX512_F-VL"],
        ["vpermilps"        , "W:ymm {kz}, ymm/m256/b64, ib/ub"                 , "RMI-FV"  , "EVEX.256.66.0F3A.W0 04 /r ib" , "AVX512_F-VL"],
        ["vpermilps"        , "W:zmm {kz}, zmm/m512/b64, ib/ub"                 , "RMI-FV"  , "EVEX.512.66.0F3A.W0 04 /r ib" , "AVX512_F"],
        ["vpermq"           , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 36 /r"    , "AVX512_F-VL"],
        ["vpermq"           , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 36 /r"    , "AVX512_F"],
        ["vpermq"           , "W:ymm {kz}, ymm/m256/b64, ib/ub"                 , "RMI-FV"  , "EVEX.256.66.0F3A.W1 00 /r ib" , "AVX512_F-VL"],
        ["vpermq"           , "W:zmm {kz}, zmm/m512/b64, ib/ub"                 , "RMI-FV"  , "EVEX.512.66.0F3A.W1 00 /r ib" , "AVX512_F"],
        ["vpermt2b"         , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.W0 7D /r"    , "AVX512_VBMI-VL"],
        ["vpermt2b"         , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.W0 7D /r"    , "AVX512_VBMI-VL"],
        ["vpermt2b"         , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.W0 7D /r"    , "AVX512_VBMI"],
        ["vpermt2d"         , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 7E /r"    , "AVX512_F-VL"],
        ["vpermt2d"         , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 7E /r"    , "AVX512_F-VL"],
        ["vpermt2d"         , "X:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 7E /r"    , "AVX512_F"],
        ["vpermt2pd"        , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 7F /r"    , "AVX512_F-VL"],
        ["vpermt2pd"        , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 7F /r"    , "AVX512_F-VL"],
        ["vpermt2pd"        , "X:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 7F /r"    , "AVX512_F"],
        ["vpermt2ps"        , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 7F /r"    , "AVX512_F-VL"],
        ["vpermt2ps"        , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 7F /r"    , "AVX512_F-VL"],
        ["vpermt2ps"        , "X:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 7F /r"    , "AVX512_F"],
        ["vpermt2q"         , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 7E /r"    , "AVX512_F-VL"],
        ["vpermt2q"         , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 7E /r"    , "AVX512_F-VL"],
        ["vpermt2q"         , "X:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 7E /r"    , "AVX512_F"],
        ["vpermt2w"         , "X:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.W1 7D /r"    , "AVX512_BW-VL"],
        ["vpermt2w"         , "X:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.W1 7D /r"    , "AVX512_BW-VL"],
        ["vpermt2w"         , "X:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.W1 7D /r"    , "AVX512_BW"],
        ["vpermw"           , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.W1 8D /r"    , "AVX512_BW-VL"],
        ["vpermw"           , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.W1 8D /r"    , "AVX512_BW-VL"],
        ["vpermw"           , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.W1 8D /r"    , "AVX512_BW"],
        ["vpexpandd"        , "W:xmm {kz}, xmm/m128"                            , "RM-T1S"  , "EVEX.128.66.0F38.W0 89 /r"    , "AVX512_F-VL"],
        ["vpexpandd"        , "W:ymm {kz}, ymm/m256"                            , "RM-T1S"  , "EVEX.256.66.0F38.W0 89 /r"    , "AVX512_F-VL"],
        ["vpexpandd"        , "W:zmm {kz}, zmm/m512"                            , "RM-T1S"  , "EVEX.512.66.0F38.W0 89 /r"    , "AVX512_F"],
        ["vpexpandq"        , "W:xmm {kz}, xmm/m128"                            , "RM-T1S"  , "EVEX.128.66.0F38.W1 89 /r"    , "AVX512_F-VL"],
        ["vpexpandq"        , "W:ymm {kz}, ymm/m256"                            , "RM-T1S"  , "EVEX.256.66.0F38.W1 89 /r"    , "AVX512_F-VL"],
        ["vpexpandq"        , "W:zmm {kz}, zmm/m512"                            , "RM-T1S"  , "EVEX.512.66.0F38.W1 89 /r"    , "AVX512_F"],
        ["vpextrb"          , "W:r32[7:0]/m8 , xmm, ib/ub"                      , "MRI-T1S" , "EVEX.128.66.0F3A.WIG 14 /r ib", "AVX512_BW"],
        ["vpextrd"          , "W:r32/m32, xmm, ib/ub"                           , "MRI-T1S" , "EVEX.128.66.0F3A.W0 16 /r ib" , "AVX512_DQ"],
        ["vpextrq"          , "W:r64/m64, xmm, ib/ub"                           , "MRI-T1S" , "EVEX.128.66.0F3A.W1 16 /r ib" , "AVX512_DQ X64"],
        ["vpextrw"          , "W:r32[15:0], xmm, ib/ub"                         , "RMI"     , "EVEX.128.66.0F.WIG C5 /r ib"  , "AVX512_BW"],
        ["vpextrw"          , "W:r32[15:0]/m16, xmm, ib/ub"                     , "MRI-T1S" , "EVEX.128.66.0F3A.WIG 15 /r ib", "AVX512_BW"],
        ["vpgatherdd"       , "X:xmm {k}, vm32x"                                , "RM-T1S"  , "EVEX.128.66.0F38.W0 90"       , "AVX512_F-VL"],
        ["vpgatherdd"       , "X:ymm {k}, vm32y"                                , "RM-T1S"  , "EVEX.256.66.0F38.W0 90"       , "AVX512_F-VL"],
        ["vpgatherdd"       , "X:zmm {k}, vm32z"                                , "RM-T1S"  , "EVEX.512.66.0F38.W0 90"       , "AVX512_F"],
        ["vpgatherdq"       , "X:xmm {k}, vm32x"                                , "RM-T1S"  , "EVEX.128.66.0F38.W1 90"       , "AVX512_F-VL"],
        ["vpgatherdq"       , "X:ymm {k}, vm32y"                                , "RM-T1S"  , "EVEX.256.66.0F38.W1 90"       , "AVX512_F-VL"],
        ["vpgatherdq"       , "X:zmm {k}, vm32z"                                , "RM-T1S"  , "EVEX.512.66.0F38.W1 90"       , "AVX512_F"],
        ["vpgatherqd"       , "X:xmm {k}, vm64x"                                , "RM-T1S"  , "EVEX.128.66.0F38.W0 91"       , "AVX512_F-VL"],
        ["vpgatherqd"       , "X:ymm {k}, vm64y"                                , "RM-T1S"  , "EVEX.256.66.0F38.W0 91"       , "AVX512_F-VL"],
        ["vpgatherqd"       , "X:zmm {k}, vm64z"                                , "RM-T1S"  , "EVEX.512.66.0F38.W0 91"       , "AVX512_F"],
        ["vpgatherqq"       , "X:xmm {k}, vm64x"                                , "RM-T1S"  , "EVEX.128.66.0F38.W1 91"       , "AVX512_F-VL"],
        ["vpgatherqq"       , "X:ymm {k}, vm64y"                                , "RM-T1S"  , "EVEX.256.66.0F38.W1 91"       , "AVX512_F-VL"],
        ["vpgatherqq"       , "X:zmm {k}, vm64z"                                , "RM-T1S"  , "EVEX.512.66.0F38.W1 91"       , "AVX512_F"],
        ["vpinsrb"          , "W:xmm {kz}, xmm, r32[7:0]/m8, ib/ub"             , "RVMI-T1S", "EVEX.128.66.0F3A.WIG 20 /r ib", "AVX512_BW"],
        ["vpinsrd"          , "W:xmm {kz}, xmm, r32/m32, ib/ub"                 , "RVMI-T1S", "EVEX.128.66.0F3A.W0 22 /r ib" , "AVX512_DQ"],
        ["vpinsrq"          , "W:xmm {kz}, xmm, r64/m64, ib/ub"                 , "RVMI-T1S", "EVEX.128.66.0F3A.W1 22 /r ib" , "AVX512_DQ X64"],
        ["vpinsrw"          , "W:xmm {kz}, xmm, r32[15:0]/m16, ib/ub"           , "RVMI-T1S", "EVEX.128.66.0F.WIG C4 /r ib"  , "AVX512_BW"],
        ["vplzcntd"         , "W:xmm {kz}, xmm/m128/b32"                        , "RM-FV"   , "EVEX.128.66.0F38.W0 44 /r"    , "AVX512_CDI-VL"],
        ["vplzcntd"         , "W:ymm {kz}, ymm/m256/b32"                        , "RM-FV"   , "EVEX.256.66.0F38.W0 44 /r"    , "AVX512_CDI-VL"],
        ["vplzcntd"         , "W:zmm {kz}, zmm/m512/b32"                        , "RM-FV"   , "EVEX.512.66.0F38.W0 44 /r"    , "AVX512_CDI"],
        ["vplzcntq"         , "W:xmm {kz}, xmm/m128/b64"                        , "RM-FV"   , "EVEX.128.66.0F38.W1 44 /r"    , "AVX512_CDI-VL"],
        ["vplzcntq"         , "W:ymm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.66.0F38.W1 44 /r"    , "AVX512_CDI-VL"],
        ["vplzcntq"         , "W:zmm {kz}, zmm/m512/b64"                        , "RM-FV"   , "EVEX.512.66.0F38.W1 44 /r"    , "AVX512_CDI"],
        ["vpmadd52luq"      , "X:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 B4 /r"    , "AVX512_IFMA-VL"],
        ["vpmadd52luq"      , "X:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 B4 /r"    , "AVX512_IFMA-VL"],
        ["vpmadd52luq"      , "X:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 B4 /r"    , "AVX512_IFMA"],
        ["vpmadd52huq"      , "X:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 B5 /r"    , "AVX512_IFMA-VL"],
        ["vpmadd52huq"      , "X:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 B5 /r"    , "AVX512_IFMA-VL"],
        ["vpmadd52huq"      , "X:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 B5 /r"    , "AVX512_IFMA"],
        ["vpmaddubsw"       , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.WIG 04 /r"   , "AVX512_BW-VL"],
        ["vpmaddubsw"       , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.WIG 04 /r"   , "AVX512_BW-VL"],
        ["vpmaddubsw"       , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.WIG 04 /r"   , "AVX512_BW"],
        ["vpmaddwd"         , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG F5 /r"     , "AVX512_BW-VL"],
        ["vpmaddwd"         , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG F5 /r"     , "AVX512_BW-VL"],
        ["vpmaddwd"         , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG F5 /r"     , "AVX512_BW"],
        ["vpmaxsb"          , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.WIG 3C /r"   , "AVX512_BW-VL"],
        ["vpmaxsb"          , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.WIG 3C /r"   , "AVX512_BW-VL"],
        ["vpmaxsb"          , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.WIG 3C /r"   , "AVX512_BW"],
        ["vpmaxsd"          , "W:xmm {kz},~xmm,~xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 3D /r"    , "AVX512_F-VL"],
        ["vpmaxsd"          , "W:ymm {kz},~ymm,~ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 3D /r"    , "AVX512_F-VL"],
        ["vpmaxsd"          , "W:zmm {kz},~zmm,~zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 3D /r"    , "AVX512_F"],
        ["vpmaxsq"          , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 3D /r"    , "AVX512_F-VL"],
        ["vpmaxsq"          , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 3D /r"    , "AVX512_F-VL"],
        ["vpmaxsq"          , "W:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 3D /r"    , "AVX512_F"],
        ["vpmaxsw"          , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG EE /r"     , "AVX512_BW-VL"],
        ["vpmaxsw"          , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG EE /r"     , "AVX512_BW-VL"],
        ["vpmaxsw"          , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG EE /r"     , "AVX512_BW"],
        ["vpmaxub"          , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG DE /r"     , "AVX512_BW-VL"],
        ["vpmaxub"          , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG DE /r"     , "AVX512_BW-VL"],
        ["vpmaxub"          , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG DE /r"     , "AVX512_BW"],
        ["vpmaxud"          , "W:xmm {kz},~xmm,~xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 3F /r"    , "AVX512_F-VL"],
        ["vpmaxud"          , "W:ymm {kz},~ymm,~ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 3F /r"    , "AVX512_F-VL"],
        ["vpmaxud"          , "W:zmm {kz},~zmm,~zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 3F /r"    , "AVX512_F"],
        ["vpmaxuq"          , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 3F /r"    , "AVX512_F-VL"],
        ["vpmaxuq"          , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 3F /r"    , "AVX512_F-VL"],
        ["vpmaxuq"          , "W:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 3F /r"    , "AVX512_F"],
        ["vpmaxuw"          , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.WIG 3E /r"   , "AVX512_BW-VL"],
        ["vpmaxuw"          , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.WIG 3E /r"   , "AVX512_BW-VL"],
        ["vpmaxuw"          , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.WIG 3E /r"   , "AVX512_BW"],
        ["vpminsb"          , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.WIG 38 /r"   , "AVX512_BW-VL"],
        ["vpminsb"          , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.WIG 38 /r"   , "AVX512_BW-VL"],
        ["vpminsb"          , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.WIG 38 /r"   , "AVX512_BW"],
        ["vpminsd"          , "W:xmm {kz},~xmm,~xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 39 /r"    , "AVX512_F-VL"],
        ["vpminsd"          , "W:ymm {kz},~ymm,~ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 39 /r"    , "AVX512_F-VL"],
        ["vpminsd"          , "W:zmm {kz},~zmm,~zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 39 /r"    , "AVX512_F"],
        ["vpminsq"          , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 39 /r"    , "AVX512_F-VL"],
        ["vpminsq"          , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 39 /r"    , "AVX512_F-VL"],
        ["vpminsq"          , "W:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 39 /r"    , "AVX512_F"],
        ["vpminsw"          , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG EA /r"     , "AVX512_BW-VL"],
        ["vpminsw"          , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG EA /r"     , "AVX512_BW-VL"],
        ["vpminsw"          , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG EA /r"     , "AVX512_BW"],
        ["vpminub"          , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F DA /r"         , "AVX512_BW-VL"],
        ["vpminub"          , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F DA /r"         , "AVX512_BW-VL"],
        ["vpminub"          , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F DA /r"         , "AVX512_BW"],
        ["vpminud"          , "W:xmm {kz},~xmm,~xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 3B /r"    , "AVX512_F-VL"],
        ["vpminud"          , "W:ymm {kz},~ymm,~ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 3B /r"    , "AVX512_F-VL"],
        ["vpminud"          , "W:zmm {kz},~zmm,~zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 3B /r"    , "AVX512_F"],
        ["vpminuq"          , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 3B /r"    , "AVX512_F-VL"],
        ["vpminuq"          , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 3B /r"    , "AVX512_F-VL"],
        ["vpminuq"          , "W:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 3B /r"    , "AVX512_F"],
        ["vpminuw"          , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38 3A /r"       , "AVX512_BW-VL"],
        ["vpminuw"          , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38 3A /r"       , "AVX512_BW-VL"],
        ["vpminuw"          , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38 3A /r"       , "AVX512_BW"],
        ["vpmovb2m"         , "W:k, xmm"                                        , "RM"      , "EVEX.128.F3.0F38.W0 29 /r"    , "AVX512_BW-VL"],
        ["vpmovb2m"         , "W:k, ymm"                                        , "RM"      , "EVEX.256.F3.0F38.W0 29 /r"    , "AVX512_BW-VL"],
        ["vpmovb2m"         , "W:k, zmm"                                        , "RM"      , "EVEX.512.F3.0F38.W0 29 /r"    , "AVX512_BW"],
        ["vpmovd2m"         , "W:k, xmm"                                        , "RM"      , "EVEX.128.F3.0F38.W0 39 /r"    , "AVX512_DQ-VL"],
        ["vpmovd2m"         , "W:k, ymm"                                        , "RM"      , "EVEX.256.F3.0F38.W0 39 /r"    , "AVX512_DQ-VL"],
        ["vpmovd2m"         , "W:k, zmm"                                        , "RM"      , "EVEX.512.F3.0F38.W0 39 /r"    , "AVX512_DQ"],
        ["vpmovdb"          , "W:xmm[31:0]/m32 {kz}, xmm"                       , "MR-QVM"  , "EVEX.128.F3.0F38.W0 31 /r"    , "AVX512_F-VL"],
        ["vpmovdb"          , "W:xmm[63:0]/m64 {kz}, ymm"                       , "MR-QVM"  , "EVEX.256.F3.0F38.W0 31 /r"    , "AVX512_F-VL"],
        ["vpmovdb"          , "W:xmm/m128 {kz}, zmm"                            , "MR-QVM"  , "EVEX.512.F3.0F38.W0 31 /r"    , "AVX512_F"],
        ["vpmovdw"          , "W:xmm[63:0]/m64 {kz}, xmm"                       , "MR-HVM"  , "EVEX.128.F3.0F38.W0 33 /r"    , "AVX512_F-VL"],
        ["vpmovdw"          , "W:xmm/m128 {kz}, ymm"                            , "MR-HVM"  , "EVEX.256.F3.0F38.W0 33 /r"    , "AVX512_F-VL"],
        ["vpmovdw"          , "W:ymm/m256 {kz}, zmm"                            , "MR-HVM"  , "EVEX.512.F3.0F38.W0 33 /r"    , "AVX512_F"],
        ["vpmovm2b"         , "W:xmm, k"                                        , "RM"      , "EVEX.128.F3.0F38.W0 28 /r"    , "AVX512_BW-VL"],
        ["vpmovm2b"         , "W:ymm, k"                                        , "RM"      , "EVEX.256.F3.0F38.W0 28 /r"    , "AVX512_BW-VL"],
        ["vpmovm2b"         , "W:zmm, k"                                        , "RM"      , "EVEX.512.F3.0F38.W0 28 /r"    , "AVX512_BW"],
        ["vpmovm2d"         , "W:xmm, k"                                        , "RM"      , "EVEX.128.F3.0F38.W0 38 /r"    , "AVX512_DQ-VL"],
        ["vpmovm2d"         , "W:ymm, k"                                        , "RM"      , "EVEX.256.F3.0F38.W0 38 /r"    , "AVX512_DQ-VL"],
        ["vpmovm2d"         , "W:zmm, k"                                        , "RM"      , "EVEX.512.F3.0F38.W0 38 /r"    , "AVX512_DQ"],
        ["vpmovm2q"         , "W:xmm, k"                                        , "RM"      , "EVEX.128.F3.0F38.W1 38 /r"    , "AVX512_DQ-VL"],
        ["vpmovm2q"         , "W:ymm, k"                                        , "RM"      , "EVEX.256.F3.0F38.W1 38 /r"    , "AVX512_DQ-VL"],
        ["vpmovm2q"         , "W:zmm, k"                                        , "RM"      , "EVEX.512.F3.0F38.W1 38 /r"    , "AVX512_DQ"],
        ["vpmovm2w"         , "W:xmm, k"                                        , "RM"      , "EVEX.128.F3.0F38.W1 28 /r"    , "AVX512_BW-VL"],
        ["vpmovm2w"         , "W:ymm, k"                                        , "RM"      , "EVEX.256.F3.0F38.W1 28 /r"    , "AVX512_BW-VL"],
        ["vpmovm2w"         , "W:zmm, k"                                        , "RM"      , "EVEX.512.F3.0F38.W1 28 /r"    , "AVX512_BW"],
        ["vpmovq2m"         , "W:k, xmm"                                        , "RM"      , "EVEX.128.F3.0F38.W1 39 /r"    , "AVX512_DQ-VL"],
        ["vpmovq2m"         , "W:k, ymm"                                        , "RM"      , "EVEX.256.F3.0F38.W1 39 /r"    , "AVX512_DQ-VL"],
        ["vpmovq2m"         , "W:k, zmm"                                        , "RM"      , "EVEX.512.F3.0F38.W1 39 /r"    , "AVX512_DQ"],
        ["vpmovqb"          , "W:xmm[15:0]/m16 {kz}, xmm"                       , "MR-OVM"  , "EVEX.128.F3.0F38.W0 32 /r"    , "AVX512_F-VL"],
        ["vpmovqb"          , "W:xmm[31:0]/m32 {kz}, ymm"                       , "MR-OVM"  , "EVEX.256.F3.0F38.W0 32 /r"    , "AVX512_F-VL"],
        ["vpmovqb"          , "W:xmm[63:0]/m64 {kz}, zmm"                       , "MR-OVM"  , "EVEX.512.F3.0F38.W0 32 /r"    , "AVX512_F"],
        ["vpmovqd"          , "W:xmm[63:0]/m64 {kz}, xmm"                       , "MR-HVM"  , "EVEX.128.F3.0F38.W0 35 /r"    , "AVX512_F-VL"],
        ["vpmovqd"          , "W:xmm/m128 {kz}, ymm"                            , "MR-HVM"  , "EVEX.256.F3.0F38.W0 35 /r"    , "AVX512_F-VL"],
        ["vpmovqd"          , "W:ymm/m256 {kz}, zmm"                            , "MR-HVM"  , "EVEX.512.F3.0F38.W0 35 /r"    , "AVX512_F"],
        ["vpmovqw"          , "W:xmm[31:0]/m32 {kz}, xmm"                       , "MR-QVM"  , "EVEX.128.F3.0F38.W0 34 /r"    , "AVX512_F-VL"],
        ["vpmovqw"          , "W:xmm[63:0]/m64 {kz}, ymm"                       , "MR-QVM"  , "EVEX.256.F3.0F38.W0 34 /r"    , "AVX512_F-VL"],
        ["vpmovqw"          , "W:xmm/m128 {kz}, zmm"                            , "MR-QVM"  , "EVEX.512.F3.0F38.W0 34 /r"    , "AVX512_F"],
        ["vpmovsdb"         , "W:xmm[31:0]/m32 {kz}, xmm"                       , "MR-QVM"  , "EVEX.128.F3.0F38.W0 21 /r"    , "AVX512_F-VL"],
        ["vpmovsdb"         , "W:xmm[63:0]/m64 {kz}, ymm"                       , "MR-QVM"  , "EVEX.256.F3.0F38.W0 21 /r"    , "AVX512_F-VL"],
        ["vpmovsdb"         , "W:xmm/m128 {kz}, zmm"                            , "MR-QVM"  , "EVEX.512.F3.0F38.W0 21 /r"    , "AVX512_F"],
        ["vpmovsdw"         , "W:xmm[63:0]/m64 {kz}, xmm"                       , "MR-HVM"  , "EVEX.128.F3.0F38.W0 23 /r"    , "AVX512_F-VL"],
        ["vpmovsdw"         , "W:xmm/m128 {kz}, ymm"                            , "MR-HVM"  , "EVEX.256.F3.0F38.W0 23 /r"    , "AVX512_F-VL"],
        ["vpmovsdw"         , "W:ymm/m256 {kz}, zmm"                            , "MR-HVM"  , "EVEX.512.F3.0F38.W0 23 /r"    , "AVX512_F"],
        ["vpmovsqb"         , "W:xmm[15:0]/m16 {kz}, xmm"                       , "MR-OVM"  , "EVEX.128.F3.0F38.W0 22 /r"    , "AVX512_F-VL"],
        ["vpmovsqb"         , "W:xmm[31:0]/m32 {kz}, ymm"                       , "MR-OVM"  , "EVEX.256.F3.0F38.W0 22 /r"    , "AVX512_F-VL"],
        ["vpmovsqb"         , "W:xmm[63:0]/m64 {kz}, zmm"                       , "MR-OVM"  , "EVEX.512.F3.0F38.W0 22 /r"    , "AVX512_F"],
        ["vpmovsqd"         , "W:xmm[63:0]/m64 {kz}, xmm"                       , "MR-HVM"  , "EVEX.128.F3.0F38.W0 25 /r"    , "AVX512_F-VL"],
        ["vpmovsqd"         , "W:xmm/m128 {kz}, ymm"                            , "MR-HVM"  , "EVEX.256.F3.0F38.W0 25 /r"    , "AVX512_F-VL"],
        ["vpmovsqd"         , "W:ymm/m256 {kz}, zmm"                            , "MR-HVM"  , "EVEX.512.F3.0F38.W0 25 /r"    , "AVX512_F"],
        ["vpmovsqw"         , "W:xmm[31:0]/m32 {kz}, xmm"                       , "MR-QVM"  , "EVEX.128.F3.0F38.W0 24 /r"    , "AVX512_F-VL"],
        ["vpmovsqw"         , "W:xmm[63:0]/m64 {kz}, ymm"                       , "MR-QVM"  , "EVEX.256.F3.0F38.W0 24 /r"    , "AVX512_F-VL"],
        ["vpmovsqw"         , "W:xmm/m128 {kz}, zmm"                            , "MR-QVM"  , "EVEX.512.F3.0F38.W0 24 /r"    , "AVX512_F"],
        ["vpmovswb"         , "W:xmm[63:0]/m64 {kz}, xmm"                       , "MR-HVM"  , "EVEX.128.F3.0F38.W0 20 /r"    , "AVX512_BW-VL"],
        ["vpmovswb"         , "W:xmm/m128 {kz}, ymm"                            , "MR-HVM"  , "EVEX.256.F3.0F38.W0 20 /r"    , "AVX512_BW-VL"],
        ["vpmovswb"         , "W:ymm/m256 {kz}, zmm"                            , "MR-HVM"  , "EVEX.512.F3.0F38.W0 20 /r"    , "AVX512_BW"],
        ["vpmovsxbd"        , "W:xmm {kz}, xmm[31:0]/m32"                       , "RM-QVM"  , "EVEX.128.66.0F38.WIG 21 /r"   , "AVX512_F-VL"],
        ["vpmovsxbd"        , "W:ymm {kz}, xmm[63:0]/m64"                       , "RM-QVM"  , "EVEX.256.66.0F38.WIG 21 /r"   , "AVX512_F-VL"],
        ["vpmovsxbd"        , "W:zmm {kz}, xmm/m128"                            , "RM-QVM"  , "EVEX.512.66.0F38.WIG 21 /r"   , "AVX512_F"],
        ["vpmovsxbq"        , "W:xmm {kz}, xmm[15:0]/m16"                       , "RM-OVM"  , "EVEX.128.66.0F38.WIG 22 /r"   , "AVX512_F-VL"],
        ["vpmovsxbq"        , "W:ymm {kz}, xmm[31:0]/m32"                       , "RM-OVM"  , "EVEX.256.66.0F38.WIG 22 /r"   , "AVX512_F-VL"],
        ["vpmovsxbq"        , "W:zmm {kz}, xmm[63:0]/m64"                       , "RM-OVM"  , "EVEX.512.66.0F38.WIG 22 /r"   , "AVX512_F"],
        ["vpmovsxbw"        , "W:xmm {kz}, xmm[63:0]/m64"                       , "RM-HVM"  , "EVEX.128.66.0F38.WIG 20 /r"   , "AVX512_BW-VL"],
        ["vpmovsxbw"        , "W:ymm {kz}, xmm/m128"                            , "RM-HVM"  , "EVEX.256.66.0F38.WIG 20 /r"   , "AVX512_BW-VL"],
        ["vpmovsxbw"        , "W:zmm {kz}, ymm/m256"                            , "RM-HVM"  , "EVEX.512.66.0F38.WIG 20 /r"   , "AVX512_BW"],
        ["vpmovsxdq"        , "W:xmm {kz}, xmm[63:0]/m64"                       , "RM-HVM"  , "EVEX.128.66.0F38.W0 25 /r"    , "AVX512_F-VL"],
        ["vpmovsxdq"        , "W:ymm {kz}, xmm/m128"                            , "RM-HVM"  , "EVEX.256.66.0F38.W0 25 /r"    , "AVX512_F-VL"],
        ["vpmovsxdq"        , "W:zmm {kz}, ymm/m256"                            , "RM-HVM"  , "EVEX.512.66.0F38.W0 25 /r"    , "AVX512_F"],
        ["vpmovsxwd"        , "W:xmm {kz}, xmm[63:0]/m64"                       , "RM-HVM"  , "EVEX.128.66.0F38.WIG 23 /r"   , "AVX512_F-VL"],
        ["vpmovsxwd"        , "W:ymm {kz}, xmm/m128"                            , "RM-HVM"  , "EVEX.256.66.0F38.WIG 23 /r"   , "AVX512_F-VL"],
        ["vpmovsxwd"        , "W:zmm {kz}, ymm/m256"                            , "RM-HVM"  , "EVEX.512.66.0F38.WIG 23 /r"   , "AVX512_F"],
        ["vpmovsxwq"        , "W:xmm {kz}, xmm[31:0]/m32"                       , "RM-QVM"  , "EVEX.128.66.0F38.WIG 24 /r"   , "AVX512_F-VL"],
        ["vpmovsxwq"        , "W:ymm {kz}, xmm[63:0]/m64"                       , "RM-QVM"  , "EVEX.256.66.0F38.WIG 24 /r"   , "AVX512_F-VL"],
        ["vpmovsxwq"        , "W:zmm {kz}, xmm/m128"                            , "RM-QVM"  , "EVEX.512.66.0F38.WIG 24 /r"   , "AVX512_F"],
        ["vpmovusdb"        , "W:xmm[31:0]/m32 {kz}, xmm"                       , "MR-QVM"  , "EVEX.128.F3.0F38.W0 11 /r"    , "AVX512_F-VL"],
        ["vpmovusdb"        , "W:xmm[63:0]/m64 {kz}, ymm"                       , "MR-QVM"  , "EVEX.256.F3.0F38.W0 11 /r"    , "AVX512_F-VL"],
        ["vpmovusdb"        , "W:xmm/m128 {kz}, zmm"                            , "MR-QVM"  , "EVEX.512.F3.0F38.W0 11 /r"    , "AVX512_F"],
        ["vpmovusdw"        , "W:xmm[63:0]/m64 {kz}, xmm"                       , "MR-HVM"  , "EVEX.128.F3.0F38.W0 13 /r"    , "AVX512_F-VL"],
        ["vpmovusdw"        , "W:xmm/m128 {kz}, ymm"                            , "MR-HVM"  , "EVEX.256.F3.0F38.W0 13 /r"    , "AVX512_F-VL"],
        ["vpmovusdw"        , "W:ymm/m256 {kz}, zmm"                            , "MR-HVM"  , "EVEX.512.F3.0F38.W0 13 /r"    , "AVX512_F"],
        ["vpmovusqb"        , "W:xmm[15:0]/m16 {kz}, xmm"                       , "MR-OVM"  , "EVEX.128.F3.0F38.W0 12 /r"    , "AVX512_F-VL"],
        ["vpmovusqb"        , "W:xmm[31:0]/m32 {kz}, ymm"                       , "MR-OVM"  , "EVEX.256.F3.0F38.W0 12 /r"    , "AVX512_F-VL"],
        ["vpmovusqb"        , "W:xmm[63:0]/m64 {kz}, zmm"                       , "MR-OVM"  , "EVEX.512.F3.0F38.W0 12 /r"    , "AVX512_F"],
        ["vpmovusqd"        , "W:xmm[63:0]/m64 {kz}, xmm"                       , "MR-HVM"  , "EVEX.128.F3.0F38.W0 15 /r"    , "AVX512_F-VL"],
        ["vpmovusqd"        , "W:xmm/m128 {kz}, ymm"                            , "MR-HVM"  , "EVEX.256.F3.0F38.W0 15 /r"    , "AVX512_F-VL"],
        ["vpmovusqd"        , "W:ymm/m256 {kz}, zmm"                            , "MR-HVM"  , "EVEX.512.F3.0F38.W0 15 /r"    , "AVX512_F"],
        ["vpmovusqw"        , "W:xmm[31:0]/m32 {kz}, xmm"                       , "MR-QVM"  , "EVEX.128.F3.0F38.W0 14 /r"    , "AVX512_F-VL"],
        ["vpmovusqw"        , "W:xmm[63:0]/m64 {kz}, ymm"                       , "MR-QVM"  , "EVEX.256.F3.0F38.W0 14 /r"    , "AVX512_F-VL"],
        ["vpmovusqw"        , "W:xmm/m128 {kz}, zmm"                            , "MR-QVM"  , "EVEX.512.F3.0F38.W0 14 /r"    , "AVX512_F"],
        ["vpmovuswb"        , "W:xmm[63:0]/m64 {kz}, xmm"                       , "MR-HVM"  , "EVEX.128.F3.0F38.W0 10 /r"    , "AVX512_BW-VL"],
        ["vpmovuswb"        , "W:xmm/m128 {kz}, ymm"                            , "MR-HVM"  , "EVEX.256.F3.0F38.W0 10 /r"    , "AVX512_BW-VL"],
        ["vpmovuswb"        , "W:ymm/m256 {kz}, zmm"                            , "MR-HVM"  , "EVEX.512.F3.0F38.W0 10 /r"    , "AVX512_BW"],
        ["vpmovw2m"         , "W:k, xmm"                                        , "RM"      , "EVEX.128.F3.0F38.W1 29 /r"    , "AVX512_BW-VL"],
        ["vpmovw2m"         , "W:k, ymm"                                        , "RM"      , "EVEX.256.F3.0F38.W1 29 /r"    , "AVX512_BW-VL"],
        ["vpmovw2m"         , "W:k, zmm"                                        , "RM"      , "EVEX.512.F3.0F38.W1 29 /r"    , "AVX512_BW"],
        ["vpmovwb"          , "W:xmm[63:0]/m64 {kz}, xmm"                       , "MR-HVM"  , "EVEX.128.F3.0F38.W0 30 /r"    , "AVX512_BW-VL"],
        ["vpmovwb"          , "W:xmm/m128 {kz}, ymm"                            , "MR-HVM"  , "EVEX.256.F3.0F38.W0 30 /r"    , "AVX512_BW-VL"],
        ["vpmovwb"          , "W:ymm/m256 {kz}, zmm"                            , "MR-HVM"  , "EVEX.512.F3.0F38.W0 30 /r"    , "AVX512_BW"],
        ["vpmovzxbd"        , "W:xmm {kz}, xmm[31:0]/m32"                       , "RM-QVM"  , "EVEX.128.66.0F38.WIG 31 /r"   , "AVX512_F-VL"],
        ["vpmovzxbd"        , "W:ymm {kz}, xmm[63:0]/m64"                       , "RM-QVM"  , "EVEX.256.66.0F38.WIG 31 /r"   , "AVX512_F-VL"],
        ["vpmovzxbd"        , "W:zmm {kz}, xmm/m128"                            , "RM-QVM"  , "EVEX.512.66.0F38.WIG 31 /r"   , "AVX512_F"],
        ["vpmovzxbq"        , "W:xmm {kz}, xmm[15:0]/m16"                       , "RM-OVM"  , "EVEX.128.66.0F38.WIG 32 /r"   , "AVX512_F-VL"],
        ["vpmovzxbq"        , "W:ymm {kz}, xmm[31:0]/m32"                       , "RM-OVM"  , "EVEX.256.66.0F38.WIG 32 /r"   , "AVX512_F-VL"],
        ["vpmovzxbq"        , "W:zmm {kz}, xmm[63:0]/m64"                       , "RM-OVM"  , "EVEX.512.66.0F38.WIG 32 /r"   , "AVX512_F"],
        ["vpmovzxbw"        , "W:xmm {kz}, xmm[63:0]/m64"                       , "RM-HVM"  , "EVEX.128.66.0F38.WIG 30 /r"   , "AVX512_BW-VL"],
        ["vpmovzxbw"        , "W:ymm {kz}, xmm/m128"                            , "RM-HVM"  , "EVEX.256.66.0F38.WIG 30 /r"   , "AVX512_BW-VL"],
        ["vpmovzxbw"        , "W:zmm {kz}, ymm/m256"                            , "RM-HVM"  , "EVEX.512.66.0F38.WIG 30 /r"   , "AVX512_BW"],
        ["vpmovzxdq"        , "W:xmm {kz}, xmm[63:0]/m64"                       , "RM-HVM"  , "EVEX.128.66.0F38.W0 35 /r"    , "AVX512_F-VL"],
        ["vpmovzxdq"        , "W:ymm {kz}, xmm/m128"                            , "RM-HVM"  , "EVEX.256.66.0F38.W0 35 /r"    , "AVX512_F-VL"],
        ["vpmovzxdq"        , "W:zmm {kz}, ymm/m256"                            , "RM-HVM"  , "EVEX.512.66.0F38.W0 35 /r"    , "AVX512_F"],
        ["vpmovzxwd"        , "W:xmm {kz}, xmm[63:0]/m64"                       , "RM-HVM"  , "EVEX.128.66.0F38.WIG 33 /r"   , "AVX512_F-VL"],
        ["vpmovzxwd"        , "W:ymm {kz}, xmm/m128"                            , "RM-HVM"  , "EVEX.256.66.0F38.WIG 33 /r"   , "AVX512_F-VL"],
        ["vpmovzxwd"        , "W:zmm {kz}, ymm/m256"                            , "RM-HVM"  , "EVEX.512.66.0F38.WIG 33 /r"   , "AVX512_F"],
        ["vpmovzxwq"        , "W:xmm {kz}, xmm[31:0]/m32"                       , "RM-QVM"  , "EVEX.128.66.0F38.WIG 34 /r"   , "AVX512_F-VL"],
        ["vpmovzxwq"        , "W:ymm {kz}, xmm[63:0]/m64"                       , "RM-QVM"  , "EVEX.256.66.0F38.WIG 34 /r"   , "AVX512_F-VL"],
        ["vpmovzxwq"        , "W:zmm {kz}, xmm/m128"                            , "RM-QVM"  , "EVEX.512.66.0F38.WIG 34 /r"   , "AVX512_F"],
        ["vpmuldq"          , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 28 /r"    , "AVX512_F-VL"],
        ["vpmuldq"          , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 28 /r"    , "AVX512_F-VL"],
        ["vpmuldq"          , "W:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 28 /r"    , "AVX512_F"],
        ["vpmulhrsw"        , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.WIG 0B /r"   , "AVX512_BW-VL"],
        ["vpmulhrsw"        , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.WIG 0B /r"   , "AVX512_BW-VL"],
        ["vpmulhrsw"        , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.WIG 0B /r"   , "AVX512_BW"],
        ["vpmulhuw"         , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG E4 /r"     , "AVX512_BW-VL"],
        ["vpmulhuw"         , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG E4 /r"     , "AVX512_BW-VL"],
        ["vpmulhuw"         , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG E4 /r"     , "AVX512_BW"],
        ["vpmulhw"          , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG E5 /r"     , "AVX512_BW-VL"],
        ["vpmulhw"          , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG E5 /r"     , "AVX512_BW-VL"],
        ["vpmulhw"          , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG E5 /r"     , "AVX512_BW"],
        ["vpmulld"          , "W:xmm {kz},~xmm,~xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 40 /r"    , "AVX512_F-VL"],
        ["vpmulld"          , "W:ymm {kz},~ymm,~ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 40 /r"    , "AVX512_F-VL"],
        ["vpmulld"          , "W:zmm {kz},~zmm,~zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 40 /r"    , "AVX512_F"],
        ["vpmullq"          , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 40 /r"    , "AVX512_DQ-VL"],
        ["vpmullq"          , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 40 /r"    , "AVX512_DQ-VL"],
        ["vpmullq"          , "W:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 40 /r"    , "AVX512_DQ"],
        ["vpmullw"          , "W:xmm {kz},~xmm,~xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG D5 /r"     , "AVX512_BW-VL"],
        ["vpmullw"          , "W:ymm {kz},~ymm,~ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG D5 /r"     , "AVX512_BW-VL"],
        ["vpmullw"          , "W:zmm {kz},~zmm,~zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG D5 /r"     , "AVX512_BW"],
        ["vpmultishiftqb"   , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 83 /r"    , "AVX512_VBMI-VL"],
        ["vpmultishiftqb"   , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 83 /r"    , "AVX512_VBMI-VL"],
        ["vpmultishiftqb"   , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 83 /r"    , "AVX512_VBMI"],
        ["vpmuludq"         , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 F4 /r"      , "AVX512_F-VL"],
        ["vpmuludq"         , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 F4 /r"      , "AVX512_F-VL"],
        ["vpmuludq"         , "W:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F.W1 F4 /r"      , "AVX512_F"],
        ["vpopcntb"         , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.66.0F38.W0 54 /r"    , "AVX512_BITALG-VL"],
        ["vpopcntb"         , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.66.0F38.W0 54 /r"    , "AVX512_BITALG-VL"],
        ["vpopcntb"         , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.66.0F38.W0 54 /r"    , "AVX512_BITALG"],
        ["vpopcntd"         , "W:xmm {kz}, xmm/m128/b32"                        , "RM-FVM"  , "EVEX.128.66.0F38.W0 55 /r"    , "AVX512_VPOPCNTDQ-VL"],
        ["vpopcntd"         , "W:ymm {kz}, ymm/m256/b32"                        , "RM-FVM"  , "EVEX.256.66.0F38.W0 55 /r"    , "AVX512_VPOPCNTDQ-VL"],
        ["vpopcntd"         , "W:zmm {kz}, zmm/m512/b32"                        , "RM-FVM"  , "EVEX.512.66.0F38.W0 55 /r"    , "AVX512_VPOPCNTDQ"],
        ["vpopcntq"         , "W:xmm {kz}, xmm/m128/b64"                        , "RM-FVM"  , "EVEX.128.66.0F38.W1 55 /r"    , "AVX512_VPOPCNTDQ-VL"],
        ["vpopcntq"         , "W:ymm {kz}, ymm/m256/b64"                        , "RM-FVM"  , "EVEX.256.66.0F38.W1 55 /r"    , "AVX512_VPOPCNTDQ-VL"],
        ["vpopcntq"         , "W:zmm {kz}, zmm/m512/b64"                        , "RM-FVM"  , "EVEX.512.66.0F38.W1 55 /r"    , "AVX512_VPOPCNTDQ"],
        ["vpopcntw"         , "W:xmm {kz}, xmm/m128"                            , "RM-FVM"  , "EVEX.128.66.0F38.W1 54 /r"    , "AVX512_BITALG-VL"],
        ["vpopcntw"         , "W:ymm {kz}, ymm/m256"                            , "RM-FVM"  , "EVEX.256.66.0F38.W1 54 /r"    , "AVX512_BITALG-VL"],
        ["vpopcntw"         , "W:zmm {kz}, zmm/m512"                            , "RM-FVM"  , "EVEX.512.66.0F38.W1 54 /r"    , "AVX512_BITALG"],
        ["vpord"            , "W:xmm {kz},~xmm,~xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F.W0 EB /r"      , "AVX512_F-VL"],
        ["vpord"            , "W:ymm {kz},~ymm,~ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F.W0 EB /r"      , "AVX512_F-VL"],
        ["vpord"            , "W:zmm {kz},~zmm,~zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F.W0 EB /r"      , "AVX512_F"],
        ["vporq"            , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 EB /r"      , "AVX512_F-VL"],
        ["vporq"            , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 EB /r"      , "AVX512_F-VL"],
        ["vporq"            , "W:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F.W1 EB /r"      , "AVX512_F"],
        ["vprold"           , "W:xmm {kz}, xmm/m128/b32, ib/ub"                 , "VMI-FV"  , "EVEX.128.66.0F.W0 72 /1 ib"   , "AVX512_F-VL"],
        ["vprold"           , "W:ymm {kz}, ymm/m256/b32, ib/ub"                 , "VMI-FV"  , "EVEX.256.66.0F.W0 72 /1 ib"   , "AVX512_F-VL"],
        ["vprold"           , "W:zmm {kz}, zmm/m512/b32, ib/ub"                 , "VMI-FV"  , "EVEX.512.66.0F.W0 72 /1 ib"   , "AVX512_F"],
        ["vprolq"           , "W:xmm {kz}, xmm/m128/b64, ib/ub"                 , "VMI-FV"  , "EVEX.128.66.0F.W1 72 /1 ib"   , "AVX512_F-VL"],
        ["vprolq"           , "W:ymm {kz}, ymm/m256/b64, ib/ub"                 , "VMI-FV"  , "EVEX.256.66.0F.W1 72 /1 ib"   , "AVX512_F-VL"],
        ["vprolq"           , "W:zmm {kz}, zmm/m512/b64, ib/ub"                 , "VMI-FV"  , "EVEX.512.66.0F.W1 72 /1 ib"   , "AVX512_F"],
        ["vprolvd"          , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 15 /r"    , "AVX512_F-VL"],
        ["vprolvd"          , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 15 /r"    , "AVX512_F-VL"],
        ["vprolvd"          , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 15 /r"    , "AVX512_F"],
        ["vprolvq"          , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 15 /r"    , "AVX512_F-VL"],
        ["vprolvq"          , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 15 /r"    , "AVX512_F-VL"],
        ["vprolvq"          , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 15 /r"    , "AVX512_F"],
        ["vprord"           , "W:xmm {kz}, xmm/m128/b32, ib/ub"                 , "VMI-FV"  , "EVEX.128.66.0F.W0 72 /0 ib"   , "AVX512_F-VL"],
        ["vprord"           , "W:ymm {kz}, ymm/m256/b32, ib/ub"                 , "VMI-FV"  , "EVEX.256.66.0F.W0 72 /0 ib"   , "AVX512_F-VL"],
        ["vprord"           , "W:zmm {kz}, zmm/m512/b32, ib/ub"                 , "VMI-FV"  , "EVEX.512.66.0F.W0 72 /0 ib"   , "AVX512_F"],
        ["vprorq"           , "W:xmm {kz}, xmm/m128/b64, ib/ub"                 , "VMI-FV"  , "EVEX.128.66.0F.W1 72 /0 ib"   , "AVX512_F-VL"],
        ["vprorq"           , "W:ymm {kz}, ymm/m256/b64, ib/ub"                 , "VMI-FV"  , "EVEX.256.66.0F.W1 72 /0 ib"   , "AVX512_F-VL"],
        ["vprorq"           , "W:zmm {kz}, zmm/m512/b64, ib/ub"                 , "VMI-FV"  , "EVEX.512.66.0F.W1 72 /0 ib"   , "AVX512_F"],
        ["vprorvd"          , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 14 /r"    , "AVX512_F-VL"],
        ["vprorvd"          , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 14 /r"    , "AVX512_F-VL"],
        ["vprorvd"          , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 14 /r"    , "AVX512_F"],
        ["vprorvq"          , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 14 /r"    , "AVX512_F-VL"],
        ["vprorvq"          , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 14 /r"    , "AVX512_F-VL"],
        ["vprorvq"          , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 14 /r"    , "AVX512_F"],
        ["vpsadbw"          , "W:xmm,~xmm,~xmm/m128"                            , "RVM-FVM" , "EVEX.128.66.0F.WIG F6 /r"     , "AVX512_BW-VL"],
        ["vpsadbw"          , "W:ymm,~ymm,~ymm/m256"                            , "RVM-FVM" , "EVEX.256.66.0F.WIG F6 /r"     , "AVX512_BW-VL"],
        ["vpsadbw"          , "W:zmm,~zmm,~zmm/m512"                            , "RVM-FVM" , "EVEX.512.66.0F.WIG F6 /r"     , "AVX512_BW"],
        ["vpscatterdd"      , "W:vm32x {k}, xmm"                                , "MR-T1S"  , "EVEX.128.66.0F38.W0 A0 /r"    , "AVX512_F-VL"],
        ["vpscatterdd"      , "W:vm32y {k}, ymm"                                , "MR-T1S"  , "EVEX.256.66.0F38.W0 A0 /r"    , "AVX512_F-VL"],
        ["vpscatterdd"      , "W:vm32z {k}, zmm"                                , "MR-T1S"  , "EVEX.512.66.0F38.W0 A0 /r"    , "AVX512_F"],
        ["vpscatterdq"      , "W:vm32x {k}, xmm"                                , "MR-T1S"  , "EVEX.128.66.0F38.W1 A0 /r"    , "AVX512_F-VL"],
        ["vpscatterdq"      , "W:vm32y {k}, ymm"                                , "MR-T1S"  , "EVEX.256.66.0F38.W1 A0 /r"    , "AVX512_F-VL"],
        ["vpscatterdq"      , "W:vm32z {k}, zmm"                                , "MR-T1S"  , "EVEX.512.66.0F38.W1 A0 /r"    , "AVX512_F"],
        ["vpscatterqd"      , "W:vm64x {k}, xmm"                                , "MR-T1S"  , "EVEX.128.66.0F38.W0 A1 /r"    , "AVX512_F-VL"],
        ["vpscatterqd"      , "W:vm64y {k}, xmm"                                , "MR-T1S"  , "EVEX.256.66.0F38.W0 A1 /r"    , "AVX512_F-VL"],
        ["vpscatterqd"      , "W:vm64z {k}, ymm"                                , "MR-T1S"  , "EVEX.512.66.0F38.W0 A1 /r"    , "AVX512_F"],
        ["vpscatterqq"      , "W:vm64x {k}, xmm"                                , "MR-T1S"  , "EVEX.128.66.0F38.W1 A1 /r"    , "AVX512_F-VL"],
        ["vpscatterqq"      , "W:vm64y {k}, ymm"                                , "MR-T1S"  , "EVEX.256.66.0F38.W1 A1 /r"    , "AVX512_F-VL"],
        ["vpscatterqq"      , "W:vm64z {k}, zmm"                                , "MR-T1S"  , "EVEX.512.66.0F38.W1 A1 /r"    , "AVX512_F"],
        ["vpshldd"          , "W:xmm {kz}, xmm, xmm/m128/b32, ib/ub"            , "RVMI-FVM", "EVEX.128.66.0F3A.W0 71 /r ib" , "AVX512_VBMI2-VL"],
        ["vpshldd"          , "W:ymm {kz}, ymm, ymm/m256/b32, ib/ub"            , "RVMI-FVM", "EVEX.256.66.0F3A.W0 71 /r ib" , "AVX512_VBMI2-VL"],
        ["vpshldd"          , "W:zmm {kz}, zmm, zmm/m512/b32, ib/ub"            , "RVMI-FVM", "EVEX.512.66.0F3A.W0 71 /r ib" , "AVX512_VBMI2"],
        ["vpshldq"          , "W:xmm {kz}, xmm, xmm/m128/b64, ib/ub"            , "RVMI-FVM", "EVEX.128.66.0F3A.W1 71 /r ib" , "AVX512_VBMI2-VL"],
        ["vpshldq"          , "W:ymm {kz}, ymm, ymm/m256/b64, ib/ub"            , "RVMI-FVM", "EVEX.256.66.0F3A.W1 71 /r ib" , "AVX512_VBMI2-VL"],
        ["vpshldq"          , "W:zmm {kz}, zmm, zmm/m512/b64, ib/ub"            , "RVMI-FVM", "EVEX.512.66.0F3A.W1 71 /r ib" , "AVX512_VBMI2"],
        ["vpshldvd"         , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FVM" , "EVEX.128.66.0F38.W0 71 /r"    , "AVX512_VBMI2-VL"],
        ["vpshldvd"         , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FVM" , "EVEX.256.66.0F38.W0 71 /r"    , "AVX512_VBMI2-VL"],
        ["vpshldvd"         , "X:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FVM" , "EVEX.512.66.0F38.W0 71 /r"    , "AVX512_VBMI2"],
        ["vpshldvq"         , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FVM" , "EVEX.128.66.0F38.W1 71 /r"    , "AVX512_VBMI2-VL"],
        ["vpshldvq"         , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FVM" , "EVEX.256.66.0F38.W1 71 /r"    , "AVX512_VBMI2-VL"],
        ["vpshldvq"         , "X:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FVM" , "EVEX.512.66.0F38.W1 71 /r"    , "AVX512_VBMI2"],
        ["vpshldvw"         , "X:xmm {kz}, xmm, xmm/m128"                       , "RVM-FV"  , "EVEX.128.66.0F38.W1 70 /r"    , "AVX512_VBMI2-VL"],
        ["vpshldvw"         , "X:ymm {kz}, ymm, ymm/m256"                       , "RVM-FV"  , "EVEX.256.66.0F38.W1 70 /r"    , "AVX512_VBMI2-VL"],
        ["vpshldvw"         , "X:zmm {kz}, zmm, zmm/m512"                       , "RVM-FV"  , "EVEX.512.66.0F38.W1 70 /r"    , "AVX512_VBMI2"],
        ["vpshldw"          , "W:xmm {kz}, xmm, xmm/m128, ib/ub"                , "RVMI-FV" , "EVEX.128.66.0F3A.W1 70 /r ib" , "AVX512_VBMI2-VL"],
        ["vpshldw"          , "W:ymm {kz}, ymm, ymm/m256, ib/ub"                , "RVMI-FV" , "EVEX.256.66.0F3A.W1 70 /r ib" , "AVX512_VBMI2-VL"],
        ["vpshldw"          , "W:zmm {kz}, zmm, zmm/m512, ib/ub"                , "RVMI-FV" , "EVEX.512.66.0F3A.W1 70 /r ib" , "AVX512_VBMI2"],
        ["vpshrdd"          , "W:xmm {kz}, xmm, xmm/m128/b32, ib/ub"            , "RVMI-FVM", "EVEX.128.66.0F3A.W0 73 /r ib" , "AVX512_VBMI2-VL"],
        ["vpshrdd"          , "W:ymm {kz}, ymm, ymm/m256/b32, ib/ub"            , "RVMI-FVM", "EVEX.256.66.0F3A.W0 73 /r ib" , "AVX512_VBMI2-VL"],
        ["vpshrdd"          , "W:zmm {kz}, zmm, zmm/m512/b32, ib/ub"            , "RVMI-FVM", "EVEX.512.66.0F3A.W0 73 /r ib" , "AVX512_VBMI2"],
        ["vpshrdq"          , "W:xmm {kz}, xmm, xmm/m128/b64, ib/ub"            , "RVMI-FVM", "EVEX.128.66.0F3A.W1 73 /r ib" , "AVX512_VBMI2-VL"],
        ["vpshrdq"          , "W:ymm {kz}, ymm, ymm/m256/b64, ib/ub"            , "RVMI-FVM", "EVEX.256.66.0F3A.W1 73 /r ib" , "AVX512_VBMI2-VL"],
        ["vpshrdq"          , "W:zmm {kz}, zmm, zmm/m512/b64, ib/ub"            , "RVMI-FVM", "EVEX.512.66.0F3A.W1 73 /r ib" , "AVX512_VBMI2"],
        ["vpshrdvd"         , "X:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FVM" , "EVEX.128.66.0F38.W0 73 /r"    , "AVX512_VBMI2-VL"],
        ["vpshrdvd"         , "X:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FVM" , "EVEX.256.66.0F38.W0 73 /r"    , "AVX512_VBMI2-VL"],
        ["vpshrdvd"         , "X:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FVM" , "EVEX.512.66.0F38.W0 73 /r"    , "AVX512_VBMI2"],
        ["vpshrdvq"         , "X:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FVM" , "EVEX.128.66.0F38.W1 73 /r"    , "AVX512_VBMI2-VL"],
        ["vpshrdvq"         , "X:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FVM" , "EVEX.256.66.0F38.W1 73 /r"    , "AVX512_VBMI2-VL"],
        ["vpshrdvq"         , "X:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FVM" , "EVEX.512.66.0F38.W1 73 /r"    , "AVX512_VBMI2"],
        ["vpshrdvw"         , "X:xmm {kz}, xmm, xmm/m128"                       , "RVM-FV"  , "EVEX.128.66.0F38.W1 72 /r"    , "AVX512_VBMI2-VL"],
        ["vpshrdvw"         , "X:ymm {kz}, ymm, ymm/m256"                       , "RVM-FV"  , "EVEX.256.66.0F38.W1 72 /r"    , "AVX512_VBMI2-VL"],
        ["vpshrdvw"         , "X:zmm {kz}, zmm, zmm/m512"                       , "RVM-FV"  , "EVEX.512.66.0F38.W1 72 /r"    , "AVX512_VBMI2"],
        ["vpshrdw"          , "W:xmm {kz}, xmm, xmm/m128, ib/ub"                , "RVMI-FV" , "EVEX.128.66.0F3A.W1 72 /r ib" , "AVX512_VBMI2-VL"],
        ["vpshrdw"          , "W:ymm {kz}, ymm, ymm/m256, ib/ub"                , "RVMI-FV" , "EVEX.256.66.0F3A.W1 72 /r ib" , "AVX512_VBMI2-VL"],
        ["vpshrdw"          , "W:zmm {kz}, zmm, zmm/m512, ib/ub"                , "RVMI-FV" , "EVEX.512.66.0F3A.W1 72 /r ib" , "AVX512_VBMI2"],
        ["vpshufb"          , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.WIG 00 /r"   , "AVX512_BW-VL"],
        ["vpshufb"          , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.WIG 00 /r"   , "AVX512_BW-VL"],
        ["vpshufb"          , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.WIG 00 /r"   , "AVX512_BW"],
        ["vpshufbitqmb"     , "W:k {k}, xmm, xmm/m128"                          , "RM-FVM"  , "EVEX.128.66.0F38.W0 8F /r"    , "AVX512_BITALG-VL"],
        ["vpshufbitqmb"     , "W:k {k}, ymm, ymm/m256"                          , "RM-FVM"  , "EVEX.256.66.0F38.W0 8F /r"    , "AVX512_BITALG-VL"],
        ["vpshufbitqmb"     , "W:k {k}, zmm, zmm/m512"                          , "RM-FVM"  , "EVEX.512.66.0F38.W0 8F /r"    , "AVX512_BITALG"],
        ["vpshufd"          , "W:xmm {kz}, xmm/m128/b32, ib/ub"                 , "RMI-FV"  , "EVEX.128.66.0F.W0 70 /r ib"   , "AVX512_F-VL"],
        ["vpshufd"          , "W:ymm {kz}, ymm/m256/b32, ib/ub"                 , "RMI-FV"  , "EVEX.256.66.0F.W0 70 /r ib"   , "AVX512_F-VL"],
        ["vpshufd"          , "W:zmm {kz}, zmm/m512/b32, ib/ub"                 , "RMI-FV"  , "EVEX.512.66.0F.W0 70 /r ib"   , "AVX512_F"],
        ["vpshufhw"         , "W:xmm {kz}, xmm/m128, ib/ub"                     , "RMI-FVM" , "EVEX.128.F3.0F.WIG 70 /r ib"  , "AVX512_BW-VL"],
        ["vpshufhw"         , "W:ymm {kz}, ymm/m256, ib/ub"                     , "RMI-FVM" , "EVEX.256.F3.0F.WIG 70 /r ib"  , "AVX512_BW-VL"],
        ["vpshufhw"         , "W:zmm {kz}, zmm/m512, ib/ub"                     , "RMI-FVM" , "EVEX.512.F3.0F.WIG 70 /r ib"  , "AVX512_BW"],
        ["vpshuflw"         , "W:xmm {kz}, xmm/m128, ib/ub"                     , "RMI-FVM" , "EVEX.128.F2.0F.WIG 70 /r ib"  , "AVX512_BW-VL"],
        ["vpshuflw"         , "W:ymm {kz}, ymm/m256, ib/ub"                     , "RMI-FVM" , "EVEX.256.F2.0F.WIG 70 /r ib"  , "AVX512_BW-VL"],
        ["vpshuflw"         , "W:zmm {kz}, zmm/m512, ib/ub"                     , "RMI-FVM" , "EVEX.512.F2.0F.WIG 70 /r ib"  , "AVX512_BW"],
        ["vpslld"           , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-M128", "EVEX.128.66.0F.W0 F2 /r"      , "AVX512_F-VL"],
        ["vpslld"           , "W:xmm {kz}, xmm/m128/b32, ib/ub"                 , "VMI-FV"  , "EVEX.128.66.0F.W0 72 /6 ib"   , "AVX512_F-VL"],
        ["vpslld"           , "W:ymm {kz}, ymm, xmm/m128"                       , "RVM-M128", "EVEX.256.66.0F.W0 F2 /r"      , "AVX512_F-VL"],
        ["vpslld"           , "W:ymm {kz}, ymm/m256/b32, ib/ub"                 , "VMI-FV"  , "EVEX.256.66.0F.W0 72 /6 ib"   , "AVX512_F-VL"],
        ["vpslld"           , "W:zmm {kz}, zmm, xmm/m128"                       , "RVM-M128", "EVEX.512.66.0F.W0 F2 /r"      , "AVX512_F"],
        ["vpslld"           , "W:zmm {kz}, zmm/m512/b32, ib/ub"                 , "VMI-FV"  , "EVEX.512.66.0F.W0 72 /6 ib"   , "AVX512_F"],
        ["vpslldq"          , "W:xmm, xmm/m128, ib/ub"                          , "VMI-FVM" , "EVEX.128.66.0F.WIG 73 /7 ib"  , "AVX512_BW-VL"],
        ["vpslldq"          , "W:ymm, ymm/m256, ib/ub"                          , "VMI-FVM" , "EVEX.256.66.0F.WIG 73 /7 ib"  , "AVX512_BW-VL"],
        ["vpslldq"          , "W:zmm, zmm/m512, ib/ub"                          , "VMI-FVM" , "EVEX.512.66.0F.WIG 73 /7 ib"  , "AVX512_BW"],
        ["vpsllq"           , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-M128", "EVEX.128.66.0F.W1 F3 /r"      , "AVX512_F-VL"],
        ["vpsllq"           , "W:xmm {kz}, xmm/m128/b64, ib/ub"                 , "VMI-FV"  , "EVEX.128.66.0F.W1 73 /6 ib"   , "AVX512_F-VL"],
        ["vpsllq"           , "W:ymm {kz}, ymm, xmm/m128"                       , "RVM-M128", "EVEX.256.66.0F.W1 F3 /r"      , "AVX512_F-VL"],
        ["vpsllq"           , "W:ymm {kz}, ymm/m256/b64, ib/ub"                 , "VMI-FV"  , "EVEX.256.66.0F.W1 73 /6 ib"   , "AVX512_F-VL"],
        ["vpsllq"           , "W:zmm {kz}, zmm, xmm/m128"                       , "RVM-M128", "EVEX.512.66.0F.W1 F3 /r"      , "AVX512_F"],
        ["vpsllq"           , "W:zmm {kz}, zmm/m512/b64, ib/ub"                 , "VMI-FV"  , "EVEX.512.66.0F.W1 73 /6 ib"   , "AVX512_F"],
        ["vpsllvd"          , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 47 /r"    , "AVX512_F-VL"],
        ["vpsllvd"          , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 47 /r"    , "AVX512_F-VL"],
        ["vpsllvd"          , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 47 /r"    , "AVX512_F"],
        ["vpsllvq"          , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 47 /r"    , "AVX512_F-VL"],
        ["vpsllvq"          , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 47 /r"    , "AVX512_F-VL"],
        ["vpsllvq"          , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 47 /r"    , "AVX512_F"],
        ["vpsllvw"          , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.W1 12 /r"    , "AVX512_BW-VL"],
        ["vpsllvw"          , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.W1 12 /r"    , "AVX512_BW-VL"],
        ["vpsllvw"          , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.W1 12 /r"    , "AVX512_BW"],
        ["vpsllw"           , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-M128", "EVEX.128.66.0F.WIG F1 /r"     , "AVX512_BW-VL"],
        ["vpsllw"           , "W:xmm {kz}, xmm/m128, ib/ub"                     , "VMI-FVM" , "EVEX.128.66.0F.WIG 71 /6 ib"  , "AVX512_BW-VL"],
        ["vpsllw"           , "W:ymm {kz}, ymm, xmm/m128"                       , "RVM-M128", "EVEX.256.66.0F.WIG F1 /r"     , "AVX512_BW-VL"],
        ["vpsllw"           , "W:ymm {kz}, ymm/m256, ib/ub"                     , "VMI-FVM" , "EVEX.256.66.0F.WIG 71 /6 ib"  , "AVX512_BW-VL"],
        ["vpsllw"           , "W:zmm {kz}, zmm, xmm/m128"                       , "RVM-M128", "EVEX.512.66.0F.WIG F1 /r"     , "AVX512_BW"],
        ["vpsllw"           , "W:zmm {kz}, zmm/m512, ib/ub"                     , "VMI-FVM" , "EVEX.512.66.0F.WIG 71 /6 ib"  , "AVX512_BW"],
        ["vpsrad"           , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-M128", "EVEX.128.66.0F.W0 E2 /r"      , "AVX512_F-VL"],
        ["vpsrad"           , "W:xmm {kz}, xmm/m128/b32, ib/ub"                 , "VMI-FV"  , "EVEX.128.66.0F.W0 72 /4 ib"   , "AVX512_F-VL"],
        ["vpsrad"           , "W:ymm {kz}, ymm, xmm/m128"                       , "RVM-M128", "EVEX.256.66.0F.W0 E2 /r"      , "AVX512_F-VL"],
        ["vpsrad"           , "W:ymm {kz}, ymm/m256/b32, ib/ub"                 , "VMI-FV"  , "EVEX.256.66.0F.W0 72 /4 ib"   , "AVX512_F-VL"],
        ["vpsrad"           , "W:zmm {kz}, zmm, xmm/m128"                       , "RVM-M128", "EVEX.512.66.0F.W0 E2 /r"      , "AVX512_F"],
        ["vpsrad"           , "W:zmm {kz}, zmm/m512/b32, ib/ub"                 , "VMI-FV"  , "EVEX.512.66.0F.W0 72 /4 ib"   , "AVX512_F"],
        ["vpsraq"           , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-M128", "EVEX.128.66.0F.W1 E2 /r"      , "AVX512_F-VL"],
        ["vpsraq"           , "W:xmm {kz}, xmm/m128/b64, ib/ub"                 , "VMI-FV"  , "EVEX.128.66.0F.W1 72 /4 ib"   , "AVX512_F-VL"],
        ["vpsraq"           , "W:ymm {kz}, ymm, xmm/m128"                       , "RVM-M128", "EVEX.256.66.0F.W1 E2 /r"      , "AVX512_F-VL"],
        ["vpsraq"           , "W:ymm {kz}, ymm/m256/b64, ib/ub"                 , "VMI-FV"  , "EVEX.256.66.0F.W1 72 /4 ib"   , "AVX512_F-VL"],
        ["vpsraq"           , "W:zmm {kz}, zmm, xmm/m128"                       , "RVM-M128", "EVEX.512.66.0F.W1 E2 /r"      , "AVX512_F"],
        ["vpsraq"           , "W:zmm {kz}, zmm/m512/b64, ib/ub"                 , "VMI-FV"  , "EVEX.512.66.0F.W1 72 /4 ib"   , "AVX512_F"],
        ["vpsravd"          , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 46 /r"    , "AVX512_F-VL"],
        ["vpsravd"          , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 46 /r"    , "AVX512_F-VL"],
        ["vpsravd"          , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 46 /r"    , "AVX512_F"],
        ["vpsravq"          , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 46 /r"    , "AVX512_F-VL"],
        ["vpsravq"          , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 46 /r"    , "AVX512_F-VL"],
        ["vpsravq"          , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 46 /r"    , "AVX512_F"],
        ["vpsravw"          , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.W1 11 /r"    , "AVX512_BW-VL"],
        ["vpsravw"          , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.W1 11 /r"    , "AVX512_BW-VL"],
        ["vpsravw"          , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.W1 11 /r"    , "AVX512_BW"],
        ["vpsraw"           , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-M128", "EVEX.128.66.0F.WIG E1 /r"     , "AVX512_BW-VL"],
        ["vpsraw"           , "W:xmm {kz}, xmm/m128, ib/ub"                     , "VMI-FVM" , "EVEX.128.66.0F.WIG 71 /4 ib"  , "AVX512_BW-VL"],
        ["vpsraw"           , "W:ymm {kz}, ymm, xmm/m128"                       , "RVM-M128", "EVEX.256.66.0F.WIG E1 /r"     , "AVX512_BW-VL"],
        ["vpsraw"           , "W:ymm {kz}, ymm/m256, ib/ub"                     , "VMI-FVM" , "EVEX.256.66.0F.WIG 71 /4 ib"  , "AVX512_BW-VL"],
        ["vpsraw"           , "W:zmm {kz}, zmm, xmm/m128"                       , "RVM-M128", "EVEX.512.66.0F.WIG E1 /r"     , "AVX512_BW"],
        ["vpsraw"           , "W:zmm {kz}, zmm/m512, ib/ub"                     , "VMI-FVM" , "EVEX.512.66.0F.WIG 71 /4 ib"  , "AVX512_BW"],
        ["vpsrld"           , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-M128", "EVEX.128.66.0F.W0 D2 /r"      , "AVX512_F-VL"],
        ["vpsrld"           , "W:xmm {kz}, xmm/m128/b32, ib/ub"                 , "VMI-FV"  , "EVEX.128.66.0F.W0 72 /2 ib"   , "AVX512_F-VL"],
        ["vpsrld"           , "W:ymm {kz}, ymm, xmm/m128"                       , "RVM-M128", "EVEX.256.66.0F.W0 D2 /r"      , "AVX512_F-VL"],
        ["vpsrld"           , "W:ymm {kz}, ymm/m256/b32, ib/ub"                 , "VMI-FV"  , "EVEX.256.66.0F.W0 72 /2 ib"   , "AVX512_F-VL"],
        ["vpsrld"           , "W:zmm {kz}, zmm, xmm/m128"                       , "RVM-M128", "EVEX.512.66.0F.W0 D2 /r"      , "AVX512_F"],
        ["vpsrld"           , "W:zmm {kz}, zmm/m512/b32, ib/ub"                 , "VMI-FV"  , "EVEX.512.66.0F.W0 72 /2 ib"   , "AVX512_F"],
        ["vpsrldq"          , "W:xmm, xmm/m128, ib/ub"                          , "VMI-FVM" , "EVEX.128.66.0F.WIG 73 /3 ib"  , "AVX512_BW-VL"],
        ["vpsrldq"          , "W:ymm, ymm/m256, ib/ub"                          , "VMI-FVM" , "EVEX.256.66.0F.WIG 73 /3 ib"  , "AVX512_BW-VL"],
        ["vpsrldq"          , "W:zmm, zmm/m512, ib/ub"                          , "VMI-FVM" , "EVEX.512.66.0F.WIG 73 /3 ib"  , "AVX512_BW"],
        ["vpsrlq"           , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-M128", "EVEX.128.66.0F.W1 D3 /r"      , "AVX512_F-VL"],
        ["vpsrlq"           , "W:xmm {kz}, xmm/m128/b64, ib/ub"                 , "VMI-FV"  , "EVEX.128.66.0F.W1 73 /2 ib"   , "AVX512_F-VL"],
        ["vpsrlq"           , "W:ymm {kz}, ymm, xmm/m128"                       , "RVM-M128", "EVEX.256.66.0F.W1 D3 /r"      , "AVX512_F-VL"],
        ["vpsrlq"           , "W:ymm {kz}, ymm/m256/b64, ib/ub"                 , "VMI-FV"  , "EVEX.256.66.0F.W1 73 /2 ib"   , "AVX512_F-VL"],
        ["vpsrlq"           , "W:zmm {kz}, zmm, xmm/m128"                       , "RVM-M128", "EVEX.512.66.0F.W1 D3 /r"      , "AVX512_F"],
        ["vpsrlq"           , "W:zmm {kz}, zmm/m512/b64, ib/ub"                 , "VMI-FV"  , "EVEX.512.66.0F.W1 73 /2 ib"   , "AVX512_F"],
        ["vpsrlvd"          , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 45 /r"    , "AVX512_F-VL"],
        ["vpsrlvd"          , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 45 /r"    , "AVX512_F-VL"],
        ["vpsrlvd"          , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F38.W0 45 /r"    , "AVX512_F"],
        ["vpsrlvq"          , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 45 /r"    , "AVX512_F-VL"],
        ["vpsrlvq"          , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 45 /r"    , "AVX512_F-VL"],
        ["vpsrlvq"          , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F38.W1 45 /r"    , "AVX512_F"],
        ["vpsrlvw"          , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F38.W1 10 /r"    , "AVX512_BW-VL"],
        ["vpsrlvw"          , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F38.W1 10 /r"    , "AVX512_BW-VL"],
        ["vpsrlvw"          , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F38.W1 10 /r"    , "AVX512_BW"],
        ["vpsrlw"           , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-M128", "EVEX.128.66.0F.WIG D1 /r"     , "AVX512_BW-VL"],
        ["vpsrlw"           , "W:xmm {kz}, xmm/m128, ib/ub"                     , "VMI-FVM" , "EVEX.128.66.0F.WIG 71 /2 ib"  , "AVX512_BW-VL"],
        ["vpsrlw"           , "W:ymm {kz}, ymm, xmm/m128"                       , "RVM-M128", "EVEX.256.66.0F.WIG D1 /r"     , "AVX512_BW-VL"],
        ["vpsrlw"           , "W:ymm {kz}, ymm/m256, ib/ub"                     , "VMI-FVM" , "EVEX.256.66.0F.WIG 71 /2 ib"  , "AVX512_BW-VL"],
        ["vpsrlw"           , "W:zmm {kz}, zmm, xmm/m128"                       , "RVM-M128", "EVEX.512.66.0F.WIG D1 /r"     , "AVX512_BW"],
        ["vpsrlw"           , "W:zmm {kz}, zmm/m512, ib/ub"                     , "VMI-FVM" , "EVEX.512.66.0F.WIG 71 /2 ib"  , "AVX512_BW"],
        ["vpsubb"           , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG F8 /r"     , "AVX512_BW-VL"],
        ["vpsubb"           , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG F8 /r"     , "AVX512_BW-VL"],
        ["vpsubb"           , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG F8 /r"     , "AVX512_BW"],
        ["vpsubd"           , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F.W0 FA /r"      , "AVX512_F-VL"],
        ["vpsubd"           , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F.W0 FA /r"      , "AVX512_F-VL"],
        ["vpsubd"           , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F.W0 FA /r"      , "AVX512_F"],
        ["vpsubq"           , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 FB /r"      , "AVX512_F-VL"],
        ["vpsubq"           , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 FB /r"      , "AVX512_F-VL"],
        ["vpsubq"           , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F.W1 FB /r"      , "AVX512_F"],
        ["vpsubsb"          , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG E8 /r"     , "AVX512_BW-VL"],
        ["vpsubsb"          , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG E8 /r"     , "AVX512_BW-VL"],
        ["vpsubsb"          , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG E8 /r"     , "AVX512_BW"],
        ["vpsubsw"          , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG E9 /r"     , "AVX512_BW-VL"],
        ["vpsubsw"          , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG E9 /r"     , "AVX512_BW-VL"],
        ["vpsubsw"          , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG E9 /r"     , "AVX512_BW"],
        ["vpsubusb"         , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG D8 /r"     , "AVX512_BW-VL"],
        ["vpsubusb"         , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG D8 /r"     , "AVX512_BW-VL"],
        ["vpsubusb"         , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG D8 /r"     , "AVX512_BW"],
        ["vpsubusw"         , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG D9 /r"     , "AVX512_BW-VL"],
        ["vpsubusw"         , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG D9 /r"     , "AVX512_BW-VL"],
        ["vpsubusw"         , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG D9 /r"     , "AVX512_BW"],
        ["vpsubw"           , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG F9 /r"     , "AVX512_BW-VL"],
        ["vpsubw"           , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG F9 /r"     , "AVX512_BW-VL"],
        ["vpsubw"           , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG F9 /r"     , "AVX512_BW"],
        ["vpternlogd"       , "X:xmm {kz}, xmm, xmm/m128/b32, ib/ub"            , "RVMI-FV" , "EVEX.128.66.0F3A.W0 25 /r ib" , "AVX512_F-VL"],
        ["vpternlogd"       , "X:ymm {kz}, ymm, ymm/m256/b32, ib/ub"            , "RVMI-FV" , "EVEX.256.66.0F3A.W0 25 /r ib" , "AVX512_F-VL"],
        ["vpternlogd"       , "X:zmm {kz}, zmm, zmm/m512/b32, ib/ub"            , "RVMI-FV" , "EVEX.512.66.0F3A.W0 25 /r ib" , "AVX512_F"],
        ["vpternlogq"       , "X:xmm {kz}, xmm, xmm/m128/b64, ib/ub"            , "RVMI-FV" , "EVEX.128.66.0F3A.W1 25 /r ib" , "AVX512_F-VL"],
        ["vpternlogq"       , "X:ymm {kz}, ymm, ymm/m256/b64, ib/ub"            , "RVMI-FV" , "EVEX.256.66.0F3A.W1 25 /r ib" , "AVX512_F-VL"],
        ["vpternlogq"       , "X:zmm {kz}, zmm, zmm/m512/b64, ib/ub"            , "RVMI-FV" , "EVEX.512.66.0F3A.W1 25 /r ib" , "AVX512_F"],
        ["vptestmb"         , "W:k {k}, xmm, xmm/m128"                          , "RVM-FVM" , "EVEX.128.66.0F38.W0 26 /r"    , "AVX512_BW-VL"],
        ["vptestmb"         , "W:k {k}, ymm, ymm/m256"                          , "RVM-FVM" , "EVEX.256.66.0F38.W0 26 /r"    , "AVX512_BW-VL"],
        ["vptestmb"         , "W:k {k}, zmm, zmm/m512"                          , "RVM-FVM" , "EVEX.512.66.0F38.W0 26 /r"    , "AVX512_BW"],
        ["vptestmd"         , "W:k {k}, xmm, xmm/m128/b32"                      , "RVM-FV"  , "EVEX.128.66.0F38.W0 27 /r"    , "AVX512_F-VL"],
        ["vptestmd"         , "W:k {k}, ymm, ymm/m256/b32"                      , "RVM-FV"  , "EVEX.256.66.0F38.W0 27 /r"    , "AVX512_F-VL"],
        ["vptestmd"         , "W:k {k}, zmm, zmm/m512/b32"                      , "RVM-FV"  , "EVEX.512.66.0F38.W0 27 /r"    , "AVX512_F"],
        ["vptestmq"         , "W:k {k}, xmm, xmm/m128/b64"                      , "RVM-FV"  , "EVEX.128.66.0F38.W1 27 /r"    , "AVX512_F-VL"],
        ["vptestmq"         , "W:k {k}, ymm, ymm/m256/b64"                      , "RVM-FV"  , "EVEX.256.66.0F38.W1 27 /r"    , "AVX512_F-VL"],
        ["vptestmq"         , "W:k {k}, zmm, zmm/m512/b64"                      , "RVM-FV"  , "EVEX.512.66.0F38.W1 27 /r"    , "AVX512_F"],
        ["vptestmw"         , "W:k {k}, xmm, xmm/m128"                          , "RVM-FVM" , "EVEX.128.66.0F38.W1 26 /r"    , "AVX512_BW-VL"],
        ["vptestmw"         , "W:k {k}, ymm, ymm/m256"                          , "RVM-FVM" , "EVEX.256.66.0F38.W1 26 /r"    , "AVX512_BW-VL"],
        ["vptestmw"         , "W:k {k}, zmm, zmm/m512"                          , "RVM-FVM" , "EVEX.512.66.0F38.W1 26 /r"    , "AVX512_BW"],
        ["vptestnmb"        , "W:k {k}, xmm, xmm/m128"                          , "RVM-FVM" , "EVEX.128.F3.0F38.W0 26 /r"    , "AVX512_BW-VL"],
        ["vptestnmb"        , "W:k {k}, ymm, ymm/m256"                          , "RVM-FVM" , "EVEX.256.F3.0F38.W0 26 /r"    , "AVX512_BW-VL"],
        ["vptestnmb"        , "W:k {k}, zmm, zmm/m512"                          , "RVM-FVM" , "EVEX.512.F3.0F38.W0 26 /r"    , "AVX512_BW"],
        ["vptestnmd"        , "W:k {k}, xmm, xmm/m128/b32"                      , "RVM-FV"  , "EVEX.128.F3.0F38.W0 27 /r"    , "AVX512_F-VL"],
        ["vptestnmd"        , "W:k {k}, ymm, ymm/m256/b32"                      , "RVM-FV"  , "EVEX.256.F3.0F38.W0 27 /r"    , "AVX512_F-VL"],
        ["vptestnmd"        , "W:k {k}, zmm, zmm/m512/b32"                      , "RVM-FV"  , "EVEX.512.F3.0F38.W0 27 /r"    , "AVX512_F"],
        ["vptestnmq"        , "W:k {k}, xmm, xmm/m128/b64"                      , "RVM-FV"  , "EVEX.128.F3.0F38.W1 27 /r"    , "AVX512_F-VL"],
        ["vptestnmq"        , "W:k {k}, ymm, ymm/m256/b64"                      , "RVM-FV"  , "EVEX.256.F3.0F38.W1 27 /r"    , "AVX512_F-VL"],
        ["vptestnmq"        , "W:k {k}, zmm, zmm/m512/b64"                      , "RVM-FV"  , "EVEX.512.F3.0F38.W1 27 /r"    , "AVX512_F"],
        ["vptestnmw"        , "W:k {k}, xmm, xmm/m128"                          , "RVM-FVM" , "EVEX.128.F3.0F38.W1 26 /r"    , "AVX512_BW-VL"],
        ["vptestnmw"        , "W:k {k}, ymm, ymm/m256"                          , "RVM-FVM" , "EVEX.256.F3.0F38.W1 26 /r"    , "AVX512_BW-VL"],
        ["vptestnmw"        , "W:k {k}, zmm, zmm/m512"                          , "RVM-FVM" , "EVEX.512.F3.0F38.W1 26 /r"    , "AVX512_BW"],
        ["vpunpckhbw"       , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG 68 /r"     , "AVX512_BW-VL"],
        ["vpunpckhbw"       , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG 68 /r"     , "AVX512_BW-VL"],
        ["vpunpckhbw"       , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG 68 /r"     , "AVX512_BW"],
        ["vpunpckhdq"       , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F.W0 6A /r"      , "AVX512_F-VL"],
        ["vpunpckhdq"       , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F.W0 6A /r"      , "AVX512_F-VL"],
        ["vpunpckhdq"       , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F.W0 6A /r"      , "AVX512_F"],
        ["vpunpckhqdq"      , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 6D /r"      , "AVX512_F-VL"],
        ["vpunpckhqdq"      , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 6D /r"      , "AVX512_F-VL"],
        ["vpunpckhqdq"      , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F.W1 6D /r"      , "AVX512_F"],
        ["vpunpckhwd"       , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG 69 /r"     , "AVX512_BW-VL"],
        ["vpunpckhwd"       , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG 69 /r"     , "AVX512_BW-VL"],
        ["vpunpckhwd"       , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG 69 /r"     , "AVX512_BW"],
        ["vpunpcklbw"       , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG 60 /r"     , "AVX512_BW-VL"],
        ["vpunpcklbw"       , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG 60 /r"     , "AVX512_BW-VL"],
        ["vpunpcklbw"       , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG 60 /r"     , "AVX512_BW"],
        ["vpunpckldq"       , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F.W0 62 /r"      , "AVX512_F-VL"],
        ["vpunpckldq"       , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F.W0 62 /r"      , "AVX512_F-VL"],
        ["vpunpckldq"       , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F.W0 62 /r"      , "AVX512_F"],
        ["vpunpcklqdq"      , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 6C /r"      , "AVX512_F-VL"],
        ["vpunpcklqdq"      , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 6C /r"      , "AVX512_F-VL"],
        ["vpunpcklqdq"      , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F.W1 6C /r"      , "AVX512_F"],
        ["vpunpcklwd"       , "W:xmm {kz}, xmm, xmm/m128"                       , "RVM-FVM" , "EVEX.128.66.0F.WIG 61 /r"     , "AVX512_BW-VL"],
        ["vpunpcklwd"       , "W:ymm {kz}, ymm, ymm/m256"                       , "RVM-FVM" , "EVEX.256.66.0F.WIG 61 /r"     , "AVX512_BW-VL"],
        ["vpunpcklwd"       , "W:zmm {kz}, zmm, zmm/m512"                       , "RVM-FVM" , "EVEX.512.66.0F.WIG 61 /r"     , "AVX512_BW"],
        ["vpxord"           , "W:xmm {kz},~xmm,~xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F.W0 EF /r"      , "AVX512_F-VL"],
        ["vpxord"           , "W:ymm {kz},~ymm,~ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F.W0 EF /r"      , "AVX512_F-VL"],
        ["vpxord"           , "W:zmm {kz},~zmm,~zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.66.0F.W0 EF /r"      , "AVX512_F"],
        ["vpxorq"           , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 EF /r"      , "AVX512_F-VL"],
        ["vpxorq"           , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 EF /r"      , "AVX512_F-VL"],
        ["vpxorq"           , "W:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F.W1 EF /r"      , "AVX512_F"],
        ["vrangepd"         , "W:xmm {kz}, xmm, xmm/m128/b64, ib/ub"            , "RVMI-FV" , "EVEX.128.66.0F3A.W1 50 /r ib" , "AVX512_DQ-VL"],
        ["vrangepd"         , "W:ymm {kz}, ymm, ymm/m256/b64, ib/ub"            , "RVMI-FV" , "EVEX.256.66.0F3A.W1 50 /r ib" , "AVX512_DQ-VL"],
        ["vrangepd"         , "W:zmm {kz}, zmm, zmm/m512/b64, ib/ub {sae}"      , "RVMI-FV" , "EVEX.512.66.0F3A.W1 50 /r ib" , "AVX512_DQ"],
        ["vrangeps"         , "W:xmm {kz}, xmm, xmm/m128/b32, ib/ub"            , "RVMI-FV" , "EVEX.128.66.0F3A.W0 50 /r ib" , "AVX512_DQ-VL"],
        ["vrangeps"         , "W:ymm {kz}, ymm, ymm/m256/b32, ib/ub"            , "RVMI-FV" , "EVEX.256.66.0F3A.W0 50 /r ib" , "AVX512_DQ-VL"],
        ["vrangeps"         , "W:zmm {kz}, zmm, zmm/m512/b32, ib/ub {sae}"      , "RVMI-FV" , "EVEX.512.66.0F3A.W0 50 /r ib" , "AVX512_DQ"],
        ["vrangesd"         , "W:xmm {kz},xmm[127:64],xmm[63:0]/m64,ib/ub {sae}", "RVMI-T1S", "EVEX.LIG.66.0F3sA.W1 51 /r ib" , "AVX512_DQ"],
        ["vrangess"         , "W:xmm {kz},xmm[127:32],xmm[31:0]/m32,ib/ub {sae}", "RVMI-T1S", "EVEX.LIG.66.0F3A.W0 51 /r ib" , "AVX512_DQ"],
        ["vrcp14pd"         , "W:xmm {kz}, xmm/m128/b64"                        , "RM-FV"   , "EVEX.128.66.0F38.W1 4C /r"    , "AVX512_F-VL"],
        ["vrcp14pd"         , "W:ymm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.66.0F38.W1 4C /r"    , "AVX512_F-VL"],
        ["vrcp14pd"         , "W:zmm {kz}, zmm/m512/b64"                        , "RM-FV"   , "EVEX.512.66.0F38.W1 4C /r"    , "AVX512_F"],
        ["vrcp14ps"         , "W:xmm {kz}, xmm/m128/b32"                        , "RM-FV"   , "EVEX.128.66.0F38.W0 4C /r"    , "AVX512_F-VL"],
        ["vrcp14ps"         , "W:ymm {kz}, ymm/m256/b32"                        , "RM-FV"   , "EVEX.256.66.0F38.W0 4C /r"    , "AVX512_F-VL"],
        ["vrcp14ps"         , "W:zmm {kz}, zmm/m512/b32"                        , "RM-FV"   , "EVEX.512.66.0F38.W0 4C /r"    , "AVX512_F"],
        ["vrcp14sd"         , "W:xmm {kz}, xmm[127:64], xmm[63:0]/m64"          , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 4D /r"    , "AVX512_F"],
        ["vrcp14ss"         , "W:xmm {kz}, xmm[127:32], xmm[31:0]/m32"          , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 4D /r"    , "AVX512_F"],
        ["vrcp28pd"         , "W:zmm {kz}, zmm/m512/b64 {sae}"                  , "RM-FV"   , "EVEX.512.66.0F38.W1 CA /r"    , "AVX512_ERI"],
        ["vrcp28ps"         , "W:zmm {kz}, zmm/m512/b32 {sae}"                  , "RM-FV"   , "EVEX.512.66.0F38.W0 CA /r"    , "AVX512_ERI"],
        ["vrcp28sd"         , "W:xmm {kz}, xmm[127:64], xmm[63:0]/m64 {sae}"    , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 CB /r"    , "AVX512_ERI"],
        ["vrcp28ss"         , "W:xmm {kz}, xmm[127:32], xmm[31:0]/m32 {sae}"    , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 CB /r"    , "AVX512_ERI"],
        ["vreducepd"        , "W:xmm {kz}, xmm/m128/b64, ib/ub"                 , "RMI-FV"  , "EVEX.128.66.0F3A.W1 56 /r ib" , "AVX512_DQ-VL"],
        ["vreducepd"        , "W:ymm {kz}, ymm/m256/b64, ib/ub"                 , "RMI-FV"  , "EVEX.256.66.0F3A.W1 56 /r ib" , "AVX512_DQ-VL"],
        ["vreducepd"        , "W:zmm {kz}, zmm/m512/b64, ib/ub"                 , "RMI-FV"  , "EVEX.512.66.0F3A.W1 56 /r ib" , "AVX512_DQ"],
        ["vreduceps"        , "W:xmm {kz}, xmm/m128/b32, ib/ub"                 , "RMI-FV"  , "EVEX.128.66.0F3A.W0 56 /r ib" , "AVX512_DQ-VL"],
        ["vreduceps"        , "W:ymm {kz}, ymm/m256/b32, ib/ub"                 , "RMI-FV"  , "EVEX.256.66.0F3A.W0 56 /r ib" , "AVX512_DQ-VL"],
        ["vreduceps"        , "W:zmm {kz}, zmm/m512/b32, ib/ub"                 , "RMI-FV"  , "EVEX.512.66.0F3A.W0 56 /r ib" , "AVX512_DQ"],
        ["vreducesd"        , "W:xmm {kz}, xmm[127:64], xmm[63:0]/m64, ib/ub"   , "RVMI-T1S", "EVEX.LIG.66.0F3A.W1 57 /r ib" , "AVX512_DQ"],
        ["vreducess"        , "W:xmm {kz}, xmm[127:32], xmm[31:0]/m32, ib/ub"   , "RVMI-T1S", "EVEX.LIG.66.0F3A.W0 57 /r ib" , "AVX512_DQ"],
        ["vrndscalepd"      , "W:xmm {kz}, xmm/m128/b64, ib/ub"                 , "RMI-FV"  , "EVEX.128.66.0F3A.W1 09 /r ib" , "AVX512_F-VL"],
        ["vrndscalepd"      , "W:ymm {kz}, ymm/m256/b64, ib/ub"                 , "RMI-FV"  , "EVEX.256.66.0F3A.W1 09 /r ib" , "AVX512_F-VL"],
        ["vrndscalepd"      , "W:zmm {kz}, zmm/m512/b64, ib/ub {sae}"           , "RMI-FV"  , "EVEX.512.66.0F3A.W1 09 /r ib" , "AVX512_F"],
        ["vrndscaleps"      , "W:xmm {kz}, xmm/m128/b32, ib/ub"                 , "RMI-FV"  , "EVEX.128.66.0F3A.W0 08 /r ib" , "AVX512_F-VL"],
        ["vrndscaleps"      , "W:ymm {kz}, ymm/m256/b32, ib/ub"                 , "RMI-FV"  , "EVEX.256.66.0F3A.W0 08 /r ib" , "AVX512_F-VL"],
        ["vrndscaleps"      , "W:zmm {kz}, zmm/m512/b32, ib/ub {sae}"           , "RMI-FV"  , "EVEX.512.66.0F3A.W0 08 /r ib" , "AVX512_F"],
        ["vrndscalesd"      , "W:xmm {kz},xmm[127:64],xmm[63:0]/m64,ib/ub {sae}", "RVMI-T1S", "EVEX.LIG.66.0F3A.W1 0B /r ib" , "AVX512_F"],
        ["vrndscaless"      , "W:xmm {kz},xmm[127:32],xmm[31:0]/m32,ib/ub {sae}", "RVMI-T1S", "EVEX.LIG.66.0F3A.W0 0A /r ib" , "AVX512_F"],
        ["vrsqrt14pd"       , "W:xmm {kz}, xmm/m128/b64"                        , "RM-FV"   , "EVEX.128.66.0F38.W1 4E /r"    , "AVX512_F-VL"],
        ["vrsqrt14pd"       , "W:ymm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.66.0F38.W1 4E /r"    , "AVX512_F-VL"],
        ["vrsqrt14pd"       , "W:zmm {kz}, zmm/m512/b64"                        , "RM-FV"   , "EVEX.512.66.0F38.W1 4E /r"    , "AVX512_F"],
        ["vrsqrt14ps"       , "W:xmm {kz}, xmm/m128/b32"                        , "RM-FV"   , "EVEX.128.66.0F38.W0 4E /r"    , "AVX512_F-VL"],
        ["vrsqrt14ps"       , "W:ymm {kz}, ymm/m256/b32"                        , "RM-FV"   , "EVEX.256.66.0F38.W0 4E /r"    , "AVX512_F-VL"],
        ["vrsqrt14ps"       , "W:zmm {kz}, zmm/m512/b32"                        , "RM-FV"   , "EVEX.512.66.0F38.W0 4E /r"    , "AVX512_F"],
        ["vrsqrt14sd"       , "W:xmm {kz}, xmm[127:64], xmm[63:0]/m64"          , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 4F /r"    , "AVX512_F"],
        ["vrsqrt14ss"       , "W:xmm {kz}, xmm[127:32], xmm[31:0]/m32"          , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 4F /r"    , "AVX512_F"],
        ["vrsqrt28pd"       , "W:zmm {kz}, zmm/m512/b64 {sae}"                  , "RM-FV"   , "EVEX.512.66.0F38.W1 CC /r"    , "AVX512_ERI"],
        ["vrsqrt28ps"       , "W:zmm {kz}, zmm/m512/b32 {sae}"                  , "RM-FV"   , "EVEX.512.66.0F38.W0 CC /r"    , "AVX512_ERI"],
        ["vrsqrt28sd"       , "W:xmm {kz}, xmm[127:64], xmm[63:0]/m64 {sae}"    , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 CD /r"    , "AVX512_ERI"],
        ["vrsqrt28ss"       , "W:xmm {kz}, xmm[127:32], xmm[31:0]/m32 {sae}"    , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 CD /r"    , "AVX512_ERI"],
        ["vscalefpd"        , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F38.W1 2C /r"    , "AVX512_F-VL"],
        ["vscalefpd"        , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F38.W1 2C /r"    , "AVX512_F-VL"],
        ["vscalefpd"        , "W:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W1 2C /r"    , "AVX512_F"],
        ["vscalefps"        , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.66.0F38.W0 2C /r"    , "AVX512_F-VL"],
        ["vscalefps"        , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.66.0F38.W0 2C /r"    , "AVX512_F-VL"],
        ["vscalefps"        , "W:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.66.0F38.W0 2C /r"    , "AVX512_F"],
        ["vscalefsd"        , "W:xmm {kz}, xmm, xmm[63:0]/m64 {er}"             , "RVM-T1S" , "EVEX.LIG.66.0F38.W1 2D /r"    , "AVX512_F"],
        ["vscalefss"        , "W:xmm {kz}, xmm, xmm[31:0]/m32 {er}"             , "RVM-T1S" , "EVEX.LIG.66.0F38.W0 2D /r"    , "AVX512_F"],
        ["vscatterdpd"      , "W:vm32x {k}, xmm"                                , "MR-T1S"  , "EVEX.128.66.0F38.W1 A2 /r"    , "AVX512_F-VL"],
        ["vscatterdpd"      , "W:vm32x {k}, ymm"                                , "MR-T1S"  , "EVEX.256.66.0F38.W1 A2 /r"    , "AVX512_F-VL"],
        ["vscatterdpd"      , "W:vm32y {k}, zmm"                                , "MR-T1S"  , "EVEX.512.66.0F38.W1 A2 /r"    , "AVX512_F"],
        ["vscatterdps"      , "W:vm32x {k}, xmm"                                , "MR-T1S"  , "EVEX.128.66.0F38.W0 A2 /r"    , "AVX512_F-VL"],
        ["vscatterdps"      , "W:vm32y {k}, ymm"                                , "MR-T1S"  , "EVEX.256.66.0F38.W0 A2 /r"    , "AVX512_F-VL"],
        ["vscatterdps"      , "W:vm32z {k}, zmm"                                , "MR-T1S"  , "EVEX.512.66.0F38.W0 A2 /r"    , "AVX512_F"],
        ["vscatterpf0dpd"   , "R:vm32y {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W1 C6 /5"    , "AVX512_PFI"],
        ["vscatterpf0dps"   , "R:vm32z {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W0 C6 /5"    , "AVX512_PFI"],
        ["vscatterpf0qpd"   , "R:vm64z {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W1 C7 /5"    , "AVX512_PFI"],
        ["vscatterpf0qps"   , "R:vm64z {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W0 C7 /5"    , "AVX512_PFI"],
        ["vscatterpf1dpd"   , "R:vm32y {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W1 C6 /6"    , "AVX512_PFI"],
        ["vscatterpf1dps"   , "R:vm32z {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W0 C6 /6"    , "AVX512_PFI"],
        ["vscatterpf1qpd"   , "R:vm64z {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W1 C7 /6"    , "AVX512_PFI"],
        ["vscatterpf1qps"   , "R:vm64z {k}"                                     , "M-T1S"   , "EVEX.512.66.0F38.W0 C7 /6"    , "AVX512_PFI"],
        ["vscatterqpd"      , "W:vm64x {k}, xmm"                                , "MR-T1S"  , "EVEX.128.66.0F38.W1 A3"       , "AVX512_F-VL"],
        ["vscatterqpd"      , "W:vm64y {k}, ymm"                                , "MR-T1S"  , "EVEX.256.66.0F38.W1 A3"       , "AVX512_F-VL"],
        ["vscatterqpd"      , "W:vm64z {k}, zmm"                                , "MR-T1S"  , "EVEX.512.66.0F38.W1 A3"       , "AVX512_F"],
        ["vscatterqps"      , "W:vm64x {k}, xmm"                                , "MR-T1S"  , "EVEX.128.66.0F38.W0 A3"       , "AVX512_F-VL"],
        ["vscatterqps"      , "W:vm64y {k}, xmm"                                , "MR-T1S"  , "EVEX.256.66.0F38.W0 A3"       , "AVX512_F-VL"],
        ["vscatterqps"      , "W:vm64z {k}, ymm"                                , "MR-T1S"  , "EVEX.512.66.0F38.W0 A3"       , "AVX512_F"],
        ["vshuff32x4"       , "W:ymm {kz}, ymm, ymm/m256/b32, ib/ub"            , "RVMI-FV" , "EVEX.256.66.0F3A.W0 23 /r ib" , "AVX512_F-VL"],
        ["vshuff32x4"       , "W:zmm {kz}, zmm, zmm/m512/b32, ib/ub"            , "RVMI-FV" , "EVEX.512.66.0F3A.W0 23 /r ib" , "AVX512_F"],
        ["vshuff64x2"       , "W:ymm {kz}, ymm, ymm/m256/b64, ib/ub"            , "RVMI-FV" , "EVEX.256.66.0F3A.W1 23 /r ib" , "AVX512_F-VL"],
        ["vshuff64x2"       , "W:zmm {kz}, zmm, zmm/m512/b64, ib/ub"            , "RVMI-FV" , "EVEX.512.66.0F3A.W1 23 /r ib" , "AVX512_F"],
        ["vshufi32x4"       , "W:ymm {kz}, ymm, ymm/m256/b32, ib/ub"            , "RVMI-FV" , "EVEX.256.66.0F3A.W0 43 /r ib" , "AVX512_F-VL"],
        ["vshufi32x4"       , "W:zmm {kz}, zmm, zmm/m512/b32, ib/ub"            , "RVMI-FV" , "EVEX.512.66.0F3A.W0 43 /r ib" , "AVX512_F"],
        ["vshufi64x2"       , "W:ymm {kz}, ymm, ymm/m256/b64, ib/ub"            , "RVMI-FV" , "EVEX.256.66.0F3A.W1 43 /r ib" , "AVX512_F-VL"],
        ["vshufi64x2"       , "W:zmm {kz}, zmm, zmm/m512/b64, ib/ub"            , "RVMI-FV" , "EVEX.512.66.0F3A.W1 43 /r ib" , "AVX512_F"],
        ["vshufpd"          , "W:xmm {kz}, xmm, xmm/m128/b32, ib/ub"            , "RVMI-FV" , "EVEX.128.66.0F.W1 C6 /r ib"   , "AVX512_F-VL"],
        ["vshufpd"          , "W:ymm {kz}, ymm, ymm/m256/b32, ib/ub"            , "RVMI-FV" , "EVEX.256.66.0F.W1 C6 /r ib"   , "AVX512_F-VL"],
        ["vshufpd"          , "W:zmm {kz}, zmm, zmm/m512/b32, ib/ub"            , "RVMI-FV" , "EVEX.512.66.0F.W1 C6 /r ib"   , "AVX512_F"],
        ["vshufps"          , "W:xmm {kz}, xmm, xmm/m128/b64, ib/ub"            , "RVMI-FV" , "EVEX.128.0F.W0 C6 /r ib"      , "AVX512_F-VL"],
        ["vshufps"          , "W:ymm {kz}, ymm, ymm/m256/b64, ib/ub"            , "RVMI-FV" , "EVEX.256.0F.W0 C6 /r ib"      , "AVX512_F-VL"],
        ["vshufps"          , "W:zmm {kz}, zmm, zmm/m512/b64, ib/ub"            , "RVMI-FV" , "EVEX.512.0F.W0 C6 /r ib"      , "AVX512_F"],
        ["vsqrtpd"          , "W:xmm {kz}, xmm/m128/b64"                        , "RM-FV"   , "EVEX.128.66.0F.W1 51 /r"      , "AVX512_F-VL"],
        ["vsqrtpd"          , "W:ymm {kz}, ymm/m256/b64"                        , "RM-FV"   , "EVEX.256.66.0F.W1 51 /r"      , "AVX512_F-VL"],
        ["vsqrtpd"          , "W:zmm {kz}, zmm/m512/b64 {er}"                   , "RM-FV"   , "EVEX.512.66.0F.W1 51 /r"      , "AVX512_F"],
        ["vsqrtps"          , "W:xmm {kz}, xmm/m128/b32"                        , "RM-FV"   , "EVEX.128.0F.W0 51 /r"         , "AVX512_F-VL"],
        ["vsqrtps"          , "W:ymm {kz}, ymm/m256/b32"                        , "RM-FV"   , "EVEX.256.0F.W0 51 /r"         , "AVX512_F-VL"],
        ["vsqrtps"          , "W:zmm {kz}, zmm/m512/b32 {er}"                   , "RM-FV"   , "EVEX.512.0F.W0 51 /r"         , "AVX512_F"],
        ["vsqrtsd"          , "W:xmm {kz}, xmm[127:64], xmm[63:0]/m64 {er}"     , "RVM-T1S" , "EVEX.LIG.F2.0F.W1 51 /r"      , "AVX512_F"],
        ["vsqrtss"          , "W:xmm {kz}, xmm[127:32], xmm[31:0]/m32 {er}"     , "RVM-T1S" , "EVEX.LIG.F3.0F.W0 51 /r"      , "AVX512_F"],
        ["vsubpd"           , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 5C /r"      , "AVX512_F-VL"],
        ["vsubpd"           , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 5C /r"      , "AVX512_F-VL"],
        ["vsubpd"           , "W:zmm {kz}, zmm, zmm/m512/b64 {er}"              , "RVM-FV"  , "EVEX.512.66.0F.W1 5C /r"      , "AVX512_F"],
        ["vsubps"           , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.0F.W0 5C /r"         , "AVX512_F-VL"],
        ["vsubps"           , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.0F.W0 5C /r"         , "AVX512_F-VL"],
        ["vsubps"           , "W:zmm {kz}, zmm, zmm/m512/b32 {er}"              , "RVM-FV"  , "EVEX.512.0F.W0 5C /r"         , "AVX512_F"],
        ["vsubsd"           , "W:xmm {kz}, xmm, xmm[63:0]/m64 {er}"             , "RVM-T1S" , "EVEX.LIG.F2.0F.W1 5C /r"      , "AVX512_F"],
        ["vsubss"           , "W:xmm {kz}, xmm, xmm[31:0]/m32 {er}"             , "RVM-T1S" , "EVEX.LIG.F3.0F.W0 5C /r"      , "AVX512_F"],
        ["vucomisd"         , "R:xmm[63:0], xmm[63:0]/m64 {sae}"                , "RM-T1S"  , "EVEX.LIG.66.0F.W1 2E /r"      , "AVX512_F OF=0 SF=0 ZF=W AF=0 PF=W CF=W"],
        ["vucomiss"         , "R:xmm[31:0], xmm[31:0]/m32 {sae}"                , "RM-T1S"  , "EVEX.LIG.0F.W0 2E /r"         , "AVX512_F OF=0 SF=0 ZF=W AF=0 PF=W CF=W"],
        ["vunpckhpd"        , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 15 /r"      , "AVX512_F-VL"],
        ["vunpckhpd"        , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 15 /r"      , "AVX512_F-VL"],
        ["vunpckhpd"        , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F.W1 15 /r"      , "AVX512_F"],
        ["vunpckhps"        , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.0F.W0 15 /r"         , "AVX512_F-VL"],
        ["vunpckhps"        , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.0F.W0 15 /r"         , "AVX512_F-VL"],
        ["vunpckhps"        , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.0F.W0 15 /r"         , "AVX512_F"],
        ["vunpcklpd"        , "W:xmm {kz}, xmm, xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 14 /r"      , "AVX512_F-VL"],
        ["vunpcklpd"        , "W:ymm {kz}, ymm, ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 14 /r"      , "AVX512_F-VL"],
        ["vunpcklpd"        , "W:zmm {kz}, zmm, zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F.W1 14 /r"      , "AVX512_F"],
        ["vunpcklps"        , "W:xmm {kz}, xmm, xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.0F.W0 14 /r"         , "AVX512_F-VL"],
        ["vunpcklps"        , "W:ymm {kz}, ymm, ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.0F.W0 14 /r"         , "AVX512_F-VL"],
        ["vunpcklps"        , "W:zmm {kz}, zmm, zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.0F.W0 14 /r"         , "AVX512_F"],
        ["vxorpd"           , "W:xmm {kz},~xmm,~xmm/m128/b64"                   , "RVM-FV"  , "EVEX.128.66.0F.W1 57 /r"      , "AVX512_DQ-VL"],
        ["vxorpd"           , "W:ymm {kz},~ymm,~ymm/m256/b64"                   , "RVM-FV"  , "EVEX.256.66.0F.W1 57 /r"      , "AVX512_DQ-VL"],
        ["vxorpd"           , "W:zmm {kz},~zmm,~zmm/m512/b64"                   , "RVM-FV"  , "EVEX.512.66.0F.W1 57 /r"      , "AVX512_DQ"],
        ["vxorps"           , "W:xmm {kz},~xmm,~xmm/m128/b32"                   , "RVM-FV"  , "EVEX.128.0F.W0 57 /r"         , "AVX512_DQ-VL"],
        ["vxorps"           , "W:ymm {kz},~ymm,~ymm/m256/b32"                   , "RVM-FV"  , "EVEX.256.0F.W0 57 /r"         , "AVX512_DQ-VL"],
        ["vxorps"           , "W:zmm {kz},~zmm,~zmm/m512/b32"                   , "RVM-FV"  , "EVEX.512.0F.W0 57 /r"         , "AVX512_DQ"]
      ]
    }
    // ${JSON:END}
    ;
    
    }).apply(this, typeof module === "object" && module && module.exports
      ? [module, "exports"] : [this.asmdb || (this.asmdb = {}), "x86data"]);
    