/**
 * @author Jiyao Wang <wangjiy@ncbi.nlm.nih.gov> / https://github.com/ncbi/icn3d
 */

class Analysis {
    constructor(icn3d) {
        this.icn3d = icn3d;
    }

    calculateArea() {var ic = this.icn3d, me = ic.icn3dui;
       ic.bCalcArea = true;
       ic.opts.surface = 'solvent accessible surface';
       ic.applyMapCls.applySurfaceOptions();
       $("#" + ic.pre + "areavalue").val(ic.areavalue);
       $("#" + ic.pre + "areatable").html(ic.areahtml);
       me.htmlCls.dialogCls.openDlg('dl_area', 'Surface area calculation');
       ic.bCalcArea = false;
    }

    calcBuriedSurface(nameArray2, nameArray) {var ic = this.icn3d, me = ic.icn3dui;
       if(nameArray2.length == 0) {
           alert("Please select the first set");
       }
       else {
           let prevHAtoms = me.hashUtilsCls.cloneHash(ic.hAtoms);
           let atomSet2 = ic.definedSetsCls.getAtomsFromNameArray(nameArray2);
           let atomSet1 = ic.definedSetsCls.getAtomsFromNameArray(nameArray);
           ic.bCalcArea = true;
           ic.opts.surface = 'solvent accessible surface';
           ic.hAtoms = me.hashUtilsCls.cloneHash(atomSet2);
           ic.applyMapCls.applySurfaceOptions();
           let area2 = ic.areavalue;
           let resid2area2 = me.hashUtilsCls.cloneHash(ic.resid2area);
           ic.hAtoms = me.hashUtilsCls.cloneHash(atomSet1);
           ic.applyMapCls.applySurfaceOptions();
           let area1 = ic.areavalue;
           let resid2area1 = me.hashUtilsCls.cloneHash(ic.resid2area);

           ic.hAtoms = me.hashUtilsCls.unionHash(ic.hAtoms, atomSet2);
           ic.applyMapCls.applySurfaceOptions();
           let areaTotal = ic.areavalue;
           let resid2areaTotal = me.hashUtilsCls.cloneHash(ic.resid2area);

           let buriedArea1 = 0, buriedArea2 = 0;
           let areaSum1 = 0, areaSum2 = 0;
           // set 1 buried
           for(let resid in resid2area2) {
               if(resid2areaTotal.hasOwnProperty(resid)) {
                   areaSum2 += parseFloat(resid2areaTotal[resid]);
               }
           }
           buriedArea2 = (area2 - areaSum2).toFixed(2);

           // set 2 buried
           for(let resid in resid2area1) {
               if(resid2areaTotal.hasOwnProperty(resid)) {
                   areaSum1 += parseFloat(resid2areaTotal[resid]);
               }
           }
           buriedArea1 = (area1 - areaSum1).toFixed(2);

           ic.bCalcArea = false;
           ic.hAtoms = me.hashUtilsCls.cloneHash(prevHAtoms);
           let buriedArea =(parseFloat(area1) + parseFloat(area2) - parseFloat(areaTotal)).toFixed(2);
           let html = '<br>Calculate solvent accessible surface area in the interface:<br><br>';
           html += 'Set 1: ' + nameArray2 + ', Surface: ' +  area2 + ' &#8491;<sup>2</sup><br>';
           html += 'Set 2: ' + nameArray + ', Surface: ' +  area1 + ' &#8491;<sup>2</sup><br>';
           html += 'Total Surface: ' +  areaTotal + ' &#8491;<sup>2</sup><br>';
           //html += '<b>Buried Surface for both Sets</b>: ' +  buriedArea + ' &#8491;<sup>2</sup><br>';
           html += '<b>Buried Surface for Set 1</b>: ' +  buriedArea2 + ' &#8491;<sup>2</sup><br>';
           html += '<b>Buried Surface for Set 2</b>: ' +  buriedArea1 + ' &#8491;<sup>2</sup><br><br>';
           $("#" + ic.pre + "dl_buriedarea_html").html(html);
           me.htmlCls.dialogCls.openDlg('dl_buriedarea', 'Buried solvent accessible surface area in the interface');
           me.htmlCls.clickMenuCls.setLogCmd('buried surface ' + buriedArea, false);
       }
    }

    measureDistTwoSets(nameArray, nameArray2) {var ic = this.icn3d, me = ic.icn3dui;
       if(nameArray.length == 0 || nameArray2.length == 0) {
           alert("Please select two sets");
       }
       else {
           let prevHAtoms = me.hashUtilsCls.cloneHash(ic.hAtoms);
           let atomSet1 = ic.definedSetsCls.getAtomsFromNameArray(nameArray);
           let atomSet2 = ic.definedSetsCls.getAtomsFromNameArray(nameArray2);

           let posArray1 = ic.contactCls.getExtent(atomSet1);
           let posArray2 = ic.contactCls.getExtent(atomSet2);

           let pos1 = new THREE.Vector3(posArray1[2][0], posArray1[2][1], posArray1[2][2]);
           let pos2 = new THREE.Vector3(posArray2[2][0], posArray2[2][1], posArray2[2][2]);

           ic.hAtoms = me.hashUtilsCls.cloneHash(prevHAtoms);

           if(ic.distPnts === undefined) ic.distPnts = [];
           ic.distPnts.push(pos1);
           ic.distPnts.push(pos2);

           let color = $("#" + ic.pre + "distancecolor2" ).val();

           this.addLine(pos1.x, pos1.y, pos1.z, pos2.x, pos2.y, pos2.z, color, true, 'distance');

           let size = 0, background = 0;
           let labelPos = pos1.clone().add(pos2).multiplyScalar(0.5);
           let distance = parseInt(pos1.distanceTo(pos2) * 10) / 10;
           let text = distance.toString() + " A";
           this.addLabel(text, labelPos.x, labelPos.y, labelPos.z, size, color, background, 'distance');
           ic.drawCls.draw();
       }
    }

    measureDistManySets(nameArray, nameArray2) {var ic = this.icn3d, me = ic.icn3dui;
        if(nameArray.length == 0 || nameArray2.length == 0) {
            alert("Please select sets for distance calculation...");
        }
        else {

            let prevHAtoms = me.hashUtilsCls.cloneHash(ic.hAtoms);

            let distHash = {};

            for(let i = 0, il = nameArray.length; i < il; ++i) {
                let set1 = nameArray[i];
                let array1 = [set1];
                distHash[set1] = {};

                for(let j = 0, jl = nameArray2.length; j < jl; ++j) {
                    let set2 = nameArray2[j];
                    let array2 = [set2];

                    if(set1 == set2) continue;

                    let atomSet1 = ic.definedSetsCls.getAtomsFromNameArray(array1);
                    let atomSet2 = ic.definedSetsCls.getAtomsFromNameArray(array2);

                    let posArray1 = ic.contactCls.getExtent(atomSet1);
                    let posArray2 = ic.contactCls.getExtent(atomSet2);
        
                    let pos1 = new THREE.Vector3(posArray1[2][0], posArray1[2][1], posArray1[2][2]);
                    let pos2 = new THREE.Vector3(posArray2[2][0], posArray2[2][1], posArray2[2][2]);
        
                    let distance = pos1.distanceTo(pos2);

                    distHash[set1][set2] = distance.toFixed(2);
                }
            }

            ic.hAtoms = me.hashUtilsCls.cloneHash(prevHAtoms);

            let tableHtml = 'Note: Click on the distance to show a dashed line in 3D view.<br><br>';
            tableHtml += '<table align=center border=1 cellpadding=10 cellspacing=0><tr><th></th>';
            for(let j = 0, jl = nameArray2.length; j < jl; ++j) {
                let set2 = nameArray2[j];
                tableHtml += '<th><b>' + set2 + '</b> (&#8491;)</th>';
            }
            tableHtml += '</tr>';

            for(let i = 0, il = nameArray.length; i < il; ++i) {
                let set1 = nameArray[i];
                tableHtml += '<tr><th><b>' + set1 + '</b> (&#8491;)</th>';

                for(let j = 0, jl = nameArray2.length; j < jl; ++j) {
                    let set2 = nameArray2[j];

                    if(distHash[set1] && distHash[set1][set2]) {
                        tableHtml += '<td><span class="icn3d-distance" sets="' + set1 + '|' + set2 + '">' + distHash[set1][set2] + '</span></td>';
                    }
                    else {
                        tableHtml += '<td>0</td>';
                    }
                }

                tableHtml += '</tr>';
            }

            tableHtml += '</table><br><br>';

            $("#" + me.pre + "dl_disttable_html").html(tableHtml);
        }
    }

    measureAngleManySets(nameArray, nameArray2) {var ic = this.icn3d, me = ic.icn3dui;
        if(nameArray.length == 0 || nameArray2.length == 0) {
            alert("Please select sets for angleance calculation...");
        }
        else {
            let angleHash = {};

            for(let i = 0, il = nameArray.length; i < il; ++i) {
                let set1 = nameArray[i];
                let array1 = [set1];
                angleHash[set1] = {};

                ic.hAtoms = ic.definedSetsCls.getAtomsFromNameArray(array1);
                let axis1 = ic.axesCls.setPc1Axes(true);

                for(let j = 0, jl = nameArray2.length; j < jl; ++j) {
                    let set2 = nameArray2[j];
                    let array2 = [set2];

                    if(set1 == set2) continue;

                    ic.hAtoms = ic.definedSetsCls.getAtomsFromNameArray(array2);
                    let axis2 = ic.axesCls.setPc1Axes(true);

                    let angleRad = new THREE.Vector3(parseFloat(axis1.x), parseFloat(axis1.y), parseFloat(axis1.z)).angleTo(new THREE.Vector3(parseFloat(axis2.x), parseFloat(axis2.y), parseFloat(axis2.z)));
                    
                    let angle = angleRad / 3.1416 * 180;
                    angle = Math.abs(angle).toFixed(0);
                    if(angle > 180) angle -= 180;
                    if(angle > 90) angle = 180 - angle;

                    angleHash[set1][set2] = angle;
                }
            }

            let tableHtml = '<table align=center border=1 cellpadding=10 cellspacing=0><tr><th></th>';
            for(let j = 0, jl = nameArray2.length; j < jl; ++j) {
                let set2 = nameArray2[j];
                tableHtml += '<th><b>' + set2 + '</b> (&deg;)</th>';
            }
            tableHtml += '</tr>';

            for(let i = 0, il = nameArray.length; i < il; ++i) {
                let set1 = nameArray[i];
                tableHtml += '<tr><th><b>' + set1 + '</b> (&deg;)</th>';

                for(let j = 0, jl = nameArray2.length; j < jl; ++j) {
                    let set2 = nameArray2[j];

                    if(angleHash[set1] && angleHash[set1][set2]) {
                        tableHtml += '<td><span>' + angleHash[set1][set2] + '</span></td>';
                    }
                    else {
                        tableHtml += '<td>0</td>';
                    }
                }

                tableHtml += '</tr>';
            }

            tableHtml += '</table><br><br>';

            $("#" + me.pre + "dl_angletable_html").html(tableHtml);
        }
    }

    //Add a line between the position (x1, y1, z1) and the position (x2, y2, z2) with the input "color".
    //The line can be dashed if "dashed" is set true.
    addLine(x1, y1, z1, x2, y2, z2, color, dashed, type, radius, opacity) {var ic = this.icn3d, me = ic.icn3dui;
        let line = {} // Each line contains 'position1', 'position2', 'color', and a boolean of 'dashed'
        line.position1 = new THREE.Vector3(x1, y1, z1);
        line.position2 = new THREE.Vector3(x2, y2, z2);
        line.color = color;
        line.dashed = dashed;
        line.radius = radius;
        line.opacity = opacity;
        if(ic.lines[type] === undefined) ic.lines[type] = [];
        if(type !== undefined) {
            ic.lines[type].push(line);
        }
        else {
            if(ic.lines['custom'] === undefined) ic.lines['custom'] = [];
            ic.lines['custom'].push(line);
        }
        ic.hlObjectsCls.removeHlObjects();
        //ic.drawCls.draw();
    }

    addLineFromPicking(type) {var ic = this.icn3d, me = ic.icn3dui;
        let size = 0, background = 0;
        let color = $("#" + ic.pre + type + "color" ).val();
        let x =(ic.pAtom.coord.x + ic.pAtom2.coord.x) / 2;
        let y =(ic.pAtom.coord.y + ic.pAtom2.coord.y) / 2;
        let z =(ic.pAtom.coord.z + ic.pAtom2.coord.z) / 2;
        let dashed =(type == 'stabilizer') ? false : true;
        me.htmlCls.clickMenuCls.setLogCmd('add line | x1 ' + ic.pAtom.coord.x.toPrecision(4)  + ' y1 ' + ic.pAtom.coord.y.toPrecision(4) + ' z1 ' + ic.pAtom.coord.z.toPrecision(4) + ' | x2 ' + ic.pAtom2.coord.x.toPrecision(4)  + ' y2 ' + ic.pAtom2.coord.y.toPrecision(4) + ' z2 ' + ic.pAtom2.coord.z.toPrecision(4) + ' | color ' + color + ' | dashed ' + dashed + ' | type ' + type, true);
        this.addLine(ic.pAtom.coord.x, ic.pAtom.coord.y, ic.pAtom.coord.z, ic.pAtom2.coord.x, ic.pAtom2.coord.y, ic.pAtom2.coord.z, color, dashed, type);
        ic.pickpair = false;
    }

    //Add a "text" at the position (x, y, z) with the input "size", "color", and "background".
    addLabel(text, x, y, z, size, color, background, type) {var ic = this.icn3d, me = ic.icn3dui;
        let label = {}; // Each label contains 'position', 'text', 'color', 'background'

        if(size === '0' || size === '' || size === 'undefined') size = undefined;
        if(color === '0' || color === '' || color === 'undefined') color = undefined;
        if(background === '0' || background === '' || background === 'undefined') background = undefined;

        let position = new THREE.Vector3();
        position.x = x;
        position.y = y;
        position.z = z;

        label.position = position;

        label.text = text;
        label.size = size;
        label.color = color;
        label.background = background;

        if(ic.labels[type] === undefined) ic.labels[type] = [];

        if(type !== undefined) {
            ic.labels[type].push(label);
        }
        else {
            if(ic.labels['custom'] === undefined) ic.labels['custom'] = [];
            ic.labels['custom'].push(label);
        }

        ic.hlObjectsCls.removeHlObjects();

        //ic.drawCls.draw();
    }

    //Display chain name in the 3D structure display for the chains intersecting with the atoms in "atomHash".
    addChainLabels(atoms) {var ic = this.icn3d, me = ic.icn3dui;
        let size = 18;
        let background = "#FFFFFF"; //"#CCCCCC";
        let atomsHash = me.hashUtilsCls.intHash(ic.hAtoms, atoms);
        if(ic.labels['chain'] === undefined) ic.labels['chain'] = [];
        let chainHash = ic.firstAtomObjCls.getChainsFromAtoms(atomsHash);
        for(let chainid in chainHash) {
            let label = {}
            label.position = ic.applyCenterCls.centerAtoms(ic.chains[chainid]).center;
            let pos = chainid.indexOf('_');
            let chainName = chainid.substr(pos + 1);
            let proteinName = ic.showSeqCls.getProteinName(chainid);
            if(proteinName.length > 20) proteinName = proteinName.substr(0, 20) + '...';
            label.text = 'Chain ' + chainName + ': ' + proteinName;
            label.size = size;
            let atomColorStr = ic.firstAtomObjCls.getFirstCalphaAtomObj(ic.chains[chainid]).color.getHexString().toUpperCase();
            label.color = (ic.opts.background != 'black') ? ic.colorWhitebkgd : ic.colorBlackbkgd; //(atomColorStr === "CCCCCC" || atomColorStr === "C8C8C8") ? "#888888" : "#" + atomColorStr;
            label.background = background;
            ic.labels['chain'].push(label);
        }
        ic.hlObjectsCls.removeHlObjects();
    }
    //Display the terminal labels for the atoms in "atomHash". The termini of proteins are labelled
    //as "N-" and "C-". The termini of nucleotides are labeled as "5'" and "3'".
    addTerminiLabels(atoms) {var ic = this.icn3d, me = ic.icn3dui;
        let size = 18;
        let background = "#FFFFFF"; //"#CCCCCC";
        let protNucl;
        protNucl = me.hashUtilsCls.unionHash(protNucl, ic.proteins);
        protNucl = me.hashUtilsCls.unionHash(protNucl, ic.nucleotides);
        let hlProtNucl = me.hashUtilsCls.intHash(ic.dAtoms, protNucl);
        let atomsHash = me.hashUtilsCls.intHash(hlProtNucl, atoms);
        if(ic.labels['chain'] === undefined) ic.labels['chain'] = [];
        let chainHash = ic.firstAtomObjCls.getChainsFromAtoms(atomsHash);
        for(let chainid in chainHash) {
            let chainAtomsHash = me.hashUtilsCls.intHash(hlProtNucl, ic.chains[chainid]);
            let serialArray = Object.keys(chainAtomsHash);
            let firstAtom = ic.atoms[serialArray[0]];
            let lastAtom = ic.atoms[serialArray[serialArray.length - 1]];
            let labelN = {}, labelC = {}
            labelN.position = firstAtom.coord;
            labelC.position = lastAtom.coord;
            labelN.text = 'N-';
            labelC.text = 'C-';
            if(ic.nucleotides.hasOwnProperty(firstAtom.serial)) {
                labelN.text = "5'";
                labelC.text = "3'";
            }
            labelN.size = size;
            labelC.size = size;
            let atomNColorStr = firstAtom.color.getHexString().toUpperCase();
            let atomCColorStr = lastAtom.color.getHexString().toUpperCase();
            labelN.color = (ic.opts.background != 'black') ? ic.colorWhitebkgd : ic.colorBlackbkgd; //(atomNColorStr === "CCCCCC" || atomNColorStr === "C8C8C8") ? "#888888" : "#" + atomNColorStr;
            labelC.color = (ic.opts.background != 'black') ? ic.colorWhitebkgd : ic.colorBlackbkgd; //(atomCColorStr === "CCCCCC" || atomCColorStr === "C8C8C8") ? "#888888" : "#" + atomCColorStr;
            labelN.background = background;
            labelC.background = background;
            ic.labels['chain'].push(labelN);
            ic.labels['chain'].push(labelC);
        }
        ic.hlObjectsCls.removeHlObjects();
    }
}

export {Analysis}
