﻿using Microsoft.AspNetCore.Http;
using Microsoft.Extensions.Logging;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using Microsoft.Extensions.Caching.Memory;
using Microsoft.Extensions.Options;

namespace Mockaco
{
    internal class RequestMatchingMiddleware
    {
        private readonly RequestDelegate _next;
        private readonly ILogger<RequestMatchingMiddleware> _logger;

        public RequestMatchingMiddleware(RequestDelegate next, ILogger<RequestMatchingMiddleware> logger)
        {
            _next = next;
            _logger = logger;
        }

        public async Task Invoke(
            HttpContext httpContext,
            IMockacoContext mockacoContext,
            IScriptContext scriptContext,
            IMockProvider mockProvider,
            ITemplateTransformer templateTransformer,
            IEnumerable<IRequestMatcher> requestMatchers,
            IMemoryCache cache,
            IOptions<MockacoOptions> options
            )
        {
            await LogHttpContext(httpContext);

            await AttachRequestToScriptContext(httpContext, mockacoContext, scriptContext);

            if (mockacoContext.Errors.Any())
            {
                return;
            }

            foreach (var mock in mockProvider.GetMocks())
            {
                if (await requestMatchers.AllAsync(_ => _.IsMatch(httpContext.Request, mock)))
                {
                    cache.Set($"{nameof(RequestMatchingMiddleware)} {httpContext.Request.Path.Value}",new
                    {
                        Route = httpContext.Request.Path.Value,
                        Timestamp = $"{DateTime.Now.ToString("t")}",
                        Headers = LoadHeaders(httpContext, options.Value.VerificationIgnoredHeaders),
                        Body = await httpContext.Request.ReadBodyStream()
                    }, DateTime.Now.AddMinutes(options.Value.MatchedRoutesCacheDuration));

                    _logger.LogInformation("Incoming request matched {mock}", mock);

                    await scriptContext.AttachRouteParameters(httpContext.Request, mock);

                    var template = await templateTransformer.TransformAndSetVariables(mock.RawTemplate, scriptContext);

                    mockacoContext.Mock = mock;
                    mockacoContext.TransformedTemplate = template;

                    await _next(httpContext);

                    return;
                }
                else
                {
                    _logger.LogDebug("Incoming request didn't match {mock}", mock);
                }
            }

            _logger.LogInformation("Incoming request didn't match any mock");

            mockacoContext.Errors.Add(new Error("Incoming request didn't match any mock"));
        }

        //TODO Remove redundant code
        private async Task AttachRequestToScriptContext(HttpContext httpContext, IMockacoContext mockacoContext, IScriptContext scriptContext)
        {
            try
            {
                await scriptContext.AttachRequest(httpContext.Request);
            }
            catch (Exception ex)
            {
                mockacoContext.Errors.Add(new Error("An error occurred while reading request", ex));

                _logger.LogWarning(ex, "An error occurred while reading request");

                return;
            }
        }

        private async Task LogHttpContext(HttpContext httpContext)
        {
            _logger.LogInformation("Incoming request from {remoteIp}", httpContext.Connection.RemoteIpAddress);

            _logger.LogDebug("Headers: {headers}", httpContext.Request.Headers.ToJson());

            var body = await httpContext.Request.ReadBodyStream();

            if (string.IsNullOrEmpty(body))
            {
                _logger.LogDebug("Body is not present");
            }
            else
            {
                _logger.LogDebug("Body: {body}", body);
            }
        }

        private static IEnumerable<object> LoadHeaders(HttpContext httpContext, IEnumerable<string> verificationIgnoredHeaders)
        {            
            return from header in httpContext.Request.Headers.ToList()
                   where !verificationIgnoredHeaders.Any(opt => opt == header.Key)
                   select new { header.Key, Value = header.Value[0] };
        }
    }
}