import unittest
import holmes_extractor as holmes

holmes_manager = holmes.Manager(model="de_core_news_lg", number_of_workers=1)
holmes_manager.register_search_phrase("Ein großer Hund jagt eine Katze")
holmes_manager.register_search_phrase("Ein ENTITYPER geht in die Stadt")


class EnglishDocumentationExamplesTest(unittest.TestCase):

    positive_examples = (
        "Der große Hund hat die Katze ständig gejagt",
        "Der große Hund, der müde war, jagte die Katze",
        "Die Katze wurde vom großen Hund gejagt",
        "Die Katze wurde immer wieder durch den großen Hund gejagt",
        "Der große Hund wollte die Katze jagen",
        "Der große Hund entschied sich, die Katze zu jagen",
        "Die Katze, die der große Hund gejagt hatte, hatte Angst",
        "Dass der große Hund die Katze jagte, war ein Problem",
        "Es gab einen großen Hund, der eine Katze jagte",
        "Die Katzenjagd durch den großen Hund",
        "Es gab einmal einen großen Hund, und er jagte eine Katze",
        "Es gab einen großen Hund. Er hieß Fido. Er jagte meine Katze",
        "Es erschien ein Hund. Er jagte eine Katze. Er war sehr groß.",
        "Die Katze schlich sich in unser Wohnzimmer zurück, weil ein großer Hund sie draußen gejagt hatte",
        "Unser großer Hund war aufgeregt, weil er eine Katze gejagt hatte",
    )

    def test_positive_examples(self):
        for positive_example in self.positive_examples:
            with self.subTest():
                assert len(holmes_manager.match(document_text=positive_example)) == 1

    negative_examples = (
        "Der Hund jagte eine große Katze",
        "Die Katze jagte den großen Hund",
        "Der große Hund und die Katze jagten",
        "Der große Hund jagte eine Maus aber die Katze war müde",
        "Der große Hund wurde ständig von der Katze gejagt",
        "Der große Hund entschloss sich, von der Katze gejagt zu werden",
        "Die Hundejagd durch den große Katze",
    )

    def test_negative_examples(self):
        for negative_example in self.negative_examples:
            with self.subTest():
                assert len(holmes_manager.match(document_text=negative_example)) == 0

    def test_complex_example(self):
        matches = holmes_manager.match(
            document_text="Letzte Woche sah ich Richard Hudson und Max Mustermann. Sie wollten nicht mehr in die Stadt gehen."
        )
        self.assertEqual(
            matches,
            [
                {
                    "search_phrase_label": "Ein ENTITYPER geht in die Stadt",
                    "search_phrase_text": "Ein ENTITYPER geht in die Stadt",
                    "document": "",
                    "index_within_document": 17,
                    "sentences_within_document": "Letzte Woche sah ich Richard Hudson und Max Mustermann. Sie wollten nicht mehr in die Stadt gehen.",
                    "negated": True,
                    "uncertain": True,
                    "involves_coreference": True,
                    "overall_similarity_measure": 1.0,
                    "word_matches": [
                        {
                            "search_phrase_token_index": 1,
                            "search_phrase_word": "ENTITYPER",
                            "document_token_index": 5,
                            "first_document_token_index": 4,
                            "last_document_token_index": 5,
                            "structurally_matched_document_token_index": 10,
                            "document_subword_index": None,
                            "document_subword_containing_token_index": None,
                            "document_word": "richard hudson",
                            "document_phrase": "Richard Hudson",
                            "match_type": "entity",
                            "negated": False,
                            "uncertain": True,
                            "similarity_measure": 1.0,
                            "involves_coreference": True,
                            "extracted_word": "richard hudson",
                            "depth": 0,
                            "explanation": "Has an entity label matching ENTITYPER.",
                        },
                        {
                            "search_phrase_token_index": 2,
                            "search_phrase_word": "gehen",
                            "document_token_index": 17,
                            "first_document_token_index": 17,
                            "last_document_token_index": 17,
                            "structurally_matched_document_token_index": 17,
                            "document_subword_index": None,
                            "document_subword_containing_token_index": None,
                            "document_word": "gehen",
                            "document_phrase": "gehen",
                            "match_type": "direct",
                            "negated": True,
                            "uncertain": False,
                            "similarity_measure": 1.0,
                            "involves_coreference": False,
                            "extracted_word": "gehen",
                            "depth": 0,
                            "explanation": "Matches GEHEN directly.",
                        },
                        {
                            "search_phrase_token_index": 3,
                            "search_phrase_word": "in",
                            "document_token_index": 14,
                            "first_document_token_index": 14,
                            "last_document_token_index": 14,
                            "structurally_matched_document_token_index": 14,
                            "document_subword_index": None,
                            "document_subword_containing_token_index": None,
                            "document_word": "in",
                            "document_phrase": "in",
                            "match_type": "direct",
                            "negated": True,
                            "uncertain": True,
                            "similarity_measure": 1.0,
                            "involves_coreference": False,
                            "extracted_word": "in",
                            "depth": 0,
                            "explanation": "Matches IN directly.",
                        },
                        {
                            "search_phrase_token_index": 5,
                            "search_phrase_word": "stadt",
                            "document_token_index": 16,
                            "first_document_token_index": 16,
                            "last_document_token_index": 16,
                            "structurally_matched_document_token_index": 16,
                            "document_subword_index": None,
                            "document_subword_containing_token_index": None,
                            "document_word": "stadt",
                            "document_phrase": "die Stadt",
                            "match_type": "direct",
                            "negated": True,
                            "uncertain": False,
                            "similarity_measure": 1.0,
                            "involves_coreference": False,
                            "extracted_word": "stadt",
                            "depth": 0,
                            "explanation": "Matches STADT directly.",
                        },
                    ],
                },
                {
                    "search_phrase_label": "Ein ENTITYPER geht in die Stadt",
                    "search_phrase_text": "Ein ENTITYPER geht in die Stadt",
                    "document": "",
                    "index_within_document": 17,
                    "sentences_within_document": "Letzte Woche sah ich Richard Hudson und Max Mustermann. Sie wollten nicht mehr in die Stadt gehen.",
                    "negated": True,
                    "uncertain": True,
                    "involves_coreference": True,
                    "overall_similarity_measure": 1.0,
                    "word_matches": [
                        {
                            "search_phrase_token_index": 1,
                            "search_phrase_word": "ENTITYPER",
                            "document_token_index": 8,
                            "first_document_token_index": 7,
                            "last_document_token_index": 8,
                            "structurally_matched_document_token_index": 10,
                            "document_subword_index": None,
                            "document_subword_containing_token_index": None,
                            "document_word": "max mustermann",
                            "document_phrase": "Max Mustermann",
                            "match_type": "entity",
                            "negated": False,
                            "uncertain": True,
                            "similarity_measure": 1.0,
                            "involves_coreference": True,
                            "extracted_word": "max mustermann",
                            "depth": 0,
                            "explanation": "Has an entity label matching ENTITYPER.",
                        },
                        {
                            "search_phrase_token_index": 2,
                            "search_phrase_word": "gehen",
                            "document_token_index": 17,
                            "first_document_token_index": 17,
                            "last_document_token_index": 17,
                            "structurally_matched_document_token_index": 17,
                            "document_subword_index": None,
                            "document_subword_containing_token_index": None,
                            "document_word": "gehen",
                            "document_phrase": "gehen",
                            "match_type": "direct",
                            "negated": True,
                            "uncertain": False,
                            "similarity_measure": 1.0,
                            "involves_coreference": False,
                            "extracted_word": "gehen",
                            "depth": 0,
                            "explanation": "Matches GEHEN directly.",
                        },
                        {
                            "search_phrase_token_index": 3,
                            "search_phrase_word": "in",
                            "document_token_index": 14,
                            "first_document_token_index": 14,
                            "last_document_token_index": 14,
                            "structurally_matched_document_token_index": 14,
                            "document_subword_index": None,
                            "document_subword_containing_token_index": None,
                            "document_word": "in",
                            "document_phrase": "in",
                            "match_type": "direct",
                            "negated": True,
                            "uncertain": True,
                            "similarity_measure": 1.0,
                            "involves_coreference": False,
                            "extracted_word": "in",
                            "depth": 0,
                            "explanation": "Matches IN directly.",
                        },
                        {
                            "search_phrase_token_index": 5,
                            "search_phrase_word": "stadt",
                            "document_token_index": 16,
                            "first_document_token_index": 16,
                            "last_document_token_index": 16,
                            "structurally_matched_document_token_index": 16,
                            "document_subword_index": None,
                            "document_subword_containing_token_index": None,
                            "document_word": "stadt",
                            "document_phrase": "die Stadt",
                            "match_type": "direct",
                            "negated": True,
                            "uncertain": False,
                            "similarity_measure": 1.0,
                            "involves_coreference": False,
                            "extracted_word": "stadt",
                            "depth": 0,
                            "explanation": "Matches STADT directly.",
                        },
                    ],
                },
            ],
        )
