import json
import sys
import os


def escape_string(s):
    return json.dumps(s)


def hid_usagepages_json2cppheader(json_file, header_file):
    print(f"Opening JSON file:      {os.path.abspath(json_file)}")
    with open(json_file, "r", encoding="utf-8") as file:
        data = json.load(file)

    print(f"Generating header file: {os.path.abspath(header_file)}")
    with open(header_file, "w", encoding="utf-8") as file:
        header_comment = """\
// This file was generated by hid_usagepages_json2cppheader.py
// ### Do not edit this file directly! ###
//
// The file HidUsageTables.json is the official HID Usage definition file
// from the standard downloaded from https://www.usb.org/hid
//
// HID Usage Tables
// The HID Usage Tables 1.5 document defines constants (Usages) that can be
// interpreted by an application to identify the purpose and meaning of a data
// field in a HID report. Usages are also used to define the meaning of groups
// of related data items. This is accomplished by the hierarchical assignment
// of Usage information to Collections.
// The HID Usage Tables 1.5 document also includes all Usage definitions as a
// JSON file as an attachment to the PDF. The PDF serves as the 'single' source
// of truth.

#pragma once

#include <QString>
#include <QVector>

namespace mixxx {

namespace hid {

struct Usage {
    uint16_t id;
    QString name;
};

struct UsagePage {
    uint16_t id;
    QString name;
    QVector<Usage> usages;
};

const QVector<UsagePage> kHidUsagePages = {
"""
        file.write(header_comment)

        for page in data["UsagePages"]:
            file.write(
                "    { 0x%04X, QStringLiteral(%s), {\n"
                % (page["Id"], escape_string(page["Name"]))
            )
            for usage in page["UsageIds"]:
                file.write(
                    "        { 0x%04X, QStringLiteral(%s) },\n"
                    % (usage["Id"], escape_string(usage["Name"]))
                )
            file.write("    } },\n")
        file.write("};\n\n")

        file.write("} // namespace hid\n\n")
        file.write("} // namespace mixxx\n")


if __name__ == "__main__":
    if len(sys.argv) != 3:
        print(
            "Usage:   python hid_usagepages_json2cppheader.py <input_json> "
            "<output_header>"
        )
        print(
            "Example: python hid_usagepages_json2cppheader.py "
            "..\\src\\controllers\\hid\\HidUsageTables.json "
            "..\\src\\controllers\\hid\\hidusagetablesdata.h"
        )
        sys.exit(1)
    hid_usagepages_json2cppheader(sys.argv[1], sys.argv[2])
